#include <string.h>
#include <errno.h>
#include <glib.h>
#include "edv_path.h"
#include "edv_device.h"
#include "edv_context.h"
#include "edv_context_private.h"
#include "edv_device_get.h"


GList *edv_devices_list(EDVContext *ctx);

EDVDevice *edv_devices_list_match_mount_path(
	EDVContext *ctx,
	const gchar *mount_path
);
EDVDevice *edv_devices_list_match_device_path(
	EDVContext *ctx,
	const gchar *device_path
);
EDVDevice *edv_devices_list_match_path(
	EDVContext *ctx,
	const gchar *path
);

void edv_devices_list_update_mount_states(EDVContext *ctx);
void edv_devices_list_update_stats(EDVContext *ctx);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the Devices list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	Returns a GList of EDVDevice * devices, the returned
 *	list must not be modified or deleted.
 */
GList *edv_devices_list(EDVContext *ctx)
{
	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return(ctx->devices_list);
}


/*
 *	Matches the EDVDevice by mount path.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The mount_path specifies the mount path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
EDVDevice *edv_devices_list_match_mount_path(
	EDVContext *ctx,
	const gchar *mount_path
)
{
	GList *glist;
	EDVDevice *d;

	if((ctx == NULL) || STRISEMPTY(mount_path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Find the Device who's mount path matches the specified
	 * mount path
	 */
	for(glist = ctx->devices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->mount_path == NULL)
			continue;

		if(!strcmp(
			(const char *)d->mount_path,
			(const char *)mount_path
		))
			return(d);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Matches the EDVDevice by device path.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The device_path specifies the device path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
EDVDevice *edv_devices_list_match_device_path(
	EDVContext *ctx,
	const gchar *device_path
)
{
	GList *glist;
	EDVDevice *d;

	if((ctx == NULL) || STRISEMPTY(device_path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Find the Device who's device path matches the specified
	 * device path
	 */
	for(glist = ctx->devices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->device_path == NULL)
			continue;

		if(!strcmp(
			(const char *)d->device_path,
			(const char *)device_path
		))
			return(d);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Matches the EDVDevice by a VFS object that resides on it.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to any object on the device
 *	including the device's mount point itself but not the device's
 *	device path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
EDVDevice *edv_devices_list_match_path(
	EDVContext *ctx,
	const gchar *path
)
{
	GList *glist;
	EDVDevice *d;

	if((ctx == NULL) || STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Find the Device who's mount path is a prefix of the
	 * specified path
	 */
	for(glist = ctx->devices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->mount_path == NULL)
			continue;

		if(edv_path_is_parent(
			d->mount_path,
			path
		))
			return(d);
	}    

	errno = ENOENT;
	 
	return(NULL);
}


/*
 *	Updates all the EDVDevices' mount states with the current
 *	mount states obtained from the system.
 *
 *	Each Device's flags member's EDV_DEVICE_MOUNTED flag will
 *	be either set or unset.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_devices_list_update_mount_states(EDVContext *ctx)
{
	if(ctx == NULL)
		return;

	g_list_foreach(
		ctx->devices_list,
		(GFunc)edv_device_update_mount_state,
		NULL
	);
}


/*
 *	Updates all the EDVDevices' statistics with the current
 *	statistics obtained from the system.
 *
 *	Each Device's blocks_total, blocks_available, blocks_free,
 *	block_size, indicies_total, indicies_available, indicies_free,
 *	and name_length_max members will be updated by this call.
 *
 *	edv_devices_list_update_mount_states() should always be called
 *	prior to this call to ensure that the mount states are up
 *	to date.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_devices_list_update_stats(EDVContext *ctx)
{
	if(ctx == NULL)
		return;

	g_list_foreach(
		ctx->devices_list,
		(GFunc)edv_device_update_stats,
		NULL
	);
}
