/*
		   Endeavour Mark II API - Device
 */

#ifndef EDV_DEVICE_H
#define EDV_DEVICE_H

#include <glib.h>
#include "edv_types.h"
#include "edv_fs_type.h"


typedef struct _EDVDevice		EDVDevice;
#define EDV_DEVICE(p)			((EDVDevice *)(p))


/*
 *	Device Flags:
 */
typedef enum {
	EDV_DEVICE_MOUNTED		= (1 << 0),	/* Device is mounted */
	EDV_DEVICE_READ_ONLY		= (1 << 1),	/* Mount as read-only,
							 * making all objects
							 * on the device
							 * read-only */
	EDV_DEVICE_NO_UNMOUNT		= (1 << 2),	/* May not be unmounted */
	EDV_DEVICE_UNLISTED		= (1 << 3),	/* Do not show on
							 * "brief" listings */
	EDV_DEVICE_NO_SCAN		= (1 << 4),	/* Do not scan this
							 * device's contents
							 * unless absolutely
							 * nessisary (useful
							 * for NFS mounted
							 * devices) */
} EDVDeviceFlags;

/*
 *	Device Icon States:
 */
typedef enum {
	EDV_DEVICE_ICON_STATE_STANDARD,
	EDV_DEVICE_ICON_STATE_SELECTED,		/* Selected or expanded */
	EDV_DEVICE_ICON_STATE_UNMOUNTED		/* Not mounted */
} EDVDeviceIconState;
#define EDV_DEVICE_TOTAL_ICON_STATES		3

#define EDV_DEVICE_ICON_STATE_NAMES	{	\
	"Standard",				\
	"Selected",				\
	"Unmounted"				\
}  


/*
 *	Device:
 */
struct _EDVDevice {

	EDVDeviceFlags	flags;

	gchar		*name;		/* Descriptive name */

	gchar		*device_path;	/* Path to device object */
	gchar		*mount_path;	/* Path to mount point directory */

	gchar		*fs_type_name;  /* Conical name for the Filesystem type */
	EDVFSTypeCode	fs_type_code;	/* Filesystem ID, one of EDV_FS_TYPE_* */

	/* Small 20x20 icons */
	gchar		*small_icon_file[EDV_DEVICE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons */
	gchar		*medium_icon_file[EDV_DEVICE_TOTAL_ICON_STATES];

	/* Large 48x48 icons */
	gchar		*large_icon_file[EDV_DEVICE_TOTAL_ICON_STATES];

	/* Mount and unmount commands, if either of these are not NULL
	 * then its respective command will be issued instead of a mount
	 * performed internally with system calls by this program
	 */
	gchar		*command_mount,
			*command_unmount,
			*command_eject;

	/* Commands to run external programs for device maintainance */
	gchar		*command_check,
			*command_tools,
			*command_format;

	/* Statistics */
	gulong		blocks_total,
			blocks_available,	/* Available to effective user
						 * id */
			blocks_free;

	gulong		block_size;		/* Bytes per block */

	gulong		indicies_total,
			indicies_available,	/* Available to effective user
                                                 * id */
			indicies_free;

	gulong		name_length_max;	/* Maximum length allowed for
						 * an object's name In bytes */

	/* Time Stamps (in seconds since EPOCH) */
	gulong		last_mount_time,
			last_check_time;	/* Last time command_check was
						 * runned */

};

#define EDV_DEVICE_IS_MOUNTED(p)	(((p) != NULL) ? \
 (((p)->flags & EDV_DEVICE_MOUNTED) ? TRUE : FALSE) : FALSE)

#define EDV_DEVICE_IS_READ_ONLY(p)	(((p) != NULL) ? \
 (((p)->flags & EDV_DEVICE_READ_ONLY) ? TRUE : FALSE) : FALSE)

#define EDV_DEVICE_IS_NO_UNMOUNT(p)	(((p) != NULL) ? \
 (((p)->flags & EDV_DEVICE_NO_UNMOUNT) ? TRUE : FALSE) : FALSE)

#define EDV_DEVICE_IS_UNLISTED(p)	(((p) != NULL) ? \
 (((p)->flags & EDV_DEVICE_UNLISTED) ? TRUE : FALSE) : FALSE)

#define EDV_DEVICE_IS_NO_SCAN(p)	(((p) != NULL) ? \
 (((p)->flags & EDV_DEVICE_NO_SCAN) ? TRUE : FALSE) : FALSE)


/*
 *	Creates a new EDVDevice.
 */
extern EDVDevice *edv_device_new(void);

/*
 *	Coppies the EDVDevice.
 */
extern EDVDevice *edv_device_copy(EDVDevice *d);

/*
 *	Deletes the EDVDevice.
 */
extern void edv_device_delete(EDVDevice *d);


/*
 *	Gets the EDVDevice's icon path.
 *
 *	The d specifies the EDVDevice.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	Returns the path to the icon file or NULL on error.
 */
extern const gchar *edv_device_get_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state
);

/*
 *	Gets the EDVDevice's icon path.
 *
 *	The d specifies the Device.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	If accept_smaller_size is TRUE then if the requested size is
 *	not available then a smaller size will be returned if one is
 *	available.
 *
 *	If accept_standard_state is TRUE then if the requested state
 *	is not available then the EDV_DEVICE_ICON_STATE_STANDARD
 *	will be returned if it is available. If the
 *	EDV_DEVICE_ICON_STATE_STANDARD is still not available then
 *	a smaller size will be returned if accept_smaller_size is TRUE.
 *
 *	Returns the path to the icon file or NULL on error.
 */
extern const gchar *edv_device_get_best_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state,
	const gboolean accept_smaller_size,
	const gboolean accept_standard_state
);


/*
 *	Updates the EDVDevice's mount state with the current mount
 *	state obtained from the system.
 *
 *	The Device's flags member's EDV_DEVICE_MOUNTED flag will
 *	be either set or unset.
 *
 *	The d specifies the EDVDevice.
 */
extern void edv_device_update_mount_state(EDVDevice *d);

/*
 *	Updates the Device's statistics with the statistics obtained
 *	from the system.
 *
 *	The Device's blocks_total, blocks_available, blocks_free,
 *	block_size, indicies_total, indicies_free, fs_id, and
 *	name_length_max members will be updated by this call.
 *
 *	edv_device_update_mount_state() should be called prior to
 *	this to ensure that the mount states are up to date.
 *
 *	The d specifies the EDVDevice.
 */
extern void edv_device_update_stats(EDVDevice *d);


#endif	/* EDV_DEVICE_H */
