#include <string.h>
#include <errno.h>
#if defined(__FreeBSD__)
# include <fstab.h>
# include <sys/param.h>
# include <sys/ucred.h>
# include <sys/mount.h>
#elif defined(__SOLARIS__)
# include <sys/mnttab.h>
# include <sys/vfstab.h>
#else
# include <mntent.h>
# include <sys/vfs.h>
#endif
#include <glib.h>
#include "edv_types.h"
#include "edv_fs_type.h"
#include "edv_device.h"
#include "config.h"


/* EDVDevice */
EDVDevice *edv_device_new(void);
EDVDevice *edv_device_copy(EDVDevice *d);
void edv_device_delete(EDVDevice *d);

/* Get Icon Path */
const gchar *edv_device_get_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state 
);
const gchar *edv_device_get_best_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state,
	const gboolean accept_smaller_size,
	const gboolean accept_standard_state
);

/* Mount States */
void edv_device_update_mount_state(EDVDevice *d);

/* Statistics */
void edv_device_update_stats(EDVDevice *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVDevice.
 */
EDVDevice *edv_device_new(void)
{
	return(EDV_DEVICE(g_malloc0(sizeof(EDVDevice))));
}

/*
 *	Coppies the EDVDevice.
 */
EDVDevice *edv_device_copy(EDVDevice *d)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;
	const EDVDevice *src = d;
	EDVDevice *tar;

	if(src == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar = edv_device_new();
	if(tar == NULL)
		return(NULL);

	tar->flags = src->flags;
	tar->name = STRDUP(src->name);
	tar->device_path = STRDUP(src->device_path);
	tar->mount_path = STRDUP(src->mount_path);
	tar->fs_type_name = STRDUP(src->fs_type_name);
	tar->fs_type_code = src->fs_type_code;

	for(i = 0; i < nicon_states; i++)
	{
		tar->small_icon_file[i] = STRDUP(src->small_icon_file[i]);
		tar->medium_icon_file[i] = STRDUP(src->medium_icon_file[i]);
		tar->large_icon_file[i] = STRDUP(src->large_icon_file[i]);
	}

	tar->command_mount = STRDUP(src->command_mount);
	tar->command_unmount = STRDUP(src->command_unmount);
	tar->command_eject = STRDUP(src->command_eject);

	tar->command_check = STRDUP(src->command_check);
	tar->command_tools = STRDUP(src->command_tools);
	tar->command_format = STRDUP(src->command_format);

	tar->blocks_total = src->blocks_total;
	tar->blocks_available = src->blocks_available;
	tar->blocks_free = src->blocks_free;

	tar->block_size = src->block_size;

	tar->indicies_total = src->indicies_total;
	tar->indicies_available = src->indicies_available;
	tar->indicies_free = src->indicies_free;

	tar->name_length_max = src->name_length_max;

	tar->last_mount_time = src->last_mount_time;
	tar->last_check_time = src->last_check_time;

	return(tar);
}

/*
 *	Deletes the EDVDevice.
 */
void edv_device_delete(EDVDevice *d)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;

	if(d == NULL)
		return;

	for(i = 0; i < nicon_states; i++)
	{
		g_free(d->small_icon_file[i]);
		g_free(d->medium_icon_file[i]);
		g_free(d->large_icon_file[i]);
	}

	g_free(d->command_mount);
	g_free(d->command_unmount);
	g_free(d->command_eject);

	g_free(d->command_check);
	g_free(d->command_tools);
	g_free(d->command_format);

	g_free(d->name);
	g_free(d->device_path);
	g_free(d->mount_path);
	g_free(d->fs_type_name);

	g_free(d);
}


/*
 *	Gets the EDVDevice's icon path.
 *
 *	The d specifies the EDVDevice.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	Returns the path to the icon file or NULL on error.
 */
const gchar *edv_device_get_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state 
)
{
	if((d == NULL) ||
	   (state < 0) || (state >= EDV_DEVICE_TOTAL_ICON_STATES)
	)
	{
		errno = EINVAL;
		return(NULL);
	}

	switch(size)
	{
	  case EDV_ICON_SIZE_16:
		return(NULL);                       /* Not supported */
		break;
	  case EDV_ICON_SIZE_20:
		return(d->small_icon_file[state]);
		break;
	  case EDV_ICON_SIZE_32:
		return(d->medium_icon_file[state]);
		break;
	  case EDV_ICON_SIZE_48:
		return(d->large_icon_file[state]);
		break;
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the EDVDevice's icon path.
 *
 *	The d specifies the Device.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 * 
 *	If accept_smaller_size is TRUE then if the requested size is
 *	not available then a smaller size will be returned if one is
 *	available.
 *
 *	If accept_standard_state is TRUE then if the requested state
 *	is not available then the EDV_DEVICE_ICON_STATE_STANDARD
 *	will be returned if it is available. If the
 *	EDV_DEVICE_ICON_STATE_STANDARD is still not available then
 *	a smaller size will be returned if accept_smaller_size is TRUE.
 *
 *	Returns the path to the icon file or NULL on error.
 */
const gchar *edv_device_get_best_icon_path(
	EDVDevice *d,
	const EDVIconSize size,
	const EDVDeviceIconState state,
	const gboolean accept_smaller_size,
	const gboolean accept_standard_state
)
{
	const gchar *path;

	if(d == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	path = edv_device_get_icon_path(
		d,
		size,
		state
	);
	if(path != NULL)
		return(path);

	if(accept_standard_state && (state != EDV_DEVICE_ICON_STATE_STANDARD))
	{
		path = edv_device_get_icon_path(
			d,
			size,
			EDV_DEVICE_ICON_STATE_STANDARD
		);
		if(path != NULL)
			return(path);
	}

	if(accept_smaller_size)
	{
		gint smaller_size;

		while(TRUE)
		{
			smaller_size = 0;
			switch(size)
			{
			  case EDV_ICON_SIZE_16:
				break;
			  case EDV_ICON_SIZE_20:
				smaller_size = EDV_ICON_SIZE_16;
				break;
			  case EDV_ICON_SIZE_32:
				smaller_size = EDV_ICON_SIZE_20;
				break;
			  case EDV_ICON_SIZE_48:
				smaller_size = EDV_ICON_SIZE_32;
				break;
			}
			if(smaller_size == 0)
				break;

			path = edv_device_get_icon_path(
				d,
				smaller_size,
				state
			);
			if(path != NULL)
				return(path);

			if(accept_standard_state)
			{
				path = edv_device_get_icon_path(
					d,
					smaller_size,
					EDV_DEVICE_ICON_STATE_STANDARD
				);
				if(path != NULL)
					return(path);
			}
		}

		errno = ENOENT;
		return(NULL);
	}
	else
	{
		errno = ENOENT;
		return(NULL);
	}
}


/*
 *	Updates the EDVDevice's mount state with the current mount
 *	state obtained from the system.
 *
 *	The Device's flags member's EDV_DEVICE_MOUNTED flag will
 *	be either set or unset.
 *
 *	The d specifies the EDVDevice.
 */
void edv_device_update_mount_state(EDVDevice *d)
{
#if defined(__FreeBSD__)
	gint            i,
			mntsize;
	struct statfs *mt_buf;
#elif defined(__SOLARIS__)
	FILE *fp;
	gint mtback;
	struct mnttab *mt_ptr;
#else
	FILE *fp;
	struct mntent *mt_ptr;
#endif
	const gchar     *device_path,
			*mount_path;

	if(d == NULL)
		return;

	/* Reset the device's mount state */
	d->flags &= ~EDV_DEVICE_MOUNTED;

	/* Open the system's mounted devices list file */
#if defined(__FreeBSD__)
	/* Do nothing */
#elif defined(__SOLARIS__)
	fp = fopen(PATH_ETC_MNTTAB, "rb");
	if(fp == NULL)
		return;
#else
	fp = setmntent(PATH_PROC_MOUNTS, "rb");
	if(fp == NULL)
		return;
#endif

	/* Begin reading the system's mounted devices list file */
#if defined(__FreeBSD__)
	mntsize = getmntinfo(
		&mt_buf,
		MNT_NOWAIT
	);
	if(mntsize == 0)
		return;
	i = mntsize - 1;
	while(i >= 0)
#elif defined(__SOLARIS__)
	mt_ptr = (struct mnttab *)g_malloc(sizeof(struct mnttab));
	mtback = getmntent(
		fp,
		mt_ptr
	);
	while(mtback != 0)
#else
	mt_ptr = getmntent(fp);
	while(mt_ptr != NULL)
#endif
	{
		/* Check if this mount path's device path matches this
		 * Device's device path
		 */
		device_path = (const gchar *)(
#if defined(__FreeBSD__)
			mt_buf[i].f_mntfromname
#elif defined(__SOLARIS__)
			mt_ptr->mnt_special
#else
			mt_ptr->mnt_fsname
#endif
		);
		if(!STRISEMPTY(device_path) && !STRISEMPTY(d->device_path))
		{
			if(!strcmp(
				(const char *)device_path,
				(const char *)d->device_path)
			)
			{
				/* Mark this device as mounted */
				d->flags |= EDV_DEVICE_MOUNTED;
				break;
			}
		}

		/* Also check if this mount path's mounted path matches
		 * this Device's mounted path
		 */
		mount_path = (const gchar *)(
#if defined(__FreeBSD__)
			mt_buf[i].f_mntonname
#elif defined(__SOLARIS__)
			mt_ptr->mnt_mountp
#else
			mt_ptr->mnt_dir
#endif
		);
		if(!STRISEMPTY(d->mount_path) && !STRISEMPTY(mount_path))
		{
			if(!strcmp(
				(const char *)d->mount_path,
				(const char *)mount_path)
			)
			{
				/* Mark this device as mounted */
				d->flags |= EDV_DEVICE_MOUNTED;
				break;
			}
		}

		/* Get the next mount entry */
#if defined(__FreeBSD__)
		i--;
#elif defined(__SOLARIS__)
		mtback = getmntent(fp, mt_ptr);
#else
		mt_ptr = getmntent(fp);
#endif
	}

	/* Close the system's mounted devices list file */
#if defined(__FreeBSD__)
	/* Do nothing */
#elif defined(__SOLARIS__)
	(void)fclose(fp);
	g_free(mt_ptr);
#else
	endmntent(fp);
#endif
}


/*
 *	Updates the Device's statistics with the statistics obtained
 *	from the system.
 *
 *	The Device's blocks_total, blocks_available, blocks_free,
 *	block_size, indicies_total, indicies_available, indicies_free,
 *	and name_length_max members will be updated by this call.
 *
 *	edv_device_update_mount_state() should be called prior to
 *	this to ensure that the mount states are up to date.
 *
 *	The d specifies the EDVDevice.
 */
void edv_device_update_stats(EDVDevice *d)
{
	const gchar *mount_path;

	if(d == NULL)
		return;

	mount_path = d->mount_path;
	if(EDV_DEVICE_IS_MOUNTED(d) && !STRISEMPTY(mount_path))
	{
#if defined(__SOLARIS__)
		struct statvfs buf;
		if(!statvfs((const char *)mount_path, &buf))
#else
		struct statfs buf;
		if(!statfs((const char *)mount_path, &buf))
#endif
		{
			/* Check if the block size transfer rate (which is
			 * really just the block size), is larger than or
			 * equal to the base of 1024 bytes per block
			 */
			const gulong block_size = (gulong)buf.f_bsize;
			if(block_size >= 1024l)
			{
				const gulong block_div = block_size / 1024l;

				d->blocks_total = (gulong)buf.f_blocks * block_div;
				d->blocks_available = (gulong)buf.f_bavail * block_div;
				d->blocks_free = (gulong)buf.f_bfree * block_div;
			}
			else if(block_size > 0l)
			{
				/* Block size is less than 1024 bytes but positive */
				const gulong block_div = 1024l / block_size;

				/* Note, block_div is in range of 1 to 1024 */
				d->blocks_total = (gulong)buf.f_blocks / block_div;
				d->blocks_available = (gulong)buf.f_bavail / block_div;
				d->blocks_free = (gulong)buf.f_bfree / block_div;
			}

			d->block_size = block_size;

			d->indicies_total = (gulong)buf.f_files;
#if defined(__SOLARIS__)
			d->indicies_available = (gulong)buf.f_favail;
#else
			d->indicies_available = (gulong)buf.f_ffree;
#endif
			d->indicies_free = (gulong)buf.f_ffree;

#if defined(__linux__)
/*			d->fs_code = (EDVFSTypeCode)*(int *)(&buf.f_fsid); */
#endif
			d->name_length_max = (gulong)buf.f_namelen;
		}
	}
	else
	{
		/* Reset this Device's statistics */
		d->blocks_total = 0l;
		d->blocks_available = 0l;
		d->blocks_free = 0l;
		d->block_size = 0l;
		d->indicies_total = 0l;
		d->indicies_available = 0l;
		d->indicies_free = 0l;
		d->name_length_max = 0l;
	}
}
