#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <glib.h>

#include "edv_date_parse.h"


void edv_date_parse_dmy(
	const gchar *s,
	gint *day, gint *month, gint *year
);
void edv_date_parse_mdy(
	const gchar *s,
	gint *month, gint *day, gint *year
);
void edv_date_parse_ymd(
	const gchar *s,
	gint *year, gint *month, gint *day
);
void edv_date_parse_any_date(
	const gchar *s,
	gint *year, gint *month, gint *day
);
void edv_date_parse_any_time(
	const gchar *s,
	gint *hour, gint *minutes, gint *seconds
);
gulong edv_date_parse_epoch(const gchar *s);


#define EDV_DATE_MONTH_NAMES_CONICAL	{		\
 "Jan", "Feb", "Mar", "Apr", "May", "Jun",	\
 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"	\
}
#define EDV_DATE_MONTH_NAMES_CONICAL_MAX_CHARS	3


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Parses the date string.
 *
 *	The s specifies the date string which must be in one of the
 *	following formats:
 *
 *	"D/M/Y"
 *	"D-M-Y"
 *	"D,M,Y"
 *	"D.M.Y"
 *	"D M Y"
 *
 *	The month M may be either the month's number (1 to 12) or the
 *	month's name.
 *
 *	The year Y must be a 4-digit year.
 *
 *	The day specifies the return value for day in the range of 1
 *	to 32.
 *
 *	The mon specifies the return value for the month in the range
 *	of 1 to 12.
 *
 *	The year specifies the return value for the year.
 */
void edv_date_parse_dmy(
	const gchar *s,
	gint *day, gint *month, gint *year
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse the day */
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);

	/* Parse the month */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = EDV_DATE_MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(!g_strncasecmp(
		    s,
		    month_name[i],
		    EDV_DATE_MONTH_NAMES_CONICAL_MAX_CHARS
		))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse the year */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);
#undef ISDATEDELIM
}

/*
 *	Same as edv_date_parse_dmy() except that the date and time
 *	string is parsed in the format of:
 *
 *	"M D, Y"
 */
void edv_date_parse_mdy(
	const gchar *s,
	gint *month, gint *day, gint *year
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse the month */
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = EDV_DATE_MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(!g_strncasecmp(
		    s,
		    month_name[i],
		    EDV_DATE_MONTH_NAMES_CONICAL_MAX_CHARS
		))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse the day */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);

	/* Parse the year */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);
#undef ISDATEDELIM
}

/*
 *	Same as edv_date_parse_dmy() except that the date and time
 *	string is parsed in the format of:
 *
 *	"Y/M/D"
 *	"Y-M-D"
 *	"Y,M,D"
 *	"Y.M.D"
 *	"Y M D"
 */
void edv_date_parse_ymd(
	const gchar *s,
	gint *year, gint *month, gint *day
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse the year */
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);

	/* Parse the month */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = EDV_DATE_MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(!g_strncasecmp(
		    s,
		    month_name[i],
		    EDV_DATE_MONTH_NAMES_CONICAL_MAX_CHARS
		))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse the day */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);
#undef ISDATEDELIM
}

/*
 *	Parses the date string in any format.
 *
 *	Calls one of EDVDateParseDate*() by determining the date
 *	format of the date string.
 *
 *	The s specifies the date string.
 */
void edv_date_parse_any_date(
	const gchar *s,
	gint *year, gint *month, gint *day
)
{
	const gchar *s2;

	if(day != NULL)
	    *day = 0;
	if(month != NULL)
	    *month = 0;
	if(year != NULL)
	    *year = 0;

	if(STRISEMPTY(s))
	    return;

	/* Seek past any spaces */
	while(ISBLANK(*s))
	    s++;

	/* Determine the format of the date and time string and handle
	 * it accordingly
	 */

	/* M D, Y
	 *
	 * Check if the first value is a month's name
	 */
	if(isalpha(*s))
	{
	    edv_date_parse_mdy(s, month, day, year);
	    return;
	}

	/* Y M D
	 *
	 * Check if the first value is a 4-digit (or greater) year
	 */
	s2 = (const gchar *)strpbrk((const char *)s, " \t/-,.:;");
	if(s2 != NULL)
	{
	    /* Is the first value at least 4 characters long and it
	     * is a numeric value?
	     */
	    if(((s2 - s) >= 4) && isdigit(*s))
	    {
		edv_date_parse_ymd(s, year, month, day);
		return;
	    }
	}
	else
	{
	    /* No deliminators, check if the string is at least 4
	     * characters long and that the first value is a numeric
	     * value
	     */
	    if((STRLEN(s) >= 4) && isdigit(*s))
	    {
		edv_date_parse_ymd(s, year, month, day);
		return;
	    }
	}

	/* D M Y */
	edv_date_parse_dmy(s, day, month, year);
}

/*
 *	Parses the time string.
 *
 *	The s specifies the time string which must be in one of the
 *	following formats:
 *
 *      "H:M[:S][AM|PM]"
 *	"H M[ S][AM|PM]"
 *
 *	To optionally indicate 12-hour time a "AM" or "PM" may be
 *	postfixed to the time string.
 *
 */
void edv_date_parse_any_time(
	const gchar *s,
	gint *hour, gint *minutes, gint *seconds
)
{
#define ISTIMEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)
	if(STRISEMPTY(s))
	{
	    if(hour != NULL)
		*hour = 0;
	    if(minutes != NULL)
		*minutes = 0;
	    if(seconds != NULL)
		*seconds = 0;
	    return;
	}

	/* Parse the hour
	 *
	 * Seek past any spaces
	 */
	while(ISBLANK(*s))
	    s++;

	/* Was a leading 0 in the hour specified? */
	if(*s == '0')
	{
	    /* Seek past the leading 0 if there is another digit after
	     * it
	     */
	    if(isdigit(s[1]))
		s++;
	}

	/* Get the hour */
	if(hour != NULL)
	    *hour = CLIP(ATOI(s), 0, 23);

	/* Parse the minutes
	 *
	 * Seek s past the next time deliminator character
	 */
	while(*s != '\0')
	{
	    if(ISTIMEDELIM(*s))
	    {
		s++;
		break;
	    }
	    s++;
	}

	/* Seek past any spaces */
	while(ISBLANK(*s))
	    s++;

	/* Was a leading 0 in the minutes specified? */
	if(*s == '0')
	{
	    /* Seek past the leading 0 if there is another digit after
	     * it
	     */
	    if(isdigit(s[1]))
		s++;
	}

	/* Get the minutes */
	if(minutes != NULL)
	    *minutes = CLIP(ATOI(s), 0, 59);

	/* Seek past the digits and any spaces to the postfix (if any) */
	while(isdigit(*s))
	    s++;
	while(ISBLANK(*s))
	    s++;

	/* Is there a PM postfix? */
	if(toupper(*s) == 'P')
	{
	    /* There is a PM postfix, add 12 hours */
	    if(hour != NULL)
	    {
		if(*hour < 12)
		    *hour += 12;
	    }
	}

	/* Parse the seconds
	 *
	 * Seek s past the next time deliminator character
	 */
	while(*s != '\0')
	{
	    if(ISTIMEDELIM(*s))
	    {
		s++;
		break;
	    }
	    s++;
	}

	/* Seek past any spaces */
	while(ISBLANK(*s))
	    s++;

	/* Was a leading 0 in the minutes specified? */
	if(*s == '0')
	{
	    /* Seek past the leading 0 if there is another digit after
	     * it
	     */
	    if(isdigit(s[1]))
		s++;
	}

	/* Get the seconds */
	if(seconds != NULL)
	    *seconds = CLIP(ATOI(s), 0, 59);

	/* Seek past the digits and any spaces to the postfix (if any) */
	while(isdigit(*s))
	    s++;
	while(ISBLANK(*s))
	    s++;

	/* Is there a PM postfix? */
	if(toupper(*s) == 'P')
	{
	    /* There is a PM postfix, add 12 hours */
	    if(hour != NULL)
	    {
		if(*hour < 12)
		    *hour += 12;
	    }
	}
#undef ISTIMEDELIM
}


/*
 *	Parses the date and time string in any format.
 *
 *	The s specifies the date and time string.
 *
 *	Returns the number of seconds since EPOCH.
 */
gulong edv_date_parse_epoch(const gchar *s)
{
	struct tm t;
	const gchar *s2;
	gint	hour = 0, minutes = 0, seconds = 0,
		year = 0, month = 0, day = 0;

	if(STRISEMPTY(s))
	    return(0l);

	/* Seek past any spaces */
	while(ISBLANK(*s))
	    s++;

	/* Check if the time was specified */
	s2 = (const gchar *)strchr((const char *)s, ':');
	if(s2 != NULL)
	{
	    /* The time was specified */
	    gchar *s_time;

	    /* Seek s2 to the start of the time */
	    if((s2 - s) >= 2)
		s2 -= 2;
	    else if((s2 - s) >= 1)
		s2 -= 1;
	    while(ISBLANK(*s2))
		s2++;

	    /* Get/copy just the time string from s2 */
	    s_time = STRDUP(s2);
	    if(s_time != NULL)
	    {
		/* Null terminate the time string */
		gchar *s3 = (gchar *)strpbrk((char *)s_time, " \t");
		if(s3 != NULL)
		    *s3 = '\0';

		/* Parse the time */
		edv_date_parse_any_time(s_time, &hour, &minutes, &seconds);
		g_free(s_time);
	    }

	    /* Was the time at the beginning? */
	    if(s2 == s)
	    {
		/* Seek past the time */
		while(*s2 != '\0')
		{
		    if(ISBLANK(*s2))
			break;

		    s2++;
		}

		/* Parse the date */
		edv_date_parse_any_date(s2, &year, &month, &day);
	    }
	    else
	    {
		/* Time was at the end */
		edv_date_parse_any_date(s, &year, &month, &day);
	    }
	}
	else
	{
	    /* No time, parse only the date */
	    edv_date_parse_any_date(s, &year, &month, &day);
	}

	t.tm_sec = seconds;
	t.tm_min = minutes;
	t.tm_hour = hour;
	t.tm_mday = day;
	t.tm_mon = month - 1;
	t.tm_year = year - 1900;
	t.tm_wday = 0;
	t.tm_yday = 0;
	t.tm_isdst = -1;

	return((gulong)mktime(&t));
}
