#include <string.h>
#include <errno.h>
#include <time.h>
#include <glib.h>
#include "edv_utils.h"
#include "edv_context.h"
#include "edv_context_private.h"
#include "edv_get.h"
#include "edv_date_format.h"
#include "edv_cfg_list.h"


const gchar *edv_date_format_current(
	EDVContext *ctx,
	const gulong t
);
const gchar *edv_date_format_absolute(
	EDVContext *ctx,
	const gulong t
);
const gchar *edv_date_format_absolute_string(
	EDVContext *ctx,
	const gulong t,
	const gchar *format
);
const gchar *edv_date_format(
	EDVContext *ctx,
	const gulong t
);
const gchar *edv_date_format_duration(
	EDVContext *ctx,
	const gulong dt
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Formats a date and time relative to the current time in
 *	accordance with the configuration specified on the context.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	Returns a statically allocated string describing the date and
 *	time or NULL on error.
 */
const gchar *edv_date_format_current(
	EDVContext *ctx,
	const gulong t
)
{
	gulong		ct,
			dt;
	gchar **s_ptr;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	s_ptr = &ctx->date_string_buf;

	if(t == 0l)
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup("");
		return(*s_ptr);
	}

	/* Calculate the delta time from the current time minus the
	 * specified time
	 */
	dt = edv_time() - t;

	/* Less than one second? */
	if(dt < 1l)
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup(
"less than a second ago"
		);
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
		ct = MAX((dt / 1l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "second" : "seconds")
		);
	}
	/* Less than one hour? */
	else if(dt < (60l * 60l))
	{
		ct = MAX((dt / 60l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "minute" : "minutes")
		);
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "hour" : "hours")
		);
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "day" : "days")
		);
	}
	/* Less than one month (30 days)? */
	else if(dt < (30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "week" : "weeks")
		);
	}
	/* Less than 6 months ago? */
	else if(dt < (6l * 30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s ago",
			ct, ((ct == 1l) ? "month" : "months")
		);
	}
	/* 6 months or longer ago */
	else
	{
		/* All else revert to using ctime() */
		time_t tt = (time_t)t;
		const gchar *ts = (const gchar *)ctime(&tt);
		if(ts != NULL)
		{
			g_free(*s_ptr);
			*s_ptr = g_strdup(ts);
			if(*s_ptr != NULL)
			{
				gchar *s = (gchar *)strchr(
					(char *)*s_ptr,
					'\n'
				);
				if(s != NULL)
					*s = '\0';
			}
		}
		else
		{
			g_free(*s_ptr);
			*s_ptr = g_strdup("");
		}
	}

	return(*s_ptr);
}

/*
 *	Formats a date and time string relative to the Common Era of
 *	current locale in accordance with the configuration specified
 *	on the context.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	Returns a statically allocated string describing the date and
 *	time or NULL on error.
 */
const gchar *edv_date_format_absolute(
	EDVContext *ctx,
	const gulong t
)
{
	return(edv_date_format_absolute_string(
		ctx,
		t,
		NULL
	));
}

/*
 *	Formats a date and time string relative to the Common Era of
 *	current locale by format.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	The format specifies the format string containing strftime(2)
 *	tokens. If format is NULL then the format specified by the
 *	configuration on the context will be used.
 *
 *	Returns a statically allocated string describing the date and
 *	time or NULL on error.
 */
const gchar *edv_date_format_absolute_string(
	EDVContext *ctx,
	const gulong t,
	const gchar *format
)
{
	time_t tt;
	const struct tm *tm_ptr;
	gchar **s_ptr;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	s_ptr = &ctx->date_string_buf;

	if(t == 0l)
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup("");
		return(*s_ptr);
	}

	/* Get the date and time format */
	if(format == NULL)
		format = edv_get_s(ctx, EDV_CFG_PARM_DATE_FORMAT);
	if(STRISEMPTY(format))
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup("");
		return(*s_ptr);
	}

	/* Convert the time value for localtime() */
	tt = (time_t)t;

	/* Get the time value break down for strftime() */
	tm_ptr = localtime(&tt);
	if(tm_ptr != NULL)
	{
		/* Calculate the length of the return string based on
		 * the number of tokens encountered
		 */
		gint len = 0;
		const gchar *s = format;
		while(*s != '\0')
		{
			if(*s == '%')
				len += 80;
			else
				len++;
			s++;
		}
		len++;				/* Count the null byte */

		/* Allocate the return string */
		g_free(*s_ptr);
		*s_ptr = (gchar *)g_malloc(len * sizeof(gchar));
		if(*s_ptr == NULL)
			return(NULL);

		/* Format the date and time string relative to the Common
		 * Era of the current locale
		 */
		if(strftime(
			(char *)*s_ptr, (size_t)len,
			(const char *)format,
			tm_ptr
		) <= 0l)
			**s_ptr = '\0';
	}
	else
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup("");
	}

	return(*s_ptr);
}

/*
 *	Formats a date and time string with relativity and format
 *	specified by the configuration.
 *
 *	The format and relativity of the date & time is specified by
 *	the configuration on the context.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	Returns a statically allocated string describing the date and
 *	time specified by the configuration or NULL on error.
 */
const gchar *edv_date_format(
	EDVContext *ctx,
	const gulong t
)
{
	EDVDateRelativity relativity;

	if(ctx == NULL)
		return(NULL);

	/* Get the relativity specified by the configuration */
	relativity = edv_get_i(ctx, EDV_CFG_PARM_DATE_RELATIVITY);

	/* Handle by relativity */
	switch(relativity)
	{
	  case EDV_DATE_RELATIVITY_ABSOLUTE:
		return(edv_date_format_absolute(ctx, t));
		break;

	  case EDV_DATE_RELATIVITY_CURRENT:
		return(edv_date_format_current(ctx, t));
		break;
	}

	return("");
}

/*
 *	Formats a date and time string describing the time lapsed in
 *	the format of "<n1> <units1> <n2> <units2>".
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The dt specifies the time lapsed in seconds. If dt is 0 then
 *	"less than a second" will be returned.
 *
 *	Returns a statically allocated string describing the date and
 *	time specified by the configuration or NULL on error.
 */
const gchar *edv_date_format_duration(
	EDVContext *ctx,
	const gulong dt
)
{
	gulong		ct,
			ct2;
	gchar **s_ptr;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	s_ptr = &ctx->date_string_buf;

	/* Less than one second? */
	if(dt < 1l)
	{
		g_free(*s_ptr);
		*s_ptr = g_strdup(
"less than a second"
		);
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
		ct = MAX((dt / 1l), 1l);
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s",
			ct, ((ct == 1l) ? "second" : "seconds")
		);
	}
	/* Less than one hour? */
	else if(dt < (60l * 60l))
	{
		ct = MAX((dt / 60l), 1l);
		ct2 = MAX((dt / 1l), 1l) % 60l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "minute" : "minutes"),
				ct2, ((ct2 == 1l) ? "second" : "seconds")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "minute" : "minutes")
			);
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l), 1l);
		ct2 = MAX((dt / 60l), 1l) % 60l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "hour" : "hours"),
				ct2, ((ct2 == 1l) ? "minute" : "minutes")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "hour" : "hours")
			);
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l), 1l);
		ct2 = MAX((dt / 60l / 60l), 1l) % 24l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "day" : "days"),
				ct2, ((ct2 == 1l) ? "hour" : "hours")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "day" : "days")
			);
	}
	/* Less than one month (30 days)? */
	else if(dt < (30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 7l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "week" : "weeks"),
				ct2, ((ct2 == 1l) ? "day" : "days")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "week" : "weeks") 
			);
	}
#if 0
	/* Less than 6 months ago? */
	else if(dt < (6l * 30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
		g_free(*s_ptr);
		*s_ptr = g_strdup_printf(
			"%ld %s %ld %s",
			ct, ((ct == 1l) ? "month" : "months"),
			ct2, ((ct2 == 1l) ? "day" : "days")
		);
	}
#endif
	/* Less than a year (365 days)? */
	else if(dt < (12l * 30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "month" : "months"),
				ct2, ((ct2 == 1l) ? "day" : "days")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "month" : "months")
			);
	}
	/* A year or longer */
	else
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l / 12l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l / 30l), 1l) % 12l;
		g_free(*s_ptr);
		if(ct2 != 0l)
			*s_ptr = g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "year" : "years"),
				ct2, ((ct2 == 1l) ? "month" : "months")
			);
		else
			*s_ptr = g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "year" : "years")
			);
	}

	return(*s_ptr);
}
