#include <glib.h>
#include "edv_path.h"
#include "edv_property.h"
#include "edv_vfs_obj.h"
#include "edv_recycled_obj.h"
#include "edv_archive_obj.h"
#include "edv_convert_obj.h"


EDVVFSObject *edv_convert_properties_list_to_vfs_object(GList *properties_list);
GList *edv_convert_vfs_object_to_properties_list(
	GList *properties_list,
	EDVVFSObject *obj
);

EDVRecycledObject *edv_convert_properties_list_to_recycled_object(GList *properties_list);
GList *edv_convert_recycled_object_to_properties_list(
	GList *properties_list,
	EDVRecycledObject *obj
);

EDVArchiveObject *edv_convert_properties_list_to_archive_object(GList *properties_list);
GList *edv_convert_archive_object_to_properties_list(
	GList *properties_list,
	EDVArchiveObject *obj
);

EDVRecycledObject *edv_convert_vfs_object_to_recycled_object(EDVVFSObject *obj);
EDVVFSObject *edv_convert_recycled_object_to_vfs_object(EDVRecycledObject *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Converts the properties list to a VFS object.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	Returns a new VFS object describing the converted values or
 *	NULL on error. Extranous properties in the properties list
 *	will be ignored. The meta data list will not be converted.
 */
EDVVFSObject *edv_convert_properties_list_to_vfs_object(GList *properties_list)
{
	EDVVFSObject *obj = edv_vfs_object_new();
	if(obj == NULL)
	    return(NULL);

	obj->type = (EDVObjectType)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE
	);
	obj->device_index = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_INDEX
	);
	obj->index = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX
	);
	obj->name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_NAME
	));
	obj->path = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_PATH
	));
	obj->size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE
	);
	obj->link_target = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET
	));
	obj->link_hint_flags = (EDVLinkHintFlags)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_LINK_HINT_FLAGS
	);
	obj->permissions = (EDVPermissionFlags)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS
	);
	obj->access_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME
	);
	obj->modify_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME
	);
	obj->change_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME
	);
	obj->owner_id = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_OWNER_ID
	);
	obj->group_id = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_GROUP_ID
	);
	obj->device_type = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_TYPE
	);
	obj->block_size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_BLOCK_SIZE
	);
	obj->blocks = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_BLOCKS
	);
	obj->hard_links = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_HARD_LINKS
	);
/*	obj->properties_list = NULL; */

	return(obj);
}

/*
 *	Converts the VFS object to a properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	The obj specifies the VFS object.
 *
 *	Returns the properties_list. Extranous properties in the
 *	properties list will be ignored. The meta data list will not be
 *	converted.
 */
GList *edv_convert_vfs_object_to_properties_list(
	GList *properties_list,
	EDVVFSObject *obj
)
{
	if(obj == NULL)
	    return(properties_list);

	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE,
	    (gint)obj->type,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_INDEX,
	    obj->device_index,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX,
	    obj->index,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_NAME,
	    obj->name,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_PATH,
	    obj->path,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE,
	    obj->size,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET,
	    obj->link_target,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_LINK_HINT_FLAGS,
	    (gint)obj->link_hint_flags,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS,
	    (gint)obj->permissions,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME,
	    obj->access_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME,
	    obj->modify_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME,
	    obj->change_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_OWNER_ID,
	    obj->owner_id,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_GROUP_ID,
	    obj->group_id,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_TYPE,
	    obj->device_type,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_BLOCK_SIZE,
	    obj->block_size,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_BLOCKS,
	    obj->blocks,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_HARD_LINKS,
	    obj->hard_links,
	    TRUE
	);
	/* obj->properties_list */

	return(properties_list);
}


/*
 *	Converts the properties list to a recycled object.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	Returns a new recycled object describing the converted values
 *	or NULL on error. Extranous properties in the properties list
 *	will be ignored. The meta data list will not be converted.
 */
EDVRecycledObject *edv_convert_properties_list_to_recycled_object(GList *properties_list)
{
	EDVRecycledObject *obj = edv_recycled_object_new();
	if(obj == NULL)
	    return(NULL);

	obj->type = (EDVObjectType)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE
	);
	obj->index = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX
	);
	obj->name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_NAME
	));
	obj->original_path = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_ORIGINAL_PATH
	));
	obj->size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE
	);
	obj->storage_size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_SIZE
	);
	obj->link_target = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET
	));
	obj->permissions = (EDVPermissionFlags)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS
	);
	obj->access_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME
	);
	obj->modify_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME
	);
	obj->change_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME
	);
	obj->deleted_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_DELETED_TIME
	);
	obj->owner_id = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_OWNER_ID
	);
	obj->group_id = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_GROUP_ID
	);

	return(obj);
}

/*
 *	Converts the recycled object to a properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	The obj specifies the recycled object.
 *
 *	Returns the properties_list. Extranous properties in the
 *	properties list will be ignored. The meta data list will not be
 *	converted.
 */
GList *edv_convert_recycled_object_to_properties_list(
	GList *properties_list,
	EDVRecycledObject *obj
)
{
	if(obj == NULL)
	    return(properties_list);

	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE,
	    (gint)obj->type,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX,
	    obj->index,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_NAME,
	    obj->name,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_ORIGINAL_PATH,
	    obj->original_path,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE,
	    obj->size,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_SIZE,
	    obj->storage_size,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET,
	    obj->link_target,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS,
	    (gint)obj->permissions,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME,
	    obj->access_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME,
	    obj->modify_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME,
	    obj->change_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_DELETED_TIME,
	    obj->deleted_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_OWNER_ID,
	    obj->owner_id,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_GROUP_ID,
	    obj->group_id,
	    TRUE
	);

	return(properties_list);
}


/*
 *	Converts the properties list to an archive object.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	Returns a new archive object describing the converted values
 *	or NULL on error. Extranous properties in the properties list
 *	will be ignored. The meta data list will not be converted.
 */
EDVArchiveObject *edv_convert_properties_list_to_archive_object(GList *properties_list)
{
	EDVArchiveObject *obj = edv_archive_object_new();
	if(obj == NULL)
	    return(NULL);

	obj->type = (EDVObjectType)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE
	);
	obj->index = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX
	);
	obj->name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_NAME
	));
	obj->path = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_PATH
	));
	obj->size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE
	);
	obj->storage_size = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_SIZE
	);
	obj->link_target = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET
	));
	obj->permissions = (EDVPermissionFlags)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS
	);
	obj->access_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME
	);
	obj->modify_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME
	);
	obj->change_time = edv_properties_list_get_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME
	);
	obj->owner_name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_OWNER_NAME
	));
	obj->group_name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_GROUP_NAME
	));
	obj->device_type = edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_TYPE
	);
	obj->encryption_name = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_ENCRYPTION_NAME
	));
	obj->compression_ratio = edv_properties_list_get_f(
	    properties_list,
	    EDV_PROP_NAME_COMPRESSION_RATIO
	);
	obj->storage_method = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_METHOD
	));
	obj->crc = STRDUP(edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_CRC
	));

	return(obj);
}

/*
 *	Converts the archive object to a properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty *properties.
 *
 *	The obj specifies the archive object.
 *
 *	Returns the properties_list. Extranous properties in the
 *	properties list will be ignored. The meta data list will not be
 *	converted.
 */
GList *edv_convert_archive_object_to_properties_list(
	GList *properties_list,
	EDVArchiveObject *obj
)
{
	if(obj == NULL)
	    return(properties_list);

	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE,
	    (gint)obj->type,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_INDEX,
	    obj->index,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_NAME,
	    obj->name,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_PATH,
	    obj->path,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_SIZE,
	    obj->size,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_SIZE,
	    obj->storage_size,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_LINK_TARGET,
	    obj->link_target,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS,
	    (gint)obj->permissions,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_ACCESS_TIME,
	    obj->access_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_MODIFY_TIME,
	    obj->modify_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_ul(
	    properties_list,
	    EDV_PROP_NAME_CHANGE_TIME,
	    obj->change_time,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_OWNER_NAME,
	    obj->owner_name,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_GROUP_NAME,
	    obj->group_name,
	    TRUE
	);
	properties_list = edv_properties_list_set_i(
	    properties_list,
	    EDV_PROP_NAME_DEVICE_TYPE,
	    obj->device_type,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_ENCRYPTION_NAME,
	    obj->encryption_name,
	    TRUE
	);
	properties_list = edv_properties_list_set_f(
	    properties_list,
	    EDV_PROP_NAME_COMPRESSION_RATIO,
	    obj->compression_ratio,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_STORAGE_METHOD,
	    obj->storage_method,
	    TRUE
	);
	properties_list = edv_properties_list_set_s(
	    properties_list,
	    EDV_PROP_NAME_CRC,
	    obj->crc,
	    TRUE
	);

	return(properties_list);
}


/*
 *	Converts the VFS object to a recycled object.
 *
 *	The obj specifies the VFS object.
 *
 *	Returns a new recycled object describing the converted values or
 *	NULL on error. Extranous properties on the VFS object will be
 *	ignored.
 */
EDVRecycledObject *edv_convert_vfs_object_to_recycled_object(EDVVFSObject *obj)
{
	EDVRecycledObject *tar_obj;
	EDVVFSObject *src_obj = obj;

	if(src_obj == NULL)
	    return(NULL);

	tar_obj = edv_recycled_object_new();
	if(tar_obj == NULL)
	    return(NULL);

	tar_obj->type = src_obj->type;
	tar_obj->index = src_obj->index;
	tar_obj->name = STRDUP(src_obj->name);
	tar_obj->original_path = (src_obj->path != NULL) ?
	    g_dirname(src_obj->path) : NULL;
	tar_obj->size = src_obj->size;
	tar_obj->link_target = STRDUP(src_obj->link_target);
	tar_obj->permissions = src_obj->permissions;
	tar_obj->access_time = src_obj->access_time;
	tar_obj->modify_time = src_obj->modify_time;
	tar_obj->change_time = src_obj->change_time;
	tar_obj->owner_id = src_obj->owner_id;
	tar_obj->group_id = src_obj->group_id;

	tar_obj->meta_data_list = edv_properties_list_copy(src_obj->meta_data_list);

	return(tar_obj);
}

/*
 *	Converts the recycled object to a VFS object.
 *
 *	The obj specifies the recycled object.
 *
 *	Returns a new VFS object describing the converted values or
 *	NULL on error. Extranous properties on the recycled object will
 *	be ignored.
 */
EDVVFSObject *edv_convert_recycled_object_to_vfs_object(EDVRecycledObject *obj)
{
	EDVVFSObject *tar_obj;
	EDVRecycledObject *src_obj = obj;

	if(src_obj == NULL)
	    return(NULL);

	tar_obj = edv_vfs_object_new();
	if(tar_obj == NULL)
	    return(NULL);

	tar_obj->type = src_obj->type;
	tar_obj->index = src_obj->index;
	tar_obj->name = STRDUP(src_obj->name);
	tar_obj->path = edv_paths_join(
	    src_obj->original_path,
	    src_obj->name
	);
	tar_obj->size = src_obj->size;
	tar_obj->link_target = STRDUP(src_obj->link_target);
	tar_obj->permissions = src_obj->permissions;
	tar_obj->access_time = src_obj->access_time;
	tar_obj->modify_time = src_obj->modify_time;
	tar_obj->change_time = src_obj->change_time;
	tar_obj->owner_id = src_obj->owner_id;
	tar_obj->group_id = src_obj->group_id;

	tar_obj->meta_data_list = edv_properties_list_copy(src_obj->meta_data_list);

	return(tar_obj);
}
