/*
                   Endeavour Mark II API - Context

	The EDVContext (Endeavour Mark II Context) is a context that
	serves as the connection instance between third-party
	epplications (and some Endeavour-related applications) and
	Endeavour Mark II. It holds specific values that are needed for
	the instance of the application to communicate with Endeavour
	Mark II using the Endeavour Mark II API library.

	All Endeavour Mark II API library functions that communicate
	with Endeavour Mark II require the EDVContext as one of its
	inputs.


	Example of creating a new EDVContext, connecting it with
	Endeavour Mark II, and instructing Endeavour Mark II to
	map the About Dialog.

	// Create a new EDVContext
        EDVContext *ctx = edv_context_new();
	// Initialize and connect to Endeavour Mark II
        edv_context_init(ctx, NULL);
	// Instruct Endeavour mark II to map the About Dialog
	EDVAbout(ctx);
	// Realize the above commands and wait/block until they are done
        edv_context_sync(ctx);
	// Disconnect, shutdown, and delete the EDVContext
        edv_context_delete(ctx);


	See also: endeavour2.h
 */

#ifndef EDV_CONTEXT_H
#define EDV_CONTEXT_H

#include <glib.h>


/*
 *	Endeavour 2 Context:
 *
 *	This is the forward-declaration, the declaration itself is
 *	private and found in edv_context_private.h in the original
 *	source. It is not inteded to be used by public code.
 */
typedef struct _EDVContext		EDVContext;
#define EDV_CONTEXT(p)			((EDVContext *)(p))


/*
 *	Creates a new Endeavour 2 Context.
 *
 *	The function edv_context_init() should be
 *	called right after this call in order to set up the new
 *	Endeavour 2 Context's values.
 *
 *	Returns the Endeavour 2 Context or NULL on error. To delete
 *	the returned Endeavour 2 Context, call edv_context_delete().
 */
extern EDVContext *edv_context_new(void);

/*
 *	Creates a new Endeavour 2 Context and initializes it.
 *
 *	This is basically:
 *
 *	EDVContext *ctx = edv_context_new();
 *	edv_context_init(ctx, cfg_path);
 *	return(ctx);
 *
 *	The cfg_path specifies the full path an alternate configuration
 *	file. If cfg_path is NULL then the default configuration file
 *	found in the user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 *
 *	Returns the Endeavour 2 Context or NULL on error. To delete
 *	the returned Endeavour 2 Context, call edv_context_delete().
 */
extern EDVContext *edv_context_new_init(const gchar *cfg_path);

/*
 *	(Re)opens the configuration and sets up the Endeavour 2 Context
 *	to be useable by all subsequent calls to Endeavour 2 API
 *	functions.
 *
 *	The ctx specifies the Endeavour 2 Context. This should be
 *	called right after edv_context_new() to ensure that the context
 *	ctx is set up properly before passing to any other function.
 *
 *	This function can also be called subsequent times to update the
 *	Endeavour 2 Context whenever the configuration has changed
 *	instead of deleting and creating a new context.
 *
 *	The cfg_path specifies the full path an alternate configuration
 *	file. If cfg_path is NULL then the default configuration file
 *	found in the user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 */
extern void edv_context_init(
	EDVContext *ctx,
	const gchar *cfg_path
);

/*
 *	Checks if the Endeavour 2 configuration file has changed.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern gboolean edv_context_configuration_changed(EDVContext *ctx);

/*
 *	Checks if the Endeavour 2 program is current running.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern gboolean edv_context_is_running(EDVContext *ctx);

/*
 *	Gets the number of queued commands.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern gint edv_context_commands_pending(EDVContext *ctx);

/*
 *	Appends a command to the queued commands list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The cmd specifies the command. This command will be queued, to
 *	actually execute it, call edv_context_flush() or edv_context_sync().
 */
extern void edv_context_queue_command(
	EDVContext *ctx,
	const gchar *cmd
);

/*
 *	Processes and flushes all pending operations and resources.
 *
 *	All queued commands will be sent (regardless if Endeavour is
 *	running or not).
 *
 *	This call will not block/wait for Endeavour to acknowlage
 *	the request, for that use edv_context_wait();
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void edv_context_flush(EDVContext *ctx);

/*
 *	Waits for a response from Endeavour 2 (if it is running) to
 *	indicate that it has processed all the commands sent to it
 *	from a prior call to edv_context_flush() (if any).
 *
 *	If Endeavour 2 is not running then this call returns
 *	immediately.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void edv_context_wait(EDVContext *ctx);

/*
 *	Calls edv_context_flush() and then edv_context_wait().
 */
extern void edv_context_sync(EDVContext *ctx);

/*
 *	Disconnects, shuts down, and deletes the Endeavour 2 Context.
 *
 *	Any queued operations will not be performed and will be
 *	discarded.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void edv_context_delete(EDVContext *ctx);


#endif	/* EDV_CONTEXT_H */
