#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include "../cfg.h"
#include "../cfg_fio.h"

#include "edv_utils.h"
#include "edv_path.h"
#include "edv_process.h"
#include "edv_interps.h"
#include "edv_id.h"
#include "edv_id_fio.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_context.h"
#include "edv_notify.h"
#include "edv_context_private.h"

/* Instruct edv_cfg_list.h to #define EDV_CONFIGURATION_LIST */
#define NEED_EDV_CFG_LIST_SOURCE
#include "edv_cfg_list.h"
#include "config.h"


static void edv_context_mime_type_append(
	EDVContext *ctx,
	EDVMIMEType *m
);
static gulong edv_context_get_path_modify_time(const gchar *path);

EDVContext *edv_context_new(void);
EDVContext *edv_context_new_init(const gchar *cfg_path);
void edv_context_init(
	EDVContext *ctx,
	const gchar *cfg_path
);
gboolean edv_context_configuration_changed(EDVContext *ctx);
static void edv_context_load_devices(EDVContext *ctx);
static void edv_context_load_mime_types(EDVContext *ctx);
gint edv_context_commands_pending(EDVContext *ctx);
void edv_context_queue_command(
	EDVContext *ctx,
	const gchar *cmd
);
void edv_context_flush(EDVContext *ctx);
void edv_context_wait(EDVContext *ctx);
void edv_context_sync(EDVContext *ctx);
void edv_context_delete(EDVContext *ctx);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Appends the MIME Type to the Endeavour 2 Context.
 */
static void edv_context_mime_type_append(
	EDVContext *ctx,
	EDVMIMEType *m
)
{
	ctx->mime_types_list = g_list_append(
		ctx->mime_types_list,
		m
	);
}

/*
 *	Gets the modify time of the path's destination.
 */
static gulong edv_context_get_path_modify_time(const gchar *path)
{
	gulong t;
	EDVVFSObject *obj = edv_vfs_object_stat(path);
	if(obj == NULL)
		return(0l);

	t = obj->modify_time;

	edv_vfs_object_delete(obj);

	return(t);
}


/*
 *	Creates a new Endeavour 2 Context.
 *
 *	The function edv_context_init() should be
 *	called right after this call in order to set up the new
 *	Endeavour 2 Context's values.
 *
 *	Returns the Endeavour 2 Context or NULL on error. To delete
 *	the returned Endeavour 2 Context, call edv_context_delete().
 */
EDVContext *edv_context_new(void)
{
	return(EDV_CONTEXT(g_malloc0(sizeof(EDVContext))));
}

/*
 *      Creates a new Endeavour 2 Context and initializes it.
 *
 *      This is basically:
 *
 *      EDVContext *ctx = edv_context_new();
 *      edv_context_init(ctx, cfg_path);
 *      return(ctx);
 *
 *	The cfg_path specifies the full path an alternate configuration
 *	file. If cfg_path is NULL then the default configuration file
 *	found in the user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 *
 *      Returns the Endeavour 2 Context or NULL on error. To delete
 *      the returned Endeavour 2 Context, call edv_context_delete().
 */
EDVContext *edv_context_new_init(const gchar *cfg_path)
{
	EDVContext *ctx = edv_context_new();
	if(ctx == NULL)
		return(NULL);

	edv_context_init(
		ctx,
		cfg_path
	);

	return(ctx);
}

/*
 *	(Re)opens the configuration and sets up the Endeavour 2 Context 
 *	to be useable by all subsequent calls to Endeavour 2 API
 *	functions.
 *
 *	The ctx specifies the Endeavour 2 Context. This should be
 *	called right after edv_context_new() to ensure that the context
 *	ctx is set up properly before passing to any other function.
 *
 *	This function can also be called subsequent times to update the
 *	Endeavour 2 Context whenever the configuration has changed
 *	instead of deleting and creating a new context.
 *
 *	The cfg_path specifies the full path an alternate configuration
 *	file. If cfg_path is NULL then the default configuration file
 *	found in the user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 */
void edv_context_init(
	EDVContext *ctx,
	const gchar *cfg_path
)
{
	gboolean need_open_cfg;
	const gchar *prog_loc[] = EDV_PROGRAM_LOCATIONS;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Need to get this pid? */
	if(ctx->pid == 0)
		ctx->pid = edv_pid_get_current();

	/* Need to get the Endeavour 2 program location? */
	if(STRISEMPTY(ctx->prog_path))
	{
		gint i;

		/* Search through the list of possible program locations */
		for(i = 0; prog_loc[i] != NULL; i++)
		{
			/* Exists at this location? */
			if(edv_path_exists(prog_loc[i]))
			{
				const gchar *path = prog_loc[i];

				g_free(ctx->prog_path);
				ctx->prog_path = g_strdup(path);

				g_free(ctx->prog_file);
				ctx->prog_file = STRDUP(g_basename(path));

				break;
			}
		}

		/* Unable to find the program location? */
		if(STRISEMPTY(ctx->prog_path))
		{
			g_printerr(
"edv_context_init(): Warning:\
 Unable to find the %s program in the following locations:\n",
				PROG_NAME
			);
			for(i = 0; prog_loc[i] != NULL; i++)
				g_printerr(
					"\t%s\n",
					prog_loc[i]
				);
		}
	}

	/* Need to create the configuration list on the context? */
	if(ctx->cfg_list == NULL)
	{
		/* Declare the default configuration list with all the
		 * parameters but no values set
		 */
		const CfgList src_cfg_list[] = EDV_CONFIGURATION_LIST;

		/* Copy the default configuration list to the context,
		 * each configuration item's value will not be set
		 */
		ctx->cfg_list = CFGItemListCopyList(src_cfg_list);
		if(ctx->cfg_list == NULL)
			return;
	}

	/* Get the configuration file path
	 *
	 * Was an alternate configuration file path specified?
	 */
	if(cfg_path != NULL)
	{
		/* Record the alternate configuration file path */
		g_free(ctx->cfg_path);
		ctx->cfg_path = g_strdup(cfg_path);
	}
	else
	{
		/* Record the default configuration file path located
		 * in the user's home directory:
		 *
		 * $HOME/EDV_NAME_DEF_USER_DATA_DIR/EDV_NAME_DEF_CONFIG_FILE
		 */
		const gchar *home = g_getenv(ENV_VAR_NAME_HOME);
		g_free(ctx->cfg_path);
		ctx->cfg_path = g_strconcat(
			(home != NULL) ? home : "/",
			G_DIR_SEPARATOR_S,
			EDV_NAME_DEF_USER_DATA_DIR,
			G_DIR_SEPARATOR_S,
			EDV_NAME_DEF_CONFIG_FILE,
			NULL
		);
	}

	/* Check if we need to open the configuration from the
	 * configuration file
	 */
	need_open_cfg = FALSE;
	if(!STRISEMPTY(ctx->cfg_path))
	{
		/* Get the configuration file's last modified time so we
		 * can tell if it is newer
		 */
		const gulong t = edv_context_get_path_modify_time(ctx->cfg_path);
		if(ctx->cfg_last_modified != t)
		{
			need_open_cfg = TRUE;

			/* Record the new modify time */
			ctx->cfg_last_modified = t;
		}
	}

	/* Need to open the configuration list from the configuration
	 * file?
	 */
	if(need_open_cfg)
	{
		/* Does the configuration file exist? */
		if(edv_path_exists(ctx->cfg_path))
		{
			/* Open the configuration list from the
			 * configuration file, setting the value of
			 * each configuration parameter read from
			 * the configuration file
			 */
			if(CFGFileOpen(
				ctx->cfg_path,
				ctx->cfg_list
			))
			{
				/* An error occured while opening
				 * the configuration file
				 */
				const gint error_code = (gint)errno;
				g_printerr(
					"%s: %s\n",
					ctx->cfg_path,
					g_strerror(error_code)
				);
			}
			else
			{
				/* Update the values on the context
				 * based on the configuration values
				 * obtained from the configuration file
				 */
				CfgList *cfg_list = ctx->cfg_list;
				const gchar *s;
				gint	version_major,
					version_minor,
					version_release;

				/* Check version obtained from the configuration file */
				version_major = CFGItemListGetValueI(
					cfg_list,
					EDV_CFG_PARM_VERSION_MAJOR
				);
				version_minor = CFGItemListGetValueI(
					cfg_list,
					EDV_CFG_PARM_VERSION_MINOR
				);
				version_release = CFGItemListGetValueI(
					cfg_list,
					EDV_CFG_PARM_VERSION_RELEASE
				);
				if((version_major != PROG_VERSION_MAJOR) ||
				   (version_minor != PROG_VERSION_MINOR) ||
				   (version_release != PROG_VERSION_RELEASE)
				)
				{
#if 0
					g_printerr(
"%s: Warning: Version from configuration file %i.%i.%i differs from\
 %s library version %i.%i.%i.\n",
						ctx->cfg_path,
						version_major, version_minor, version_release,
						PROG_NAME,
						PROG_VERSION_MAJOR, PROG_VERSION_MINOR, PROG_VERSION_RELEASE
					);
#endif
				}

				/* Set the recycled objects index file */
				s = CFGItemListGetValueS(
					cfg_list,
					EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX
				);
				g_free(ctx->recycle_bin_index_path);
				ctx->recycle_bin_index_path = STRDUP(s);
			}
		}
	}

	/* Reopen the UIDs and GIDs lists */
	if(ctx->uids_list != NULL)
	{
		g_list_foreach(
			ctx->uids_list,
			(GFunc)edv_uid_delete,
			NULL
		);
		g_list_free(ctx->uids_list);
		ctx->uids_list = NULL;
	}
	ctx->uids_list = edv_uids_list_open_from_system(
		ctx->uids_list,
		-1				/* Append */
	);

	if(ctx->gids_list != NULL)
	{
		g_list_foreach(
			ctx->gids_list,
			(GFunc)edv_gid_delete,
			NULL
		);
		g_list_free(ctx->gids_list);
		ctx->gids_list = NULL;
	}
	ctx->gids_list = edv_gids_list_open_from_system(
		ctx->gids_list,
		-1				/* Append */
	);


	/* Reopen the Devices list from the Devices file */
	if(ctx->devices_list != NULL)
	{
		g_list_foreach(
			ctx->devices_list,
			(GFunc)edv_device_delete,
			NULL
		);
		g_list_free(ctx->devices_list);
		ctx->devices_list = NULL;
	}
	edv_context_load_devices(ctx);


	/* Reopen the MIME Types list from the MIME Types file */
	if(ctx->mime_types_list != NULL)
	{
		g_list_foreach(
			ctx->mime_types_list,
			(GFunc)edv_mime_type_delete,
			NULL
		);
		g_list_free(ctx->mime_types_list);
		ctx->mime_types_list = NULL;
	}
	edv_context_load_mime_types(ctx);
}

/*
 *	Checks if the Endeavour 2 configuration file has changed.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
gboolean edv_context_configuration_changed(EDVContext *ctx)
{
	gulong t;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	t = edv_context_get_path_modify_time(ctx->cfg_path);
	if(t == 0l)
		return(FALSE);

	errno = 0;

	if(ctx->cfg_last_modified == t)
		return(FALSE);

	return(TRUE);
}

/*
 *	Called by edv_context_init().
 *
 *	Opens the Devices List file to the Endeavour 2 Context.
 */
static void edv_context_load_devices(EDVContext *ctx)
{
	/* Open the local Devices List file */   
	ctx->devices_list = edv_devices_list_file_open(
		ctx->devices_list,
		CFGItemListGetValueS(
			ctx->cfg_list,
			EDV_CFG_PARM_FILE_DEVICES
		)
	);
}

/*
 *	Called by edv_context_init().
 *
 *	Loads default/system, local, and global MIME Types.
 */
static void edv_context_load_mime_types(EDVContext *ctx)
{
	const gchar *s;
	gchar *icons_dir_global;
	EDVMIMEType *m;

	/* Get the global icons directory */
	s = CFGItemListGetValueS(
		ctx->cfg_list,
		EDV_CFG_PARM_DIR_GLOBAL
	);
	icons_dir_global = g_strconcat(
		s,
		G_DIR_SEPARATOR_S,
		EDV_NAME_ICONS_SUBDIR,
		NULL
	);

#define SET_MIMETYPE_ICON_FILES(state,sfile,mfile,lfile) { \
 if(!STRISEMPTY(icons_dir_global)) {		\
						\
 g_free(m->small_icon_file[(state)]);		\
 m->small_icon_file[(state)] = g_strconcat(	\
  icons_dir_global,				\
  G_DIR_SEPARATOR_S,				\
  (sfile),					\
  NULL						\
 );						\
						\
 g_free(m->medium_icon_file[(state)]);		\
 m->medium_icon_file[(state)] = g_strconcat(	\
  icons_dir_global,				\
  G_DIR_SEPARATOR_S,				\
  (mfile),					\
  NULL						\
 );						\
						\
 g_free(m->large_icon_file[(state)]);		\
 m->large_icon_file[(state)] = g_strconcat(	\
  icons_dir_global,				\
  G_DIR_SEPARATOR_S,				\
  (lfile),					\
  NULL						\
 );						\
} }

	/* Begin loading the system MIME Types */

	/* Unknown */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_UNKNOWN,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_object_misc_20x20.xpm",
		"icon_object_misc_32x32.xpm",
		"icon_object_misc_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Error */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_ERROR,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_object_error_20x20.xpm",
		"icon_object_error_32x32.xpm",
		"icon_object_error_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* File */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_FILE,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_file_20x20.xpm",
		"icon_file_32x32.xpm",
		"icon_file_48x48.xpm"
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_HIDDEN,
		"icon_file_hidden_20x20.xpm",
		"icon_file_hidden_32x32.xpm",
		"icon_file_hidden_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Executable */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_executable_20x20.xpm",
		"icon_executable_32x32.xpm",
		"icon_executable_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Directory */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_folder_closed_20x20.xpm",
		"icon_folder_closed_32x32.xpm",
		"icon_folder_closed_48x48.xpm"
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_OPENED,
		"icon_folder_opened_20x20.xpm",
		"icon_folder_opened_32x32.xpm",
		"icon_folder_opened_48x48.xpm"
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE,
		"icon_folder_noaccess_20x20.xpm",
		"icon_folder_noaccess_32x32.xpm",
		"icon_folder_noaccess_48x48.xpm"
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_HIDDEN,
		"icon_folder_hidden_20x20.xpm",
		"icon_folder_hidden_32x32.xpm",
		"icon_folder_hidden_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Link */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_LINK,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_link2_20x20.xpm",
		"icon_link2_32x32.xpm",
		"icon_link2_48x48.xpm"
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE,
		"icon_link_broken_20x20.xpm",
		"icon_link_broken_32x32.xpm",
		"icon_link_broken_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Device Block */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_device_block_20x20.xpm",
		"icon_device_block_32x32.xpm",
		"icon_device_block_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Device Character */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_device_character_20x20.xpm",
		"icon_device_character_32x32.xpm",
		"icon_device_character_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* FIFO Pipe */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_FIFO,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_pipe_20x20.xpm",
		"icon_pipe_32x32.xpm",
		"icon_pipe_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);

	/* Socket */
	m = edv_mime_type_new_values(
		EDV_MIME_TYPE_CLASS_SYSTEM,
		NULL,
		EDV_MIME_TYPE_TYPE_INODE_SOCKET,
		NULL
	);
	SET_MIMETYPE_ICON_FILES(
		EDV_MIME_TYPE_ICON_STATE_STANDARD,
		"icon_socket_20x20.xpm",
		"icon_socket_32x32.xpm",
		"icon_socket_48x48.xpm"
	);
	edv_context_mime_type_append(ctx, m);


	/* Begin opening the MIME Types from file */

	/* Open the local MIME Types file */
	ctx->mime_types_list = edv_mime_types_list_file_open(
		ctx->mime_types_list,
		CFGItemListGetValueS(
			ctx->cfg_list,
			EDV_CFG_PARM_FILE_MIME_TYPES
		)
	);

	/* Open the global MIME Types file */
	ctx->mime_types_list = edv_mime_types_list_file_open(
		ctx->mime_types_list,
		CFGItemListGetValueS(
			ctx->cfg_list,
			EDV_CFG_PARM_FILE_MIME_TYPES_GLOBAL
		)
	);

#undef SET_MIMETYPE_ICON_FILES

	g_free(icons_dir_global);
}


/*
 *	Checks if the Endeavour 2 program is current running.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
gboolean edv_context_is_running(EDVContext *ctx)
{
	if(ctx == NULL)
		return(FALSE);

	/* Is the process running? */
	if(edv_interps_get_lock(ctx->cfg_list) > 0)
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Gets the number of queued commands.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
gint edv_context_commands_pending(EDVContext *ctx)
{
	return((ctx != NULL) ? ctx->total_queued_commands : 0);
}

/*
 *	Appends a command to the queued commands list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The cmd specifies the command. This command will be queued, to
 *	actually execute it, call edv_context_flush() or edv_context_sync().
 */
void edv_context_queue_command(
	EDVContext *ctx,
	const gchar *cmd
)
{
	gint i;

	if((ctx == NULL) || STRISEMPTY(cmd))
		return;

	/* Append the command plus one NULL pointer to the list */
	i = MAX(ctx->total_queued_commands, 0);
	ctx->total_queued_commands = i + 1;
	ctx->queued_command = (gchar **)g_realloc(
		ctx->queued_command,
		(ctx->total_queued_commands + 1) * sizeof(gchar *)
	);
	if(ctx->queued_command == NULL)
	{
		ctx->total_queued_commands = 0;
		return;
	}

	ctx->queued_command[i] = g_strdup(cmd);
	ctx->queued_command[i + 1] = NULL;
}

/*
 *	Processes and flushes all pending operations and resources.
 *
 *	All queued commands will be sent (regardless if Endeavour is
 *	running or not).
 *
 *	This call will not block/wait for Endeavour to acknowlage
 *	the request, for that use edv_context_wait();
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_context_flush(EDVContext *ctx)
{
	if(ctx == NULL)
		return;

	/* Get InterPS lock link to find out the pid of the currently
	 * running Endeavour (if it is running)
	 *
	 * Note that we do not store the pid on the context ctx because
	 * Endeavour might exit and restart, creating a new lock link
	 * and have a different pid
	 */
	if(ctx->queued_command != NULL)
	{
		const gint p = edv_interps_get_lock(ctx->cfg_list);
		if(p > 0)
		{
			/* Got pid and is running
			 *
			 * Send out any queued commands
			 */
			edv_interps_send_commands(
				ctx->cfg_list,
				p,
				(const gchar **)ctx->queued_command
			);
		}

		/* Remove all queued commands (regardless of if they
		 * were sent or not)
		 */
		g_strfreev(ctx->queued_command);
		ctx->queued_command = NULL;
		ctx->total_queued_commands = 0;
	}
}

/*
 *	Waits for a response from Endeavour 2 (if it is running) to
 *	indicate that it has processed all the commands sent to it
 *	from a prior call to edv_context_flush() (if any).
 *
 *	If Endeavour 2 is not running then this call returns
 *	immediately.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_context_wait(EDVContext *ctx)
{
	gint p;
	CfgList *cfg_list;

	if(ctx == NULL)
		return;

	cfg_list = ctx->cfg_list;

	/* Is the process running? */
	p = edv_interps_get_lock(cfg_list);
	if(p > 0)
	{
		int nretries = 0;
		gulong elapsed_sec;
		GTimer *timer = g_timer_new();
		g_timer_start(timer);

		/* Wait until the commands have been processed */
		while(edv_interps_command_pending(cfg_list) &&
			  edv_pid_exists(p)
		)
		{
			elapsed_sec = (gulong)(g_timer_elapsed(timer, NULL) / 1000000.0f);

			/* Resend the signal every 3 seconds */
			if(elapsed_sec > 3l)
			{
				/* Resend the signal */
				edv_interps_send_command_notify(
					cfg_list,
					p
				);

				nretries++;

				/* Give up after 60 tries */
				if(nretries > 60)
					break;

				g_timer_start(timer);
			}

			edv_usleep(8000l);
		}

		g_timer_destroy(timer);
	}
}

/*
 *      Calls edv_context_flush() and then edv_context_wait().
 */
void edv_context_sync(EDVContext *ctx)
{
	edv_context_flush(ctx);
	edv_context_wait(ctx);
}

/*
 *	Disconnects, shuts down, and deletes the Endeavour 2 Context.
 *
 *	Any queued operations will not be performed and will be
 *	discarded.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_context_delete(EDVContext *ctx)
{
	if(ctx == NULL)
		return;

	/* Delete any queued commands */
	if(ctx->queued_command != NULL)
	{
		g_strfreev(ctx->queued_command);
		ctx->queued_command = NULL;
		ctx->total_queued_commands = 0;
	}

	/* Delete the Devices List */
	if(ctx->devices_list != NULL)
	{
		g_list_foreach(
			ctx->devices_list,
			(GFunc)edv_device_delete,
			NULL
		);
		g_list_free(ctx->devices_list);
		ctx->devices_list = NULL;
	}

	/* Delete the MIME Types List */
	if(ctx->mime_types_list != NULL)
	{
		g_list_foreach(
			ctx->mime_types_list,
			(GFunc)edv_mime_type_delete,
			NULL
		);
		g_list_free(ctx->mime_types_list);
		ctx->mime_types_list = NULL;
	}

	/* Delete the UIDs list */
	if(ctx->uids_list != NULL)
	{                         
		g_list_foreach(
			ctx->uids_list,
			(GFunc)edv_uid_delete,
			NULL
		);
		g_list_free(ctx->uids_list);
		ctx->uids_list = NULL;
	}

	/* Delete the GIDs list */
	if(ctx->gids_list != NULL)
	{
		g_list_foreach(
			ctx->gids_list,
			(GFunc)edv_gid_delete,
			NULL
		);
		g_list_free(ctx->gids_list);
		ctx->gids_list = NULL;
	}

	/* Delete the configuration parameter names list */
	if(ctx->cfg_list_parameters != NULL)
	{
		g_list_foreach(
			ctx->cfg_list_parameters,
			(GFunc)g_free,
			NULL
		);
		g_list_free(ctx->cfg_list_parameters);
		ctx->cfg_list_parameters = NULL;
	}

	/* Delete the configuration list */
	CFGItemListDeleteList(ctx->cfg_list);

	g_free(ctx->cfg_path);
	g_free(ctx->recycle_bin_index_path);
	g_free(ctx->prog_path);
	g_free(ctx->prog_file);
	g_free(ctx->last_error_buf);
	g_free(ctx->date_string_buf);

	g_free(ctx);
}
