#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_property.h"
#include "edv_archive_obj.h"


EDVArchiveObject *edv_archive_object_new(void);
EDVArchiveObject *edv_archive_object_new_type(const EDVObjectType type);
EDVArchiveObject *edv_archive_object_copy(EDVArchiveObject *obj);
void edv_archive_object_set_object(
	EDVArchiveObject *obj,			/* Target */
	EDVArchiveObject *obj_values		/* Source */
);
void edv_archive_object_clear(EDVArchiveObject *obj);
void edv_archive_object_delete(EDVArchiveObject *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVArchiveObject.
 *
 *	Returns a new dynamically allocated EDVArchiveObject with all of its
 *	values zero'ed or NULL on error.
 */
EDVArchiveObject *edv_archive_object_new(void)
{
	return(EDV_ARCHIVE_OBJECT(g_malloc0(sizeof(EDVArchiveObject))));
}

/*
 *	Creates a new EDVArchiveObject of a specific type.
 *
 *	The type specifies the EDVArchiveObject's type.
 *
 *	Returns a new dynamically allocated EDVArchiveObject with its type
 *	value set or NULL on error.
 */
EDVArchiveObject *edv_archive_object_new_type(const EDVObjectType type)
{
	EDVArchiveObject *obj = edv_archive_object_new();
	if(obj == NULL)
            return(NULL);

        obj->type = type;

        return(obj);
}

/*
 *	Coppies the EDVArchiveObject.
 *
 *	The obj specifies the EDVArchiveObject to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVArchiveObject or
 *	NULL on error.
 */
EDVArchiveObject *edv_archive_object_copy(EDVArchiveObject *obj)
{
	EDVArchiveObject *new_obj;

	if(obj == NULL)
	    return(NULL);

	new_obj = edv_archive_object_new();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->path = STRDUP(obj->path);

	new_obj->size = obj->size;
	new_obj->storage_size = obj->storage_size;

	new_obj->link_target = STRDUP(obj->link_target);

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;

	new_obj->owner_name = STRDUP(obj->owner_name);
	new_obj->group_name = STRDUP(obj->group_name);

	new_obj->device_type = obj->device_type;

	new_obj->encryption_name = STRDUP(obj->encryption_name);
	new_obj->compression_ratio = obj->compression_ratio;
	new_obj->storage_method = STRDUP(obj->storage_method);
	new_obj->crc = STRDUP(obj->crc);

	new_obj->meta_data_list = edv_properties_list_copy(obj->meta_data_list);

	return(new_obj);
}

/*
 *	Sets the EDVArchiveObject's values from another EDVArchiveObject.
 *
 *	The obj specifies the EDVArchiveObject to be set.
 *
 *	The obj_values specifies the values to set obj with.
 */
void edv_archive_object_set_object(
	EDVArchiveObject *obj,
	EDVArchiveObject *obj_values
)
{
	EDVArchiveObject	*tar_obj = obj,
				*src_obj = obj_values;

	if((src_obj == NULL) || (tar_obj == NULL))
	    return;

	tar_obj->type = src_obj->type;

	tar_obj->index = src_obj->index;

	g_free(tar_obj->name);
	tar_obj->name = STRDUP(src_obj->name);
	g_free(tar_obj->path);
	tar_obj->path = STRDUP(src_obj->path);

	tar_obj->size = src_obj->size;
	tar_obj->storage_size = src_obj->storage_size;

	g_free(tar_obj->link_target);
	tar_obj->link_target = STRDUP(src_obj->link_target);

	tar_obj->permissions = src_obj->permissions;

	tar_obj->access_time = src_obj->access_time;
	tar_obj->modify_time = src_obj->modify_time;
	tar_obj->change_time = src_obj->change_time;

	g_free(tar_obj->owner_name);
	tar_obj->owner_name = STRDUP(src_obj->owner_name);
	g_free(tar_obj->group_name);
	tar_obj->group_name = STRDUP(src_obj->group_name);

	tar_obj->device_type = src_obj->device_type;

	g_free(tar_obj->encryption_name);
	tar_obj->encryption_name = STRDUP(src_obj->encryption_name);
	tar_obj->compression_ratio = src_obj->compression_ratio;
	g_free(tar_obj->storage_method);
	tar_obj->storage_method = STRDUP(src_obj->storage_method);
	g_free(tar_obj->crc);
	tar_obj->crc = STRDUP(src_obj->crc);

	tar_obj->meta_data_list = edv_properties_list_delete(tar_obj->meta_data_list);
	tar_obj->meta_data_list = edv_properties_list_copy(src_obj->meta_data_list);
}

/*
 *	Deletes and zeros all the values on the EDVArchiveObject.
 *
 *	The obj specifies the EDVArchiveObject who's values are to be cleared.
 */
void edv_archive_object_clear(EDVArchiveObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);

	g_free(obj->link_target);

	g_free(obj->owner_name);
	g_free(obj->group_name);

	g_free(obj->encryption_name);

	g_free(obj->storage_method);
	g_free(obj->crc);

	(void)memset(
	    obj,
	    0x00,
	    sizeof(EDVArchiveObject)
	);
}

/*
 *	Deletes the EDVArchiveObject.
 *
 *	The obj specifies the EDVArchiveObject to delete.
 */
void edv_archive_object_delete(EDVArchiveObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);

	g_free(obj->link_target);

	g_free(obj->owner_name);
	g_free(obj->group_name);

	g_free(obj->encryption_name);

	g_free(obj->storage_method);
	g_free(obj->crc);

	(void)edv_properties_list_delete(obj->meta_data_list);

	g_free(obj);
}
