/*
	Create/Map Window
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


static void print_help(const gchar *prog_name);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s <win_type> [arg...]\n",
		prog_name
	);
	g_print("\n\
    The <win_type> specifies the window type, which can be any of the\n\
    following:\n\
\n\
	about [page]            About Dialog\n\
	file_browser [path] [tree_origin]   File Browser\n\
	image_browser [path]    Image Browser\n\
	archiver [path] [password]   Archiver\n\
	recycle_bin             Recycle Bin\n\
	mime_types [type]       MIME Types List Window\n\
	devices [path]          Devices List Window\n\
	history [event_index]   History List Window\n\
	options [page]          Options Window\n\
	customize [page]        Customize Window\n\
	properties <path> [page]   Properties Dialog\n\
	find [location] [value]   Find Window\n\
	run [command] [working_directory]   Run Dialog\n\
	help [subject]          Help\n\
\n\
    The [arg...] specifies the argument(s) based on the <win_type>.\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
\n"
	);
}

int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;
	GList *args_list = NULL;
	EDVWindowType win_type = EDV_WINDOW_NONE;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
 if(args_list != NULL) {		\
  g_list_foreach(			\
   args_list, (GFunc)g_free, NULL	\
  );					\
  g_list_free(args_list);		\
 }					\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);		\
					\
 return(_v_);				\
}

	/* Insufficient arguments? */
	if(argc <= 1)
	{
		print_help(argv[0]);
		CLEANUP_RETURN(2);
	}

	/* Handle the arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help? */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			print_help(argv[0]);
			CLEANUP_RETURN(0);
		}
		/* Version? */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
"Endeavour Mark II Window Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			CLEANUP_RETURN(0);
		}
		else
		{
			if(win_type == EDV_WINDOW_NONE)
			{
				win_type = edv_window_name_to_window_type(arg);
			}
			else
			{
				args_list = g_list_append(
					args_list,
					g_strdup(arg)
				);
			}
		}
	}

	status = 0;

	switch(win_type)
	{
	  case EDV_WINDOW_NONE:
		print_help(argv[0]);
		status = 2;
		break;

	  case EDV_WINDOW_ABOUT_DIALOG:
		edv_window_about_dialog_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_VFS_BROWSER:
		edv_window_vfs_browser_new(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0),
			(const gchar *)g_list_nth_data(args_list, 1)
		);
		break;

	  case EDV_WINDOW_IMAGE_BROWSER:
		edv_window_image_browser_new(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_ARCHIVER:
		edv_window_archiver_new(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0),
			(const gchar *)g_list_nth_data(args_list, 1)
		);
		break;

	  case EDV_WINDOW_RECYCLE_BIN:
		edv_window_recycle_bin_map(ctx);
		break;

	  case EDV_WINDOW_MIME_TYPES_LIST:
		edv_window_mime_types_list_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_DEVICES_LIST:
		edv_window_devices_list_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_HISTORY_LIST:
		arg = (const gchar *)g_list_nth_data(args_list, 0);
		edv_window_history_list_map(
			ctx,
			(arg != NULL) ? (gint)atoi((const char *)arg) : -1
		);
		break;

	  case EDV_WINDOW_OPTIONS:
		edv_window_options_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_CUSTOMIZE:
		edv_window_customize_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;

	  case EDV_WINDOW_PROPERTIES_DIALOG:
		edv_window_properties_dialog_new(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0),
			(const gchar *)g_list_nth_data(args_list, 1)
		);
		break;

	  case EDV_WINDOW_FIND:
		edv_window_find_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0),
			(const gchar *)g_list_nth_data(args_list, 1)
		);
		break;

	  case EDV_WINDOW_OBJECT_OPERATIONS_DIALOG:
		g_printerr("This win_type is not yet supported.\n");
		status = 2;
		break;

	  case EDV_WINDOW_RUN_DIALOG:
		edv_window_run_dialog_map(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0),
			(const gchar *)g_list_nth_data(args_list, 1)
		);
		break;

	  case EDV_WINDOW_HELP:
		EDVHelpSubject(
			ctx,
			(const gchar *)g_list_nth_data(args_list, 0)
		);
		break;
	}



	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(status);

#undef CLEANUP_RETURN
}
