/*
 *	Prints a list of all Endeavour2 configuration parameters or
 *	a specific configuration parameter.
 */

#include <glib.h>
#include <endeavour2.h>

static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s [parameter]\n",
		prog_name
	);
}

static void match_print_param(EDVContext *ctx, const gchar *param)
{
	switch(edv_get_type(ctx, param))
	{
	  case EDV_GET_TYPE_NONE:
		g_printerr("%s: No such parameter.\n", param);
		break;

	  case EDV_GET_TYPE_INT8:
	  case EDV_GET_TYPE_UINT8:
	  case EDV_GET_TYPE_INT16:
	  case EDV_GET_TYPE_UINT16:
	  case EDV_GET_TYPE_INT32:
	  case EDV_GET_TYPE_UINT32:
		g_print(
"%s = %i\n",
			param, edv_get_i(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_INT64:
		g_print(
"%s = %ld\n",
			param, edv_get_l(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_UINT64:
		g_print(
"%s = %ld\n",
			param, edv_get_ul(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_FLOAT:
		g_print(
"%s = %f\n",
			param, edv_get_f(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_DOUBLE:
		g_print(
"%s = %f\n",
			param, edv_get_d(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_STRING:
		g_print(
"%s = \"%s\"\n",
			param, edv_get_s(ctx, param)
		);
		break;
	  case EDV_GET_TYPE_INT_LIST:
		g_print(
"%s = (int list)\n",
			param
		);
		break;
	  case EDV_GET_TYPE_STRING_LIST:
		g_print(
"%s = (string list)\n",
			param
		);
		break;
	  case EDV_GET_TYPE_COLOR:
		g_print(
"%s = (color)\n",
			param
		);
		break;
	  case EDV_GET_TYPE_ACCELKEY_LIST:
		g_print(
"%s = (accelkey list)\n",
			param
		);
		break;
	  case EDV_GET_TYPE_STYLE:
		g_print(
"%s = (style)\n",
			param
		);
		break;
	  case EDV_GET_TYPE_MENU:
		g_print(
"%s = (menu)\n",
			param
		);
		break;
	}
}

int main(int argc, char *argv[])
{
	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Specific configuration parameter? */
	if(argc > 1)
	{
		const gchar *arg = argv[1];

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			print_help(argv[0]);
		}
		else
		{
			/* Print the specified configuration parameter */
			match_print_param(ctx, arg);
		}
	}
	/* Print all the configuration parameters */
	else
	{
		gint i;
		GList	*glist,
			*parameter_names_list = edv_get_cfg_list_parameters(ctx);
		for(glist = parameter_names_list, i = 0;
		    glist != NULL;
		    glist = g_list_next(glist), i++
		)
		{
			match_print_param(
				ctx,
				(const gchar *)glist->data
			);
		}
		g_print(
"%i %s found\n",
			i,
			(i == 1) ? "parameter" : "parameters"
		);
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	/* Shutdown the Endeavour2 context */
	edv_context_delete(ctx);

	return(0);
}
