/*
	Displays Object Name, Icon, and Type

	Demostration on the usage of Endeavour Mark II's
	EDVMatchObject*() functions to obtain and display the
	appropriate information for an object.

	Note that this program uses GTK.
 */

#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <endeavour2.h>
#include "../config.h"


static void print_usage(const gchar *prog_name);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s <path>\n",
	    prog_name
	);
}


int main(int argc, char *argv[])
{
	const gchar *path = NULL;
	GdkWindow *window;
	GtkWidget	*toplevel,
			*w,
			*parent,
			*parent2;
	EDVVFSObject *obj;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);			\
					\
 return(_v_);				\
}

	if(argc > 1)
	{
	    gint i;
	    const gchar *arg;

	    /* Parse arguments */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(arg == NULL)
		    continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
		    print_usage(argv[0]);
		    CLEANUP_RETURN(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
		        !g_strcasecmp(arg, "-version")
		)
		{
		    g_print(
			"%s",
"Endeavour Mark II Object Info Version " PROG_VERSION "\n"
PROG_COPYRIGHT
		    );
		    CLEANUP_RETURN(0);
		}
		/* Path */
		else if((*arg != '-') && (*arg != '+'))
		{
		    path = arg;
		}
	    }
	}

	/* No path specified? */
	if(path == NULL)
	{
	    /* Print usage message and return invalid value */
	    print_usage(argv[0]);
	    CLEANUP_RETURN(2);
	}

	/* Initialize GTK+ */
	gtk_init(&argc, &argv);

	/* Create the toplevel GtkWindow */
	toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(gtk_main_quit), NULL
	);

	/* Realize and get the GtkWindow's GdkWindow so that we can
	 * use it to open the XPM file for the icon
	 */
	gtk_widget_realize(w);
	window = w->window;

	parent = w;

	/* Create the GtkHBox to arrange and contain the icon and labels */
	w = gtk_hbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(w), 5);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Get the specified object's statistics */
	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
	    /* An error occured whle getting the object's statistics,
	     * display the error message as a GtkLahel
	     */
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
		"%s:\n\n    %s",
		g_strerror(error_code),
		path
	    );
	    w = gtk_label_new(msg);
	    g_free(msg);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	else
	{
	    gchar	*icon_closed_path,
			*icon_opened_path,
			*icon_extended_path,
			*icon_hidden_path,
			*type_string;

	    /* Get the icon appropriate to display the object with */
	    if(edv_match_vfs_object_icon(
		ctx,
		obj,
		EDV_ICON_SIZE_20,		/* Small icons */
		&icon_closed_path,
		&icon_opened_path,
		&icon_extended_path,
		&icon_hidden_path
	    ) != EDV_MATCH_OBJECT_INFO_NONE)
	    {
		/* Got the appropriate icon file paths, now determine
		 * which one we want to use
		 */
		const gchar *icon_path = icon_closed_path;
		GdkBitmap *mask;
		GdkPixmap *pixmap;

		if(edv_path_is_hidden(icon_path))
		{
		    if(icon_hidden_path != NULL)
			icon_path = icon_hidden_path;
		}

		/* Open the XPM file to a GdkPixmap and GdkBitmap mask */
		pixmap = gdk_pixmap_create_from_xpm(
		    window,
		    &mask,
		    NULL,			/* No background color */
		    icon_path
		);
		if(pixmap != NULL)
		{
		    /* Create a GtkPixmap to display the icon */
		    w = gtk_pixmap_new(pixmap, mask);
		    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		    gtk_widget_show(w);

		    gdk_pixmap_unref(pixmap);
		    if(mask != NULL)
			gdk_bitmap_unref(mask);
		}
		else
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
			"%s:\n\n    %s",
			g_strerror(error_code),
			icon_path
		    );
		    w = gtk_label_new(msg);
		    g_free(msg);
		    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		    gtk_widget_show(w);
		}
	    }

	    g_free(icon_closed_path);
	    g_free(icon_opened_path);
	    g_free(icon_extended_path);
	    g_free(icon_hidden_path);

	    /* Create a GtkLabel to display the object's name */
	    w = gtk_label_new(g_basename(path));
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Get the object's MIME Type */
	    if(edv_match_vfs_object_type_string(
		ctx,
		obj,
		&type_string
	    ) != EDV_MATCH_OBJECT_INFO_NONE)
	    {
		/* Create a GtkLabel to display the object's MIME Type */
		w = gtk_label_new(type_string);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    edv_vfs_object_delete(obj);
	    g_free(type_string);
	}

	/* Map the toplevel GtkWindow */
	gtk_widget_show(toplevel);

	/* Display until the user closes the toplevel GtkWindow */
	gtk_main();

	/* Destroy all the GtkWidgets */
	gtk_widget_destroy(toplevel);

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(0);

#undef CLEANUP_RETURN
}
