/*
	MIME Types List

	See also; object_info_gtk.c
 */

#define _GNU_SOURCE				/* For FNM_CASEFOLD */
#include <fnmatch.h>				/* For fnmatch() */
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


typedef enum {
	MLS_PRINT_ALL		= (1 << 0),
	MLS_PRINT_LONG		= (1 << 1),
	MLS_PRINT_DEREFERENCE_COMMANDS	= (1 << 2),
	MLS_PRINT_FULL_TIME	= (1 << 3)
} MLSPrintFlags;


static void print_usage(const gchar *prog_name);

static gchar *mls_get_time_stamp_str(
	EDVContext *ctx,
	const gulong t,
	const gulong ct,
	MLSPrintFlags print_flags
);

static void mls_print_time_stamp(
	EDVContext *ctx,
	const gulong t,
	const gulong ct,
	MLSPrintFlags print_flags
);
static gint mls_print_mimetype(
	EDVContext *ctx,
	EDVMIMEType *m,
	const gulong ct,
	const MLSPrintFlags print_flags
);

static gint mls_list_mimetypes(
	EDVContext *ctx,
	GList *types_list,
	const gulong ct,
	const MLSPrintFlags print_flags
);

static gint mls(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [type(s)...] [options]\n",
		prog_name
	);

	g_print(
		"%s",
"\n\
    The [type(s)...] specifies the type names of the MIME Types to\n\
    list.\n\
\n\
    Note that if you specify wildcards (* or ?) then you need to put\n\
    them between double quotes (\") such as \"*\" or else the shell\n\
    will expand them as regular object names instead of MIME Types.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	-a                      Print all information.\n\
	-l                      Print class and value.\n\
	--dereference-commands  Print only the referenced command\n\
				(instead of both the referenced\n\
				command and the reference).\n\
	-L                      Same as --dereference-commands.\n\
	--full-time             Print full date and time.\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
\n\
    To find a MIME Type assocated with an object, use:\n\
\n\
	open -l <object>\n\
\n"
	);
}


/*
 *	Returns a dynamically allocated string describing the timestamp.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	The ct specifies the current time in seconds since EPOCH.
 */
static gchar *mls_get_time_stamp_str(
	EDVContext *ctx,
	const gulong t,
	const gulong ct,
	MLSPrintFlags print_flags
)
{
	gchar *s;

	if(t == 0l)
		return(g_strdup(""));

	if(print_flags & MLS_PRINT_FULL_TIME)
	{
		s = g_strconcat(
			"                        ",	/* 24 spaces */
			edv_date_format_absolute_string(
				ctx,
				t,
				"%a %b %e %H:%M:%S %Y"
			),
			NULL
		);
		if(s != NULL)
			s[24] = '\0';
	}
	else
	{   
		/* Time is in the future? */
		if(t > ct)
			s = g_strconcat(
				"            ",	/* 12 spaces */
				edv_date_format_absolute_string(
					ctx,
					t,
					"%b %e %H:%M"
				),
				NULL
			);
		/* Less than a year old? */
		else if((gulong)(ct - t) < (365l * 24l * 60l * 60l))
			s = g_strconcat(
				"            ",	/* 12 spaces */
				edv_date_format_absolute_string(
					ctx,
					t,
					"%b %e %H:%M"
				),
				NULL
			);
		else
			s = g_strconcat(
				"            ",	/* 12 spaces */
				edv_date_format_absolute_string(
					ctx,
					t,
					"%b %e  %Y"
				),
				NULL
			);
		if(s != NULL)
			s[12] = '\0';
	}

	return(s);
}


/*
 *	Prints the timestamp.
 *
 *	The t specifies the time in seconds since EPOCH.
 *
 *	The ct specifies the current time in seconds since EPOCH.
 */
static void mls_print_time_stamp(
	EDVContext *ctx,
	const gulong t,
	const gulong ct,
	MLSPrintFlags print_flags
)
{
	gchar *s = mls_get_time_stamp_str(
		ctx,
		t,
		ct,
		print_flags
	);
	if(s != NULL)
	{
		g_print("%s", s);
		g_free(s);
	}
}

/*
 *	Prints the specified MIME Type.
 *
 *	The ct specifies the current time in seconds since EPOCH.
 */
static gint mls_print_mimetype(
	EDVContext *ctx,
	EDVMIMEType *m,
	const gulong ct,
	const MLSPrintFlags print_flags
)
{
	gint i;
	const gchar *s;

	if(m == NULL)
		return(-2);

	if((print_flags & MLS_PRINT_ALL) ||
	   (print_flags & MLS_PRINT_LONG)
	)
	{
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
			g_print("%s: System Object\n", m->type);
			break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
			g_print("%s: Format: \"%s\"\n", m->type, m->value);
			break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
			g_print("%s: Program: \"%s\"\n", m->type, m->value);
			break;
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
			g_print("%s: Unique: \"%s\"\n", m->type, m->value);
			break;
		}
	}
	else
	{
		g_print("%s\n", m->type);
	}


	/* Print the icon files */
	if(print_flags & MLS_PRINT_ALL)
	{
		g_print("\tStandard Icons:\n");
		i = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		s = m->small_icon_file[i];
		g_print("\t\tSmall: \"%s\"\n", (s != NULL) ? s : "");
		s = m->medium_icon_file[i];
		g_print("\t\tMedium: \"%s\"\n", (s != NULL) ? s : "");
		s = m->large_icon_file[i];
		g_print("\t\tLarge: \"%s\"\n", (s != NULL) ? s : "");

		g_print("\tOpened Icons:\n");
		i = EDV_MIME_TYPE_ICON_STATE_OPENED;
		s = m->small_icon_file[i];
		g_print("\t\tSmall: \"%s\"\n", (s != NULL) ? s : "");
		s = m->medium_icon_file[i];
		g_print("\t\tMedium: \"%s\"\n", (s != NULL) ? s : "");
		s = m->large_icon_file[i];
		g_print("\t\tLarge: \"%s\"\n", (s != NULL) ? s : "");

		g_print("\tInaccessable Icons:\n");
		i = EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE;
		s = m->small_icon_file[i];
		g_print("\t\tSmall: \"%s\"\n", (s != NULL) ? s : "");
		s = m->medium_icon_file[i];
		g_print("\t\tMedium: \"%s\"\n", (s != NULL) ? s : "");
		s = m->large_icon_file[i];
		g_print("\t\tLarge: \"%s\"\n", (s != NULL) ? s : "");

		g_print("\tHidden Icons:\n");
		i = EDV_MIME_TYPE_ICON_STATE_HIDDEN;
		s = m->small_icon_file[i];
		g_print("\t\tSmall: \"%s\"\n", (s != NULL) ? s : "");
		s = m->medium_icon_file[i];
		g_print("\t\tMedium: \"%s\"\n", (s != NULL) ? s : "");
		s = m->large_icon_file[i];
		g_print("\t\tLarge: \"%s\"\n", (s != NULL) ? s : "");
	}

	/* Print the commands */
	if((print_flags & MLS_PRINT_ALL) &&
	   (m->commands_list != NULL)
	)
	{
		const gint ncmds = g_list_length(m->commands_list);
		const gchar *command;
		GList *glist;
		EDVMIMETypeCommand *cmd;

		g_print("\tCommand%s:\n", (ncmds == 1) ? "" : "s");

		for(glist = m->commands_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			cmd = EDV_MIME_TYPE_COMMAND(glist->data);
			if(cmd == NULL)
				continue;

			command = cmd->command;
			if(command == NULL)
				continue;

			/* Dereference? */
			if(print_flags & MLS_PRINT_DEREFERENCE_COMMANDS)
			{
				/* Does this command refer to another MIME Type? */
				if(*command != G_DIR_SEPARATOR)
				{
					/* Print the referenced MIME Type */
					const EDVMIMEType *m2 = edv_mime_types_list_match_type(
						ctx,
						command
					);
					if(m2 != NULL)
					{
						if(m2->commands_list != NULL)
						{
							GList *glist = m2->commands_list;
							EDVMIMETypeCommand *cmd2 = EDV_MIME_TYPE_COMMAND(glist->data);
							if(cmd2 != NULL)
								g_print(
									"\t\t%s = \"%s\"\n",
									cmd->name,
									cmd2->command
								);
							else
								g_print(
									"\t\t%s = \"\"\n",
									cmd->name
								);
						}
						else
						{
							g_print(
								"\t\t%s = \"\"\n",
								cmd->name
							);
						}
					}
					else
					{
						g_print(
							"\t\t%s = \"%s\" (reference missing)\n",
							cmd->name,
							command
						);
					}
				}
				else
				{
					g_print(
						"\t\t%s = \"%s\"\n",
						cmd->name,
						command
					);
				}
			}
			else
			{
				g_print(
					"\t\t%s = \"%s\"",
					cmd->name,
					command
				);

				/* Does this command refer to another MIME Type? */
				if(*command != G_DIR_SEPARATOR)
				{
					/* Print the referenced MIME Type */
					const EDVMIMEType *m2 = edv_mime_types_list_match_type(
						ctx,
						command
					);
					if(m2 != NULL)
					{
						if(m2->commands_list != NULL)
						{
							GList *glist = m2->commands_list;
							EDVMIMETypeCommand *cmd2 = EDV_MIME_TYPE_COMMAND(glist->data);
							if(cmd2 != NULL)
								g_print(" -> \"%s\"", cmd2->command);
						}
					}
				}
				g_print("\n");
			}
			/* Shell Command */
			if(!STRISEMPTY(cmd->shell_command))
			{
				g_print(
					"\t\t\tShell Command = \"%s\"\n",
					cmd->shell_command
				);
			}
			/* Run In Terminal */
			if(cmd->flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL)
			{
				g_print(
					"\t\t\tRun In Terminal\n"
				);
			}
		}
	}

	/* Time stamps */
	if(print_flags & MLS_PRINT_ALL)
	{
		g_print("\tLast Accessed: ");
		mls_print_time_stamp(
			ctx,
			m->access_time,
			ct,
			print_flags
		);
		g_print("\n");
		g_print("\tLast Modified: ");
		mls_print_time_stamp(
			ctx,
			m->modify_time,
			ct,
			print_flags
		);
		g_print("\n");
		g_print("\tLast Changed: ");
		mls_print_time_stamp(
			ctx,
			m->change_time,
			ct,
			print_flags
		);
		g_print("\n");
	}

	return(0);
}


/*
 *	Prints the list of MIME Types.
 *
 *	The types_list specifies the MIME Type type name strings of
 *	the MIME Types to print (wildcards are allowed). If types_list
 *	is NULL then all the MIME Types will be printed.
 *
 *	The ct specifies the current time in seconds since EPOCH.
 */
static gint mls_list_mimetypes(
	EDVContext *ctx,
	GList *types_list,
	const gulong ct,
	const MLSPrintFlags print_flags
)
{
	gint status = 0;
	const gchar *type;
	GList *mime_types_list = edv_mime_types_list(ctx);
	EDVMIMEType *m;

	if(types_list != NULL)
	{
		gint	nmime_types_matched = 0,
			status2;
		GList	*glist,
			*glist2;

		/* Iterate through the types list */
		for(glist = types_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			type = (const gchar *)glist->data;
			if(type == NULL)
				continue;

			/* Iterate through the MIME Types list and print the
			 * MIME types that match this type
			 */
			for(glist2 = mime_types_list;
			    glist2 != NULL;
			    glist2 = g_list_next(glist2)
			)
			{
				m = EDV_MIME_TYPE(glist2->data);
				if(m == NULL)
					continue;

				if(m->type == NULL)
					continue;

				if(fnmatch(
					(const char *)type,
					(const char *)m->type,
					FNM_CASEFOLD
				) == 0)
				{
					status2 = mls_print_mimetype(
						ctx,
						m,
						ct,
						print_flags
					);
					if(status2 != 0)
					{
						if(status == 0)
							status = status2;
					}
					nmime_types_matched++;
				}
			}
		}
		if(status == 0)
		{
			/* No mime types matched? */
			if(nmime_types_matched == 0)
				status = -2;
		}
	}
	else
	{
		gint status2;
		GList *glist;
		for(glist = mime_types_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			status2 = mls_print_mimetype(
				ctx,
				EDV_MIME_TYPE(glist->data),
				ct,
				print_flags
			);
			if(status2 != 0)
			{
				if(status == 0)
					status = status2;
			}
		}
	}

	return(status);
}


/*
 *	MIME Types List.
 */
static gint mls(const gint argc, const gchar **argv)
{
	MLSPrintFlags print_flags = 0;
	gint		i,
			status;
	gulong ct;
	const gchar *arg;
	GList *types_list = NULL;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
 if(types_list != NULL) {		\
  g_list_foreach(			\
   types_list, (GFunc)g_free, NULL	\
  );					\
  g_list_free(types_list);		\
 }					\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);		\
					\
 return(_v_);				\
}

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* All */
		if(!strcmp(arg, "--a") ||
		   !strcmp(arg, "-a")
		)
		{
			print_flags |= MLS_PRINT_ALL;
		}
		/* Long */
		else if(!strcmp(arg, "--l") ||
			!strcmp(arg, "-l")
		)
		{
			print_flags |= MLS_PRINT_LONG;
		}
		/* Dereference Commands */
		else if(!g_strcasecmp(arg, "--dereference-commands") ||
			!g_strcasecmp(arg, "-dereference-commands") ||
			!strcmp(arg, "--L") ||
			!strcmp(arg, "-L")
		)
		{
			print_flags |= MLS_PRINT_DEREFERENCE_COMMANDS;
		}
		/* Full Time */
		else if(!g_strcasecmp(arg, "--full-time") ||
				!g_strcasecmp(arg, "-full-time")
		)
		{
			print_flags |= MLS_PRINT_FULL_TIME;
		}
		/* Switch */
		else if(*arg == '-')
		{
			while(*arg == '-')
				arg++;

			while(*arg != '\0')
			{
				switch(*arg)
				{
				  case 'a':
					print_flags |= MLS_PRINT_ALL;
					break;
				  case 'l':
					print_flags |= MLS_PRINT_LONG;
					break;
				  case 'L':
					print_flags |= MLS_PRINT_DEREFERENCE_COMMANDS;
					break;
				  default:
				    g_printerr(
"-%c: Unsupported argument.\n",
						*arg
					);
					CLEANUP_RETURN(-2);
					break;
				}
				arg++;
			}
		}
		/* All else assume MIME Type type */
		else if((*arg != '-') && (*arg != '+'))
		{
			types_list = g_list_append(
				types_list,
				g_strdup(arg)
			);
		}
	}

	/* Get the current time in seconds since EPOCH */
	ct = edv_time();

	/* Print the MIME Types */
	status = mls_list_mimetypes(
		ctx,
		types_list,
		ct,
		print_flags
	);

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(status);

#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = (const gchar *)argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			print_usage(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s",
"Endeavour Mark II MIME Types List Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* MIME Types List */
	status = mls((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
