/*
	Install New MIME Type

	Sends the new MIME Type values to Endeavour2 and instructs it
	to install it.

	Note, since there are so many input arguments needed to run
	this, there is a convience script called "mime_type_install.sh"
	that is used to call this and supply the arguments which will
	be used as the values for the new MIME Type.
 */

#include <stdlib.h>
#include <glib.h>
#include <endeavour2.h>


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s <class> [type] [value] [description] [handler] \
[icon_small_closed] [icon_small_opened] [icon_small_extended] [icon_small_hidden] \
[icon_medium_closed] [icon_medium_opened] [icon_medium_extended] [icon_medium_hidden] \
[icon_large_closed] [icon_large_opened] [icon_large_extended] [icon_large_hidden] \
[<command_name> <command>] [<command_name> <command>...]\n",
	    prog_name
	);
}


int main(int argc, char *argv[])
{
        /* Initialize the Endeavour 2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	if(argc > 1)
	{
	    gint i;
	    const EDVMIMETypeClass mt_class = (EDVMIMETypeClass)atoi(argv[1]);
	    const gchar *type = (argc > 2) ? argv[2] : NULL;

	    /* Check if the MIME Type already exists */
	    EDVMIMEType *m = edv_mime_types_list_match_type(
		ctx,
		type
	    );
	    if(m != NULL)
	    {
		/* The MIME Type already exists, we will be updating it */
		g_print(
		    "Updating MIME Type \"%s\"...\n",
		    type
		);

		/* Any checks on the current MIME Type's values should
		 * be done here to decide if we want to update it or not
		 */
	    }
	    else
	    {
		g_print(
		    "Installing MIME Type \"%s\"...\n",
		    type
		);
	    }

	    /* Create a new MIME Type */
	    m = edv_mime_type_new_values(
		mt_class,			/* Class */
		(argc > 3) ? argv[3] : NULL,	/* Value */
		type,				/* Type */
		(argc > 4) ? argv[4] : NULL	/* Description */
	    );
	    if(m == NULL)
	    {
		/* Memory allocation error */
		edv_context_delete(ctx);
		return(3);
	    }

	    /* Set the specified values to the new MIME Type */
	    m->handler = (EDVMIMETypeHandler)((argc > 5) ?
		atoi(argv[5]) : EDV_MIME_TYPE_HANDLER_COMMAND
	    );

	    if(argc > 6)
		m->small_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_STANDARD
		] = g_strdup(argv[6]);
            if(argc > 7)
                m->small_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_OPENED
                ] = g_strdup(argv[7]);
            if(argc > 8)
                m->small_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
                ] = g_strdup(argv[8]);
            if(argc > 9)
                m->small_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_HIDDEN
                ] = g_strdup(argv[9]);
            if(argc > 10)
                m->medium_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_STANDARD
                ] = g_strdup(argv[10]);
            if(argc > 11)
                m->medium_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_OPENED
                ] = g_strdup(argv[11]);
            if(argc > 12)
                m->medium_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
                ] = g_strdup(argv[12]);
            if(argc > 13)
                m->medium_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_HIDDEN
                ] = g_strdup(argv[13]);
            if(argc > 14)
                m->large_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_STANDARD
                ] = g_strdup(argv[14]);
            if(argc > 15)
                m->large_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_OPENED
                ] = g_strdup(argv[15]);
            if(argc > 16)
                m->large_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
                ] = g_strdup(argv[16]);
            if(argc > 17)
                m->large_icon_file[
                    EDV_MIME_TYPE_ICON_STATE_HIDDEN
                ] = g_strdup(argv[17]);

	    i = 18;
	    while(argc > i)
	    {
		EDVMIMETypeCommand *cmd = edv_mime_type_command_new();
		if(cmd != NULL)
		{
		    cmd->name = (argc > i) ? g_strdup(argv[i]) : NULL;
		    i++;
		    cmd->command = (argc > i) ? g_strdup(argv[i]) : NULL;
		    i++;

		    m->commands_list = g_list_append(
			m->commands_list,
			cmd
		    );
		}
		else
		{
		    i += 2;
		}
	    }

	    /* Since edv_mime_type_install() requires that Endeavour be
	     * running we need to map the MIME Types List Window to
	     * ensure that Endeavour is runned
	     */
	    g_print("Mapping the MIME Types list...\n");
	    edv_window_mime_types_list_map(
		ctx,
		type
	    );
	    edv_context_sync(ctx);

	    /* Send the new the MIME Type values to Endeavour and let
	     * Endeavour install the MIME Type
	     *
	     * Note that Endeavour will query the user to confirm the
	     * installation of the MIME Type, if the user clicks on
	     * "No" then the new MIME Type will not be installed
	     */
	    g_print("Sending the MIME Type values...\n");
	    edv_mime_type_install(ctx, m);
	    edv_context_sync(ctx);

	    g_print("Done\n");

	    edv_mime_type_delete(m);
	}
	else
	{
	    print_usage(argv[0]);
	}

        /* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

        /* Shutdown the Endeavour2 context */
	edv_context_delete(ctx);

	return(0);
}
