/*
	Directory List

	Usage of the edv_directory_*() functions to list contents of
	directories.
 */

#include <errno.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


typedef enum {
	DIR_PRINT_SORTED		= (1 << 0),	/* Sort alphabetically */
	DIR_PRINT_NOTATIONS		= (1 << 1),	/* Include ".." and "." */
	DIR_PRINT_FULL_PATHS		= (1 << 2),
	DIR_PRINT_POSITIONS		= (1 << 3)	/* Print stream locations */
} DirPrintFlags;


static void print_usage(const gchar *prog_name);

static gint dir_print_path(
	EDVContext *ctx,
	const gchar *path,
	const gulong ct,
	const DirPrintFlags print_flags
);

static gint dir_print_paths(
	EDVContext *ctx,
	GList *paths_list,
	const gulong ct,
	const DirPrintFlags print_flags
);

static gint dir(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [path(s)...] [options]\n",
		prog_name
	);

	g_print(
		"%s",
"\n\
    The [path(s)...] specifies the full paths to the directories.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	-f                      Unsorted.\n\
	-A                      Exclude \"..\" and \".\".\n\
	-l                      Print full paths.\n\
        -i			Print the stream position of each object.\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
\n"
	);
}


/*
 *	Prints the directory.
 */
static gint dir_print_path(
	EDVContext *ctx,
	const gchar *path,
	const gulong ct,
	const DirPrintFlags print_flags
)
{
	const gchar *name;

	/* Open the directory stream */
	EDVDirectory *dp = edv_directory_open(
		path,
		(print_flags & DIR_PRINT_SORTED) ? TRUE : FALSE,
		(print_flags & DIR_PRINT_NOTATIONS) ? TRUE : FALSE
	);
	if(dp == NULL)
	{
		const gint error_code = (gint)errno;
		g_printerr(
"%s: %s.\n",
			path,
			g_strerror(error_code)
		);
		return(-1);
	}

	/* Print each name in the directory stream */
	for(name = edv_directory_next(dp);
	    name != NULL;
	    name = edv_directory_next(dp)
	)
	{
		if(print_flags & DIR_PRINT_POSITIONS)
			g_print(
"%ld\t",
				edv_directory_tell(dp)
			);
		if(print_flags & DIR_PRINT_FULL_PATHS)
		{
			g_print(
"%s%c%s\n",
				path,
				G_DIR_SEPARATOR,
				name
			);
		}
		else
		{
			g_print(
"%s\n",
				name
			);
		}
	}

	/* Close the directory stream */
	edv_directory_close(dp);

	return(0);
}


/*
 *	Prints the directories.
 */
static gint dir_print_paths(
	EDVContext *ctx,
	GList *paths_list,
	const gulong ct,
	const DirPrintFlags print_flags
)
{
	gint		status = 0,
			status2;
	GList *glist;

	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		status2 = dir_print_path(
			ctx,
			(const gchar *)glist->data,
			ct,
			print_flags
		);
		if(status2 != 0)
		{
			if(status == 0)
				status = status2;
		}
	}

	return(status);
}


/*
 *	Directory List.
 */
static gint dir(const gint argc, const gchar **argv)
{
	DirPrintFlags print_flags =	DIR_PRINT_SORTED |
					DIR_PRINT_NOTATIONS;
	gint		i,
			status;
	gulong ct;
	const gchar *arg;
	GList *paths_list = NULL;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
 if(paths_list != NULL) {		\
  g_list_foreach(			\
   paths_list, (GFunc)g_free, NULL	\
  );					\
  g_list_free(paths_list);		\
 }					\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);		\
					\
 return(_v_);				\
}

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Unsorted */
		if(!strcmp(arg, "--f") ||
		   !strcmp(arg, "-f")
		)
		{
			print_flags &= ~DIR_PRINT_SORTED;
		}
		/* Exclude Notations */
		else if(!strcmp(arg, "--A") ||
		        !strcmp(arg, "-A")
		)
		{
			print_flags &= ~DIR_PRINT_NOTATIONS;
		}
		/* Full Paths */
		else if(!strcmp(arg, "--l") ||
		        !strcmp(arg, "-l")
		)
		{
			print_flags |= DIR_PRINT_FULL_PATHS;
		}
		/* Positions */
		else if(!strcmp(arg, "--i") ||
		        !strcmp(arg, "-i")
		)
		{
			print_flags |= DIR_PRINT_POSITIONS;
		}
		/* Switch */
		else if(*arg == '-')
		{
			while(*arg == '-')
				arg++;

			while(*arg != '\0')
			{
				switch(*arg)
				{
				  case 'l':
					print_flags |= DIR_PRINT_FULL_PATHS;
					break;
				  case 'f':
					print_flags &= ~DIR_PRINT_SORTED;
					break;
				  case 'A':
					print_flags &= ~DIR_PRINT_NOTATIONS;
					break;
				  case 'i':
					print_flags |= DIR_PRINT_POSITIONS;
					break;
				  default:
					g_printerr(
"-%c: Unsupported argument.\n",
						*arg
					);
					CLEANUP_RETURN(-2);
					break;
				}
				arg++;
			}
		}
		/* All else assume path */
		else if((*arg != '-') && (*arg != '+'))
		{
			paths_list = g_list_append(
				paths_list,
				g_strdup(arg)
			);
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(-2);
		}
	}

	/* Get the current time in seconds since EPOCH */
	ct = edv_time();

	/* If no paths were specified then list the current directory */
	if(paths_list == NULL)
		paths_list = g_list_append(
			paths_list,
			edv_getcwd()
		);

	/* Print the MIME Types */
	status = dir_print_paths(
		ctx,
		paths_list,
		ct,
		print_flags
	);

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(status);

#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = (const gchar *)argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			print_usage(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s",
"Endeavour Mark II Directory List Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* Directory List */
	status = dir((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
