/*
	Display Date & Time

	Usage of Endeavour2's date formatting capabilities.

	The date format is specified by the configuration, you can
	change the output of the date formatting here by running
	Endeavour2 and going to Settings->Options->Date & Time.

	Note, this module is named "edv_date" and not just "date" in
	order to avoid conflicts with the "date" command found on
	most systems.
 */

#include <stdlib.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


static void print_usage(const gchar *prog_name);

static void print_date(EDVContext *ctx, const gulong t);
static void print_delta(EDVContext *ctx, const gulong t);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [seconds] [options]\n",
	    prog_name
	);

	g_print(
		"%s",
"\n\
    The [seconds] specifies the time value in seconds since EPOCH.\n\
\n\
    If this value is not specified then the current time value is\n\
    used.\n\
\n\
    If the --delta argument is used then this value specifies the\n\
    delta time in seconds.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	-d, --delta             Print delta time (requires [seconds]\n\
				argument).\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
\n"
	);
}


/*
 *	Print the date and time.
 */
static void print_date(
	EDVContext *ctx,
	const gulong t
)
{
	const gchar *s = edv_date_format(ctx, t);
	if(s == NULL)
	    return;

	g_print("%s\n", s);
}

/*
 *	Print the delta time.
 */
static void print_delta(
	EDVContext *ctx,
	const gulong t
)
{
	const gchar *s = edv_date_format_duration(ctx, t);
	if(s == NULL)
	    return;

	g_print("%s\n", s);
}


int main(int argc, char *argv[])
{
	gboolean	opt_print_delta = FALSE,
			opt_t_set = FALSE;
	gint i;
	gulong t = 0l;
	const gchar *arg;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);			\
					\
 return(_v_);				\
}

	if(argc > 1)
	{
	    /* Parse arguments */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(arg == NULL)
		    continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
		    print_usage(argv[0]);
		    CLEANUP_RETURN(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
		        !g_strcasecmp(arg, "-version")
		)
		{
		    g_print(
			"%s",
"Endeavour Mark II Date Version " PROG_VERSION "\n"
PROG_COPYRIGHT
		    );
		    CLEANUP_RETURN(0);
		}
		/* Switch */
		else if(*arg == '-')
		{
		    arg++;
		    if(*arg == '-')
		    {
			arg++;
			if(!g_strcasecmp(arg, "delta"))
			{
			    opt_print_delta = TRUE;
			}
			else
			{
			    g_printerr(
"--%s: Unsupported argument.\n",
				arg
			    );
			    CLEANUP_RETURN(2);
			}
		    }
		    else
		    {
			while(*arg != '\0')
			{
			    switch(*arg)
			    {
			      case 'd':		/* Delta */
				opt_print_delta = TRUE;
				break;
			      default:
				g_printerr(
"-%c: Unsupported argument.\n",
				    *arg
				);
				CLEANUP_RETURN(2);
				break;
			    }
			    arg++;
			}
		    }
		}
		/* All else assume time value */
		else
		{
		    t = ATOL(arg);
		    opt_t_set = TRUE;
		}
	    }

	    /* Print delta time? */
	    if(opt_print_delta)
	    {
		if(!opt_t_set)
		{
		    g_printerr(
"Time value not specified.\n"
		    );
		    CLEANUP_RETURN(2);
		}

		print_delta(ctx, t);
	    }
	    /* Print date and time */
	    else
	    {
		/* Get the current time as needed */
		if(!opt_t_set)
		{
		    t = edv_time();
		    opt_t_set = TRUE;
		}
		print_date(ctx, t);
	    }
	}
	else
	{
	    /* No argument specified, print the current date and time */
	    print_date(ctx, edv_time());
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(0);

#undef CLEANUP_RETURN
}
