/*
	List Devices

	You can also mount/unmount/eject a device by using the following
	arguments:

	-m				Mount the device.
	-u				Unmount the device.
	-e				Eject the device
 */

#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


static void print_usage(const gchar *prog_name);

static void print_device_info(
	EDVContext *ctx,
	EDVDevice *d
);

static EDVDevice *match_device(
	EDVContext *ctx,
	const gchar *path
);

static void list_devices(EDVContext *ctx);

static void mount_device(EDVContext *ctx, EDVDevice *d);
static void unmount_device(EDVContext *ctx, EDVDevice *d);
static void eject_device(EDVContext *ctx, EDVDevice *d);


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [mount_path|device_path]\n\
	%s -m [mount_path|device_path]\n\
	%s -u [mount_path|device_path]\n\
	%s -e [mount_path|device_path]\n",
			prog_name,
			prog_name,
			prog_name,
			prog_name
	);
}


/*
 *	Prints information about the specified Device.
 */
static void print_device_info(
	EDVContext *ctx,
	EDVDevice *d
)
{
	if(d == NULL)
		return;

	g_print(
		"Device: \"%s\" (%s)\n",
		d->name,
		EDV_DEVICE_IS_MOUNTED(d) ? "Mounted" : "Unmounted"
	);
	g_print(
		"\tFilesystem: %s 0x%.8X\n",
		d->fs_type_name,
		d->fs_type_code
	);
	g_print(
		"\tDevice Path: %s\n",
		d->device_path
	);
	g_print(
		"\tMount Path: %s\n",
		d->mount_path
	);
	if(EDV_DEVICE_IS_MOUNTED(d))
	{
		gulong used;

		if(d->blocks_total > d->blocks_free)
			used = d->blocks_total - d->blocks_free;
		else
			used = 0l;
		g_print(
"\tCapacity Total: %ld kb (%ld mb)   Used: %ld kb (%ld mb) %i%%\n",
			d->blocks_total,
			d->blocks_total / 1000l,
			used,
			used / 1000l,
			(d->blocks_total > 0l) ?
				(gint)((gfloat)used / (gfloat)d->blocks_total * 100.0f) : 0
		);

		g_print(
"\tOptimul Transfer Block Size: %ld bytes\n",
			d->block_size
		);

		if(d->indicies_total > d->indicies_free)
			used = d->indicies_total - d->indicies_free;
		else
			used = 0l;
		g_print(
"\tIndicies (INodes) Total: %ld   Used: %ld %i%% \n",
			d->indicies_total,
			used,
			(d->indicies_total > 0l) ?
				(gint)((gfloat)used / (gfloat)d->indicies_total * 100.0f) : 0
		);

		g_print(
"\tName Length Max: %ld chars\n",
			d->name_length_max
		);
	}
	if(d->last_mount_time > 0l)
		g_print(
			"\tLast Mounted: %s\n",
			edv_date_format(ctx, d->last_mount_time)
		);
	if(d->last_check_time > 0l)
		g_print(
			"\tLast Checked: %s\n",
			edv_date_format(ctx, d->last_check_time)
		);
}


/*
 *	Matches a Device by the specified mount path or device path.
 */
static EDVDevice *match_device(
	EDVContext *ctx,
	const gchar *path
)
{
	EDVDevice *d;

	/* Update the Device's mount states and statistics */
	edv_devices_list_update_mount_states(ctx);
	edv_devices_list_update_stats(ctx);

	/* Match the Device by its mount path */
	d = edv_devices_list_match_mount_path(
		ctx,
		path
	);
	if(d != NULL)
		return(d);

	/* Match the Device by its device path */
	d = edv_devices_list_match_device_path(
		ctx,
		path
	);
	if(d != NULL)
		return(d);

	return(NULL);
}

/*
 *	Prints a list of all the Devices.
 */
static void list_devices(EDVContext *ctx)
{
	GList *glist, *devices_list = edv_devices_list(ctx);

	/* Update device mount states and stats */
	edv_devices_list_update_mount_states(ctx);
	edv_devices_list_update_stats(ctx);

	/* Print the information of all the Devices */
	for(glist = devices_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
		print_device_info(
			ctx,
			EDV_DEVICE(glist->data)
		);
}


/*
 *	Mounts the specified Device.
 */
static void mount_device(EDVContext *ctx, EDVDevice *d)
{
	if(edv_device_mount(ctx, d, TRUE))
		g_printerr("%s\n", edv_device_mount_get_error(ctx));
}

/*
 *	Unmounts the specified Device.
 */
static void unmount_device(EDVContext *ctx, EDVDevice *d)
{
	if(edv_device_unmount(ctx, d, TRUE))
		g_printerr("%s\n", edv_device_mount_get_error(ctx));
}

/*
 *	Ejects the specified Device.
 */
static void eject_device(EDVContext *ctx, EDVDevice *d)
{
	if(edv_device_eject(ctx, d, TRUE))
		g_printerr("%s\n", edv_device_mount_get_error(ctx));
}


int main(int argc, char *argv[])
{
	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)     {       \
					\
 /* Shutdown the Endeavour2 context */  \
 edv_context_delete(ctx);		\
					\
 return(_v_);                           \
}

	if(argc > 1)
	{
		gint i;
		const gchar *arg;

		/* Parse arguments */
		for(i = 1; i < argc; i++)
		{
			arg = argv[i];
			if(arg == NULL)
				continue;

			/* Help */
			if(!g_strcasecmp(arg, "--help") ||
			   !g_strcasecmp(arg, "-help") ||
			   !g_strcasecmp(arg, "--h") ||
			   !g_strcasecmp(arg, "-h")
			)
			{
				print_usage(argv[0]);
				CLEANUP_RETURN(0);
			}
			/* Version */
			else if(!g_strcasecmp(arg, "--version") ||
					!g_strcasecmp(arg, "-version")
			)
			{
				g_print(
					"%s",
"Endeavour Mark II Devices Version " PROG_VERSION "\n"
PROG_COPYRIGHT
				);
				CLEANUP_RETURN(0);
			}
		}
	}

	/* Multiple arguments specified? */
	if(argc > 2)
	{
		const gchar	*op = argv[1],
					*path = argv[2];
		EDVDevice *d = match_device(ctx, path);
		if(d != NULL)
		{
			if(!g_strcasecmp(op, "-m"))
				mount_device(ctx, d);
			else if(!g_strcasecmp(op, "-u"))
				unmount_device(ctx, d);
			else if(!g_strcasecmp(op, "-e"))
				eject_device(ctx, d);
			else
				g_printerr("%s: Invalid argument.\n", op);
		}
		else
			g_printerr("%s: No such device.\n", path);
	}
	/* Print a specific Device? */
	else if(argc > 1)
	{
		const gchar *path = argv[1];
		EDVDevice *d = match_device(ctx, path);
		if(d != NULL)
			print_device_info(ctx, d);
		else
			g_printerr("%s: No such device.\n", path);
	}
	/* List all the Devices */
	else
	{
		/* Print the list of devices */
		list_devices(ctx);
		/* Print the usage afterwards */
		g_print("\n");
		print_usage(argv[0]);
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(0);

#undef CLEANUP_RETURN
}
