#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "cfg.h"


/* Item Types */
CfgIntList *CFGIntListNew(GList *glist);
void CFGIntListDelete(CfgIntList *intlist);

CfgStringList *CFGStringListNew(GList *glist);
void CFGStringListDelete(CfgStringList *string_list);

CfgColor *CFGColorNew(
	const gfloat r,
	const gfloat g,
	const gfloat b,
	const gfloat a
);
void CFGColorDelete(CfgColor *color);

CfgAccelkey *CFGAccelkeyNew(
	const gint op_id,
	const guint key,
	const guint modifiers
);
void CFGAccelkeyDelete(CfgAccelkey *accelkey);
CfgAccelkeyList *CFGAccelkeyListNew(GList *accelkeys_list);
void CFGAccelkeyListDelete(CfgAccelkeyList *accelkeys_list);

CfgStyle *CFGStyleNew(void);
void CFGStyleDelete(CfgStyle *style);

CfgMenuItem *CFGMenuItemNew(
	const CfgMenuItemFlags flags,
	const gchar *label,
	const gchar *command,
	const gchar *icon_file,
	const gchar *description,
	const gchar *ext_data
);
void CFGMenuItemDelete(CfgMenuItem *mi);
CfgMenu *CFGMenuNew(GList *menu_items_list);
void CFGMenuDelete(CfgMenu *m);


/* CfgItem */
CfgItem *CFGItemNew(
	const CfgItemType type,
	const gchar *parameter
);
gpointer CFGItemGetValue(
	const CfgItem *item,
	CfgItemType *type_rtn
);
void CFGItemSetValue(
	CfgItem *item,
	gconstpointer value
);
void CFGItemResetValue(CfgItem *item);
void CFGItemReset(CfgItem *item);
void CFGItemDelete(CfgItem *item);

/* CfgList Utilities */
CfgList *CFGItemListCopyList(const CfgList *list);
void CFGItemListDeleteList(CfgList *list);
gint CFGItemListMatchParameter(
	const CfgList *list,
	const gchar *parameter
);

/* Get CfgItem Value From CfgList */
gpointer CFGItemListMatchGetValue(
	const CfgList *list,
	const gchar *parameter,
	CfgItemType *type_rtn
);
gint CFGItemListGetValueI(
	const CfgList *list,
	const gchar *parameter
);
glong CFGItemListGetValueL(
	const CfgList *list,
	const gchar *parameter
);
gulong CFGItemListGetValueUL(
	const CfgList *list,
	const gchar *parameter
);
gfloat CFGItemListGetValueF(
	const CfgList *list,
	const gchar *parameter
);
gdouble CFGItemListGetValueD(
	const CfgList *list,
	const gchar *parameter
);
gchar *CFGItemListGetValueS(
	const CfgList *list,
	const gchar *parameter
);
CfgColor *CFGItemListGetValueColor(
	const CfgList *list,
	const gchar *parameter
);
CfgIntList *CFGItemListGetValueIntList(
	const CfgList *list,
	const gchar *parameter
);
CfgStringList *CFGItemListGetValueStringList(
	const CfgList *list,
	const gchar *parameter
);
CfgAccelkeyList *CFGItemListGetValueAccelkeyList(
	const CfgList *list,
	const gchar *parameter
);
CfgStyle *CFGItemListGetValueStyle(
	const CfgList *list,
	const gchar *parameter
);
CfgMenu *CFGItemListGetValueMenu(
	const CfgList *list,
	const gchar *parameter
);


/* Cfg Item Value Setting */
void CFGItemListSetValue(
	CfgList *list,
	const gchar *parameter,
	gconstpointer value,
	const gboolean create_as_needed
);
void CFGItemListSetValueI(
	CfgList *list,
	const gchar *parameter,
	const gint value,
	const gboolean create_as_needed
);
void CFGItemListSetValueL(
	CfgList *list,
	const gchar *parameter,
	const glong value,
	const gboolean create_as_needed
);
void CFGItemListSetValueUL(
	CfgList *list,
	const gchar *parameter,
	const gulong value,
	const gboolean create_as_needed
);
void CFGItemListSetValueF(
	CfgList *list,
	const gchar *parameter,
	const gfloat value,
	const gboolean create_as_needed
);
void CFGItemListSetValueD(
	CfgList *list,
	const gchar *parameter,
	const gdouble value,
	const gboolean create_as_needed
);
void CFGItemListSetValueS(
	CfgList *list,
	const gchar *parameter,
	const gchar *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueColor(
	CfgList *list,
	const gchar *parameter,
	const CfgColor *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueIntList(
	CfgList *list,
	const gchar *parameter,
	const CfgIntList *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueStringList(
	CfgList *list,
	const gchar *parameter,
	const CfgStringList *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueAccelkeyList(
	CfgList *list,
	const gchar *parameter,
	const CfgAccelkeyList *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueStyle(
	CfgList *list,
	const gchar *parameter,
	const CfgStyle *value,
	const gboolean create_as_needed
);
void CFGItemListSetValueMenu(
	CfgList *list,
	const gchar *parameter,
	const CfgMenu *value,
	const gboolean create_as_needed
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new IntList.
 *
 *	The values from the specified GList will be coppied.
 */
CfgIntList *CFGIntListNew(GList *glist)
{
	CfgIntList *intlist = CFG_INT_LIST(
	    g_malloc(sizeof(CfgIntList))
	);
	if(intlist == NULL)
	    return(NULL);

	intlist->list = NULL;
	while(glist != NULL)
	{
	    intlist->list = g_list_append(intlist->list, glist->data);
	    glist = g_list_next(glist);
	}

	return(intlist);
}

/*
 *	Deletes the IntList.
 */
void CFGIntListDelete(CfgIntList *intlist)
{
	if(intlist == NULL)
	    return;

	g_list_free(intlist->list);
	g_free(intlist);
}


/*
 *	Creates a new StringList.
 */
CfgStringList *CFGStringListNew(GList *glist)
{
	CfgStringList *string_list = CFG_STRING_LIST(
	    g_malloc(sizeof(CfgStringList))
	);
	if(string_list == NULL)
	    return(NULL);

	string_list->list = NULL;
	while(glist != NULL)
	{
	    string_list->list = g_list_append(
		string_list->list,
		STRDUP(glist->data)
	    );
	    glist = g_list_next(glist);
	}

	return(string_list);
}

/*
 *	Deletes the StringList.
 */
void CFGStringListDelete(CfgStringList *string_list)
{
	if(string_list == NULL)
	    return;

	if(string_list->list != NULL)
	{
	    g_list_foreach(
		string_list->list,
		(GFunc)g_free,
		NULL
	    );
	    g_list_free(string_list->list);
	}
	g_free(string_list);
}


/*
 *	Creates a new Color.
 */
CfgColor *CFGColorNew(
	const gfloat r,
	const gfloat g,
	const gfloat b,
	const gfloat a
)
{
	CfgColor *color = CFG_COLOR(
	    g_malloc(sizeof(CfgColor))
	);
	if(color == NULL)
	    return(NULL);

	color->r = r;
	color->g = g;
	color->b = b;
	color->a = a;

	return(color);
}

/*
 *	Deletes the Color.
 */
void CFGColorDelete(CfgColor *color)
{
	if(color == NULL)
	    return;

	g_free(color);
}

/*
 *	Creates a new Accelkey.
 */
CfgAccelkey *CFGAccelkeyNew(
	const gint op_id,
	const guint key,
	const guint modifiers
)
{
	CfgAccelkey *akey = CFG_ACCELKEY(
	    g_malloc(sizeof(CfgAccelkey))
	);
	if(akey == NULL)
	    return(NULL);

	akey->op_id = op_id;
	akey->key = key;
	akey->modifiers = modifiers;

	return(akey);
}

/*
 *	Deletes the Accelkey.
 */
void CFGAccelkeyDelete(CfgAccelkey *accelkey)
{
	if(accelkey == NULL)
	    return;

	g_free(accelkey);
}

/*
 *	Creates a new Accelkey List.
 *
 *	The values from the specified GList will be coppied.
 */
CfgAccelkeyList *CFGAccelkeyListNew(GList *accelkeys_list)
{
	GList *glist;
	CfgAccelkey *akey;
	CfgAccelkeyList *cfg_accelkeys_list = CFG_ACCELKEY_LIST(
	    g_malloc(sizeof(CfgAccelkeyList))
	);
	if(cfg_accelkeys_list == NULL)
	    return(NULL);

	cfg_accelkeys_list->list = NULL;

	for(glist = accelkeys_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    akey = CFG_ACCELKEY(glist->data);
	    if(akey == NULL)
		continue;

	    cfg_accelkeys_list->list = g_list_append(
		cfg_accelkeys_list->list,
		CFGAccelkeyNew(
		    akey->op_id,
		    akey->key,
		    akey->modifiers
		)
	    );
	}

	return(cfg_accelkeys_list);
}

/*
 *	Deletes the Accelkey List.
 */
void CFGAccelkeyListDelete(CfgAccelkeyList *accelkeys_list)
{
	if(accelkeys_list == NULL)
	    return;

	if(accelkeys_list->list != NULL)
	{
	    g_list_foreach(
		accelkeys_list->list,
		(GFunc)CFGAccelkeyDelete,
		NULL
	    );
	    g_list_free(accelkeys_list->list);
	}

	g_free(accelkeys_list);
}

/*
 *	Creates a new Cfg Style.
 */
CfgStyle *CFGStyleNew(void)
{
	CfgStyle *style = CFG_STYLE(
	    g_malloc0(sizeof(CfgStyle))
	);
	if(style == NULL)
	    return(NULL);

	return(style);
}

/*
 *	Deletes the Cfg Style.
 */
void CFGStyleDelete(CfgStyle *style)
{
	gint i;

	if(style == NULL)
	    return;

	g_free(style->font_name);

	for(i = 0; i < CFG_STYLE_STATES; i++)
	    g_free(style->bg_pixmap_name[i]);

	g_free(style);
}


/*
 *	Creates a new Cfg Menu Item.
 */
CfgMenuItem *CFGMenuItemNew(
	const CfgMenuItemFlags flags,
	const gchar *label,
	const gchar *command,
	const gchar *icon_file,
	const gchar *description,
	const gchar *ext_data
)
{
	CfgMenuItem *mi = CFG_MENU_ITEM(g_malloc(
	    sizeof(CfgMenuItem)
	));
	if(mi == NULL)
	    return(NULL);

	mi->flags = flags;
	mi->label = STRDUP(label);
	mi->command = STRDUP(command);
	mi->icon_file = STRDUP(icon_file);
	mi->description = STRDUP(description);
	mi->ext_data = STRDUP(ext_data);

	return(mi);
}

/*
 *	Deletes the Cfg Menu Item.
 */
void CFGMenuItemDelete(CfgMenuItem *mi)
{
	if(mi == NULL)
	    return;

	g_free(mi->label);
	g_free(mi->command);
	g_free(mi->icon_file);
	g_free(mi->description);
	g_free(mi);
}

/*
 *	Creates a new Cfg Menu.
 */
CfgMenu *CFGMenuNew(GList *menu_items_list)
{
	GList *glist;
	CfgMenuItem *mi;
	CfgMenu *m = CFG_MENU(g_malloc0(
	    sizeof(CfgMenu)
	));
	if(m == NULL)
	    return(NULL);

	m->list = NULL;

	for(glist = menu_items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    mi = CFG_MENU_ITEM(glist->data);
	    if(mi == NULL)
		continue;

	    m->list = g_list_append(
		m->list,
		CFGMenuItemNew(
		    mi->flags,
		    mi->label,
		    mi->command,
		    mi->icon_file,
		    mi->description,
		    mi->ext_data
		)
	    );
	}

	return(m);
}

/*
 *	Deletes the Cfg Menu.
 */
void CFGMenuDelete(CfgMenu *m)
{
	if(m == NULL)
	    return;

	if(m->list != NULL)
	{
	    g_list_foreach(
		m->list,
		(GFunc)CFGMenuItemDelete,
		NULL
	    );
	    g_list_free(m->list);
	}

	g_free(m);
}


/*
 *	Creates a new Cfg Item.
 */
CfgItem *CFGItemNew(
	const CfgItemType type,
	const gchar *parameter
)
{
	CfgItem *item = CFG_ITEM(g_malloc(sizeof(CfgItem)));
	if(item == NULL)
	    return(NULL);

	item->type = type;
	item->parameter = STRDUP(parameter);
	item->value = NULL;

	return(item);
}

/*
 *	Returns the Cfg Item's value and type.
 */
gpointer CFGItemGetValue(
	const CfgItem *item,
	CfgItemType *type_rtn
)
{
	if(item != NULL)
	{
	    if(type_rtn != NULL)
		*type_rtn = item->type;
	    return(item->value);
	}
	else
	{
	    if(type_rtn != NULL)
		*type_rtn = CFG_ITEM_TYPE_NONE;
	    return(NULL);
	}
}

/*
 *	Sets the Cfg Item's value.
 *
 *	The type of data that value points to must match the same
 *	type of data of the item's type.
 *
 *	Note that item->value may be reallocated during this call.
 */
void CFGItemSetValue(
	CfgItem *item,
	gconstpointer value
)
{
	CfgIntList *tar_intlist;
	const CfgIntList *src_intlist;
	CfgStringList *tar_string_list;
	const CfgStringList *src_string_list;
	CfgColor *tar_color;
	const CfgColor *src_color;
	CfgAccelkeyList *tar_ak_list;
	const CfgAccelkeyList *src_ak_list;
	CfgStyle *tar_style;
	CfgMenu *tar_menu;
	const CfgMenu *src_menu;

	if(item == NULL)
	    return;

	/* Delete the item's existing value */
	CFGItemResetValue(item);

	/* No value specified? */
	if(value == NULL)
	    return;

	/* Set by the item's type */
	switch(item->type)
	{
	  case CFG_ITEM_TYPE_NONE:
	    break;

	  case CFG_ITEM_TYPE_INT8:
	    item->value = g_memdup(value, sizeof(gint8));
	    break;
	  case CFG_ITEM_TYPE_UINT8:
	    item->value = g_memdup(value, sizeof(guint8));
	    break;

	  case CFG_ITEM_TYPE_INT16:
	    item->value = g_memdup(value, sizeof(gint16));
	    break;
	  case CFG_ITEM_TYPE_UINT16:
	    item->value = g_memdup(value, sizeof(guint16));
	    break;

	  case CFG_ITEM_TYPE_INT32:
	    item->value = g_memdup(value, sizeof(gint32));
	    break;
	  case CFG_ITEM_TYPE_UINT32:
	    item->value = g_memdup(value, sizeof(guint32));
	    break;

	  case CFG_ITEM_TYPE_INT64:
	    item->value = g_memdup(value, sizeof(gint64));
	    break;
	  case CFG_ITEM_TYPE_UINT64:
	    item->value = g_memdup(value, sizeof(guint64));
	    break;

	  case CFG_ITEM_TYPE_FLOAT:
	    item->value = g_memdup(value, sizeof(gfloat));
	    break;
	  case CFG_ITEM_TYPE_DOUBLE:
	    item->value = g_memdup(value, sizeof(gdouble));
	    break;

	  case CFG_ITEM_TYPE_STRING:
	    item->value = STRDUP((const gchar *)value);
	    break;

	  case CFG_ITEM_TYPE_INT_LIST:
	    src_intlist = CFG_INT_LIST(value);
	    tar_intlist = CFGIntListNew(src_intlist->list);
	    item->value = tar_intlist;
	    break;

	  case CFG_ITEM_TYPE_STRING_LIST:
	    src_string_list = CFG_STRING_LIST(value);
	    tar_string_list = CFGStringListNew(src_string_list->list);
	    item->value = tar_string_list;
	    break;

	  case CFG_ITEM_TYPE_COLOR:
	    src_color = CFG_COLOR(value);
	    tar_color = CFGColorNew(
		src_color->r,
		src_color->g,
		src_color->b,
		src_color->a
	    );
	    item->value = tar_color;
	    break;

	  case CFG_ITEM_TYPE_ACCELKEY_LIST:
	    src_ak_list = CFG_ACCELKEY_LIST(value);
	    tar_ak_list = CFGAccelkeyListNew(
		src_ak_list->list
	    );
	    item->value = tar_ak_list;
	    break;

	  case CFG_ITEM_TYPE_STYLE:
	    tar_style = CFGStyleNew();
	    if(tar_style != NULL)
	    {
		gint state;
		const CfgStyle *src_style = CFG_STYLE(value);

		/* Copy values from source style to target style */
		tar_style->font_name = STRDUP(src_style->font_name);
		memcpy(
		    tar_style->color_flags, src_style->color_flags,
		    CFG_STYLE_STATES * sizeof(guint)
		);
		memcpy(
		    tar_style->fg, src_style->fg,
		    CFG_STYLE_STATES * sizeof(CfgColor)
		);
		memcpy(
		    tar_style->bg, src_style->bg,
		    CFG_STYLE_STATES * sizeof(CfgColor)
		);
		memcpy(
		    tar_style->text, src_style->text,
		    CFG_STYLE_STATES * sizeof(CfgColor)
		);
		memcpy(
		    tar_style->base, src_style->base,
		    CFG_STYLE_STATES * sizeof(CfgColor)
		);
		for(state = 0; state < CFG_STYLE_STATES; state++)
		    tar_style->bg_pixmap_name[state] = STRDUP(
			src_style->bg_pixmap_name[state]
		    );
		item->value = tar_style;
	    }
	    break;

	  case CFG_ITEM_TYPE_MENU:
	    src_menu = CFG_MENU(value);
	    tar_menu = CFGMenuNew(
		src_menu->list
	    );
	    item->value = tar_menu;
	    break;
	}
}

/*
 *	Deletes the Item's value and then resets it to NULL.
 */
void CFGItemResetValue(CfgItem *item)
{
	if(item == NULL)
	    return;

	/* Value already deleted? */
	if(item->value == NULL)
	    return;

	/* Delete the value by the item's type */
	switch(item->type)
	{
	  case CFG_ITEM_TYPE_NONE:
	    break;
	  case CFG_ITEM_TYPE_INT8:
	    break;
	  case CFG_ITEM_TYPE_UINT8:
	    break;
	  case CFG_ITEM_TYPE_INT16:
	    break;
	  case CFG_ITEM_TYPE_UINT16:
	    break;
	  case CFG_ITEM_TYPE_INT32:
	    break;
	  case CFG_ITEM_TYPE_UINT32:
	    break;
	  case CFG_ITEM_TYPE_INT64:
	    break;
	  case CFG_ITEM_TYPE_UINT64:
	    break;
	  case CFG_ITEM_TYPE_FLOAT:
	    break;
	  case CFG_ITEM_TYPE_DOUBLE:
	    break;
	  case CFG_ITEM_TYPE_STRING:
	    break;
	  case CFG_ITEM_TYPE_INT_LIST:
	    CFGIntListDelete(CFG_INT_LIST(item->value));
	    item->value = NULL;
	    break;
	  case CFG_ITEM_TYPE_STRING_LIST:
	    CFGStringListDelete(CFG_STRING_LIST(item->value));
	    item->value = NULL;
	    break;
	  case CFG_ITEM_TYPE_COLOR:
	    CFGColorDelete(CFG_COLOR(item->value));
	    item->value = NULL;
	    break;
	  case CFG_ITEM_TYPE_ACCELKEY_LIST:
	    CFGAccelkeyListDelete(CFG_ACCELKEY_LIST(item->value));
	    item->value = NULL;
	    break;
	  case CFG_ITEM_TYPE_STYLE:
	    CFGStyleDelete(CFG_STYLE(item->value));
	    item->value = NULL;
	    break;
	  case CFG_ITEM_TYPE_MENU:
	    CFGMenuDelete(CFG_MENU(item->value));
	    item->value = NULL;
	    break;
	}

	/* Delete value in case it was not deleted above and reset it to
	 * NULL
	 */
	g_free(item->value);
	item->value = NULL;
}

/*
 *	Deletes the Cfg Item's value and parameter and resets its type
 *	to CFG_ITEM_TYPE_NONE.
 */
void CFGItemReset(CfgItem *item)
{
	if(item == NULL)
	    return;

	/* Delete the value */
	CFGItemResetValue(item);

	/* Delete the parameter */
	g_free(item->parameter);
	item->parameter = NULL;

	/* Reset the type */
	item->type = CFG_ITEM_TYPE_NONE;
}

/*
 *	Deletes the Cfg Item.
 */
void CFGItemDelete(CfgItem *item)
{
	if(item == NULL)
	    return;

	/* Delete the Cfg Item's value and parameter */
	CFGItemReset(item);

	/* Delete the Cfg Item */
	g_free(item);
}


/*
 *	Coppies the list of Cfg Items.
 *
 *	Each member of each Cfg Item will also be coppied.
 */
CfgList *CFGItemListCopyList(const CfgList *list)
{
	gint i;
	const CfgItem		*src_list = list,
				*src_cfg_item;
	CfgItem		*tar_list = NULL,
			*tar_cfg_item;

	if(src_list == NULL)
	    return(tar_list);


	/* Iterate through source list, copying each configuration
	 * item to the target list
	 */
	i = 0;
	src_cfg_item = &src_list[i];
	while((src_cfg_item->type != CFG_ITEM_TYPE_NONE) ||
	      (src_cfg_item->parameter != NULL) ||
	      (src_cfg_item->value != NULL)
	)
	{
	    /* Aappend a new cfg item on the target list */
	    tar_list = CFG_ITEM(g_realloc(
		tar_list,
		(i + 1) * sizeof(CfgItem)
	    ));
	    if(tar_list == NULL)
	    {
		i = 0;
		break;
	    }

	    /* Get the pointer to new target cfg item */
	    tar_cfg_item = &tar_list[i];

	    /* Reset the target cfg item */
	    memset(tar_cfg_item, 0x00, sizeof(CfgItem));


	    /* Begin copying values from source cfg item to the
	     * target cfg item
	     */

	    /* Type */
	    tar_cfg_item->type = src_cfg_item->type;

	    /* Parameter */
	    tar_cfg_item->parameter = STRDUP(src_cfg_item->parameter);

	    /* Value */
	    if(src_cfg_item->value != NULL)
	    {
		switch(tar_cfg_item->type)
		{
		  case CFG_ITEM_TYPE_NONE:
		  case CFG_ITEM_TYPE_INT8:
		  case CFG_ITEM_TYPE_UINT8:
		  case CFG_ITEM_TYPE_INT16:
		  case CFG_ITEM_TYPE_UINT16:
		  case CFG_ITEM_TYPE_INT32:
		  case CFG_ITEM_TYPE_UINT32:
		    CFGItemListSetValueI(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueI(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_INT64:
		    CFGItemListSetValueL(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueL(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_UINT64:
		    CFGItemListSetValueUL(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueUL(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_FLOAT:
		    CFGItemListSetValueF(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueF(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_DOUBLE:
		    CFGItemListSetValueD(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueD(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_STRING:
		    CFGItemListSetValueS(
			tar_cfg_item, tar_cfg_item->parameter,
	CFGItemListGetValueS(src_cfg_item, src_cfg_item->parameter),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_INT_LIST:
		    CFGItemListSetValueIntList(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_INT_LIST(src_cfg_item->value),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_STRING_LIST:
		    CFGItemListSetValueStringList(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_STRING_LIST(src_cfg_item->value),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_COLOR:
		    CFGItemListSetValueColor(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_COLOR(src_cfg_item->value),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_ACCELKEY_LIST:
		    CFGItemListSetValueAccelkeyList(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_ACCELKEY_LIST(src_cfg_item->value),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_STYLE:
		    CFGItemListSetValueStyle(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_STYLE(src_cfg_item->value),
			FALSE
		    );
		    break;
		  case CFG_ITEM_TYPE_MENU:
		    CFGItemListSetValueMenu(
			tar_cfg_item, tar_cfg_item->parameter,
			CFG_MENU(src_cfg_item->value),
			FALSE
		    );
		    break;
		}
	    }

	    /* Go on to the next source Cfg Item */
	    i++;
	    src_cfg_item = &src_list[i];
	}

	/* Create the last Cfg Item on the target list to be all 0's,
	 * thus marking the end of the list
	 */
	tar_list = CFG_ITEM(g_realloc(
	    tar_list,
	    (i + 1) * sizeof(CfgItem)
	));
	if(tar_list != NULL)
	{
	    tar_cfg_item = &tar_list[i];
	    memset(tar_cfg_item, 0x00, sizeof(CfgItem));
	}

	return(tar_list);
}

/*
 *	Deletes the list of Cfg Items.
 */
void CFGItemListDeleteList(CfgList *list)
{
	gint i;
	CfgItem *item;

	if(list == NULL)
	    return;

	/* Reset each Cfg Item in the List, thus deleting each Cfg
	 * Item's members
	 */
	i = 0;
	item = &list[i];
	while((item->type != CFG_ITEM_TYPE_NONE) ||
	      (item->parameter != NULL) ||
	      (item->value != NULL)
	)
	{
	    /* Delete the cfg item's parameter and value */
	    CFGItemReset(item);

	    /* Go on to the next cfg item */
	    i++;
	    item = &list[i];
	}

	/* Delete the Cfg Items List */
	g_free(list);
}

/*
 *	Scans through the sequential list of configuration items and
 *	returns the index number of the one who's parameter matches the
 *	given parameter (case insensitive).
 *
 *	The last item in the sequential list needs to be all 0.
 *
 *	Returns the index of the matched item or negative on error.
 */
gint CFGItemListMatchParameter(
	const CfgList *list,
	const gchar *parameter
)
{
	gint i;
	const CfgItem *item;

	if((list == NULL) || STRISEMPTY(parameter))
	    return(-2);

	i = 0;
	item = &list[i];
	while(item->type != CFG_ITEM_TYPE_NONE)
	{
	    if(item->parameter != NULL)
	    {
		if(!g_strcasecmp(item->parameter, parameter))
		    return(i);
	    }

	    i++;
	    item = &list[i];
	}

	return(-1);
}


/*
 *	Returns the value pointer and type of the Cfg Item that matches
 *	the given parameter found in the given Cfg Items List..
 *
 *      Returns NULL on failed match or error.
 *
 *	The returned pointer must not be modified or deleted.
 */
gpointer CFGItemListMatchGetValue(
	const CfgList *list,
	const gchar *parameter,
	CfgItemType *type_rtn
)
{
	gint i;

	/* Check if the specified parameter matches the parameter of a
	 * cfg item in the list
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i < 0)
	{
	    /* No such parameter */
	    if(type_rtn != NULL)
		*type_rtn = CFG_ITEM_TYPE_NONE;
	    return(NULL);
	}

	/* Return the parameter's value and type */
	return(CFGItemGetValue(&list[i], type_rtn));
}

/*
 *	Returns the gint value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 */
gint CFGItemListGetValueI(
	const CfgList *list,
	const gchar *parameter
)
{
#define RTN_VAL_T	gint
	RTN_VAL_T rtn_val = (RTN_VAL_T)0;
	gint item_num;
	const CfgItem *item_ptr;

	gint8 *ptr_8;
	guint8 *ptr_u8;
	gint16 *ptr_16;
	guint16 *ptr_u16;
	gint32 *ptr_32;
	guint32 *ptr_u32;

	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(rtn_val);
	else
	    item_ptr = &list[item_num];

	/* No value? */
	if(item_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type */
	switch((gint)item_ptr->type)
	{
	  case CFG_ITEM_TYPE_INT8: 
	    ptr_8 = (gint8 *)item_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_8); 
	    break;
	  case CFG_ITEM_TYPE_UINT8: 
	    ptr_u8 = (guint8 *)item_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_u8); 
	    break;

	  case CFG_ITEM_TYPE_INT16:
	    ptr_16 = (gint16 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_16);
	    break;
	  case CFG_ITEM_TYPE_UINT16:
	    ptr_u16 = (guint16 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u16);
	    break;

	  case CFG_ITEM_TYPE_INT32:
	    ptr_32 = (gint32 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_32);
	    break;
	  case CFG_ITEM_TYPE_UINT32:
	    ptr_u32 = (guint32 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u32);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns the glong value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 *		CFG_ITEM_TYPE_INT64
 *		CFG_ITEM_TYPE_UINT64
 */
glong CFGItemListGetValueL(
	const CfgList *list,
	const gchar *parameter
)
{
#define RTN_VAL_T	glong
	RTN_VAL_T rtn_val = (RTN_VAL_T)0;
	gint item_num;
	const CfgItem *item_ptr;

	gint64 *ptr_64;
	guint64 *ptr_u64;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(rtn_val);
	else
	    item_ptr = &list[item_num];

	/* No value? */
	if(item_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type */
	switch((gint)item_ptr->type)  
	{
	  case CFG_ITEM_TYPE_INT8:
	  case CFG_ITEM_TYPE_UINT8:
	  case CFG_ITEM_TYPE_INT16:
	  case CFG_ITEM_TYPE_UINT16:
	  case CFG_ITEM_TYPE_INT32:
	  case CFG_ITEM_TYPE_UINT32:
	    rtn_val = (RTN_VAL_T)CFGItemListGetValueI(
		list, parameter
	    );
	    break;

	  case CFG_ITEM_TYPE_INT64:
	    ptr_64 = (gint64 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_64);
	    break;
	  case CFG_ITEM_TYPE_UINT64:
	    ptr_u64 = (guint64 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u64);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T  
}           

/*
 *	Returns the gulong value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 *		CFG_ITEM_TYPE_INT64
 *		CFG_ITEM_TYPE_UINT64
 */
gulong CFGItemListGetValueUL(
	const CfgList *list,
	const gchar *parameter
)
{
#define RTN_VAL_T       gulong
	RTN_VAL_T rtn_val = (RTN_VAL_T)0;
	gint item_num;
	const CfgItem *item_ptr;

	gint64 *ptr_64;
	guint64 *ptr_u64;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(rtn_val);
	else
	    item_ptr = &list[item_num];

	/* No value? */
	if(item_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type */
	switch((gint)item_ptr->type)
	{
	  case CFG_ITEM_TYPE_INT8:
	  case CFG_ITEM_TYPE_UINT8:
	  case CFG_ITEM_TYPE_INT16:
	  case CFG_ITEM_TYPE_UINT16:
	  case CFG_ITEM_TYPE_INT32:
	  case CFG_ITEM_TYPE_UINT32:
	    rtn_val = (RTN_VAL_T)CFGItemListGetValueI(
		list, parameter
	    );
	    break;

	  case CFG_ITEM_TYPE_INT64:
	    ptr_64 = (gint64 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_64);
	    break;
	  case CFG_ITEM_TYPE_UINT64:
	    ptr_u64 = (guint64 *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u64);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns the gfloat value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 *		CFG_ITEM_TYPE_INT64
 *		CFG_ITEM_TYPE_UINT64
 *		CFG_ITEM_TYPE_FLOAT
 */
gfloat CFGItemListGetValueF(
	const CfgList *list,
	const gchar *parameter
)
{
#define RTN_VAL_T	float
	RTN_VAL_T rtn_val = (float)0.0;
	gint item_num;
	const CfgItem *item_ptr;

	gfloat *ptr_f;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)  
	    return(rtn_val);
	else
	    item_ptr = &list[item_num];

	/* No value? */
	if(item_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type */
	switch((gint)item_ptr->type)
	{
	  case CFG_ITEM_TYPE_INT8:
	  case CFG_ITEM_TYPE_UINT8:
	  case CFG_ITEM_TYPE_INT16:
	  case CFG_ITEM_TYPE_UINT16:
	  case CFG_ITEM_TYPE_INT32:
	  case CFG_ITEM_TYPE_UINT32:
	  case CFG_ITEM_TYPE_INT64:
	  case CFG_ITEM_TYPE_UINT64:
	    rtn_val = (RTN_VAL_T)CFGItemListGetValueL(
		list, parameter
	    );
	    break;

	  case CFG_ITEM_TYPE_FLOAT:
	    ptr_f = (float *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_f);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns the gdouble value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8 
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16  
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 *		CFG_ITEM_TYPE_INT64
 *		CFG_ITEM_TYPE_UINT64
 *		CFG_ITEM_TYPE_FLOAT
 *		CFG_ITEM_TYPE_DOUBLE
 */
gdouble CFGItemListGetValueD(
	const CfgList *list,
	const gchar *parameter
)
{
#define RTN_VAL_T	gdouble
	RTN_VAL_T rtn_val = (RTN_VAL_T)0.0;
	gint item_num;
	const CfgItem *item_ptr;

	gdouble *ptr_d;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(rtn_val);
	else
	    item_ptr = &list[item_num];

	/* No value? */
	if(item_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type */
	switch((gint)item_ptr->type)
	{
	  case CFG_ITEM_TYPE_INT8:
	  case CFG_ITEM_TYPE_UINT8:
	  case CFG_ITEM_TYPE_INT16:
	  case CFG_ITEM_TYPE_UINT16:
	  case CFG_ITEM_TYPE_INT32:
	  case CFG_ITEM_TYPE_UINT32:
	  case CFG_ITEM_TYPE_INT64:
	  case CFG_ITEM_TYPE_UINT64:
	  case CFG_ITEM_TYPE_FLOAT:
	    rtn_val = (RTN_VAL_T)CFGItemListGetValueF(
		list, parameter
	    );
	    break;

	  case CFG_ITEM_TYPE_DOUBLE:
	    ptr_d = (double *)item_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_d);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns the string value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_STRING
 */
gchar *CFGItemListGetValueS(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_STRING) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return((gchar *)item_ptr->value);
}

/*
 *	Returns the color value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_COLOR
 */
CfgColor *CFGItemListGetValueColor(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_COLOR) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_COLOR(item_ptr->value));
}

/*
 *	Returns the intlist value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_INT_LIST
 */
CfgIntList *CFGItemListGetValueIntList(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;


	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_INT_LIST) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_INT_LIST(item_ptr->value));
}

/*
 *	Gets the StringList value from the CfgItem in the CfgList.
 *
 *	Only CfgItems of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_STRING_LIST
 */
CfgStringList *CFGItemListGetValueStringList(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;

	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_STRING_LIST) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_STRING_LIST(item_ptr->value));
}

/*
 *	Returns the accelkey list value of the Cfg Item who matches
 *	the given parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_ACCELKEY_LIST
 */
CfgAccelkeyList *CFGItemListGetValueAccelkeyList(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;

	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_ACCELKEY_LIST) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_ACCELKEY_LIST(item_ptr->value));
}

/*
 *	Returns the style value of the Cfg Item who matches the given
 *	parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_STYLE
 */
CfgStyle *CFGItemListGetValueStyle(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;

	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_STYLE) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_STYLE(item_ptr->value));
}

/*
 *	Returns the menu value of the Cfg Item who matches
 *	the given parameter found in the given Cfg Items List.
 *
 *	Only Cfg Items of the following types will be matched:
 *
 *		CFG_ITEM_TYPE_MENU
 */
CfgMenu *CFGItemListGetValueMenu(
	const CfgList *list,
	const gchar *parameter
)
{
	gint item_num;
	const CfgItem *item_ptr;

	/* Match by parameter */
	item_num = CFGItemListMatchParameter(list, parameter);
	if(item_num < 0)
	    return(NULL);
	else
	    item_ptr = &list[item_num];

	/* Not the correct type or no value? */
	if((item_ptr->type != CFG_ITEM_TYPE_MENU) ||
	   (item_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return(CFG_MENU(item_ptr->value));
}


/*
 *      Sets the given value to the sequential list item found match
 *	the given parameter.
 *
 *      The last item in the sequential list needs to be all 0.
 *
 *	Returns the pointer to the new sequential list, if
 *	create_as_needed is false then pointer will be exactly the same
 *	as the list pointer value. If create_as_needed is true then
 *	the returned pointer will be a new pointer to the new
 *	sequential list and the old list will be deallocated.
 *
 *	Can return NULL on error if create_as_needed is true.
 */
void CFGItemListSetValue(
	CfgList *list,
	const gchar *parameter,
	gconstpointer value,
	const gboolean create_as_needed
)
{
	gint i;


	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of cfg items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    /* Matched existing configuration item */
	    CFGItemSetValue(&list[i], value);
	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *      Sets value as a gint, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_INT8
 *              CFG_ITEM_TYPE_UINT8
 *              CFG_ITEM_TYPE_INT16
 *              CFG_ITEM_TYPE_UINT16
 *              CFG_ITEM_TYPE_INT32
 *              CFG_ITEM_TYPE_UINT32
 *              CFG_ITEM_TYPE_INT64
 *              CFG_ITEM_TYPE_UINT64
 *              CFG_ITEM_TYPE_FLOAT
 *              CFG_ITEM_TYPE_DOUBLE
 */
void CFGItemListSetValueI(
	CfgList *list,
	const gchar *parameter,
	const gint value,
	const gboolean create_as_needed
)
{
	CFGItemListSetValueL(
	    list,
	    parameter,
	    value,
	    create_as_needed
	);
}

/*
 *      Sets value as a gint, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_INT8
 *              CFG_ITEM_TYPE_UINT8
 *              CFG_ITEM_TYPE_INT16
 *              CFG_ITEM_TYPE_UINT16
 *              CFG_ITEM_TYPE_INT32
 *              CFG_ITEM_TYPE_UINT32
 *              CFG_ITEM_TYPE_INT64
 *              CFG_ITEM_TYPE_UINT64
 *              CFG_ITEM_TYPE_FLOAT
 *              CFG_ITEM_TYPE_DOUBLE
 */
void CFGItemListSetValueL(
	CfgList *list,
	const gchar *parameter,
	const glong value,
	const gboolean create_as_needed
)
{
	gint i;
	gpointer ptr;
	gint8 v_8;
	guint8 v_u8;
	gint16 v_16;
	guint16 v_u16;
	gint32 v_32;
	guint32 v_u32;
	gint64 v_64;
	guint64 v_u64;
	gfloat v_float;
	gdouble v_double;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    ptr = NULL;
	    switch((gint)item->type)
	    {
	      case CFG_ITEM_TYPE_INT8:
		v_8 = (gint8)value;
		ptr = (gpointer)&v_8;
		break;
	      case CFG_ITEM_TYPE_UINT8:
		v_u8 = (guint8)value;
		ptr = (gpointer)&v_u8;
		break;

	      case CFG_ITEM_TYPE_INT16:
		v_16 = (gint16)value;
		ptr = (gpointer)&v_16;
		break;
	      case CFG_ITEM_TYPE_UINT16:
		v_u16 = (guint16)value;
		ptr = (gpointer)&v_u16;
		break;

	      case CFG_ITEM_TYPE_INT32:
		v_32 = (gint32)value;
		ptr = (gpointer)&v_32;
		break;
	      case CFG_ITEM_TYPE_UINT32:
		v_u32 = (guint32)value;
		ptr = (gpointer)&v_u32;
		break;

	      case CFG_ITEM_TYPE_INT64:
		v_64 = (gint64)value;
		ptr = (gpointer)&v_64;
		break;
	      case CFG_ITEM_TYPE_UINT64:
		v_u64 = (guint64)value;
		ptr = (gpointer)&v_u64;
		break;

	      case CFG_ITEM_TYPE_FLOAT:
		v_float = (gfloat)value;
		ptr = (gpointer)&v_float;
		break;
	      case CFG_ITEM_TYPE_DOUBLE:
		v_double = (gdouble)value;
		ptr = (gpointer)&v_double;
		break;
	    }
	    if(ptr == NULL)
		return;

	    CFGItemSetValue(item, ptr);
	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *      Sets value as a gulong, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_INT8
 *              CFG_ITEM_TYPE_UINT8
 *              CFG_ITEM_TYPE_INT16
 *              CFG_ITEM_TYPE_UINT16
 *              CFG_ITEM_TYPE_INT32
 *              CFG_ITEM_TYPE_UINT32
 *              CFG_ITEM_TYPE_INT64
 *              CFG_ITEM_TYPE_UINT64
 *              CFG_ITEM_TYPE_FLOAT
 *              CFG_ITEM_TYPE_DOUBLE
 */
void CFGItemListSetValueUL(
	CfgList *list,
	const gchar *parameter,
	const gulong value,
	const gboolean create_as_needed
)
{
	CFGItemListSetValueL(
	    list,
	    parameter,
	    value,
	    create_as_needed
	);
}

/*
 *	Sets value as a gfloat, the parameter must match to an item
 *	who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_INT8
 *		CFG_ITEM_TYPE_UINT8
 *		CFG_ITEM_TYPE_INT16
 *		CFG_ITEM_TYPE_UINT16
 *		CFG_ITEM_TYPE_INT32
 *		CFG_ITEM_TYPE_UINT32
 *		CFG_ITEM_TYPE_INT64
 *		CFG_ITEM_TYPE_UINT64
 *		CFG_ITEM_TYPE_FLOAT
 */
void CFGItemListSetValueF(
	CfgList *list,
	const gchar *parameter,
	const gfloat value,
	const gboolean create_as_needed
)
{
	CFGItemListSetValueD(
	    list,
	    parameter,
	    (gdouble)value,
	    create_as_needed
	);
}

/*
 *      Sets value as a gdouble, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_INT8
 *              CFG_ITEM_TYPE_UINT8
 *              CFG_ITEM_TYPE_INT16
 *              CFG_ITEM_TYPE_UINT16
 *              CFG_ITEM_TYPE_INT32
 *              CFG_ITEM_TYPE_UINT32
 *              CFG_ITEM_TYPE_INT64
 *              CFG_ITEM_TYPE_UINT64
 *              CFG_ITEM_TYPE_FLOAT
 *              CFG_ITEM_TYPE_DOUBLE
 */
void CFGItemListSetValueD(
	CfgList *list,
	const gchar *parameter,
	const gdouble value,
	const gboolean create_as_needed
)
{
	gint i;
	gpointer ptr;
	gint8 v_8;
	guint8 v_u8;
	gint16 v_16;
	guint16 v_u16;
	gint32 v_32;
	guint32 v_u32;
	gint64 v_64;
	guint64 v_u64;
	gfloat v_float;
	gdouble v_double;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    ptr = NULL;
	    switch((gint)item->type)
	    {
	      case CFG_ITEM_TYPE_INT8:
		v_8 = (gint8)value;
		ptr = (gpointer)&v_8;
		break;
	      case CFG_ITEM_TYPE_UINT8:
		v_u8 = (guint8)value;
		ptr = (gpointer)&v_u8;
		break;

	      case CFG_ITEM_TYPE_INT16:
		v_16 = (gint16)value;
		ptr = (gpointer)&v_16;
		break;
	      case CFG_ITEM_TYPE_UINT16:
		v_u16 = (guint16)value;
		ptr = (gpointer)&v_u16;
		break;

	      case CFG_ITEM_TYPE_INT32:
		v_32 = (gint32)value;
		ptr = (gpointer)&v_32;
		break;
	      case CFG_ITEM_TYPE_UINT32:
		v_u32 = (guint32)value;
		ptr = (gpointer)&v_u32;
		break;

	      case CFG_ITEM_TYPE_INT64:
		v_64 = (gint64)value;
		ptr = (gpointer)&v_64;
		break;
	      case CFG_ITEM_TYPE_UINT64:
		v_u64 = (guint64)value;
		ptr = (gpointer)&v_u64;
		break;

	      case CFG_ITEM_TYPE_FLOAT:
		v_float = (gfloat)value;
		ptr = (gpointer)&v_float;
		break;
	      case CFG_ITEM_TYPE_DOUBLE:
		v_double = (gdouble)value;
		ptr = (gpointer)&v_double;
		break;
	    }
	    if(ptr == NULL)
		return;

	    CFGItemSetValue(item, ptr);
	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *      Sets value as a string, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_STRING
 */
void CFGItemListSetValueS(
	CfgList *list,
	const gchar *parameter,
	const gchar *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_STRING)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *	Sets value as a CfgColor, the parameter must match to
 *	an item who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_COLOR
 */
void CFGItemListSetValueColor(
	CfgList *list,
	const gchar *parameter,
	const CfgColor *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_COLOR)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *	Sets value as a CfgIntList, the parameter must match
 *	to an item who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_INT_LIST
 */
void CFGItemListSetValueIntList(
	CfgList *list,
	const gchar *parameter,
	const CfgIntList *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the given
	 * list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_INT_LIST)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *	Sets value as a CfgStringList, the parameter must match
 *	to an item who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_STRING_LIST
 */
void CFGItemListSetValueStringList(
	CfgList *list,
	const gchar *parameter,
	const CfgStringList *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the given
	 * list of configuration items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_STRING_LIST)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *	Sets value as a CfgAccelkeyList, the parameter must
 *	match to an item who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_ACCELKEY_LIST
 */
void CFGItemListSetValueAccelkeyList(
	CfgList *list,
	const gchar *parameter,
	const CfgAccelkeyList *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of cfg items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_ACCELKEY_LIST)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *      Sets value as a CfgStyle, the parameter must match to
 *	an item who's type matches any of the following:
 *
 *              CFG_ITEM_TYPE_STYLE
 */
void CFGItemListSetValueStyle(
	CfgList *list,
	const gchar *parameter,
	const CfgStyle *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of cfg items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_STYLE)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}

/*
 *	Sets value as a CfgMenu, the parameter must
 *	match to an item who's type matches any of the following:
 *
 *		CFG_ITEM_TYPE_MENU
 */
void CFGItemListSetValueMenu(
	CfgList *list,
	const gchar *parameter,
	const CfgMenu *value,
	const gboolean create_as_needed
)
{
	gint i;

	if(list == NULL)
	    return;

	/* First try to match an existing configuration item in the
	 * given list of cfg items
	 */
	i = CFGItemListMatchParameter(list, parameter);
	if(i > -1)
	{
	    CfgItem *item = &list[i];
	    if(item->type == CFG_ITEM_TYPE_MENU)
		CFGItemSetValue(item, (gconstpointer)value);

	    return;
	}
	else
	{
	    /* Does not exist, are we allowed to create one? */
/* TODO */
	}
}
