#ifdef HAVE_LIBXPM
#include <stdio.h>
#include <stdlib.h>				/* mkstemp() */
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <X11/xpm.h>				/* libxpm */
#include <unistd.h>				/* close() */
#include "rgba_to_cidx.h"
#include "imgio.h"
#include "config.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0
#endif


typedef struct _RGBAColor		RGBAColor;
#define RGBA_COLOR(p)			((RGBAColor *)(p))

typedef struct _SymbolicRGBAColor	SymbolicRGBAColor;
#define SYM_RGBA_COLOR(p)		((SymbolicRGBAColor *)(p))

typedef struct _RGBAToCIdxData		RGBAToCIdxData;
#define RGBA_TO_CIDX_DATA(p)		((RGBAToCIdxData *)(p))


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


/* Color Utilities */
static int ImgXPMGetSymbolicColor(
	ColormapColor *c,
	const char *name
);

/* Version */
void ImgXPMVersion(int *major, int *minor, int *release);


/* Open */
static char *ImgXPMGetCIDFromFileName(const char *path);
int ImgFileOpenXPMRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);


/* Save */
static int ImgXpmRGBAToCIdxCB(int i, int m, void *data);
static int ImgXPMWriteXPMImageToXPMFile(
	const char *path,
	const char *c_id,
	XpmImage *img,
	XpmInfo *info
);
static int ImgFileSaveXPMRGBAToColor(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int max_colors,			/* -1 for no color limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgFileSaveXPMRGBAToGreyscale(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgFileSaveXPMRGBAToBW(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments, 
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileSaveXPMRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale    
						 * 2 = Color */
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_XPM_PROGRESS_RESOLUTION	10


/*
 *	XPM Pixel ASCII Characters:
 *
 *	These are ASCII character values that represent a range of
 *	ASCII characters that may be used as characters to represent
 *	a pixel value in an XPM file.
 *
 *	The range here must not contain the '"' or '\' characters since
 *	these characters will be parsed by C interpreters when the
 *	XPM file is used in C source code.
 */
#define XPM_ASCII_CHAR_MIN		35	/* '#' */
#define XPM_ASCII_CHAR_MAX		90	/* 'Z' */
#define XPM_ASCII_CHAR_RANGE		(XPM_ASCII_CHAR_MAX - XPM_ASCII_CHAR_MIN + 1)


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)

#define ISSPACE(c)	(((c) == ' ') || ((c) == '\t') ||	\
			 ((c) == '\v') || ((c) == '\f') ||	\
			 ((c) == '\r') || ((c) == '\n')		\
			)

/*
 *	RGBA Color:
 */
struct _RGBAColor {
	u_int8_t	r, g, b, a;
};

/*
 *	Symbolic RGBA Color:
 */
struct _SymbolicRGBAColor {
	char		*name;
	u_int8_t	r, g, b, a;
};


/*
 *	RGBAToCIdx Callback Data:
 */
struct _RGBAToCIdxData {

	int		width, height,
			bpp, bpl;
	const u_int8_t	*data;

	ImgProgressFunc progress_cb;
	void		*progress_data;
	int		*user_aborted;
};


/*
 *	Symbolic Colors List:
 */
#define IMG_XPM_SYM_RGBA_COLOR_LIST	{		\
{	"Black",	0x00, 0x00, 0x00, 0xff },	\
{	"White",	0xff, 0xff, 0xff, 0xff },	\
{	"None",		0x00, 0x00, 0x00, 0x00 },	\
{	"Transparent",	0x00, 0x00, 0x00, 0x00 },	\
{	"Background",	0x00, 0x00, 0x00, 0x00 },	\
{	"LightGrey",	0xc0, 0xc0, 0xc0, 0xff },	\
{	"Grey",		0x80, 0x80, 0x80, 0xff },	\
{	"DarkGrey",	0x40, 0x40, 0x40, 0xff },	\
{	"Red",		0xff, 0x00, 0x00, 0xff },	\
{	"Green",	0x00, 0xff, 0x00, 0xff },	\
{	"Blue",		0x00, 0x00, 0xff, 0xff }	\
}
static SymbolicRGBAColor sym_rgba_color_list[] = IMG_XPM_SYM_RGBA_COLOR_LIST;


/*
 *	Sets the color to the symbolic color specified by name.
 *
 *	Returns 1 if set or 0 if not.
 */
static int ImgXPMGetSymbolicColor(
	ColormapColor *c,
	const char *name
)
{
	int i;
	const SymbolicRGBAColor *sym_c;
	const int m = sizeof(sym_rgba_color_list) / sizeof(SymbolicRGBAColor);

	for(i = 0; i < m; i++)
	{
		sym_c = &sym_rgba_color_list[i];
		if(!strcasecmp(name, sym_c->name))
		{
			c->r = sym_c->r;
			c->g = sym_c->g;
			c->b = sym_c->b;
			return(1);
		}
	}

	return(0);
}

/*
 *	Gets the C ID from the XPM file.
 */
static char *ImgXPMGetCIDFromFileName(const char *path)
{
	const int buf_len = 4 * 80;		/* First 4 lines */
	char		*s,
			*buf,
			*c_id = NULL;
	size_t units_read;
	FILE *fp = fopen(path, "rb");
	if(fp == NULL)
		return(NULL);

	buf = (char *)malloc((buf_len + 1) * sizeof(char));
	if(buf == NULL)
	{
		(void)fclose(fp);
		return(NULL);
	}

	/* Read a few characters at the start of the file, hoping to
	 * find the C ID
	 */
	units_read = fread(
		buf,
		sizeof(char),
		(size_t)buf_len,
		fp
	);

	(void)fclose(fp);

	if((units_read > 0l) && (units_read <= (size_t)buf_len))
	{
		buf[units_read] = '\0';

		/* Seek to type declaration of C ID (if any) */
		s = strstr(buf, "char");
		if(s != NULL)
			s = strchr(s, '*');
		if(s != NULL)
		{
			/* Found declaration of C ID, now see s to start of the
			 * C ID
			 */
			char *s_end;

			while(*s == '*')
				s++;
			while(ISSPACE(*s))
				s++;

			/* Seek s_end to end of C ID */
			s_end = s;
			while((*s_end != '\0') && (*s_end != '[') &&
				  (*s_end != '=') && !ISSPACE(*s_end)
			)
				s_end++;

			/* Copy C ID to return string */
			if(s_end > s)
			{
				int c_id_len = s_end - s;
				c_id = (char *)malloc((c_id_len + 1) * sizeof(char));
				if(c_id != NULL)
				{
					memcpy(c_id, s, c_id_len);
					c_id[c_id_len] = '\0';
				}
			}
		}
	}

	free(buf);

	return(c_id);
}


/*
 *	Gets the XPM library's version.
 */
void ImgXPMVersion(int *major, int *minor, int *release)
{
	const int n = XpmLibraryVersion();
	if(major != NULL)
		*major = n / 10000;
/*		*major = XpmFormat; */
	if(minor != NULL)
		*minor = (n % 10000) / 100;
/*		*minor = XpmVersion; */
	if(release != NULL)
		*release = 'a' + (n % 10000) % 100 - 1;
/*		*release = XpmRevision; */
}


/*
 *	Reads the XPM file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value (invalid format/not a xpm file)
 *	-3	Systems error
 *	-4	User abort
 */
int ImgFileOpenXPMRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		width, height,
			bpl,
			ncolors = 0,
			transparent_color_num = -1;
	unsigned long info_flags;
	u_int8_t *rgba = NULL;
	ColormapColor *cmap = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;

#define CLEANUP_RETURN(_v_)	{	\
 if(img != NULL) {			\
  XpmFreeXpmImage(img);			\
  free(img);				\
 }					\
 if(info != NULL) {			\
  XpmFreeXpmInfo(info);			\
  free(info);				\
 }					\
 free(rgba);				\
 free(cmap);				\
 return(_v_);				\
}

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip if user aborted */
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		return(-4);
	}

	/* Allocate the XPM image and XPM info */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));	
	if((img == NULL) || (info == NULL))
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}

	/* Read the XPM image file and get the XpmImage and XpmInfo */
	switch(XpmReadFileToXpmImage((char *)path, img, info))
	{
	  case XpmColorError:
		imgio_last_open_error = "XPM color error"; 
		CLEANUP_RETURN(-1);
		break;
	  case XpmSuccess:
		break;
	  case XpmOpenFailed:
		imgio_last_open_error = "Unable to open the XPM file for reading"; 
		CLEANUP_RETURN(-1);
		break;
	  case XpmFileInvalid:
		imgio_last_open_error = "Invalid XPM file";
		CLEANUP_RETURN(-2);
		break;
	  case XpmNoMemory:
		imgio_last_open_error = "Memory allocation error"; 
		CLEANUP_RETURN(-3);
		break;
	  case XpmColorFailed:
		imgio_last_open_error = "XPM color failed";
		CLEANUP_RETURN(-1);
		break;
	  default:
		imgio_last_open_error = "Error reading XPM file";
		CLEANUP_RETURN(-1);
		break;
	}

	/* Get the XPM values */
	info_flags = info->valuemask;
	width = img->width;
	height = img->height;
	bpl = width * bpp;
	if((width <= 0) || (height <= 0))
	{
		imgio_last_open_error = "Invalid image size";
		CLEANUP_RETURN(-2);
	}
	if(img->data == NULL)
	{
		imgio_last_open_error = "No image data";
		CLEANUP_RETURN(-2);
	}

	/* Create the colormap from the XPM colormap */
	ncolors = img->ncolors;
	if(ncolors > 0)
	{
		const int	po = 0,
					pm = 2 * ncolors;
		int i;
		const char *s;
		ColormapColor *c;
		const XpmColor *xpm_color = img->colorTable, *xpm_c;

		cmap = (ColormapColor *)realloc(
			cmap,
			ncolors * sizeof(ColormapColor)
		);
		if(cmap == NULL)
		{
			ncolors = 0;
			imgio_last_open_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}

		/* Get each color from the XPM colormap */
		for(i = 0; i < ncolors; i++)
		{
			xpm_c = &xpm_color[i];
			c = &cmap[i];

			/* Report progress */
			if((progress_cb != NULL) && ((i % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + i, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{
					*user_aborted = 1;
					break;
				}
			}

#define IS_COLOR_STRING_TRANSPARENT(_s_)	\
 (!strcasecmp((_s_), "None") || !strcasecmp((_s_), "Transparent"))

			/* Symbolic color? */
			s = xpm_c->symbolic;
			if(s != NULL)
			{
				if(IS_COLOR_STRING_TRANSPARENT(s))
				{
					if(transparent_color_num < 0)
						transparent_color_num = i;
				}
				if(!ImgXPMGetSymbolicColor(c, s))
				{
					c->r = 0xff;
					c->g = 0xff;
					c->b = 0xff;
				}
				continue;
			}

			/* Get color string and parse it */
			s = xpm_c->c_color;
			if(s != NULL)
			{
				/* Symbolic color? */
				if(*s != '#')
				{
					if(IS_COLOR_STRING_TRANSPARENT(s))
					{
						if(transparent_color_num < 0)
							transparent_color_num = i;
					}
					if(!ImgXPMGetSymbolicColor(c, s))
					{
						c->r = 0xff;
						c->g = 0xff;
						c->b = 0xff;
					}
					continue;
				}

				/* Seek past blanks and the '#' prefix */
				while((*s == ' ') || (*s == '\t') || (*s == '#'))
					s++;
				if(STRLEN(s) >= 6)
				{
#define HEX_CTOI(_c_)	(u_int8_t)(((_c_) >= 'a') ?	\
 ((_c_) - 'a' + 10) : ((_c_) - '0')			\
)
					c->r = (HEX_CTOI(tolower(s[0])) * 16) +
						HEX_CTOI(tolower(s[1]));
					c->g = (HEX_CTOI(tolower(s[2])) * 16) +
						HEX_CTOI(tolower(s[3]));
					c->b = (HEX_CTOI(tolower(s[4])) * 16) +
						HEX_CTOI(tolower(s[5]));
#undef HEX_CTOI
				}
				continue;
			}
#undef IS_COLOR_STRING_TRANSPARENT
		}
	}
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}

	/* Allocate the RGBA image data */
	rgba = (u_int8_t *)realloc(
		rgba,
		bpl * height * sizeof(u_int8_t)
	);
	if(rgba == NULL)
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}

	/* Copy/convert the XPM color index image data to the RGBA
	 * image data
	 */
	if(!(*user_aborted))
	{
		const int	po = height,
					pm = 2 * height;
		int y;
		unsigned int ci;
		const unsigned int	*cidx_ptr,
							*cidx_end;
		u_int8_t *rgba_ptr;
		const ColormapColor *c;

		/* Iterate through the XPM color index image */
		for(y = 0; y < height; y++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((y % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + y, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{
					*user_aborted = 1;
					break;
				}
			}

			cidx_ptr = img->data + (y * width);
			cidx_end = cidx_ptr + width;
			rgba_ptr = rgba + (y * bpl);
			while(cidx_ptr < cidx_end)
			{
				/* Get the colormap color index for this pixel */
				ci = *cidx_ptr++;
				c = (ci < ncolors) ? &cmap[ci] : NULL;

				/* Transparent? */
				if((int)ci == transparent_color_num)
				{
					*rgba_ptr++ = c->r;
					*rgba_ptr++ = c->g;
					*rgba_ptr++ = c->b;
					*rgba_ptr++ = 0x00;
				}
				else if(c != NULL)
				{
					*rgba_ptr++ = c->r;
					*rgba_ptr++ = c->g;
					*rgba_ptr++ = c->b;
					*rgba_ptr++ = def_alpha_value;
				}
				else
				{
					rgba_ptr += bpp;
				}
			}
		}
	}
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}

	/* Handle extensions */
	if(info_flags & XpmExtensions)
	{
		unsigned int i;
		XpmExtension *ext;

		for(i = 0; i < info->nextensions; i++)
		{
			ext = &info->extensions[i];
#if (DEBUG_LEVEL >= 1)
printf(
 "ImgFileOpenXPMRGBA(): Extension \"%s\" encountered.\n",
 ext->name
);
#endif

		}
	}

	/* Set the return values */
	if(width_rtn != NULL)
		*width_rtn = width;
	if(height_rtn != NULL)
		*height_rtn = height;
	if(bpl_rtn != NULL)
		*bpl_rtn = bpl;
	if(rgba_rtn != NULL)
	{
		free(*rgba_rtn);
		*rgba_rtn = rgba;
		rgba = NULL;
	}
	if(x_rtn != NULL)
		*x_rtn = (info_flags & XpmHotspot) ? (int)info->x_hotspot : 0;
	if(y_rtn != NULL)
		*y_rtn = (info_flags & XpmHotspot) ? (int)info->y_hotspot : 0;
	if(base_width_rtn != NULL)
		*base_width_rtn = width;
	if(base_height_rtn != NULL)
		*base_height_rtn = height;

	/* Get the C ID (if any) and return it as the title */
	if(title_rtn != NULL)
	{
		free(*title_rtn);
		*title_rtn = ImgXPMGetCIDFromFileName(path);
	}

	if(comments_rtn != NULL)
	{
		free(*comments_rtn);
		*comments_rtn = NULL;
#if defined(XpmComments)
/* Comments support has been dropped since libXpm 3.0c,
 * but we support it only if its still there */
		if(info_flags & XpmComments)
		{
			if(!STRISEMPTY(info->hints_cmt))
				*comments_rtn = strdup(info->hints_cmt);
			else if(!STRISEMPTY(info->colors_cmt))
				*comments_rtn = strdup(info->colors_cmt);
			else if(!STRISEMPTY(info->pixels_cmt))
				*comments_rtn = strdup(info->pixels_cmt);
		}
#endif	/* XpmComments */
	}

	/* Report the final progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			progress_data,
			height, height,
			width, height,
			bpl, bpp,
			rgba
		))
			*user_aborted = 1;
	}

	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}
	else
	{
		CLEANUP_RETURN(0);
	}
#undef CLEANUP_RETURN
}


/*
 *	RGBAToCIdx progress callback.
 */
static int ImgXpmRGBAToCIdxCB(int i, int m, void *data)
{
	RGBAToCIdxData *d = RGBA_TO_CIDX_DATA(data);
	if(d == NULL)
		return(1);

	if((d->progress_cb != NULL) && ((i % IMG_XPM_PROGRESS_RESOLUTION) == 0))
	{
		if(!d->progress_cb(
			d->progress_data,
			i / 3, m,
			d->width, d->height,
			d->bpl, d->bpp,
			d->data
		))
			*d->user_aborted = 1;
	}

	return(*d->user_aborted);
}
 
/*
 *	Writes the XpmImage and XpmInfo to file.
 */
static int ImgXPMWriteXPMImageToXPMFile(
	const char *path,
	const char *c_id,
	XpmImage *img,
	XpmInfo *info
)
{
	int		i, c,
			len,
			fd;
	char *tmp_xpm_path;
	const char *tmp_dir_path;
	u_int8_t	*io_buf;
	size_t		io_buf_len,
			units_read,
			units_to_write,
			units_written;
	FILE		*in_fp,
			*out_fp;
	struct stat stat_buf;

	/* Get the temporary files directory path */
	tmp_dir_path = getenv(ENV_VAR_NAME_TMPDIR);
        if(tmp_dir_path == NULL)
#if defined(P_tmpdir)
                tmp_dir_path = P_tmpdir;
#elif defined(_WIN32)
                tmp_dir_path = "C:\\TEMP";
#else
                tmp_dir_path = "/tmp";
#endif

	/* Format the a temporary file path for writing the initial
	 * XPM image to using XpmWriteFileFromXpmImage()
	 */
	len = strlen(tmp_dir_path) + strlen("/XXXXXX");
	tmp_xpm_path = (char *)malloc((len + 1) * sizeof(char));
	if(tmp_xpm_path == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		return(-3);
	}
	(void)strcpy(
		tmp_xpm_path,
		tmp_dir_path
	);
	(void)strcat(
		tmp_xpm_path,
		"/XXXXXX"
	);
        fd = mkstemp(tmp_xpm_path);
	if(fd < 0)
	{
		free(tmp_xpm_path);
		imgio_last_save_error = "Unable to create a temporary file";
		return(-3);
	}
	(int)close(fd);

	/* Write a temporary XPM file data from the XpmImage and
	 * XpmInfo
	 *
	 * This is used to work around a bug in
	 * XpmCreateBufferFromXpmImage() that does not allocate
	 * memory correctly
	 */
	switch(XpmWriteFileFromXpmImage(
		tmp_xpm_path,
		img,
		info
	))
	{
	    case XpmColorError:
		free(tmp_xpm_path);
		imgio_last_save_error = "XPM color error";
		return(-1);
		break;
	    case XpmSuccess:
		break;
	    case XpmOpenFailed:
		free(tmp_xpm_path);
		imgio_last_save_error = "Unable to open the XPM file for writing";
		return(-1);
		break;
	    case XpmFileInvalid:
		free(tmp_xpm_path);
		imgio_last_save_error = "Invalid XPM file";
		return(-2);
		break;
	    case XpmNoMemory:
		free(tmp_xpm_path);
		imgio_last_save_error = "Memory allocation error";
		return(-3);
		break;
	    case XpmColorFailed:
		free(tmp_xpm_path);
		imgio_last_save_error = "XPM color failed";
		return(-1);
		break;
	    default:
		free(tmp_xpm_path);
		imgio_last_save_error = "Error writing XPM file";
		return(-1);
		break;
	}

	/* Open the temporary XPM image file that was just written */
	in_fp = fopen(tmp_xpm_path, "rb");
	if(in_fp == NULL)
	{
		imgio_last_save_error = "Error writing XPM file";
		free(tmp_xpm_path);
		return(-1);
	}

	/* Create/truncate the output file */
	out_fp = fopen(path, "wb");
	if(out_fp == NULL)
	{
		imgio_last_save_error = "Error writing XPM file";
		(void)fclose(in_fp);
		free(tmp_xpm_path);
		return(-1);
	}

	/* Get the initial size of the IO buffer */
	io_buf_len = 0l;
	if(!fstat(fileno(in_fp), &stat_buf))
	{
#if !defined(_WIN32)
		io_buf_len = (size_t)stat_buf.st_blksize;
#endif
	}
	if(io_buf_len <= 0l)
		io_buf_len = 4l;

	/* Allocate the IO buffer */
	io_buf = (u_int8_t *)malloc(io_buf_len * sizeof(u_int8_t));
	if(io_buf == NULL)
	{
		(void)fclose(in_fp);
		(void)unlink(tmp_xpm_path);
		free(tmp_xpm_path);
		(void)fclose(out_fp);
		imgio_last_save_error = "Memory allocation error";
		return(-3);
	}

	/* Replace the C ID in the generated buffer with our specified
	 * C ID
	 */
	while(!feof(in_fp))
	{
		/* Read to the IO buffer until a newline or end of
		 * file is reached
		 */
		for(c = fgetc(in_fp), i = 0;
		    c != EOF;
		    c = fgetc(in_fp)
		)
		{
			/* Need to increase the IO buffer allocation? */
			if(i >= io_buf_len)
			{
				io_buf_len = i + 4;
				io_buf = (u_int8_t *)realloc(
					io_buf,
					io_buf_len * sizeof(u_int8_t)
				);
				if(io_buf == NULL)
				{
					(void)fclose(in_fp);
					(void)unlink(tmp_xpm_path);
					free(tmp_xpm_path);
					(void)fclose(out_fp);
					imgio_last_save_error = "Memory allocation error";
					return(-3);
				}
			}

			io_buf[i] = (u_int8_t)c;
			i++;

			/* End of line (the newline character will be
			 * stored)?
			 */
			if((char)c == '\n')
				break;
		}
		units_read = (size_t)i;

		/* Filter */
		if(!strncmp(
			(const char *)io_buf,
			"static char",
			strlen("static char")
		))
		{
			i = strlen(
"static char * [] = {\n"
			) + strlen(c_id) + 1;
			if(i >= io_buf_len)
			{
				io_buf_len = i + 4;
				io_buf = (u_int8_t *)realloc(
					io_buf,
					io_buf_len * sizeof(u_int8_t)
				);
				if(io_buf == NULL)
				{
					(void)fclose(in_fp);
					(void)unlink(tmp_xpm_path);
					free(tmp_xpm_path);
					(void)fclose(out_fp);
					imgio_last_save_error = "Memory allocation error";
					return(-3);
				}
			}
			(void)sprintf(
				(char *)io_buf,
				"static char * %s[] = {\n",
				c_id
			);
			units_read = strlen((const char *)io_buf);
		}

		/* Write the IO buffer to the output file */
		units_to_write = units_read;
		units_written = fwrite(
			io_buf,
			sizeof(u_int8_t),
			units_to_write,
			out_fp
		);
		if(units_written != units_to_write)
		{
			(void)fclose(in_fp);
			(void)unlink(tmp_xpm_path);
		        free(tmp_xpm_path);
			(void)fclose(out_fp);
			imgio_last_save_error = "An error occured while writing the XPM buffer to the file";
			return(-1);
		}
	}

	/* Delete the IO buffer */
	free(io_buf);

	/* Close and remove the temporary file */
	(void)fclose(in_fp);
	(void)unlink(tmp_xpm_path);
	free(tmp_xpm_path);

	/* Close the output file */
	if(fclose(out_fp))
	{
		imgio_last_save_error = "Error writing XPM file";
		return(-1);
	}

	return(0);
}

/*
 *	Writes the image data to an XPM file in color.
 */
static int ImgFileSaveXPMRGBAToColor(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int max_colors,			/* -1 for no color limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		transparent_color_num = -1,
			ncolors = 0,
			_max_colors = max_colors;
	u_int8_t *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;
	RGBAToCIdxData *d;

#define CLEANUP_RETURN(_v_)	{		\
 if(img != NULL) {				\
  XpmFreeXpmImage(img);				\
  free(img);					\
 }						\
 if(info != NULL) {				\
  XpmFreeXpmInfo(info);				\
  free(info);					\
 }						\
 free(cmap);					\
 free(cidx);					\
 return(_v_);					\
}
	/* If specifying no limit on colors, then we still need to
	 * limit it to 255 since RGBAToCIdx() can only reduce to 256
	 */
	if(_max_colors <= 0)
		_max_colors = 255;

	/* Limit the maximum number of colors to 255, so we have room
	 * to add the transparent color which would make it 256
	 * colors afterwards
	 */
	if(_max_colors > 255)
		_max_colors = 255;

	/* Allocate the RGBAToCIdx() callback data */
	d = RGBA_TO_CIDX_DATA(calloc(1, sizeof(RGBAToCIdxData)));
	d->progress_cb = progress_cb;
	d->progress_data = progress_data;
	d->width = width;
	d->height = height;
	d->bpp = bpp;
	d->bpl = bpl;
	d->data = rgba;				/* RGBA */
	d->user_aborted = user_aborted;

	/* Generate the color index image data and the corresponding
	 * colormap
	 */
	RGBAToCIdx(
		rgba,
		width, height,
		bpl,
		_max_colors,
		&cidx,
		&cmap,
		&ncolors,
		ImgXpmRGBAToCIdxCB,
		d
	);

	/* Delete the callback data */
	free(d);

	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Apply transparency to the color index image data */
	if(1)
	{
		int i = ncolors;

		/* Insert the transparent color at the beginning of the
		 * colormap
		 */
		ncolors = i + 1;
		cmap = (ColormapColor *)realloc(
			cmap,
			ncolors * sizeof(ColormapColor)
		);
		if(cmap != NULL)
		{
			int y;
			const u_int8_t *rgba_ptr;
			u_int8_t	*cidx_ptr,
					*cidx_end;

			transparent_color_num = 0;

			/* Shift colors on the colormap */
			for(i = ncolors - 1; i > 0; i--)
				(void)memcpy(
					&cmap[i],
					&cmap[i - 1],
					sizeof(ColormapColor)
				);

			/* Increment each color value on the color index image
			 * data and set all transparent colors
			 */
			for(y = 0; y < height; y++)
			{
				rgba_ptr = rgba + (y * bpl);
				cidx_ptr = cidx + (y * width);
				cidx_end = cidx_ptr + width;
				while(cidx_ptr < cidx_end)
				{
					if(rgba_ptr[3] < threshold)
						*cidx_ptr = (u_int8_t)transparent_color_num;
					else
						*cidx_ptr = (*cidx_ptr) + 1;
					rgba_ptr += bpp;
					cidx_ptr++;
				}
			}
		}
		else
		{
			ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	info->valuemask = XpmHotspot;
	info->x_hotspot = x;
	info->y_hotspot = y;
	if(!STRISEMPTY(comments))
	{
#if defined(XpmComments)
/* Comments support has been dropped since libXpm 3.0c,
 * but we support it only if its still there */
		info->valuemask |= XpmComments;
		info->hints_cmt = strdup(comments);
#endif	/* XpmComments */
	}
	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
		imgio_last_save_error = "Memory allocation error"; 
		CLEANUP_RETURN(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
		/* Calculate the number of ACSII characters needed to
		 * represent one pixel
		 */
		unsigned int m = XPM_ASCII_CHAR_RANGE;
		for(img->cpp = 1;
		    m < ncolors;
		    img->cpp++, m *= XPM_ASCII_CHAR_RANGE
		);
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
		const int	po = ncolors,
				pm = 3 * ncolors;
		const unsigned int cpp = img->cpp;
		int i;
		char	*s,
			*c_string = (char *)malloc((cpp + 1) * sizeof(char));
		const ColormapColor *src_c;
		XpmColor *tar_c;

		/* Reset the color string */
		(void)memset(
			c_string,
			XPM_ASCII_CHAR_MIN,
			(size_t)cpp
		);
		c_string[cpp] = '\0';

		/* Allocate the XpmColors on the XPM colormap */
		img->ncolors = ncolors;
		img->colorTable = (XpmColor *)calloc(
			img->ncolors,
			sizeof(XpmColor)
		);
		if(img->colorTable == NULL)
		{
			free(c_string);
			img->ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}

		/* Iterate through the colormap and create the XpmColors */
		for(i = 0; i < ncolors; i++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((i % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + i, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{ 
					*user_aborted = 1;
					break;
				}
			}

			/* Set this colormap color to the XPM colormap */
			src_c = &cmap[i];
			tar_c = &img->colorTable[i];

			/* Set the XpmColor's ASCII characters string that
			 * represent this color
			 */
			tar_c->string = strdup(c_string);

			/* Set the XpmColor's color value string
			 *
			 * Check for symbolic colors
			 */
			if(i == transparent_color_num)
			{
				tar_c->c_color = strdup("None");
			}
			else
			{
				tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
				(void)sprintf(
					s,
					"#%.2X%.2X%.2X",
					src_c->r, src_c->g, src_c->b
				);
			}

			/* Increment characters in the color string */
			for(s = c_string; *s != '\0'; s++)
			{
				if(*s < XPM_ASCII_CHAR_MAX)
				{
					*s = (*s) + 1;
					break;
				}
				else
				{
					*s = XPM_ASCII_CHAR_MIN;
				}
			}
		}

		free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Allocate the XPM image data */
	img->data = (unsigned int *)malloc(
		width * height * sizeof(unsigned int)
	);
	if(img->data == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}

	/* Copy/convert the color index image data to the XPM color
	 * index image data
	 */
	if((img->data != NULL) && (cidx != NULL))
	{
		const int	po = 2 * height,
				pm = 3 * height;
		int y;
		unsigned int *tar_ptr;
		const u_int8_t *src_ptr, *src_end;

		for(y = 0; y < height; y++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((y % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + y, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{ 
					*user_aborted = 1;
					break;
				}
			}

			/* Copy/convert this line */
			src_ptr = cidx + (y * width);
			src_end = src_ptr + width;
			tar_ptr = img->data + (y * width);
			while(src_ptr < src_end)
				*tar_ptr++ = (unsigned int)*src_ptr++;
		}
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteXPMImageToXPMFile(
		path,
		c_id,
		img,
		info
	))
	{
		CLEANUP_RETURN(-1);
	}
	else
	{
		CLEANUP_RETURN(0);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the image data to an XPM file in greyscale.
 */
static int ImgFileSaveXPMRGBAToGreyscale(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		ncolors = 0,
			transparent_color_num = -1;
	unsigned int *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(img != NULL) {				\
  XpmFreeXpmImage(img);				\
  free(img);					\
 }						\
 if(info != NULL) {				\
  XpmFreeXpmInfo(info);				\
  free(info);					\
 }						\
 free(cidx);					\
 free(cmap);					\
 return(_v_);					\
}
 
	/* Allocate the color index image data */
	cidx = (unsigned int *)realloc(
		cidx,
		width * height * sizeof(unsigned int)
	);
	if(cidx == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}

	/* Create the colormap and the color index image data from the
	 * RGBA image data
	 *
	 * Note that we are ignoring max_colors, as many as 257 colors
	 * may be generated (256 + transparent color).
	 */
	if(cidx != NULL)
	{
		const int	po = 0,
				pm = 2 * height;
		int	i,
			y;
		u_int32_t gamma;
		unsigned int *cidx_ptr;
		const u_int8_t	*rgba_ptr,
				*rgba_end;
		ColormapColor	tc,
				*c;

		/* Append the transparent color */
		i = ncolors;
		ncolors = i + 1;
		cmap = (ColormapColor *)realloc(
			cmap, ncolors * sizeof(ColormapColor)
		);
		if(cmap == NULL)
		{
			ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}
		c = &cmap[i];
		c->r = 0x00;
		c->g = 0x00;
		c->b = 0x00;
		transparent_color_num = i;

		/* Iterate through each line */
		for(y = 0; y < height; y++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((y % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + y, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{
					*user_aborted = 1;
					break;
				}
			}

			/* Iterate through this line */
			cidx_ptr = cidx + (y * width);
			rgba_ptr = rgba + (y * bpl);
			rgba_end = rgba_ptr + (width * bpp);
			while(rgba_ptr < rgba_end)
			{
				/* Skip transparent pixels */
				if(rgba_ptr[3] < threshold)
				{
					*cidx_ptr++ = (unsigned int)transparent_color_num;
					rgba_ptr += bpp;
					continue;
				}

				/* Get the grey value of the current RGBA pixel */
				gamma = ((u_int32_t)rgba_ptr[0] +
					 (u_int32_t)rgba_ptr[1] +
					 (u_int32_t)rgba_ptr[2]
					) / 3;
				tc.r = tc.g = tc.b = (u_int8_t)gamma;

				/* Check if the color exists in the colormap */
				for(i = 0; i < ncolors; i++)
				{
					/* Skip transparent color */
					if(i == transparent_color_num)
						continue;

					c = &cmap[i];
					if((c->r == tc.r) &&
					   (c->g == tc.g) &&
					   (c->b == tc.b)
					)
						break;
				}
				if(i >= ncolors)
				{
					/* Need to append this color */
					const int n = ncolors;
					ncolors = n + 1;
					cmap = (ColormapColor *)realloc(
						cmap,
						ncolors * sizeof(ColormapColor)
					);
					if(cmap != NULL)
					{
						c = &cmap[n];
						c->r = tc.r;
						c->g = tc.g;
						c->b = tc.b;
					}
					else
					{
						ncolors = 0;
						imgio_last_save_error =
							"Memory allocation error";
						CLEANUP_RETURN(-3);
					}
					*cidx_ptr++ = (unsigned int)n;
					rgba_ptr += bpp;
				}
				else
				{
					*cidx_ptr++ = (unsigned int)i;
					rgba_ptr += bpp;
				}
			}
		}
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	info->valuemask = XpmHotspot;
	info->x_hotspot = x;
	info->y_hotspot = y;
	if(!STRISEMPTY(comments))
	{
#if defined(XpmComments)
/* Comments support has been dropped since libXpm 3.0c,
 * but we support it only if its still there */
		info->valuemask |= XpmComments;
		info->hints_cmt = strdup(comments);
#endif	/* XpmComments */
	}

	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
		unsigned int m = XPM_ASCII_CHAR_RANGE;
		for(img->cpp = 1;
			m < ncolors;
			img->cpp++, m *= XPM_ASCII_CHAR_RANGE
		);
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
		const int	po = ncolors,
				pm = 2 * ncolors;
		const unsigned int cpp = img->cpp;
		int i;
		char	*s,
			*c_string = (char *)malloc((cpp + 1) * sizeof(char));
		const ColormapColor *src_c;
		XpmColor *tar_c;

		/* Reset the color string */
		(void)memset(
			c_string,
			XPM_ASCII_CHAR_MIN,
			(size_t)cpp
		);
		c_string[cpp] = '\0';

		/* Allocate XpmColors on the XPM colormap */
		img->ncolors = ncolors;
		img->colorTable = (XpmColor *)calloc(
			img->ncolors, sizeof(XpmColor)
		);
		if(img->colorTable == NULL)
		{
			free(c_string);
			img->ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}

		/* Iterate through the colormap and create the XpmColors */
		for(i = 0; i < ncolors; i++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((i % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + i, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{ 
					*user_aborted = 1;
					break;
				}
			}

			/* Set this colormap color to the XPM colormap */
			src_c = &cmap[i];
			tar_c = &img->colorTable[i];

			/* Set the XpmColor's ASCII characters string that
			 * represent this color
			 */
			tar_c->string = STRDUP(c_string);

			/* Set the XpmColor's color value string
			 *
			 * Check for symbolic colors
			 */
			if(i == transparent_color_num)
			{
				tar_c->c_color = strdup("None");
			}
			else
			{
				tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
				(void)sprintf(
					s,
					"#%.2X%.2X%.2X",
					src_c->r, src_c->g, src_c->b
				);
			}

			/* Increment characters in the color string */
			for(s = c_string; *s != '\0'; s++)
			{
				if(*s < XPM_ASCII_CHAR_MAX)
				{
					*s = (*s) + 1;
					break;
				}
				else
				{
					*s = XPM_ASCII_CHAR_MIN;
				}
			}
		}

		free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Set the color index image data as the XPM image data */
	img->data = cidx;
	cidx = NULL;

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteXPMImageToXPMFile(
		path,
		c_id,
		img,
		info
	))
	{
		CLEANUP_RETURN(-1);
	}
	else
	{
		CLEANUP_RETURN(0);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the image data to an XPM file in black & white.
 */
static int ImgFileSaveXPMRGBAToBW(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments, 
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		ncolors = 0,
			transparent_color_num = -1;
	unsigned int *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(img != NULL) {				\
  XpmFreeXpmImage(img);				\
  free(img);					\
 }						\
 if(info != NULL) {				\
  XpmFreeXpmInfo(info);				\
  free(info);					\
 }						\
 free(cidx);					\
 free(cmap);					\
 return(_v_);					\
}
 
	/* Allocate the color index image data */
	cidx = (unsigned int *)realloc(
		cidx,
		width * height * sizeof(unsigned int)
	);
	if(cidx == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}

	/* Create the colormap and the color index image data from the
	 * RGBA image data
	 *
	 * Note that we are ignoring max_colors, as many as 257 colors
	 * may be generated (256 + transparent color).
	 */
	if(cidx != NULL)
	{
		const int	po = 0,
				pm = 2 * height;
		int	i,
			y;
		u_int32_t gamma;
		unsigned int *cidx_ptr;
		const u_int8_t	*rgba_ptr,
				*rgba_end;
		ColormapColor	tc,
				*c;

		/* Append the transparent color */
		i = ncolors;
		ncolors = i + 1;
		cmap = (ColormapColor *)realloc(
			cmap, ncolors * sizeof(ColormapColor)
		);
		if(cmap == NULL)
		{
			ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}
		c = &cmap[i];
		c->r = 0x00;
		c->g = 0x00;
		c->b = 0x00;
		transparent_color_num = i;

		/* Iterate through each line */
		for(y = 0; y < height; y++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((y % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + y, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{
					*user_aborted = 1;
					break;
				}
			}

			/* Iterate through this line */
			cidx_ptr = cidx + (y * width);
			rgba_ptr = rgba + (y * bpl);
			rgba_end = rgba_ptr + (width * bpp);
			while(rgba_ptr < rgba_end)
			{
				/* Skip transparent pixels */
				if(rgba_ptr[3] < threshold)
				{
					*cidx_ptr++ = (unsigned int)transparent_color_num;
					rgba_ptr += bpp;
					continue;
				}

				/* Get the grey value of the current RGBA pixel */
				gamma = ((u_int32_t)rgba_ptr[0] +
					 (u_int32_t)rgba_ptr[1] +
					 (u_int32_t)rgba_ptr[2]
					) / 3;
				tc.r = tc.g = tc.b = ((u_int8_t)gamma < 0x80) ?
					0x00 : 0xff;

				/* Check if the color exists in the colormap */
				for(i = 0; i < ncolors; i++)
				{
					/* Skip transparent color */
					if(i == transparent_color_num)
						continue;

					c = &cmap[i];
					if((c->r == tc.r) &&
					   (c->g == tc.g) &&
					   (c->b == tc.b)
					)
						break;
				}
				if(i >= ncolors)
				{
					/* Need to append this color */
					const int n = ncolors;
					ncolors = n + 1;
					cmap = (ColormapColor *)realloc(
						cmap,
						ncolors * sizeof(ColormapColor)
					);
					if(cmap != NULL)
					{
						c = &cmap[n];
						c->r = tc.r;
						c->g = tc.g;
						c->b = tc.b;
					}
					else
					{
						ncolors = 0;
						imgio_last_save_error = "Memory allocation error";
						CLEANUP_RETURN(-3);
					}
					*cidx_ptr++ = (unsigned int)n;
					rgba_ptr += bpp;
				}
				else
				{
					*cidx_ptr++ = (unsigned int)i;
					rgba_ptr += bpp;
				}
			}
		}
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	info->valuemask = XpmHotspot;
	info->x_hotspot = x;
	info->y_hotspot = y;
	if(!STRISEMPTY(comments))
	{
#if defined(XpmComments)
/* Comments support has been dropped since libXpm 3.0c,
 * but we support it only if its still there */
		info->valuemask |= XpmComments;
		info->hints_cmt = strdup(comments);
#endif	/* XpmComments */
	}

	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
		unsigned int m = XPM_ASCII_CHAR_RANGE;
		for(img->cpp = 1;
			m < ncolors;
			img->cpp++, m *= XPM_ASCII_CHAR_RANGE
		);
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
		const int	po = ncolors,
				pm = 2 * ncolors;
		const unsigned int cpp = img->cpp;
		int i;
		char	*s,
			*c_string = (char *)malloc((cpp + 1) * sizeof(char));
		const ColormapColor *src_c;
		XpmColor *tar_c;

		/* Reset color string */
		(void)memset(
			c_string,
			XPM_ASCII_CHAR_MIN,
			(size_t)cpp
		);
		c_string[cpp] = '\0';

		/* Allocate XpmColors on the XPM colormap */
		img->ncolors = ncolors;
		img->colorTable = (XpmColor *)calloc(
			img->ncolors, sizeof(XpmColor)
		);
		if(img->colorTable == NULL)
		{
			free(c_string);
			img->ncolors = 0;
			imgio_last_save_error = "Memory allocation error";
			CLEANUP_RETURN(-3);
		}

		/* Iterate through the colormap and create the XpmColors */
		for(i = 0; i < ncolors; i++)
		{
			/* Report progress */
			if((progress_cb != NULL) && ((i % IMG_XPM_PROGRESS_RESOLUTION) == 0))
			{
				if(!progress_cb(
					progress_data,
					po + i, pm,
					width, height,
					bpl, bpp,
					rgba
				))
				{ 
					*user_aborted = 1;
					break;
				}
			}

			/* Set this colormap color to the XPM colormap */
			src_c = &cmap[i];
			tar_c = &img->colorTable[i];

			/* Set the XpmColor's ASCII characters string that
			 * represent this color
			 */
			tar_c->string = strdup(c_string);

			/* Set the XpmColor's color value string
			 *
			 * Check for symbolic colors
			 */
			if(i == transparent_color_num)
			{
				tar_c->c_color = strdup("None");
			}
			else
			{
				tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
				(void)sprintf(
					s,
					"#%.2X%.2X%.2X",
					src_c->r, src_c->g, src_c->b
				);
			}

			/* Increment characters in the color string */
			for(s = c_string; *s != '\0'; s++)
			{
				if(*s < XPM_ASCII_CHAR_MAX)
				{
					*s = (*s) + 1;
					break;
				}
				else
				{
					*s = XPM_ASCII_CHAR_MIN;
				}
			}
		}

		free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
		CLEANUP_RETURN(-4);
	}

	/* Set the color index image data as the XPM image data */
	img->data = cidx;
	cidx = NULL;

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteXPMImageToXPMFile(
		path,
		c_id,
		img,
		info
	))
	{
		CLEANUP_RETURN(-1);
	}
	else
	{
		CLEANUP_RETURN(0);
	}
#undef CLEANUP_RETURN
}

/*
 *	Saves the RGBA image data to an XPM image file.
 */
int ImgFileSaveXPMRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale    
						 * 2 = Color */
	const int max_colors,			/* -1 for no color limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
			user_aborted = 0,
			_bpl = bpl;
	char		*s,
			*new_c_id;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((path == NULL) || (rgba == NULL) ||
	   (width <= 0) || (height <= 0)
	)
	{
		imgio_last_save_error = "Invalid value used to describe image";
		return(-1); 
	}

	/* Automatically calculate bytes per line? */
	if(_bpl <= 0)
		_bpl = width * bpp;

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgFileSaveXPMRGBA(): Writing XPM file \"%s\"...\n",
 path
);
#endif

	/* Report initial progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			progress_data,
			0, height,
			width, height,
			_bpl, bpp,
			rgba
		))
			user_aborted = 1;
	}

	/* Generate the C ID as needed */
	if(STRISEMPTY(c_id))
	{
		const char *s = strrchr(path, '/');
		new_c_id = (s != NULL) ? strdup(s + 1) : strdup(path);
	}
	else
	{
		new_c_id = strdup(c_id);
	}
	/* Replace any unacceptable characters in the C ID */
	for(s = new_c_id; *s != '\0'; s++)
	{
		if(!isalnum(*s))
			*s = '_';
	}

	/* Replace any unacceptable characters in the comments */
	if(!STRISEMPTY(comments))
	{
/* TODO */

	}

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgFileSaveXPMRGBA(): C ID=\"%s\"\n",
 new_c_id
);
#endif

	/* Save by the color type */
	status = -2;
	switch(color_type)
	{
	    case 0:				/* Black & white */
		status = ImgFileSaveXPMRGBAToBW(
			path,
			width, height,
			_bpl,
			rgba,
			bg_color,
			x, y,
			base_width, base_height,
			new_c_id, comments,
			max_colors, threshold,
			progress_cb, progress_data,
			&user_aborted
		);
		break;

	    case 1:				/* Greyscale */
		status = ImgFileSaveXPMRGBAToGreyscale(
			path,
			width, height,
			_bpl,
			rgba,
			bg_color,
			x, y,
			base_width, base_height,
			new_c_id, comments,
			max_colors, threshold,
			progress_cb, progress_data,
			&user_aborted
		);
		break;

	    case 2:				/* Color */
		status = ImgFileSaveXPMRGBAToColor(
			path,
			width, height,
			_bpl,
			rgba,
			bg_color,
			x, y,
			base_width, base_height,
			new_c_id, comments,
			max_colors, threshold,
			progress_cb, progress_data,
			&user_aborted
		);
		break;

	    default:
		imgio_last_save_error = "Unsupported XPM color format";
		status = -2;
		break;
	}

	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
		if(!progress_cb(
			progress_data,
			height, height,
			width, height,
			_bpl, bpp,
			rgba
		))
			user_aborted = 1;
	}

	free(new_c_id);

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgFileSaveXPMRGBA(): Done.\n"
);
#endif

	if(user_aborted)
	{
		imgio_last_save_error = "User aborted operation";
		return(-4);
	}
	else
		return(status);
}
#endif	/* HAVE_LIBXPM */
