#ifdef HAVE_LIBTIFF
#include <stdarg.h>				/* va_list */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>				/* isdigit() */
#include <tiffio.h>				/* libtiff */
#include "imgio.h"
#include "config.h"


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;

typedef struct _ImgTIFFReadContext	ImgTIFFReadContext;
#define IMG_TIFF_READ_CONTEXT(p)	((ImgTIFFReadContext *)p)
typedef struct _ImgTIFFWriteContext	ImgTIFFWriteContext;
#define IMG_TIFF_WRITE_CONTEXT(p)	((ImgTIFFWriteContext *)p)


/* Version */
void ImgTIFFVersion(int *major, int *minor, int *release);

/* Check Type */
int ImgBufferIsTIFF(const u_int8_t *data, const int len);
int ImgStreamIsTIFF(FILE *fp);
int ImgFileIsTIFF(const char *path);

/* Open */
static void ImgReadTIFFWarningCB(
	const char *module,
	const char *fmt,
	va_list ap
);
static void ImgReadTIFFErrorCB(
	const char *module,
	const char *fmt,
	va_list ap
);
static int ImgReadTIFFReportProgress(
	ImgTIFFReadContext *ctx,
	const int i, const int n
);
static void ImgReadTIFFInformation(ImgTIFFReadContext *ctx);
static int ImgReadTIFFRGBAGreyscaleAlphaIterate(ImgTIFFReadContext *ctx);
static int ImgReadTIFFRGBAIterate(ImgTIFFReadContext *ctx);
static int ImgReadTIFFRGBA(ImgTIFFReadContext *ctx);
int ImgStreamReadTIFFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenTIFFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgReadTIFFRGBAThumbGreyscaleAlphaIterate(ImgTIFFReadContext *ctx);
static int ImgReadTIFFRGBAThumbIterate(ImgTIFFReadContext *ctx);
static int ImgReadTIFFRGBAThumb(ImgTIFFReadContext *ctx);
int ImgStreamReadTIFFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenTIFFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);

/* Save */
static void ImgWriteTIFFWarningCB(
	const char *module,
	const char *fmt,
	va_list ap
);
static void ImgWriteTIFFErrorCB(
	const char *module,
	const char *fmt,
	va_list ap
);
static int ImgWriteTIFFReportProgress(
	ImgTIFFWriteContext *ctx,
	const int i, const int n,
	const int frame_num
);
static int ImgWriteTIFFHeader(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBAToBW(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBAToGreyscale(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBAToGreyscaleAlpha(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBAToRGB(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBAToRGBA(
	ImgTIFFWriteContext *ctx,
	const int frame_num
);
static int ImgWriteTIFFRGBA(ImgTIFFWriteContext *ctx);
int ImgStreamWriteTIFFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,                   /* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveTIFFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,                   /* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_TIFF_PROGRESS_RESOLUTION	10

/*
 *	Default Frame Delay (in milliseconds):
 *
 *	Since tiff images do not store frame delays, we need to set
 *	a default delay when reading images with multiple frames.
 */
#define IMG_TIFF_DEF_FRAME_DELAY_MS	1000

/*
 *	Default Rows Per Strip Estimate:
 *
 *	Used when writing TIFF images by TIFFDefaultStripSize() to
 *	calculate the desired value the TIFFTAG_ROWSPERSTRIP tag.
 */
#define IMG_TIFF_DEF_ROWS_PER_STRIP_ESTIMATE	\
					8


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)


/*
 *	Read Context:
 */
struct _ImgTIFFReadContext {
	TIFF		*tp;

	tdir_t		tiff_ndirectories;	/* Total number of frames */
	uint32		tiff_bits_per_sample,
			tiff_samples_per_pixel,
			tiff_planar_config,
			tiff_rows_per_strip,
			tiff_width, tiff_height,
			tiff_x, tiff_y,
			tiff_base_width, tiff_base_height;

	int		req_width, req_height,	/* Used only when reading as
						 * a thumb */
			nframes,		/* Number of frames read */
			data_width, data_height,/* Description of data_list */
			data_bpp, data_bpl;
	u_int8_t	**data_list;		/* Read RGBA frames list */
	unsigned long   *delay_ms_list;		/* Read delay list in milliseconds */

	u_int8_t	def_alpha_value;

	u_int8_t	*bg_color;		/* 4 bytes RGBA */

	char		**creator,
			**title,
			**author,
			**comments;

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		cur_progress_level,
			nprogress_levels;
	int		*user_aborted;
	int		status;

};

/*
 *	Write Context:
 */
struct _ImgTIFFWriteContext {
	TIFF		*tp;

	int		nframes,		/* Number of frames to write */
			data_width, data_height,/* Description of data_list */
			data_bpp, data_bpl;
	const u_int8_t	**data_list;		/* RGBA frames list to write */
	const unsigned long   *delay_ms_list;	/* Delay list to write */

	const u_int8_t	*bg_color;		/* 4 bytes RGBA */

	int		x, y,
			base_width, base_height;

	const char	*creator,
			*title,
			*author,
			*comments;

	int		compression_type;	/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	float		jpeg_quality;		/* 0.0 to 1.0 */
	int		color_type;		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	int		rows_per_strip;

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		cur_progress_level,
			nprogress_levels;
	int		*user_aborted;
	int		status;

};


/*
 *	Gets the TIFF library's version.
 */
void ImgTIFFVersion(int *major, int *minor, int *release)
{
	/* Get the version string, format "SOME WORDS M.m.r" */
	const char *s = TIFFGetVersion();
	if(s == NULL)
	{
		if(major != NULL)
			*major = 0;
		if(minor != NULL)
			*minor = 0;
		if(release != NULL)
			*release = 0;
		return;
	}

	/* Seek s in the version to the first numeric character */
	while((*s != '\0') && (*s != '\n'))
	{
		if(isdigit(*s))
			break;
		s++;
	}
	if(major != NULL)
		*major = atoi(s);

	while((*s != '\0') && (*s != '\n'))
	{
		if(*s == '.')
		{
			s++;
			break;
		}
		s++;
	}
	if(minor != NULL)
		*minor = atoi(s);

	while((*s != '\0') && (*s != '\n'))
	{
		if(*s == '.')
		{
			s++;
			break;
		}
		s++;
	}
	if(release != NULL)
		*release = atoi(s);
}


/* 
 *	Checks for the TIFF type ID.
 */
int ImgBufferIsTIFF(const u_int8_t *data, const int len)
{
	if((data == NULL) || (len < 8))
	{
	    errno = EINVAL;
	    return(0);
	}

	/* Intel TIFF header */
	if((data[0] == 0x49) &&
	   (data[1] == 0x49) &&
	   (data[2] == 0x2A) &&
	   (data[3] == 0x00)
	)
	    return(1);

	/* Macintosh TIFF header */
	if((data[0] == 0x4D) &&
	   (data[1] == 0x4D) &&
	   (data[2] == 0x00) &&
	   (data[3] == 0x2A)
	)
	    return(1);

	errno = EINVAL;
	return(0);
}

int ImgStreamIsTIFF(FILE *fp)
{
	u_int8_t buf[8];
	size_t units_read;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(0);
	}

	units_read = fread(
		buf,
		sizeof(u_int8_t),
		sizeof(buf) / sizeof(u_int8_t),
		fp
	);
	if(units_read == (sizeof(buf) / sizeof(u_int8_t)))
		return(ImgBufferIsTIFF(buf, sizeof(buf)));
	else
		return(0);
}

int ImgFileIsTIFF(const char *path)
{
	int		status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
		return(0);

	status = ImgStreamIsTIFF(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	Warning callback.
 */
static void ImgReadTIFFWarningCB(
	const char *module,
	const char *fmt,
	va_list ap
)
{
	/* Do nothing */
}

/*
 *	Error callback.
 */
static void ImgReadTIFFErrorCB(
	const char *module,
	const char *fmt,
	va_list ap
)
{
	int		len,
			ntokens = 0;
	char *msg;
	const char *s;

	for(s = fmt; *s != '\0'; s++)
	{
		if(*s == '%')
			ntokens++;
	}

	len = strlen(fmt) + (ntokens * 1024);
	msg = (char *)malloc((len + 1) * sizeof(char));
	if(msg != NULL)
	{
		(void)vsprintf(
			msg,
			fmt,
			ap
		);
		ImgOpenSetError(msg);
		free(msg);
	}
}

/*
 *	Reports the read progress on the ImgTIFFReadContext and checks
 *	for and handles any user abort.
 *
 *	Returns 0 to indicate continue or -4 to abort.
 */
static int ImgReadTIFFReportProgress(
	ImgTIFFReadContext *ctx,
	const int i, const int n
)
{
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			(ctx->cur_progress_level * n) + i,
			ctx->nprogress_levels * n,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			(ctx->nframes > 0) ?
				ctx->data_list[ctx->nframes - 1] : NULL
		))
		{
			*ctx->user_aborted = 1;
			if(imgio_last_open_error == NULL)
				imgio_last_open_error = "User aborted operation";
			errno = EINTR;
			return(-4);
		}
	}

	return(0);
}

/*
 *	Reads the TIFF image information and stores it on the
 *	ImgTIFFReadContext.
 */
static void ImgReadTIFFInformation(ImgTIFFReadContext *ctx)
{
	TIFF *tp = ctx->tp;

	/* Format & Data Type */
	(void)TIFFGetField(
		tp,
		TIFFTAG_BITSPERSAMPLE,
		&ctx->tiff_bits_per_sample
	);
	(void)TIFFGetField(
		tp,
		TIFFTAG_SAMPLESPERPIXEL,
		&ctx->tiff_samples_per_pixel
	);
	(void)TIFFGetField(
		tp,
		TIFFTAG_PLANARCONFIG,
		&ctx->tiff_planar_config
	);
	(void)TIFFGetField(
		tp,
		TIFFTAG_ROWSPERSTRIP,
		&ctx->tiff_rows_per_strip
	);

	/* Size */
	(void)TIFFGetField(tp, TIFFTAG_IMAGEWIDTH, &ctx->tiff_width);
	(void)TIFFGetField(tp, TIFFTAG_IMAGELENGTH, &ctx->tiff_height);

#ifdef TIFFTAG_XPOSITION
	(void)TIFFGetField(tp, TIFFTAG_XPOSITION, &ctx->tiff_x);
#endif
#ifdef TIFFTAG_YPOSITION
	(void)TIFFGetField(tp, TIFFTAG_YPOSITION, &ctx->tiff_y);
#endif
#ifdef TIFFTAG_PIXAR_IMAGEFULLWIDTH
	(void)TIFFGetField(tp, TIFFTAG_PIXAR_IMAGEFULLWIDTH, &ctx->tiff_base_width);
#endif
#ifdef TIFFTAG_PIXAR_IMAGEFULLLENGTH
	(void)TIFFGetField(tp, TIFFTAG_PIXAR_IMAGEFULLLENGTH, &ctx->tiff_base_height);
#endif

	ctx->tiff_ndirectories = TIFFNumberOfDirectories(tp);

	/* Get the creator, title, author, and comments */
#ifdef TIFFTAG_SOFTWARE
	if(ctx->creator != NULL)
	{
		const char *cs;
		if(TIFFGetField(tp, TIFFTAG_SOFTWARE, &cs))
		{
			free(*ctx->creator);
			*ctx->creator = strdup(cs);
		}
	}
#endif
#ifdef TIFFTAG_DOCUMENTNAME
	if(ctx->title != NULL)
	{
		const char *cs;
		if(TIFFGetField(tp, TIFFTAG_DOCUMENTNAME, &cs))
		{
			free(*ctx->title);
			*ctx->title = strdup(cs);
		}
	}
#endif
#ifdef TIFFTAG_ARTIST
	if(ctx->author != NULL)
	{
		const char *cs;
		if(TIFFGetField(tp, TIFFTAG_ARTIST, &cs))
		{
			free(*ctx->author);
			*ctx->author = strdup(cs);
		}
	}
#endif
#ifdef TIFFTAG_IMAGEDESCRIPTION
	if(ctx->comments != NULL)
	{
		const char *cs;
		if(TIFFGetField(tp, TIFFTAG_IMAGEDESCRIPTION, &cs))
		{
			free(*ctx->comments);
			*ctx->comments = strdup(cs);
		}
	}
#endif
}

/*
 *	Special handler for reading TIFF data in Greyscale Alpha.
 */
static int ImgReadTIFFRGBAGreyscaleAlphaIterate(ImgTIFFReadContext *ctx)
{
	int frame_num;
	u_int32_t *tar;
	TIFF *tp = ctx->tp;

	/* Append/allocate a new frame on the ImgTIFFReadContext */
	frame_num = ctx->nframes;
	ctx->nframes++;
	ctx->data_list = (u_int8_t **)realloc(
		ctx->data_list,
		ctx->nframes * sizeof(u_int8_t *)
	);
	if(ctx->data_list == NULL)
	{
		ctx->nframes = 0;
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
	ctx->data_list[frame_num] = (u_int8_t *)malloc(
		ctx->data_height * ctx->data_bpl * sizeof(u_int8_t)
	);

	/* Append the delay of this frame to the delay list */
	ctx->delay_ms_list = (unsigned long *)realloc(
		ctx->delay_ms_list,
		ctx->nframes * sizeof(unsigned long)
	);
	if(ctx->delay_ms_list == NULL)
	{
		int i;
		for(i = 0; i < ctx->nframes; i++)
			free(ctx->data_list[i]);
		free(ctx->data_list);
		ctx->data_list = NULL;
		ctx->nframes = 0;
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
/* TODO: when tiff images support frame delay times */
	ctx->delay_ms_list[frame_num] = IMG_TIFF_DEF_FRAME_DELAY_MS;

	tar = (u_int32_t *)ctx->data_list[frame_num];
	if(tar != NULL)
	{
		if(ctx->tiff_planar_config == PLANARCONFIG_CONTIG)
		{
			const tsize_t tiff_bpl = TIFFScanlineSize(tp);
			const int	sbpp = 2,	/* Grayscale Alpha */
					sbpl = tiff_bpl,
					tbpp = ctx->data_bpp,
					tbpl = ctx->data_bpl;
			u_int8_t *tiff_row_buf = (tiff_bpl > 0l) ?
				(u_int8_t *)_TIFFmalloc(tiff_bpl * sizeof(u_int8_t)) : NULL;
			if(tiff_row_buf != NULL)
			{
				const int	width = ctx->data_width,
						height = ctx->data_height;
				int y;
				u_int8_t	*tar_ptr,
						*tar_end;
				const u_int8_t	*src_ptr,
						*src_end;

				for(y = 0; y < height; y++)
				{
					/* Report the progress */
					if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
					{
						if(ImgReadTIFFReportProgress(ctx, y, height))
							break;
					}

					/* Read the next row */
					if(TIFFReadScanline(
						tp,
						tiff_row_buf,
						y,
						0	/* Sample bit (ignored for
							 * PLANARCONFIG_CONTIG) */
					) == -1)
						continue;

					/* Copy/convert this Greyscale Alpha
					 * row to our RGBA frame data
					 */
					for(src_ptr = tiff_row_buf,
					    src_end = src_ptr + sbpl,
					    tar_ptr = (u_int8_t *)tar + (y * tbpl),
					    tar_end = tar_ptr + (width * tbpp);
					    (src_ptr < src_end) && (tar_ptr < tar_end);
					    src_ptr += sbpp,
					    tar_ptr += tbpp
					)
					{
						tar_ptr[0] = src_ptr[0];
						tar_ptr[1] = src_ptr[0];
						tar_ptr[2] = src_ptr[0];
						tar_ptr[3] = src_ptr[1];
					}
				}
				_TIFFfree(tiff_row_buf);
			}
		}
		else
		{
			if(imgio_last_open_error == NULL)
				imgio_last_open_error = "Unsupported planar configuration for Greyscale Alpha";
			ctx->status = -2;
			errno = EINVAL;
		}
	}

	return(ctx->status);
}

/*
 *	Reads the next frame/directory/subfile in the TIFF stream on
 *	the ImgTIFFReadContext.
 */
static int ImgReadTIFFRGBAIterate(ImgTIFFReadContext *ctx)
{
	int frame_num;
	char emsg[1024];
	u_int32_t *tar;
	uint32 *tiff_rgba;
	TIFF *tp = ctx->tp;
	TIFFRGBAImage tiff_frame;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Check for any errors on this TIFF frame and if it can be
	 * processed as RGBA data
	 */
	if(!TIFFRGBAImageOK(tp, emsg))
	{
		/* Check if the data is in Greyscale Alpha format */
		if((ctx->tiff_bits_per_sample == 8) &&
		   (ctx->tiff_samples_per_pixel == 2)
		)
		{
			uint16	ntags,
				*tags_list;
			if(TIFFGetField(
	                        tp,
				TIFFTAG_EXTRASAMPLES,
				&ntags, &tags_list
			))
			{
				if(ntags >= 1)
				{
					if(tags_list[0] == EXTRASAMPLE_ASSOCALPHA)
					{
						return(ImgReadTIFFRGBAGreyscaleAlphaIterate(ctx));
					}
				}
			}
		}
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Encountered a TIFF frame that could not be processed as RGBA data";
		errno = EINVAL;
		return(-2);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Begin reading this TIFF frame and construct a TIFF decoder
	 * state block
	 */
	if(!TIFFRGBAImageBegin(
		&tiff_frame,
		tp,
		0,                              /* Do not stop on errors */
		emsg
	))
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to construct a TIFF decoder state block";
		return(-1);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Allocate the RGBA data buffer for this TIFF frame that
	 * TIFFRGBAImageGet() will read the TIFF frame data to
	 */
	tiff_rgba = (uint32 *)_TIFFmalloc(
		tiff_frame.width * tiff_frame.height * sizeof(uint32)
	);
	if(tiff_rgba == NULL)
	{
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Render the TIFF frame data to our RGBA data buffer
	 *
	 * Note that the data will be inverted on the Y axis
	 */
	if(!TIFFRGBAImageGet(
		&tiff_frame,
		tiff_rgba,
		tiff_frame.width, tiff_frame.height
	))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to read or convert the TIFF frame data to RGBA data";
		return(-1);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Append/allocate a new frame on the ImgTIFFReadContext */
	frame_num = ctx->nframes;
	ctx->nframes++;
	ctx->data_list = (u_int8_t **)realloc(
		ctx->data_list,
		ctx->nframes * sizeof(u_int8_t *)
	);
	if(ctx->data_list == NULL)
	{
		ctx->nframes = 0;
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
	ctx->data_list[frame_num] = (u_int8_t *)malloc(
		ctx->data_height * ctx->data_bpl * sizeof(u_int8_t)
	);

	/* Append the delay of this frame to the delay list */
	ctx->delay_ms_list = (unsigned long *)realloc(
		ctx->delay_ms_list,
		ctx->nframes * sizeof(unsigned long)
	);
	if(ctx->delay_ms_list == NULL)
	{
		int i;
		for(i = 0; i < ctx->nframes; i++)
			free(ctx->data_list[i]);
		free(ctx->data_list);
		ctx->data_list = NULL;
		ctx->nframes = 0;
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
/* TODO: when tiff images support frame delay times */
	ctx->delay_ms_list[frame_num] = IMG_TIFF_DEF_FRAME_DELAY_MS;

	tar = (u_int32_t *)ctx->data_list[frame_num];
	if(tar != NULL)
	{
		const int       swidth = (int)tiff_frame.width,
				sheight = (int)tiff_frame.height,
				twidth = ctx->data_width,
				theight = ctx->data_height;

		/* Check if this TIFF frame size is different from
		 * our image size (which was determined from the size
		 * of the first TIFF frame
		 */
		if((swidth != twidth) || (sheight != theight))
		{
			/* Copy/resize the TIFF RGBA frame raster data
			 * to our new RGBA frame data
			 */
			const u_int32_t bg_pixel = 0x00000000;
/*                      ((u_int32_t)ctx->def_alpha_value << 24); */
			int     sx, sy,
				tx, ty;
			const int	sppl = swidth,
					tppl = twidth;
			const u_int32_t *src = (const u_int32_t *)tiff_rgba,
					*src_line_ptr;
			u_int32_t       *tar_line_ptr,
					*tar_line_end;

			/* Clear our RGBA frame data if this TIFF
			 * frame is smaller
			 */
			if((swidth < twidth) || (sheight < theight))
			{
				for(ty = 0; ty < theight; ty++)
				{
					for(tar_line_ptr = tar + (ty * tppl),
					    tar_line_end = tar_line_ptr + twidth;
					    (tar_line_ptr < tar_line_end);
					    tar_line_ptr++
					)
						*tar_line_ptr = bg_pixel;
				}
			}

			/* Copy/resize the TIFF RGBA frame raster
			 * data to our RGBA frame data
			 */
			for(ty = 0; ty < theight; ty++)
			{
				/* TIFF frame data is in RGBA raster
				 * format, so the Y axis is inverted
				 */
				sy = sheight - 1 - (ty * sheight / theight);

				/* Report the progress */
				if((ty % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
				{
					if(ImgReadTIFFReportProgress(ctx, ty, theight))
						break;
				}

				/* Copy/resize the TIFF raster buffer line to
				 * our RGBA frame data line
				 */
				for(tx = 0,
				    src_line_ptr = src + (sy * sppl),
				    tar_line_ptr = tar + (ty * tppl);
				    tx < twidth;
				    tx++
				)
				{
					sx = tx * swidth / twidth;
					tar_line_ptr[tx] = src_line_ptr[sx];
				}
			}
		}
		else
		{
			/* The size of this TIFF frame is the same as
			 * the size of our image, so just copy the
			 * TIFF RGBA frame raster data to our RGBA
			 * frame data
			 */
			int		sy,
					ty;
			const int	sbpp = 4,	/* RGBA */
					sppl = swidth,
					tppl = twidth;
			const u_int32_t *src = (const u_int32_t *)tiff_rgba;

			for(ty = 0; ty < theight; ty++)
			{
				sy = sheight - ty - 1;

				/* Report the progress */
				if((ty % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
				{
					if(ImgReadTIFFReportProgress(ctx, ty, theight))
						break;
				}

				_TIFFmemcpy(
					(tdata_t)(
						tar + (ty * tppl)
					),
					(const tdata_t)(
						src + (sy * sppl)
					),
					(tsize_t)(swidth * sbpp * sizeof(u_int8_t))
				);
			}

		}
	}

	/* Delete the TIFF raster buffer */
	_TIFFfree(tiff_rgba);

	/* End the reading of this TIFF frame */
	TIFFRGBAImageEnd(&tiff_frame);

	return(ctx->status);
}

/*
 *	Reads the TIFF image data to the ImgTIFFReadContext.
 */
static int ImgReadTIFFRGBA(ImgTIFFReadContext *ctx)
{
	const int bpp = 4;		/* RGBA */
	int status;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Get the TIFF image information */
	ImgReadTIFFInformation(ctx);

	/* Set the number of progress levels to report to match
	 * the number of frames in the TIFF image
	 */
	ctx->nprogress_levels = (int)ctx->tiff_ndirectories;

	/* Set our RGBA image data size to match the size of the
	 * first TIFF frame
	 */
	ctx->data_width = (int)ctx->tiff_width;
	ctx->data_height = (int)ctx->tiff_height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = ctx->data_width * ctx->data_bpp;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Read each TIFF frame */
	do {
		status = ImgReadTIFFRGBAIterate(ctx);
		if(status != 0)
		{
			if(ctx->status == 0)
				ctx->status = status;
			break;
		}

		ctx->cur_progress_level++;

	} while(TIFFReadDirectory(tp));

	/* Report the final progress */
	ImgReadTIFFReportProgress(ctx, 1, 1);

	return(ctx->status);
}

/*
 *	Reads the TIFF image stream to RGBA image data.
 */
int ImgStreamReadTIFFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int status;
/*	long fp_pos; */
	TIFF *tp;
	ImgTIFFReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(ctx != NULL) {				\
  if(ctx->tp != NULL)				\
   TIFFClose(ctx->tp);				\
  if(ctx->data_list != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->nframes; i++)		\
    free(ctx->data_list[i]);			\
   free(ctx->data_list);			\
  }						\
  free(ctx->delay_ms_list);			\
  free(ctx);					\
 }						\
						\
 return(_v_);					\
}

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip if user aborted */
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		return(-4);
	}

#if 0
/* This does not work because it seeks the integer descriptor and causes
 * libtiff to be unable to read the magic number */
 	/* Check if the stream data does not have a TIFF signature */
	fp_pos = ftell(fp);
	if(!ImgStreamIsTIFF(fp))
	{
		(void)fseek(fp, fp_pos, SEEK_SET);
		imgio_last_open_error = "Not a TIFF image";
		errno = EINVAL;
		return(-2);
	}
	if(fseek(fp, fp_pos, SEEK_SET))
	{
		const int error_code = errno;
		ImgOpenSetError(strerror(error_code));
		errno = error_code;
		return(-1);
	}
#endif

	/* Set the libtiff warning and error callbacks */
	(void)TIFFSetWarningHandler(ImgReadTIFFWarningCB);
	(void)TIFFSetErrorHandler(ImgReadTIFFErrorCB);

	/* Create our TIFF read context */
	ctx = IMG_TIFF_READ_CONTEXT(calloc(1, sizeof(ImgTIFFReadContext)));
	if(ctx == NULL)
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->def_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = user_aborted;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		CLEANUP_RETURN(-4);
	}

	/* Open the TIFF image stream for reading */
	ctx->tp = tp = TIFFFdOpen(
		fileno(fp),
		"/dev/null",			/* No file name */
		"r"
	);
	if(tp == NULL)
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to open the image for reading";
		CLEANUP_RETURN(-1);
	}

	/* Read the TIFF image to our TIFF read context */
	status = ImgReadTIFFRGBA(ctx);

	/* Get the return values from our TIFF read context */
	if(width_rtn != NULL)
		*width_rtn = ctx->data_width;
	if(height_rtn != NULL)
		*height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
		*bpl_rtn = ctx->data_bpl;
	if(rgba_list_rtn != NULL)
	{
		if(*rgba_list_rtn == NULL)
		{
			*rgba_list_rtn = ctx->data_list;
			ctx->data_list = NULL;
		}
	}
	if(delay_ms_list_rtn != NULL)
	{
		if(*delay_ms_list_rtn == NULL)
		{
			*delay_ms_list_rtn = ctx->delay_ms_list;
			ctx->delay_ms_list = NULL;
		}
	}
	if(nframes_rtn != NULL)
		*nframes_rtn = ctx->nframes;
 
	if(x_rtn != NULL)
		*x_rtn = (int)ctx->tiff_x;
	if(y_rtn != NULL)
		*y_rtn = (int)ctx->tiff_y;
	if(base_width_rtn != NULL)
		*base_width_rtn = (int)ctx->tiff_base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = (int)ctx->tiff_base_width;

	if(*ctx->user_aborted)
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}
	else
	{
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Opens the TIFF image file to RGBA image data.
 */
int ImgFileOpenTIFFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int status;
	FILE *fp = fopen(path, "rb");
	if(fp == NULL)
	{
		imgio_last_open_error = "Unable to open the file for reading";
		return(-1);
	}

	status = ImgStreamReadTIFFRGBA(
		fp,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_list_rtn,
		delay_ms_list_rtn,
		nframes_rtn,
		bg_color,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	);

	(void)fclose(fp);

	return(status);
}

/*
 *	Special handler for reading TIFF data in Greyscale Alpha.
 */
static int ImgReadTIFFRGBAThumbGreyscaleAlphaIterate(ImgTIFFReadContext *ctx)
{
	int frame_num;
	u_int32_t *tar;
	TIFF *tp = ctx->tp;

	/* Append/allocate a new frame on the ImgTIFFReadContext */
	frame_num = ctx->nframes;
	ctx->nframes++;
	ctx->data_list = (u_int8_t **)realloc(
		ctx->data_list,
		ctx->nframes * sizeof(u_int8_t *)
	);
	if(ctx->data_list == NULL)
	{
		ctx->nframes = 0;
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
	ctx->data_list[frame_num] = (u_int8_t *)malloc(
		ctx->data_height * ctx->data_bpl * sizeof(u_int8_t)
	);

	/* Append the delay of this frame to the delay list */
	ctx->delay_ms_list = (unsigned long *)realloc(
		ctx->delay_ms_list,
		ctx->nframes * sizeof(unsigned long)
	);
	if(ctx->delay_ms_list == NULL)
	{
		int i;
		for(i = 0; i < ctx->nframes; i++)
			free(ctx->data_list[i]);
		free(ctx->data_list);
		ctx->data_list = NULL;
		ctx->nframes = 0;
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
/* TODO: when tiff images support frame delay times */
	ctx->delay_ms_list[frame_num] = IMG_TIFF_DEF_FRAME_DELAY_MS;

	tar = (u_int32_t *)ctx->data_list[frame_num];
	if(tar != NULL)
	{
		uint32	tiff_width = 0,
			tiff_height = 0;

		/* Get the size of this TIFF frame */
		(void)TIFFGetField(tp, TIFFTAG_IMAGEWIDTH, &tiff_width);
		(void)TIFFGetField(tp, TIFFTAG_IMAGELENGTH, &tiff_height);

		if(ctx->tiff_planar_config == PLANARCONFIG_CONTIG)
		{
			const int	sbpp = 2,	/* Grayscale Alpha */
					tbpp = ctx->data_bpp,
					tbpl = ctx->data_bpl;
			const tsize_t tiff_bpl = TIFFScanlineSize(tp);
			u_int8_t *tiff_row_buf = (tiff_bpl > 0l) ?
				(u_int8_t *)_TIFFmalloc(tiff_bpl * sizeof(u_int8_t)) : NULL;
			if(tiff_row_buf != NULL)
			{
				const u_int32_t bg_pixel = 0x00000000;
				const int	swidth = (int)tiff_width,
						sheight = (int)tiff_height,
						twidth = ctx->data_width,
						theight = ctx->data_height;
				int		sx, sy,
						tx, ty;
				const u_int8_t	*src_line,
						*src_ptr;
				u_int8_t	*tar_line,
						*tar_line_end,
						*tar_ptr;

				/* Clear our RGBA frame data */
				for(ty = 0; ty < theight; ty++)
				{
					for(tar_ptr = (u_int8_t *)tar + (ty * tbpl),
					    tar_line_end = tar_ptr + (twidth * tbpp);
					    tar_ptr < tar_line_end;
					    tar_ptr += tbpp
					)
						*(u_int32_t *)tar_ptr = bg_pixel;
				}

				for(sy = 0; sy < sheight; sy++)
				{
					/* Report the progress */
					if((sy % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
					{
						if(ImgReadTIFFReportProgress(ctx, sy, sheight))
							break;
					}

					/* Read the next row */
					if(TIFFReadScanline(
						tp,
						tiff_row_buf,
						sy,
						0	/* Sample bit (ignored for
							 * PLANARCONFIG_CONTIG) */
					) == -1)
						continue;

					ty = sy * theight / sheight;

					/* Copy/convert this Greyscale Alpha
					 * row to our RGBA frame data
					 */
					for(tx = 0,
					    src_line = tiff_row_buf,
					    tar_line = (u_int8_t *)tar + (ty * tbpl);
					    tx < twidth;
					    tx++
					)
					{
						sx = tx * swidth / twidth;
						src_ptr = src_line + (sx * sbpp);
						tar_ptr = tar_line + (tx * tbpp);
						tar_ptr[0] = src_ptr[0];
						tar_ptr[1] = src_ptr[0];
						tar_ptr[2] = src_ptr[0];
						tar_ptr[3] = src_ptr[1];
					}
				}
				_TIFFfree(tiff_row_buf);
			}
		}
		else
		{
			if(imgio_last_open_error == NULL)
				imgio_last_open_error = "Unsupported planar configuration for Greyscale Alpha";
			ctx->status = -2;
			errno = EINVAL;
		}
	}

	return(ctx->status);
}

/*
 *	Reads the next frame/directory/subfile in the TIFF stream on
 *	the ImgTIFFReadContext.
 */
static int ImgReadTIFFRGBAThumbIterate(ImgTIFFReadContext *ctx)
{
	int frame_num;
	char emsg[1024];
	u_int32_t *tar;
	uint32 *tiff_rgba;
	TIFF *tp = ctx->tp;
	TIFFRGBAImage tiff_frame;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Check for any errors on this TIFF frame and if it can be
	 * processed as RGBA data
	 */
	if(!TIFFRGBAImageOK(tp, emsg))
	{
		/* Check if the data is in Greyscale Alpha format */
		if((ctx->tiff_bits_per_sample == 8) &&
		   (ctx->tiff_samples_per_pixel == 2)
		)
		{
			uint16	ntags,
				*tags_list;
			if(TIFFGetField(
	                        tp,
				TIFFTAG_EXTRASAMPLES,
				&ntags, &tags_list
			))
			{
				if(ntags >= 1)
				{
					if(tags_list[0] == EXTRASAMPLE_ASSOCALPHA)
					{
						return(ImgReadTIFFRGBAThumbGreyscaleAlphaIterate(ctx));
					}
				}
			}
		}
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Encountered a TIFF frame that could not be processed as RGBA data";
		errno = EINVAL;
		return(-2);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Begin reading this TIFF frame and construct a TIFF decoder
	 * state block
	 */
	if(!TIFFRGBAImageBegin(
		&tiff_frame,
		tp,
		0,                              /* Do not stop on errors */
		emsg
	))
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to construct a TIFF decoder state block";
		return(-1);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Allocate the RGBA data buffer for this TIFF frame that
	 * TIFFRGBAImageGet() will read the TIFF frame data to
	 */
	tiff_rgba = (uint32 *)_TIFFmalloc(
		tiff_frame.width * tiff_frame.height * sizeof(uint32)
	);
	if(tiff_rgba == NULL)
	{
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Render the TIFF frame data to our RGBA data buffer
	 *
	 * Note that the data will be inverted on the Y axis
	 */
	if(!TIFFRGBAImageGet(
		&tiff_frame,
		tiff_rgba,
		tiff_frame.width, tiff_frame.height
	))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to read or convert the TIFF frame data to RGBA data";
		return(-1);
	}

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		return(-4);
	}

	/* Append/allocate a new frame on the ImgTIFFReadContext */
	frame_num = ctx->nframes;
	ctx->nframes++;
	ctx->data_list = (u_int8_t **)realloc(
		ctx->data_list,
		ctx->nframes * sizeof(u_int8_t *)
	);
	if(ctx->data_list == NULL)
	{
		ctx->nframes = 0;
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
	ctx->data_list[frame_num] = (u_int8_t *)malloc(
		ctx->data_height * ctx->data_bpl * sizeof(u_int8_t)
	);

	/* Append the delay of this frame to the delay list */
	ctx->delay_ms_list = (unsigned long *)realloc(
		ctx->delay_ms_list,
		ctx->nframes * sizeof(unsigned long)
	);
	if(ctx->delay_ms_list == NULL)
	{
		int i;
		for(i = 0; i < ctx->nframes; i++)
			free(ctx->data_list[i]);
		free(ctx->data_list);
		ctx->data_list = NULL;
		ctx->nframes = 0;
		_TIFFfree(tiff_rgba);
		TIFFRGBAImageEnd(&tiff_frame);
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Memory allocation error";
		return(-3);
	}
/* TODO: when tiff images support frame delay times */
	ctx->delay_ms_list[frame_num] = IMG_TIFF_DEF_FRAME_DELAY_MS;

	tar = (u_int32_t *)ctx->data_list[frame_num];
	if(tar != NULL)
	{
		/* Render the TIFF raster buffer to our new frame */
		const u_int32_t bg_pixel = 0x00000000;
/*			((u_int32_t)ctx->def_alpha_value << 24); */
		int	sx, sy,
			tx, ty;
		const int	swidth = (int)tiff_frame.width,
				sheight = (int)tiff_frame.height,
				sppl = swidth,
				twidth = ctx->data_width,
				theight = ctx->data_height,
				tppl = twidth;
		const u_int32_t	*src = (const u_int32_t *)tiff_rgba,
				*src_line_ptr;
		u_int32_t	*tar_line_ptr,
				*tar_line_end;

		/* Clear our RGBA frame data */
		for(ty = 0; ty < theight; ty++)
		{
			for(tar_line_ptr = tar + (ty * tppl),
			    tar_line_end = tar_line_ptr + twidth;
			    (tar_line_ptr < tar_line_end);
			    tar_line_ptr++
			)
				*tar_line_ptr = bg_pixel;
		}

		/* Copy/resize the TIFF raster buffer data to our
		 * RGBA frame buffer
		 */
		for(ty = 0; ty < theight; ty++)
		{
			/* TIFF frame data is in RGBA raster format,
			 * so the Y axis is inverted
			 */
			sy = sheight - 1 - (ty * sheight / theight);

			/* Report the progress */
			if((ty % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgReadTIFFReportProgress(ctx, ty, theight))
					break;
			}

			/* Copy/resize the TIFF raster buffer line to
			 * our RGBA frame data line
			 */
			for(tx = 0,
			    src_line_ptr = src + (sy * sppl),
			    tar_line_ptr = tar + (ty * tppl);
			    tx < twidth;
			    tx++
			)
			{
				sx = tx * swidth / twidth;
				tar_line_ptr[tx] = src_line_ptr[sx];
			}
		}
	}

	/* Delete the TIFF raster buffer */
	_TIFFfree(tiff_rgba);

	/* End the reading of this TIFF frame */
	TIFFRGBAImageEnd(&tiff_frame);

	return(ctx->status);
}

/*
 *	Reads the TIFF image data to the ImgTIFFReadContext as a thumb.
 */
static int ImgReadTIFFRGBAThumb(ImgTIFFReadContext *ctx)
{
	const int bpp = 4;			/* RGBA */
	int status;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Get the TIFF image information */
	ImgReadTIFFInformation(ctx);

	/* Set the number of progress levels to 1, since we will
	 * only read the first frame as the thumb
	 */
	ctx->nprogress_levels = 1;

	/* Calculate the size for our RGBA image data based on the
	 * requested size of the thumb
	 */
	ImgCalculateThumbSize(
		(int)ctx->tiff_width, (int)ctx->tiff_height,
		ctx->req_width, ctx->req_height,
		&ctx->data_width, &ctx->data_height
	);
	ctx->data_bpp = bpp;
	ctx->data_bpl = ctx->data_width * ctx->data_bpp;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
		return(-4);

	/* Read only the first TIFF frame */
	do {
		status = ImgReadTIFFRGBAThumbIterate(ctx);
		if(status != 0)
		{
			if(ctx->status == 0)
				ctx->status = status;
			break;
		}

		ctx->cur_progress_level++;

		/* Only get the first frame */
		if(ctx->nframes >= 1)
			break;

	} while(TIFFReadDirectory(tp));

	/* Report the final progress */
	ImgReadTIFFReportProgress(ctx, 1, 1);

	return(ctx->status);
}

/*
 *	Reads the TIFF image stream to RGBA image data as a thumb.
 */
int ImgStreamReadTIFFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int status;
/*	long fp_pos; */
	TIFF *tp;
	ImgTIFFReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(ctx != NULL) {				\
  if(ctx->tp != NULL)				\
   TIFFClose(ctx->tp);				\
  if(ctx->data_list != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->nframes; i++)		\
    free(ctx->data_list[i]);			\
   free(ctx->data_list);			\
  }						\
  free(ctx->delay_ms_list);			\
  free(ctx);					\
 }						\
						\
 return(_v_);					\
}

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip if user aborted */
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		return(-4);
	}

#if 0
/* This does not work because it seeks the integer descriptor and causes
 * libtiff to be unable to read the magic number */
 	/* Check if the stream data does not have a TIFF signature */
	fp_pos = ftell(fp);
	if(!ImgStreamIsTIFF(fp))
	{
		(void)fseek(fp, fp_pos, SEEK_SET);
		imgio_last_open_error = "Not a TIFF image";
		errno = EINVAL;
		return(-2);
	}
	if(fseek(fp, fp_pos, SEEK_SET))
	{
		const int error_code = errno;
		ImgOpenSetError(strerror(error_code));
		errno = error_code;
		return(-1);
	}
#endif

	/* Set the libtiff warning and error callbacks */
	(void)TIFFSetWarningHandler(ImgReadTIFFWarningCB);
	(void)TIFFSetErrorHandler(ImgReadTIFFErrorCB);

	/* Create our TIFF read context */
	ctx = IMG_TIFF_READ_CONTEXT(calloc(1, sizeof(ImgTIFFReadContext)));
	if(ctx == NULL)
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->req_width = req_width;		/* Open as a thumb */
	ctx->req_height = req_height;
	ctx->def_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = user_aborted;

	/* Report the progress */
	if(ImgReadTIFFReportProgress(ctx, 0, 1))
	{
		CLEANUP_RETURN(-4);
	}

	/* Open the TIFF image stream for reading */
	ctx->tp = tp = TIFFFdOpen(
		fileno(fp),
		"/dev/null",			/* No file name */
		"r"
	);
	if(tp == NULL)
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "Unable to open the image for reading";
		CLEANUP_RETURN(-1);
	}

	/* Read the TIFF image to our TIFF read context */
	status = ImgReadTIFFRGBAThumb(ctx);

	/* Get the return values from our TIFF read context */
	if(width_rtn != NULL)
		*width_rtn = ctx->data_width;
	if(height_rtn != NULL)
		*height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
		*bpl_rtn = ctx->data_bpl;
	if(rgba_rtn != NULL)
	{
		if(ctx->nframes > 0)
		{
			*rgba_rtn = ctx->data_list[0];
			ctx->data_list[0] = NULL;
		}
	}
	if(play_time_ms_rtn != NULL)
	{
		int i;
		*play_time_ms_rtn = 0l;
		for(i = 0; i < ctx->nframes; i++)
			*play_time_ms_rtn = (*play_time_ms_rtn) + ctx->delay_ms_list[i];
	}
	if(orig_width_rtn != NULL)
		*orig_width_rtn = ctx->tiff_width;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = ctx->tiff_height;
	if(nframes_rtn != NULL)
		*nframes_rtn = ctx->nframes;
 
	if(x_rtn != NULL)
		*x_rtn = (int)ctx->tiff_x;
	if(y_rtn != NULL)
		*y_rtn = (int)ctx->tiff_y;
	if(base_width_rtn != NULL)
		*base_width_rtn = (int)ctx->tiff_base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = (int)ctx->tiff_base_width;

	if(*ctx->user_aborted)
	{
		if(imgio_last_open_error == NULL)
			imgio_last_open_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}
	else
	{
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Opens the TIFF image file to RGBA image data as a thumb.
 */
int ImgFileOpenTIFFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int status;
	FILE *fp = fopen(path, "rb");
	if(fp == NULL)
	{
		imgio_last_open_error = "Unable to open the file for reading";
		return(-1);
	}

	status = ImgStreamReadTIFFRGBAThumb(
		fp,
		req_width, req_height,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		orig_width_rtn, orig_height_rtn,
		nframes_rtn,
		play_time_ms_rtn,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	);

	(void)fclose(fp);

	return(status);
}


/*
 *	TIFF write warning callback.
 */
static void ImgWriteTIFFWarningCB(
	const char *module,
	const char *fmt,
	va_list ap
)
{
	/* Do nothing */
}

/*
 *	TIFF write error callback.
 */
static void ImgWriteTIFFErrorCB(
	const char *module,
	const char *fmt,
	va_list ap
)
{
	int		len,
			ntokens = 0;
	char *msg;
	const char *s;

	for(s = fmt; *s != '\0'; s++)
	{
		if(*s == '%')
			ntokens++;
	}

	len = strlen(fmt) + (ntokens * 1024);
	msg = (char *)malloc((len + 1) * sizeof(char));
	if(msg != NULL)
	{
		(void)vsprintf(
			msg,
			fmt,
			ap
		);
		ImgSaveSetError(msg);
		free(msg);
	}
}

/*
 *	Reports the write progress on the ImgTIFFWriteContext and
 *	checks for and handles any user abort.
 *
 *	Returns 0 to indicate continue or -4 to abort.
 */
static int ImgWriteTIFFReportProgress(
	ImgTIFFWriteContext *ctx,
	const int i, const int n,
	const int frame_num
)
{
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			(ctx->cur_progress_level * n) + i,
			ctx->nprogress_levels * n,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			((frame_num >= 0) && (frame_num < ctx->nframes)) ?
				ctx->data_list[frame_num] : NULL
		))
		{
			*ctx->user_aborted = 1;
			if(imgio_last_save_error == NULL)
				imgio_last_save_error = "User aborted operation";
			errno = EINTR;
			return(-4);
		}
	}

	return(0);
}

/*
 *	Writes the TIFF header from the values in the
 *	ImgTIFFWriteContext.
 *
 *	The ctx->rows_per_strip will be updated to the desired rows
 *	per strip by this call.
 */
static int ImgWriteTIFFHeader(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	int		tiff_bits_per_sample = 0,
			tiff_samples_per_pixel = 0,
			tiff_compression_type = COMPRESSION_NONE,
			tiff_photo_metric = PHOTOMETRIC_MINISWHITE;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	switch(ctx->compression_type)
	{
	    case 0:				/* None */
		tiff_compression_type = COMPRESSION_NONE;
		break;
	    case 1:				/* GZIP */
		tiff_compression_type = COMPRESSION_DEFLATE;
		break;
	    case 2:				/* JPEG */
		tiff_compression_type = COMPRESSION_JPEG;
		break;
	    case 3:				/* CCITT RLE */
		tiff_compression_type = COMPRESSION_CCITTRLE;
		break;
	    case 4:				/* CCITT FAX3 */
		tiff_compression_type = COMPRESSION_CCITTFAX3;
		break;
	    case 5:				/* CCITT FAX4 */
		tiff_compression_type = COMPRESSION_CCITTFAX4;
		break;
	}
	switch(ctx->color_type)
	{
	    case 0:				/* Black & White */
		tiff_bits_per_sample = 1;
		tiff_samples_per_pixel = 1;
		tiff_photo_metric = PHOTOMETRIC_MINISBLACK;
		break;
	    case 1:				/* Greyscale */
		tiff_bits_per_sample = 8;
		tiff_samples_per_pixel = 1;
		tiff_photo_metric = PHOTOMETRIC_MINISBLACK;
		break;
	    case 2:				/* Greyscale Alpha */
		tiff_bits_per_sample = 8;
		tiff_samples_per_pixel = 2;
		tiff_photo_metric = PHOTOMETRIC_MINISBLACK;
		break;
	    case 3:				/* RGB */
		tiff_bits_per_sample = 8;
		tiff_samples_per_pixel = 3;
		tiff_photo_metric = PHOTOMETRIC_RGB;
		break;
	    case 4:				/* RGBA */
		tiff_bits_per_sample = 8;
		tiff_samples_per_pixel = 4;
		tiff_photo_metric = PHOTOMETRIC_RGB;
		break;
	}

	/* Begin writing the header
	 *
	 * Number of frames/directories/pages
	 */
	(void)TIFFSetField(
		tp,
		TIFFTAG_SUBFILETYPE,
		FILETYPE_PAGE
	);
	(void)TIFFSetField(
		tp,
		TIFFTAG_PAGENUMBER,
		frame_num,
		ctx->nframes
	);

	/* Size */
	(void)TIFFSetField(
		tp,
		TIFFTAG_IMAGEWIDTH,
		ctx->data_width
	);
	(void)TIFFSetField(
		tp,
		TIFFTAG_IMAGELENGTH,
		ctx->data_height
	);
	if(tiff_bits_per_sample > 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_BITSPERSAMPLE,
			tiff_bits_per_sample
		);
	if(tiff_samples_per_pixel > 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_SAMPLESPERPIXEL,
			tiff_samples_per_pixel
		);

	/* Compression */
	(void)TIFFSetField(
		tp,
		TIFFTAG_COMPRESSION,
		tiff_compression_type
	);
	switch(tiff_compression_type)
	{
	    case COMPRESSION_CCITTRLE:
		break;
	    case COMPRESSION_CCITTFAX3:
		/* 32 flag bits for COMPRESSION_CCITTFAX3 */
		(void)TIFFSetField(
			tp,
			TIFFTAG_GROUP3OPTIONS,
			GROUP3OPT_2DENCODING +
/*			GROUP3OPT_UNCOMPRESSED + */
			GROUP3OPT_FILLBITS
		);
		break;
	    case COMPRESSION_CCITTFAX4:
		/* 32 flag bits for COMPRESSION_CCITTFAX4 */
		(void)TIFFSetField(
			tp,
			TIFFTAG_GROUP4OPTIONS,
/*			GROUP4OPT_UNCOMPRESSED + */
			0
		);
		break;
	    case COMPRESSION_LZW:		/* No longer supported by
						 * libtiff (patent issues) */
		/* Prediction scheme with LZW compression */
		(void)TIFFSetField(
			tp,
			TIFFTAG_PREDICTOR,
			2			/* 1 = no horizontal scanline
						 *     differencing
						 * 2 = scanline differencing */
		);
		break;
	    case COMPRESSION_OJPEG:
	    case COMPRESSION_JPEG:
		(void)TIFFSetField(
			tp,
			TIFFTAG_JPEGQUALITY,
			CLIP(
				(int)(ctx->jpeg_quality * 100.0f),/* 0 to 100 */
				0,
				100
			)
		);
		(void)TIFFSetField(
			tp,
			TIFFTAG_JPEGCOLORMODE,
			JPEGCOLORMODE_RAW	/* No auto RGB to YCbCr
						 * conversion */
		);
		break;
	    case COMPRESSION_DEFLATE:		/* GZIP */
		/* Prediction scheme with GZIP compression */
#if 0
		(void)TIFFSetField(
			tp,
			TIFFTAG_PREDICTOR,
			???
		);
#endif
		break;
	}

	/* Color type */
	(void)TIFFSetField(
		tp,
		TIFFTAG_PHOTOMETRIC,
		tiff_photo_metric
	);

	/* Data order within a byte */
	(void)TIFFSetField(
		tp,
		TIFFTAG_FILLORDER,
		FILLORDER_MSB2LSB		/* Most to least significant */
	);

	/* Storage organization */
	(void)TIFFSetField(
		tp,
		TIFFTAG_PLANARCONFIG,
		PLANARCONFIG_CONTIG		/* Single image plane */
	);

	/* Calculate the desired rows per strip after the
	 * TIFFTAG_IMAGEWIDTH, TIFFTAG_BITSPERSAMPLE, and
	 * TIFFTAG_SAMPLESPERPIXEL tags have been set
	 */
	ctx->rows_per_strip = (int)TIFFDefaultStripSize(
		tp,
		IMG_TIFF_DEF_ROWS_PER_STRIP_ESTIMATE
	);
	/* Set the desired rows per strip */
	(void)TIFFSetField(
		tp,
		TIFFTAG_ROWSPERSTRIP,
		(uint32)ctx->rows_per_strip
	);

	/* If alpha is specified then we need to write the marker that
	 * indicatest hat the extra samples are alpha values
	 */
	if((ctx->color_type == 2) || (ctx->color_type == 4))
	{
		const uint16 v[] = { EXTRASAMPLE_ASSOCALPHA };
		(void)TIFFSetField(
			tp,
			TIFFTAG_EXTRASAMPLES,
			sizeof(v) / sizeof(uint16),
			v
		);
	}

	/* Printout metrics */
	(void)TIFFSetField(
		tp,
		TIFFTAG_ORIENTATION,
		ORIENTATION_TOPLEFT
	);
	(void)TIFFSetField(
		tp,
		TIFFTAG_XRESOLUTION,
		150.0
	);
	(void)TIFFSetField(
		tp,
		TIFFTAG_YRESOLUTION,
		150.0
	);
	(void)TIFFSetField(
		tp,
		TIFFTAG_RESOLUTIONUNIT,
		RESUNIT_INCH
	);

	if(frame_num == 0)
	{
#ifdef TIFFTAG_SOFTWARE
		/* Creator */
		if(!STRISEMPTY(ctx->creator))
			(void)TIFFSetField(
				tp,
				TIFFTAG_SOFTWARE,
				ctx->creator
			);
#endif
#ifdef TIFFTAG_DOCUMENTNAME
		/* Title */
		if(!STRISEMPTY(ctx->title))
			(void)TIFFSetField(
				tp,
				TIFFTAG_DOCUMENTNAME,
				ctx->title
			);
#endif
#ifdef TIFFTAG_ARTIST
		/* Author */
		if(!STRISEMPTY(ctx->author))
			(void)TIFFSetField(
				tp,
				TIFFTAG_ARTIST,
				ctx->author
			);
#endif
#ifdef TIFFTAG_IMAGEDESCRIPTION
		/* Comments */
		if(!STRISEMPTY(ctx->comments))
			(void)TIFFSetField(
				tp,
				TIFFTAG_IMAGEDESCRIPTION,
				ctx->comments
			);
#endif
	}

	/* Image offset */
#ifdef TIFFTAG_XPOSITION
	if(ctx->x != 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_XPOSITION,
			ctx->x
		);
#endif
#ifdef TIFFTAG_YPOSITION
	if(ctx->y != 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_YPOSITION,
			ctx->y
		);
#endif

	/* Base size */
#ifdef TIFFTAG_PIXAR_IMAGEFULLWIDTH
	if(ctx->base_width > 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_PIXAR_IMAGEFULLWIDTH,
			ctx->base_width
		);
#endif
#ifdef TIFFTAG_PIXAR_IMAGEFULLLENGTH
	if(ctx->base_height > 0)
		(void)TIFFSetField(
			tp,
			TIFFTAG_PIXAR_IMAGEFULLLENGTH,
			ctx->base_height
		);
#endif

	return(ctx->status);
}


/*
 *	Writes a single TIFF image plane in Black & White format.
 */
static int ImgWriteTIFFRGBAToBW(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	const int	tbpp = 1,		/* BW */
			sbpp = ctx->data_bpp,
			sbpl = ctx->data_bpl,
			width = ctx->data_width,
			height = ctx->data_height,
			tbpl = ((width / 8) + ((width % 8) ? 1 : 0)) * tbpp,
			trows_per_strip = ctx->rows_per_strip;
	const u_int8_t *src;
	u_int8_t *tar;
	size_t tar_nbytes;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	/* Allocate the TIFF frame data */
	tar_nbytes = (size_t)(trows_per_strip * tbpl);
	tar = (u_int8_t *)_TIFFmalloc(tar_nbytes * sizeof(u_int8_t));
	if(tar == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Memory allocation error";
		return(-3);
	}

	/* Render our RGBA frame data to the TIFF frame data */
	src = ctx->data_list[frame_num];
	if(src != NULL)
	{
		int	x, y = 0,
			row,
			strip_num,
			nbytes_to_write;
		const u_int8_t	bg_grey = ImgConvertPixelRGBToGreyscale(ctx->bg_color),
				*src_line;
		u_int8_t	bit_mask,
				*tar_byte_ptr,
				*tar_line;
		while(y < height)
		{
			/* Report the progress */
			if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgWriteTIFFReportProgress(ctx, y, height, frame_num))
					break;
			}

			/* Calculate the TIFF strip number */
			strip_num = y / trows_per_strip;

			/* Copy our RGBA data to the TIFF Greyscale
			 * Alpha strips and increment y
			 */
			for(row = 0; row < trows_per_strip; row++)
			{
				if(y >= height)
					break;

				src_line = src + (y * sbpl);
				tar_line = tar + (row * tbpl);
				for(x = 0; x < width; x++)
				{
					bit_mask = (u_int8_t)(1 << (7 - (x % 8)));
					tar_byte_ptr = tar_line + (x / 8);
					if(ImgConvertPixelRGBAToGreyscale(
						src_line + (x * sbpp),
						bg_grey
					) >= 0x80)
						*tar_byte_ptr |= bit_mask;
					else
						*tar_byte_ptr &= ~bit_mask;
				}

				y++;
			}
			nbytes_to_write = row * tbpl;

			/* Write these TIFF strips */
			if(TIFFWriteEncodedStrip(
				tp,
				(tstrip_t)strip_num,
				(tdata_t)tar,
				(tsize_t)nbytes_to_write
			) != (tsize_t)nbytes_to_write)
			{
				if(imgio_last_save_error == NULL)
					imgio_last_save_error = "Unable to write the TIFF frame";
				ctx->status = -1;
				break;
			}
		}
	}

	/* Delete the TIFF frame data */
	_TIFFfree(tar);

	return(ctx->status);
}

/*
 *	Writes a single TIFF image plane in Greyscale format.
 */
static int ImgWriteTIFFRGBAToGreyscale(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	const int	tbpp = 1,		/* Greyscale */
			sbpp = ctx->data_bpp,
			sbpl = ctx->data_bpl,
			width = ctx->data_width,
			height = ctx->data_height,
			tbpl = width * tbpp,
			trows_per_strip = ctx->rows_per_strip;
	const u_int8_t *src;
	u_int8_t *tar;
	size_t tar_nbytes;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	/* Allocate the TIFF frame data */
	tar_nbytes = (size_t)(trows_per_strip * tbpl);
	tar = (u_int8_t *)_TIFFmalloc(tar_nbytes * sizeof(u_int8_t));
	if(tar == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Memory allocation error";
		return(-3);
	}  

	/* Render our RGBA frame data to the TIFF frame data */
	src = ctx->data_list[frame_num];
	if(src != NULL)
	{
		int	y = 0,
			row,
			strip_num,
			nbytes_to_write;
		const u_int8_t	bg_grey = ImgConvertPixelRGBToGreyscale(ctx->bg_color),
				*src_ptr,
				*src_line_end;
		u_int8_t *tar_ptr;
		while(y < height)
		{
			/* Report the progress */
			if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgWriteTIFFReportProgress(ctx, y, height, frame_num))
					break;
			}

			/* Calculate the TIFF strip number */
			strip_num = y / trows_per_strip;

			/* Copy our RGBA data to the TIFF Greyscale
			 * Alpha strips and increment y
			 */
			for(row = 0; row < trows_per_strip; row++)
			{
				if(y >= height)
					break;

				src_ptr = src + (y * sbpl);
				src_line_end = src_ptr + (width * sbpp);
				tar_ptr = tar + (row * tbpl);
				while(src_ptr < src_line_end)
				{
					tar_ptr[0] = ImgConvertPixelRGBAToGreyscale(
						src_ptr,
						bg_grey
					);
					src_ptr += sbpp;
					tar_ptr += tbpp;
				}

				y++;
			}
			nbytes_to_write = row * tbpl;

			/* Write these TIFF strips */
			if(TIFFWriteEncodedStrip(
				tp,
				(tstrip_t)strip_num,
				(tdata_t)tar,
				(tsize_t)nbytes_to_write
			) != (tsize_t)nbytes_to_write)
			{
				if(imgio_last_save_error == NULL)
					imgio_last_save_error = "Unable to write the TIFF frame";
				ctx->status = -1;
				break;
			}
		}
	}

	/* Delete the TIFF frame data */
	_TIFFfree(tar);

	return(ctx->status);
}

/*
 *	Writes a single TIFF image plane in Greyscale Alpha format.
 */
static int ImgWriteTIFFRGBAToGreyscaleAlpha(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	const int	tbpp = 2,		/* Greyscale Alpha */
			sbpp = ctx->data_bpp,
			sbpl = ctx->data_bpl,
			width = ctx->data_width,
			height = ctx->data_height,
			tbpl = width * tbpp,
			trows_per_strip = ctx->rows_per_strip;
	const u_int8_t *src;
	u_int8_t *tar;
	size_t tar_nbytes;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	/* Allocate the TIFF frame data */
	tar_nbytes = (size_t)(trows_per_strip * tbpl);
	tar = (u_int8_t *)_TIFFmalloc(tar_nbytes * sizeof(u_int8_t));
	if(tar == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Memory allocation error";
		return(-3);
	}  

	/* Render our RGBA frame data to the TIFF frame data */
	src = ctx->data_list[frame_num];
	if(src != NULL)
	{
		int	y = 0,
			row,
			strip_num,
			nbytes_to_write;
		const u_int8_t	*src_ptr,
				*src_line_end;
		u_int8_t *tar_ptr;
		while(y < height)
		{
			/* Report the progress */
			if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgWriteTIFFReportProgress(ctx, y, height, frame_num))
					break;
			}

			/* Calculate the TIFF strip number */
			strip_num = y / trows_per_strip;

			/* Copy our RGBA data to the TIFF Greyscale
			 * Alpha strips and increment y
			 */
			for(row = 0; row < trows_per_strip; row++)
			{
				if(y >= height)
					break;

				src_ptr = src + (y * sbpl);
				src_line_end = src_ptr + (width * sbpp);
				tar_ptr = tar + (row * tbpl);
				while(src_ptr < src_line_end)
				{
					tar_ptr[0] = ImgConvertPixelRGBToGreyscale(src_ptr);
					tar_ptr[1] = src_ptr[3];
					src_ptr += sbpp;
					tar_ptr += tbpp;
				}

				y++;
			}
			nbytes_to_write = row * tbpl;

			/* Write these TIFF strips */
			if(TIFFWriteEncodedStrip(
				tp,
				(tstrip_t)strip_num,
				(tdata_t)tar,
				(tsize_t)nbytes_to_write
			) != (tsize_t)nbytes_to_write)
			{
				if(imgio_last_save_error == NULL)
					imgio_last_save_error = "Unable to write the TIFF frame";
				ctx->status = -1;
				break;
			}
		}
	}

	/* Delete the TIFF frame data */
	_TIFFfree(tar);

	return(ctx->status);
}

/*
 *	Writes a single TIFF image plane in RGB format.
 */
static int ImgWriteTIFFRGBAToRGB(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	const int	tbpp = 3,		/* RGB */
			sbpp = ctx->data_bpp,
			sbpl = ctx->data_bpl,
			width = ctx->data_width,
			height = ctx->data_height,
			tbpl = width * tbpp,
			trows_per_strip = ctx->rows_per_strip;
	const u_int8_t *src;
	u_int8_t *tar;
	size_t tar_nbytes;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	/* Allocate the TIFF frame data */
	tar_nbytes = (size_t)(trows_per_strip * tbpl);
	tar = (u_int8_t *)_TIFFmalloc(tar_nbytes * sizeof(u_int8_t));
	if(tar == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Memory allocation error";
		return(-3);
	}  

	/* Render our RGBA frame data to the TIFF frame data */
	src = ctx->data_list[frame_num];
	if(src != NULL)
	{
		int	y = 0,
			row,
			strip_num,
			nbytes_to_write;
		const u_int8_t	*bg_color = ctx->bg_color,
				*src_ptr,
				*src_line_end;
		u_int8_t *tar_ptr;
		while(y < height)
		{
			/* Report the progress */
			if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgWriteTIFFReportProgress(ctx, y, height, frame_num))
					break;
			}

			/* Calculate the TIFF strip number */
			strip_num = y / trows_per_strip;

			/* Copy our RGBA data to the TIFF RGB strips
			 * and increment y
			 */
			for(row = 0; row < trows_per_strip; row++)
			{
				if(y >= height)
					break;

				src_ptr = src + (y * sbpl);
				src_line_end = src_ptr + (width * sbpp);
				tar_ptr = tar + (row * tbpl);
				while(src_ptr < src_line_end)
				{
					ImgConvertPixelRGBAToRGB(
						src_ptr,
						bg_color,
						tar_ptr
					);
					src_ptr += sbpp;
					tar_ptr += tbpp;
				}

				y++;
			}
			nbytes_to_write = row * tbpl;

			/* Write these TIFF strips */
			if(TIFFWriteEncodedStrip(
				tp,
				(tstrip_t)strip_num,
				(tdata_t)tar,
				(tsize_t)nbytes_to_write
			) != (tsize_t)nbytes_to_write)
			{
				if(imgio_last_save_error == NULL)
					imgio_last_save_error = "Unable to write the TIFF frame";
				ctx->status = -1;
				break;
			}
		}
	}

	/* Delete the TIFF frame data */
	_TIFFfree(tar);

	return(ctx->status);
}

/*
 *	Writes a single TIFF image plane in RGBA format.
 */
static int ImgWriteTIFFRGBAToRGBA(
	ImgTIFFWriteContext *ctx,
	const int frame_num
)
{
	const int	tbpp = 4,		/* RGBA */
			sbpp = ctx->data_bpp,
			sbpl = ctx->data_bpl,
			width = ctx->data_width,
			height = ctx->data_height,
			tbpl = width * tbpp,
			trows_per_strip = ctx->rows_per_strip;
	const u_int8_t *src;
	u_int8_t *tar;
	size_t tar_nbytes;
	TIFF *tp = ctx->tp;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
		return(-4);

	/* Allocate the TIFF frame data */
	tar_nbytes = (size_t)(trows_per_strip * tbpl);
	tar = (u_int8_t *)_TIFFmalloc(tar_nbytes * sizeof(u_int8_t));
	if(tar == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Memory allocation error";
		return(-3);
	}  

	/* Render our RGBA frame data to the TIFF frame data */
	src = ctx->data_list[frame_num];
	if(src != NULL)
	{
		const int min_bpl = MIN(
			width * sbpp,
			width * tbpp
		);
		int	y = 0,
			row,
			strip_num,
			nbytes_to_write;
		while(y < height)
		{
			/* Report the progress */
			if((y % IMG_TIFF_PROGRESS_RESOLUTION) == 0)
			{
				if(ImgWriteTIFFReportProgress(ctx, y, height, frame_num))
					break;
			}

			/* Calculate the TIFF strip number */
			strip_num = y / trows_per_strip;

			/* Copy our RGBA data to the TIFF RGBA strips
			 * and increment y
			 */
			for(row = 0; row < trows_per_strip; row++)
			{
				if(y >= height)
					break;

				(void)_TIFFmemcpy(
					(tdata_t)(tar + (row * tbpl)),
					(const tdata_t)(src + (y * sbpl)),
					(size_t)(min_bpl * sizeof(u_int8_t))
				);
				y++;
			}
			nbytes_to_write = row * tbpl;

			/* Write these TIFF strips */
			if(TIFFWriteEncodedStrip(
				tp,
				(tstrip_t)strip_num,
				(tdata_t)tar,
				(tsize_t)nbytes_to_write
			) != (tsize_t)nbytes_to_write)
			{
				if(imgio_last_save_error == NULL)
					imgio_last_save_error = "Unable to write the TIFF frame";
				ctx->status = -1;
				break;
			}
		}
	}

	/* Delete the TIFF frame data */
	_TIFFfree(tar);

	return(ctx->status);
}

/*
 *	Writes the RGBA image data on the ImgTIFFWriteContext to a
 *	TIFF image.
 */
static int ImgWriteTIFFRGBA(ImgTIFFWriteContext *ctx)
{
	int i;
	int (*write_frame_func)(
		ImgTIFFWriteContext *ctx,
		const int frame_num
	) = NULL;
	TIFF *tp = ctx->tp;

	/* Determine the write frame function */
	switch(ctx->color_type)
	{
	    case 0:				/* Black & White */
		write_frame_func = ImgWriteTIFFRGBAToBW;
		break;
	    case 1:				/* Greyscale */
		write_frame_func = ImgWriteTIFFRGBAToGreyscale;
		break;
	    case 2:				/* Greyscale Alpha */
		write_frame_func = ImgWriteTIFFRGBAToGreyscaleAlpha;
		break;
	    case 3:				/* RGB */
		write_frame_func = ImgWriteTIFFRGBAToRGB;
		break;
	    case 4:				/* RGBA */
		write_frame_func = ImgWriteTIFFRGBAToRGBA;
		break;
	}
	if(write_frame_func == NULL)
	{
		imgio_last_save_error = "Unsupported color type";
		return(-2);
	}

	/* Write all of our RGBA image frames to the TIFF image */
	for(i = 0; i < ctx->nframes; i++)
	{
		/* Write the header for this frame/directory/subfile */
		ctx->status = ImgWriteTIFFHeader(
			ctx,
			i
		);
		if(ctx->status != 0)
			break;
		if(*ctx->user_aborted)
			break;

		/* Write the frame data for this frame/directory/subfile */
		ctx->status = write_frame_func(
			ctx,
			i
		);
		if(ctx->status != 0)
			break;
		if(*ctx->user_aborted)
			break;

		/* Write the current current contents of the
		 * directory/subfile to the stream
		 */
		if(TIFFWriteDirectory(tp) == 0)
		{
			if(imgio_last_save_error == NULL)
				imgio_last_save_error = "Unable to write a new TIFF directory";
			ctx->status = -1;
			break;
		}

		ctx->cur_progress_level++;
	}
	if(*ctx->user_aborted)
		return(-4);

	/* Report the final progress */
	if(ImgWriteTIFFReportProgress(ctx, 1, 1, ctx->nframes - 1))
		return(-4);

	return(ctx->status);
}

/*
 *	Writes the RGBA image data to a TIFF image stream.
 */
int ImgStreamWriteTIFFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,                   /* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
			user_aborted = 0;
	TIFF *tp;
	ImgTIFFWriteContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(ctx != NULL) {				\
  if(ctx->tp != NULL)				\
   TIFFClose(ctx->tp);				\
  free(ctx);					\
 }						\
						\
 return(_v_);					\
}

	/* Reset the last save error message pointer */
	imgio_last_save_error = NULL;

	if(fp == NULL)
	{
		imgio_last_save_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Set the libtiff warning and error callbacks */
	(void)TIFFSetWarningHandler(ImgWriteTIFFWarningCB);
	(void)TIFFSetErrorHandler(ImgWriteTIFFErrorCB);

	/* Create our TIFF write context */
	ctx = IMG_TIFF_WRITE_CONTEXT(calloc(1, sizeof(ImgTIFFWriteContext)));
	if(ctx == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->nframes = nframes;
	ctx->data_width = width;
	ctx->data_height = height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = bpl;
	ctx->data_list = rgba_list;
	ctx->delay_ms_list = delay_ms_list;
	ctx->bg_color = bg_color;
	ctx->x = x;
	ctx->y = y;
	ctx->base_width = base_width;
	ctx->base_height = base_height;
	ctx->creator = creator;
	ctx->title = title;
	ctx->author = author;
	ctx->comments = comments;
	ctx->compression_type = compression_type;
	ctx->jpeg_quality = jpeg_quality;
	ctx->color_type = color_type;
	ctx->rows_per_strip = IMG_TIFF_DEF_ROWS_PER_STRIP_ESTIMATE;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->nprogress_levels = ctx->nframes;
	ctx->user_aborted = &user_aborted;

	/* Report the progress */
	if(ImgWriteTIFFReportProgress(ctx, 0, 1, 0))
	{
		CLEANUP_RETURN(-4);
	}

	/* Open the TIFF image stream for writing */
	ctx->tp = tp = TIFFFdOpen(
		fileno(fp),
		"/dev/null",                    /* No file name */
		"w+b"				/* libtiff must open TIFF
						 * images for read and write
						 * when writing */
	);
	if(tp == NULL)
	{
		if(imgio_last_save_error == NULL)
			imgio_last_save_error = "Unable to open the image for writing";
		CLEANUP_RETURN(-1);
	}

	/* Write the TIFF image on our TIFF write context */
	status = ImgWriteTIFFRGBA(ctx);

	if(*ctx->user_aborted)
	{
		imgio_last_save_error = "User aborted operation";
		CLEANUP_RETURN(-4);
	}
	else
	{
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Saves the RGBA image data to a TIFF image file.
 */
int ImgFileSaveTIFFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,                   /* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int		status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
		imgio_last_save_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Open/create the TIFF image file for writing */
	fp = fopen(
		path,
		"w+b"				/* libtiff must open TIFF
						 * images for read and write
						 * when writing */
	);
	if(fp == NULL)
	{
		ImgSaveSetError(strerror(errno));
		return(-1);
	}

	/* Write the stream */
	status = ImgStreamWriteTIFFRGBA(
		fp,
		width, height,
		bpl,
		rgba_list,
		delay_ms_list,
		nframes,
		bg_color,
		x, y,
		base_width, base_height,
		creator, title,
		author, comments,
		compression_type,
		jpeg_quality,
		color_type,
		progress_cb, progress_data
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}
#endif	/* HAVE_LIBTIFF */
