#include <stdlib.h>
#include <string.h>
#include "../include/tga.h"
#include "imgio.h"


/* Last error message pointer */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


/* Open */
int ImgFileOpenTGARGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);

/* Save */
static int ImgTGAWriteRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *comments,
        const int color_type,			/* 0 = Greyscale
                                                 * 1 = RGB
                                                 * 2 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileSaveTGARGBA(
        const char *path,
        const int width, const int height,
        const int bpl,
        const u_int8_t *rgba,
        const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
        const int x, const int y,
        const int base_width, const int base_height,
        const char *comments,
        const int color_type,			/* 0 = Greyscale
                                                 * 1 = RGB
                                                 * 2 = RGBA */
        ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_TGA_PROGRESS_RESOLUTION		10


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))     

#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : 1)


/*
 *	TGA library read front end for RGBA (4 bpp) image data.
 *
 *      Return values:
 *
 *      0       Success
 *      -1      General error
 *      -2      Bad value (invalid format/not a tga file)
 *      -3      Systems error
 *      -4      User abort
 */
int ImgFileOpenTGARGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;
	int		width, height,
			bpl;
	u_int8_t *rgba;

	/* Reset global load error string pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    return(-4);
	}

	/* Open the TGA image */
	rgba = TgaReadFromFileFastRGBA(
	    path,
	    &width, &height,
	    (bg_color != NULL) ? *(u_int32_t *)bg_color : 0x00000000
	);
	bpl = width * bpp;

	if(rgba_rtn != NULL)
	    *rgba_rtn = rgba;
	if(width_rtn != NULL)
	    *width_rtn = width;
	if(height_rtn != NULL)
	    *height_rtn = height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = bpl;

	/* Read TGA header for additional information */
	if(rgba != NULL)
	{
	    tga_data_struct tga_data, *td = &tga_data;
	    if(TgaReadHeaderFromFile(path, td) == TgaSuccess)
	    {
		if(x_rtn != NULL)
		    *x_rtn = td->x;
		if(y_rtn != NULL)
		    *y_rtn = td->y;
		if(comments_rtn != NULL)
		{
		    free(*comments_rtn);
		    *comments_rtn = STRDUP(td->comments);
		}
	    }
	    TgaDestroyData(td);
	}

	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    return(-4);
	}
	else
	    return((rgba != NULL) ? 0 : -1);
}

/*
 *      TGA library write front end for RGBA (4 bpp) image data.
 *
 *      Return values:
 *
 *      0       Success
 *      -1      General error
 *      -2      Bad value (invalid format)
 *      -3      Systems error
 *      -4      User abort
 */
static int ImgTGAWriteRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *comments,
        const int color_type,			/* 0 = Greyscale
                                                 * 1 = RGB
                                                 * 2 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
			tga_depth;
	tga_data_struct td_data, *td = &td_data;

	if(*user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    return(-4);
	}

	/* Save by the color type */
	switch(color_type)
	{
	  case 1:				/* 24 bits RGB */
	    tga_depth = 24;
	    break;
	  case 0:				/* 8 bits greyscale */
	    tga_depth = 8;
	    break;
	  default:				/* 32 bits RGBA */
	    tga_depth = 32;
	    break;
	}

	/* Set up tga data for writing */
	(void)memset(td, 0x00, sizeof(tga_data_struct));
	td->x = x;
	td->y = y;
	td->width = width;
	td->height = height;
	td->depth = 32;	/* Ignored */
	td->data = (u_int8_t *)malloc(bpl * height);
	td->data_depth = bpp * 8;
	td->comments = STRDUP(comments);

	/* Copy and convert source image in RGBA format to target
	 * image in BGRA format
	 */
	if(td->data != NULL)
	{
	    int x, y;
	    u_int8_t *tar = td->data, *tar_line, *tar_ptr;
	    const u_int8_t *src = rgba, *src_line, *src_ptr;

	    for(y = 0; y < height; y++)
	    {
		tar_line = tar + (y * bpl);
		src_line = src + (y * bpl);

		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_line + (x * bpp);
		    src_ptr = src_line + (x * bpp);

		    *tar_ptr++ = src_ptr[2];
		    *tar_ptr++ = src_ptr[1];
		    *tar_ptr++ = src_ptr[0];
		    *tar_ptr = src_ptr[3];
		}

		/* Report progress */
		if((progress_cb != NULL) && ((y % IMG_TGA_PROGRESS_RESOLUTION) == 0))
		{
		    if(!progress_cb(
			progress_data,
			y, height,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}
	    }
	}

	/* Write tga image */
	if(*user_aborted)
	    status = TgaNoAccess;
	else
	    status = TgaWriteToFile(
		path,
		td,
		tga_depth	/* Target depth */
	    );

	/* Delete tga data */
	TgaDestroyData(td);

	if(*user_aborted)
	    return(-4);
	else
	    return((status == TgaSuccess) ? 0 : -1);
}

/*
 *	Writes the specified image data to a TGA file.
 *
 *      Return values:
 *
 *      0       Success
 *      -1      General error
 *      -2      Bad value
 *      -3      Systems error
 *      -4      User abort
 */
int ImgFileSaveTGARGBA(
        const char *path,
        const int width, const int height,
        const int bpl,
        const u_int8_t *rgba,
        const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
        const int x, const int y,
        const int base_width, const int base_height,
        const char *comments,
        const int color_type,			/* 0 = Greyscale
                                                 * 1 = RGB
                                                 * 2 = RGBA */
        ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
			user_aborted = 0,
			_bpl = bpl;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((path == NULL) || (rgba == NULL) ||
	   (width <= 0) || (height <= 0)
	)
	{
	    imgio_last_save_error = "Invalid value used to describe image";
	    return(-1);
	}

	/* Automatically calculate bytes per line? */
	if(_bpl <= 0)
	    _bpl = width * bpp;

	/* Call TGA writing function for RGBA image data format */
	status = ImgTGAWriteRGBA(
	    path,
	    width, height,
	    _bpl,
	    rgba,
	    bg_color,
	    x, y,
	    base_width, base_height,
	    comments,
	    color_type,
	    progress_cb, progress_data,
	    &user_aborted
	);

	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
	    if(!progress_cb(
		progress_data,
		height, height,
		width, height,
		_bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	if(user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}
