#ifdef HAVE_LIBPNG
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
/* png.h and setjmp.h may not be combined */
/* #include <setjmp.h> */
#include <png.h>				/* libpng */
#if defined(_WIN32)
# include "../include/string.h"
#endif
#include "imgio.h"


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


typedef struct _ImgPNGReadContext	ImgPNGReadContext;
#define IMG_PNG_READ_CONTEXT(p)		((ImgPNGReadContext *)(p))
typedef struct _ImgPNGWriteContext	ImgPNGWriteContext;
#define IMG_PNG_WRITE_CONTEXT(p)	((ImgPNGWriteContext *)(p))


typedef enum {
	IMG_PNG_DATA_STORAGE_BUFFER,
	IMG_PNG_DATA_STORAGE_STREAM
} ImgPNGDataStorageType;


/*	png_jmpbuf() macro used in error handling
 *
 *	This macro became available in libpng version 1.0.6
 *
 *	Notes: If you want to be able to run your code with older
 *	versions of libpng, then you must define the macro yourself (but
 *	only if it is not already defined by libpng)
 */
#ifndef png_jmpbuf
# define png_jmpbuf(png_ptr)	((png_ptr)->jmpbuf)
#endif


/* Version */
void ImgPNGVersion(int *major, int *minor, int *release);

/* Check Type */
int ImgBufferIsPNG(const u_int8_t *data, const int len);
int ImgStreamIsPNG(FILE *fp);
int ImgFileIsPNG(const char *path);

/* Open */
static void ImgReadPNGErrorCB(png_structp png_ptr, png_const_charp msg);
static void ImgReadPNGWarningCB(png_structp png_ptr, png_const_charp msg);
static void ImgReadPNGInfoCB(png_structp png_ptr, png_infop png_info_ptr);
static void ImgReadPNGReadRowRGBACB(
	png_structp png_ptr,
	png_bytep new_row,
	png_uint_32 row_num,
	int pass
);
static void ImgReadPNGEndCB(png_structp png_ptr, png_infop png_info_ptr);
static void ImgReadPNGInsertPNGLineToRGBA(
	const int png_color_type,
	const u_int8_t *png_row_data,
	const int png_width,
	const int png_bpl,
	u_int8_t *rgba,
	const int rgba_width, const int rgba_height,
	const int rgba_bpl,
	const int rgba_y,
	u_int8_t *transparent_color_rgba,
	const u_int8_t def_alpha_value
);
static void ImgReadPNGGetTextChunks(ImgPNGReadContext *ctx);
static int ImgReadPNGRGBA(
	const ImgPNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadPNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadPNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenPNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgReadPNGRGBAThumb(
	const ImgPNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadPNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadPNGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenPNGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);


/* Save */
static void ImgWritePNGErrorCB(png_structp png_ptr, png_const_charp msg);
static void ImgWritePNGWarningCB(png_structp png_ptr, png_const_charp msg);
static void ImgWritePNGWriteCB(
	png_structp png_ptr,
	png_bytep data, png_size_t length
);
static void ImgWritePNGFlushCB(png_structp png_ptr);
static void ImgWritePNGConvertLineRGBAToGreyscale(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line,
	const u_int8_t *bg_color
);
static void ImgWritePNGConvertLineRGBAToGreyscaleAlpha(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line
);
static void ImgWritePNGConvertLineRGBAToRGB(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line,
	const u_int8_t *bg_color
);
static int ImgWritePNGRGBA(
	const ImgPNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWritePNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W  
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB  
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWritePNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W  
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB  
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSavePNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W  
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB  
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Read Context:
 */
struct _ImgPNGReadContext {

	ImgPNGDataStorageType	data_storage_type;
	const void	*bp;
	unsigned long	bp_len,
					bp_pos;
	FILE		*fp;
	unsigned long	fp_len;

	void		*io_buf;		/* Read buffer */
	unsigned long	io_buf_len;

	int		req_width, req_height;	/* Positive if opening as a
						 * thumb */

	/* PNG */
	png_structp	png_ptr;
	png_infop	png_info_ptr;
	int		png_x, png_y,
			png_width, png_height,
			png_bpp, png_bpl,
			png_bit_depth,
			png_color_type,		/* One of PNG_COLOR_TYPE_* */
			png_interlace_type,	/* One of PNG_INTERLACE_* */
			png_interlace_pass,	/* Interlace pass number */
			png_ninterlace_passes,	/* Total interlace passes */
			png_last_interlace_pass_reported,
						/* Last interlace pass
						 * reported to progress */
			png_last_row;
	png_bytep	png_last_row_ptr;	/* Pointer to last row, used
						 * for interlaced loading */
	int		png_encountered_rows;	/* Highest number of rows
						 * encountered, think of it
						 * as the *current* allocated
						 * height */

	/* PNG row buffer used for reading progressive (non-interlaced)
	 * PNG images, allocated to the size of png_bpl in
	 * ImgReadPNGInfoCB()
	 */
	png_bytep	png_row_buffer;

	/* PNG image data used for reading interlaced PNG images,
	 * allocated to png_height row buffers each allocated to the
	 * size of png_bpl in ImgReadPNGInfoCB()
	 */
	png_bytep	*png_data;

	/* Loaded image data */
	int		data_width, data_height,
					data_bpp, data_bpl;
	u_int8_t	*data;
	int		data_y_open_thumb_last;	/* For opening thumbs */

	/* Transparent color in 4 bytes RGBA
	 *
	 * If this is not NULL then it hints that a tRNS chunk was
	 * encountered
	 *
	 * This needs to be deleted if it is not NULL
	 */
	u_int8_t	*transparent_color_rgba;

	u_int8_t	default_alpha_value;

	u_int8_t	*bg_color;		/* 4 bytes RGBA */

	char		**creator,
					**title,
					**author,
					**comments;

	unsigned long	*modified_time_sec;	/* In seconds since EPOCH */

	int		operation_completed;

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;

	int		status;
};

/*
 *	Write Context:
 */
struct _ImgPNGWriteContext {

	ImgPNGDataStorageType	data_storage_type;
	void		**bp;
	unsigned long	*bp_len;
	FILE		*fp;

	unsigned long	io_size;

	/* PNG */
	png_structp	png_ptr;
	png_infop	png_info_ptr;
	int		png_x, png_y,
			png_width, png_height,
			png_bpp, png_bpl,
			png_bit_depth;
	int		png_color_type;		/* One of PNG_COLOR_TYPE_* */
	int		png_interlace_type;	/* One of PNG_INTERLACE_* */

	/* PNG image data used for writing interlaced PNG images,
	 * allocated to png_height row buffers each allocated to the
	 * size of png_bpl
	 */
	png_bytep	*png_data;

	/* Image data to write */
	int		data_width, data_height,
					data_bpp, data_bpl;
	const u_int8_t	*data;

	const u_int8_t	*bg_color;		/* 4 bytes RGBA */

	const char	*creator,
					*title,
					*author,
					*comments;

	unsigned long	modified_time_sec;	/* In seconds since EPOCH */

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;

	int		status;
};


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_PNG_PROGRESS_RESOLUTION	10


/*
 *	Suggested IO Buffer Length (in bytes):
 */
#define IMG_PNG_DEF_IO_SIZE		1024

/*
 *	Minimum IO Buffer Length (in bytes):
 */
#define IMG_PNG_MIN_IO_SIZE		4


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))     

#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : 1)


/*
 *	Gets the PNG library's version.
 */
void ImgPNGVersion(int *major, int *minor, int *release)
{
	if(major != NULL)   
#ifdef PNG_LIBPNG_VER_MAJOR
		*major = PNG_LIBPNG_VER_MAJOR;
#else
#warning "PNG_LIBPNG_VER_MAJOR (libpng version major number) not defined."
		*major = 0;
#endif
	if(minor != NULL)
#ifdef PNG_LIBPNG_VER_MINOR
		*minor = PNG_LIBPNG_VER_MINOR;
#else
		*minor = 0;
#endif
	if(release != NULL)
#ifdef PNG_LIBPNG_VER_RELEASE
		*release = PNG_LIBPNG_VER_RELEASE;
#else
		*release = 0;
#endif
}


/*
 *	Checks for the PNG type ID.
 */
int ImgBufferIsPNG(const u_int8_t *data, const int len)
{
	if((data == NULL) || (len < 8))
	{
		errno = EINVAL;
		return(0);
	}

	if(png_check_sig((png_bytep)data, len))
		return(1);

	errno = EINVAL;
	return(0);
}

int ImgStreamIsPNG(FILE *fp)
{
	u_int8_t buf[8];

	size_t units_read;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(0);
	}

	units_read = fread(
		buf,
		sizeof(u_int8_t),
		sizeof(buf) / sizeof(u_int8_t),
		fp
	);
	if(units_read == (sizeof(buf) / sizeof(u_int8_t)))
		return(ImgBufferIsPNG(buf, sizeof(buf)));
	else
		return(0);
}

int ImgFileIsPNG(const char *path)
{
	int             status,
					error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
		return(0);

	status = ImgStreamIsPNG(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	PNG read error callback.
 *
 *	Records the error message on the local png message buffer.
 */
static void ImgReadPNGErrorCB(png_structp png_ptr, png_const_charp msg)
{
	ImgPNGReadContext *ctx = IMG_PNG_READ_CONTEXT(
		png_get_error_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Set the error message from libpng only if no other error
	 * message has been set yet
	 */
	if(imgio_last_open_error == NULL)
		ImgOpenSetError(msg);

	if(ctx->status == 0)
		ctx->status = -1;

	/* Return control to the setjmp point (this function does not
	 * return to libpng)
	 */
	longjmp(png_jmpbuf(ctx->png_ptr), 1);
}

/*
 *	PNG read warning callback.
 *
 *	Records the warning message on the local png message buffer.
 */
static void ImgReadPNGWarningCB(png_structp png_ptr, png_const_charp msg)
{
	ImgPNGReadContext *ctx = IMG_PNG_READ_CONTEXT(
		png_get_error_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Ignore warnigs */
}

/*
 *	PNG library read info callback.
 */
static void ImgReadPNGInfoCB(png_structp png_ptr, png_infop png_info_ptr)
{
	int		color_type,
					width, height,
					bpl,
					bit_depth,
					interlace_type,
					buf_len;
	ImgPNGReadContext *ctx = IMG_PNG_READ_CONTEXT(
		png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Update the PNG read instance pointer on the context */
	ctx->png_ptr = png_ptr;
	ctx->png_info_ptr = png_info_ptr;

	/* Get the PNG color type */
	ctx->png_color_type = color_type = ((png_info_ptr != NULL) ?
		png_info_ptr->color_type : PNG_COLOR_TYPE_GRAY
	);

	/* Get the PNG bit depth */
	ctx->png_bit_depth = bit_depth = (png_info_ptr != NULL) ? png_info_ptr->bit_depth : 1;


	/* Insert transformations here */

	/* Swap bytes of 16 bit files to least significant byte first */
/*	png_set_swap(png_ptr); */

	/* Reduce 16 bits data down to 8 bits data */
	if(bit_depth == 16)
		png_set_strip_16(png_ptr);

	/* Swap the RGBA or GA data to ARGB or AG (or BGRA to ABGR) */
/*	png_set_swap_alpha(png_ptr); */

	/* Flip the RGB pixels to BGR (or RGBA to BGRA) */
/*
	if(color_type & PNG_COLOR_MASK_COLOR)
		png_set_bgr(png_ptr);
 */

	/* Extract multiple pixels with bit depths of 1, 2, and 4 from
	 * a single byte into separate bytes (useful for paletted and
	 * grayscale images)
	 */
/*	png_set_packing(png_ptr); */

	/* Change the order of packed pixels to least significant bit
	 * first (not useful if you are using png_set_packing)
	 */
/*	png_set_packswap(png_ptr); */

	/* Expand paletted colors into true RGB triplets, this will
	 * make the actual bpp value to be 3 instead of the reported 1
	 */
	if(color_type == PNG_COLOR_TYPE_PALETTE)
		png_set_expand(png_ptr);

	/* Expand grayscale images to the full 8 bits from 1, 2, or 4
	 * bits per pixel
	 */
	if((color_type == PNG_COLOR_TYPE_GRAY) && (bit_depth < 8))
		png_set_expand(png_ptr);

	/* Get the tEXt chunks */
	ImgReadPNGGetTextChunks(ctx);

#if defined(PNG_INFO_tIME) && defined(PNG_tIME_SUPPORTED)
	/* Get the modified time from the tIME chunk */
	if(png_get_valid(
		png_ptr,
		png_info_ptr,
		PNG_INFO_tIME
	) != 0x00000000)
	{
		png_timep png_modified_time = NULL;
		if(png_get_tIME(
			png_ptr,
			png_info_ptr,
			&png_modified_time
		) != 0)
		{
			if(png_modified_time != NULL)
			{
				struct tm tm_buf;
				tm_buf.tm_sec = (int)png_modified_time->second;
				tm_buf.tm_min = (int)png_modified_time->minute;
				tm_buf.tm_hour = (int)png_modified_time->hour;
				tm_buf.tm_mday = (int)png_modified_time->day;
				tm_buf.tm_mon = (int)png_modified_time->month - 1;
				tm_buf.tm_year = (int)png_modified_time->year - 1900;
				tm_buf.tm_wday = 0;	/* Not used */
				tm_buf.tm_yday = 0;	/* Not used */
				tm_buf.tm_isdst = -1;	/* DST not available */
				if(ctx->modified_time_sec != NULL)
				{
					*ctx->modified_time_sec = (unsigned long)mktime(&tm_buf);
					/* MNG stores time values relative to GMT so 
					 * we need to account for the time zone in
					 * order to have the time value in local time
					 */
					*ctx->modified_time_sec = (*ctx->modified_time_sec) - (unsigned long)timezone;
				}
			}
		}
	}
#endif	/* PNG_INFO_tIME && PNG_tIME_SUPPORTED */

#if defined(PNG_INFO_tRNS)
	/* Expand paletted or RGB images with transparency to full
	 * alpha channels so that the PNG data will be available as
	 * RGBA and get the transparent color
	 *
	 * Check if there is a transparent color tRNS chunk
	 */
	if(png_get_valid(
		png_ptr,
		png_info_ptr,
		PNG_INFO_tRNS
	) != 0x00000000)
	{
		/* Add an alpha channel so that the PNG paletted or RGB
		 * image becomes RGBA
		 */
		png_set_expand(png_ptr);

		/* Set the transparent color on our context, this will
		 * serve as a hint that the tRNS chunk was encountered
		 */
		ctx->transparent_color_rgba = (u_int8_t *)realloc(
			ctx->transparent_color_rgba,
			4 * sizeof(u_int8_t)
		);
		if(ctx->transparent_color_rgba != NULL)
		{
#if defined(PNG_tRNS_SUPPORTED)
			int ntransparent_colors;
			u_int8_t *transparent_color_rgba = ctx->transparent_color_rgba;
			png_bytep a;
			png_color_16p transparent_colors_list;

			/* Get the transparency information */
			if(png_get_tRNS(
				png_ptr,
				png_info_ptr,
				&a,		/* Alpha value for
						 * PNG_COLOR_TYPE_PALETTE images
						 * from 0x00 to 0xFF */
				&ntransparent_colors,
				&transparent_colors_list
			) != 0)
			{
				/* Palette image? */
				if(color_type == PNG_COLOR_TYPE_PALETTE)
				{
#if 0
/* TODO apply the transparent colors to the colors */
					int i;
					png_color *palette;
					png_get_PLTE(png_ptr, info_ptr, &palette, &ntransparent_colors);
					for(i = 0; i < ntransparent_colors; i++)
					{
						colormap->c[i].a = 0xff - (u_int8_t)a[i];
						colormap->c[i].r = (u_int8_t)palette[i].red;
						colormap->c[i].g = (u_int8_t)palette[i].green;
						colormap->c[i].b = (u_int8_t)palette[i].blue;
					}
#endif
				}
				/* RGB image */
				else if(transparent_colors_list != NULL)
				{
					/* Get the single transparent color for RGB images */
					png_color_16p c16 = transparent_colors_list;
					transparent_color_rgba[0] = (u_int8_t)(c16->red >> 8);
					transparent_color_rgba[1] = (u_int8_t)(c16->green >> 8);
					transparent_color_rgba[2] = (u_int8_t)(c16->blue >> 8);
					transparent_color_rgba[3] = ctx->default_alpha_value;
				}
			}
#endif	/* PNG_tRNS_SUPPORTED */
		}
	}
#endif	/* PNG_INFO_tRNS */

#if defined(PNG_bKGD_SUPPORTED) && defined(PNG_INFO_bKGD)
	/* Get the background color from the bKGD chunk */
	if(png_get_valid(
		png_ptr,
		png_info_ptr,
		PNG_INFO_bKGD
	) != 0x00000000)
	{
		png_color_16p pBackground;
		u_int8_t *bg_color = ctx->bg_color;

		/* It is not obvious from the libpng documentation but
		 * png_get_bKGD() takes a pointer to a pointer and it
		 * always returns valid RGB values regardless of the
		 * PNG color type
		 */
		if(png_get_bKGD(
			png_ptr,
			png_info_ptr,
			&pBackground
		) != 0)
		{
			/* png_get_bKGD() always returns the raw bKGD chunk
			 * data regardless of any bit-depth transformations so
			 * we need to check the depth and adjust if necessary
			 */
			if(bit_depth == 16)
			{
				if(bg_color != NULL)
				{
					bg_color[0] = (u_int8_t)(pBackground->red >> 8);
					bg_color[1] = (u_int8_t)(pBackground->green >> 8);
					bg_color[2] = (u_int8_t)(pBackground->blue >> 8);
					bg_color[3] = ctx->default_alpha_value;
				}
			}
			else if((color_type == PNG_COLOR_TYPE_GRAY) && (bit_depth < 8))
			{
				if(bg_color != NULL)
				{
					if(bit_depth == 1)
						bg_color[0] = bg_color[1] = bg_color[2] =
							pBackground->gray ? 255 : 0;
					else if(bit_depth == 2)
						bg_color[0] = bg_color[1] = bg_color[2] =
							(255 / 3) * pBackground->gray;
					else			/* bit_depth == 4 */
						bg_color[0] = bg_color[1] = bg_color[2] =
							(255 / 15) * pBackground->gray;
					bg_color[3] = ctx->default_alpha_value;
				}
			}
			else			/* 8 bits */
			{
				if(bg_color != NULL)
				{
					bg_color[0] = (u_int8_t)pBackground->red;
					bg_color[1] = (u_int8_t)pBackground->green;
					bg_color[2] = (u_int8_t)pBackground->blue;
					bg_color[3] = ctx->default_alpha_value;
				}
			}
		}
	}
#endif	/* PNG_bKGD_SUPPORTED && PNG_INFO_bKGD */

	/* Set the interlace handling and get the number of expected
	 * interlace passes if using interlace
	 */
	ctx->png_ninterlace_passes = png_set_interlace_handling(png_ptr);

	/* Get the interlace type */
	ctx->png_interlace_type = interlace_type = png_get_interlace_type(
		png_ptr,
		png_info_ptr
	);

	/* Get the size of the PNG image */
	ctx->png_width = width = (int)png_get_image_width(png_ptr, png_info_ptr);
	ctx->png_height = height = (int)png_get_image_height(png_ptr, png_info_ptr);

	/* Calculate the PNG bytes per pixel */
	if(color_type & PNG_COLOR_MASK_COLOR)
	{
		if((color_type & PNG_COLOR_MASK_ALPHA) ||
		   (ctx->transparent_color_rgba != NULL)
		)
			ctx->png_bpp = 4;
		else
			ctx->png_bpp = 3;
	}
	else
	{
		if((color_type & PNG_COLOR_MASK_ALPHA) ||
		   (ctx->transparent_color_rgba != NULL)
		)
			ctx->png_bpp = 2;
		else
			ctx->png_bpp = 1;
	}

	/* Calculate the PNG bytes per line */
	bpl = png_get_rowbytes(
		png_ptr,
		png_info_ptr
	);
	/* png_get_rowbytes is not reliable, it should return the number
	 * of bytes needed to hold a transformed row on a PNG image but
	 * it does not take into account the expansion cases like
	 * paletted to RGBA specified by png_set_expand()
	 */
	if(bpl < (width * ctx->png_bpp))
		bpl = width * ctx->png_bpp;
	ctx->png_bpl = bpl;

#if defined(PNG_oFFs_SUPPORTED) && defined(PNG_INFO_oFFs)
	/* Get the offset */
	if(png_get_valid(
		png_ptr,
		png_info_ptr,
		PNG_INFO_oFFs
	) != 0x00000000)
	{
		int unit_type;
		png_int_32 x, y;
		if(png_get_oFFs(
			png_ptr,
			png_info_ptr,
			&x, &y,
			&unit_type
		) != 0)
		{
			switch(unit_type)
			{
			  case PNG_OFFSET_PIXEL:
				ctx->png_x = (int)x;
				ctx->png_y = (int)y;
				break;
			  case PNG_OFFSET_MICROMETER:
/* TODO Offset in micrometers (1/10^6 meter) */
				break;
			}
		}
	}
#endif	/* PNG_oFFs_SUPPORTED && PNG_INFO_oFFs */

	/* Allocate the PNG read buffer for reading image data returned
	 * by libpng in ImgPNGRowCB()
	 *
	 * If the image is interlaced then use an array of PNG row
	 * buffers, otherwise use a single PNG row buffer
	 */
#if defined(PNG_READ_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
	if(interlace_type == PNG_INTERLACE_ADAM7)
#else
	if(0)
#endif
	{
		int y;
		if(ctx->png_data != NULL)
		{
			for(y = 0; y < height; y++)
				free(ctx->png_data[y]);
			free(ctx->png_data);
		}
		ctx->png_data = (png_bytep *)malloc(height * sizeof(png_bytep));
		if(ctx->png_data != NULL)
		{
			for(y = 0; y < height; y++)
				ctx->png_data[y] = (png_bytep)malloc(ctx->png_bpl * sizeof(png_byte));
		}
	}
	else
	{
		if(ctx->png_row_buffer != NULL)
			free(ctx->png_row_buffer);
		ctx->png_row_buffer = (png_bytep)malloc(ctx->png_bpl * sizeof(png_byte));
	}

	/* Calculate the size of the target RGBA image data
	 *
	 * If opening as a thumb then limit the size of the target
	 * RGBA image data to the requested size
	 */
	if((ctx->req_width > 0) && (ctx->req_height > 0))
	{
		/* Calculate the size of the target RGBA image in order to
		 * accomidate for the thumb
		 */
		ImgCalculateThumbSize(
			width, height,			/* Original size */
			ctx->req_width, ctx->req_height,/* Thumb size limit */
			&ctx->data_width, &ctx->data_height
		);
		ctx->data_bpl = ctx->data_width * ctx->data_bpp;
	}
	else	/* IMG_PNG_MODE_OPEN */
	{
		ctx->data_width = width;
		ctx->data_height = height;
		ctx->data_bpl = ctx->data_bpp * ctx->data_width;
	}

	/* Allocate the target RGBA image data */
	buf_len = ctx->data_bpl * ctx->data_height;
	if(buf_len > 0)
	{
		if(ctx->data != NULL)
			ctx->data = (u_int8_t *)realloc(
				ctx->data,
				buf_len * sizeof(u_int8_t)
			);
		else if(ctx->progress_cb != NULL)
			ctx->data = (u_int8_t *)calloc(
				buf_len,
				sizeof(u_int8_t)
			);
		else
			ctx->data = (u_int8_t *)malloc(buf_len * sizeof(u_int8_t));
	}

	/* Transformations set, now begin reading by combining
	 * background with palette (this is required)
	 */
	png_start_read_image(png_ptr);
}

/*
 *	PNG library read row in RGBA callback.
 */
static void ImgReadPNGReadRowRGBACB(
	png_structp png_ptr,
	png_bytep new_row,
	png_uint_32 row_num,
	int pass
)
{
	int png_last_row;
	png_bytep	png_row_buffer,
			png_last_row_ptr;
	ImgPNGReadContext *ctx = IMG_PNG_READ_CONTEXT(
		png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Update the PNG read instance pointer on the context */
	ctx->png_ptr = png_ptr;

	/* Update the interlace pass number */
	ctx->png_interlace_pass = pass;

	/* Get the previous row index and pointer (might be NULL) */
	png_last_row = ctx->png_last_row;
	png_last_row_ptr = ctx->png_last_row_ptr;

	/* Increment the total number of rows encountered? */
	if((int)row_num >= ctx->png_encountered_rows)
		ctx->png_encountered_rows = MAX((int)row_num + 1, 0);


	/* Get the pointer to the PNG row buffer
	 *
	 * If the image is interlaced then use the PNG row buffer
	 * array, otherwise use the single PNG row buffer
	 */
#if defined(PNG_READ_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
	if(ctx->png_interlace_type == PNG_INTERLACE_ADAM7)
#else
	if(0)
#endif
	{
		const int png_y = (int)row_num;
		if((png_y >= 0) && (png_y < ctx->png_height) &&
		   (ctx->png_data != NULL)
		)
			png_row_buffer = ctx->png_data[png_y];
		else
			png_row_buffer = NULL;
	}
	else
	{
		png_row_buffer = ctx->png_row_buffer;
	}

	/* Insert/convert the PNG row into the RGBA image data */
	if((png_row_buffer != NULL) && (new_row != NULL))
	{
		const int png_y = (int)row_num;

		/* Opening as a thumb? */
		if((ctx->req_width > 0) && (ctx->req_height > 0))
		{
			if((png_y >= 0) && (png_y < ctx->png_height))
			{
				const int rgba_y = ctx->data_height * png_y /
					ctx->png_height;

				png_progressive_combine_row(
					png_ptr,
					png_row_buffer,
					new_row
				);

				if(rgba_y > ctx->data_y_open_thumb_last)
				{
					/* Insert this PNG row into the corresponding
					 * line in our target RGBA image data
					 */
					ImgReadPNGInsertPNGLineToRGBA(
						ctx->png_color_type,
						(const u_int8_t *)png_row_buffer,	/* Source PNG row */
						ctx->png_width,
						ctx->png_bpl,
						ctx->data,/* Target RGBA image data */
						ctx->data_width, ctx->data_height,
						ctx->data_bpl,
						rgba_y,/* Y coordinate in target */
						ctx->transparent_color_rgba,
						ctx->default_alpha_value
					);
					ctx->data_y_open_thumb_last = rgba_y;
				}
			}
		}
		else
		{
			if((png_y >= 0) && (png_y < ctx->png_height))
			{
				const int rgba_y = png_y;

				png_progressive_combine_row(
					png_ptr,
					png_row_buffer,
					new_row
				);

				/* Insert this PNG row into the corresponding
				 * line in our target RGBA image data
				 */
				ImgReadPNGInsertPNGLineToRGBA(
					ctx->png_color_type,
					(const u_int8_t *)png_row_buffer,/* Source PNG row */
					ctx->png_width,
					ctx->png_bpl,
					ctx->data,/* Target RGBA image data */
					ctx->data_width, ctx->data_height,
					ctx->data_bpl,
					rgba_y,	/* Y coordinate in target */
					ctx->transparent_color_rgba,
					ctx->default_alpha_value
				);
			}
		}
	}

	/* Update last row index and pointer with the current row
	 * index and pointer
	 */
	ctx->png_last_row = (int)row_num;
	ctx->png_last_row_ptr = new_row;
}

/*
 *	PNG library read end callback.
 */
static void ImgReadPNGEndCB(png_structp png_ptr, png_infop png_info_ptr)
{
	ImgPNGReadContext *ctx = IMG_PNG_READ_CONTEXT(
		png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Update the PNG read instance pointes */
	ctx->png_ptr = png_ptr;
	ctx->png_info_ptr = png_info_ptr;

	/* Mark that the PNG reading has been completed */
	ctx->operation_completed = 1;
}

/*
 *	Insert/converts a PNG row into the RGBA image data.
 *
 *	The png_color_type specifies the PNG color type, one of
 *	PNG_COLOR_TYPE_* or any of PNG_COLOR_MASK_*.
 *
 *	The png_row_data, png_width, and png_bpl, specifies the PNG
 *	row data to insert.
 *
 *	The rgba, rgba_width, rgba_height, and rgba_bpl specifies the
 *	target RGBA image data.
 *
 *	The rgba_y specifies the Y coordinate in the RGBA image data
 *	to insert the PNG row at.
 *
 *	The transparent_color_rgba specifies a 4 byte RGBA transparent
 *	color. This is used as a hint to determine if there is an
 *	alpha channel when png_color_type is set to
 *	PNG_COLOR_TYPE_PALETTE.
 *
 *	The def_alpha_value specifies the default alpha value.
 */
static void ImgReadPNGInsertPNGLineToRGBA(
	const int png_color_type,
	const u_int8_t *png_row_data,
	const int png_width,
	const int png_bpl,
	u_int8_t *rgba,
	const int rgba_width, const int rgba_height,
	const int rgba_bpl,
	const int rgba_y,
	u_int8_t *transparent_color_rgba,
	const u_int8_t def_alpha_value
)
{
	const int	tar_bpp = 4,		/* RGBA */
			tar_width = rgba_width,
			src_width = png_width;
	int src_bpp;
	u_int8_t	*tar_row = rgba + (rgba_y * rgba_bpl),
			*tar_ptr;
	const u_int8_t	*src_row = png_row_data,
			*src_end,
			*src_ptr;

	/* Determine the format of the PNG row data
	 *
	 * The PNG row data should be expanded or reduced to 8 bits
	 * from its original bit depth on file
	 *
	 * If transparent_color_rgba is not NULL then it suggests
	 * that the PNG row data is in RGBA or Greyscale Alpha
	 * format otherwise it should be RGB or Greyscale format
	 *
	 * Greyscale / Greyscale Alpha
	 */
	if((png_color_type == PNG_COLOR_TYPE_GRAY) ||
#ifdef PNG_COLOR_TYPE_GA
	   (png_color_type == PNG_COLOR_TYPE_GA)
#else
	   (png_color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
#endif
	)
	{
		/* Greyscale Alpha */
#ifdef PNG_COLOR_TYPE_GA
		if((png_color_type == PNG_COLOR_TYPE_GA) ||
#else
		if((png_color_type == PNG_COLOR_TYPE_GRAY_ALPHA) ||
#endif
		   (transparent_color_rgba != NULL)
		)
		{
			src_bpp = 2;
			src_end = src_row + (src_width * src_bpp);
			if(tar_width < src_width)
			{
				/* Copy/convert the PNG row to the RGBA thumb line */
				u_int8_t grey;
				int	sx,
					tx,
					tx_last = -1;
				for(sx = 0,
				    src_ptr = src_row;
				    sx < src_width;
				    sx++,
				    src_ptr += src_bpp
				)
				{
					tx = tar_width * sx / src_width;
					if(tx <= tx_last)
						continue;

					grey = src_ptr[0];
					tar_ptr = tar_row + (tx * tar_bpp);
					tar_ptr[0] = grey;
					tar_ptr[1] = grey;
					tar_ptr[2] = grey;
					tar_ptr[3] = src_ptr[1];

					tx_last = tx;
				}
			}
			else
			{
				/* Copy/convert the PNG row to the RGBA line */
				u_int8_t grey;
				src_ptr = src_row;
				tar_ptr = tar_row;
				while(src_ptr < src_end)
				{
					grey = src_ptr[0];
					tar_ptr[0] = grey;
					tar_ptr[1] = grey;
					tar_ptr[2] = grey;
					tar_ptr[3] = src_ptr[1];
					src_ptr += src_bpp;
					tar_ptr += tar_bpp;
				}
			}
		}
		/* Greyscale */
		else
		{
			src_bpp = 1;
			src_end = src_row + (src_width * src_bpp);
			if(tar_width < src_width)
			{
				/* Copy/convert the PNG row to the RGBA thumb line */
				u_int8_t grey;
				int	sx,
					tx,
					tx_last = -1;
				for(sx = 0,
				    src_ptr = src_row;
				    sx < src_width;
				    sx++,
				    src_ptr += src_bpp
				)
				{
					tx = tar_width * sx / src_width;
					if(tx <= tx_last)
						continue;

					grey = src_ptr[0];
					tar_ptr = tar_row + (tx * tar_bpp);
					tar_ptr[0] = grey;
					tar_ptr[1] = grey;
					tar_ptr[2] = grey;
					tar_ptr[3] = def_alpha_value;

					tx_last = tx;
				}
			}
			else
			{
				/* Copy/convert the PNG row to the RGBA line */
				u_int8_t grey;
				src_ptr = src_row;
				tar_ptr = tar_row;
				while(src_ptr < src_end)
				{
					grey = src_ptr[0];
					tar_ptr[0] = grey;
					tar_ptr[1] = grey;
					tar_ptr[2] = grey;
					tar_ptr[3] = def_alpha_value;
					src_ptr += src_bpp;
					tar_ptr += tar_bpp;
				}
			}
		}
	}
	/* RGB / RGBA */
	else if(((png_color_type == PNG_COLOR_TYPE_PALETTE) ||
		     (png_color_type == PNG_COLOR_TYPE_RGB)) ||
#ifdef PNG_COLOR_TYPE_RGBA
			 (png_color_type == PNG_COLOR_TYPE_RGBA)
#else
			 (png_color_type == PNG_COLOR_TYPE_RGB_ALPHA)
#endif
	)
	{
		/* RGBA */
#ifdef PNG_COLOR_TYPE_RGBA
		if((png_color_type == PNG_COLOR_TYPE_RGBA) ||
#else
		if((png_color_type == PNG_COLOR_TYPE_RGB_ALPHA) ||
#endif
		   (transparent_color_rgba != NULL)
		)
		{
			src_bpp = 4;
			src_end = src_row + (src_width * src_bpp);
			if(tar_width < src_width)
			{
				/* Copy/convert the PNG row to the RGBA thumb line */
				int	sx,
					tx,
					tx_last = -1;
				for(sx = 0,
				    src_ptr = src_row;
				    sx < src_width;
				    sx++,
				    src_ptr += src_bpp
				)
				{
					tx = tar_width * sx / src_width;
					if(tx <= tx_last)
						continue;

					tar_ptr = tar_row + (tx * tar_bpp);
					*(u_int32_t *)tar_ptr = *(const u_int32_t *)src_ptr;

					tx_last = tx;
				}
			}
			else
			{
				/* Copy/convert the PNG row to the RGBA line */
				src_ptr = src_row;
				tar_ptr = tar_row;
				while(src_ptr < src_end)
				{
					*(u_int32_t *)tar_ptr = *(const u_int32_t *)src_ptr;
					src_ptr += src_bpp;
					tar_ptr += tar_bpp;
				}
			}
		}
		/* RGB */
		else
		{
			src_bpp = 3;
			src_end = src_row + (src_width * src_bpp);
			if(tar_width < src_width)
			{
				/* Copy/convert the PNG row to the RGBA thumb line */
				int	sx,
					tx,
					tx_last = -1;
				for(sx = 0,
				    src_ptr = src_row;
				    sx < src_width;
				    sx++,
				    src_ptr += src_bpp
				)
				{
					tx = tar_width * sx / src_width;
					if(tx <= tx_last)
						continue;

					tar_ptr = tar_row + (tx * tar_bpp);
					tar_ptr[0] = src_ptr[0];
					tar_ptr[1] = src_ptr[1];
					tar_ptr[2] = src_ptr[2];
					tar_ptr[3] = def_alpha_value;

					tx_last = tx;
				}
			}
			else
			{
				/* Copy/convert the PNG row to the RGBA line */
				src_ptr = src_row;
				tar_ptr = tar_row;
				while(src_ptr < src_end)
				{
					tar_ptr[0] = src_ptr[0];
					tar_ptr[1] = src_ptr[1];
					tar_ptr[2] = src_ptr[2];
					tar_ptr[3] = def_alpha_value;
					src_ptr += src_bpp;
					tar_ptr += tar_bpp;
				}
			}
		}
	}
}

/*
 *	Gets the PNG tEXt chunks
 */
static void ImgReadPNGGetTextChunks(ImgPNGReadContext *ctx)
{
#if defined(PNG_TEXT_SUPPORTED)
	int	i,
			nchunks;
	const char	*keyword,
			*value;
	png_textp	text,
			text_ptr;

	if(ctx->png_info_ptr == NULL)
		return;

	/* Get the text chunks */
	if(png_get_text(
		ctx->png_ptr,
		ctx->png_info_ptr,
		&text,
		&nchunks
	) == 0)
		return;

	for(i = 0; i < nchunks; i++)
	{
		text_ptr = &text[i];

		keyword = NULL;
		value = NULL;
		switch(text_ptr->compression)
		{
#ifdef PNG_TEXT_COMPRESSION_NONE
		  case PNG_TEXT_COMPRESSION_NONE:
			keyword = (const char *)text_ptr->key;
			value = (const char *)text_ptr->text;
			break;
#endif
#ifdef PNG_TEXT_COMPRESSION_zTXt
		  case PNG_TEXT_COMPRESSION_zTXt:
			break;
#endif
#if defined(PNG_iTXt_SUPPORTED)
#ifdef PNG_ITXT_COMPRESSION_NONE
		  case PNG_ITXT_COMPRESSION_NONE:
			break;
#endif
#ifdef PNG_ITXT_COMPRESSION_zTXt
		  case PNG_ITXT_COMPRESSION_zTXt:
			break;
#endif
#endif	/* PNG_iTXt_SUPPORTED */
		}
		if(STRISEMPTY(keyword) || STRISEMPTY(value))
			continue;

		/* Title */
		if(!strcasecmp(keyword, "Title") ||
		   !strcasecmp(keyword, "Name")
		)
		{
			if(ctx->title != NULL)
			{
				free(*ctx->title);
				*ctx->title = strdup(value);
			}
		}
		/* Author */
		else if(!strcasecmp(keyword, "Author") ||
				!strcasecmp(keyword, "Artist")
		)
		{
			if(ctx->author != NULL)
			{
				free(*ctx->author);
				*ctx->author = strdup(value);
			}
		}
		/* E-mail */
		else if(!strcasecmp(keyword, "E-mail"))
		{
			if(ctx->author != NULL)
			{
				/* Set author field as e-mail only if author
				 * field has not been set yet
				 */
				if(*ctx->author == NULL)
					*ctx->author = strdup(value);
			}
		}
		/* Creator */
		else if(!strcasecmp(keyword, "Creator") ||
				!strcasecmp(keyword, "Generator") ||
				!strcasecmp(keyword, "Software")
		)
		{
			if(ctx->creator != NULL)
			{
				free(*ctx->creator);
				*ctx->creator = strdup(value);
			}
		}
		/* Comment */
		else if(!strcasecmp(keyword, "Comment") ||
				!strcasecmp(keyword, "Comments")
		)
		{
			if(ctx->comments != NULL)
			{
				free(*ctx->comments);
				*ctx->comments = strdup(value);
			}
		}
		/* Description */
		else if(!strcasecmp(keyword, "Description"))
		{
			if(ctx->comments != NULL)
			{
				/* Set comments field as description only if the
				 * comments field was set yet
				 */
				if(*ctx->comments == NULL)
					*ctx->comments = strdup(value);
			}
		}
	}
#endif	/* PNG_TEXT_SUPPORTED */
}

/*
 *	Reads the PNG image data from a buffer or a stream to RGBA image
 *	data.
 */
static int ImgReadPNGRGBA(
	const ImgPNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	unsigned long bytes_processed;
	ImgPNGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the callback data */				\
 if(ctx != NULL) {					\
  /* Delete the PNG read structure and info pair */	\
  if(ctx->png_ptr != NULL) {				\
   png_destroy_read_struct(				\
    &ctx->png_ptr,					\
    (ctx->png_info_ptr != NULL) ?			\
     &ctx->png_info_ptr : NULL,				\
    (png_infopp)NULL		/* No end info */	\
   );							\
  }							\
							\
  /* Delete the PNG row buffer */			\
  free(ctx->png_row_buffer);				\
							\
  /* Delete the PNG image data */			\
  if(ctx->png_data != NULL) {				\
   const int height = ctx->png_height;			\
   int y;						\
   for(y = 0; y < height; y++)				\
    free(ctx->png_data[y]);				\
   free(ctx->png_data);					\
  }							\
							\
  /* Delete the loaded image data */			\
  free(ctx->data);					\
							\
  /* Delete the transparent color */			\
  free(ctx->transparent_color_rgba);			\
							\
  /* Delete the read buffer */				\
  free(ctx->io_buf);					\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	/* Create our PNG read context */
	ctx = IMG_PNG_READ_CONTEXT(calloc(1, sizeof(ImgPNGReadContext)));
	if(ctx == NULL)
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
#if 0
	ctx->fp_len = 0l;
	ctx->io_buf = NULL;
	ctx->io_buf_len = 0l;
	ctx->req_width = 0;			/* Open actual size */
	ctx->req_height = 0;
	ctx->png_ptr = NULL;
	ctx->png_info_ptr = NULL;
	ctx->png_x = 0;
	ctx->png_y = 0;
	ctx->png_width = 0;			/* Obtained in ImgReadPNGInfoCB() */
	ctx->png_height = 0;
	ctx->png_bpp = 0;
	ctx->png_bpl = 0;
	ctx->png_bit_depth = 0;
#endif
	ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	ctx->png_interlace_type = PNG_INTERLACE_NONE;
#if 0
	ctx->png_interlace_pass = 0;
	ctx->png_ninterlace_passes = 0;
#endif
	ctx->png_last_interlace_pass_reported = -1;
#if 0
	ctx->png_last_row_ptr = NULL;
	ctx->png_encountered_rows = 0;
	ctx->png_row_buffer = NULL;		/* Allocated in ImgReadPNGInfoCB() */
	ctx->png_data = NULL;			/* Allocated in ImgReadPNGInfoCB() */
	ctx->data_width = 0;
	ctx->data_height = 0;
#endif
	ctx->data_bpp = 4;			/* RGBA */
#if 0
	ctx->data_bpl = 0;			/* Calculate later */
	ctx->data = NULL;			/* Allocated in ImgReadPNGInfoCB() */
#endif
	ctx->data_y_open_thumb_last = -1;
#if 0
	ctx->transparent_color_rgba = NULL;	/* Allocated in ImgReadPNGInfoCB() */
#endif
	ctx->default_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->modified_time_sec = modified_time_sec_rtn;
/*	ctx->operation_completed = 0; */
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
/*	ctx->status = 0; */
	ctx->user_aborted = user_aborted;

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			0, 0,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		CLEANUP_RETURN(-4);
	}

	/* Check if the input was specified and get its statistics */
	switch(ctx->data_storage_type)
	{
	  case IMG_PNG_DATA_STORAGE_BUFFER:
		if((ctx->bp != NULL) && (ctx->bp_len > 0l))
		{
			ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;

			/* Check if the buffer data does not have a PNG signature */
			if(!ImgBufferIsPNG((const u_int8_t *)ctx->bp, (int)ctx->bp_len))
			{
				imgio_last_open_error = "Not a PNG image";
				errno = EINVAL;
				CLEANUP_RETURN(-2);
			}
		}
		else
		{
			imgio_last_open_error = "Buffer not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;

	  case IMG_PNG_DATA_STORAGE_STREAM:
		if(ctx->fp != NULL)
		{
			long fp_pos;
			struct stat stat_buf;

			/* Get the stream's statistics */
#if defined(_WIN32)
			ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;
#else
			if(fstat(fileno(ctx->fp), &stat_buf))
			{
				ctx->fp_len = 0l;
				ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;
			}
			else
			{
				ctx->fp_len = (unsigned long)stat_buf.st_size;
				ctx->io_buf_len = MAX(
					stat_buf.st_blksize,
					IMG_PNG_MIN_IO_SIZE
				);
			}
#endif
			/* Allocate the IO buffer */
			ctx->io_buf = (char *)malloc(ctx->io_buf_len * sizeof(char));
			if(ctx->io_buf == NULL)
			{
				imgio_last_open_error = "Memory allocation error";
				CLEANUP_RETURN(-3);
			}

			/* Check if the stream data does not have a PNG signature */
			fp_pos = ftell(ctx->fp);
			if(!ImgStreamIsPNG(ctx->fp))
			{
				(void)fseek(ctx->fp, fp_pos, SEEK_SET);
				imgio_last_open_error = "Not a PNG image";
				errno = EINVAL;
				CLEANUP_RETURN(-2);
			}
			(void)fseek(ctx->fp, fp_pos, SEEK_SET);
		}
		else
		{
			imgio_last_open_error = "Stream not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;
	}

	/* Create the libpng read structure */
	ctx->png_ptr = png_create_read_struct(
#ifdef PNG_LIBPNG_VER_STRING
		PNG_LIBPNG_VER_STRING,
#else
#warning "PNG_LIBPNG_VER_STRING (libpng version string) was not defined."
		"0.0.0",
#endif
		ctx,
		ImgReadPNGErrorCB,
		ImgReadPNGWarningCB
	);
	if(ctx->png_ptr == NULL)
	{
		imgio_last_open_error = "Unable to create a new libpng read structure";
		CLEANUP_RETURN(-3);
	}

	/* Create the libpng info structure for pairing with the
	 * libpng read structure
	 */
	ctx->png_info_ptr = png_create_info_struct(ctx->png_ptr);
	if(ctx->png_info_ptr == NULL)
	{
		imgio_last_open_error = "Unable to create a new libpng info structure";
		CLEANUP_RETURN(-3);
	}

	/* Set the libpng error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libpng's functions encounters an error, libpng will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(png_jmpbuf(ctx->png_ptr)))
	{
		const int status = ctx->status;
		CLEANUP_RETURN(status);
	}

	/* Set the libpng progressive read callbacks */
	png_set_progressive_read_fn(
		ctx->png_ptr,
		ctx,				/* User data */
		ImgReadPNGInfoCB,		/* Read info callback */
		ImgReadPNGReadRowRGBACB,	/* Read row callback */
		ImgReadPNGEndCB			/* Read end callback */
	);

	/* Begin reading the PNG image */
	do
	{
		/* Get the data from the specified input and give it to
		 * libpng for processing
		 */
		bytes_processed = 0l;
		switch(ctx->data_storage_type)
		{
		  case IMG_PNG_DATA_STORAGE_BUFFER:
			if(ctx->bp_pos < ctx->bp_len)
			{
				/* Calculate the number of bytes to read */
				const size_t bytes_to_read = (size_t)(ctx->bp_len - ctx->bp_pos);

				/* Give this segment of the input buffer to
				 * libpng
				 */
				png_process_data(
					ctx->png_ptr,
					ctx->png_info_ptr,
					(png_bytep)(ctx->bp + ctx->bp_pos),
				    (png_size_t)bytes_to_read
				);

		        /* Increment the input buffer position */
		        ctx->bp_pos += bytes_to_read;

				/* Record the number of bytes processed on this iteration */
				bytes_processed = (unsigned long)bytes_to_read;
			}
			break;

		  case IMG_PNG_DATA_STORAGE_STREAM:
			if((ctx->fp != NULL) && (ctx->io_buf != NULL))
			{
				const size_t	units_to_read = (size_t)ctx->io_buf_len / sizeof(char),
				units_read = fread(
					ctx->io_buf,
					sizeof(char),
					units_to_read,
					ctx->fp
				);
				if(units_read != units_to_read)
				{
					/* Check for read error */
					if(ferror(ctx->fp))
						ImgOpenSetError(strerror(errno));
				}

				/* Record the number of bytes processed on this iteration */
				bytes_processed = (unsigned long)(units_read * sizeof(char));

				/* Give the data read from the stream to libpng */
				png_process_data(
					ctx->png_ptr,
					ctx->png_info_ptr,
					ctx->io_buf,
					(png_size_t)bytes_processed
				);
			}
			break;
		}

		/* Report the progress */
		if(ctx->progress_cb != NULL)
		{
			/* Interlaced? */
#if defined(PNG_READ_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
			if(ctx->png_interlace_type == PNG_INTERLACE_ADAM7)
#else
			if(0)
#endif
			{
				/* Report each interlace pass */
				if(ctx->png_interlace_pass > ctx->png_last_interlace_pass_reported)
				{
					if(!ctx->progress_cb(
						ctx->progress_data,
						ctx->png_interlace_pass, ctx->png_ninterlace_passes,
						ctx->data_width, ctx->data_height,
						ctx->data_bpl, ctx->data_bpp,
						ctx->data
					))
					{
						*ctx->user_aborted = 1;
						break;
					}
					ctx->png_last_interlace_pass_reported = ctx->png_interlace_pass;
				}
			}
			else
			{
				/* Report progressive (non-interlaced) */
				const int i = ctx->png_encountered_rows;
				if((i % IMG_PNG_PROGRESS_RESOLUTION) == 0)
				{
					if(!ctx->progress_cb(
						ctx->progress_data,
						i, ctx->png_height,
						ctx->data_width, ctx->data_height,
						ctx->data_bpl, ctx->data_bpp,
						ctx->data
					))
					{
						*ctx->user_aborted = 1;
						break;
					}
				}
			}
		}

	} while(!ctx->operation_completed && (bytes_processed > 0l));

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			ctx->data_height, ctx->data_height,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
		*width_rtn = ctx->data_width;
	if(height_rtn != NULL)
		*height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
		*bpl_rtn = ctx->data_bpl;
	if(x_rtn != NULL)          
		*x_rtn = ctx->png_x;
	if(y_rtn != NULL)          
		*y_rtn = ctx->png_y;
	if(base_width_rtn != NULL)
		*base_width_rtn = ctx->png_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = ctx->png_height;

	if(rgba_rtn != NULL)
	{
		/* Transfer the target RGBA image data to the return value */
		free(*rgba_rtn);
		*rgba_rtn = ctx->data;
		ctx->data = NULL;
	}

	if(*ctx->user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		CLEANUP_RETURN(-4);
	}
	else
	{
		const int status = ctx->status;
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the PNG image data from a buffer to RGBA image data.
 */
int ImgBufferReadPNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadPNGRGBA(
		IMG_PNG_DATA_STORAGE_BUFFER,
		bp, bp_len,
		NULL,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	));
}

/*
 *	Reads the PNG image data from a stream to RGBA image data.
 */
int ImgStreamReadPNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadPNGRGBA(
		IMG_PNG_DATA_STORAGE_STREAM,
		NULL, 0l,
		fp,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	));
}

/*
 *	Opens the PNG image file to RGBA image data.
 */
int ImgFileOpenPNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	u_int8_t def_alpha_value,
	ImgProgressFunc	progress_cb, void *progress_data,
	int *user_aborted
)
{
	int             status,
					error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		return(-4);
	}

	if(STRISEMPTY(path))
	{
		imgio_last_open_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Open the PNG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
		ImgOpenSetError(strerror(errno));
		return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadPNGRGBA(
		fp,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

/*
 *	Reads the PNG image data from a buffer or a stream to RGBA
 *	image data as a thumb.
 */
static int ImgReadPNGRGBAThumb(
	const ImgPNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	unsigned long bytes_processed;
	ImgPNGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the callback data */				\
 if(ctx != NULL) {					\
  /* Delete the PNG read structure and info pair */	\
  if(ctx->png_ptr != NULL) {				\
   png_destroy_read_struct(				\
    &ctx->png_ptr,					\
    (ctx->png_info_ptr != NULL) ?			\
     &ctx->png_info_ptr : NULL,				\
    (png_infopp)NULL		/* No end info */	\
   );							\
  }							\
							\
  /* Delete the PNG row buffer */			\
  free(ctx->png_row_buffer);				\
							\
  /* Delete the PNG image data */			\
  if(ctx->png_data != NULL) {				\
   const int height = ctx->png_height;			\
   int y;						\
   for(y = 0; y < height; y++)				\
    free(ctx->png_data[y]);				\
   free(ctx->png_data);					\
  }							\
							\
  /* Delete the loaded image data */			\
  free(ctx->data);					\
							\
  /* Delete the transparent color */			\
  free(ctx->transparent_color_rgba);			\
							\
  /* Delete the IO buffer */				\
  free(ctx->io_buf);					\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	/* Create our PNG read context */
	ctx = IMG_PNG_READ_CONTEXT(calloc(1, sizeof(ImgPNGReadContext)));
	if(ctx == NULL)
	{
		imgio_last_open_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
/*	ctx->fp_len = 0l; */
	ctx->req_width = req_width;		/* Open as a thumb */
	ctx->req_height = req_height;
#if 0
	ctx->io_buf = NULL;
	ctx->io_buf_len = 0l;
	ctx->png_ptr = NULL;
	ctx->png_info_ptr = NULL;
	ctx->png_x = 0;
	ctx->png_y = 0;
	ctx->png_width = 0;			/* Obtained in ImgReadPNGInfoCB() */
	ctx->png_height = 0;
	ctx->png_bpp = 0;
	ctx->png_bpl = 0;
	ctx->png_bit_depth = 0;
#endif
	ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	ctx->png_interlace_type = PNG_INTERLACE_NONE;
#if 0
	ctx->png_interlace_pass = 0;
	ctx->png_ninterlace_passes = 0;
#endif
	ctx->png_last_interlace_pass_reported = -1;
#if 0
	ctx->png_last_row_ptr = NULL;
	ctx->png_encountered_rows = 0;
	ctx->png_row_buffer = NULL;		/* Allocated in ImgReadPNGInfoCB() */
	ctx->png_data = NULL;			/* Allocated in ImgReadPNGInfoCB() */
	ctx->data_width = 0;
	ctx->data_height = 0;
#endif
	ctx->data_bpp = 4;			/* RGBA */
#if 0
	ctx->data_bpl = 0;			/* Calculate later */
	ctx->data = NULL;			/* Allocated in ImgReadPNGInfoCB() */
#endif
	ctx->data_y_open_thumb_last = -1;
#if 0
	ctx->transparent_color_rgba = NULL;	/* Allocated in ImgReadPNGInfoCB() */
#endif
	ctx->default_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->modified_time_sec = modified_time_sec_rtn;
/*	ctx->operation_completed = 0; */
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = user_aborted;
/*	ctx->status = 0; */

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			0, 0,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		CLEANUP_RETURN(-4);
	}

	/* Check if the input was specified and get its statistics */
	switch(ctx->data_storage_type)
	{
	    case IMG_PNG_DATA_STORAGE_BUFFER:
		if((ctx->bp != NULL) && (ctx->bp_len > 0l))
		{
			ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;

			/* Check if the buffer data does not have a PNG signature */
			if(!ImgBufferIsPNG((const u_int8_t *)ctx->bp, (int)ctx->bp_len))
			{
				imgio_last_open_error = "Not a PNG image";
				errno = EINVAL;
				CLEANUP_RETURN(-2);
			}
		}
		else
		{
			imgio_last_open_error = "Buffer not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;

	    case IMG_PNG_DATA_STORAGE_STREAM:
		if(ctx->fp != NULL)
		{
			long fp_pos;
			struct stat stat_buf;

			/* Get the stream's statistics */
#if defined(_WIN32)
			ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;
#else
			if(fstat(fileno(ctx->fp), &stat_buf))
			{
				ctx->fp_len = 0l;
				ctx->io_buf_len = IMG_PNG_DEF_IO_SIZE;
			}
			else
			{
				ctx->fp_len = (unsigned long)stat_buf.st_size;
				ctx->io_buf_len = MAX(
					stat_buf.st_blksize,
					IMG_PNG_MIN_IO_SIZE
				);
			}
#endif
			/* Allocate the IO buffer */
			ctx->io_buf = (char *)malloc(ctx->io_buf_len * sizeof(char));
			if(ctx->io_buf == NULL)
			{
				imgio_last_open_error = "Memory allocation error";
				CLEANUP_RETURN(-3);
			}

			/* Check if the stream data's signature is a PNG image */
			fp_pos = ftell(ctx->fp);
			if(!ImgStreamIsPNG(ctx->fp))
			{
				(void)fseek(ctx->fp, fp_pos, SEEK_SET);
				imgio_last_open_error = "Not a PNG image";
				errno = EINVAL;
				CLEANUP_RETURN(-2);
			}
			(void)fseek(ctx->fp, fp_pos, SEEK_SET);
		}
		else
		{
			imgio_last_open_error = "Stream not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;
	}

	/* Create the libpng read structure */
	ctx->png_ptr = png_create_read_struct(
#ifdef PNG_LIBPNG_VER_STRING
		PNG_LIBPNG_VER_STRING,
#else
#warning "PNG_LIBPNG_VER_STRING (libpng version string) was not defined."
		"0.0.0",
#endif
		ctx,
		ImgReadPNGErrorCB,
		ImgReadPNGWarningCB
	);
	if(ctx->png_ptr == NULL)
	{
		imgio_last_open_error = "Unable to create a new libpng read structure";
		CLEANUP_RETURN(-3);
	}

	/* Create the libpng info structure for pairing with the
	 * libpng read structure
	 */
	ctx->png_info_ptr = png_create_info_struct(ctx->png_ptr);
	if(ctx->png_info_ptr == NULL)
	{
		imgio_last_open_error = "Unable to create a new libpng info structure";
		CLEANUP_RETURN(-3);
	}

	/* Set the libpng error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libpng's functions encounters an error, libpng will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(png_jmpbuf(ctx->png_ptr)))
	{
		const int status = ctx->status;
		CLEANUP_RETURN(status);
	}

	/* Set the libpng progressive read callbacks */
	png_set_progressive_read_fn(
		ctx->png_ptr,
		ctx,				/* User data */
		ImgReadPNGInfoCB,		/* Read info callback */
		ImgReadPNGReadRowRGBACB,	/* Read row callback */
		ImgReadPNGEndCB			/* Read end callback */
	);

	/* Begin reading the PNG image */
	do
	{
		/* Get the data from the specified input and give it to
		 * libpng for processing
		 */
		bytes_processed = 0l;
		switch(ctx->data_storage_type)
		{
		  case IMG_PNG_DATA_STORAGE_BUFFER:
			if(ctx->bp_pos < ctx->bp_len)
			{
				/* Calculate the number of bytes to read */
				const size_t bytes_to_read = (size_t)(ctx->bp_len - ctx->bp_pos);

				/* Give this segment of the input buffer to
				 * libpng
				 */
				png_process_data(
					ctx->png_ptr,
					ctx->png_info_ptr,
					(png_bytep)(ctx->bp + ctx->bp_pos),
					(png_size_t)bytes_to_read
				);

				/* Increment the input buffer position */
				ctx->bp_pos += bytes_to_read;

				/* Record the number of bytes processed on this iteration */
				bytes_processed = (unsigned long)bytes_to_read;
			}
			break;

		  case IMG_PNG_DATA_STORAGE_STREAM:
			if((ctx->fp != NULL) && (ctx->io_buf != NULL))
			{
				const size_t	units_to_read = (size_t)ctx->io_buf_len / sizeof(char),
						units_read = fread(
					ctx->io_buf,
					sizeof(char),
					units_to_read,
					ctx->fp
				);
				if(units_read != units_to_read)
				{
					/* Check for read error */
					if(ferror(ctx->fp))
						ImgOpenSetError(strerror(errno));
				}

				/* Record the number of bytes processed on this iteration */
				bytes_processed = (unsigned long)(units_read * sizeof(char));

				/* Give the data read from the stream to libpng */
				png_process_data(
					ctx->png_ptr,
					ctx->png_info_ptr,
					ctx->io_buf,
					(png_size_t)bytes_processed
				);
			}
			break;
		}

		/* Report the progress */
		if(ctx->progress_cb != NULL)
		{
			/* Interlaced? */
#if defined(PNG_READ_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
			if(ctx->png_interlace_type == PNG_INTERLACE_ADAM7)
#else
			if(0)
#endif
			{
				/* Report each interlace pass */
				if(ctx->png_interlace_pass > ctx->png_last_interlace_pass_reported)
				{
					if(!ctx->progress_cb(
						ctx->progress_data,
						ctx->png_interlace_pass, ctx->png_ninterlace_passes,
						ctx->data_width, ctx->data_height,
						ctx->data_bpl, ctx->data_bpp,
						ctx->data
					))
					{
						*ctx->user_aborted = 1;
						break;
					}
					ctx->png_last_interlace_pass_reported = ctx->png_interlace_pass;
				}
			}
			else
			{
				/* Report progressive (non-interlaced) */
				const int i = ctx->png_encountered_rows;
				if((i % IMG_PNG_PROGRESS_RESOLUTION) == 0)
				{
					if(!ctx->progress_cb(
						ctx->progress_data,
						i, ctx->png_height,
						ctx->data_width, ctx->data_height,
						ctx->data_bpl, ctx->data_bpp,
						ctx->data
					))
					{
						*ctx->user_aborted = 1;
						break;
					}
				}
			}
		}

	} while(!ctx->operation_completed && (bytes_processed > 0l));

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			ctx->data_height, ctx->data_height,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
		*width_rtn = ctx->data_width;
	if(height_rtn != NULL)
		*height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
		*bpl_rtn = ctx->data_bpl;
	if(orig_width_rtn != NULL)
		*orig_width_rtn = ctx->png_width;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = ctx->png_height;
	if(x_rtn != NULL)          
		*x_rtn = ctx->png_x;
	if(y_rtn != NULL)          
		*y_rtn = ctx->png_y;
	if(base_width_rtn != NULL)
		*base_width_rtn = ctx->png_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = ctx->png_height;

	if(rgba_rtn != NULL)
	{
		/* Transfer the target RGBA image data to the return value */
		free(*rgba_rtn);
		*rgba_rtn = ctx->data;
		ctx->data = NULL;
	}

	if(*ctx->user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		CLEANUP_RETURN(-4);
	}
	else
	{
		const int status = ctx->status;
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the PNG image data from a buffer to RGBA image data
 *	as a thumb.
 */
int ImgBufferReadPNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadPNGRGBAThumb(
		IMG_PNG_DATA_STORAGE_BUFFER,
		bp, bp_len,
		NULL,
		req_width, req_height,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		orig_width_rtn, orig_height_rtn,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	));
}

/*
 *	Reads the PNG image data from a stream to RGBA image data
 *	as a thumb.
 */
int ImgStreamReadPNGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadPNGRGBAThumb(
		IMG_PNG_DATA_STORAGE_STREAM,
		NULL, 0l,
		fp,
		req_width, req_height,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		orig_width_rtn, orig_height_rtn,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	));
}

/*
 *	Opens PNG image file to RGBA image data as a thumb.
 */
int ImgFileOpenPNGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int             status,
			error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		return(-4);
	}

	if(STRISEMPTY(path))
	{
		imgio_last_open_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Open the PNG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
		ImgOpenSetError(strerror(errno));
		return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadPNGRGBAThumb(
		fp,
		req_width, req_height,
		width_rtn, height_rtn,
		bpl_rtn,
		rgba_rtn,
		bg_color,
		orig_width_rtn, orig_height_rtn,
		x_rtn, y_rtn,
		base_width_rtn, base_height_rtn,
		creator_rtn, title_rtn,
		author_rtn, comments_rtn,
		modified_time_sec_rtn,
		def_alpha_value,
		progress_cb, progress_data,
		user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	Set write error message callback.
 */
static void ImgWritePNGErrorCB(png_structp png_ptr, png_const_charp msg)
{
	ImgPNGWriteContext *ctx = IMG_PNG_WRITE_CONTEXT(
		png_get_error_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Set the error message from libpng only if no other error
	 * message has been set yet
	 */
	if(imgio_last_save_error == NULL)
		ImgSaveSetError(msg);

	if(ctx->status == 0)
		ctx->status = -1;

	/* Return control to the setjmp point (this function does not
	 * return to libpng)
	 */
	longjmp(png_jmpbuf(ctx->png_ptr), 1);
}

/*
 *	Set write warning message callback.
 */
static void ImgWritePNGWarningCB(png_structp png_ptr, png_const_charp msg)
{
	ImgPNGWriteContext *ctx = IMG_PNG_WRITE_CONTEXT(
		png_get_error_ptr(png_ptr)
	);
	if(ctx == NULL)
		return;

	/* Do nothing */
}

/*
 *	Write calblack.
 */
static void ImgWritePNGWriteCB(
	png_structp png_ptr,
	png_bytep data, png_size_t length
)
{
	ImgPNGWriteContext *ctx = IMG_PNG_WRITE_CONTEXT(png_get_io_ptr(png_ptr));
	if(ctx == NULL)
		return;

	if((data == NULL) || (length == 0))
		return;

	switch(ctx->data_storage_type)
	{
	  case IMG_PNG_DATA_STORAGE_BUFFER:
		if((ctx->bp != NULL) && (ctx->bp_len != NULL))
		{
			/* Append all the bytes provided to the output buffer
			 * in one iteration
			 */
			const unsigned long	bytes_to_write = (unsigned long)length * sizeof(u_int8_t),
						bp_pos = *ctx->bp_len;

			/* Increase the allocation */
			*ctx->bp_len = bp_pos + bytes_to_write;
			*ctx->bp = (void *)realloc(
				*ctx->bp,
				(*ctx->bp_len) * sizeof(u_int8_t)
			);
			if(*ctx->bp == NULL)
			{
				*ctx->bp_len = 0l;
				imgio_last_save_error = "Memory allocation error";
				ctx->status = -3;
				return;
			}

			/* Copy/append the data */
			(void)memcpy(
				(*ctx->bp) + bp_pos,
				data,
				bytes_to_write
			);
		}
		break;

	  case IMG_PNG_DATA_STORAGE_STREAM:
		if((ctx->fp != NULL) && (ctx->io_size > 0l))
		{
			FILE *fp = ctx->fp;
			png_size_t length_left = length;
			const size_t io_size = ctx->io_size;
			size_t	units_to_write,
					units_written;
			png_bytep data_ptr = data;
			while(length_left > 0)
			{
				/* Calculate the number of units to write, which
				 * will be no more than the IO size
				 */
				units_to_write = MIN(
					length_left,
					io_size
				) / sizeof(u_int8_t);

				/* Write this block */
				units_written = fwrite(
					data_ptr,
					sizeof(u_int8_t),
					units_to_write,
					fp
				);

				/* Increment the number of bytes left to write
				 * and the libpng buffer pointer
				 */
				length_left -= units_written * sizeof(u_int8_t);
				data_ptr += units_written * sizeof(u_int8_t);

				/* Check for any write errors */
				if(units_written != units_to_write)
				{
					ImgSaveSetError(strerror(errno));
					ctx->status = -1;
					return;
				}
			}
		}
		break;
	}
}

/*
 *	Write flush callback.
 */
static void ImgWritePNGFlushCB(png_structp png_ptr)
{
	ImgPNGWriteContext *ctx = IMG_PNG_WRITE_CONTEXT(png_get_io_ptr(png_ptr));
	if(ctx == NULL)
		return;

	/* Do nothing */
}

/*
 *	Converts the RGBA line to Greyscale.
 */
static void ImgWritePNGConvertLineRGBAToGreyscale(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line,
	const u_int8_t *bg_color
)
{
	const int	src_bpp = 4,		/* RGBA */
			tar_bpp = 1;		/* Grayscale */
	const u_int8_t	*src_ptr = src_line,
					*src_end = src_ptr + (src_width * src_bpp),
					bg_grey = ImgConvertPixelRGBToGreyscale(bg_color);
	u_int8_t *tar_ptr = tar_line;
	while(src_ptr < src_end)
	{
		tar_ptr[0] = ImgConvertPixelRGBAToGreyscale(
			src_ptr,		/* Source RGBA */
			bg_grey			/* Background Greyscale */
		);
		src_ptr += src_bpp;
		tar_ptr += tar_bpp;
	}
}

/*
 *	Converts the RGBA line to Greyscale Alpha.
 */
static void ImgWritePNGConvertLineRGBAToGreyscaleAlpha(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line
)
{
	const int	src_bpp = 4,		/* RGBA */
			tar_bpp = 2;		/* Grayscale Alpha */
	const u_int8_t	*src_ptr = src_line,
					*src_end = src_ptr + (src_width * src_bpp);
	u_int8_t *tar_ptr = tar_line;
	while(src_ptr < src_end)
	{
		tar_ptr[0] = (u_int8_t)(
			(0.299 * src_ptr[0]) + (0.587 * src_ptr[1]) +
			(0.114 * src_ptr[2])
		);
		tar_ptr[1] = (u_int8_t)src_ptr[3];
		src_ptr += src_bpp;
		tar_ptr += tar_bpp;
	}
}

/*
 *	Converts the RGBA line to RGB.
 */
static void ImgWritePNGConvertLineRGBAToRGB(
	const u_int8_t *src_line,
	const int src_width,
	u_int8_t *tar_line,
	const u_int8_t *bg_color
)
{
	const int	src_bpp = 4,		/* RGBA */
			tar_bpp = 3;		/* RGB */
	const u_int8_t	*src_ptr = src_line,
					*src_end = src_ptr + (src_width * src_bpp);
	u_int8_t *tar_ptr = tar_line;
	while(src_ptr < src_end)
	{
		ImgConvertPixelRGBAToRGB(
			src_ptr,			/* Source RGBA */
			bg_color,			/* Background RGB */
			tar_ptr				/* Target RGB */
		);
		src_ptr += src_bpp;
		tar_ptr += tar_bpp;
	}
}

/*
 *	Writes the RGBA image data to a PNG image buffer or stream.
 */
static int ImgWritePNGRGBA(
	const ImgPNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : width * bpp;
	int user_aborted = 0;
	ImgPNGWriteContext *ctx;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the callback data */				\
 if(ctx != NULL) {					\
  /* Delete the PNG write structure and info pair */	\
  if(ctx->png_ptr != NULL) {				\
   png_destroy_write_struct(				\
    &ctx->png_ptr,					\
    (ctx->png_info_ptr != NULL) ?			\
     &ctx->png_info_ptr : NULL				\
   );							\
  }							\
							\
  /* Delete the PNG image data */			\
  if(ctx->png_data != NULL) {				\
   const int height = ctx->png_height;			\
   int y;						\
   for(y = 0; y < height; y++)				\
    free(ctx->png_data[y]);				\
   free(ctx->png_data);					\
  }							\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (rgba == NULL)
	)
	{
		imgio_last_save_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Create our PNG write context */
	ctx = IMG_PNG_WRITE_CONTEXT(calloc(1, sizeof(ImgPNGWriteContext)));
	if(ctx == NULL)
	{
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
#if 0
	ctx->io_size = 0l;
	ctx->png_ptr = NULL;
	ctx->png_info_ptr = NULL;
#endif
	ctx->png_x = x;
	ctx->png_y = y;
	ctx->png_width = width;
	ctx->png_height = height;
#if 0
	ctx->png_bpp = 0;			/* Set later */
	ctx->png_bpl = 0;			/* Set later */
#endif
	ctx->png_bit_depth = 8;			/* Always 8 bits when saving */
	ctx->png_color_type = PNG_COLOR_TYPE_GRAY;	/* Set later */
	ctx->png_interlace_type = PNG_INTERLACE_NONE;	/* Set later */
#if 0
	ctx->png_last_row_ptr = NULL;
	ctx->png_encountered_rows = 0;
	ctx->png_row_buffer = NULL;		/* Not used when saving */
	ctx->png_data = NULL;			/* Allocated later */
#endif
	ctx->data_width = width;
	ctx->data_height = height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = _bpl;
	ctx->data = rgba;
	ctx->bg_color = bg_color;
	ctx->creator = creator;
	ctx->title = title;
	ctx->author = author;
	ctx->comments = comments;
	ctx->modified_time_sec = modified_time_sec;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = &user_aborted;
/*	ctx->status = 0; */

	/* Set the PNG color type based on the specified format */
	switch(color_type)
	{
	  case 0:				/* Black & White */
	  case 1:				/* Greyscale */
		ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
		break;
	  case 2:				/* Greyscale Alpha */
		ctx->png_color_type =
#ifdef PNG_COLOR_TYPE_GA
			PNG_COLOR_TYPE_GA
#else
			PNG_COLOR_TYPE_GRAY_ALPHA
#endif
		;
		break;
	  case 3:				/* RGB */
		ctx->png_color_type = PNG_COLOR_TYPE_RGB;
		break;
	  case 4:				/* RGBA */
		ctx->png_color_type =
#ifdef PNG_COLOR_TYPE_RGBA
			PNG_COLOR_TYPE_RGBA
#else
			PNG_COLOR_TYPE_RGB_ALPHA
#endif
		;
		break;
	}

	/* Set the interlace type */
	if(interlaced)
		ctx->png_interlace_type =
#if defined(PNG_WRITE_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
			PNG_INTERLACE_ADAM7
#elif defined(PNG_INTERLACE_NONE)
			PNG_INTERLACE_NONE
#else
			0
#endif
		;
	else
		ctx->png_interlace_type =
#ifdef PNG_INTERLACE_NONE
			PNG_INTERLACE_NONE
#else
			0
#endif
		;

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			0, ctx->data_height,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
		imgio_last_save_error = "User aborted operation";
		errno = EINTR;
		return(-4);
	}

	/* Check if the output was specified and get its statistics */
	switch(ctx->data_storage_type)
	{
	  case IMG_PNG_DATA_STORAGE_BUFFER:
		if((ctx->bp != NULL) && (ctx->bp_len != NULL))
		{

		}
		else
		{
			imgio_last_save_error = "Output buffer not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;

	  case IMG_PNG_DATA_STORAGE_STREAM:
		if(ctx->fp != NULL)
		{
			struct stat stat_buf;
			FILE *fp = ctx->fp;

			/* Get the stream's statistics */
#if defined(_WIN32)
			ctx->io_size = IMG_PNG_DEF_IO_SIZE;
#else
			if(fstat(fileno(fp), &stat_buf))
			{
				ctx->io_size = IMG_PNG_DEF_IO_SIZE;
			}
			else
			{
				ctx->io_size = MAX(
					stat_buf.st_blksize,
					IMG_PNG_MIN_IO_SIZE
				);
			}
#endif
		}
		else
		{
			imgio_last_save_error = "Output stream not specified";
			errno = EINVAL;
			CLEANUP_RETURN(-2);
		}
		break;
	}

	/* Create the libpng write structure */
	ctx->png_ptr = png_create_write_struct(
#ifdef PNG_LIBPNG_VER_STRING
		PNG_LIBPNG_VER_STRING,
#else
#warning "PNG_LIBPNG_VER_STRING (libpng version string) was not defined."
		"0.0.0",
#endif
		ctx,
		ImgWritePNGErrorCB,
		ImgWritePNGWarningCB
	);
	if(ctx->png_ptr == NULL)
	{
		imgio_last_save_error = "Unable to create a new libpng write structure";
		CLEANUP_RETURN(-3);
	}

	/* Create the libpng info structure for pairing with the
	 * libpng write structure
	 */
	ctx->png_info_ptr = png_create_info_struct(ctx->png_ptr);
	if(ctx->png_info_ptr == NULL)
	{
		imgio_last_save_error = "Unable to create a new libpng info structure";
		CLEANUP_RETURN(-3);
	}

	/* Set the libpng error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libpng's functions encounters an error, libpng will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(png_jmpbuf(ctx->png_ptr)))
	{
		int status = ctx->status;
		CLEANUP_RETURN(status);
	}

	/* Set the libpng progressive read callbacks */
	png_set_write_fn(
		ctx->png_ptr,
		ctx,				/* User data */
		ImgWritePNGWriteCB,		/* Write callback */
		ImgWritePNGFlushCB		/* Flush callback */
	);

	/* Begin setting the PNG write parameters */

	/* Set the zlib compression level (0 to 9) */
	png_set_compression_level(
		ctx->png_ptr,
		CLIP(compression_level, 0, 9)
	);

#if 0
/* this is default for no filtering; Z_FILTERED is default otherwise: */
	png_set_compression_strategy(ctx->png_ptr, Z_DEFAULT_STRATEGY);
/* these are all defaults: */
	png_set_compression_mem_level(ctx->png_ptr, 8);
	png_set_compression_window_bits(ctx->png_ptr, 15);
	png_set_compression_method(ctx->png_ptr, 8);
#endif

	/* Begin setting the PNG chunks */

	/* Set the PNG IHDR (image header) chunk */
	png_set_IHDR(
		ctx->png_ptr,
		ctx->png_info_ptr,
		ctx->png_width, ctx->png_height,
		ctx->png_bit_depth,
		ctx->png_color_type,
		ctx->png_interlace_type,
		PNG_COMPRESSION_TYPE_DEFAULT,
		PNG_FILTER_TYPE_DEFAULT
	);

#if 0
/* #if defined(PNG_gAMA_SUPPORTED) && defined(PNG_FLOATING_POINT_SUPPORTED) */
	/* Set the PNG gAMA (gamma) chunk */
	if(gamma > 0.0f)
		png_set_gAMA(
			png_ptr,
			png_info_ptr,
			gamma
		);
#endif	/* PNG_gAMA_SUPPORTED && PNG_FLOATING_POINT_SUPPORTED */

#if defined(PNG_tIME_SUPPORTED)
	/* Set the PNG tIME (modified time) chunk */
	if(ctx->modified_time_sec > 0l)
	{
		png_time png_mtime;
		png_convert_from_time_t(
			&png_mtime,		/* PNG time */
			(time_t)ctx->modified_time_sec	/* Seconds since EPOCH */
		);
		png_set_tIME(
			ctx->png_ptr,
			ctx->png_info_ptr,
			&png_mtime
		);
	}
#endif	/* PNG_tIME_SUPPORTED */

#if defined(PNG_TEXT_SUPPORTED)
	/* Set the PNG tEXt (text) chunks */
	if(!(*ctx->user_aborted))
	{
#ifdef PNG_TEXT_COMPRESSION_NONE
		const int compression = PNG_TEXT_COMPRESSION_NONE;
#else
		const int compression = -1;
#endif
		int nchunks = 0;
		png_textp chunks_list = NULL;

#define APPEND_TEXT(_parameter_,_value_) {		\
 if(!STRISEMPTY(_value_)) {				\
  const int i = nchunks;				\
  nchunks = i + 1;					\
  chunks_list = (png_textp)realloc(			\
   chunks_list,						\
   nchunks * sizeof(png_text)				\
  );							\
  if(chunks_list != NULL) {				\
   png_textp t = &chunks_list[i];			\
   t->compression = compression;			\
   t->key = (png_charp)strdup(_parameter_);		\
   t->text = (png_charp)strdup(_value_);		\
   t->text_length = (png_size_t)strlen((char *)t->text);\
  } else {						\
   nchunks = 0;						\
  }							\
 }							\
}
		APPEND_TEXT(
			"Creator",
			ctx->creator
		);
		APPEND_TEXT(
			"Title",
			ctx->title
		);
		APPEND_TEXT(
			"Author",
			ctx->author
		);
		APPEND_TEXT(
			"Comment",
			ctx->comments
		);
#undef APPEND_TEXT
		if(chunks_list != NULL)
		{
			int i;
			png_textp t;

			if(nchunks > 0)
				png_set_text(
					ctx->png_ptr,
					ctx->png_info_ptr,
					chunks_list,
					nchunks
				);

			for(i = 0; i < nchunks; i++)
			{
				t = &chunks_list[i];
				free(t->key);
				free(t->text);
			}
			free(chunks_list);
		}
	}
#endif	/* PNG_TEXT_SUPPORTED */

#if defined(PNG_bKGD_SUPPORTED)
	/* Set the PNG bKGD (background color) chunk */
	if(ctx->bg_color != NULL)
	{
		const u_int8_t *c = ctx->bg_color;
		png_color_16 c16;

		/* Colormap color */
		c16.index = (png_byte)0;

		/* Set the 8 bit RGB background color */
		c16.red = (png_uint_16)c[0];
		c16.green = (png_uint_16)c[1];
		c16.blue = (png_uint_16)c[2];

		/* Set the 8 bit greyscale background color */
		c16.gray = (png_uint_16)ImgConvertPixelRGBToGreyscale(c);

		/* Set the raw PNG background color data
		 *
		 * Even though png_set_bKGD() uses a 16 bit pointer for
		 * the background color, we set the background color as
		 * 8 bits because png_set_bKGD() allows for raw data and
		 * the correct bit depth was set in the PNG's header
		 */
		png_set_bKGD(
			ctx->png_ptr,
			ctx->png_info_ptr,
			&c16
		);
	}
#endif	/* PNG_bKGD_SUPPORTED */

#if defined(PNG_oFFs_SUPPORTED)
	/* Set the PNG offsets (oFFs) chunk */
	if((ctx->png_x != 0) || (ctx->png_y != 0))
	{
		png_int_32	offset_x = (png_int_32)ctx->png_x,
					offset_y = (png_int_32)ctx->png_y;
		png_set_oFFs(
			ctx->png_ptr,
			ctx->png_info_ptr,
			offset_x, offset_y,
#ifdef PNG_OFFSET_PIXEL
			PNG_OFFSET_PIXEL
#else
			0			/* Offset in pixels */
#endif
		);
	}
#endif	/* PNG_oFFs_SUPPORTED */

	/* Write all of the PNG chunks set above before writing the
	 * IDAT chunk
	 */
	png_write_info(
		ctx->png_ptr,
		ctx->png_info_ptr
	);


	/* Set up the transformations:  for now, just pack low-bit-depth
	 * pixels into bytes (one, two or four pixels per byte)
	 */
	png_set_packing(ctx->png_ptr);
#if 0
	/* to scale low-bit-depth values */
	png_set_shift(ctx->png_ptr, &sig_bit);
#endif


	/* Begin writing the IDAT (image data) chunk */
	if(!(*ctx->user_aborted))
	{
		/* Save interlaced? */
#if defined(PNG_WRITE_INTERLACING_SUPPORTED) && defined(PNG_INTERLACE_ADAM7)
		if(ctx->png_interlace_type == PNG_INTERLACE_ADAM7)
#else
		if(0)
#endif
		{
			/* For interlacing we need to generate the entire
			 * PNG image data and write the entire image at once 
			 *
			 * Allocate the PNG row buffers for the PNG image data
			 */
			if(ctx->png_data != NULL)
			{
				int y;
				for(y = 0; y < ctx->png_height; y++)
					free(ctx->png_data[y]);
				free(ctx->png_data);
			}
			ctx->png_data = (png_bytep *)malloc(
				ctx->png_height * sizeof(png_bytep)
			);
			if(ctx->png_data != NULL)
			{
				const u_int8_t	def_bg_color[] = { 0x00, 0x00, 0x00, 0xFF };
				const int	src_width = ctx->data_width,
						src_height = ctx->data_height,
						src_bpp = ctx->data_bpp,
						src_bpl = ctx->data_bpl,
						tar_width = ctx->png_width;
				int	y,
					tar_bpp, tar_bpl;
				const u_int8_t	*src_data = ctx->data,	/* RGBA */
						*src_row,
						*bg_color_ptr = (bg_color != NULL) ? bg_color : def_bg_color;
				png_bytep	*tar_data = ctx->png_data,
						tar_row;


#define REPORT_PROGRESS	{			\
 /* Report progress from 0% to 49% */		\
 if((ctx->progress_cb != NULL) &&		\
    ((y % IMG_PNG_PROGRESS_RESOLUTION) == 0)	\
 ) {						\
  if(!ctx->progress_cb(				\
   ctx->progress_data,				\
   y, (2 * src_height),				\
   src_width, src_height,			\
   src_bpl, src_bpp,				\
   src_data					\
  )) {						\
   *ctx->user_aborted = 1;			\
   break;					\
  }						\
 }						\
}

				/* Copy/convert the specified RGBA image data to
				 * the libpng image data based on the specified
				 * color type
				 *
				 *	0	Black & White
				 *	1	Greyscale
				 *	2	Greyscale Alpha
				 *	3	RGB
				 *	4	RGBA
				 */
				switch(color_type)
				{
				    case 0:	/* Black & White */
				    case 1:	/* Greyscale */
					ctx->png_bpp = tar_bpp = 1;
					ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						tar_data[y] = tar_row = (png_bytep)malloc(tar_bpl);
						if(tar_row == NULL)
							continue;

						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToGreyscale(
							src_row,
							src_width,
							(u_int8_t *)tar_row,
							bg_color_ptr
						);
					}
					break;

				    case 2:	/* Greyscale Alpha */
					ctx->png_bpp = tar_bpp = 2;
					ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						tar_data[y] = tar_row = (png_bytep)malloc(tar_bpl);
						if(tar_row == NULL)
							continue;

						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToGreyscaleAlpha(
							src_row,
							src_width,
							(u_int8_t *)tar_row
						);
					}
					break;

				    case 3:	/* RGB */
					ctx->png_bpp = tar_bpp = 3;
					ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						tar_data[y] = tar_row = (png_bytep)malloc(tar_bpl);
						if(tar_row == NULL)
							continue;

						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToRGB(
							src_row,
							src_width,
							(u_int8_t *)tar_row,
							bg_color_ptr
						);
					}
					break;

				    case 4:	/* RGBA */
					ctx->png_bpp = tar_bpp = 4;
					ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						tar_data[y] = tar_row = (png_bytep)malloc(tar_bpl);
						if(tar_row == NULL)
							continue;

						REPORT_PROGRESS

						(void)memcpy(
							tar_row,
							src_row,
							src_width * src_bpp
						);
					}
					break;
				}
#undef REPORT_PROGRESS
			}

			/* Write the entire image interlaced from the PNG
			 * image data
			 */
			if(!(*ctx->user_aborted))
			{
				if(ctx->png_data != NULL)
					png_write_image(
						ctx->png_ptr,
						ctx->png_data
					);
			}

			/* Report progress */
			if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
			{
				if(!ctx->progress_cb(
					ctx->progress_data,
					(2 * ctx->png_height) - 1, 2 * ctx->png_height,
					ctx->data_width, ctx->data_height,
					ctx->data_bpl, ctx->data_bpp,
					ctx->data
				))
					*ctx->user_aborted = 1;
			}

			/* Delete the PNG image data */
			if(ctx->png_data != NULL)
			{
				int y;
				for(y = 0; y < ctx->png_height; y++)
					free(ctx->png_data[y]);
				free(ctx->png_data);
				ctx->png_data = NULL;
			}
		}
		else
		{
			/* Save progressive (non-interlaced) */
			const u_int8_t	def_bg_color[] = { 0x00, 0x00, 0x00, 0xFF };
			const int	src_width = ctx->data_width,
					src_height = ctx->data_height,
					src_bpp = ctx->data_bpp,
					src_bpl = ctx->data_bpl,
					tar_width = ctx->png_width;
			int	y,
				tar_bpp, tar_bpl;
			const u_int8_t	*src_data = ctx->data,	/* RGBA */
					*src_row,
					*bg_color_ptr = (bg_color != NULL) ? bg_color : def_bg_color;
			png_bytep	tar_row;

#define REPORT_PROGRESS	{			\
 /* Report progress from 0 to 99% */		\
 if((ctx->progress_cb != NULL) &&		\
    ((y % IMG_PNG_PROGRESS_RESOLUTION) == 0)	\
 ) {						\
  if(!ctx->progress_cb(				\
   ctx->progress_data,				\
   y, src_height,				\
   src_width, src_height,			\
   src_bpl, src_bpp,				\
   src_data					\
  )) {						\
   *ctx->user_aborted = 1;			\
   break;					\
  }						\
 }						\
}

			/* Convert and write each PNG row based on the
			 * specified color type
			 *
			 *	0	Black & White
			 *	1	Greyscale
			 *	2	Greyscale Alpha
			 *	3	RGB
			 *	4	RGBA
			 */
			switch(color_type)
			{
			    case 0:		/* Black & White */
			    case 1:		/* Greyscale */
				ctx->png_bpp = tar_bpp = 1;
				ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
				tar_row = (png_bytep)malloc(ctx->png_bpl);
				if(tar_row != NULL)
				{
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToGreyscale(
							src_row,
							src_width,
							(u_int8_t *)tar_row,
							bg_color_ptr
						);

						png_write_row(
							ctx->png_ptr,
							tar_row
						);
					}
					free(tar_row);
				}
				break;

			    case 2:		/* Greyscale Alpha */
				ctx->png_bpp = tar_bpp = 2;
				ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
				tar_row = (png_bytep)malloc(ctx->png_bpl);
				if(tar_row != NULL)
				{
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToGreyscaleAlpha(
							src_row,
							src_width,
							(u_int8_t *)tar_row
						);

						png_write_row(
							ctx->png_ptr,
							tar_row
						);
					}
					free(tar_row);
				}
				break;

			    case 3:		/* RGB */
				ctx->png_bpp = tar_bpp = 3;
				ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
				tar_row = (png_bytep)malloc(ctx->png_bpl);
				if(tar_row != NULL)
				{
					for(y = 0,
						src_row = src_data;
						y < src_height;
						y++,
						src_row += src_bpl
					)
					{
						REPORT_PROGRESS

						ImgWritePNGConvertLineRGBAToRGB(
							src_row,
							src_width,
							(u_int8_t *)tar_row,
							bg_color_ptr
						);

						png_write_row(
							ctx->png_ptr,
							tar_row
						);
					}
					free(tar_row);
				}
				break;

			    case 4:		/* RGBA */
				ctx->png_bpp = tar_bpp = 4;
				ctx->png_bpl = tar_bpl = tar_width * tar_bpp;
				for(y = 0,
					src_row = src_data;
					y < src_height;
					y++,
					src_row += src_bpl
				)
				{
					REPORT_PROGRESS

					/* Write the RGBA image data row since it is
					 * already in RGBA format
					 */
					png_write_row(
						ctx->png_ptr,
						(png_bytep)src_row
					);
				}
				break;
			}
#undef REPORT_PROGRESS
		}
	}

	/* End the writing of the PNG image file
	 *
	 * If we had any text or time info to write after the IDATs,
	 * second argument would be info_ptr, but we optimize slightly
	 * by sending a NULL pointer
	 */
	png_write_end(
		ctx->png_ptr,
		NULL				/* Do not write the info,
						 * we already did */
	);

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
		if(!ctx->progress_cb(
			ctx->progress_data,
			ctx->data_height, ctx->data_height,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			ctx->data
		))
			*ctx->user_aborted = 1;
	}

	if(*ctx->user_aborted)
	{
		imgio_last_save_error = "User aborted operation";
		errno = EINTR;
		CLEANUP_RETURN(-4);
	}
	else
	{
		const int status = ctx->status;
		CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the RGBA image data to a buffer as a PNG image.
 */
int ImgBufferWritePNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWritePNGRGBA(
		IMG_PNG_DATA_STORAGE_BUFFER,
		bp, bp_len,
		NULL,
		width, height,
		bpl,
		rgba,
		bg_color,
		x, y,
		base_width, base_height,
		creator, title, author, comments,
		modified_time_sec,
		compression_level,
		interlaced,
		color_type,
		progress_cb, progress_data
	));
}

/*
 *	Writes the RGBA image data to a stream as a PNG image.
 */
int ImgStreamWritePNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWritePNGRGBA(
		IMG_PNG_DATA_STORAGE_STREAM,
		NULL, NULL,
		fp,
		width, height,
		bpl,
		rgba,
		bg_color,
		x, y,
		base_width, base_height,
		creator, title, author, comments,
		modified_time_sec,
		compression_level,
		interlaced,
		color_type,
		progress_cb, progress_data
	));
}

/*
 *	Saves the RGBA image data to a PNG image file.
 */
int ImgFileSavePNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 or 1 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	FILE *fp;
	int		status,
			error_code;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if(STRISEMPTY(path))
	{
		imgio_last_save_error = "Invalid value";
		errno = EINVAL;
		return(-2);
	}

	/* Open/create the PNG image file for writing */
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
		ImgSaveSetError(strerror(errno));
		return(-1);
	}

	/* Write the stream */
	status = ImgStreamWritePNGRGBA(
		fp,
		width, height,
		bpl,
		rgba,
		bg_color,
		x, y,
		base_width, base_height,
		creator, title, author, comments,
		modified_time_sec,
		compression_level,
		interlaced,
		color_type,
		progress_cb, progress_data
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

#endif	/* HAVE_LIBPNG */
