#ifdef HAVE_LIBMNG
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <zlib.h>				/* libz */
#include <libmng.h>				/* libmng */
#if defined(_WIN32)
# include "../include/string.h"
#endif
#include "imgio.h"


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


typedef struct _ImgMNGReadContext	ImgMNGReadContext;
#define IMG_MNG_READ_CONTEXT(p)		((ImgMNGReadContext *)p)
typedef struct _ImgMNGWriteContext	ImgMNGWriteContext;
#define IMG_MNG_WRITE_CONTEXT(p)	((ImgMNGWriteContext *)p)


typedef enum {
	IMG_MNG_DATA_STORAGE_BUFFER,
	IMG_MNG_DATA_STORAGE_STREAM
} ImgMNGDataStorageType;

typedef enum {
	IMG_MNG_COLOR_TYPE_BW		= 0,	/* Black & White */
	IMG_MNG_COLOR_TYPE_GRAYSCALE	= 1,	/* Grayscale */
	IMG_MNG_COLOR_TYPE_GA		= 2,	/* Grayscale Alpha */
	IMG_MNG_COLOR_TYPE_RGB		= 3,	/* RGB */
	IMG_MNG_COLOR_TYPE_RGBA		= 4	/* RGBA */
} ImgMNGColorType;

typedef enum {
	IMG_JNG_COLOR_TYPE_GRAYSCALE	= 0,	/* Greyscale [Alpha] */
	IMG_JNG_COLOR_TYPE_COLOR	= 1	/* RGB[A] */
} ImgJNGColorType;


/* Version */
void ImgMNGVersion(int *major, int *minor, int *release);

/* Check Type */
int ImgBufferIsJNG(const u_int8_t *data, const int len);
int ImgBufferIsMNG(const u_int8_t *data, const int len);
int ImgStreamIsJNG(FILE *fp);
int ImgStreamIsMNG(FILE *fp);
int ImgFileIsJNG(const char *path);
int ImgFileIsMNG(const char *path);

/* Callbacks */
static mng_ptr ImgMNGNewCB(mng_size_t len);
static void ImgMNGDeleteCB(mng_ptr p, mng_size_t len);

/* Open */
static mng_bool ImgMNGReadErrorCB(
	mng_handle mng_h,
	mng_int32 error_code,
	mng_int8 severity_code,
	mng_chunkid chunk_id,
	mng_uint32 chunk_sequence,
	mng_int32 extra1,
	mng_int32 extra2,
	mng_pchar msg
);
static mng_bool ImgMNGReadOpenCB(mng_handle mng_h);
static mng_bool ImgMNGReadDataCB(
	mng_handle mng_h,
	mng_ptr buffer,
	mng_uint32 bytes_requested,
	mng_uint32p bytes_read
);
static mng_bool ImgMNGReadProcessHeaderCB(
	mng_handle mng_h,
	mng_uint32 width, mng_uint32 height
);
static mng_bool ImgMNGReadChunkIteratorCB(
	mng_handle mng_h,
	mng_handle chunk_h,
	mng_chunkid chunk_type,
	mng_uint32 chunk_index
);
static mng_bool ImgMNGReadSetTimerCB(
	mng_handle mng_h,
	mng_uint32 delay
);
static mng_uint32 ImgMNGReadGetTickCountCB(mng_handle mng_h);
static mng_ptr ImgMNGReadGetCanvasLineCB(
	mng_handle mng_h,
	mng_uint32 line
);
static mng_bool ImgMNGReadRefreshCB(
	mng_handle mng_h,
	mng_uint32 x, mng_uint32 y,
	mng_uint32 width, mng_uint32 height
);
static mng_bool ImgMNGReadProcessTextCB(
	mng_handle mng_h,
	mng_uint8 type,
	mng_pchar keyword,
	mng_pchar text,
	mng_pchar language,
	mng_pchar translation
);
static mng_bool ImgMNGReadCloseCB(mng_handle mng_h);
static int ImgReadMNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadMNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadMNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenMNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgReadMNGRGBAThumb(
	const ImgMNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadMNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadMNGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenMNGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);

/* Save */
static mng_bool ImgMNGWriteOpenCB(mng_handle mng_h);
static mng_bool ImgMNGWriteDataCB(
	mng_handle mng_h,
	mng_ptr buffer,
	mng_uint32 bytes_provided,
	mng_uint32p bytes_written
);
static mng_bool ImgMNGWriteCloseCB(mng_handle mng_h);
static int ImgJPEGWriteProgressCB(
	void *progress_data,
	const int i, const int n,
	const int width, const int height,
	const int bpl, const int bpp,
	const u_int8_t *image_data
);
static mng_ptr ImgMNGCompressDataZLib(
	ImgMNGWriteContext *ctx,
	const int compression_level,		/* 0 to 9 */
	mng_ptr data,
	const mng_uint32 length,
	mng_uint32 *length_rtn
);
static mng_ptr ImgMNGConvertRGBAToIDATGreyscale(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
);
static mng_ptr ImgMNGConvertRGBAToIDATGreyscaleAlpha(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
);
static mng_ptr ImgMNGConvertRGBAToIDATRGB(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
);
static mng_ptr ImgMNGConvertRGBAToIDATRGBA(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
);
static int ImgWriteJNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const ImgJNGColorType color_type,
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWriteJNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWriteJNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveJNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
static int ImgFileWriteMNGFrame(
	ImgMNGWriteContext *ctx,
	const int frame_num,
	const unsigned long prev_frame_delay_ms,
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const ImgMNGColorType color_type
);
static int ImgWriteMNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const ImgMNGColorType color_type,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWriteMNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWriteMNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveMNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Read Context:
 */
struct _ImgMNGReadContext {

	ImgMNGDataStorageType	data_storage_type;
	const void	*bp;
	unsigned long	bp_len,
			bp_pos;
	FILE		*fp;
	unsigned long	fp_len;
	unsigned long	io_size;		/* Read block size in bytes */

	int		req_width, req_height;	/* Positive if opening as a
						 * thumb */

	mng_handle	mng_h;
	mng_uint32	mng_simplicity_flags;	/* Specified by MHDR chunk */
	mng_uint8	mng_color_type;
	mng_uint32	mng_width, mng_height,
			mng_ticks_per_second,	/* Specified by MHDR chunk */
			mng_nlayers,		/* Specified by MHDR chunk */
			mng_nframes,		/* Specified by MHDR chunk */
			mng_total_play_time;	/* In ticks */

	int		mng_nihdrs;		/* Number of IHDR chunks encountered */
	unsigned long	mng_cur_play_time_ms,
			mng_total_play_time_ms;

	int		mng_drawing_buffer_bpp;
	u_int8_t	*mng_drawing_buffer;	/* mng_width * mng_height *
						 * mng_drawing_buffer_bpp */

	int		nframes,
			data_width, data_height,
			data_bpp, data_bpl;
	u_int8_t	**data_list;		/* Loaded image data frames
						 * list */
	unsigned long	*delay_ms_list;		/* Loaded delay in milliseconds
						 * list */

	u_int8_t	def_alpha_value;

	int		got_bg_color;
	u_int8_t	*bg_color;		/* 4 bytes RGBA */

	char		**creator,
			**title,
			**author,
			**comments;

	unsigned long	*modified_time_sec;	/* In seconds since EPOCH */

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		cur_progress_level,
			nprogress_levels;
	int		*user_aborted;
	int		status;
};

/*
 *	Write Context:
 */
struct _ImgMNGWriteContext {

	ImgMNGDataStorageType	data_storage_type;
	void		**bp;
	unsigned long	*bp_len,
			bp_pos;
	FILE		*fp;
	unsigned long	io_size;		/* write block size in bytes */

	mng_handle	mng_h;
	mng_uint8	mng_color_type;
	mng_uint32	mng_width, mng_height,
			mng_ticks_per_second,
			mng_nlayers,
			mng_nframes,
			mng_total_play_time;	/* In ticks */

	unsigned long	mng_total_play_time_ms;

	int		nframes,
			data_width, data_height,
			data_bpp, data_bpl;
	const u_int8_t	**data_list;		/* Image data frames list to
						 * save */
	const unsigned long	*delay_ms_list;	/* Delay in milliseconds list
						 * to save */

	const u_int8_t	*bg_color;		/* 4 bytes RGBA */

	const char	*creator,
			*title,
			*author,
			*comments;

	unsigned long	modified_time_sec;	/* In seconds since EPOCH */

	/* Progress callback */
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		cur_progress_level,
			nprogress_levels;
	int		*user_aborted;
	int		status;
};


/*
 *	Progress update resolution in lines per update):
 */
#define IMG_MNG_PROGRESS_RESOLUTION	10


/*
 *	Suggested IO Buffer Length in bytes):
 */
#define IMG_MNG_DEF_IO_SIZE		1024

/*
 *	Minimum IO Buffer Length in bytes):
 */
#define IMG_MNG_MIN_IO_SIZE		4


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (((MIN(MAX(a),(l)),(h)))
#define ABSOLUTE(x)	((((((x) < 0) ? x) * -1) : x))     

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? ((*s) == '\0') : 1)


/*
 *	Gets the MNG library's version.
 */
void ImgMNGVersion(int *major, int *minor, int *release)
{
	if(major != NULL)
#ifdef MNG_VERSION_MAJOR
	    *major = MNG_VERSION_MAJOR;
#else
	    *major = 0;
#endif
	if(minor != NULL)
#ifdef MNG_VERSION_MINOR
	    *minor = MNG_VERSION_MINOR;
#else
	    *minor = 0;
#endif
	if(release != NULL)
#ifdef MNG_VERSION_RELEASE
	    *release = MNG_VERSION_RELEASE;
#else
	    *release = 0;
#endif
}


/*
 *	Checks for the JNG type ID.
 */
int ImgBufferIsJNG(const u_int8_t *data, const int len)
{
	const u_int8_t signature[] = {
		0x8B,
		'J',
		'N',
		'G',
		0x0D,
		0x0A,
		0x1A,
		0x0A
	};
	const int signature_length = sizeof(signature) / sizeof(u_int8_t);

	if((data == NULL) || (len < signature_length))
	{
	    errno = EINVAL;
	    return(0);
	}

	if(memcmp(data, signature, signature_length) == 0)
	    return(1);

	errno = EINVAL;
	return(0);
}

/*
 *	Checks for the MNG type ID.
 */
int ImgBufferIsMNG(const u_int8_t *data, const int len)
{
	const u_int8_t signature[] = {
		0x8A,
		'M',
		'N',
		'G',
		0x0D,
		0x0A,
		0x1A,
		0x0A
	};
	const int signature_length = sizeof(signature) / sizeof(u_int8_t);

	if((data == NULL) || (len < signature_length))
	{
	    errno = EINVAL;
	    return(0);
	}

	if(memcmp(data, signature, signature_length) == 0)
	    return(1);

	errno = EINVAL;
	return(0);
}

int ImgStreamIsJNG(FILE *fp)
{
	u_int8_t buf[8];

	size_t units_read;

	if(fp == NULL)
	{
	    errno = EINVAL;
	    return(0);
	}

	units_read = fread(
	    buf,
	    sizeof(u_int8_t),
	    sizeof(buf) / sizeof(u_int8_t),
	    fp
	);
	if((units_read == sizeof(buf) / sizeof(u_int8_t)))
	    return(ImgBufferIsJNG(buf, sizeof(buf)));
	else
	    return(0);
}

int ImgStreamIsMNG(FILE *fp)
{
	u_int8_t buf[8];

	size_t units_read;

	if(fp == NULL)
	{
	    errno = EINVAL;
	    return(0);
	}

	units_read = fread(
	    buf,
	    sizeof(u_int8_t),
	    sizeof(buf) / sizeof(u_int8_t),
	    fp
	);
	if((units_read == sizeof(buf) / sizeof(u_int8_t)))
	    return(ImgBufferIsMNG(buf, sizeof(buf)));
	else
	    return(0);
}

int ImgFileIsJNG(const char *path)
{
	int             status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
	    return(0);

	status = ImgStreamIsJNG(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}

int ImgFileIsMNG(const char *path)
{
	int             status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
	    return(0);

	status = ImgStreamIsMNG(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	libmng new callback.
 *
 *	The returned memory segment must be set to all 0's.
 */
static mng_ptr ImgMNGNewCB(mng_size_t len)
{
	return((mng_ptr)(calloc(1, (size_t)len)));
}

/*
 *	libmng delete callback.
 */
static void ImgMNGDeleteCB(mng_ptr p, mng_size_t len)
{
	free(p);
}


/*
 *	libmng read error callback.
 */
static mng_bool ImgMNGReadErrorCB(
	mng_handle mng_h,
	mng_int32 error_code,
	mng_int8 severity_code,
	mng_chunkid chunk_id,
	mng_uint32 chunk_sequence,
	mng_int32 extra1,
	mng_int32 extra2,
	mng_pchar msg
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	    return(MNG_FALSE);

	if(imgio_last_open_error == NULL)
	    ImgOpenSetError(msg);

	if(ctx->status == 0)
	    ctx->status = -1;

	return(MNG_TRUE);
}

/*
 *	libmng read open callback.
 */
static mng_bool ImgMNGReadOpenCB(mng_handle mng_h)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	switch(ctx->data_storage_type)
	{
	  case IMG_MNG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		const void *bp = ctx->bp;
		const unsigned long bp_len = ctx->bp_len;

		/* Use the default IO size */
		ctx->io_size = IMG_MNG_DEF_IO_SIZE;

		/* Check if the buffer has a MNG or JNG header
		 * signature
		  */
		if(!ImgBufferIsMNG((const u_int8_t *)bp, (int)bp_len))
		{
		    if(!ImgBufferIsJNG((const u_int8_t *)bp, (int)bp_len))
		    {
			imgio_last_open_error = "Not a MNG or JNG image";
			errno = EINVAL;
			return(MNG_FALSE);
		    }
		}
	    }
	    else
	    {
		imgio_last_open_error = "Buffer not specified";
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;

	  case IMG_MNG_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		long fp_pos;
		FILE *fp = ctx->fp;
		struct stat stat_buf;
		if(fstat(fileno(fp), &stat_buf))
		{
		    ctx->fp_len = 0l;
		    ctx->io_size = IMG_MNG_DEF_IO_SIZE;
		}
		else
		{
		    ctx->fp_len = (unsigned long)stat_buf.st_size;
		    ctx->io_size = MAX(
			stat_buf.st_blksize,
			IMG_MNG_MIN_IO_SIZE
		    );
		}

		/* Check if the stream has a MNG or JNG header
		 * signature
		 */
		fp_pos = ftell(fp);
		if(!ImgStreamIsMNG(fp))
		{
		    (void)fseek(fp, fp_pos, SEEK_SET);
		    if(!ImgStreamIsJNG(fp))
		    {
			(void)fseek(fp, fp_pos, SEEK_SET);
			imgio_last_open_error = "Not a MNG or JNG image";
			errno = EINVAL;
			return(MNG_FALSE);
		    }
		}
		(void)fseek(fp, fp_pos, SEEK_SET);
	    }
	    else
	    {
		imgio_last_open_error = "Stream not specified";
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;
	}

	return(MNG_TRUE);
}

/*
 *	libmng read data callback.
 */
static mng_bool ImgMNGReadDataCB(
	mng_handle mng_h,
	mng_ptr buffer,
	mng_uint32 bytes_requested,
	mng_uint32p bytes_read
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));

	*bytes_read = 0;

	if(*ctx->user_aborted)
	{
	    errno = EINTR;
	    return(MNG_FALSE);
	}

	/* Nothing to read? */
	if((buffer == NULL) || (bytes_requested == 0))
	    return(MNG_TRUE);

	switch(ctx->data_storage_type)
	{
	  case IMG_MNG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		size_t	bp_bytes_remaining,
			bytes_to_read;
			
		/* End of input buffer reached? */
		if(ctx->bp_pos >= ctx->bp_len)
		{
		    errno = ENODATA;
		    return(MNG_TRUE);
		}

		/* Calculate the number of bytes remaining in the
		 * input buffer
		 */
		bp_bytes_remaining = ((size_t)ctx->bp_len - ctx->bp_pos);

		/* Calculate the number of bytes to read */
		bytes_to_read = (size_t)bytes_requested;
		if(bytes_to_read > bp_bytes_remaining)
		    bytes_to_read = bp_bytes_remaining;

		/* Report progress */
		if(ctx->progress_cb != NULL)
		{
		    const int progress_level_size = (int)ctx->bp_len;
		    if(!ctx->progress_cb(
			ctx->progress_data,
			(progress_level_size * ctx->cur_progress_level) +
			    (int)ctx->bp_pos,
			progress_level_size * ctx->nprogress_levels,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
		    ))
		    {
			*ctx->user_aborted = 1;
			errno = EINTR;
			return(MNG_FALSE);
		    }
		}

		/* Copy this segment of the input buffer to the
		 * libmng supplied read buffer
		 */
		(void)memcpy(
		    buffer,
		    ctx->bp + ctx->bp_pos,
		    bytes_to_read
		);

		/* Increment the input buffer position */
		ctx->bp_pos += bytes_to_read;

		/* Count the number of bytes read to libmng */
		*bytes_read = (*bytes_read) + (mng_uint32)bytes_to_read;
	    }
	    else
	    {
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;

	  case IMG_MNG_DATA_STORAGE_STREAM:
	    if((ctx->fp != NULL) && (ctx->io_size > 0l))
	    {
		FILE *fp = ctx->fp;
		const size_t io_size = ctx->io_size;
		size_t	units_to_read,
			units_read,
			bytes_left = bytes_requested;
		mng_ptr buf_ptr = buffer;
		while(bytes_left > 0l)
		{
		    /* Calculate the number of units to read, which
		     * will be no more than the IO size
		     */
		    units_to_read = MIN(
			bytes_left,
			io_size
		    ) / sizeof(u_int8_t);

		    /* Report progress */
		    if(ctx->progress_cb != NULL)
		    {
			const int progress_level_size = (int)ctx->fp_len;
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (progress_level_size * ctx->cur_progress_level) +
				(int)ftell(fp),
			    progress_level_size * ctx->nprogress_levels,
			    ctx->data_width, ctx->data_height,
			    ctx->data_bpl, ctx->data_bpp,
			    (ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
			))
			{
			    *ctx->user_aborted = 1;
			    errno = EINTR;
			    return(MNG_FALSE);
			}
		    }

		    /* Read this block */
		    units_read = fread(
			buf_ptr,
			sizeof(u_int8_t),
			units_to_read,
			fp
		    );

		    /* Count the number of bytes read to libmng */
		    *bytes_read = (*bytes_read) +
			((mng_uint32)units_read * sizeof(u_int8_t));

		    /* Increment the number of bytes left to read
		     * and the libmng buffer pointer
		     */
		    bytes_left -= units_read * sizeof(u_int8_t);
		    buf_ptr += units_read * sizeof(u_int8_t);

		    /* Check for any read errors or end of file */
		    if(units_read != units_to_read)
		    {
			if(ferror(fp))
			{
			    ImgOpenSetError(strerror(errno));
			    return(MNG_FALSE);
			}
			else if(feof(fp))
			{
			    errno = ENODATA;
			    return(MNG_TRUE);
			}
		    }
		    if(units_read == 0)
		    {
			errno = ENODATA;
			return(MNG_TRUE);
		    }
		}
	    }
	    else
	    {
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;
	}

	return(MNG_TRUE);
}

/*
 *	libmng process header callback.
 */
static mng_bool ImgMNGReadProcessHeaderCB(
	mng_handle mng_h,
	mng_uint32 width, mng_uint32 height
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(MNG_FALSE);
	}

	/* Opening as a thumb? */
	if((ctx->req_width > 0) && (ctx->req_height > 0))
	{
	    /* Open as a thumb */
	    ctx->mng_width = width;
	    ctx->mng_height = height;
	    /* Calculate the size of the target RGBA image in order to
	     * accomidate for the thumb
	     */
	    ImgCalculateThumbSize(
		width, height,			/* Original size */
		ctx->req_width, ctx->req_height,/* Thumb size limit */
		&ctx->data_width, &ctx->data_height
	    );
	}
	else
	{
	    /* Open as actual size */
	    ctx->data_width = ctx->mng_width = width;
	    ctx->data_height = ctx->mng_height = height;
	}

	/* Reallocate the MNG drawing buffer */
	if((ctx->mng_drawing_buffer_bpp > 0) &&
	   (ctx->mng_width > 0) && (ctx->mng_height > 0)
	)
	    ctx->mng_drawing_buffer = (u_int8_t *)realloc(
		ctx->mng_drawing_buffer,
		(size_t)(ctx->mng_width * ctx->mng_height * ctx->mng_drawing_buffer_bpp)
	    );

	/* Calculate the RGBA bytes per line */
	ctx->data_bpl = ctx->data_width * ctx->data_bpp;

	return(MNG_TRUE);
}

/*
 *	Read chunk iterator callback.
 */
static mng_bool ImgMNGReadChunkIteratorCB(
	mng_handle mng_h,
	mng_handle chunk_h,
	mng_chunkid chunk_type,
	mng_uint32 chunk_index
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(MNG_FALSE);
	}

	switch((int)chunk_type)
	{
	  case MNG_UINT_MHDR:
	    if((mng_h != MNG_NULL) && (chunk_h != MNG_NULL))
	    {
		mng_uint32	width,
				height,
				ticks_per_second,
				nlayers,
				nframes,
				total_play_time,
				simplicity_flags;
		if(mng_getchunk_mhdr(
		    mng_h,
		    chunk_h,
		    &width, &height,
		    &ticks_per_second,
		    &nlayers,
		    &nframes,
		    &total_play_time,
		    &simplicity_flags
		) == MNG_NOERROR)
		{
		    ctx->mng_ticks_per_second = ticks_per_second;
		    ctx->mng_nlayers = nlayers;
		    ctx->mng_nframes = nframes;
		    ctx->mng_total_play_time = total_play_time;
		    ctx->mng_total_play_time_ms = (unsigned long)(
			(ticks_per_second > 0) ?
			    (total_play_time * 1000l / ticks_per_second) :
			    total_play_time
		    );
		    ctx->mng_simplicity_flags = simplicity_flags;
		}
	    }
	    break;

	  case MNG_UINT_MEND:
	    errno = 0;
	    return(MNG_FALSE);
	    break;

	  case MNG_UINT_BACK:
	    if((mng_h != MNG_NULL) && (chunk_h != MNG_NULL))
	    {
		mng_uint16 r16, g16, b16;
		mng_uint8 mandatory, tile;
		mng_uint16 tile_image_id;

		if(mng_getchunk_back(
		    mng_h,
		    chunk_h,
		    &r16, &g16, &b16,
		    &mandatory,
		    &tile_image_id,
		    &tile
		) == MNG_NOERROR)
		{
		    /* Get the background color if we have not
		     * obtained it yet
		     */
		    if(!ctx->got_bg_color && (ctx->bg_color != NULL))
		    {
			u_int8_t *tar_bg_rgba = ctx->bg_color;
			tar_bg_rgba[0] = (u_int8_t)(r16 >> 8);
			tar_bg_rgba[1] = (u_int8_t)(g16 >> 8);
			tar_bg_rgba[2] = (u_int8_t)(b16 >> 8);
			tar_bg_rgba[3] = ctx->def_alpha_value;

			/* Mark that we got the background color so
			 * that we ignore it on subsequent BACK or
			 * bKGD chunks
			 */
			ctx->got_bg_color = 1;
		    }
		}
	    }
	    break;

	  case MNG_UINT_IHDR:
	    if((mng_h != MNG_NULL) && (chunk_h != MNG_NULL))
	    {
		mng_uint32	width,
				height;
		mng_uint8	bit_depth,
				color_type,
				compression,
				filter,
				interlace;
		if(mng_getchunk_ihdr(
		    mng_h,
		    chunk_h,
		    &width, &height,
		    &bit_depth,
		    &color_type,
		    &compression,
		    &filter,
		    &interlace
		) == MNG_NOERROR)
		{
		    /* Record certain values for the first frame */
		    if(ctx->mng_nihdrs == 0)
		    {
			ctx->mng_color_type = color_type;
		    }

		    /* Count this IHDR chunk */
		    ctx->mng_nihdrs++;
		}
	    }
	    break;

	  case MNG_UINT_tIME:
	    if((mng_h != MNG_NULL) && (chunk_h != MNG_NULL))
	    {
		mng_uint16 year;
		mng_uint8	month,
				day,
				hour,
				minute,
				second;
		if(mng_getchunk_time(
		    mng_h,
		    chunk_h,
		    &year,
		    &month,
		    &day,
		    &hour,
		    &minute,
		    &second
		) == MNG_NOERROR)
		{
		    struct tm tm_buf;
		    tm_buf.tm_sec = (int)second;
		    tm_buf.tm_min = (int)minute;
		    tm_buf.tm_hour = (int)hour;
		    tm_buf.tm_mday = (int)day;
		    tm_buf.tm_mon = (int)month - 1;
		    tm_buf.tm_year = (int)year - 1900;
		    tm_buf.tm_wday = 0;		/* Not used */
		    tm_buf.tm_yday = 0;		/* Not used */
		    tm_buf.tm_isdst = -1;	/* DST not available */
		    if(ctx->modified_time_sec != NULL)
		    {
			*ctx->modified_time_sec = (unsigned long)mktime(&tm_buf);
			/* MNG stores time values relative to GMT so
			 * we need to account for the time zone in
			 * order to have the time value in local time
			 */
			*ctx->modified_time_sec = (*ctx->modified_time_sec) - (unsigned long)timezone;
		    }
		}
	    }
	    break;

	  case MNG_UINT_bKGD:
	    if((mng_h != MNG_NULL) && (chunk_h != MNG_NULL))
	    {
		mng_bool empty;
		mng_uint8	type,
				index;
		mng_uint16	grey16,
				r16,
				g16,
				b16;
		if(mng_getchunk_bkgd(
		    mng_h,
		    chunk_h,
		    &empty,
		    &type,
		    &index,
		    &grey16,
		    &r16,
		    &g16,
		    &b16
		) == MNG_NOERROR)
		{
		    /* Get the background color if we have not
		     * obtained it yet
		     */
		    if(!ctx->got_bg_color && (ctx->bg_color != NULL))
		    {
			u_int8_t *tar_bg_rgba = ctx->bg_color;
			tar_bg_rgba[0] = (u_int8_t)(r16 >> 8);
			tar_bg_rgba[1] = (u_int8_t)(g16 >> 8);
			tar_bg_rgba[2] = (u_int8_t)(b16 >> 8);
			tar_bg_rgba[3] = ctx->def_alpha_value;

			/* Mark that we got the background color so
			 * that we ignore it on subsequent BACK or
			 * bKGD chunks
			 */
			ctx->got_bg_color = 1;
		    }
		}
	    }
	    break;
	}

	/* Return MNG_TRUE to indicate get next chunk if any */
	return(MNG_TRUE);
}

/*
 *	libmng set timer callback.
 */
static mng_bool ImgMNGReadSetTimerCB(
	mng_handle mng_h,
	mng_uint32 delay
)
{
	int i;
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(MNG_FALSE);
	}

	/* Set the delay for the current frame
	 *
	 * If the specified delay is 0 then it means we should use
	 * the value specified by the ticks per second
	 */
	i = ctx->nframes - 1;
	if((i >= 0) && (ctx->delay_ms_list != NULL))
	    ctx->delay_ms_list[i] = (delay > 0) ?
		(unsigned long)delay :
		(unsigned long)ctx->mng_ticks_per_second;

	return(MNG_TRUE);
}

/*
 *	libmng get tick count callback.
 */
static mng_uint32 ImgMNGReadGetTickCountCB(mng_handle mng_h)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(0);
	}

	/* Always returning 0 so that the next refresh is called
	 * immediately
	 */
	return(0);
}

/*
 *	libmng get canvas line callback.
 */
static mng_ptr ImgMNGReadGetCanvasLineCB(
	mng_handle mng_h,
	mng_uint32 line
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(ctx->mng_drawing_buffer == NULL)
	    return(NULL);

	return(
	    ctx->mng_drawing_buffer +
		(line * ctx->mng_width * ctx->mng_drawing_buffer_bpp)
	);
}

/*
 *	MNG library refresh redraw) callback.
 */
static mng_bool ImgMNGReadRefreshCB(
	mng_handle mng_h,
	mng_uint32 x, mng_uint32 y,
	mng_uint32 width, mng_uint32 height
)
{
	int cur_frame_num;
	u_int8_t *rgba;
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(MNG_FALSE);
	}

#if 0
/* Do not return if *ctx->user_aborted is true allow the calling
 * function to check for it and break out of any display loops
 */
	if(*ctx->user_aborted)
	{
	    errno = EINTR;
	    return(MNG_TRUE);
	}
#endif

	cur_frame_num = ctx->nframes;

	/* Increment the number of frames loaded and increase the
	 * RGBA image frames list and delay list allocation
	 */
	ctx->nframes++;
	ctx->data_list = (u_int8_t **)realloc(
	    ctx->data_list,
	    ctx->nframes * sizeof(u_int8_t *)
	);
	if(ctx->data_list == NULL)
	    return(MNG_TRUE);

	ctx->delay_ms_list = (unsigned long *)realloc(
	    ctx->delay_ms_list,
	    ctx->nframes * sizeof(unsigned long)
	);
	if(ctx->delay_ms_list == NULL)
	    return(MNG_TRUE);

	/* Create a new RGBA image data frame */
	if(ctx->progress_cb != NULL)
	    rgba = (u_int8_t *)calloc(
		ctx->data_height * ctx->data_bpl,
		sizeof(u_int8_t)
	    );
	else
	    rgba = (u_int8_t *)malloc(
		ctx->data_height * ctx->data_bpl * sizeof(u_int8_t)
	    );
	ctx->data_list[cur_frame_num] = rgba;
	if(rgba == NULL)
	    return(MNG_TRUE);

	/* Render the contents of the current MNG drawing buffer to
	 * our new RGBA image data frame
	 *
	 * Render as a thumb?
	 */
	if((ctx->req_width > 0) && (ctx->req_height > 0))
	{
	    /* RGBA */
	    if((ctx->mng_drawing_buffer != NULL) &&
	       (ctx->mng_drawing_buffer_bpp == 4) &&
	       (ctx->data_bpp == 4)
	    )
	    {
		const int	bpp = ctx->data_bpp,
				twidth = ctx->data_width,
				theight = ctx->data_height,
				tbpl = ctx->data_bpl,
				swidth = (int)ctx->mng_width,
				sheight = (int)ctx->mng_height,
				sbpl = swidth * (int)ctx->mng_drawing_buffer_bpp;
		u_int8_t	*tar = rgba,
				*tar_row,
				*tar_ptr;
		const u_int8_t	*src = ctx->mng_drawing_buffer,
				*src_ptr;
		int	sx, sy,
			tx, ty,
			tx_last,
			ty_last = -1;

		for(sy = 0; sy < sheight; sy++)
		{
		    ty = theight * sy / sheight;
		    if(ty <= ty_last)
			continue;

		    for(sx = 0,
			src_ptr = src + (sy * sbpl),
			tar_row = tar + (ty * tbpl),
			tx_last = -1;
			sx < swidth;
			sx++,
			src_ptr += bpp
		    )
		    {
			tx = twidth * sx / swidth;
			if(tx <= tx_last)
			    continue;

			tar_ptr = tar_row + (tx * bpp);
			*(u_int32_t *)tar_ptr = *(const u_int32_t *)src_ptr;

			/* Report progress */
			if((ctx->progress_cb != NULL) && ((sy % IMG_MNG_PROGRESS_RESOLUTION) == 0))
			{
			    int i, n;
			    if((ctx->mng_simplicity_flags & MNG_SIMPLICITY_COMPLEXFEATURES) &&
			       (ctx->mng_total_play_time_ms > 0l)
			    )
			    {
				const int progress_level_size = (int)ctx->mng_total_play_time_ms;
				i = (progress_level_size * ctx->cur_progress_level) +
				    (int)ctx->mng_cur_play_time_ms;
				n = progress_level_size * ctx->nprogress_levels;
			    }
			    else
			    {
				const int progress_level_size = ctx->mng_nihdrs * sheight;
				i = (progress_level_size * ctx->cur_progress_level) +
				    ((cur_frame_num * sheight) + sy);
				n = progress_level_size * ctx->nprogress_levels;
			    }
			    if(!ctx->progress_cb(
				ctx->progress_data,
				i, n,
				twidth, theight,
				tbpl, bpp,
				rgba
			    ))
			    {
				*ctx->user_aborted = 1;
				break;
			    }
			}

			tx_last = tx;
		    }

		    ty_last = ty;
		}
	    }
	}
	else
	{
	    /* Render actual size
	     *
	     * RGBA
	     */
	    if((ctx->mng_drawing_buffer != NULL) &&
	       (ctx->mng_drawing_buffer_bpp == 4) &&
	       (ctx->data_bpp == 4)
	    )
	    {
		const int	bpp = ctx->data_bpp,
				_width = (int)ctx->mng_width,
				_height = (int)ctx->mng_height,
				tbpl = ctx->data_bpl,
				sbpl = _width * (int)ctx->mng_drawing_buffer_bpp,
				min_bpl = MIN(tbpl, sbpl);
		u_int8_t	*tar = rgba;
		const u_int8_t	*src = ctx->mng_drawing_buffer;
		int _y;

		/* Render each line from the current MNG drawing
		 * buffer to the RGBA image data frame
		 */
		for(_y = 0; _y < _height; _y++)
		{
		    /* Render/copy this line */
		    (void)memcpy(
			tar + (_y * tbpl),
			src + (_y * sbpl),
			(size_t)min_bpl
		    );

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((_y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
		    {
			int i, n;
			if((ctx->mng_simplicity_flags & MNG_SIMPLICITY_COMPLEXFEATURES) &&
			   (ctx->mng_total_play_time_ms > 0l)
			)
			{
			    const int progress_level_size = (int)ctx->mng_total_play_time_ms;
			    i = (progress_level_size * ctx->cur_progress_level) +
				(int)ctx->mng_cur_play_time_ms;
			    n = progress_level_size * ctx->nprogress_levels;
			}
			else
			{
			    const int progress_level_size = ctx->mng_nihdrs * _height;
			    i = (progress_level_size * ctx->cur_progress_level) +
				((cur_frame_num * _height) + _y);
			    n = progress_level_size * ctx->nprogress_levels;
			}
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    i, n,
			    _width, _height,
			    tbpl, bpp,
			    rgba
			))
			{
			    *ctx->user_aborted = 1;
			    break;
			}
		    }
		}
	    }
	}

	ctx->delay_ms_list[cur_frame_num] = 0;	/* Set later in
						 * ImgMNGSetTimerCB) */

	return(MNG_TRUE);			/* Keep displaying */
}

/*
 *	MNG library process text callback.
 */
static mng_bool ImgMNGReadProcessTextCB(
	mng_handle mng_h,
	mng_uint8 type,
	mng_pchar keyword,
	mng_pchar text,
	mng_pchar language,
	mng_pchar translation
)
{
	ImgMNGReadContext *ctx = IMG_MNG_READ_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	{
	    errno = EINVAL;
	    return(MNG_FALSE);
	}

	if(keyword == NULL)
	    return(MNG_TRUE);

	switch(type)
	{
	  case MNG_TYPE_TEXT:
#ifdef MNG_TEXT_TITLE
	    if(!strcmp(keyword, MNG_TEXT_TITLE))
#else
	    if(!strcmp(keyword, "Title"))
#endif
	    {
		if(ctx->title != NULL)
		{
		    free(*ctx->title);
		    *ctx->title = STRDUP(text);
		}
	    }
#ifdef MNG_TEXT_AUTHOR
	    else if(!strcmp(keyword, MNG_TEXT_AUTHOR))
#else
	    else if(!strcmp(keyword, "Author"))
#endif
	    {
		if(ctx->author != NULL)
		{
		    free(*ctx->author);
		    *ctx->author = STRDUP(text);
		}
	    }
#ifdef MNG_TEXT_DESCRIPTION
	    else if(!strcmp(keyword, MNG_TEXT_DESCRIPTION))
#else
	    else if(!strcmp(keyword, "Description"))
#endif
	    {
		if(ctx->comments != NULL)
		{
		    /* Set comments field as description only if the
		     * comments field was set yet
		     */
		    if(*ctx->comments == NULL)
			*ctx->comments = STRDUP(text);
		}
	    }
#ifdef MNG_TEXT_COPYRIGHT
	    else if(!strcmp(keyword, MNG_TEXT_COPYRIGHT))
#else
	    else if(!strcmp(keyword, "Copyright"))
#endif
	    {

	    }
#ifdef MNG_TEXT_CREATIONTIME
	    else if(!strcmp(keyword, MNG_TEXT_CREATIONTIME))
#else
	    else if(!strcmp(keyword, "Creation Time"))
#endif
	    {

	    }
#ifdef MNG_TEXT_SOFTWARE
	    else if(!strcmp(keyword, MNG_TEXT_SOFTWARE))
#else
	    else if(!strcmp(keyword, "Software"))
#endif
	    {
		if(ctx->creator != NULL)
		{
		    free(*ctx->creator);
		    *ctx->creator = STRDUP(text);
		}
	    }
#ifdef MNG_TEXT_DISCLAIMER
	    else if(!strcmp(keyword, MNG_TEXT_DISCLAIMER))
#else
	    else if(!strcmp(keyword, "Disclaimer"))
#endif
	    {

	    }
#ifdef MNG_TEXT_WARNING
	    else if(!strcmp(keyword, MNG_TEXT_WARNING))
#else
	    else if(!strcmp(keyword, "Warning"))
#endif
	    {

	    }
#ifdef MNG_TEXT_SOURCE
	    else if(!strcmp(keyword, MNG_TEXT_SOURCE))
#else
	    else if(!strcmp(keyword, "Source"))
#endif
	    {

	    }
#ifdef MNG_TEXT_COMMENT
	    else if(!strcmp(keyword, MNG_TEXT_COMMENT))
#else
	    else if(!strcmp(keyword, "Comment"))
#endif
	    {
		if(ctx->comments != NULL)
		{
		    free(*ctx->comments);
		    *ctx->comments = STRDUP(text);
		}
	    }
	    break;

	  case MNG_TYPE_ZTXT:
	    break;

	  case MNG_TYPE_ITXT:
	    break;
	}

	return(MNG_TRUE);
}

/*
 *	libmng read close callback.
 */
static mng_bool ImgMNGReadCloseCB(mng_handle mng_h)
{
	/* Do nothing and return success */
	return(MNG_TRUE);
}

/*
 *	Reads the JNG or MNG image data from a buffer or a stream to
 *	RGBA image data.
 */
static int ImgReadMNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	mng_retcode mng_rc;
	mng_handle mng_h;
	ImgMNGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the ImgMNGReadContext */			\
 if(ctx != NULL) {					\
  /* Shutdown libmng */					\
  if(ctx->mng_h != MNG_NULL) {				\
   (void)mng_cleanup(&ctx->mng_h);			\
   ctx->mng_h = mng_h = MNG_NULL;			\
  }							\
							\
  /* Delete the MNG drawing buffer */			\
  free(ctx->mng_drawing_buffer);			\
							\
  /* Delete the loaded image data */			\
  if(ctx->data_list != NULL) {				\
   int i;						\
   for(i = 0; i < ctx->nframes; i++)			\
    free(ctx->data_list[i]);				\
   free(ctx->data_list);				\
  }							\
							\
  /* Delete the loaded delay list */			\
  free(ctx->delay_ms_list);				\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	/* Create our MNG read context */
	ctx = IMG_MNG_READ_CONTEXT(calloc(1, sizeof(ImgMNGReadContext)));
	if(ctx == NULL)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/*	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->fp_len = 0l; */
#if 0
	ctx->io_size = 0l;
	ctx->req_width = 0;			/* Open actual size */
	ctx->req_height = 0;
#endif
	ctx->mng_h = mng_h = MNG_NULL;
#if 0
	ctx->mng_simplicity_flags = 0;
	ctx->mng_color_type = 0;
	ctx->mng_width = 0;
	ctx->mng_height = 0;
	ctx->mng_ticks_per_second = 0;
	ctx->mng_nframes = 0;
	ctx->mng_nlayers = 0;
	ctx->mng_total_play_time = 0;
	ctx->mng_nihdrs = 0;
	ctx->mng_cur_play_time_ms = 0l;
	ctx->mng_total_play_time_ms = 0l;
	ctx->mng_drawing_buffer_bpp = 0;
	ctx->mng_drawing_buffer = NULL;
	ctx->nframes = 0;
	ctx->data_width = 0;
	ctx->data_height = 0;
#endif
	ctx->data_bpp = 4;			/* RGBA */
#if 0
	ctx->data_bpl = 0;			/* Calculated in
						 * ImgMNGReadProcessHeaderCB) */
	ctx->data_list = NULL;
	ctx->delay_ms_list = NULL;
#endif
	ctx->def_alpha_value = def_alpha_value;
/*	ctx->got_bg_color = 0; */
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->modified_time_sec = modified_time_sec_rtn;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
/*	ctx->cur_progress_level = 0; */
	ctx->nprogress_levels = 2;
	ctx->user_aborted = user_aborted;
/*	ctx->status = 0; */

	/* Initialize libmng */
	ctx->mng_h = mng_h = mng_initialize(
	    ctx,				/* User data */
	    ImgMNGNewCB,
	    ImgMNGDeleteCB,
	    MNG_NULL				/* No trace */
	);
	if(mng_h == MNG_NULL)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}
	if((mng_setcb_errorproc(mng_h, ImgMNGReadErrorCB) != MNG_NOERROR) ||
	   (mng_setcb_openstream(mng_h, ImgMNGReadOpenCB) != MNG_NOERROR) ||
	   (mng_setcb_closestream(mng_h, ImgMNGReadCloseCB) != MNG_NOERROR) ||
	   (mng_setcb_readdata(mng_h, ImgMNGReadDataCB) != MNG_NOERROR) ||
	   (mng_setcb_processheader(mng_h, ImgMNGReadProcessHeaderCB) != MNG_NOERROR) ||
	   (mng_setcb_settimer(mng_h, ImgMNGReadSetTimerCB) != MNG_NOERROR) ||
	   (mng_setcb_gettickcount(mng_h, ImgMNGReadGetTickCountCB) != MNG_NOERROR) ||
	   (mng_setcb_getcanvasline(mng_h, ImgMNGReadGetCanvasLineCB) != MNG_NOERROR) ||
	   (mng_setcb_refresh(mng_h, ImgMNGReadRefreshCB) != MNG_NOERROR) ||
	   (mng_setcb_processtext(mng_h, ImgMNGReadProcessTextCB) != MNG_NOERROR)
	)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}

	/* Set the MNG data to be provided to us as 4 bytes per pixel
	 * RGBA when rendered to the MNG drawing buffer
	 */
	if(mng_set_canvasstyle(
	    mng_h,
	    MNG_CANVAS_RGBA8
	) != MNG_NOERROR)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to set the MNG Library canvas style to MNG_CANVAS_RGBA8";
	    CLEANUP_RETURN(-1);
	}
	ctx->mng_drawing_buffer_bpp = 4;


	ctx->cur_progress_level = 0;		/* Progress level 0 */

	/* Set the libmng reading process to suspension mode SMOD) so
	 * that ImgMNGOpenReadDataCB) can return less than the
	 * requested amount of data
	 */
	(void)mng_set_suspensionmode(mng_h, MNG_TRUE);

	/* Read the entire MNG image in suspension mode
	 *
	 * This will call ImgMNGOpenCB),
	 * ImgMNGOpenReadDataCB), ImgMNGOpenProcessHeaderCB),
	 * ImgMNGReadProcessTextCB), and ImgMNGCloseCB)
	 *
	 * Each RGBA image frame will not be loaded until mng_display)
	 * is called further below
	 */
	mng_rc = mng_read(mng_h);
	while((mng_rc == MNG_NEEDMOREDATA) &&
	      !(*ctx->user_aborted)
	)
	{
	    mng_rc = mng_read_resume(mng_h);
	}
	if(mng_rc != MNG_NOERROR)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "An error occured while reading the file";
	    CLEANUP_RETURN(-1);
	}
	if(*ctx->user_aborted)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Iterate through each MNG chunk to get the information
	 * for the context members; mng_ticks_per_second,
	 * mng_nlayers, mng_nframes, mny_nihdrs and bg_rgba
	 *
	 * Note that is the only way to get the number of IHDR
	 * chunks, thus the true number of frames in the MNG file
	 * excluding those created by the fPRI and TERM chunk
	 * effects
	 */
	(void)mng_iterate_chunks(
	    mng_h,
	    0,					/* Start at the first chunk */
	    ImgMNGReadChunkIteratorCB
	);

	/* Get the total play time if it was not specified in the MHDR */
	if(ctx->mng_total_play_time_ms == 0l)
	    ctx->mng_total_play_time_ms = (unsigned long)mng_get_totalplaytime(mng_h);

	/* Render the MNG image
	 *
	 * This will call ImgMNGSetTimerCB), ImgMNGGetTickCountCB),
	 * ImgMNGGetCanvasLineCB), and ImgMNGReadRefreshCB)
	 *
	 * Each call to ImgMNGReadRefreshCB) will append an RGBA
	 * frame from a libmng rendered drawing buffer
	 *
	 * Note that we use ctx->mng_nihdrs instead of
	 * ctx->mng_nframes to check when we have obtained all the
	 * frames because ctx->mng_nframes is sometimes unspecified
	 * or includes repeated frames caused by the fPRI or TERM
	 * effects chunk
	 */
	ctx->cur_progress_level++;		/* Progress level 1 */
	if(!(*ctx->user_aborted))
	{
	    /* If the complex features flag was found in the MHDR
	     * then we need to render for the entire display time
	     */
	    if((ctx->mng_simplicity_flags & MNG_SIMPLICITY_COMPLEXFEATURES) &&
	       (ctx->mng_total_play_time_ms > 0l)
	    )
	    {
		int prev_nframes;

		mng_rc = mng_display(mng_h);
		ctx->mng_cur_play_time_ms = (unsigned long)mng_get_currentplaytime(mng_h);
		while((mng_rc == MNG_NEEDTIMERWAIT) &&
		      (ctx->mng_cur_play_time_ms < ctx->mng_total_play_time_ms) &&
		      !(*ctx->user_aborted)
		)
		{
		    prev_nframes = ctx->nframes;

		    /* Render the next frame */
		    mng_rc = mng_display_resume(mng_h);

		    /* Was ImgMNGReadRefreshCB) unable to append a frame? */
		    if(prev_nframes == ctx->nframes)
			break;

		    /* Update the current play time after appending a frame */
		    ctx->mng_cur_play_time_ms = (unsigned long)mng_get_currentplaytime(mng_h);
		}
	    }
	    else
	    {
		/* All else render using the number of IHDR chunks
		 * encountered as the total number of frames we
		 * do not use ctx->mng_nframes because it is
		 * sometimes unsepcified or include srepeated frames
		 * caused by the fPRI or TERM effects chunks)
		 */
		int prev_nframes;
		mng_rc = mng_display(mng_h);
		while((mng_rc == MNG_NEEDTIMERWAIT) &&
		      (ctx->nframes < ctx->mng_nihdrs) &&
		      !(*ctx->user_aborted)
		)
		{
		    prev_nframes = ctx->nframes;

		    /* Render the next frame */
		    mng_rc = mng_display_resume(mng_h);

		    /* Was ImgMNGReadRefreshCB) unable to append a frame? */
		    if(prev_nframes == ctx->nframes)
			break;
		}
	    }
	}

	/* Report the final progress */
	if((ctx->progress_cb != NULL) &&
	   !(*ctx->user_aborted)
	)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0,
		ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->data_width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->data_bpl;
	if(nframes_rtn != NULL)
	    *nframes_rtn = ctx->nframes;
	if(rgba_list_rtn != NULL)
	{
	    if(*rgba_list_rtn == NULL)
	    {
		*rgba_list_rtn = ctx->data_list;
		ctx->data_list = NULL;
	    }
	}
	if(delay_ms_list_rtn != NULL)
	{
	    if(*delay_ms_list_rtn == NULL)
	    {
		*delay_ms_list_rtn = ctx->delay_ms_list;
		ctx->delay_ms_list = NULL;
	    }
	}
	if(x_rtn != NULL)          
	    *x_rtn = 0;
	if(y_rtn != NULL)
	    *y_rtn = 0;
	if(base_width_rtn != NULL)
	    *base_width_rtn = ctx->mng_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = ctx->mng_height;

	if(*ctx->user_aborted)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the JNG or MNG image data from a buffer to RGBA image data.
 */
int ImgBufferReadMNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadMNGRGBA(
	    IMG_MNG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_ms_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Reads the JNG or MNG image data from a stream to RGBA image data.
 */
int ImgStreamReadMNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadMNGRGBA(
	    IMG_MNG_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_ms_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the JNG or MNG image file to RGBA image data.
 */
int ImgFileOpenMNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    return(-4);
	}

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the MNG or JNG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadMNGRGBA(
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_ms_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

/*
 *	Reads the JNG or MNG image data from a buffer or a stream to
 *	RGBA image data as a thumb.
 */
static int ImgReadMNGRGBAThumb(
	const ImgMNGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc	progress_cb, void *progress_data,
	int *user_aborted
)
{
	mng_retcode mng_rc;
	mng_handle mng_h;
	ImgMNGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the ImgMNGReadContext */			\
 if(ctx != NULL) {					\
  /* Shutdown libmng */					\
  if(ctx->mng_h != MNG_NULL) {				\
   (void)mng_cleanup(&ctx->mng_h);			\
   ctx->mng_h = mng_h = MNG_NULL;			\
  }							\
							\
  /* Delete the MNG drawing buffer */			\
  free(ctx->mng_drawing_buffer);			\
							\
  /* Delete the loaded image data */			\
  if(ctx->data_list != NULL) {				\
   int i;						\
   for(i = 0; i < ctx->nframes; i++)			\
    free(ctx->data_list[i]);				\
   free(ctx->data_list);				\
  }							\
							\
  /* Delete the loaded delay list */			\
  free(ctx->delay_ms_list);				\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Create our MNG read context */
	ctx = IMG_MNG_READ_CONTEXT(calloc(1, sizeof(ImgMNGReadContext)));
	if(ctx == NULL)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/*	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->fp_len = 0l; */
#if 0
	ctx->io_size = 0l;
#endif
	ctx->req_width = req_width;		/* Open as a thumb */
	ctx->req_height = req_height;
	ctx->mng_h = mng_h = MNG_NULL;
#if 0
	ctx->mng_simplicity_flags = 0;
	ctx->mng_color_type = 0;
	ctx->mng_width = 0;
	ctx->mng_height = 0;
	ctx->mng_ticks_per_second = 0;
	ctx->mng_nframes = 0;
	ctx->mng_nlayers = 0;
	ctx->mng_total_play_time = 0;
	ctx->mng_nihdrs = 0;
	ctx->mng_cur_play_time_ms = 0l;
	ctx->mng_total_play_time_ms = 0l;
	ctx->mng_drawing_buffer_bpp = 0;
	ctx->mng_drawing_buffer = NULL;
	ctx->nframes = 0;
#endif
#if 0
	ctx->data_width = 0;
	ctx->data_height = 0;
#endif
	ctx->data_bpp = 4;			/* RGBA */
#if 0
	ctx->data_bpl = 0;			/* Calculated in
						 * ImgMNGReadProcessHeaderCB) */
	ctx->data_list = NULL;
	ctx->delay_ms_list = NULL;
#endif
	ctx->def_alpha_value = def_alpha_value;
/*	ctx->got_bg_color = 0; */
	ctx->bg_color = bg_color;
	ctx->creator = creator_rtn;
	ctx->title = title_rtn;
	ctx->author = author_rtn;
	ctx->comments = comments_rtn;
	ctx->modified_time_sec = modified_time_sec_rtn;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
/*	ctx->cur_progress_level = 0; */
	ctx->nprogress_levels = 2;
	ctx->user_aborted = user_aborted;
/*	ctx->status = 0; */

	/* Initialize libmng */
	ctx->mng_h = mng_h = mng_initialize(
	    ctx,				/* User data */
	    ImgMNGNewCB,
	    ImgMNGDeleteCB,
	    MNG_NULL				/* No trace */
	);
	if(mng_h == MNG_NULL)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}
	if((mng_setcb_errorproc(mng_h, ImgMNGReadErrorCB) != MNG_NOERROR) ||
	   (mng_setcb_openstream(mng_h, ImgMNGReadOpenCB) != MNG_NOERROR) ||
	   (mng_setcb_closestream(mng_h, ImgMNGReadCloseCB) != MNG_NOERROR) ||
	   (mng_setcb_readdata(mng_h, ImgMNGReadDataCB) != MNG_NOERROR) ||
	   (mng_setcb_processheader(mng_h, ImgMNGReadProcessHeaderCB) != MNG_NOERROR) ||
	   (mng_setcb_settimer(mng_h, ImgMNGReadSetTimerCB) != MNG_NOERROR) ||
	   (mng_setcb_gettickcount(mng_h, ImgMNGReadGetTickCountCB) != MNG_NOERROR) ||
	   (mng_setcb_getcanvasline(mng_h, ImgMNGReadGetCanvasLineCB) != MNG_NOERROR) ||
	   (mng_setcb_refresh(mng_h, ImgMNGReadRefreshCB) != MNG_NOERROR) ||
	   (mng_setcb_processtext(mng_h, ImgMNGReadProcessTextCB) != MNG_NOERROR)
	)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}

	/* Set the MNG data to be provided to us as 4 bytes per pixel
	 * RGBA when rendered to the MNG drawing buffer
	 */
	if(mng_set_canvasstyle(
	    mng_h,
	    MNG_CANVAS_RGBA8
	) != MNG_NOERROR)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "Unable to set the MNG Library canvas style to MNG_CANVAS_RGBA8";
	    CLEANUP_RETURN(-1);
	}
	ctx->mng_drawing_buffer_bpp = 4;

	/* Read the entire MNG image in suspension mode
	 *
	 * This will call ImgMNGOpenCB),
	 * ImgMNGOpenReadDataCB), ImgMNGOpenProcessHeaderCB),
	 * ImgMNGReadProcessTextCB), and ImgMNGCloseCB)
	 *
	 * Each RGBA image frame will not be loaded until mng_display)
	 * is called further below
	 */
	ctx->cur_progress_level = 0;		/* Progress level 0 */
	/* Set the libmng reading process to suspension mode SMOD) so
	 * that ImgMNGOpenReadDataCB) can return less than the
	 * requested amount of data
	 */
	(void)mng_set_suspensionmode(mng_h, MNG_TRUE);
	/* Call ImgMNGOpenReadDataCB) to read the MNG file */
	mng_rc = mng_read(mng_h);
	while((mng_rc == MNG_NEEDMOREDATA) &&
	      !(*ctx->user_aborted)
	)
	{
	    mng_rc = mng_read_resume(mng_h);
	}
	if(mng_rc != MNG_NOERROR)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "An error occured while reading the file";
	    CLEANUP_RETURN(-1);
	}
	if(*ctx->user_aborted)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Iterate through each MNG chunk to get the information
	 * for the context members; mng_ticks_per_second,
	 * mng_nlayers, mng_nframes, mny_nihdrs and bg_rgba
	 *
	 * Note that is the only way to get the number of IHDR
	 * chunks, thus the true number of frames in the MNG file
	 * excluding those created by the fPRI and TERM chunk
	 * effects
	 */
	(void)mng_iterate_chunks(
	    mng_h,
	    0,					/* Start at the first chunk */
	    ImgMNGReadChunkIteratorCB
	);

	/* Get the total play time if it was not specified in the MHDR */
	if(ctx->mng_total_play_time_ms == 0l)
	    ctx->mng_total_play_time_ms = (unsigned long)mng_get_totalplaytime(mng_h);

	/* Render the MNG image
	 *
	 * This will call ImgMNGSetTimerCB), ImgMNGGetTickCountCB),
	 * ImgMNGGetCanvasLineCB), and ImgMNGReadRefreshCB)
	 *
	 * Each call to ImgMNGReadRefreshCB) will append an RGBA
	 * frame from a libmng rendered drawing buffer
	 *
	 * Note that we use ctx->mng_nihdrs instead of
	 * ctx->mng_nframes to check when we have obtained all the
	 * frames because ctx->mng_nframes is sometimes unspecified
	 * or includes repeated frames caused by the fPRI or TERM
	 * effects chunk
	 */
	ctx->cur_progress_level++;		/* Progress level 1 */
	if(!(*ctx->user_aborted))
	{
	    /* If the complex features flag was found in the MHDR
	     * then we need to render for the entire display time
	     */
	    if((ctx->mng_simplicity_flags & MNG_SIMPLICITY_COMPLEXFEATURES) &&
	       (ctx->mng_total_play_time_ms > 0l)
	    )
	    {
		int prev_nframes;

		mng_rc = mng_display(mng_h);
		ctx->mng_cur_play_time_ms = (unsigned long)mng_get_currentplaytime(mng_h);
		while((mng_rc == MNG_NEEDTIMERWAIT) &&
		      (ctx->mng_cur_play_time_ms < ctx->mng_total_play_time_ms) &&
		      !(*ctx->user_aborted)
		)
		{
		    /* When opening as a thumb, only render the first frame */
		    if(ctx->nframes >= 1)
			break;

		    prev_nframes = ctx->nframes;

		    /* Render the next frame */
		    mng_rc = mng_display_resume(mng_h);

		    /* Was ImgMNGReadRefreshCB) unable to append a frame? */
		    if(prev_nframes == ctx->nframes)
			break;

		    /* Update the current play time after appending a frame */
		    ctx->mng_cur_play_time_ms = (unsigned long)mng_get_currentplaytime(mng_h);
		}
	    }
	    else
	    {
		/* All else render using the number of IHDR chunks
		 * encountered as the total number of frames we
		 * do not use ctx->mng_nframes because it is
		 * sometimes unsepcified or include srepeated frames
		 * caused by the fPRI or TERM effects chunks)
		 */
		int prev_nframes;
		mng_rc = mng_display(mng_h);
		while((mng_rc == MNG_NEEDTIMERWAIT) &&
		      (ctx->nframes < ctx->mng_nihdrs) &&
		      !(*ctx->user_aborted)
		)
		{
		    /* When opening as a thumb, only render the first frame */
		    if(ctx->nframes >= 1)
			break;

		    prev_nframes = ctx->nframes;

		    /* Render the next frame */
		    mng_rc = mng_display_resume(mng_h);

		    /* Was ImgMNGReadRefreshCB) unable to append a frame? */
		    if(prev_nframes == ctx->nframes)
			break;
		}
	    }
	}

	/* Report the final progress */
	if((ctx->progress_cb != NULL) &&
	   !(*ctx->user_aborted)
	)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0,
		ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->data_width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->data_bpl;
	if(orig_width_rtn != NULL)
	    *orig_width_rtn = ctx->mng_width;
	if(orig_height_rtn != NULL)
	    *orig_height_rtn = ctx->mng_height;
	if(nframes_rtn != NULL)
	{
	    /* When opening as a thumb we need to report the number of
	     * frames from values other than ctx->nframes since it will
	     * always be 0 or 1 because only at most one frame will be
	     * loaded
	     */
	    if(ctx->mng_nframes > 0)
		*nframes_rtn = ctx->mng_nframes;
	    else if(ctx->mng_nlayers > 0)
		*nframes_rtn = ctx->mng_nlayers;
	    else if(ctx->mng_nihdrs > 0)
		*nframes_rtn = ctx->mng_nihdrs;
	    else
		*nframes_rtn = ctx->nframes;	/* All else use 1 */
	}
	if(play_time_ms_rtn != NULL)
	    *play_time_ms_rtn = ctx->mng_total_play_time_ms;
	if(rgba_rtn != NULL)
	{
	    if((ctx->nframes > 0) && (ctx->data_list != NULL))
	    {
		free(*rgba_rtn);
		*rgba_rtn = ctx->data_list[0];
		ctx->data_list[0] = NULL;
	    }
	}
	if(x_rtn != NULL)          
	    *x_rtn = 0;
	if(y_rtn != NULL)
	    *y_rtn = 0;
	if(base_width_rtn != NULL)
	    *base_width_rtn = ctx->mng_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = ctx->mng_height;

	if(*ctx->user_aborted)
	{
	    if(imgio_last_open_error == NULL)
		imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}

#undef CLEANUP_RETURN
}

/*
 *	Opens the JNG or MNG image from a buffer to RGBA image data as
 *	a thumb.
 */
int ImgBufferReadMNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadMNGRGBAThumb(
	    IMG_MNG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the JNG or MNG image from a stream to RGBA image data as
 *	a thumb.
 */
int ImgStreamReadMNGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadMNGRGBAThumb(
	    IMG_MNG_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the JNG or MNG image file to RGBA image data as a thumb.
 */
int ImgFileOpenMNGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc	progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    return(-4);
	}

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the MNG or JNG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadMNGRGBAThumb(
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    modified_time_sec_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	libmng write error callback.
 */
static mng_bool ImgMNGWriteErrorCB(
	mng_handle mng_h,
	mng_int32 error_code,
	mng_int8 severity_code,
	mng_chunkid chunk_id,
	mng_uint32 chunk_sequence,
	mng_int32 extra1,
	mng_int32 extra2,
	mng_pchar msg
)
{
	ImgMNGWriteContext *ctx = IMG_MNG_WRITE_CONTEXT(mng_get_userdata(mng_h));
	if(ctx == NULL)
	    return(MNG_FALSE);

	if(imgio_last_save_error == NULL)
	    ImgSaveSetError(msg);

	if(ctx->status == 0)
	    ctx->status = -1;

	return(MNG_TRUE);
}

/*
 *	libmng write open callback.
 */
static mng_bool ImgMNGWriteOpenCB(mng_handle mng_h)
{
	ImgMNGWriteContext *ctx = IMG_MNG_WRITE_CONTEXT(mng_get_userdata(mng_h));
	switch(ctx->data_storage_type)
	{
	  case IMG_MNG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL))
	    {
		ctx->io_size = IMG_MNG_DEF_IO_SIZE;
	    }
	    else
	    {
		imgio_last_save_error = "Buffer not specified";
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;

	  case IMG_MNG_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		FILE *fp = ctx->fp;
		struct stat stat_buf;
		if(fstat(fileno(fp), &stat_buf))
		{
		    ctx->io_size = IMG_MNG_DEF_IO_SIZE;
		}
		else
		{
		    ctx->io_size = stat_buf.st_blksize;
		}
	    }
	    else
	    {
		imgio_last_save_error = "Stream not specified";
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;
	}

	return(MNG_TRUE);
}

/*
 *	libmng write data callback.
 */
static mng_bool ImgMNGWriteDataCB(
	mng_handle mng_h,
	mng_ptr buffer,
	mng_uint32 bytes_provided,
	mng_uint32p bytes_written
)
{
	ImgMNGWriteContext *ctx = IMG_MNG_WRITE_CONTEXT(mng_get_userdata(mng_h));

	*bytes_written = 0;

	if(*ctx->user_aborted)
	{
	    errno = EINTR;
	    return(MNG_FALSE);
	}

	/* Nothing to write? */
	if((buffer == NULL) || (bytes_provided == 0))
	    return(MNG_TRUE);

	switch(ctx->data_storage_type)
	{
	  case IMG_MNG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL) &&
	       (ctx->io_size > 0l)
	    )
	    {
		/* Append all the bytes provided to the output buffer
		 * in one iteration
		 */
		const unsigned long bytes_to_write = (unsigned long)bytes_provided;

		/* Increase the allocation */
		ctx->bp_pos = *ctx->bp_len;
		*ctx->bp_len = ctx->bp_pos + bytes_to_write;
		*ctx->bp = (void *)realloc(
		    *ctx->bp,
		    (*ctx->bp_len)
		);
		if(*ctx->bp == NULL)
		{
		    *ctx->bp_len = 0l;
		    ctx->bp_pos = 0l;
		    imgio_last_save_error = "Memory allocation error";
		    return(MNG_FALSE);
		}

 		/* Copy/append the data */
		(void)memcpy(
		    (*ctx->bp) + ctx->bp_pos,
		    buffer,
		    bytes_to_write
		);

		/* Update the buffer position */
		ctx->bp_pos = *ctx->bp_len;

		/* Count the number of bytes written to libmng */
		*bytes_written = (*bytes_written) + (mng_uint32)bytes_to_write;
	    }
	    else
	    {
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;

	  case IMG_MNG_DATA_STORAGE_STREAM:
	    if((ctx->fp != NULL) && (ctx->io_size > 0l))
	    {
		FILE *fp = ctx->fp;
		const size_t io_size = (size_t)ctx->io_size;
		size_t	units_to_write,
			units_written,
			bytes_left = bytes_provided;
		mng_ptr buf_ptr = buffer;
		while(bytes_left > 0l)
		{
		    /* Calculate the number of units to write, which
		     * will be no more than the IO size
		     */
		    units_to_write = MIN(
			bytes_left,
			io_size
		    ) / sizeof(u_int8_t);

		    /* Write this block */
		    units_written = fwrite(
			buf_ptr,
			sizeof(u_int8_t),
			units_to_write,
			fp
		    );

		    /* Count the number of bytes written to libmng */
		    *bytes_written = (*bytes_written) +
			((mng_uint32)units_written * sizeof(u_int8_t));

		    /* Increment the number of bytes left to write
		     * and the libmng buffer pointer
		     */
		    bytes_left -= units_written * sizeof(u_int8_t);
		    buf_ptr += units_written * sizeof(u_int8_t);

		    /* Check for any write errors */
		    if(units_written != units_to_write)
		    {
			ImgSaveSetError(strerror(errno));
			return(MNG_FALSE);
		    }
		}
	    }
	    else
	    {
		errno = EINVAL;
		return(MNG_FALSE);
	    }
	    break;
	}

	return(MNG_TRUE);
}

/*
 *	libmng write close callback.
 */
static mng_bool ImgMNGWriteCloseCB(mng_handle mng_h)
{
	/* Do nothing and return success */
	return(MNG_TRUE);
}

/*
 *	ImgIO JPEG write progress callback.
 */
static int ImgJPEGWriteProgressCB(
	void *progress_data,
	const int i, const int n,
	const int width, const int height,
	const int bpl, const int bpp,
	const u_int8_t *image_data
)
{
	ImgMNGWriteContext *ctx = IMG_MNG_WRITE_CONTEXT(progress_data);
	if(ctx == NULL)
	    return(0);

	if(ctx->progress_cb != NULL)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		(n * ctx->cur_progress_level) + i,
		(n * ctx->nprogress_levels),
		width, height,
		bpl, bpp,
		image_data
	    ))
	    {
		*ctx->user_aborted = 1;
		return(0);
	    }
	}

	return(1);
}

/*
 *	Compresses the data using zlib.
 */
static mng_ptr ImgMNGCompressDataZLib(
	ImgMNGWriteContext *ctx,
	const int compression_level,		/* 0 to 9 */
	mng_ptr data,
	const mng_uint32 length,
	mng_uint32 *length_rtn
)
{
	uLongf compressed_length_ulf = (uLongf)length;
	mng_ptr compressed_data = (mng_ptr)malloc(length * sizeof(mng_uint8));

	*length_rtn = 0;

	if(compressed_data == NULL)
	    return(NULL);

	/* Use zlib compression */
	(void)compress2(
	    (Bytef *)compressed_data,
	    &compressed_length_ulf,
	    (const Bytef *)data,
	    (uLong)length,
	    compression_level			/* 0 to 9 */
	);

	*length_rtn = (mng_uint32)compressed_length_ulf;

	return(compressed_data);
}

/*
 *	Converts the RGBA image data to MNG Greyscale image data.
 */
static mng_ptr ImgMNGConvertRGBAToIDATGreyscale(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
)
{
	const int	rgba_bpp = 4,
			mng_bpp = 1,		/* MNG Greyscale size */
			/* Each MNG image data line has a filter byte at
			 * the beginning */
			mng_bpl = 1 + (width * mng_bpp),
			len = height * mng_bpl;
	int x, y;
	const u_int8_t	*rgba_row;
	u_int8_t	bg_grey = ImgConvertPixelRGBToGreyscale(bg_color),
			*mng_idat_g = (u_int8_t *)malloc(len * sizeof(u_int8_t)),
			*mng_idat_g_row;
	if(mng_idat_g == NULL)
	{
	    *length_rtn = 0;
	    return(NULL);
	}

	*length_rtn = (mng_uint32)len;

	/* Copy/convert each row */
	for(y = 0; y < height; y++)
	{
	    rgba_row = rgba + (y * rgba_bpl);
	    mng_idat_g_row = mng_idat_g + (y * mng_bpl);

	    /* Set the filter byte first byte) */
	    *mng_idat_g_row = 0x00;
	    mng_idat_g_row++;

	    /* Copy the rest of the pixels on this row */
	    for(x = 0; x < width; x++)
	    {
		mng_idat_g_row[0] = ImgConvertPixelRGBAToGreyscale(
		    rgba_row,
		    bg_grey
		);
		rgba_row += rgba_bpp;
		mng_idat_g_row += mng_bpp;
	    }

	    if((ctx->progress_cb != NULL) && ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
	    {
		if(!ctx->progress_cb(
		    ctx->progress_data,
		    (ctx->cur_progress_level * height) + y,
		    (ctx->nprogress_levels * height),
		    width, height,
		    rgba_bpl, rgba_bpp,
		    rgba
		))
		{
		    *ctx->user_aborted = 1;
		    break;
		}
	    }
	}

	return((mng_ptr)mng_idat_g);
}

/*
 *	Converts the RGBA image data to MNG Greyscale Alpha image data.
 */
static mng_ptr ImgMNGConvertRGBAToIDATGreyscaleAlpha(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
)
{
	const int	rgba_bpp = 4,
			mng_bpp = 2,		/* MNG Greyscale Alpha size */
			/* Each MNG image data line has a filter byte at
			 * the beginning */
			mng_bpl = 1 + (width * mng_bpp),
			len = height * mng_bpl;
	int x, y;
	const u_int8_t	*rgba_row;
	u_int8_t	*mng_idat_ga = (u_int8_t *)malloc(len * sizeof(u_int8_t)),
			*mng_idat_ga_row;
	if(mng_idat_ga == NULL)
	{
	    *length_rtn = 0;
	    return(NULL);
	}

	*length_rtn = (mng_uint32)len;

	/* Copy/convert each row */
	for(y = 0; y < height; y++)
	{
	    rgba_row = rgba + (y * rgba_bpl);
	    mng_idat_ga_row = mng_idat_ga + (y * mng_bpl);

	    /* Set the filter byte first byte) */
	    *mng_idat_ga_row = 0x00;
	    mng_idat_ga_row++;

	    /* Copy the rest of the pixels on this row */
	    for(x = 0; x < width; x++)
	    {
		mng_idat_ga_row[0] = ImgConvertPixelRGBToGreyscale(rgba_row);
		mng_idat_ga_row[1] = rgba_row[3];

		rgba_row += rgba_bpp;
		mng_idat_ga_row += mng_bpp;
	    }

	    if((ctx->progress_cb != NULL) && ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
	    {
		if(!ctx->progress_cb(
		    ctx->progress_data,
		    (ctx->cur_progress_level * height) + y,
		    (ctx->nprogress_levels * height),
		    width, height,
		    rgba_bpl, rgba_bpp,
		    rgba
		))
		{
		    *ctx->user_aborted = 1;
		    break;
		}
	    }
	}

	return((mng_ptr)mng_idat_ga);
}

/*
 *	Converts the RGBA image data to MNG RGB image data.
 */
static mng_ptr ImgMNGConvertRGBAToIDATRGB(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
)
{
	const int	rgba_bpp = 4,
			mng_bpp = 3,		/* MNG RGB size */
			/* Each MNG image data line has a filter byte at
			 * the beginning */
			mng_bpl = 1 + (width * mng_bpp),
			len = height * mng_bpl;
	int x, y;
	const u_int8_t	*rgba_row;
	u_int8_t	*mng_idat_rgb = (u_int8_t *)malloc(len * sizeof(u_int8_t)),
			*mng_idat_rgb_row;
	if(mng_idat_rgb == NULL)
	{
	    *length_rtn = 0;
	    return(NULL);
	}

	*length_rtn = (mng_uint32)len;

	/* Copy/convert each row */
	for(y = 0; y < height; y++)
	{
	    rgba_row = rgba + (y * rgba_bpl);
	    mng_idat_rgb_row = mng_idat_rgb + (y * mng_bpl);

	    /* Set the filter byte first byte) */
	    *mng_idat_rgb_row = 0x00;
	    mng_idat_rgb_row++;

	    /* Copy the rest of the pixels on this row */
	    for(x = 0; x < width; x++)
	    {
		ImgConvertPixelRGBAToRGB(
		    rgba_row,
		    bg_color,
		    mng_idat_rgb_row
		);
		rgba_row += rgba_bpp;
		mng_idat_rgb_row += mng_bpp;
	    }

	    if((ctx->progress_cb != NULL) && ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
	    {
		if(!ctx->progress_cb(
		    ctx->progress_data,
		    (ctx->cur_progress_level * height) + y,
		    (ctx->nprogress_levels * height),
		    width, height,
		    rgba_bpl, rgba_bpp,
		    rgba
		))
		{
		    *ctx->user_aborted = 1;
		    break;
		}
	    }
	}

	return((mng_ptr)mng_idat_rgb);
}

/*
 *	Converts the RGBA image data to MNG RGBA image data.
 */
static mng_ptr ImgMNGConvertRGBAToIDATRGBA(
	ImgMNGWriteContext *ctx,
	const int width, const int height,
	const int rgba_bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,
	mng_uint32 *length_rtn
)
{
	const int	rgba_bpp = 4,
			mng_bpp = 4,		/* MNG RGBA size */
			/* Each MNG image data line has a filter byte at
			 * the beginning */
			mng_bpl = 1 + (width * mng_bpp),
			len = height * mng_bpl;
	int x, y;
	const u_int8_t	*rgba_row;
	u_int8_t	*mng_idat_rgba = (u_int8_t *)malloc(len * sizeof(u_int8_t)),
			*mng_idat_rgba_row;
	if(mng_idat_rgba == NULL)
	{
	    *length_rtn = 0;
	    return(NULL);
	}

	*length_rtn = (mng_uint32)len;

	/* Copy/convert each row */
	for(y = 0; y < height; y++)
	{
	    rgba_row = rgba + (y * rgba_bpl);
	    mng_idat_rgba_row = mng_idat_rgba + (y * mng_bpl);

	    /* Set the filter byte first byte) */
	    *mng_idat_rgba_row = 0x00;
	    mng_idat_rgba_row++;

	    /* Copy the rest of the pixels on this row */
	    for(x = 0; x < width; x++)
	    {
		*(u_int32_t *)mng_idat_rgba_row = *(const u_int32_t *)rgba_row;
		rgba_row += rgba_bpp;
		mng_idat_rgba_row += mng_bpp;
	    }

	    if((ctx->progress_cb != NULL) && ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
	    {
		if(!ctx->progress_cb(
		    ctx->progress_data,
		    (ctx->cur_progress_level * height) + y,
		    (ctx->nprogress_levels * height),
		    width, height,
		    rgba_bpl, rgba_bpp,
		    rgba
		))
		{
		    *ctx->user_aborted = 1;
		    break;
		}
	    }
	}

	return((mng_ptr)mng_idat_rgba);
}

/*
 *	Writes the RGBA image data to a buffer or a stream as a JNG
 *	image.
 */
static int ImgWriteJNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const ImgJNGColorType color_type,
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : width * bpp,
			nframes = 1;
	int user_aborted = 0;
	const u_int8_t *rgba_list[nframes];
	mng_handle mng_h;
	ImgMNGWriteContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the ImgMNGWriteContext */			\
 if(ctx != NULL) {					\
  /* Shutdown libmng */					\
  if(ctx->mng_h != MNG_NULL) {				\
   (void)mng_cleanup(&ctx->mng_h);			\
   ctx->mng_h = mng_h = MNG_NULL;			\
  }							\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (rgba == NULL)
	)
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	rgba_list[0] = rgba;

	/* Create our MNG write context */
	ctx = IMG_MNG_WRITE_CONTEXT(calloc(1, sizeof(ImgMNGWriteContext)));
	if(ctx == NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/*	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->io_size = 0l; */
	ctx->mng_h = mng_h = MNG_NULL;
	ctx->mng_color_type = (color_type == IMG_JNG_COLOR_TYPE_COLOR) ?
	    ((transparency) ?
		MNG_COLORTYPE_JPEGCOLORA : MNG_COLORTYPE_JPEGCOLOR) :
	    ((transparency) ?
		MNG_COLORTYPE_JPEGGRAYA : MNG_COLORTYPE_JPEGGRAY);
	ctx->mng_width = width;
	ctx->mng_height = height;
	ctx->mng_ticks_per_second = 1000;	/* Not used when saving JNG */
	ctx->mng_nlayers = nframes + 1;		/* nframes + base layer */
	ctx->mng_nframes = nframes;		/* Always 1 when saving JNG */
	ctx->mng_total_play_time = 0;		/* Not used when saving JNG */
	ctx->mng_total_play_time_ms = 0;	/* Not used when saving JNG */
	ctx->nframes = nframes;
	ctx->data_width = width;
	ctx->data_height = height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = _bpl;
	ctx->data_list = rgba_list;
	ctx->delay_ms_list = NULL;		/* Not used when saving JNG */
	ctx->bg_color = bg_color;
	ctx->creator = creator;
	ctx->title = title;
	ctx->author = author;
	ctx->comments = comments;
	ctx->modified_time_sec = modified_time_sec;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
/*	ctx->cur_progress_level = 0; */
	if(transparency)
	    ctx->nprogress_levels = 4;
	else
	    ctx->nprogress_levels = 1;
	ctx->user_aborted = &user_aborted;
/*	ctx->status = 0; */

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[0] : NULL
	    ))
		*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Initialize libmng */
	ctx->mng_h = mng_h = mng_initialize(
	    ctx,				/* User data */
	    ImgMNGNewCB,
	    ImgMNGDeleteCB,
	    MNG_NULL				/* No trace */
	);
	if(mng_h == MNG_NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}
	if((mng_setcb_errorproc(mng_h, ImgMNGWriteErrorCB) != MNG_NOERROR) ||
	   (mng_setcb_openstream(mng_h, ImgMNGWriteOpenCB) != MNG_NOERROR) ||
	   (mng_setcb_closestream(mng_h, ImgMNGWriteCloseCB) != MNG_NOERROR) ||
	   (mng_setcb_writedata(mng_h, ImgMNGWriteDataCB) != MNG_NOERROR)
	)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}

	/* Set libmng to write mode */
	if(mng_create(mng_h) != MNG_NOERROR)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to set the MNG library into write mode";
	    CLEANUP_RETURN(-1);
	}

	/* Begin creating the JNG image in memory, the mng_putchunk*)
	 * functions only create the JNG image in memory, the actual
	 * JNG image file will be written further below when
	 * mng_write) is called
	 */

	/* Write the JHDR (JNG Header) chunk */
	if(!(*ctx->user_aborted))
	{
 	    if(mng_putchunk_jhdr(
		mng_h,
		ctx->mng_width, ctx->mng_height,
		ctx->mng_color_type,
		MNG_BITDEPTH_JPEG8,
		MNG_COMPRESSION_BASELINEJPEG,	/* 8: ISO-10918-1 Huffman-coded baseline JPEG */
		MNG_INTERLACE_PROGRESSIVE,
/*		MNG_INTERLACE_SEQUENTIAL, */
		(transparency) ? 8 : 0,		/* Alpha compression method */
		(transparency) ? 8 : 0,		/* JNG 8-bit grayscale JDAA format */
		0,				/* Alpha filter */
		0				/* Alpha interlace */
	    ) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
 "Unable to append the JNG header (JHDR) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Write the tEXt (text) chunks */
	if(!(*ctx->user_aborted))
	{
	    const char *s = ctx->title;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_TITLE),
		    MNG_TEXT_TITLE,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->author;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_AUTHOR),
		    MNG_TEXT_AUTHOR,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->creator;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_SOFTWARE),
		    MNG_TEXT_SOFTWARE,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->comments;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_COMMENT),
		    MNG_TEXT_COMMENT,
		    strlen(s),
		    (mng_pchar)s
		);
	}

	/* Write the tIME (modified time) chunk */
	if(!(*ctx->user_aborted))
	{
	    if(ctx->modified_time_sec > 0l)
	    {
		/* MNG requires that time values be stored in GMT */
		time_t t = (time_t)ctx->modified_time_sec;
		struct tm *tm_ptr = gmtime(&t);
		if(tm_ptr != NULL)
		{
		    mng_putchunk_time(
			mng_h,
			(mng_uint16)tm_ptr->tm_year + 1900,
			(mng_uint8)tm_ptr->tm_mon + 1,
			(mng_uint8)tm_ptr->tm_mday,
			(mng_uint8)tm_ptr->tm_hour,
			(mng_uint8)tm_ptr->tm_min,
			(mng_uint8)tm_ptr->tm_sec
		    );
		}
	    }
	}

	/* Write the bKGD (background) chunks */
	if(!(*ctx->user_aborted) && (ctx->bg_color != NULL))
	{
	    const u_int8_t	*c = ctx->bg_color,
				c_grey = ImgConvertPixelRGBToGreyscale(c);

	    if(mng_putchunk_bkgd(
		mng_h,
		MNG_FALSE,			/* Not empty */
		(color_type == IMG_JNG_COLOR_TYPE_COLOR) ?
		    MNG_COLORTYPE_RGB : MNG_COLORTYPE_GRAY,
		0,				/* Index */
		((mng_uint16)c_grey << 8) | (mng_uint16)c_grey,
		((mng_uint16)c[0] << 8) | (mng_uint16)c[0],
		((mng_uint16)c[1] << 8) | (mng_uint16)c[1],
		((mng_uint16)c[2] << 8) | (mng_uint16)c[2]
	    ) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
 "Unable to append the background color (bKGD) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Write the JDAA (JPEG alpha mask image data) chunk */
	if(!(*ctx->user_aborted) && transparency)
	{
	    /* Create a greyscale image data from the alpha channel
	     * of the RGBA image data to be saved as a greyscale JPEG
	     * image buffer
	     */
	    const int	tar_width = ctx->data_width,
			tar_height = ctx->data_height,
			tar_bpp = 1,		/* Greyscale */
			tar_bpl = tar_width * tar_bpp;
	    u_int8_t *tar = (u_int8_t *)malloc(
		tar_height * tar_bpl * sizeof(u_int8_t)
	    );
	    if(tar != NULL)
	    {
		const int	src_bpp = ctx->data_bpp,
				src_bpl = ctx->data_bpl;
		int y;
		const u_int8_t	*src = ctx->data_list[0],/* RGBA */
				*src_ptr;
		u_int8_t	*tar_ptr,
				*tar_row_end;
		void *jpeg_bp = NULL;
		unsigned long jpeg_bp_len = 0l;

		for(y = 0; y < tar_height; y++)
		{
		    src_ptr = src + (y * src_bpl);
		    tar_ptr = tar + (y * tar_bpl);
		    tar_row_end = tar_ptr + (tar_width * tar_bpp);
		    while(tar_ptr < tar_row_end)
		    {
			*tar_ptr = src_ptr[3];
			src_ptr += src_bpp;
			tar_ptr += tar_bpp;
		    }

		    /* Report the progress */
		    if((ctx->progress_cb != NULL) && ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0))
		    {
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (tar_height * ctx->cur_progress_level) + y,
			    (tar_height * ctx->nprogress_levels),
			    ctx->data_width, ctx->data_height,
			    ctx->data_bpl, ctx->data_bpp,
			    (ctx->nframes > 0) ? ctx->data_list[0] : NULL
			))
			{
			    *ctx->user_aborted = 1;
			    break;
			}
		    }
		}

		ctx->cur_progress_level++;

		/* Save the greyscale image data to a greyscale JPEG
		 * image buffer
		 */
		ctx->status = ImgBufferWriteJPEGGreyscale(
		    &jpeg_bp, &jpeg_bp_len,
		    tar_width, tar_height,
		    tar_bpl,
		    tar,			/* Greyscale */
		    quality,
		    ImgJPEGWriteProgressCB, ctx
		);

		/* Delete the greyscale image data */
		free(tar);

		/* Write the JDAA (JPEG alpha mask image data) chunk */
		if((jpeg_bp != NULL) && (jpeg_bp_len > 0l) &&
		   (ctx->status == 0)
		)
		{
		    if(mng_putchunk_jdaa(
			mng_h,
			(mng_uint32)jpeg_bp_len,
			(mng_ptr)jpeg_bp
		    ) != MNG_NOERROR)
		    {
			free(jpeg_bp);
			if(imgio_last_save_error == NULL)
			    imgio_last_save_error =
 "Unable to append the JPEG alpha mask image data (JDAA) chunk";
			CLEANUP_RETURN(-1);
		    }
		}

		/* Delete the JPEG image buffer */
		free(jpeg_bp);
	    }
	    else
	    {
		imgio_last_save_error = "Memory allocation error";
		CLEANUP_RETURN(-3);
	    }

	    ctx->cur_progress_level++;
	}

	/* Write the JDAT (JPEG image data) chunk */
	if(!(*ctx->user_aborted))
	{
	    /* Was an alpha mask written? */
	    if(transparency)
	    {
		/* We need to create an RGBA image data from the
		 * specified RGBA image data with the alpha channel
		 * set to fully opaque, otherwise
		 * ImgBufferWriteJPEGRGBA() will blend the specified
		 * RGBA image data with the background color if its
		 * alpha channel was not fully opaque
		 */
		const int	tar_width = ctx->data_width,
				tar_height = ctx->data_height,
				tar_bpp = 4,		/* RGBA */
				tar_bpl = tar_width * tar_bpp;
		u_int8_t *tar = (u_int8_t *)malloc(
		    tar_height * tar_bpl * sizeof(u_int8_t)
		);
		if(tar != NULL)
		{
		    const int	src_bpp = ctx->data_bpp,
				src_bpl = ctx->data_bpl;
		    int y;
		    const u_int8_t	*src = ctx->data_list[0],/* RGBA */
					*src_ptr;
		    u_int8_t	*tar_ptr,
				*tar_row_end;
		    void *jpeg_bp = NULL;
		    unsigned long jpeg_bp_len = 0l;

		    for(y = 0; y < tar_height; y++)
		    {
			src_ptr = src + (y * src_bpl);
			tar_ptr = tar + (y * tar_bpl);
			tar_row_end = tar_ptr + (tar_width * tar_bpp);
			while(tar_ptr < tar_row_end)
			{
			    tar_ptr[0] = src_ptr[0];
			    tar_ptr[1] = src_ptr[1];
			    tar_ptr[2] = src_ptr[2];
			    tar_ptr[3] = 0xFF;
			    src_ptr += src_bpp;
			    tar_ptr += tar_bpp;
			}

			/* Report the progress */
			if((ctx->progress_cb != NULL) &&
			   ((y % IMG_MNG_PROGRESS_RESOLUTION) == 0)
			)
			{
			    if(!ctx->progress_cb(
				ctx->progress_data,
				(tar_height * ctx->cur_progress_level) + y,
				(tar_height * ctx->nprogress_levels),
				ctx->data_width, ctx->data_height,
				ctx->data_bpl, ctx->data_bpp,
				(ctx->nframes > 0) ? ctx->data_list[0] : NULL
			    ))
			    {
				*ctx->user_aborted = 1;
				break;
			    }
			}
		    }

		    ctx->cur_progress_level++;

		    /* Save the coppied RGBA image data to a color
		     * JPEG image buffer
		     */
		    ctx->status = ImgBufferWriteJPEGRGBA(
			&jpeg_bp, &jpeg_bp_len,
			tar_width, tar_height,
			tar_bpl,
			tar,			/* RGBA */
			ctx->bg_color,
			quality,
			(int)color_type,
			ImgJPEGWriteProgressCB, ctx
		    );

		    /* Delete the coppied RGBA image data image data */
		    free(tar);

		    /* Write the JDAT (JPEG image data) chunk */
		    if((jpeg_bp != NULL) && (jpeg_bp_len > 0l) &&
		       (ctx->status == 0)
		    )
		    {
			if(mng_putchunk_jdat(
			    mng_h,
			    (mng_uint32)jpeg_bp_len,
			    (mng_ptr)jpeg_bp
			) != MNG_NOERROR)
			{
			    free(jpeg_bp);
			    if(imgio_last_save_error == NULL)
				imgio_last_save_error =
"Unable to append the JPEG image data (JDAT) chunk";
			    CLEANUP_RETURN(-1);
			}
		    }

		    /* Delete the JPEG image buffer */
		    free(jpeg_bp);
		}
		else
		{
		    imgio_last_save_error = "Memory allocation error";
		    CLEANUP_RETURN(-3);
		}
	    }
	    else
	    {
		/* We did not write an alpha mask, so save the
		 * specified RGBA image data to a color JPEG image
		 * buffer, ImgBufferWriteJPEGRGBA() will blend any
		 * not fully opaque pixels with the background color
		 */
		void *jpeg_bp = NULL;
		unsigned long jpeg_bp_len = 0l;
		ctx->status = ImgBufferWriteJPEGRGBA(
		    &jpeg_bp, &jpeg_bp_len,
		    ctx->data_width, ctx->data_height,
		    ctx->data_bpl,
		    ctx->data_list[0],		/* RGBA */
		    ctx->bg_color,
		    quality,
		    (int)color_type,
		    ImgJPEGWriteProgressCB, ctx
		);

		/* Write the JDAT (JPEG image data) chunk */
		if((jpeg_bp != NULL) && (jpeg_bp_len > 0l) &&
		   (ctx->status == 0)
		)
		{
		    if(mng_putchunk_jdat(
			mng_h,
			(mng_uint32)jpeg_bp_len,
			(mng_ptr)jpeg_bp
		    ) != MNG_NOERROR)
		    {
			free(jpeg_bp);
			if(imgio_last_save_error == NULL)
			    imgio_last_save_error =
"Unable to append the JPEG image data (JDAT) chunk";
			CLEANUP_RETURN(-1);
		    }
		}

		/* Delete the JPEG image buffer */
		free(jpeg_bp);
	    }

	    ctx->cur_progress_level++;
	}

	/* Write the IEND chunk */
	if(!(*ctx->user_aborted))
	{
	    if(mng_putchunk_iend(mng_h) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
"Unable to append JNG tailer (IEND) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Write the JNG image data in memory to file */
	if(!(*ctx->user_aborted))
	{
	    if(mng_write(mng_h) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
"Unable to write the JNG image to file";
		CLEANUP_RETURN(-1);
	    }
	}

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		ctx->data_height, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		rgba
	    ))
		*ctx->user_aborted = 1;
	}

	if(*ctx->user_aborted)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the RGBA image data to a buffer as a JNG image.
 */
int ImgBufferWriteJNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJNGRGBA(
	    IMG_MNG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    creator, title, author, comments,
	    modified_time_sec,
	    quality,
	    (ImgJNGColorType)color_type,
	    transparency,
	    progress_cb, progress_data
	));
}

/*
 *	Writes the RGBA image data to a stream as a JNG image.
 */
int ImgStreamWriteJNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJNGRGBA(
	    IMG_MNG_DATA_STORAGE_STREAM,
	    NULL, NULL,
	    fp,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    creator, title, author, comments,
	    modified_time_sec,
	    quality,
	    (ImgJNGColorType)color_type,
	    transparency,
	    progress_cb, progress_data
	));
}

/*
 *	Saves the RGBA image data to a JNG image file.
 */
int ImgFileSaveJNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int             status,
			error_code;
	FILE *fp;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the JNG image file for writing */
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    ImgSaveSetError(strerror(errno));
	    return(-1);
	}

	/* Write to the stream */
	status = ImgStreamWriteJNGRGBA(
	    fp,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    creator, title, author, comments,
	    modified_time_sec,
	    quality,
	    color_type,
	    transparency,
	    progress_cb, progress_data
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	Appends a FRAM and IHDR/IEND chunks.
 */
static int ImgFileWriteMNGFrame(
	ImgMNGWriteContext *ctx,
	const int frame_num,
	const unsigned long prev_frame_delay_ms,
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const ImgMNGColorType color_type
)
{
	const int	width = ctx->mng_width,
			height = ctx->mng_height;
	int		*user_aborted = ctx->user_aborted;
	mng_uint32	mng_idat_data_length,
			mng_idat_data_compressed_length;
	const u_int8_t	**rgba_list = ctx->data_list,
			*rgba = rgba_list[frame_num],
			*bg_color = ctx->bg_color;
	const unsigned long	*delay_ms_list = ctx->delay_ms_list,
				delay_ms = (delay_ms_list != NULL) ? delay_ms_list[frame_num] : 0l;
	mng_ptr		mng_idat_data,
			mng_idat_data_compressed;
	mng_retcode mng_rc;
	mng_handle mng_h = ctx->mng_h;

	/* Skip this frame if no RGBA data was specified even though
	 * this should be an error, we still want to save the rest of
	 * the frames so we do not report this as an error
	 */
	if(rgba == NULL)
	    return(0);

	/* Write the FRAM (frame) chunk only if the delay is different
	 * from the previous frame
	 */
	if(delay_ms != prev_frame_delay_ms)
	{
	    if(mng_putchunk_fram(
		mng_h,
		MNG_FALSE,			/* Not empty */
		((color_type == IMG_MNG_COLOR_TYPE_GA) ||
		 (color_type == IMG_MNG_COLOR_TYPE_RGBA)) ?
		    MNG_FRAMINGMODE_3 : MNG_FRAMINGMODE_1,
		0, MNG_NULL,			/* No name for this frame */
		MNG_CHANGEDELAY_NEXTSUBFRAME,	/* Change the delay for this
						 * frame only */
		MNG_CHANGETIMOUT_NO,		/* No change in timeout and
						 * termination */
		MNG_CHANGECLIPPING_NO,		/* No change in clipping */
		MNG_CHANGESYNCID_NO,		/* No change in sync */
		(mng_uint32)(			/* Delay in ticks */
		    delay_ms * ctx->mng_ticks_per_second / 1000l
		),
		0,				/* Timeout (omitted) */
		MNG_BOUNDARY_ABSOLUTE,		/* Boundary type (omitted) */
		0,				/* Left boundary (omitted) */
		0,				/* Right boundary (omitted) */
		0,				/* Top boundary (omitted) */
		0,				/* Bottom boundary (omitted) */
		0,				/* Number of sync IDs (omitted) */
		MNG_NULL			/* Sync IDs list (omitted) */
	    ) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error = "Unable to append MNG chunk FRAM frame control)";
		return(-1);
	    }
	}

	/* Interlace is not supported */
	if(interlaced)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Interlace option is not implmented by the MNG library";
	    errno = EINVAL;
	    return(-2);
	}

	/* Put the IHDR chunk image header) */
	if(mng_putchunk_ihdr(
	    mng_h,
	    width, height,
	    MNG_BITDEPTH_8,
	    ctx->mng_color_type,
	    MNG_COMPRESSION_DEFLATE,
	    MNG_FILTER_ADAPTIVE,
	    MNG_INTERLACE_NONE
	) != MNG_NOERROR)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to append MNG chunk IHDR frame header)";
	    return(-1);
	}
	if(*user_aborted)
	{
	    errno = EINTR;
	    return(-4);
	}

	/* Write the tEXt (text) chunks and tIME chunk if this is the
	 * first frame
	 */
	if(frame_num == 0)
	{
	    const char *s = ctx->title;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_TITLE),
		    MNG_TEXT_TITLE,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->author;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_AUTHOR),
		    MNG_TEXT_AUTHOR,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->creator;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_SOFTWARE),
		    MNG_TEXT_SOFTWARE,
		    strlen(s),
		    (mng_pchar)s
		);
	    s = ctx->comments;
	    if(!STRISEMPTY(s))
		mng_putchunk_text(
		    mng_h,
		    strlen(MNG_TEXT_COMMENT),
		    MNG_TEXT_COMMENT,
		    strlen(s),
		    (mng_pchar)s
		);

	    if(ctx->modified_time_sec > 0l)
	    {
		/* MNG requires that time values be stored in GMT */
		time_t t = (time_t)ctx->modified_time_sec;
		struct tm *tm_ptr = gmtime(&t);
		if(tm_ptr != NULL)
		{
		    mng_putchunk_time(
			mng_h,
			(mng_uint16)tm_ptr->tm_year + 1900,
			(mng_uint8)tm_ptr->tm_mon + 1,
			(mng_uint8)tm_ptr->tm_mday,
			(mng_uint8)tm_ptr->tm_hour,
			(mng_uint8)tm_ptr->tm_min,
			(mng_uint8)tm_ptr->tm_sec
		    );
		}
	    }
	}

	/* Convert the RGBA image data to the MNG image data based
	 * on the specified color type
	 */
	mng_idat_data = MNG_NULL;
	switch(color_type)
	{
	  case IMG_MNG_COLOR_TYPE_BW:
	  case IMG_MNG_COLOR_TYPE_GRAYSCALE:
	    mng_idat_data = ImgMNGConvertRGBAToIDATGreyscale(
		ctx,
		width, height,
		ctx->data_bpl,
		rgba,
		bg_color,
		&mng_idat_data_length
	    );
	    break;

	  case IMG_MNG_COLOR_TYPE_GA:
	    mng_idat_data = ImgMNGConvertRGBAToIDATGreyscaleAlpha(
		ctx,
		width, height,
		ctx->data_bpl,
		rgba,
		bg_color,
		&mng_idat_data_length
	    );
	    break;

	  case IMG_MNG_COLOR_TYPE_RGB:
	    mng_idat_data = ImgMNGConvertRGBAToIDATRGB(
		ctx,
		width, height,
		ctx->data_bpl,
		rgba,
		bg_color,
		&mng_idat_data_length
	    );
	    break;

	  case IMG_MNG_COLOR_TYPE_RGBA:
	    mng_idat_data = ImgMNGConvertRGBAToIDATRGBA(
		ctx,
		width, height,
		ctx->data_bpl,
		rgba,
		bg_color,
		&mng_idat_data_length
	    );
	    break;
	}
	if(mng_idat_data == MNG_NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error =
"Unable to convert the RGBA data to MNG image data";
	    return(-1);
	}
	if(*user_aborted)
	{
	    free(mng_idat_data);
	    errno = EINTR;
	    return(-4);
	}

	/* Compress the converted MNG image data */
	mng_idat_data_compressed = ImgMNGCompressDataZLib(
	    ctx,
	    compression_level,
	    mng_idat_data,
	    mng_idat_data_length,
	    &mng_idat_data_compressed_length
	);

	/* Delete the converted MNG image data */
	free(mng_idat_data);

	if(mng_idat_data_compressed == MNG_NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to compress the MNG image data";
	    return(-1);
	}
	if(*user_aborted)
	{
	    free(mng_idat_data_compressed);
	    errno = EINTR;
	    return(-4);
	}

	/* Write the IDAT (image data) chunk with the compressed
	 * converted MNG image data
	 */
	mng_rc = mng_putchunk_idat(
	    mng_h,
	    mng_idat_data_compressed_length,
	    mng_idat_data_compressed
	);

	/* Delete the compressed converted MNG image data */
	free(mng_idat_data_compressed);

	if(mng_rc != MNG_NOERROR)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error =
"Unable to append the image data (IDAT) chunk";
	    return(-1);
	}
	if(*user_aborted)
	{
	    errno = EINTR;
	    return(-4);
	}

	/* Write the IEND (image data tailer) chunk */
	if(mng_putchunk_iend(mng_h) != MNG_NOERROR)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error =
"Unable to append the image data tailer (IEND) chunk";
	    return(-1);
	}
	if(*user_aborted)
	{
	    errno = EINTR;
	    return(-4);
	}

	return(0);
}

/*
 *	Writes the RGBA image data to a buffer or a stream as a MNG
 *	image.
 */
static int ImgWriteMNGRGBA(
	const ImgMNGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const ImgMNGColorType color_type,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : width * bpp;
	int user_aborted = 0;
	mng_handle mng_h;
	ImgMNGWriteContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 const int error_code = errno;				\
							\
 /* Delete the ImgMNGWriteContext */			\
 if(ctx != NULL) {					\
  /* Shutdown libmng */					\
  if(ctx->mng_h != MNG_NULL) {				\
   (void)mng_cleanup(&ctx->mng_h);			\
   ctx->mng_h = mng_h = MNG_NULL;			\
  }							\
							\
  free(ctx);						\
 }							\
							\
 errno = error_code;					\
							\
 return(_v_);						\
}

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (nframes <= 0) || (rgba_list == NULL)
	)
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Create our MNG write context */
	ctx = IMG_MNG_WRITE_CONTEXT(calloc(1, sizeof(ImgMNGWriteContext)));
	if(ctx == NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/*	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->io_size = 0l; */
	ctx->mng_h = mng_h = MNG_NULL;
	switch(color_type)
	{
	  case IMG_MNG_COLOR_TYPE_BW:
	  case IMG_MNG_COLOR_TYPE_GRAYSCALE:
	    ctx->mng_color_type = MNG_COLORTYPE_GRAY;
	    break;
	  case IMG_MNG_COLOR_TYPE_GA:
	    ctx->mng_color_type = MNG_COLORTYPE_GRAYA;
	    break;
	  case IMG_MNG_COLOR_TYPE_RGB:
	    ctx->mng_color_type = MNG_COLORTYPE_RGB;
	    break;
	  case IMG_MNG_COLOR_TYPE_RGBA:
	    ctx->mng_color_type = MNG_COLORTYPE_RGBA;
	    break;
	}
	ctx->mng_width = width;
	ctx->mng_height = height;
	ctx->mng_ticks_per_second = 1000;	/* Units of milliseconds per
						 * second */
	ctx->mng_nlayers = nframes + 1;		/* nframes + base layer */
	ctx->mng_nframes = nframes;
	ctx->mng_total_play_time_ms = 0l;
	if((nframes > 1) && (delay_ms_list != NULL))
	{
	    int i;
	    for(i = 0; i < nframes; i++)
		ctx->mng_total_play_time_ms += delay_ms_list[i];
	}
	ctx->mng_total_play_time = ctx->mng_total_play_time_ms * ctx->mng_ticks_per_second / 1000;
	ctx->nframes = nframes;
	ctx->data_width = width;
	ctx->data_height = height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = _bpl;
	ctx->data_list = rgba_list;
	ctx->delay_ms_list = delay_ms_list;
	ctx->bg_color = bg_color;
	ctx->creator = creator;
	ctx->title = title;
	ctx->author = author;
	ctx->comments = comments;
	ctx->modified_time_sec = modified_time_sec;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
/*	ctx->cur_progress_level = 0; */
	ctx->nprogress_levels = nframes;
	ctx->user_aborted = &user_aborted;
/*	ctx->status = 0; */

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[0] : NULL
	    ))
		*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Initialize libmng */
	ctx->mng_h = mng_h = mng_initialize(
	    ctx,				/* User data */
	    ImgMNGNewCB,
	    ImgMNGDeleteCB,
	    MNG_NULL				/* No trace */
	);
	if(mng_h == MNG_NULL)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}
	if((mng_setcb_errorproc(mng_h, ImgMNGWriteErrorCB) != MNG_NOERROR) ||
	   (mng_setcb_openstream(mng_h, ImgMNGWriteOpenCB) != MNG_NOERROR) ||
	   (mng_setcb_closestream(mng_h, ImgMNGWriteCloseCB) != MNG_NOERROR) ||
	   (mng_setcb_writedata(mng_h, ImgMNGWriteDataCB) != MNG_NOERROR)
	)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to initialize the MNG Library";
	    CLEANUP_RETURN(-1);
	}

	/* Set libmng to write mode */
	if(mng_create(mng_h) != MNG_NOERROR)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Unable to set the MNG library into write mode";
	    CLEANUP_RETURN(-1);
	}

	/* Begin creating the MNG image in memory, the mng_putchunk*)
	 * functions only create the MNG image in memory, the actual
	 * MNG image file will be written further below when
	 * mng_write) is called
	 */

	/* Write the MHDR (MNG header) chunk */
	if(!(*ctx->user_aborted))
	{
	    if(mng_putchunk_mhdr(
		mng_h,
		width, height,
		ctx->mng_ticks_per_second,
		ctx->mng_nlayers,
		ctx->mng_nframes,
		ctx->mng_total_play_time,	/* Play time in ticks */
		MNG_SIMPLICITY_VALID |		/* Simplicity bits present */
		MNG_SIMPLICITY_SIMPLEFEATURES |	/* Have DEFI, FRAM, MAGN, or global PLTE and tRNS chunks */
/*		MNG_SIMPLICITY_COMPLEXFEATURES | */ 	/* Have BASI, CLON, DHDR/IEND, PAST, DISC, MOVE, CLIP, and SHOW chunks */
		(((color_type == IMG_MNG_COLOR_TYPE_GA) ||
		  (color_type == IMG_MNG_COLOR_TYPE_RGBA)) ?
		    MNG_SIMPLICITY_TRANSPARENCY : 0) |
/*		MNG_SIMPLICITY_DELTAPNG | */
		/* 0x240 */ 0
	    ) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
"Unable to append the MNG header (MHDR) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Write the TERM (termination action) chunk */
	if(!(*ctx->user_aborted))
	{
	    if(looping)
	    {
		if(mng_putchunk_term(
		    mng_h,
		    MNG_TERMACTION_REPEAT,
		    MNG_ITERACTION_LASTFRAME,	/* Show the last frame
						 * indefinitely after all
						 * iterations */
		    (delay_ms_list != NULL) ?
			MAX(delay_ms_list[nframes - 1], 1) : 1,
		    0x7FFFFFFF			/* Repeat infinitly */
		) != MNG_NOERROR)
		{
		    if(imgio_last_save_error == NULL)
			imgio_last_save_error =
 "Unable to append terminate action (TERM) chunk";
		     CLEANUP_RETURN(-1);
		}
	    }
	    else
	    {
		if(mng_putchunk_term(
		    mng_h,
		    MNG_TERMACTION_LASTFRAME,
		    MNG_ITERACTION_LASTFRAME,	/* Show the last frame
						 * indefinitely after the
						 * iteration */
		    (delay_ms_list != NULL) ?
			MAX(delay_ms_list[nframes - 1], 1) : 1,
		    1				/* Play once (one iteration) */
		) != MNG_NOERROR)
		{
		    if(imgio_last_save_error == NULL)
			imgio_last_save_error =
"Unable to append the terminate action (TERM) chunk";
		     CLEANUP_RETURN(-1);
		}
	    }
	}

	/* Write the BACK and bKGD (background) chunks */
	if(!(*ctx->user_aborted) && (ctx->bg_color != NULL))
	{
	    const u_int8_t	*c = ctx->bg_color,
				c_grey = ImgConvertPixelRGBToGreyscale(c);

	    /* Write the BACK chunk only if we do not want
	     * transparency, since the BACK chunk will cause any
	     * transparent pixels found in any frame to be rendered
	     * against the background and result in fully opaque
	     * pixels
	     */
	    switch(color_type)
	    {
	      case IMG_MNG_COLOR_TYPE_BW:
	      case IMG_MNG_COLOR_TYPE_GRAYSCALE:
	      case IMG_MNG_COLOR_TYPE_RGB:
		if(mng_putchunk_back(
		    mng_h,
		    ((mng_uint16)c[0] << 8) | (mng_uint16)c[0],
		    ((mng_uint16)c[1] << 8) | (mng_uint16)c[1],
		    ((mng_uint16)c[2] << 8) | (mng_uint16)c[2],
		    MNG_BACKGROUNDCOLOR_MANDATORY,
		    0,				/* No background image ID */
		    MNG_BACKGROUNDIMAGE_NOTILE
		) != MNG_NOERROR)
		{
		    if(imgio_last_save_error == NULL)
			imgio_last_save_error =
 "Unable to append the background color and image (BACK) chunk";
		    CLEANUP_RETURN(-1);
		}
		break;
	      case 2:				/* Greyscale Alpha */
	      case 4:				/* RGBA */
		break;
	    }

	    if(mng_putchunk_bkgd(
		mng_h,
		MNG_FALSE,			/* Not empty */
		MNG_COLORTYPE_RGB,		/* Must be MNG_COLORTYPE_RGB
						 * regardless of what is set
						 * in the MHDR */
		0,				/* Index */
		((mng_uint16)c_grey << 8) | (mng_uint16)c_grey,
		((mng_uint16)c[0] << 8) | (mng_uint16)c[0],
		((mng_uint16)c[1] << 8) | (mng_uint16)c[1],
		((mng_uint16)c[2] << 8) | (mng_uint16)c[2]
	    ) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
 "Unable to append the background color (bKGD) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Append each frame */
	if(!(*ctx->user_aborted))
	{
	    int i;
	    unsigned long prev_frame_delay_ms = 0l;

	    for(i = 0; i < nframes; i++)
	    {
		ctx->cur_progress_level = i;
		if(!progress_cb(
		    progress_data,
		    i, ctx->nframes,
		    ctx->data_width, ctx->data_height,
		    ctx->data_bpl, ctx->data_bpp,
		    ctx->data_list[i]
		))
		{
		    *ctx->user_aborted = 1;
		    break;
		}

		ctx->status = ImgFileWriteMNGFrame(
		    ctx,
		    i,
		    prev_frame_delay_ms,
		    compression_level,
		    interlaced,
		    color_type
		);
		if(ctx->status != 0)
		    break;

		if(*ctx->user_aborted)
		{
		    ctx->status = -4;
		    break;
		}

		if(delay_ms_list != NULL)
		    prev_frame_delay_ms = delay_ms_list[i];
	    }
	}

	/* Tailer */
	if(!(*ctx->user_aborted))
	{
	    if(mng_putchunk_mend(mng_h) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error =
 "Unable to append the MNG tailer (MEND) chunk";
		 CLEANUP_RETURN(-1);
	    }
	}

	/* Write the MNG image data in memory to file */
	if(!(*ctx->user_aborted))
	{
	    if(mng_write(mng_h) != MNG_NOERROR)
	    {
		if(imgio_last_save_error == NULL)
		    imgio_last_save_error = "Unable to write the MNG image data to file";
		CLEANUP_RETURN(-1);
	    }
	}

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		ctx->data_height, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	if(*ctx->user_aborted)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the RGBA image data to a buffer as a MNG image.
 */
int ImgBufferWriteMNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteMNGRGBA(
	    IMG_MNG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    x, y,
	    base_width, base_height,
	    creator, title, author, comments,
	    modified_time_sec,
	    compression_level,
	    interlaced,
	    (ImgMNGColorType)color_type,
	    looping,
	    progress_cb, progress_data
	));
}

/*
 *	Writes the RGBA image data to a stream as a MNG image.
 */
int ImgStreamWriteMNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteMNGRGBA(
	    IMG_MNG_DATA_STORAGE_STREAM,
	    NULL, NULL,
	    fp,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    x, y,
	    base_width, base_height,
	    creator, title, author, comments,
	    modified_time_sec,
	    compression_level,
	    interlaced,
	    (ImgMNGColorType)color_type,
	    looping,
	    progress_cb, progress_data
	));
}

/*
 *	Saves the RGBA image data to an MNG image file.
 */
int ImgFileSaveMNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = progressive
						 * 1 = interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the last save error message pointer */
	imgio_last_save_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the MNG image file for writing */
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    ImgSaveSetError(strerror(errno));
	    return(-1);
	}

	/* Write to the stream */
	status = ImgStreamWriteMNGRGBA(
	    fp,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    x, y,
	    base_width, base_height,
	    creator, title, author, comments,
	    modified_time_sec,
	    compression_level,
	    interlaced,
	    (ImgMNGColorType)color_type,
	    looping,
	    progress_cb, progress_data
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

#endif	/* HAVE_LIBPNG */
