#ifdef HAVE_LIBJPEG
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <setjmp.h>
#include <jpeglib.h>				/* libjpeg */
#include "imgio.h"


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


typedef struct _ImgJPEGErrorContext		ImgJPEGErrorContext;
#define IMG_JPEG_ERROR_CONTEXT(p)		((ImgJPEGErrorContext *)(p))
typedef struct _ImgJPEGProgressContext		ImgJPEGProgressContext;
#define IMG_JPEG_PROGRESS_CONTEXT(p)		((ImgJPEGProgressContext *)(p))
typedef struct _ImgJPEGReadContext		ImgJPEGReadContext;
#define IMG_JPEG_READ_CONTEXT(p)		((ImgJPEGReadContext *)(p))
typedef struct _ImgJPEGWriteContext		ImgJPEGWriteContext;
#define IMG_JPEG_WRITE_CONTEXT(p)		((ImgJPEGWriteContext *)(p))


/*
 *	Data Storage Type:
 */
typedef enum {
	IMG_JPEG_DATA_STORAGE_BUFFER,
	IMG_JPEG_DATA_STORAGE_STREAM
} ImgJPEGDataStorageType;


/* Version */
void ImgJPEGVersion(int *major, int *minor, int *release);

/* Check Type */
int ImgBufferIsJPEG(const u_int8_t *data, const int len);
int ImgStreamIsJPEG(FILE *fp);
int ImgFileIsJPEG(const char *path);

/* Callbacks */
static void ImgJPEGErrorExitCB(j_common_ptr jinfo);
static void ImgJPEGErrorEmitMessageCB(j_common_ptr jinfo, int level);
static void ImgJPEGErrorOutputMessageCB(j_common_ptr jinfo);
static void ImgJPEGProgressMonitorCB(j_common_ptr jinfo);

/* Open */
static void ImgJPEGReadOpenCB(j_decompress_ptr dinfo);
static boolean ImgJPEGReadBufferCB(j_decompress_ptr dinfo);
static boolean ImgJPEGReadStreamCB(j_decompress_ptr dinfo);
static void ImgJPEGReadBufferSeekCB(j_decompress_ptr dinfo, long nbytes);
static void ImgJPEGReadStreamSeekCB(j_decompress_ptr dinfo, long nbytes);
static void ImgJPEGReadCloseCB(j_decompress_ptr dinfo);
static j_decompress_ptr ImgJPEGCreateDecompressionManager(
	const ImgJPEGDataStorageType data_storage_type,
	ImgJPEGErrorContext **error_ctx_rtn,
	ImgJPEGProgressContext **progress_ctx_rtn,
	ImgJPEGReadContext **read_ctx_rtn
);
static int ImgReadJPEGRGBA(
	const ImgJPEGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadJPEGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadJPEGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenJPEGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgReadJPEGRGBAThumb(
	const ImgJPEGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadJPEGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadJPEGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenJPEGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);

/* Save */
static void ImgJPEGWriteOpenCB(j_compress_ptr cinfo);
static boolean ImgJPEGWriteBufferCB(j_compress_ptr cinfo);
static boolean ImgJPEGWriteStreamCB(j_compress_ptr cinfo);
static void ImgJPEGWriteCloseCB(j_compress_ptr cinfo);
static j_compress_ptr ImgJPEGCreateCompressionManager(
	const ImgJPEGDataStorageType data_storage_type,
	ImgJPEGErrorContext **error_ctx_rtn,
	ImgJPEGProgressContext **progress_ctx_rtn,
	ImgJPEGWriteContext **write_ctx_rtn
);
static int ImgWriteJPEGGreyscale(
	const ImgJPEGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,				/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWriteJPEGGreyscale(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWriteJPEGGreyscale(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveJPEGGreyscale(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
static int ImgWriteJPEGRGBA(
	const ImgJPEGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWriteJPEGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWriteJPEGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveJPEGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */         
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Error Context:
 */
struct _ImgJPEGErrorContext {

	/* JPEG members */
	struct jpeg_error_mgr	jpeg_err;	/* JPEG members (must be the
						 * first member in this struct) */

	/* Our members */
	jmp_buf		jpeg_jmp_buf;		/* For return to caller */

	int		status;
};

/*
 *	Progress Context:
 */
struct _ImgJPEGProgressContext {

	/* JPEG members */
	struct jpeg_progress_mgr	jpeg_progress;	/* JPEG members (must be the
						 * first member in this struct) */

	/* Our members */
	int		width, height,
			bpp, bpl;
	const u_int8_t	*data;
	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;
};

/*
 *	Read Context:
 */
struct _ImgJPEGReadContext {

	/* JPEG members */
	struct jpeg_source_mgr	jpeg_src;	/* JPEG members (must be the
						 * first member in this struct) */

	/* Our members */
	ImgJPEGDataStorageType	data_storage_type;
	boolean		input_opened;
	const void	*bp;
	unsigned long	bp_len,
			bp_pos;
	FILE		*fp;
	unsigned long	fp_len;

	JOCTET		*io_buf;
	size_t		io_buf_len;

	int		jpeg_width,		/* Original size of image */
			jpeg_height;

	/* Loaded image data */
	int		width, height,
			bpp, bpl;
	u_int8_t	*data;
};

/*
 *	Write Context:
 */
struct _ImgJPEGWriteContext {

	/* JPEG members */
	struct jpeg_destination_mgr	jpeg_dest;	/* JPEG members (must be the
						 * first member in this struct) */

	/* Our members */
	ImgJPEGDataStorageType	data_storage_type;
	boolean		output_opened;
	void		**bp;
	unsigned long	*bp_len;
	FILE		*fp;

	JOCTET		*io_buf;
	size_t		io_buf_len;

	/* Image data to be written */
	int		width, height,
			bpp, bpl;
	const u_int8_t	*data;

	int		quality;		/* 0 to 100 */
};


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_JPEG_PROGRESS_RESOLUTION	10


/*
 *	Suggested IO Buffer Length (in bytes):
 */
#define IMG_JPEG_DEF_IO_SIZE		1024

/*
 *	Minimum IO Buffer Length (in bytes):
 *
 *	Must be big enough to hold one JPEG marker.
 */
#define IMG_JPEG_MIN_IO_SIZE		4


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the JPEG library's version.
 */
void ImgJPEGVersion(int *major, int *minor, int *release) 
{
	if(major != NULL)
	    *major = JPEG_LIB_VERSION / 10;
	if(minor != NULL)
	    *minor = JPEG_LIB_VERSION % 10;
	if(release != NULL)
	    *release = 0;
}


/*
 *      Checks for the JPEG type ID.
 */
int ImgBufferIsJPEG(const u_int8_t *data, const int len)
{
	if((data == NULL) || (len < 2))
	{
	    errno = EINVAL;
	    return(0);
	}

	if((data[0] == 0xFF) && (data[1] == 0xD8))
	    return(1);

	errno = EINVAL;
	return(0);
}

int ImgStreamIsJPEG(FILE *fp)
{
	u_int8_t buf[2];

	size_t units_read;

	if(fp == NULL)
	{
	    errno = EINVAL;
	    return(0);
	}

	units_read = fread(
	    buf,
	    sizeof(u_int8_t),
	    sizeof(buf) / sizeof(u_int8_t),
	    fp
	);
	if(units_read == (sizeof(buf) / sizeof(u_int8_t)))
	    return(ImgBufferIsJPEG(buf, sizeof(buf)));
	else
	    return(0);
}

int ImgFileIsJPEG(const char *path)
{
	int             status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
	    return(0);

	status = ImgStreamIsJPEG(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	JPEG error exit callback.
 */
static void ImgJPEGErrorExitCB(j_common_ptr jinfo)
{
	ImgJPEGErrorContext *ctx = IMG_JPEG_ERROR_CONTEXT(jinfo->err);

	/* Return control to the setjmp point (this function does not
	 * return to libjpeg)
	 */
	longjmp(ctx->jpeg_jmp_buf, 1);
}

/*
 *	JPEG error emit message callback.
 */
static void ImgJPEGErrorEmitMessageCB(j_common_ptr jinfo, int level)
{
	ImgJPEGErrorContext *ctx = IMG_JPEG_ERROR_CONTEXT(jinfo->err);
	struct jpeg_error_mgr *err = &ctx->jpeg_err;

	/* Count this message */
	err->num_warnings++;

	/* Severe error? */
	if(level < 0)
	{
	    if((*err->output_message) != NULL)
		(*err->output_message)(jinfo);
	    if(ctx->status == 0)
		ctx->status = -1;
	}
}

/*
 *	JPEG error output message callback.
 */
static void ImgJPEGErrorOutputMessageCB(j_common_ptr jinfo)
{
	ImgJPEGErrorContext *ctx = IMG_JPEG_ERROR_CONTEXT(jinfo->err);
	struct jpeg_error_mgr *err = &ctx->jpeg_err;
	char *msg = (char *)malloc((JMSG_LENGTH_MAX + 1) * sizeof(char));
	if(msg != NULL)
	{
	    if((*err->format_message) != NULL)
	    {
		(*err->format_message)(jinfo, msg);
		if(jinfo->is_decompressor)
		{
		    if(imgio_last_open_error == NULL)
			ImgOpenSetError(msg);
		}
		else
		{
		    if(imgio_last_save_error == NULL)
			ImgSaveSetError(msg);
		}
	    }
	    free(msg);
	}
}


/*
 *	JPEG read progress callback.
 */
static void ImgJPEGProgressMonitorCB(j_common_ptr jinfo)
{
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(jinfo->err);
	ImgJPEGProgressContext *progress_ctx = IMG_JPEG_PROGRESS_CONTEXT(jinfo->progress);
	struct jpeg_progress_mgr *progress = &progress_ctx->jpeg_progress;

	if(progress_ctx->user_aborted != NULL)
	{
	    if(*progress_ctx->user_aborted)
		return;
	}

	if((progress_ctx->progress_cb != NULL) &&
	   ((progress->pass_counter % IMG_JPEG_PROGRESS_RESOLUTION) == 0)
	)
	{
	    const int	i = (int)progress->pass_counter,
			n = (int)progress->pass_limit,
			cur_level = progress->completed_passes,
			nlevels = progress->total_passes;
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		(cur_level * n) + i,
		(nlevels * n),
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
	    {
		if(error_ctx->status == 0)
		    error_ctx->status = -4;
		*progress_ctx->user_aborted = 1;
		errno = EINTR;
	    }
	}
}


/*
 *	JPEG read open callback.
 */
static void ImgJPEGReadOpenCB(j_decompress_ptr dinfo)
{
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(dinfo->err);
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);
	switch(ctx->data_storage_type)
	{
	  case IMG_JPEG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		/* Check if there is no JPEG header signature in the data */
		const void *bp = ctx->bp;
		const unsigned long bp_len = ctx->bp_len;
		if(!ImgBufferIsJPEG((const u_int8_t *)bp, (int)bp_len))
		{
		    imgio_last_open_error = "Not a JPEG image";
		    error_ctx->status = -2;
		    return;
		}

		/* Allocate the IO buffer */
		if(ctx->io_buf == NULL)
		{
		    ctx->io_buf_len = IMG_JPEG_MIN_IO_SIZE;
		    ctx->io_buf = (JOCTET *)(*dinfo->mem->alloc_small)(
			(j_common_ptr)dinfo,
			JPOOL_PERMANENT,	/* Stay allocated until the
						 * master record is destroyed */
			ctx->io_buf_len * sizeof(JOCTET)
		    );
		    if(ctx->io_buf == NULL)
		    {
			ctx->io_buf_len = 0l;
			imgio_last_open_error = "Memory allocation error";
			error_ctx->status = -3;
			return;
		    }
		}
	    }
	    else
	    {
		imgio_last_open_error = "Buffer not specified";
		error_ctx->status = -2;
		return;
	    }
	    break;

	  case IMG_JPEG_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		/* Check if there is not JPEG header signature in the stream */
		FILE *fp = ctx->fp;
		const long fp_pos = ftell(fp);
		int status = ImgStreamIsJPEG(fp);
		(void)fseek(fp, fp_pos, SEEK_SET);
		if(!status)
		{
		    imgio_last_open_error = "Not a JPEG image";
		    error_ctx->status = -2;
		    return;
		}

		/* Allocate the read buffer */
		if(ctx->io_buf == NULL)
		{
		    struct stat stat_buf;
		    if(fstat(fileno(fp), &stat_buf))
		    {
			ctx->fp_len = 0l;
			ctx->io_buf_len = IMG_JPEG_DEF_IO_SIZE;
		    }
		    else
		    {
			ctx->fp_len = (unsigned long)stat_buf.st_size;
			ctx->io_buf_len = MAX(
			    stat_buf.st_blksize,
			    IMG_JPEG_MIN_IO_SIZE
			);
		    }
		    ctx->io_buf = (JOCTET *)(*dinfo->mem->alloc_small)(
			(j_common_ptr)dinfo,
			JPOOL_PERMANENT,	/* Stay allocated until the
						 * master record is destroyed */
			ctx->io_buf_len * sizeof(JOCTET)
		    );
		    if(ctx->io_buf == NULL)
		    {
			ctx->io_buf_len = 0l;
			imgio_last_open_error = "Memory allocation error";
			error_ctx->status = -3;
			return;
		    }
		}
	    }
	    else
	    {
		imgio_last_open_error = "Stream not specified";
		error_ctx->status = -2;
		return;
	    }
	    break;
	}

	/* Mark that the input has been opened */
	ctx->input_opened = TRUE;
}

/*
 *	JPEG read buffer callback.
 */
static boolean ImgJPEGReadBufferCB(j_decompress_ptr dinfo)
{
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(dinfo->err);
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);
	struct jpeg_source_mgr *src = &ctx->jpeg_src;
	const void *bp = ctx->bp;
	const unsigned long bp_len = ctx->bp_len;
	JOCTET *io_buf = ctx->io_buf;
	const unsigned long io_buf_len = ctx->io_buf_len;

	/* Buffer not specified or error? */
	if((bp == NULL) || (bp_len == 0l) ||
	   (io_buf == NULL) || (io_buf_len == 0l)
	)
	{
	    src->next_input_byte = NULL;
	    src->bytes_in_buffer = 0l;
	    if(error_ctx->status == 0)
		error_ctx->status = -1;
	    return(FALSE);
	}

	/* Is there more data in the input buffer to read? */
	if(ctx->bp_pos < bp_len)
	{
	    /* Calculate the remaining number of bytes in the input
	     * buffer
	     */
	    unsigned long len = bp_len - ctx->bp_pos;

	    /* Tell libjpeg to read the remaining data in the input
	     * buffer
	     */
	    src->next_input_byte = (JOCTET *)(bp + ctx->bp_pos);
	    src->bytes_in_buffer = (size_t)len;

	    /* Increment the buffer position */
	    ctx->bp_pos += len;

	    return(TRUE);
	}
	else
	{
	    /* End of buffer reached
	     *
	     * Set the fake EOI marker in the IO buffer and give it to
	     * libjpeg so that libjpeg will detect a premature end of
	     * data and stop calling this function
	     */
	    JOCTET *tar = io_buf;
	    tar[0] = 0xFF;
	    tar[1] = JPEG_EOI;
	    src->next_input_byte = tar;
	    src->bytes_in_buffer = 2l * sizeof(JOCTET);
	    return(FALSE);
	}
}

/*
 *	JPEG read stream or file callback.
 */
static boolean ImgJPEGReadStreamCB(j_decompress_ptr dinfo)
{
	size_t		units_read,
			units_to_read;
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(dinfo->err);
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);
	struct jpeg_source_mgr *src = &ctx->jpeg_src;
	FILE *fp = ctx->fp;
	JOCTET *io_buf = ctx->io_buf;
	const unsigned long io_buf_len = ctx->io_buf_len;

	/* Stream not opened and/or error or IO buffer not allocated? */
	if((fp == NULL) ||
	   (io_buf == NULL) || (io_buf_len == 0l)
	)
	{
	    src->next_input_byte = NULL;
	    src->bytes_in_buffer = 0;
	    if(error_ctx->status == 0)
		error_ctx->status = -1;
	    return(FALSE);
	}

	/* Set the amount of data to read as the entire size of the
	 * IO buffer
	 */
	units_to_read = (size_t)(io_buf_len / sizeof(JOCTET));

	/* Read the next block */
	units_read = fread(
	    io_buf,
	    sizeof(JOCTET),
	    units_to_read,
	    fp
	);
	if(units_read != units_to_read)
	{
	    if(ferror(fp))
	    {
		ImgOpenSetError(strerror(errno));
		error_ctx->status = -1;
	    }
	}

	if(units_read > 0l)
	{
	    src->next_input_byte = (JOCTET *)io_buf;
	    src->bytes_in_buffer = units_read * sizeof(JOCTET);
	    return(TRUE);
	}
	else
	{
#if 0
/* Setting src->bytes_in_buffer to 0 would produce a "suspend"
 * behavior, meaning that this function will be called repeatedly
 * until all the data is received
 */
	    src->next_input_byte = NULL;
	    src->bytes_in_buffer = 0l;
#endif
	    /* End of buffer reached
	     *
	     * Set the fake EOI marker in the IO buffer and give it to
	     * libjpeg so that libjpeg will detect a premature end of
	     * data and stop calling this function
	     */
	    JOCTET *tar = io_buf;
	    tar[0] = 0xFF;
	    tar[1] = JPEG_EOI;
	    src->next_input_byte = (JOCTET *)tar;
	    src->bytes_in_buffer = 2l * sizeof(JOCTET);
	    return(FALSE);
	}
}

/*
 *	JPEG read buffer seek callback.
 */
static void ImgJPEGReadBufferSeekCB(j_decompress_ptr dinfo, long nbytes)
{
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);
	struct jpeg_source_mgr *src = &ctx->jpeg_src;

	while(nbytes >= (long)src->bytes_in_buffer)
	{
	    nbytes -= (long)src->bytes_in_buffer;
	    if(ImgJPEGReadBufferCB(dinfo) == FALSE)
		return;
	}

	src->bytes_in_buffer -= nbytes;
	src->next_input_byte += nbytes;
}

/*
 *	JPEG read stream or file seek callback.
 */
static void ImgJPEGReadStreamSeekCB(j_decompress_ptr dinfo, long nbytes)
{
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);
	struct jpeg_source_mgr *src = &ctx->jpeg_src;

	while(nbytes >= (long)src->bytes_in_buffer)
	{
	    nbytes -= (long)src->bytes_in_buffer;
	    if(ImgJPEGReadStreamCB(dinfo) == FALSE)
		return;
	}

	src->bytes_in_buffer -= nbytes;
	src->next_input_byte += nbytes;
}

/*
 *	JPEG read close callback.
 */
static void ImgJPEGReadCloseCB(j_decompress_ptr dinfo)
{
	ImgJPEGReadContext *ctx = IMG_JPEG_READ_CONTEXT(dinfo->src);

	/* Delete the read image data */
	free(ctx->data);
	ctx->data = NULL;

	/* Close the input */
	switch(ctx->data_storage_type)
	{
	  case IMG_JPEG_DATA_STORAGE_BUFFER:
	    ctx->bp = NULL;			/* Read-only, do not delete  */
	    ctx->bp_len = 0l;
	    ctx->bp_pos = 0l;
	    break;
	  case IMG_JPEG_DATA_STORAGE_STREAM:
	    ctx->fp = NULL;			/* Set by calling function,
						 * do not close  */
	    ctx->fp_len = 0l;
	    break;
	}

	/* Mark that the inputs have been closed */
	ctx->input_opened = FALSE;
}

/*
 *	Creates and sets up a new JPEG decompression manager for
 *	reading from a file.
 */
static j_decompress_ptr ImgJPEGCreateDecompressionManager(
	const ImgJPEGDataStorageType data_storage_type,
	ImgJPEGErrorContext **error_ctx_rtn,
	ImgJPEGProgressContext **progress_ctx_rtn,
	ImgJPEGReadContext **read_ctx_rtn
)
{
	j_decompress_ptr dinfo;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGReadContext *ctx = NULL;

	/* Reset the return values */
	*error_ctx_rtn = error_ctx;
	*progress_ctx_rtn = progress_ctx;
	*read_ctx_rtn = ctx;

	/* Allocate the JPEG decompression manager */
	dinfo = (j_decompress_ptr)calloc(1, sizeof(struct jpeg_decompress_struct));
	if(dinfo == NULL)
	    return(NULL);

	/* Initialize the JPEG decompression manager */
	jpeg_create_decompress(dinfo);

	/* Allocate our error context who's size will be large enough
	 * to accomidate the jpeg_error_mgr and our error context
	 * members, this allows this error context to be used as both
	 * the jpeg_error_mgr and our error context
	 */
	if(dinfo->err == NULL)
	{
	    struct jpeg_error_mgr *err;
	    error_ctx = (ImgJPEGErrorContext *)(*dinfo->mem->alloc_small)(
		(j_common_ptr)dinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGErrorContext)
	    );
	    if(error_ctx == NULL)
	    {
		jpeg_destroy_decompress(dinfo);
		free(dinfo);
		return(NULL);
	    }

	    dinfo->err = jpeg_std_error(&error_ctx->jpeg_err);

	    err = &error_ctx->jpeg_err;
	    err->error_exit = ImgJPEGErrorExitCB;
	    err->emit_message = ImgJPEGErrorEmitMessageCB;
	    err->output_message = ImgJPEGErrorOutputMessageCB;

	    (void)memset(
		error_ctx->jpeg_jmp_buf,
		0x00,
		sizeof(jmp_buf)
	    );
	    error_ctx->status = 0;
	}

	/* Allocate our progress context who's size will be large
	 * enough to accomidate the jpeg_progress_mgr and our
	 * progress context members, this allows this progress
	 * context to be used as both the jpeg_progress_mgr and our
	 * progress context
	 */
	if(dinfo->progress == NULL)
	{
	    struct jpeg_progress_mgr *progress;
	    dinfo->progress = (struct jpeg_progress_mgr *)(*dinfo->mem->alloc_small)(
		(j_common_ptr)dinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGProgressContext)
	    );
	    if(dinfo->progress == NULL)
	    {
		jpeg_destroy_decompress(dinfo);
		free(dinfo);
		return(NULL);
	    }

	    progress_ctx = (ImgJPEGProgressContext *)dinfo->progress;

	    progress = &progress_ctx->jpeg_progress;
	    progress->progress_monitor = ImgJPEGProgressMonitorCB;
	    progress->pass_counter = 0l;
	    progress->pass_limit = 0l;
	    progress->completed_passes = 0;
	    progress->total_passes = 0;

	    progress_ctx->width = 0;
	    progress_ctx->height = 0;
	    progress_ctx->bpp = 0;
	    progress_ctx->bpl = 0;
	    progress_ctx->data = NULL;
	    progress_ctx->progress_cb = NULL;
	    progress_ctx->progress_data = NULL;
	    progress_ctx->user_aborted = NULL;
	}

	/* Allocate our read context who's size will be large enough
	 * to accomidate the jpeg_source_mgr and our read context
	 * members, this allows the read context to be used as both
	 * the jpeg_source_mgr and our read context
	 */
	if(dinfo->src == NULL)
	{
	    struct jpeg_source_mgr *src;
	    dinfo->src = (struct jpeg_source_mgr *)(*dinfo->mem->alloc_small)(
		(j_common_ptr)dinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGReadContext)
	    );
	    if(dinfo->src == NULL)
	    {
		jpeg_destroy_decompress(dinfo);
		free(dinfo);
		return(NULL);
	    }

	    ctx = (ImgJPEGReadContext *)dinfo->src;

	    src = &ctx->jpeg_src;
	    src->next_input_byte = NULL;
	    src->bytes_in_buffer = 0;
	    switch(data_storage_type)
	    {
	      case IMG_JPEG_DATA_STORAGE_BUFFER:
		src->init_source = ImgJPEGReadOpenCB;
		src->fill_input_buffer = ImgJPEGReadBufferCB;
		src->skip_input_data = ImgJPEGReadBufferSeekCB;
		src->resync_to_restart = jpeg_resync_to_restart;	/* Use default */
		src->term_source = ImgJPEGReadCloseCB;
		break;
	      case IMG_JPEG_DATA_STORAGE_STREAM:
		src->init_source = ImgJPEGReadOpenCB;
		src->fill_input_buffer = ImgJPEGReadStreamCB;
		src->skip_input_data = ImgJPEGReadStreamSeekCB;
		src->resync_to_restart = jpeg_resync_to_restart;	/* Use default */
		src->term_source = ImgJPEGReadCloseCB;
		break;
	    }
	    ctx->data_storage_type = data_storage_type;
	    ctx->input_opened = FALSE;
	    ctx->bp = NULL;
	    ctx->bp_len = 0l;
	    ctx->bp_pos = 0l;
	    ctx->fp = NULL;
	    ctx->fp_len = 0l;
	    ctx->io_buf = NULL;
	    ctx->io_buf_len = 0;
	    ctx->jpeg_width = 0;
	    ctx->jpeg_height = 0;
	    ctx->width = 0;
	    ctx->height = 0;
	    ctx->bpp = 0;
	    ctx->bpl = 0;
	    ctx->data = NULL;
	}

	*error_ctx_rtn = error_ctx;
	*progress_ctx_rtn = progress_ctx;
	*read_ctx_rtn = ctx;

	return(dinfo);
}

/*
 *	Opens the JPEG image from a buffer or a stream to RGBA image
 *	data.
 */
static int ImgReadJPEGRGBA(
	const ImgJPEGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		jpeg_samples_per_row,
			jpeg_compoents_per_pixel;
	JSAMPARRAY jpeg_row_buf;
	J_COLOR_SPACE jpeg_color_space;
	j_decompress_ptr dinfo = NULL;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_rtn_val_)	{	\
 const int error_code = errno;			\
						\
 /* Delete the loaded image data, close the	\
  * file, and delete the JPEG decompression	\
  * manager					\
  */						\
 if(dinfo != NULL) {				\
  if(ctx != NULL) {				\
   /* If the user aborted or an error occured	\
    * then jpeg_finish_decompress() would not	\
    * have been called and we need to call	\
    * ImgJPEGReadCloseCB() manually here	\
    */						\
   if(ctx->input_opened) {			\
    ImgJPEGReadCloseCB(dinfo);			\
   }						\
  }						\
						\
  jpeg_destroy_decompress(dinfo);		\
  free(dinfo);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_rtn_val_);				\
}

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	/* Create and set up the JPEG decompression manager */
	dinfo = ImgJPEGCreateDecompressionManager(
	    data_storage_type,
	    &error_ctx,
	    &progress_ctx,
	    &ctx
	);
	if((dinfo == NULL) || (error_ctx == NULL) ||
	   (progress_ctx == NULL) || (ctx == NULL)
	)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
	ctx->bpp = bpp;
	progress_ctx->progress_cb = progress_cb;
	progress_ctx->progress_data = progress_data;
	progress_ctx->user_aborted = user_aborted;

	/* Set the libjpeg error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libjpeg's functions encounters an error, libjpeg will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(error_ctx->jpeg_jmp_buf))
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Check for user abort */
	if(*progress_ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}


	/* Read the header
	 *
	 * This will call ImgJPEGReadOpenCB() to open the file
	 */
	jpeg_read_header(dinfo, TRUE);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Modify the output values */

	/* Check the color space (JPEG image data format) specified by
	 * the JPEG header, the output color space must be RGB or
	 * Greyscale
	 */
	switch(dinfo->jpeg_color_space)
	{
	  case JCS_UNKNOWN:
	  case JCS_RGB:
	  case JCS_YCbCr:			/* YUV */
	  case JCS_CMYK:
	  case JCS_YCCK:
	    dinfo->out_color_space = JCS_RGB;
	    break;
	  case JCS_GRAYSCALE:
	    dinfo->out_color_space = JCS_GRAYSCALE;
	    break;
	}

	/* Add other output options here */


	/* With the output values set we can now start decompressing
	 * the JPEG data
	 */
	jpeg_start_decompress(dinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Calculate the number of JSAMPLEs per row in the output
	 * buffer
	 */
	jpeg_color_space = dinfo->jpeg_color_space;
	jpeg_compoents_per_pixel = dinfo->output_components;
	jpeg_samples_per_row = dinfo->output_width * jpeg_compoents_per_pixel;

	/* Allocate a single row sample array for decompressing the
	 * JPEG image data, this allocation will automatically be deleted
	 * when the JPEG decompression object is deleted
	 */
	jpeg_row_buf = (*dinfo->mem->alloc_sarray)(
	    (j_common_ptr)dinfo,
	    JPOOL_IMAGE,
	    jpeg_samples_per_row,
	    1
	);

	/* Get the JPEG image data obtained from the header */
	ctx->jpeg_width = (int)dinfo->output_width;
	ctx->jpeg_height = (int)dinfo->output_height;

	/* Calculate the target RGBA image values */
	ctx->width = ctx->jpeg_width;
	ctx->height = ctx->jpeg_height;
	ctx->bpl = ctx->width * ctx->bpp;

	/* Allocate the target RGBA image data */
	if(ctx->data != NULL)
	    ctx->data = (u_int8_t *)realloc(
		ctx->data,
		ctx->bpl * ctx->height
	    );
	else
	    ctx->data = (u_int8_t *)calloc(
		ctx->bpl * ctx->height,
		sizeof(u_int8_t)
	    );
	if(ctx->data == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}

	/* Set the progress context values */
	progress_ctx->width = ctx->width;
	progress_ctx->height = ctx->height;
	progress_ctx->bpp = ctx->bpp;
	progress_ctx->bpl = ctx->bpl;
	progress_ctx->data = ctx->data;

	/* Report the initial progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		0,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* Begin opening the JPEG image file to RGBA image data */
	if(!(*progress_ctx->user_aborted))
	{
	    const int	jpeg_height = ctx->jpeg_height,
			twidth = ctx->width,
			tbpp = ctx->bpp,
			tbpl = ctx->bpl;
	    int y;
	    u_int8_t	grey,
			*rgba = ctx->data,	/* RGBA */
			*rgba_ptr,
			*rgba_end;
	    JSAMPROW	src_row = jpeg_row_buf[0],
			src_ptr;

	    /* Begin reading each line from the JPEG decompressor
	     *
	     * Use the JPEG library's state variable
	     * dinfo->output_scanline as the row counter
	     */
	    while((dinfo->output_scanline < jpeg_height) &&
		  (error_ctx->status == 0) &&
		  !(*progress_ctx->user_aborted)
	    )
	    {
		y = dinfo->output_scanline;

		/* Read the next line */
		jpeg_read_scanlines(
		    dinfo,
		    jpeg_row_buf,
		    1				/* Max lines to read */
		);

		/* Copy the line to the target RGBA buffer */
		src_ptr = src_row;
		rgba_ptr = rgba + (y * tbpl);
		rgba_end = rgba_ptr + (twidth * tbpp);
		switch(jpeg_compoents_per_pixel)
		{
		  case 4:
		    while(rgba_ptr < rgba_end)
		    {
			*(u_int32_t *)rgba_ptr = *(const u_int32_t *)src_ptr;
			src_ptr += jpeg_compoents_per_pixel;
			rgba_ptr += tbpp;
		    }
		    break;
		  case 3:
		    while(rgba_ptr < rgba_end)
		    {
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = 0xff;
		    }
		    break;
		  case 1:
		    while(rgba_ptr < rgba_end)
		    {
			grey = (u_int8_t)*src_ptr++;
			*rgba_ptr++ = grey;
			*rgba_ptr++ = grey;
			*rgba_ptr++ = grey;
			*rgba_ptr++ = 0xff;
		    }
		    break;
		}
 	    }
	}

	/* Report the final progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		progress_ctx->height,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->bpl;
	if(rgba_rtn != NULL)
	{
	    free(*rgba_rtn);
	    *rgba_rtn = ctx->data;		/* RGBA */
	    ctx->data = NULL;
	}

	/* End the decompression
	 *
	 * This will call ImgJPEGReadCloseCB() to delete the
	 * loaded image data and close the file on our read context
	 */
	jpeg_finish_decompress(dinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	if(*progress_ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the JPEG image data from a buffer to RGBA image data.
 */
int ImgBufferReadJPEGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadJPEGRGBA(
	    IMG_JPEG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Reads the JPEG image data from a stream to RGBA image data.
 */
int ImgStreamReadJPEGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadJPEGRGBA(
	    IMG_JPEG_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the JPEG image file to RGBA image data.
 */
int ImgFileOpenJPEGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    return(-4);
	}

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the JPEG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadJPEGRGBA(
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

/*
 *	Reads the JPEG image data from a buffer or a stream to RGBA
 *	image data as a thumb.
 */
static int ImgReadJPEGRGBAThumb(
	const ImgJPEGDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		jpeg_samples_per_row,
			jpeg_compoents_per_pixel;
	JSAMPARRAY jpeg_row_buf;
	J_COLOR_SPACE jpeg_color_space;
	j_decompress_ptr dinfo = NULL;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGReadContext *ctx = NULL;

#define CLEANUP_RETURN(_rtn_val_)	{	\
 const int error_code = errno;			\
						\
 /* Delete the loaded image data, close the	\
  * file, and delete the JPEG decompression	\
  * manager					\
  */						\
 if(dinfo != NULL) {				\
  if(ctx != NULL) {				\
   /* If the user aborted or an error occured	\
    * then jpeg_finish_decompress() would not	\
    * have been called and we need to call	\
    * ImgJPEGReadCloseCB() manually here	\
    */						\
   if(ctx->input_opened) {			\
    ImgJPEGReadCloseCB(dinfo);			\
   }						\
  }						\
						\
  jpeg_destroy_decompress(dinfo);		\
  free(dinfo);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_rtn_val_);				\
}

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	/* Create and set up the JPEG decompression manager */
	dinfo = ImgJPEGCreateDecompressionManager(
	    data_storage_type,
	    &error_ctx,
	    &progress_ctx,
	    &ctx
	);
	if((dinfo == NULL) || (error_ctx == NULL) ||
	   (progress_ctx == NULL) || (ctx == NULL)
	)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
	ctx->bpp = bpp;
	progress_ctx->progress_cb = progress_cb;
	progress_ctx->progress_data = progress_data;
	progress_ctx->user_aborted = user_aborted;

	/* Set the libjpeg error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libjpeg's functions encounters an error, libjpeg will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(error_ctx->jpeg_jmp_buf))
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Check for user abort */
	if(*progress_ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}


	/* Read the header
	 *
	 * This will call ImgJPEGReadOpenCB() to open the file
	 */
	jpeg_read_header(dinfo, TRUE);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Modify the output values */

	/* Check the color space (JPEG image data format) specified by
	 * the JPEG header, the output color space must be RGB or
	 * Greyscale
	 */
	switch(dinfo->jpeg_color_space)
	{
	  case JCS_UNKNOWN:
	  case JCS_RGB:
	  case JCS_YCbCr:			/* YUV */
	  case JCS_CMYK:
	  case JCS_YCCK:
	    dinfo->out_color_space = JCS_RGB;
	    break;
	  case JCS_GRAYSCALE:
	    dinfo->out_color_space = JCS_GRAYSCALE;
	    break;
	}

	/* Add other output options here */


	/* With the output values set we can now start decompressing
	 * the JPEG data
	 */
	jpeg_start_decompress(dinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Calculate the number of JSAMPLEs per row in the output
	 * buffer
	 */
	jpeg_color_space = dinfo->jpeg_color_space;
	jpeg_compoents_per_pixel = dinfo->output_components;
	jpeg_samples_per_row = dinfo->output_width * jpeg_compoents_per_pixel;

	/* Allocate a single row sample array for decompressing the
	 * JPEG image data, this allocation will automatically be deleted
	 * when the JPEG decompression object is deleted
	 */
	jpeg_row_buf = (*dinfo->mem->alloc_sarray)(
	    (j_common_ptr)dinfo, JPOOL_IMAGE, jpeg_samples_per_row, 1
	);

	/* Get the JPEG image data obtained from the header */
	ctx->jpeg_width = (int)dinfo->output_width;
	ctx->jpeg_height = (int)dinfo->output_height;

	/* Calculate the size of the target RGBA image in order to
	 * accomidate for the thumb
	 */
	ImgCalculateThumbSize(
	    ctx->jpeg_width, ctx->jpeg_height,	/* Original size */
	    req_width, req_height,		/* Thumb size limit */
	    &ctx->width, &ctx->height
	);

	ctx->bpl = ctx->width * ctx->bpp;

	/* Allocate the target RGBA image data */
	if(ctx->data != NULL)
	    ctx->data = (u_int8_t *)realloc(
		ctx->data,
		ctx->bpl * ctx->height
	    );
	else
	    ctx->data = (u_int8_t *)calloc(
		ctx->bpl * ctx->height,
		sizeof(u_int8_t)
	    );
	if(ctx->data == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}

	/* Set the progress context values */
	progress_ctx->width = ctx->width;
	progress_ctx->height = ctx->height;
	progress_ctx->bpp = ctx->bpp;
	progress_ctx->bpl = ctx->bpl;
	progress_ctx->data = ctx->data;

	/* Report the initial progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		0,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* Read the JPEG image */
	if(!(*progress_ctx->user_aborted))
	{
	    const int	jpeg_width = ctx->jpeg_width,
			jpeg_height = ctx->jpeg_height,
			thumb_width = ctx->width,
			thumb_height = ctx->height,
			tbpp = ctx->bpp,
			tbpl = ctx->bpl;
	    int		tx, ty,			/* Target RGBA contexts */
			tx_last,
			ty_last,
			sx, sy;			/* Source JPEG contexts */
	    u_int8_t	g,			/* Gamma */
			*rgba = ctx->data,	/* RGBA */
			*rgba_row,
			*rgba_ptr;
	    JSAMPROW	src_row = jpeg_row_buf[0],	/* Source JPEG image pointers */
			src_ptr,
			src_end;

	    /* Begin reading each line from the JPEG decompressor and
	     * only process lines that are visibly scaled to the thumb
	     * size
	     */
	    ty_last = -1;
	    while((dinfo->output_scanline < jpeg_height) &&
		  (error_ctx->status == 0) &&
		  !(*progress_ctx->user_aborted)
	    )
	    {
		sy = dinfo->output_scanline;

		/* Calculate the y position on the target RGBA image data */
		ty = thumb_height * sy / jpeg_height;

		/* Read the next line */
		jpeg_read_scanlines(
		    dinfo,
		    jpeg_row_buf,
		    1				/* Max lines to read */
		);

		/* If the y position on the target is still on the same
		 * target line as the previous line then do not process
		 * this line
		 */
		if(ty <= ty_last)
		    continue;

		rgba_row = rgba + (ty * tbpl);

		/* Copy the line to the target RGBA buffer */
		tx_last = -1;
		switch(jpeg_compoents_per_pixel)
		{
		  case 4:
		    for(sx = 0,
			src_ptr = src_row,
			src_end = src_ptr + (jpeg_width * jpeg_compoents_per_pixel);
			src_ptr < src_end;
			sx++,
			src_ptr += jpeg_compoents_per_pixel
		    )
		    {
			tx = thumb_width * sx / jpeg_width;
			if(tx <= tx_last)
			    continue;

			rgba_ptr = rgba_row + (tx * tbpp);
			*(u_int32_t *)rgba_ptr = *(const u_int32_t *)src_ptr;

			tx_last = tx;
		    }
		    break;
		  case 3:
		    for(sx = 0,
			src_ptr = src_row,
			src_end = src_ptr + (jpeg_width * jpeg_compoents_per_pixel);
			src_ptr < src_end;
			sx++,
			src_ptr += jpeg_compoents_per_pixel
		    )
		    {
			tx = thumb_width * sx / jpeg_width;
			if(tx <= tx_last)
			    continue;

			rgba_ptr = rgba_row + (tx * tbpp);
			*rgba_ptr++ = src_ptr[0];
			*rgba_ptr++ = src_ptr[1];
			*rgba_ptr++ = src_ptr[2];
			*rgba_ptr = 0xff;

			tx_last = tx;
		    }
		    break;
		  case 1:
		    for(sx = 0,
			src_ptr = src_row,
			src_end = src_ptr + (jpeg_width * jpeg_compoents_per_pixel);
			src_ptr < src_end;
			sx++,
			src_ptr += jpeg_compoents_per_pixel
		    )
		    {
			tx = thumb_width * sx / jpeg_width;
			if(tx <= tx_last)
			    continue;

			g = *src_ptr;
			rgba_ptr = rgba_row + (tx * tbpp);
			*rgba_ptr++ = g;
			*rgba_ptr++ = g;
			*rgba_ptr++ = g;
			*rgba_ptr = 0xff;

			tx_last = tx;
		    }
		    break;
		}

		ty_last = ty;
 	    }
	}

	/* Report the final progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		progress_ctx->height,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->bpl;
	if(rgba_rtn != NULL)
	{
	    free(*rgba_rtn);
	    *rgba_rtn = ctx->data;		/* RGBA */
	    ctx->data = NULL;
	}
	if(orig_width_rtn != NULL)
	    *orig_width_rtn = ctx->jpeg_width;
	if(orig_height_rtn != NULL)
	    *orig_height_rtn = ctx->jpeg_height;

	/* End the decompression
	 *
	 * This will call ImgJPEGReadCloseCB() to delete the
	 * loaded image data and close the file on our read context
	 */
	jpeg_finish_decompress(dinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	if(*progress_ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the JPEG image data from a buffer to RGBA image data
 *	as a thumb.
 */
int ImgBufferReadJPEGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadJPEGRGBAThumb(
	    IMG_JPEG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    orig_width_rtn, orig_height_rtn,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Reads the JPEG image data from a stream to RGBA image data
 *	as a thumb.
 */
int ImgStreamReadJPEGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadJPEGRGBAThumb(
	    IMG_JPEG_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    orig_width_rtn, orig_height_rtn,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the JPEG image file to RGBA image data.
 */
int ImgFileOpenJPEGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last open error message pointer */
	imgio_last_open_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    return(-4);
	}

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the JPEG image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadJPEGRGBAThumb(
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    orig_width_rtn, orig_height_rtn,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	JPEG write open callback.
 */
static void ImgJPEGWriteOpenCB(j_compress_ptr cinfo)
{
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(cinfo->err);
	ImgJPEGWriteContext *ctx = IMG_JPEG_WRITE_CONTEXT(cinfo->dest);
	struct jpeg_destination_mgr *dest = &ctx->jpeg_dest;
	switch(ctx->data_storage_type)
	{
	  case IMG_JPEG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL))
	    {
		/* Allocate the IO buffer */
		if(ctx->io_buf == NULL)
		{
		    ctx->io_buf_len = IMG_JPEG_DEF_IO_SIZE;
		    ctx->io_buf = (JOCTET *)(*cinfo->mem->alloc_small)(
			(j_common_ptr)cinfo,
			JPOOL_PERMANENT,	/* Stay allocated until the
						 * master record is destroyed */
			ctx->io_buf_len * sizeof(JOCTET)
		    );
		    if(ctx->io_buf == NULL)
		    {
			ctx->io_buf_len = 0l;
			imgio_last_save_error = "Memory allocation error";
			error_ctx->status = -3;
			break;
		    }
		}
	    }
	    else
	    {
		imgio_last_save_error = "Buffer return pointer not specified";
		errno = EINVAL;
		error_ctx->status = -2;
		return;
	    }
	    break;

	  case IMG_JPEG_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		FILE *fp = ctx->fp;

		/* Allocate the IO buffer */
		if(ctx->io_buf == NULL)
		{
		    struct stat stat_buf;
		    if(fstat(fileno(fp), &stat_buf))
			ctx->io_buf_len = IMG_JPEG_DEF_IO_SIZE;
		    else
			ctx->io_buf_len = MAX(
			    stat_buf.st_blksize,
			    IMG_JPEG_MIN_IO_SIZE
			);
		    ctx->io_buf = (JOCTET *)(*cinfo->mem->alloc_small)(
			(j_common_ptr)cinfo,
			JPOOL_PERMANENT,	/* Stay allocated until the
						 * master record is destroyed */
			ctx->io_buf_len * sizeof(JOCTET)
		    );
		    if(ctx->io_buf == NULL)
		    {
			ctx->io_buf_len = 0l;
			imgio_last_save_error = "Memory allocation error";
			error_ctx->status = -3;
			break;
		    }
		}
	    }
	    else
	    {
		imgio_last_save_error = "Stream not specified";
		errno = EINVAL;
		error_ctx->status = -2;
		return;
	    }
	    break;
	}

	/* Mark that the entire write buffer is available */
	dest->next_output_byte = ctx->io_buf;
	dest->free_in_buffer = ctx->io_buf_len;

	/* Mark that the output has been opened */
	ctx->output_opened = TRUE;
}

/*
 *	JPEG write buffer callback.
 */
static boolean ImgJPEGWriteBufferCB(j_compress_ptr cinfo)
{
	unsigned long bp_pos;
	size_t units_written;
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(cinfo->err);
	ImgJPEGWriteContext *ctx = IMG_JPEG_WRITE_CONTEXT(cinfo->dest);
	struct jpeg_destination_mgr *dest = &ctx->jpeg_dest;
	JOCTET *io_buf = ctx->io_buf;
	const unsigned long io_buf_len = ctx->io_buf_len;

	/* Buffer return pointers not specified? */
	if((ctx->bp == NULL) || (ctx->bp_len == NULL) ||
	   (io_buf == NULL) || (io_buf_len == 0l)
	)
	{
	    dest->next_output_byte = NULL;
	    dest->free_in_buffer = 0;
	    if(error_ctx->status == 0)
		error_ctx->status = -1;
	    return(FALSE);
	}

	/* Get the current position in the output buffer to append the
	 * data to
	 */
	bp_pos = *ctx->bp_len;

	/* Append the data to the output buffer */
	if(io_buf_len > 0l)
	{
	    /* Increase the allocation of the output buffer */
	    const size_t len = io_buf_len;
	    *ctx->bp_len = bp_pos + (unsigned long)len;
	    *ctx->bp = (void *)realloc(
		*ctx->bp,
		(*ctx->bp_len) * sizeof(u_int8_t)
	    );
	    if(*ctx->bp == NULL)
	    {
		*ctx->bp_len = 0l;
		imgio_last_save_error = "Memory allocation error";
		error_ctx->status = -3;
		return(FALSE);
	    }

	    /* Copy/append all the data from the IO buffer to the
	     * newly allocated output buffer segment
	     */
	    (void)memcpy(
		(*ctx->bp) + bp_pos,
		io_buf,
		len * sizeof(u_int8_t)
	    );

	    units_written = len / sizeof(u_int8_t);
	}
	else
	    units_written = 0l;

	/* Mark that the entire IO buffer is now available to be filled
	 * again
	 */
	dest->next_output_byte = io_buf;
	dest->free_in_buffer = io_buf_len;

	if(units_written > 0l)
	    return(TRUE);
	else
	    return(FALSE);
}

/*
 *	JPEG write stream or file callback.
 */
static boolean ImgJPEGWriteStreamCB(j_compress_ptr cinfo)
{
	size_t units_written;
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(cinfo->err);
	ImgJPEGWriteContext *ctx = IMG_JPEG_WRITE_CONTEXT(cinfo->dest);
	struct jpeg_destination_mgr *dest = &ctx->jpeg_dest;
	FILE *fp = ctx->fp;
	JOCTET *io_buf = ctx->io_buf;
	const unsigned long io_buf_len = ctx->io_buf_len;

	/* Stream not opened and/or error or IO buffer not allocated? */
	if((fp == NULL) ||
	   (io_buf == NULL) || (io_buf_len == 0l)
	)
	{
	    dest->next_output_byte = NULL;
	    dest->free_in_buffer = 0;
	    if(error_ctx->status == 0)
		error_ctx->status = -1;
	    return(FALSE);
	}

	/* Write the next block
	 *
	 * Always assume libjpeg filled the IO buffer completely and
	 * write everything in it
	 */
	if(io_buf_len > 0l)
	{
	    const size_t units_to_write = (size_t)(io_buf_len / sizeof(JOCTET));
	    units_written = fwrite(
		io_buf,
		sizeof(JOCTET),
		units_to_write,
		fp
	    );
	    if(units_written != units_to_write)
	    {
		ImgSaveSetError(strerror(errno));
		error_ctx->status = -1;
	    }
	}
	else
	    units_written = 0l;

	/* Mark that the entire IO buffer is now available to be filled
	 * again
	 */
	dest->next_output_byte = io_buf;
	dest->free_in_buffer = io_buf_len;

	if(units_written > 0)
	    return(TRUE);
	else
	    return(FALSE);
}

/*
 *	JPEG write close buffer, stream, or file callback.
 */
static void ImgJPEGWriteCloseCB(j_compress_ptr cinfo)
{
	ImgJPEGErrorContext *error_ctx = IMG_JPEG_ERROR_CONTEXT(cinfo->err);
	ImgJPEGWriteContext *ctx = IMG_JPEG_WRITE_CONTEXT(cinfo->dest);
	struct jpeg_destination_mgr *dest = &ctx->jpeg_dest;

	ctx->data = NULL;			/* Read-only, do not delete */

	/* Close the output */
	switch(ctx->data_storage_type)
	{
	  case IMG_JPEG_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL))
	    {
		/* Write any remaining data to the buffer */
		const size_t len = ctx->io_buf_len - dest->free_in_buffer;
		if(len > 0l)
		{
		    /* Get the current position in the output buffer
		     * to append the data to
		     */
		    const unsigned long bp_pos = *ctx->bp_len;

		    /* Increase the allocation of the output buffer */
		    *ctx->bp_len = bp_pos + (unsigned long)len;
		    *ctx->bp = (void *)realloc(
			*ctx->bp,
			(*ctx->bp_len) * sizeof(u_int8_t)
		    );
		    if(*ctx->bp != NULL)
		    {
			/* Copy/append all the data from the IO buffer
			 * to the newly allocated output buffer segment
			 */
			(void)memcpy(
			    (*ctx->bp) + bp_pos,
			    ctx->io_buf,
			    len * sizeof(u_int8_t)
			);
		    }
		    else
		    {
			*ctx->bp_len = 0l;
			imgio_last_save_error = "Memory allocation error";
			error_ctx->status = -3;
		    }
		}
	    }
	    break;

	  case IMG_JPEG_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		/* Write any remaining data to the stream */
		const size_t len = ctx->io_buf_len - dest->free_in_buffer;
		if(len > 0l)
		{
		    const size_t units_to_write = (size_t)(len / sizeof(JOCTET));
		    if(fwrite(
			ctx->io_buf,
			sizeof(JOCTET),
			units_to_write,
			ctx->fp
		    ) != units_to_write)
		    {
			ImgSaveSetError(strerror(errno));
			error_ctx->status = -1;
		    }
		}
		/* Mark the stream as closed and let the calling
		 * function decide what to do
		 */
		ctx->fp = NULL;
	    }
	    break;
	}

	/* Mark that the outputs have been closed */
	ctx->output_opened = FALSE;
}

/*
 *	Creates and sets up a new JPEG decompression manager for
 *	reading from a file.
 */
static j_compress_ptr ImgJPEGCreateCompressionManager(
	const ImgJPEGDataStorageType data_storage_type,
	ImgJPEGErrorContext **error_ctx_rtn,
	ImgJPEGProgressContext **progress_ctx_rtn,
	ImgJPEGWriteContext **write_ctx_rtn
)
{
	j_compress_ptr cinfo;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGWriteContext *ctx = NULL;

	/* Reset the return values */
	*error_ctx_rtn = error_ctx;
	*progress_ctx_rtn = progress_ctx;
	*write_ctx_rtn = ctx;

	/* Allocate the JPEG decompression manager */
	cinfo = (j_compress_ptr)calloc(1, sizeof(struct jpeg_compress_struct));
	if(cinfo == NULL)
	    return(NULL);

	/* Initialize the JPEG compression manager */
	jpeg_create_compress(cinfo);

	/* Allocate our error context who's size will be large enough
	 * to accomidate the jpeg_error_mgr and our error context
	 * members, this allows this error context to be used as both
	 * the jpeg_error_mgr and our error context
	 */
	if(cinfo->err == NULL)
	{
	    struct jpeg_error_mgr *err;
	    error_ctx = (ImgJPEGErrorContext *)(*cinfo->mem->alloc_small)(
		(j_common_ptr)cinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGErrorContext)
	    );
	    if(error_ctx == NULL)
	    {
		jpeg_destroy_compress(cinfo);
		free(cinfo);
		return(NULL);
	    }

	    cinfo->err = jpeg_std_error(&error_ctx->jpeg_err);

	    err = &error_ctx->jpeg_err;
	    err->error_exit = ImgJPEGErrorExitCB;
	    err->emit_message = ImgJPEGErrorEmitMessageCB;
	    err->output_message = ImgJPEGErrorOutputMessageCB;

	    (void)memset(
		error_ctx->jpeg_jmp_buf,
		0x00,
		sizeof(jmp_buf)
	    );
	    error_ctx->status = 0;
	}

	/* Allocate our progress context who's size will be large
	 * enough to accomidate the jpeg_progress_mgr and our
	 * progress context members, this allows this progress
	 * context to be used as both the jpeg_progress_mgr and our
	 * progress context
	 */
	if(cinfo->progress == NULL)
	{
	    struct jpeg_progress_mgr *progress;
	    cinfo->progress = (struct jpeg_progress_mgr *)(*cinfo->mem->alloc_small)(
		(j_common_ptr)cinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGProgressContext)
	    );
	    if(cinfo->progress == NULL)
	    {
		jpeg_destroy_compress(cinfo);
		free(cinfo);
		return(NULL);
	    }

	    progress_ctx = (ImgJPEGProgressContext *)cinfo->progress;

	    progress = &progress_ctx->jpeg_progress;
	    progress->progress_monitor = ImgJPEGProgressMonitorCB;
	    progress->pass_counter = 0l;
	    progress->pass_limit = 0l;
	    progress->completed_passes = 0;
	    progress->total_passes = 0;

	    progress_ctx->width = 0;
	    progress_ctx->height = 0;
	    progress_ctx->bpp = 0;
	    progress_ctx->bpl = 0;
	    progress_ctx->data = NULL;
	    progress_ctx->progress_cb = NULL;
	    progress_ctx->progress_data = NULL;
	    progress_ctx->user_aborted = NULL;
	}

	/* Allocate our read context who's size will be large enough
	 * to accomidate the jpeg_source_mgr and our read context
	 * members, this allows the read context to be used as both
	 * the jpeg_source_mgr and our read context
	 */
	if(cinfo->dest == NULL)
	{
	    struct jpeg_destination_mgr *dest;
	    cinfo->dest = (struct jpeg_destination_mgr *)(*cinfo->mem->alloc_small)(
		(j_common_ptr)cinfo,
		JPOOL_PERMANENT,		/* Stay allocated until the
						 * master record is destroyed */
		sizeof(ImgJPEGWriteContext)
	    );
	    if(cinfo->dest == NULL)
	    {
		jpeg_destroy_compress(cinfo);
		free(cinfo);
		return(NULL);
	    }

	    ctx = (ImgJPEGWriteContext *)cinfo->dest;

	    dest = &ctx->jpeg_dest;
	    dest->next_output_byte = NULL;
	    dest->free_in_buffer = 0;
	    switch(data_storage_type)
	    {
	      case IMG_JPEG_DATA_STORAGE_BUFFER:
		dest->init_destination = ImgJPEGWriteOpenCB;
		dest->empty_output_buffer = ImgJPEGWriteBufferCB;
		dest->term_destination = ImgJPEGWriteCloseCB;
		break;
	      case IMG_JPEG_DATA_STORAGE_STREAM:
		dest->init_destination = ImgJPEGWriteOpenCB;
		dest->empty_output_buffer = ImgJPEGWriteStreamCB;
		dest->term_destination = ImgJPEGWriteCloseCB;
		break;
	    }

	    ctx->data_storage_type = data_storage_type;
	    ctx->output_opened = FALSE;
	    ctx->bp = NULL;
	    ctx->bp_len = NULL;
	    ctx->fp = NULL;
	    ctx->io_buf = NULL;
	    ctx->io_buf_len = 0;
	    ctx->width = 0;
	    ctx->height = 0;
	    ctx->bpp = 0;
	    ctx->bpl = 0;
	    ctx->data = NULL;
	    ctx->quality = 0;
	}

	*error_ctx_rtn = error_ctx;
	*progress_ctx_rtn = progress_ctx;
	*write_ctx_rtn = ctx;

	return(cinfo);
}

/*
 *	Writes the greyscale image data to a buffer or stream in JPEG
 *	format.
 */
static int ImgWriteJPEGGreyscale(
	const ImgJPEGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,				/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 1,		/* Greyscale */
			_bpl = (bpl > 0) ? bpl : width * bpp;
	int user_aborted = 0;
	j_compress_ptr cinfo = NULL;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGWriteContext *ctx = NULL;

#define CLEANUP_RETURN(_rtn_val_)	{	\
 const int error_code = errno;			\
						\
 /* Close the stream and delete the JPEG	\
  * compression manager				\
  */						\
 if(cinfo != NULL) {				\
  if(ctx != NULL) {				\
   /* If the user aborted or an error occured	\
    * then jpeg_finish_compress() would not	\
    * have been called and we need to call	\
    * ImgJPEGFileCloseWriteCB() manually here	\
    */						\
   if(ctx->output_opened) {			\
    struct jpeg_destination_mgr *dest =		\
     &ctx->jpeg_dest;				\
    /* Mark that we have no data to write so	\
     * that ImgJPEGWriteCloseCB() does		\
     * not write any remaining data		\
     */						\
    dest->free_in_buffer = ctx->io_buf_len;	\
    ImgJPEGWriteCloseCB(cinfo);			\
   }						\
  }						\
						\
  jpeg_destroy_compress(cinfo);			\
  free(cinfo);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_rtn_val_);				\
}

	/* Reset the last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (grey == NULL)
	)
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Create and set up the JPEG compression manager */
	cinfo = ImgJPEGCreateCompressionManager(
	    data_storage_type,
	    &error_ctx,
	    &progress_ctx,
	    &ctx
	);
	if((cinfo == NULL) || (error_ctx == NULL) ||
	   (progress_ctx == NULL) || (ctx == NULL)
	)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
	ctx->width = width;
	ctx->height = height;
	ctx->bpp = bpp;
	ctx->bpl = _bpl;
	ctx->data = grey;			/* Greyscale */
	ctx->quality = CLIP(quality, 1, 100);

	/* Set the progress context values */
	progress_ctx->width = ctx->width;
	progress_ctx->height = ctx->height;
	progress_ctx->bpp = ctx->bpp;
	progress_ctx->bpl = ctx->bpl;
	progress_ctx->data = ctx->data;
	progress_ctx->progress_cb = progress_cb;
	progress_ctx->progress_data = progress_data;
	progress_ctx->user_aborted = &user_aborted;

	/* Report the initial progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		0,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}
	if(*progress_ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Set the libjpeg error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libjpeg's functions encounters an error, libjpeg will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(error_ctx->jpeg_jmp_buf))
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Set the image values on to the JPEG compression manager */
	cinfo->image_width = width;
	cinfo->image_height = height;
	cinfo->input_components = 1;		/* Greyscale */
	cinfo->in_color_space = JCS_GRAYSCALE;

	/* Set compression parameters to default values */
	jpeg_set_defaults(cinfo);

	/* Begin setting specific compression parameters */

	/* Quality */
	jpeg_set_quality(
	    cinfo,
	    ctx->quality,			/* 1 to 100 */
	    TRUE				/* limit to baseline JPEG values */
	);


	/* Start the JPEG compression
	 *
	 * This will call ImgJPEGWriteOpenCB() to open the buffer,
	 * stream, or file
	 */
	jpeg_start_compress(cinfo, TRUE);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Begin saving the RGBA image data to the JPEG image file */
	if(!(*progress_ctx->user_aborted))
	{
	    /* Allocate JSAMPLEs row buffer for use in writing the JPEG
	     * image
	     *
	     * Each line from the source RGBA image data will be
	     * copy/converted to this JSAMPLEs buffer and written to the
	     * JPEG image using jpeg_write_scanlines()
	     */
	    int samples_per_pixel = cinfo->input_components;
	    JSAMPROW row_ptr = (JSAMPROW)malloc(ctx->width * samples_per_pixel);
	    JSAMPARRAY buf = (JSAMPARRAY)malloc(1 * sizeof(JSAMPROW));
	    if((buf != NULL) && (row_ptr != NULL))
	    {
		const int	src_width = ctx->width,
				src_height = ctx->height,
				src_bpp = ctx->bpp,
				src_bpl = ctx->bpl;
		const u_int8_t	*src = ctx->data,/* Greyscale */
				*src_line,
				*src_line_end,
				*src_end = src + (src_height * src_bpl),
				*src_ptr;
		JSAMPROW	tar_ptr;

		buf[0] = row_ptr;

		/* Write each line */
		src_line = src;
		while((src_line < src_end) &&
		      (error_ctx->status == 0) &&
		      !(*progress_ctx->user_aborted)
		)
		{
		    /* Copy/convert this line */
		    src_ptr = src_line;
		    src_line_end = src_ptr + (src_width * src_bpp);
		    tar_ptr = row_ptr;
		    while(src_ptr < src_line_end)
		    {
			*tar_ptr++ = (JSAMPLE)(*src_ptr);
			src_ptr += src_bpp;
		    }

		    /* Write this line */
		    jpeg_write_scanlines(cinfo, buf, 1);

		    /* Seek to the next line */
		    src_line += src_bpl;
		}
	    }

	    free(row_ptr);
	    free(buf);
	}

	/* Report the final progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		progress_ctx->height,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* End the JPEG compression
	 *
	 * This will call ImgJPEGWriteCloseCB() to write out
	 * any remaining data and close the stream
	 */
	jpeg_finish_compress(cinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	if(*progress_ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the greyscale image data to a buffer in JPEG format.
 */
int ImgBufferWriteJPEGGreyscale(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJPEGGreyscale(
	    IMG_JPEG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width, height,
	    bpl,
	    grey,
	    quality,
	    progress_cb, progress_data
	));
}

/*
 *	Writes the greyscale image data to a stream in JPEG format.
 */
int ImgStreamWriteJPEGGreyscale(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJPEGGreyscale(
	    IMG_JPEG_DATA_STORAGE_STREAM,
	    NULL, NULL,
	    fp,
	    width, height,
	    bpl,
	    grey,
	    quality,
	    progress_cb, progress_data
	));
}

/*
 *	Saves the greyscale image data to a JPEG image file.
 */
int ImgFileSaveJPEGGreyscale(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the JPEG file for writing */ 
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    ImgSaveSetError(strerror(errno));
	    return(-1);
	}

	/* Write the RGBA image data to a JPEG image file */
	status = ImgStreamWriteJPEGGreyscale(
	    fp,
	    width, height,
	    bpl,
	    grey,
	    quality,
	    progress_cb, progress_data
	);
	error_code = errno;

	/* Close the JPEG file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

/*
 *	Writes the RGBA image data to a buffer or stream in JPEG
 *	format.
 */
static int ImgWriteJPEGRGBA(
	const ImgJPEGDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : width * bpp;
	int user_aborted = 0;
	j_compress_ptr cinfo = NULL;
	ImgJPEGErrorContext *error_ctx = NULL;
	ImgJPEGProgressContext *progress_ctx = NULL;
	ImgJPEGWriteContext *ctx = NULL;

#define CLEANUP_RETURN(_rtn_val_)	{	\
 const int error_code = errno;			\
						\
 /* Close the stream and delete the JPEG	\
  * compression manager				\
  */						\
 if(cinfo != NULL) {				\
  if(ctx != NULL) {				\
   /* If the user aborted or an error occured	\
    * then jpeg_finish_compress() would not	\
    * have been called and we need to call	\
    * ImgJPEGFileCloseWriteCB() manually here	\
    */						\
   if(ctx->output_opened) {			\
    struct jpeg_destination_mgr *dest =		\
     &ctx->jpeg_dest;				\
    /* Mark that we have no data to write so	\
     * that ImgJPEGWriteCloseCB() does		\
     * not write any remaining data		\
     */						\
    dest->free_in_buffer = ctx->io_buf_len;	\
    ImgJPEGWriteCloseCB(cinfo);			\
   }						\
  }						\
						\
  jpeg_destroy_compress(cinfo);			\
  free(cinfo);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_rtn_val_);				\
}

	/* Reset the last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (rgba == NULL)
	)
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Create and set up the JPEG compression manager */
	cinfo = ImgJPEGCreateCompressionManager(
	    data_storage_type,
	    &error_ctx,
	    &progress_ctx,
	    &ctx
	);
	if((cinfo == NULL) || (error_ctx == NULL) ||
	   (progress_ctx == NULL) || (ctx == NULL)
	)
	{
	    if(imgio_last_save_error == NULL)
		imgio_last_save_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
	ctx->width = width;
	ctx->height = height;
	ctx->bpp = bpp;
	ctx->bpl = _bpl;
	ctx->data = rgba;			/* RGBA */
	ctx->quality = CLIP(quality, 1, 100);

	/* Set the progress context values */
	progress_ctx->width = ctx->width;
	progress_ctx->height = ctx->height;
	progress_ctx->bpp = ctx->bpp;
	progress_ctx->bpl = ctx->bpl;
	progress_ctx->data = ctx->data;
	progress_ctx->progress_cb = progress_cb;
	progress_ctx->progress_data = progress_data;
	progress_ctx->user_aborted = &user_aborted;

	/* Report the initial progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		0,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}
	if(*progress_ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Set the libjpeg error return point
	 *
	 * When using custom error callbacks, whenever a call to one
	 * of libjpeg's functions encounters an error, libjpeg will
	 * longjmp() back to the setjmp() point which is expected to
	 * delete any allocated memory and return an error
	 */
	if(setjmp(error_ctx->jpeg_jmp_buf))
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Set the specified image values on to the JPEG compression
	 * manager
	 */
	cinfo->image_width = width;
	cinfo->image_height = height;
	switch(color_type)
	{
	  case 1:				/* Color */
	    cinfo->input_components = 3;
	    cinfo->in_color_space = JCS_RGB;
	    break;
	  default:				/* Greyscale */
	    cinfo->input_components = 1;
	    cinfo->in_color_space = JCS_GRAYSCALE;
	    break;
	}

	/* Set compression parameters to default values */
	jpeg_set_defaults(cinfo);

	/* Begin setting specific compression parameters */

	/* Quality */
	jpeg_set_quality(
	    cinfo,
	    ctx->quality,			/* 1 to 100 */
	    TRUE				/* limit to baseline JPEG values */
	);


	/* Start the JPEG compression
	 *
	 * This will call ImgJPEGWriteOpenCB() to open the buffer,
	 * stream, or file
	 */
	jpeg_start_compress(cinfo, TRUE);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	/* Begin saving the RGBA image data to the JPEG image file */
	if(!(*progress_ctx->user_aborted))
	{
	    /* Allocate JSAMPLEs row buffer for use in writing the JPEG
	     * image
	     *
	     * Each line from the source RGBA image data will be
	     * copy/converted to this JSAMPLEs buffer and written to the
	     * JPEG image using jpeg_write_scanlines()
	     */
	    int samples_per_pixel = cinfo->input_components;
	    JSAMPROW row_ptr = (JSAMPROW)malloc(ctx->width * samples_per_pixel);
	    JSAMPARRAY buf = (JSAMPARRAY)malloc(1 * sizeof(JSAMPROW));
	    if((buf != NULL) && (row_ptr != NULL))
	    {
		const u_int8_t	def_bg_color[] = { 0x00, 0x00, 0x00, 0xFF };
		const int	src_width = ctx->width,
				src_height = ctx->height,
				src_bpp = ctx->bpp,
				src_bpl = ctx->bpl,
				tar_bpp = samples_per_pixel;
		const u_int8_t	*src = ctx->data,	/* RGBA */
				*src_end = src + (src_height * src_bpl),
				*src_line,
				*src_line_end,
				*src_ptr,
				*bg_color_ptr = (bg_color != NULL) ? bg_color : def_bg_color;
		u_int8_t bg_grey;
		JSAMPROW	tar_ptr;

		buf[0] = row_ptr;

		/* Write by the specified color type */
		switch(color_type)
		{
		  case 1:			/* Color */
		    /* Write each line */
		    src_line = src;
		    while((src_line < src_end) &&
			  (error_ctx->status == 0) &&
			  !(*progress_ctx->user_aborted)
		    )
		    {
			/* Copy/convert this line from RGBA to RGB */
			src_ptr = src_line;
			src_line_end = src_ptr + (src_width * src_bpp);
			tar_ptr = row_ptr;
			while(src_ptr < src_line_end)
			{
			    ImgConvertPixelRGBAToRGB(
				src_ptr,	/* RGBA */
				bg_color_ptr,
				(u_int8_t *)tar_ptr	/* RGB */
			    );
			    tar_ptr += tar_bpp;
			    src_ptr += src_bpp;
			}

			/* Write this line */
			jpeg_write_scanlines(cinfo, buf, 1);

			/* Seek to the next line */
			src_line += src_bpl;
		    }
		    break;

		  default:			/* Greyscale */
		    /* Precalculate the background color in greyscale */
		    bg_grey = ImgConvertPixelRGBToGreyscale(bg_color_ptr);
		    /* Write each line */
		    src_line = src;
		    while((src_line < src_end) &&
			  (error_ctx->status == 0) &&
			  !(*progress_ctx->user_aborted)
		    )
		    {
			/* Copy/convert this line from RGBA to greyscale */
			src_ptr = src_line;
			src_line_end = src_ptr + (src_width * src_bpp);
			tar_ptr = row_ptr;
			while(src_ptr < src_line_end)
			{
			    tar_ptr[0] = (JSAMPLE)ImgConvertPixelRGBAToGreyscale(
				src_ptr,	/* RGBA */
				bg_grey		/* Greyscale */
			    );
			    tar_ptr += tar_bpp;
			    src_ptr += src_bpp;
			}

			/* Write this line */
			jpeg_write_scanlines(cinfo, buf, 1);

			/* Seek to the next line */
			src_line += src_bpl;
		    }
		    break;
		}
	    }

	    free(row_ptr);
	    free(buf);
	}

	/* Report the final progress */
	if((progress_ctx->progress_cb != NULL) &&
	   !(*progress_ctx->user_aborted)
	)
	{
	    if(!progress_ctx->progress_cb(
		progress_ctx->progress_data,
		progress_ctx->height,
		progress_ctx->height,
		progress_ctx->width, progress_ctx->height,
		progress_ctx->bpl, progress_ctx->bpp,
		progress_ctx->data
	    ))
		*progress_ctx->user_aborted = 1;
	}

	/* End the JPEG compression
	 *
	 * This will call ImgJPEGWriteCloseCB() to write out
	 * any remaining data and close the stream
	 */
	jpeg_finish_compress(cinfo);
	if(error_ctx->status != 0)
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}

	if(*progress_ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = error_ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the RGBA image data to a buffer in JPEG format.
 */
int ImgBufferWriteJPEGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJPEGRGBA(
	    IMG_JPEG_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    quality,
	    color_type,
	    progress_cb, progress_data
	));
}

/*
 *	Writes the RGBA image data to a stream in JPEG format.
 */
int ImgStreamWriteJPEGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteJPEGRGBA(
	    IMG_JPEG_DATA_STORAGE_STREAM,
	    NULL, NULL,
	    fp,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    quality,
	    color_type,
	    progress_cb, progress_data
	));
}

/*
 *	Saves the RGBA image data to a JPEG image file.
 */
int ImgFileSaveJPEGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */         
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the JPEG file for writing */ 
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    ImgSaveSetError(strerror(errno));
	    return(-1);
	}

	/* Write the RGBA image data to a JPEG image file */
	status = ImgStreamWriteJPEGRGBA(
	    fp,
	    width, height,
	    bpl,
	    rgba,
	    bg_color,
	    quality,
	    color_type,
	    progress_cb, progress_data
	);
	error_code = errno;

	/* Close the JPEG file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

#endif	/* HAVE_LIBJPEG */
