#ifdef HAVE_LIBGIF
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gif_lib.h>				/* libungif | libgif */

#include "rgba_to_cidx.h"
#include "imgio.h"


/* Last error message pointers */
extern const char	*imgio_last_open_error,
			*imgio_last_save_error;


typedef struct _ImgGIFReadContext	ImgGIFReadContext;
#define IMG_GIF_READ_CONTEXT(p)		((ImgGIFReadContext *)(p))
typedef struct _ImgGIFWriteContext	ImgGIFWriteContext;
#define IMG_GIF_WRITE_CONTEXT(p)	((ImgGIFWriteContext *)(p))

typedef struct _RGBAToCIdxData		RGBAToCIdxData;
#define RGBA_TO_CIDX_DATA(p)		((RGBAToCIdxData *)(p))


typedef enum {
	IMG_GIF_DATA_STORAGE_BUFFER,
	IMG_GIF_DATA_STORAGE_STREAM
} ImgGIFDataStorageType;


/* Version */
void ImgGIFVersion(int *major, int *minor, int *release); 

/* Check Type */
int ImgBufferIsGIF(const u_int8_t *data, const int len);
int ImgStreamIsGIF(FILE *fp);
int ImgFileIsGIF(const char *path);

/* Error Code To String */
static const char *ImgGIFErrorStringDecode(const int gif_error_code);
#define DGIF_STR_ERROR		ImgGIFErrorStringDecode
static const char *ImgGIFErrorStringEncode(const int gif_error_code);
#define EGIF_STR_ERROR		ImgGIFErrorStringEncode

/* Open */
static int ImgReadGIFReadCB(
	GifFileType *ft,
	GifByteType *buf, int buf_len
);
static void ImgReadGIFRenderGIFScreenToRGBA(
	u_int8_t *rgba,
	const int rgba_width, const int rgba_height,
	const int rgba_bpl,
	const GifRowType *gif_screen,
	const int gif_width, const int gif_height,
	const int gif_bpl,
	const ColorMapObject *colormap,
	const u_int8_t default_alpha_value,
	const int transparent_color_num,
	const u_int8_t prev_disposal_method,
	const u_int8_t *prev_rgba,
	const u_int8_t *cwp_rgba		/* Clear with previous RGBA */
);
static int ImgReadGIFChunksRGBA(ImgGIFReadContext *ctx);
static int ImgReadGIFRGBA(
	const ImgGIFDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadGIFRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadGIFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenGIFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgReadGIFRGBAThumb(
	const ImgGIFDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgBufferReadGIFRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgStreamReadGIFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenGIFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);

/* Save */
static int ImgWriteRGBAToCIdxCB(int i, int m, void *data);
static int ImgWriteGIFDitherRGBA(
	const int frame_num,
	const int nframes,
	const u_int8_t *rgba,
	const int width, const int height,
	const int bpl,
	const u_int8_t *bg_color,
	int *bg_color_num,
	const u_int8_t alpha_threshold,
	int *trans_color_num,
	ColorMapObject **colormap_rtn,
	GifByteType **color_index_rtn,
	int *color_index_bpl_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
static int ImgWriteGIFLoopingBlock(GifFileType *ft);
static int ImgWriteGIFWriteCB(
	GifFileType *ft,
	const GifByteType *buf, const int buf_len
);
static int ImgWriteGIFRGBAToColor(ImgGIFWriteContext *ctx);
static int ImgWriteGIFRGBAToGreyscale(ImgGIFWriteContext *ctx);
static int ImgWriteGIFRGBAToBW(ImgGIFWriteContext *ctx);
static int ImgWriteGIFRGBA(
	const ImgGIFDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgBufferWriteGIFRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgStreamWriteGIFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileSaveGIFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_GIF_PROGRESS_RESOLUTION	10


/*
 *	Suggested IO Buffer Length (in bytes):
 */
#define IMG_GIF_DEF_IO_SIZE		1024

/*
 *	Minimum IO Buffer Length (in bytes):
 */
#define IMG_GIF_MIN_IO_SIZE		4


/*
 *	Graphic Control Flags:
 */
#define IMG_GIF_GRAPHIC_CONTROL_TRANSPARENCY	(1 << 0)
#define IMG_GIF_GRAPHIC_CONTROL_USER_INPUT	(1 << 1)
#define IMG_GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK	((1 << 2) | (1 << 3) | (1 << 4))


/*
 *	Disposal Method Flags:
 *
 *	Obtained from the graphic control flags as follows:
 *
 *	disposal_method_flags = ((graphic_control_flags &
 *		IMG_GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK) >> 2);
 */
#define IMG_GIF_DISPOSAL_METHOD_NONE		0
#define IMG_GIF_DISPOSAL_METHOD_DO_NOTHING	(1 << 0)	/* Same as IMG_GIF_DISPOSAL_METHOD_NONE */
#define IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG	(1 << 1)	/* Clear with the background color */
#define IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV	(1 << 2)	/* Clear with the previous image */
#define IMG_GIF_DISPOSAL_METHOD_MASK		((1 << 0) | (1 << 1) | (1 << 2))


/*
 *	Read Context:
 */
struct _ImgGIFReadContext {

	ImgGIFDataStorageType data_storage_type;
	const void	*bp;
	unsigned long	bp_len,
			bp_pos;
	FILE		*fp;
	unsigned long	fp_len;

	unsigned long	io_buf_len;

	int		req_width, req_height;	/* Positive if opening as a
						 * thumb */

	u_int8_t	default_alpha_value;

	GifFileType	*gif_file;		/* GIF file handle */
	GifRowType	*gif_screen;		/* GIF screen buffer used to
						 * render and store each
						 * GIF frame's pixels during
						 * read */
	int		gif_width,		/* GIF screen geometry */
			gif_height,
			gif_bpp,
			gif_bpl;

	int		nframes,
			data_width, data_height,
			data_bpp, data_bpl;
	u_int8_t	**data_list;		/* Loaded image data frames
						 * list */
	unsigned long	*delay_ms_list;		/* Loaded delay in milliseconds
						 * list */

	u_int8_t	*bg_color;
	int		bg_color_num,
			transparent_color_num;

	char		*comments;

	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;
	int		status;
};

/*
 *	Write Context:
 */
struct _ImgGIFWriteContext {

	ImgGIFDataStorageType data_storage_type;
	void		**bp;
	unsigned long	*bp_len;
	FILE		*fp;

	unsigned long	io_buf_len;

	GifFileType	*gif_file;		/* GIF file handle */
	GifRowType	*gif_screen;		/* GIF screen buffer used to
						 * render and store each
						 * GIF frame's pixels during
						 * read */
	int		gif_width,		/* GIF screen geometry */
			gif_height,
			gif_bpp,
			gif_bpl;

	int		nframes,
			data_width, data_height,
			data_bpp, data_bpl;
	const u_int8_t	**data_list;		/* Image data frames list to
						 * save */
	const unsigned long	*delay_ms_list;	/* Delay in milliseconds list
						 * to save */

	const u_int8_t	*bg_color;
	int		bg_color_num,
			transparent_color_num;

	const char	*comments;

	int		interlaced;		/* 0 = Progressive
						 * 1 = Interlaced */
	int		color_type;		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	int		transparency;		/* 0 = No transparent colors
						 * 1 = Include transparent color */
	u_int8_t	alpha_threshold;
	int		looping;		/* 0 = Play once
						 * 1 = Repeating */

	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;
	int		status;
};

/*
 *	RGBAToCIdx Callback Data:
 */
struct _RGBAToCIdxData {

	int		frame_num,
			nframes;
	const u_int8_t	*rgba;
	int		width, height,
			bpp, bpl;

	ImgProgressFunc	progress_cb;
	void		*progress_data;
	int		*user_aborted;
};


static int	gif_interlace_offset[] = { 0, 4, 2, 1 },
		gif_interlace_jumps[] = { 8, 8, 4, 2 };


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the GIF library's version.
 */
void ImgGIFVersion(int *major, int *minor, int *release) 
{
	const char *ver_str = GIF_LIB_VERSION, *s;

	s = strstr(ver_str, "Version");
	while((*s != '\0') && (*s != ' '))
	    s++;
	while(*s == ' ')
	    s++;
	if(major != NULL)
	    *major = ATOI(s);

	while((*s != '\0') && (*s != '.'))
	    s++;
	while(*s == '.')
	    s++;
	if(minor != NULL)
	    *minor = ATOI(s);

	while((*s != '\0') && (*s != '.'))
	    s++;
	while(*s == '.')
	    s++;
	if(release != NULL)
	    *release = ATOI(s);
}


/*
 *	Checks for the GIF type ID.
 */
int ImgBufferIsGIF(const u_int8_t *data, const int len)
{
	if((data == NULL) || (len < 6))
	{
	    errno = EINVAL;
	    return(0);
	}

	if(!memcmp(data, "GIF87a", 6))
	    return(1);

	if(!memcmp(data, "GIF89a", 6))
	    return(1);

	errno = EINVAL;
	return(0);
}

int ImgStreamIsGIF(FILE *fp)
{
	u_int8_t buf[6];

	size_t units_read;

	if(fp == NULL)
	{
	    errno = EINVAL;
	    return(0);
	}

	units_read = fread(
	    buf,
	    sizeof(u_int8_t),
	    sizeof(buf) / sizeof(u_int8_t),
	    fp
	);
	if(units_read == (sizeof(buf) / sizeof(u_int8_t)))
	    return(ImgBufferIsGIF(buf, sizeof(buf)));
	else
	    return(0);
}

int ImgFileIsGIF(const char *path)
{
	int		status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(0);
	}

	fp = fopen(path, "rb");
	if(fp == NULL)
	    return(0);

	status = ImgStreamIsGIF(fp);
	error_code = (int)errno;

	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	Convert the GIF decode error code into an error string.
 */
static const char *ImgGIFErrorStringDecode(const int gif_error_code)
{
	switch(gif_error_code)
	{
	  case D_GIF_ERR_OPEN_FAILED:
	    return("Unable to open the file for reading");
	    break;
	  case D_GIF_ERR_READ_FAILED:
	    return("Unable to read the file");
	    break;
	  case D_GIF_ERR_NOT_GIF_FILE:
	    return("Not a GIF image");
	    break;
	  case D_GIF_ERR_NO_SCRN_DSCR:
	    return("Missing screen descriptor");
	    break;
	  case D_GIF_ERR_NO_IMAG_DSCR:
	    return("Missing image descriptor");
	    break;
	  case D_GIF_ERR_NO_COLOR_MAP:
	    return("Missing colormap");
	    break;
	  case D_GIF_ERR_WRONG_RECORD:
	    return("Wrong record");
	    break;
	  case D_GIF_ERR_DATA_TOO_BIG:
	    return("Data too big");
	    break;
	  case D_GIF_ERR_NOT_ENOUGH_MEM:
	    return("Memory allocation error");
	    break;
	  case D_GIF_ERR_CLOSE_FAILED:
	    return("Unable to close the file");
	    break;
	  case D_GIF_ERR_NOT_READABLE:
	    return("Permission denied");
	    break;
	  case D_GIF_ERR_IMAGE_DEFECT:
	    return("Corrupt image");
	    break;
	  case D_GIF_ERR_EOF_TOO_SOON:
	    return("Premature end of image encountered");
	    break;
	}

	return("An unknown error occured");
}

/*
 *	Convert the GIF encode error code into an error string.
 */
static const char *ImgGIFErrorStringEncode(const int gif_error_code)
{
	switch(gif_error_code)
	{
	  case E_GIF_ERR_OPEN_FAILED:
	    return("Unable to open the file for writing");
	    break;
	  case E_GIF_ERR_WRITE_FAILED:
	    return("Unable to write to the file");
	    break;
	  case E_GIF_ERR_HAS_SCRN_DSCR:
	    return("Has screen descriptor");
	    break;
	  case E_GIF_ERR_HAS_IMAG_DSCR:
	    return("Has image descriptor");
	    break;
	  case E_GIF_ERR_NO_COLOR_MAP:
	    return("Missing colormap");
	    break;
	  case E_GIF_ERR_DATA_TOO_BIG:
	    return("Data too big");
	    break;
	  case E_GIF_ERR_NOT_ENOUGH_MEM:
	    return("Memory allocation error");
	    break;
	  case E_GIF_ERR_DISK_IS_FULL:
	    return("Out of disk space");
	    break;
	  case E_GIF_ERR_CLOSE_FAILED:
	    return("Unable to close the file");
	    break;
	  case E_GIF_ERR_NOT_WRITEABLE:
	    return("Permission denied");
	    break;
	}

	return("An unknown error occured");
}

/*
 *	libgif read callback.
 *
 *	Returns the number of bytes read.
 */
static int ImgReadGIFReadCB(
	GifFileType *ft,
	GifByteType *buf, int buf_len
)
{
	ImgGIFReadContext *ctx = IMG_GIF_READ_CONTEXT(ft->UserData);
	if((ctx == NULL) || (buf == NULL) || (buf_len <= 0))
	    return(0);

	switch(ctx->data_storage_type)
	{
	  case IMG_GIF_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		unsigned long total_bytes_read = 0l;
		unsigned long len = (ctx->bp_len - ctx->bp_pos) * sizeof(u_int8_t);
		if(len > (buf_len * sizeof(GifByteType)))
		    len = buf_len * sizeof(GifByteType);

		(void)memcpy(
		    buf,
		    ctx->bp + ctx->bp_pos,
		    len
		);

		ctx->bp_pos += len;

		total_bytes_read += len;		

		/* Report progress */
		if(ctx->progress_cb != NULL)
		{
		    const int nframes = ctx->nframes;
		    if(!ctx->progress_cb(
			ctx->progress_data,
			(int)ctx->bp_pos,
			(int)ctx->bp_len,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl, ctx->data_bpp,
			(nframes > 0) ? ctx->data_list[nframes - 1] : NULL
		    ))
			*ctx->user_aborted = 1;
		}

		return(total_bytes_read);
	    }
	    break;

	  case IMG_GIF_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		const int io_buf_len = ctx->io_buf_len;
		unsigned long total_bytes_read = 0l;
		size_t	units_to_read,
			units_read;
		FILE *fp = ctx->fp;
		GifByteType	*buf_ptr = buf,
				*buf_end = buf_ptr + buf_len;
		while(buf_ptr < buf_end)
		{
		    units_to_read = (size_t)MIN(
			(buf_end - buf_ptr),
			io_buf_len
		    ) / sizeof(GifByteType);

		    units_read = fread(
			buf_ptr,
			sizeof(GifByteType),
			units_to_read,
			fp
		    );

		    total_bytes_read += units_read * sizeof(GifByteType);

		    /* Report progress */
		    if(ctx->progress_cb != NULL)
		    {
			const int nframes = ctx->nframes;
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (int)ftell(fp),
			    (int)ctx->fp_len,
			    ctx->data_width, ctx->data_height,
			    ctx->data_bpl, ctx->data_bpp,
			    (nframes > 0) ? ctx->data_list[nframes - 1] : NULL
			))
			    *ctx->user_aborted = 1;
		    }

		    if(units_read != units_to_read)
		    {
			if(ferror(fp))
			{
			    ImgOpenSetError(strerror(errno));
			    ctx->status = -1;
			    return((int)total_bytes_read);
			}
			else if(feof(fp))
			{
			    return((int)total_bytes_read);
			}
		    }

		    buf_ptr += units_read;
		}

		return((int)total_bytes_read);
	    }
	    break;
	}

	return(0);
}

/*
 *	Renders the GIF screen image data to the RGBA image data.
 */
static void ImgReadGIFRenderGIFScreenToRGBA(
	u_int8_t *rgba,
	const int rgba_width, const int rgba_height,
	const int rgba_bpl,
	const GifRowType *gif_screen,
	const int gif_width, const int gif_height,
	const int gif_bpl,
	const ColorMapObject *colormap,
	const u_int8_t default_alpha_value,
	const int transparent_color_num,
	const u_int8_t prev_disposal_method,
	const u_int8_t *prev_rgba,
	const u_int8_t *cwp_rgba		/* Clear with previous RGBA */
)
{
	const int	rgba_bpp = 4;		/* RGBA */
	int		tx, ty,
			tx_last, ty_last = -1,
			sx, sy,
			c_num;
	u_int8_t	*rgba_row,
			*rgba_ptr;
	GifRowType gif_screen_row;

	if(1)
	{
	    ty_last = -1;
	    for(sy = 0; sy < gif_height; sy++)
	    {
		ty = rgba_height * sy / gif_height;
		if(ty <= ty_last)
		    continue;

		gif_screen_row = gif_screen[sy];
		rgba_row = rgba + (ty * rgba_bpl);

		tx_last = -1;
		for(sx = 0; sx < gif_width; sx++)
		{
		    tx = rgba_width * sx / gif_width;
		    if(tx <= tx_last)
			continue;

		    rgba_ptr = rgba_row + (tx * rgba_bpp);

		    /* The GIF screen's pixel value equals the color
		     * index on the colormap
		     */
		    c_num = (int)gif_screen_row[sx];

		    /* Not transparent? */
		    if(c_num != transparent_color_num)
		    {
			const GifColorType *c = &colormap->Colors[c_num];
			*rgba_ptr++ = (u_int8_t)c->Red;
			*rgba_ptr++ = (u_int8_t)c->Green;
			*rgba_ptr++ = (u_int8_t)c->Blue;
			*rgba_ptr = default_alpha_value;
		    }
		    else
		    {
			/* Is transparent, handle by the previous
			 * image's disposal method
			 */
			const GifColorType *c;
			switch(prev_disposal_method)
			{
			  case IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV:
			    /* Use the clear with previous image's pixel */
			    if(cwp_rgba != NULL)
			    {
				(void)memcpy(
				    rgba_ptr,
				    cwp_rgba + (ty * rgba_bpl) +
					(tx * rgba_bpp),
				    rgba_bpp
				);
			    }
			    else
			    {
				/* No clear with previous image,
				 * set to transparent
				 */
				c = &colormap->Colors[c_num];
				*rgba_ptr++ = (u_int8_t)c->Red;
				*rgba_ptr++ = (u_int8_t)c->Green;
				*rgba_ptr++ = (u_int8_t)c->Blue;
				*rgba_ptr = 0x00;
			    }
			    break;

			  /* Clear with background */
			  case IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG:
			    /* Set to transparent */
			    c = &colormap->Colors[c_num];
			    *rgba_ptr++ = (u_int8_t)c->Red;
			    *rgba_ptr++ = (u_int8_t)c->Green;
			    *rgba_ptr++ = (u_int8_t)c->Blue;
			    *rgba_ptr = 0x00;
			    break;

			  case IMG_GIF_DISPOSAL_METHOD_DO_NOTHING:
			  case IMG_GIF_DISPOSAL_METHOD_NONE:
			  default:
			    /* Use the previous image's pixel */
			    if(prev_rgba != NULL)
			    {
				(void)memcpy(
				    rgba_ptr,
				    prev_rgba + (ty * rgba_bpl) +
					(tx * rgba_bpp),
				    rgba_bpp
				);
			    }
			    else
			    {
				/* No previous image, set to
				 * transparent
				 */
				const GifColorType *c = &colormap->Colors[c_num];
				*rgba_ptr++ = (u_int8_t)c->Red;
				*rgba_ptr++ = (u_int8_t)c->Green;
				*rgba_ptr++ = (u_int8_t)c->Blue;
				*rgba_ptr = 0x00;
			    }
			    break;
			}

		    }

		    tx_last = tx;
		}

		ty_last = ty;
	    }
	}
}

/*
 *	Reads the frames and other data from the GIF image file and
 *	stores them on the ImgGIFContext.
 */
static int ImgReadGIFChunksRGBA(ImgGIFReadContext *ctx)
{
	int		status = 0,
			gif_status;
	u_int8_t	disposal_method = IMG_GIF_DISPOSAL_METHOD_NONE,
			prev_disposal_method = disposal_method;
	const u_int8_t *cwp_rgba = NULL;	/* Clear With Previous RGBA image data */
	GifFileType *gif_file = ctx->gif_file;
	GifRecordType record_type;

	/* Begin reading the GIF file and open each frame
	 *
	 * Each frame will be opened to the GIF screen and then
	 * rendered to the RGBA image data
	 */
	do
	{
	    /* User aborted? */
	    if(*ctx->user_aborted)
	    {
		status = -4;
		break;
	    }

	    /* Get the next record type */
	    gif_status = DGifGetRecordType(
		gif_file,
		&record_type
	    );
	    if(gif_status != GIF_OK)
	    {
		ImgOpenSetError(DGIF_STR_ERROR(gif_status));
		status = -1;
		break;
	    }

	    if(*ctx->user_aborted)
	    {
		status = -4;
		break;
	    }

	    switch(record_type)
	    {
	      case UNDEFINED_RECORD_TYPE:
		break;

	      case SCREEN_DESC_RECORD_TYPE:
		break;

	      case IMAGE_DESC_RECORD_TYPE:
		gif_status = DGifGetImageDesc(gif_file);
		if(gif_status == GIF_OK)
		{
		    const int	rgba_bpl = ctx->data_bpl,
				frame_num = ctx->nframes,
				prev_frame_num = frame_num - 1;
		    const u_int8_t	*prev_rgba = (prev_frame_num >= 0) ?
			ctx->data_list[prev_frame_num] : NULL;
		    u_int8_t *rgba;
		    GifRowType *gif_screen = ctx->gif_screen;
		    GifImageDesc *gif_img = &gif_file->Image;

		    /* If this GIF frame has a local colormap then
		     * use it, otherwise use the GIF image file's
		     * global colormap
		     */
		    const ColorMapObject *colormap = (gif_img->ColorMap != NULL) ?
			gif_img->ColorMap : gif_file->SColorMap;

		    /* User aborted? */
		    if(*ctx->user_aborted)
		    {
			status = -4;
			break;
		    }

		    /* No local or global GIF colormap? */
		    if(colormap == NULL)
		    {
			imgio_last_open_error = "No GIF colormap found";
			status = -2;
			break;
		    }
		    /* Empty GIF colormap? */
		    if(colormap->ColorCount <= 0)
		    {
			imgio_last_open_error = "Empty GIF colormap";
			status = -2;
			break;
		    }

		    /* Read the pixels from this GIF frame on to the
		     * GIF screen
		     *
		     * Interlaced?
		     */
		    if(gif_img->Interlace)
		    {
			/* Read interlaced */
			const int	top = gif_img->Top,
					left = gif_img->Left,
					width = gif_img->Width,
					height = gif_img->Height;
			int i, j;

			/* Interlace needs 4 passes on the image */
			for(i = 0; i < 4; i++)
			{
			    for(j = top + gif_interlace_offset[i];
				j < (top + height);
				j += gif_interlace_jumps[i]
			    )
			    {
				gif_status = DGifGetLine(
				    gif_file,
				    (GifPixelType *)&gif_screen[j][left],
				    width
				);
				if(gif_status != GIF_OK)
				{
				    ImgOpenSetError(DGIF_STR_ERROR(gif_status));
				    status = -1;
				    break;
				}

				/* Check for user abort */
				if(*ctx->user_aborted)
				{
				    status = -4;
				    break;
				}
			    }
			}
		    }
		    else
		    {
			/* Read progressive */
			const int	left = gif_img->Left,
					width = gif_img->Width,
					height = gif_img->Height;
			int	i,
				top = gif_img->Top;
			for(i = 0; i < height; i++)
			{
			    gif_status = DGifGetLine(
				gif_file,
				(GifPixelType *)&gif_screen[top++][left],
				width
			    );
			    if(gif_status != GIF_OK)
			    {
				ImgOpenSetError(DGIF_STR_ERROR(gif_status));
				status = -1;
				break;
			    }

			    /* Check for user abort */
			    if(*ctx->user_aborted)
			    {
				status = -4;
				break;
			    }
			}
		    }
		    if(*ctx->user_aborted)
		    {
			status = -4;
			break;
		    }

		    /* Opening as a thumb? */
		    if((ctx->req_width > 0) && (ctx->req_height > 0))
		    {
			/* Count subsequent frames and increase the
			 * frames list allocation but do not append
			 * any new frames to it
			 */
			if(frame_num >= 1)
			{
			    ctx->nframes = frame_num + 1;
			    ctx->data_list = (u_int8_t **)realloc(
				ctx->data_list,
				ctx->nframes * sizeof(u_int8_t *)
			    );
			    if(ctx->data_list != NULL)
			    {
				ctx->data_list[frame_num] = NULL;
			    }
			    else
			    {
				ctx->nframes = 0;
				imgio_last_open_error = "Memory allocation error";
				status = -3;
			    }
			    break;
			}
		    }

		    /* Allocate the RGBA image data for this frame */
		    rgba = (u_int8_t *)malloc(
			rgba_bpl * ctx->data_height * sizeof(u_int8_t)
		    );
		    if(rgba == NULL)
		    {
			imgio_last_open_error = "Memory allocation error";
			status = -3;
			break;
		    }

		    /* Render the GIF screen on to the RGBA image for
		     * this frame
		     */
		    ImgReadGIFRenderGIFScreenToRGBA(
			rgba,
			ctx->data_width, ctx->data_height,
			ctx->data_bpl,
			ctx->gif_screen,
			ctx->gif_width, ctx->gif_height,
			ctx->gif_bpl,
			colormap,
			ctx->default_alpha_value,
			ctx->transparent_color_num,
			prev_disposal_method,
			prev_rgba,
			cwp_rgba
		    );

		    /* Append this RGBA image data to the RGBA images list */
		    ctx->nframes = frame_num + 1;
		    ctx->data_list = (u_int8_t **)realloc(
			ctx->data_list,
			ctx->nframes * sizeof(u_int8_t *)
		    );
		    if(ctx->data_list != NULL)
		    {
			ctx->data_list[frame_num] = rgba;
		    }
		    else
		    {
			ctx->nframes = 0;
			free(rgba);
			rgba = NULL;
			imgio_last_open_error = "Memory allocation error";
			status = -3;
			break;
		    }

		    /* Clear the GIF screen with the background after
		     * rendering this frame?
		     */
		    if(disposal_method & IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG)
		    {
			int i;

			/* Clear the first line */
			(void)memset(
			    gif_screen[0],
			    ctx->bg_color_num,
			    ctx->gif_width
			);

			/* Copy the first line to the subsequent lines */
			for(i = 1; i < ctx->gif_height; i++)
			    (void)memcpy(
				gif_screen[i],
				gif_screen[0],
				ctx->gif_width
			    );
		    }

		    /* If this frame's disposal method is not Clear With
		     * Previous then we need to unset the Clear With
		     * Previous RGBA image data pointer
		     */
		    if(!(disposal_method & IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV))
			cwp_rgba = NULL;
		}
		else
		{
		    ImgOpenSetError(DGIF_STR_ERROR(gif_status));
		    status = -1;
		}
		break;

	      case EXTENSION_RECORD_TYPE:
		if(1)
		{
		    u_int8_t get_error_need_break;
		    int	code,
			ext_len;
		    GifByteType *ext;

		    /* Get the first extension */
		    gif_status = DGifGetExtension(
			gif_file,
			&code,
			&ext
		    );
		    if(gif_status == GIF_OK)
		    {
			/* Iterate through the extensions */
			while((ext != NULL) && !(*ctx->user_aborted))
			{
			    /* Get this extension's length */
			    ext_len = (int)ext[0];

			    /* Reset the get extension error marker */
			    get_error_need_break = 0;

			    /* Handle by extension code */
			    switch(code)
			    {
			      case COMMENT_EXT_FUNC_CODE:	/* Comment */
				if(ext_len > 0)
				{
				    const int comment_len = STRLEN(ctx->comments);
				    if(ctx->comments != NULL)
				    {
					ctx->comments = (char *)realloc(
					    ctx->comments,
					    (comment_len + ext_len + 1) * sizeof(char)
					);
				    }
				    else
				    {
					ctx->comments = (char *)malloc(
					    (ext_len + 1) * sizeof(char)
					);
					if(ctx->comments != NULL)
					    *ctx->comments = '\0';
				    }
				    if(ctx->comments != NULL)
				    {
					(void)memcpy(
					    ctx->comments + comment_len,
					    ext + 1,
					    ext_len
					);
					ctx->comments[comment_len + ext_len] = '\0';
				    }
				}
				break;

			      case GRAPHICS_EXT_FUNC_CODE:	/* Graphics Control */
				/* Format (in bytes):
				 *
				 * <len>
				 * <flags> <delay_time> <delay_time_cnt>
				 * <transparent_color>
				 */
				if(ext_len >= 4)
				{
				    const int nframes = ctx->nframes;
				    const u_int8_t flags = (u_int8_t)ext[1];
				    char user_input;

				    /* Get the delay in milliseconds
				     *
				     * Needed to add 1 to nframes since
				     * Graphic Controls come before
				     * each frame
				     *
				     * Delay stored on GIF files are in
				     * hundreths of a second, need to
				     * * 10 to convert to milliseconds
				     */
				    ctx->delay_ms_list = (unsigned long *)realloc(
					ctx->delay_ms_list,
					(nframes + 1) * sizeof(unsigned long)
				    );
				    if(ctx->delay_ms_list != NULL)
					ctx->delay_ms_list[nframes] = (unsigned long)(
					    (u_int16_t)ext[2] |
					    ((u_int16_t)ext[3] << 8)
					) * 10l;

				    /* Get the transparent color number */
				    ctx->transparent_color_num = (flags & IMG_GIF_GRAPHIC_CONTROL_TRANSPARENCY) ?
					(int)ext[4] : -1;

				    /* Get user input */
				    user_input = (flags & IMG_GIF_GRAPHIC_CONTROL_USER_INPUT) ?
					1 : 0;
				    if(user_input)
				    {
					/* Wait for user input */

				    }

				    /* Record the previous disposal
				     * method and get the disposal
				     * method for this image
				     */
				    prev_disposal_method = disposal_method;
				    disposal_method = ((flags &
					IMG_GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK) >> 2);

				    /* If this disposal method is clear
				     * with prevous then update the
				     * Clear With Previous RGBA image
				     * data pointer if it was not set
				     * so that subsequent clear with
				     * previous disposal methods use
				     * this RGBA image data
				     */
				    if(disposal_method & IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV)
				    {
					if(cwp_rgba == NULL)
					{
					    const int frame_num = ctx->nframes - 1;
					    if(frame_num >= 0)
						cwp_rgba = ctx->data_list[frame_num];
					}
				    }
				}
				break;

			      case PLAINTEXT_EXT_FUNC_CODE:
				break;

			      case APPLICATION_EXT_FUNC_CODE:
				if(ext_len > 0)
				{
				    /* Netscape 2.0 looping control? */
				    if((ext_len >= 11) ?
					!memcmp(ext + 1, "NETSCAPE2.0", 11) : 0
				    )
				    {
					/* Get the value from the next extension */
					gif_status = DGifGetExtensionNext(
					    gif_file,
					    &ext
					);
					if(gif_status != GIF_OK)
					{
					    ImgOpenSetError(DGIF_STR_ERROR(gif_status));
					    status = -1;
					    get_error_need_break = 1;
					    break;
					}
					ext_len = (int)ext[0];
					if(ext_len >= 3)
					{
#if 0
					    const int	v1 = (int)ext[1],	/* v1 is always 1 */
							loop_count = (int)ext[2] |
							    ((int)ext[3] << 8);
#endif
					}
				    }
				}
				break;
			    }

			    /* Get extension error? */
			    if(get_error_need_break)
				break;

			    /* Get the next extension */
			    gif_status = DGifGetExtensionNext(
				gif_file,
				&ext
			    );
			    if(gif_status != GIF_OK)
			    {
				ImgOpenSetError(DGIF_STR_ERROR(gif_status));
				status = -1;
				break;
			    }
			}
		    }
		    else
		    {
			ImgOpenSetError(DGIF_STR_ERROR(gif_status));
			status = -1;
		    }
		}
		break;

	      case TERMINATE_RECORD_TYPE:
		break;
	    }
	}
	while((record_type != TERMINATE_RECORD_TYPE) && (status == 0));

	return(status);
}

/*
 *	Reads the GIF image data from a buffer or a stream to RGBA
 *	image data.
 */
static int ImgReadGIFRGBA(
	const ImgGIFDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		i,
			gif_status;
	GifFileType *gif_file;
	GifRowType gif_screen_row;
	ImgGIFReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)		{	\
 const int error_code = errno;			\
						\
 /* Delete our GIF read context */		\
 if(ctx != NULL) {				\
  /* Close the GIF file */			\
  if(ctx->gif_file != NULL)			\
   (void)DGifCloseFile(ctx->gif_file);		\
						\
  /* Delete the GIF screen */			\
  if(ctx->gif_screen != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->gif_height; i++)		\
    free(ctx->gif_screen[i]);			\
   free(ctx->gif_screen);			\
  }						\
						\
  /* Delete the loaded image frame data list */	\
  if(ctx->data_list != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->nframes; i++)		\
    free(ctx->data_list[i]);			\
   free(ctx->data_list);			\
  }						\
						\
  /* Delete the loaded delay list */		\
  free(ctx->delay_ms_list);			\
						\
  free(ctx->comments);				\
						\
  free(ctx);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_v_);					\
}

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	/* Create our GIF read context */
	ctx = IMG_GIF_READ_CONTEXT(calloc(1, sizeof(ImgGIFReadContext)));
	if(ctx == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/* 	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->fp_len = 0; */
#if 0
	ctx->req_width = 0;			/* Open actual size */
	ctx->req_height = 0;
#endif
	ctx->data_bpp = bpp;
	ctx->default_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = user_aborted;
/*	ctx->status = 0; */

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Check if the input is valid and get its statistics */
	switch(ctx->data_storage_type)
	{
	  case IMG_GIF_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		const void *bp = ctx->bp;
		const unsigned long bp_len = ctx->bp_len;

		ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;

		/* Check if the buffer data's signature is a GIF image */
		if(!ImgBufferIsGIF((const u_int8_t *)bp, (int)bp_len))
		{
		    imgio_last_open_error = "Not a GIF image";
		    CLEANUP_RETURN(-2);
		}
	    }
	    else
	    {
		imgio_last_open_error = "Buffer not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;

	  case IMG_GIF_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		long fp_pos;
		FILE *fp = ctx->fp;
		struct stat stat_buf;
		if(fstat(fileno(fp), &stat_buf))
		{
		    ctx->fp_len = 0l;
		    ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;
		}
		else
		{
		    ctx->fp_len = (unsigned long)stat_buf.st_size;
		    ctx->io_buf_len = MAX(
			stat_buf.st_blksize,
			IMG_GIF_MIN_IO_SIZE
		    );
		}

		/* Check if the stream data's signature is a GIF image */
		fp_pos = ftell(fp);
		if(!ImgStreamIsGIF(fp))
		{
		    (void)fseek(fp, fp_pos, SEEK_SET);
		    imgio_last_open_error = "Not a GIF image";
		    CLEANUP_RETURN(-2);
		}
		(void)fseek(fp, fp_pos, SEEK_SET);
	    }
	    else
	    {
		imgio_last_open_error = "Stream not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;
	}

	/* Set the GIF decode read callback and get the GIF decode
	 * file handle
	 */
	ctx->gif_file = gif_file = DGifOpen(
	    ctx,
	    ImgReadGIFReadCB
	);
	if(gif_file == NULL)
	{
	    ImgOpenSetError(DGIF_STR_ERROR(GifLastError()));
	    CLEANUP_RETURN(-1);
	}

	/* Get the GIF screen values */
	ctx->gif_width = gif_file->SWidth;
	ctx->gif_height = gif_file->SHeight;
	ctx->gif_bpp = sizeof(GifPixelType);
	ctx->gif_bpl = ctx->gif_width * ctx->gif_bpp;

	/* Invalid GIF screen geometry? */
	if((ctx->gif_width <= 0) || (ctx->gif_height <= 0) ||
	   (ctx->gif_bpp <= 0) || (ctx->gif_bpl <= 0)
	)
	{
	    imgio_last_open_error = "Invalid or empty GIF screen geometry";
	    errno = EINVAL;
	    CLEANUP_RETURN(-2);
	}

	/* Set our target RGBA image geometry to match the GIF screen */
	ctx->data_width = ctx->gif_width;
	ctx->data_height = ctx->gif_height;
	ctx->data_bpl = ctx->data_width * ctx->data_bpp;
#if 0
	ctx->nframes = 0;
	ctx->data_list = NULL;
	ctx->delay_ms_list = NULL;
#endif

	/* Get the GIF background color */
	ctx->bg_color_num = gif_file->SBackGroundColor;
	if((gif_file->SColorMap != NULL) && (ctx->bg_color != NULL))
	{
	    const ColorMapObject *colormap = gif_file->SColorMap;
	    const GifColorType *c = &colormap->Colors[ctx->bg_color_num];
	    ctx->bg_color[0] = (u_int8_t)c->Red;
	    ctx->bg_color[1] = (u_int8_t)c->Green;
	    ctx->bg_color[2] = (u_int8_t)c->Blue;
	    ctx->bg_color[3] = ctx->default_alpha_value;
	}

	/* Allocate the GIF screen, an array of GifRowType *, each
	 * GIF frame's pixels will be stored on the GIF screen before
	 * being rendered on to the RGBA image
	 */
	ctx->gif_screen = (GifRowType *)malloc(
	    ctx->gif_height * sizeof(GifRowType *)
	);
	if(ctx->gif_screen == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}

	/* Allocate the GIF screen and initialize all of its pixels
	 * to the background color, this will be used to store each
	 * GIF frame's pixels
	 */
	ctx->gif_screen[0] = gif_screen_row = (GifRowType)malloc(ctx->gif_bpl);
	if(gif_screen_row != NULL)
	    (void)memset(
		gif_screen_row,
		ctx->bg_color_num,
		ctx->gif_width
	    );
	for(i = 1; i < ctx->gif_height; i++)
	{
	    ctx->gif_screen[i] = gif_screen_row = (GifRowType)malloc(ctx->gif_bpl);
	    if(gif_screen_row != NULL)
		(void)memcpy(
		    gif_screen_row,
		    ctx->gif_screen[0],
		    ctx->gif_width
		);
	}

	/* Reset the transparent color number, this will be set in
	 * ImgReadGIFChunksRGBA() when an extension is encountered that
	 * specifies a transparent color number
	 */
	ctx->transparent_color_num = -1;

	/* Open all the frames and other data from the GIF image file
	 * and store them on the context
	 */
	ctx->status = ImgReadGIFChunksRGBA(ctx);

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted) &&
	   (ctx->status == 0)
	)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		ctx->data_height, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->data_width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->data_bpl;
	if(nframes_rtn != NULL)
	    *nframes_rtn = ctx->nframes;
	if(rgba_list_rtn != NULL)
	{
	    if(*rgba_list_rtn == NULL)
	    {
		*rgba_list_rtn = ctx->data_list;
		ctx->data_list = NULL;
	    }
	}
	if(delay_list_rtn != NULL)
	{
	    if(*delay_list_rtn == NULL)
	    {
		*delay_list_rtn = ctx->delay_ms_list;
		ctx->delay_ms_list = NULL;
	    }
	}
	if(x_rtn != NULL)
	    *x_rtn = 0;
	if(y_rtn != NULL)
	    *y_rtn = 0;
	if(base_width_rtn != NULL)
	    *base_width_rtn = ctx->gif_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = ctx->gif_height;
	if(comments_rtn != NULL)
	{
	    free(*comments_rtn);
	    *comments_rtn = ctx->comments;
	    ctx->comments = NULL;
	}

	/* End the GIF decoding */
	gif_status = DGifCloseFile(ctx->gif_file);
	ctx->gif_file = NULL;
	if(gif_status != GIF_OK)
	{
	    ImgOpenSetError(DGIF_STR_ERROR(gif_status));
	    ctx->status = -1;
	}

	if(*ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the GIF image file from a buffer to RGBA image data.
 */
int ImgBufferReadGIFRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadGIFRGBA(
	    IMG_GIF_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Reads the GIF image file from a stream to RGBA image data.
 */
int ImgStreamReadGIFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadGIFRGBA(
	    IMG_GIF_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the GIF image file to RGBA image data.
 */
int ImgFileOpenGIFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the GIF image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadGIFRGBA(
	    fp,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_list_rtn,
	    delay_list_rtn,
	    nframes_rtn,
	    bg_color,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

/*
 *	Opens the GIF image file to RGBA image data as a thumb.
 */
static int ImgReadGIFRGBAThumb(
	const ImgGIFDataStorageType data_storage_type,
	const void *bp, const unsigned long bp_len,
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		i,
			gif_status;
	GifFileType *gif_file;
	GifRowType gif_screen_row;
	ImgGIFReadContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)		{	\
 const int error_code = errno;			\
						\
 /* Delete our GIF read context */		\
 if(ctx != NULL) {				\
  /* Close the GIF file */			\
  if(ctx->gif_file != NULL)			\
   (void)DGifCloseFile(ctx->gif_file);		\
						\
  /* Delete the GIF screen */			\
  if(ctx->gif_screen != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->gif_height; i++)		\
    free(ctx->gif_screen[i]);			\
   free(ctx->gif_screen);			\
  }						\
						\
  /* Delete the loaded image frame data list */	\
  if(ctx->data_list != NULL) {			\
   int i;					\
   for(i = 0; i < ctx->nframes; i++)		\
    free(ctx->data_list[i]);			\
   free(ctx->data_list);			\
  }						\
						\
  /* Delete the loaded delay list */		\
  free(ctx->delay_ms_list);			\
						\
  free(ctx->comments);				\
						\
  free(ctx);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_v_);					\
}

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	/* Create our GIF read context */
	ctx = IMG_GIF_READ_CONTEXT(calloc(1, sizeof(ImgGIFReadContext)));
	if(ctx == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
/* 	ctx->bp_pos = 0l; */
	ctx->fp = fp;
/*	ctx->fp_len = 0; */
	ctx->req_width = req_width;		/* Open as a thumb */
	ctx->req_height = req_height;
	ctx->data_bpp = bpp;
	ctx->default_alpha_value = def_alpha_value;
	ctx->bg_color = bg_color;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = user_aborted;
/*	ctx->status = 0l; */

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Check if the input is valid and get its statistics */
	switch(ctx->data_storage_type)
	{
	  case IMG_GIF_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len > 0l))
	    {
		const void *bp = ctx->bp;
		const unsigned long bp_len = ctx->bp_len;

		ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;

		/* Check if the buffer data's signature is a GIF image */
		if(!ImgBufferIsGIF((const u_int8_t *)bp, (int)bp_len))
		{
		    imgio_last_open_error = "Not a GIF image";
		    CLEANUP_RETURN(-2);
		}
	    }
	    else
	    {
		imgio_last_open_error = "Buffer not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;

	  case IMG_GIF_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		long fp_pos;
		FILE *fp = ctx->fp;
		struct stat stat_buf;
		if(fstat(fileno(fp), &stat_buf))
		{
		    ctx->fp_len = 0l;
		    ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;
		}
		else
		{
		    ctx->fp_len = (unsigned long)stat_buf.st_size;
		    ctx->io_buf_len = MAX(
			stat_buf.st_blksize,
			IMG_GIF_MIN_IO_SIZE
		    );
		}

		/* Check if the stream data's signature is a GIF image */
		fp_pos = ftell(fp);
		if(!ImgStreamIsGIF(fp))
		{
		    (void)fseek(fp, fp_pos, SEEK_SET);
		    imgio_last_open_error = "Not a GIF image";
		    CLEANUP_RETURN(-2);
		}
		(void)fseek(fp, fp_pos, SEEK_SET);
	    }
	    else
	    {
		imgio_last_open_error = "Stream not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;
	}

	/* Set the GIF decode read callback and get the GIF decode
	 * file handle
	 */
	ctx->gif_file = gif_file = DGifOpen(
	    ctx,
	    ImgReadGIFReadCB
	);
	if(gif_file == NULL)
	{
	    ImgOpenSetError(DGIF_STR_ERROR(GifLastError()));
	    CLEANUP_RETURN(-1);
	}

	/* Get the GIF screen values */
	ctx->gif_width = gif_file->SWidth;
	ctx->gif_height = gif_file->SHeight;
	ctx->gif_bpp = sizeof(GifPixelType);
	ctx->gif_bpl = ctx->gif_width * ctx->gif_bpp;

	/* Invalid GIF screen geometry? */
	if((ctx->gif_width <= 0) || (ctx->gif_height <= 0) ||
	   (ctx->gif_bpp <= 0) || (ctx->gif_bpl <= 0)
	)
	{
	    imgio_last_open_error = "Invalid or empty GIF screen geometry";
	    errno = EINVAL;
	    CLEANUP_RETURN(-2);
	}

	/* Calculate the size of the target RGBA image in order to
	 * accomidate for the thumb
	 */
	ImgCalculateThumbSize(
	    ctx->gif_width, ctx->gif_height,	/* Original size */
	    ctx->req_width, ctx->req_height,	/* Thumb size limit */
	    &ctx->data_width, &ctx->data_height
	);

	ctx->data_bpl = ctx->data_width * ctx->data_bpp;
#if 0
	ctx->nframes = 0;
	ctx->data_list = NULL;
	ctx->delay_ms_list = NULL;
#endif

	/* Get the GIF background color */
	ctx->bg_color_num = gif_file->SBackGroundColor;
	if((gif_file->SColorMap != NULL) && (ctx->bg_color != NULL))
	{
	    const ColorMapObject *colormap = gif_file->SColorMap;
	    const GifColorType *c = &colormap->Colors[ctx->bg_color_num];
	    ctx->bg_color[0] = (u_int8_t)c->Red;
	    ctx->bg_color[1] = (u_int8_t)c->Green;
	    ctx->bg_color[2] = (u_int8_t)c->Blue;
	    ctx->bg_color[3] = ctx->default_alpha_value;
	}

	/* Allocate the GIF screen, an array of GifRowType *, each
	 * GIF frame's pixels will be stored on the GIF screen before
	 * being rendered on to the RGBA image
	 */
	ctx->gif_screen = (GifRowType *)malloc(
	    ctx->gif_height * sizeof(GifRowType *)
	);
	if(ctx->gif_screen == NULL)
	{
	    imgio_last_open_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}

	/* Allocate the GIF screen and initialize all of its pixels
	 * to the background color, this will be used to store each
	 * GIF frame's pixels
	 */
	ctx->gif_screen[0] = gif_screen_row = (GifRowType)malloc(ctx->gif_bpl);
	if(gif_screen_row != NULL)
	    (void)memset(
		gif_screen_row,
		ctx->bg_color_num,
		ctx->gif_width
	    );
	for(i = 1; i < ctx->gif_height; i++)
	{
	    ctx->gif_screen[i] = gif_screen_row = (GifRowType)malloc(ctx->gif_bpl);
	    if(gif_screen_row != NULL)
		(void)memcpy(
		    gif_screen_row,
		    ctx->gif_screen[0],
		    ctx->gif_width
		);
	}

	/* Reset the transparent color number, this will be set in
	 * ImgReadGIFChunksRGBA() when an extension is encountered that
	 * specifies a transparent color number
	 */
	ctx->transparent_color_num = -1;

	/* Open all the frames and other data from the GIF image file
	 * and store them on our context
	 */
	ctx->status = ImgReadGIFChunksRGBA(ctx);

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted) &&
	   (ctx->status == 0)
	)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		ctx->data_height, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	/* Set the return values */
	if(width_rtn != NULL)
	    *width_rtn = ctx->data_width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->data_height;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->data_bpl;
	if(orig_width_rtn != NULL)
	    *orig_width_rtn = ctx->gif_width;
	if(orig_height_rtn != NULL)
	    *orig_height_rtn = ctx->gif_height;
	if(nframes_rtn != NULL)
	    *nframes_rtn = ctx->nframes;
	if(play_time_ms_rtn != NULL)
	{
	    if(ctx->delay_ms_list != NULL)
	    {
		const int n = ctx->nframes;
		int i;
		for(i = 0; i < n; i++)
		    *play_time_ms_rtn = (*play_time_ms_rtn) + ctx->delay_ms_list[i];
	    }
	}
	if(rgba_rtn != NULL)
	{
	    if((ctx->nframes > 0) && (ctx->data_list != NULL))
	    {
		free(*rgba_rtn);
		*rgba_rtn = ctx->data_list[0];
		ctx->data_list[0] = NULL;
	    }
	}
	if(x_rtn != NULL)
	    *x_rtn = 0;
	if(y_rtn != NULL)
	    *y_rtn = 0;
	if(base_width_rtn != NULL)
	    *base_width_rtn = ctx->gif_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = ctx->gif_height;
	if(comments_rtn != NULL)
	{
	    free(*comments_rtn);
	    *comments_rtn = ctx->comments;
	    ctx->comments = NULL;
	}

	/* End the GIF decoding */
	gif_status = DGifCloseFile(ctx->gif_file);
	ctx->gif_file = NULL;
	if(gif_status != GIF_OK)
	{
	    ImgOpenSetError(DGIF_STR_ERROR(gif_status));
	    ctx->status = -1;
	}

	/* Check for user abort */
	if(*ctx->user_aborted)
	{
	    imgio_last_open_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Reads the GIF image data from a buffer to RGBA image data as
 *	a thumb.
 */
int ImgBufferReadGIFRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadGIFRGBAThumb(
	    IMG_GIF_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Reads the GIF image data from a stream to RGBA image data as
 *	a thumb.
 */
int ImgStreamReadGIFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	return(ImgReadGIFRGBAThumb(
	    IMG_GIF_DATA_STORAGE_STREAM,
	    NULL, 0l,
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	));
}

/*
 *	Opens the GIF image file to RGBA image data as a thumb.
 */
int ImgFileOpenGIFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	int		status,
			error_code;
	FILE *fp;

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	if(STRISEMPTY(path))
	{
	    imgio_last_open_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the GIF image file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    ImgOpenSetError(strerror(errno));
	    return(-1);
	}

	/* Read the stream */
	status = ImgStreamReadGIFRGBAThumb(
	    fp,
	    req_width, req_height,
	    width_rtn, height_rtn,
	    bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    orig_width_rtn, orig_height_rtn,
	    nframes_rtn,
	    play_time_ms_rtn,
	    x_rtn, y_rtn,
	    base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn,
	    author_rtn, comments_rtn,
	    def_alpha_value,
	    progress_cb, progress_data,
	    user_aborted
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}


/*
 *	RGBAToCIdx progress callback.
 */
static int ImgWriteRGBAToCIdxCB(int i, int m, void *data)
{
	RGBAToCIdxData *d = (RGBAToCIdxData *)data;
	if(d == NULL)
	    return(1);

	if((d->progress_cb != NULL) && ((i % IMG_GIF_PROGRESS_RESOLUTION) == 0))
	{
	    if(!d->progress_cb(
		d->progress_data,
		(d->frame_num * m) + (i / 2),
		(d->nframes * m),
		d->width, d->height,
		d->bpl, d->bpp,
		d->rgba
	    ))
		*d->user_aborted = 1;
	}

	return(*d->user_aborted);
}

/*
 *	Creates a dithered GIF ColorMapObject and GIF image from RGBA
 *	image data.
 *
 *	If bg_color and bg_color_num are not NULL then the background
 *	color specified by bg_color will be added to the colormap.
 *	*bg_color_num will be updated with the background color number.
 *
 *	If trans_color_num is not NULL then transparency will be
 *	processed and a transparent color will be added to the colormap.
 *	*trans_color_num will be updated with the transparent color
 *	number.
 *
 *	Inputs assumed valid except as noted above.
 */
static int ImgWriteGIFDitherRGBA(
	const int frame_num,
	const int nframes,
	const u_int8_t *rgba,
	const int width, const int height,
	const int bpl,
	const u_int8_t *bg_color,
	int *bg_color_num,
	const u_int8_t alpha_threshold,
	int *trans_color_num,
	ColorMapObject **colormap_rtn,
	GifByteType **color_index_rtn,
	int *color_index_bpl_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4,		/* RGBA */
			color_index_bpp = 1;	/* Color Index */
	int		i, n,
			color_index_bpl,
			ncolors,
			max_colors = 256;
	GifByteType *color_index;
	GifColorType *gif_c;
	ColormapColor	*c,
			*colors_list;
	ColorMapObject *colormap;
	RGBAToCIdxData *d;

	/* Reduce the maximum number of dither colors allowed to
	 * accomidate for the background and transparent colors
	 *
	 * We only need to reduce max_colors by one since the
	 * background color and the transparent color will be set as
	 * the same if either one is specified
	 */
	if((bg_color != NULL) || (trans_color_num != NULL))
	    max_colors--;

	/* Allocate the RGBAToCIdx() callback data */
	d = (RGBAToCIdxData *)calloc(1, sizeof(RGBAToCIdxData));
	if(d == NULL)
	    return(-3);

	d->frame_num = frame_num;
	d->nframes = nframes;
	d->width = width;
	d->height = height;
	d->bpp = bpp;
	d->bpl = bpl;
	d->rgba = rgba;
	d->progress_cb = progress_cb;
	d->progress_data = progress_data;
	d->user_aborted = user_aborted;

	/* If there is no transparency then we need to create an RGBA
	 * image data with the background color applied and use that
	 * to generate the color index image data with
	 */
	if((trans_color_num == NULL) && (bg_color != NULL))
	{
	    u_int8_t *rgba2 = (u_int8_t *)malloc(
		height * bpl * sizeof(u_int8_t)
	    );
	    if(rgba2 != NULL)
	    {
		int y;
		const u_int8_t	*src = rgba,
				*src_ptr;
		u_int8_t	*tar = rgba2,
				*tar_ptr,
				*tar_end;

		for(y = 0; y < height; y++)
		{
		    src_ptr = src + (y * bpl);
		    tar_ptr = tar + (y * bpl);
		    tar_end = tar_ptr + (width * bpp);
		    while(tar_ptr < tar_end)
		    {
			ImgConvertPixelRGBAToRGB(
			    src_ptr,		/* RGBA */
			    bg_color,		/* Background RGB */
			    tar_ptr		/* RGB */
			);
			tar_ptr[3] = 0xFF;
			src_ptr += bpp;
			tar_ptr += bpp;
		    }
		}
	    }
	    (void)RGBAToCIdx(
		rgba2,
		width, height,
		bpl,
		max_colors,
		(u_int8_t **)&color_index,
		&colors_list,
		&ncolors,
		ImgWriteRGBAToCIdxCB, d
	    );
	    free(rgba2);
	}
	else
	{
	    /* Generate the color index image data and colormap from
	     * the specified RGBA image data
	     */
	    (void)RGBAToCIdx(
		rgba,
		width, height,
		bpl,
		max_colors,
		(u_int8_t **)&color_index,
		&colors_list,
		&ncolors,
		ImgWriteRGBAToCIdxCB, d
	    );
	}

	free(d);

	/* User aborted? */
	if(*user_aborted)
	{
	    free(color_index);
	    free(colors_list);
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    return(-4);
	}

	/* Unable to create the color index image data? */
	if(color_index == NULL)
	{
	    const int error_code = errno;
	    free(colors_list);
	    imgio_last_save_error = "Unable to generate color index image";
	    errno = error_code;
	    return(-1);
	}

	color_index_bpl = width * color_index_bpp;

	/* Unable to generate the colors list? */
	if(colors_list == NULL)
	{
	    const int error_code = errno;
	    free(color_index);
	    imgio_last_save_error = "Unable to generate color palette";
	    errno = error_code;
	    return(-1);
	}

	/* Append the background color? */
	if(bg_color != NULL)
	{
	    const int i = ncolors;
	    ncolors = i + 1;
	    colors_list = (ColormapColor *)realloc(
		colors_list,
		ncolors * sizeof(ColormapColor)
	    );
	    if(colors_list == NULL)
	    {
		const int error_code = errno;
		free(color_index);
		imgio_last_save_error = "Memory allocation error";
		errno = error_code;
		return(-3);
	    }
	    c = &colors_list[i];
	    c->r = bg_color[0];
	    c->g = bg_color[1];
	    c->b = bg_color[2];
	    if(bg_color_num != NULL)
		*bg_color_num = i;
	}

	/* Need to append the transparent color to the colors list
	 * and apply it to the color index image data?
	 */
	if(trans_color_num != NULL)
	{
	    /* Was the background color specified? */
	    if((bg_color_num != NULL) && (bg_color != NULL))
	    {
		/* Use the background color as the transparent color */
		*trans_color_num = *bg_color_num;
	    }
	    else
	    {
		/* The background color was not specified, so append
		 * the transparent color to the colors list
		 */
		const int i = ncolors;
		ncolors = i + 1;
		colors_list = (ColormapColor *)realloc(
		    colors_list,
		    ncolors * sizeof(ColormapColor)
		);
		if(colors_list == NULL)
		{
		    const int error_code = errno;
		    imgio_last_save_error = "Memory allocation error";
		    errno = error_code;
		    return(-3);
		}
		c = &colors_list[i];
		c->r = 0x00;
		c->g = 0x00;
		c->b = 0x00;
		*trans_color_num = i;
	    }

	    /* Apply the transparency to the color index image data
	     * if an alpha threshold is specified as non-zero
	     */
	    if(alpha_threshold > 0x00)
	    {
		const GifByteType v = (GifByteType)(*trans_color_num);
		int y;
		const u_int8_t *rgba_ptr;
		GifByteType	*cidx_ptr,
				*cidx_end;

		/* Scan the RGBA image data, set all values on the
		 * color index image to the transparent color number
		 * v that correspond to a transparent pixel on the
		 * RGBA image data
		 */
		for(y = 0; y < height; y++)
		{
		    rgba_ptr = rgba + (y * bpl);
		    cidx_ptr = color_index + (y * color_index_bpl);
		    cidx_end = cidx_ptr + (width * color_index_bpp);
		    while(cidx_ptr < cidx_end)
		    {
			/* Transparent? */
			if(rgba_ptr[3] < alpha_threshold)
			    *cidx_ptr = v;
			rgba_ptr += bpp;
			cidx_ptr += color_index_bpp;
		    }
		}
	    }
	}

	/* Create the GIF colormap from the colors list */
	colormap = MakeMapObject(
	    256,				/* Must be 256 or else fails */
	    NULL
	);
	if(colormap == NULL)
	{
	    const int error_code = errno;
	    free(color_index);
	    free(colors_list);
	    imgio_last_save_error = "Memory allocation error";
	    errno = error_code;
	    return(-3);
	}
	n = MIN(ncolors, colormap->ColorCount);
	for(i = 0; i < n; i++)
	{
	    c = &colors_list[i];
	    gif_c = &colormap->Colors[i];
	    gif_c->Red = (GifByteType)c->r;
	    gif_c->Green = (GifByteType)c->g;
	    gif_c->Blue = (GifByteType)c->b;
	}

	/* Delete the colors list */
	free(colors_list);

	/* Set the return values */
	*color_index_rtn = color_index;
	*color_index_bpl_rtn = color_index_bpl;
	*colormap_rtn = colormap;

	return(0);
}


/*
 *	Writes a Netscape 2.0 looping application control block.
 *
 *	This should be called right after each EGifPutScreenDesc().
 */
static int ImgWriteGIFLoopingBlock(GifFileType *ft)
{
	const u_int8_t buf[] = {
		'N',			/* Bytes 0 to 7 = "NETSCAPE" */
		'E',
		'T',
		'S',
		'C',
		'A',
		'P',
		'E',
		'2',			/* Bytes 8 to 10 = "2.0" */
		'.',
		'0',
	};
	const u_int8_t buf2[] = {
		0x01,			/* 0x01 (unknown why as to this value) */
		0x00,			/* Lo byte of a u_int16_t loop count */
		0x00			/* Hi byte of a u_int16_t loop count */
	    };

	int gif_status = EGifPutExtensionFirst(
	    ft,
	    APPLICATION_EXT_FUNC_CODE,
	    sizeof(buf), buf
	);
	if(gif_status != GIF_OK)
	{
	    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
	    return(-1);
	}

	gif_status = EGifPutExtensionLast(
	    ft,
	    APPLICATION_EXT_FUNC_CODE,
	    sizeof(buf2), buf2
	);
	if(gif_status != GIF_OK)
	{
	    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
	    return(-1);
	}

	return(0);
}


/*
 *	libgif write callback.
 *
 *	Returns the number of bytes written.
 */
static int ImgWriteGIFWriteCB(
	GifFileType *ft,
	const GifByteType *buf, const int buf_len
)
{
	ImgGIFWriteContext *ctx = IMG_GIF_WRITE_CONTEXT(ft->UserData);
	if((ctx == NULL) || (buf == NULL) || (buf_len <= 0))
	    return(0);

	switch(ctx->data_storage_type)
	{
	  case IMG_GIF_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL))
	    {
		unsigned long bp_pos = *ctx->bp_len;
		*ctx->bp_len = bp_pos + buf_len;
		*ctx->bp = (void *)realloc(
		    *ctx->bp,
		    (*ctx->bp_len) * sizeof(u_int8_t)
		);
		if(*ctx->bp != NULL)
		{
		    (void)memcpy(
			(*ctx->bp) + bp_pos,
			buf,
			buf_len
		    );
		}
		else
		{
		    *ctx->bp_len = 0l;
		    imgio_last_save_error = "Memory allocation error";
		    ctx->status = -3;
		}
	    }
	    break;
	  case IMG_GIF_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		const unsigned long io_buf_len = ctx->io_buf_len;
		unsigned long total_bytes_written = 0l;
		FILE *fp = ctx->fp;
		size_t	units_to_write,
			units_written;
		const GifByteType	*buf_ptr = buf,
					*buf_end = buf_ptr + buf_len;
		while(buf_ptr < buf_end)
		{
		    units_to_write = (size_t)MIN(
			(buf_end - buf_ptr),
			io_buf_len
		    ) / sizeof(GifByteType);

		    units_written = fwrite(
			buf_ptr,
			sizeof(GifByteType),
			units_to_write,
			fp
		    );

		    total_bytes_written += units_written * sizeof(GifByteType);

		    if(units_written != units_to_write)
		    {
			ImgSaveSetError(strerror(errno));
			ctx->status = -1;
			return((int)total_bytes_written);
		    }

		    buf_ptr += units_written;
		}

		return((int)total_bytes_written);
	    }
	    break;
	}

	return(0);
}

/*
 *	Writes the image data to a GIF image file in color.
 */
static int ImgWriteGIFRGBAToColor(ImgGIFWriteContext *ctx)
{
	const int	width = ctx->data_width,
			height = ctx->data_height,
			bpp = ctx->data_bpp,
			bpl = ctx->data_bpl,
			nframes = ctx->nframes;
	int		status = 0,
			status2,
			gif_status,
			frame_num,
			cidx_bpl,
			color_resolution = 8,
			bg_color_num = 0,
			trans_color_num = -1;
	char user_input = 0;
	unsigned long delay_ms;
	const unsigned long *delay_ms_list = ctx->delay_ms_list;
	u_int8_t disposal_method;
	const u_int8_t	**rgba_list = ctx->data_list,
			*rgba_cur,
			*bg_color = ctx->bg_color,
			alpha_threshold = ctx->alpha_threshold;
	GifByteType *cidx = NULL;
	GifFileType *ft = ctx->gif_file;
	ColorMapObject *colormap = NULL;

	/* Write each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*ctx->user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba_list[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame in milliseconds */
	    delay_ms = (delay_ms_list != NULL) ? delay_ms_list[frame_num] : 0l;

	    /* Dither this RGBA image data frame and get the GIF
	     * colormap and the GIF color index image data
	     */
	    status2 = ImgWriteGIFDitherRGBA(
		frame_num,
		nframes,
		rgba_cur,
		width, height,
		bpl,
		bg_color,
		&bg_color_num,
		alpha_threshold,
		(ctx->transparency) ? &trans_color_num : NULL,
		&colormap,
		&cidx,
		&cidx_bpl,
		ctx->progress_cb, ctx->progress_data,
		ctx->user_aborted
	    );
	    if((status2 != 0) || (colormap == NULL) || (cidx == NULL))
	    {
		if(colormap != NULL)
		    FreeMapObject(colormap);
		free(cidx);
		continue;
	    }
	    if(*ctx->user_aborted)
	    {
		FreeMapObject(colormap);
		free(cidx);
		status = -4;
		break;
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		gif_status = EGifPutScreenDesc(
		    ft,
		    width, height,
		    color_resolution,
		    bg_color_num,
		    colormap
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    FreeMapObject(colormap);
		    free(cidx);
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && ctx->looping)
		{
		    if(ImgWriteGIFLoopingBlock(ft) != 0)
		    {
			FreeMapObject(colormap);
			free(cidx);
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(ctx->transparency)
		disposal_method = IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = IMG_GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Write the graphic control as needed */
	    if(ctx->transparency || user_input || (delay_ms > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay_ms / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (ctx->transparency ? IMG_GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? IMG_GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
	        };
		gif_status = EGifPutExtension(
		    ft,
		    GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    FreeMapObject(colormap);
		    free(cidx);
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    gif_status = EGifPutImageDesc(
		ft,
		0, 0,				/* Top & left */
		width, height,			/* Width & height */
		ctx->interlaced,
		/* Specify the local colormap only if multiple frames */
		(nframes > 1) ? colormap : NULL
	    );
	    if(gif_status != GIF_OK)
	    {
		ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		FreeMapObject(colormap);
		free(cidx);
		status = -1;
		break;
	    }

	    /* Delete the GIF colormap */
	    FreeMapObject(colormap);

	    /* Write the image data for this frame */
	    if(ctx->interlaced)
	    {
		const int	po = height,
				pm = 2 * height;
		int	i,
			y,
			lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
			y < height;
			y += gif_interlace_jumps[i]
		    )
		    {
			gif_status = EGifPutLine(
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			);
			if(gif_status != GIF_OK)
			{
			    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((ctx->progress_cb != NULL) && ((lines_written % IMG_GIF_PROGRESS_RESOLUTION) == 0))
			{
			    if(!ctx->progress_cb(
				ctx->progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*ctx->user_aborted = 1;
				status = -4;
				break;
			    }
			} 
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		const int	po = height,
				pm = 2 * height;
		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    gif_status = EGifPutLine(
			ft,
			(GifPixelType *)cidx_ptr,
			cidx_bpl
		    );
		    if(gif_status != GIF_OK)
		    {
			ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % IMG_GIF_PROGRESS_RESOLUTION) == 0))
		    {
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *ctx->user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
	    {
		free(cidx);
		break;
	    }

	    /* Delete the GIF color index image data */
	    free(cidx);
	}

	return(status);
}

/*
 *	Writes the image data to a GIF image file in greyscale.
 */
static int ImgWriteGIFRGBAToGreyscale(ImgGIFWriteContext *ctx)
{
	const int	width = ctx->data_width,
			height = ctx->data_height,
			bpp = ctx->data_bpp,
			bpl = ctx->data_bpl,
			nframes = ctx->nframes;
	int		status,
			gif_status,
			frame_num,
			cidx_bpl,
			color_resolution = 8,
			colormap_ncolors_set = 256,	/* Must be 256 or else fails */
			bg_color_num = 0,
			trans_color_num = -1;
	char user_input = 0;
	unsigned long delay_ms;
	const unsigned long *delay_ms_list = ctx->delay_ms_list;
	const u_int8_t	**rgba_list = ctx->data_list,
			*rgba_cur,
			*bg_color = ctx->bg_color;
	u_int8_t	disposal_method,
			alpha_threshold = ctx->alpha_threshold,
			bg_grey = (bg_color != NULL) ?
			    ImgConvertPixelRGBToGreyscale(bg_color) : 0x00;
	GifByteType *cidx;
	GifFileType *ft = ctx->gif_file;

	/* Create a greyscale GIF colormap */
	ColorMapObject *colormap = (ColorMapObject *)malloc(sizeof(ColorMapObject));
	if(colormap == NULL)
	{
	    imgio_last_save_error = "Memory allocation error";
	    return(-3);
	}
	colormap->ColorCount = colormap_ncolors_set;
	colormap->BitsPerPixel = 8;
	colormap->Colors = (GifColorType *)calloc(
	    colormap->ColorCount, sizeof(GifColorType)
	);
	/* Generate the greyscale colors */
	if(colormap->Colors != NULL)
	{
	    /* Calculate the greyscale background color */
	    if(ctx->transparency)
	    {
		/* Generate 255 colors from black to white, leaving
		 * color 255 for the transparent pixel
		 */
		const int	m = colormap->ColorCount - 1,
				n = MAX(m - 1, 1);
		int i;
		GifColorType *c;
		for(i = 0; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue =
			(GifByteType)(i * 0xFF / n);

		    /* If this color matches the background color then
		     * set this color as the background color
		     */
		    if(c->Red == (GifByteType)bg_grey)
			bg_color_num = i;
		}

		/* Set the transparent color */
		trans_color_num = m;
		c = &colormap->Colors[trans_color_num];
		c->Red = c->Green = c->Blue = 0xFF;
	    }
	    else
	    {
		/* Generate 256 colors from black to white */
		const int	m = colormap->ColorCount,
				n = MAX(m - 1, 1);
		int i;
		GifColorType *c;
		for(i = 0; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)(
			i * 0xFF / n
		    );

		    /* If this color matches the background color then
		     * set this color as the background color
		     */
		    if(c->Red == (GifByteType)bg_grey)
			bg_color_num = i;
		}
	    }
	}

	/* Allocate the GIF color index image data */
	cidx_bpl = width * sizeof(GifByteType);
	cidx = (GifByteType *)malloc(cidx_bpl * height);
	if(cidx == NULL)
	{
	    const int error_code = errno;
	    FreeMapObject(colormap);
	    imgio_last_save_error = "Memory allocation error";
	    errno = error_code;
	    return(-3);
	}

	status = 0;

	/* Write each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*ctx->user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba_list[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame in milliseconds */
	    delay_ms = (delay_ms_list != NULL) ? delay_ms_list[frame_num] : 0l;

	    /* Copy/convert the RGBA image data to the GIF color
	     * index image data in greyscale
	     */
	    if(ctx->transparency && (trans_color_num > -1))
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (cidx_bpl * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
	        {
		    /* Opaque? */
		    if(rgba_ptr[3] >= alpha_threshold)
			*cidx_ptr++ = (GifByteType)(ImgConvertPixelRGBAToGreyscale(
			    rgba_ptr,
			    bg_grey
			) * 255 / 256);
		    else
		        *cidx_ptr++ = (GifByteType)trans_color_num;
		    rgba_ptr += bpp;
	        }
	    }
	    else
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (cidx_bpl * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
		{
		    *cidx_ptr++ = (GifByteType)ImgConvertPixelRGBAToGreyscale(
			rgba_ptr,
			bg_grey
		    );
		    rgba_ptr += bpp;
		}
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		gif_status = EGifPutScreenDesc(
		    ft,
		    width, height,
		    color_resolution,
		    bg_color_num,
		    colormap
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && ctx->looping)
		{
		    if(ImgWriteGIFLoopingBlock(ft) != 0)
		    {
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(ctx->transparency)
		disposal_method = IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = IMG_GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Need to write the graphic control? */
	    if(ctx->transparency || user_input || (delay_ms > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay_ms / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (ctx->transparency ? IMG_GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? IMG_GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
		};
		gif_status = EGifPutExtension(
		    ft,
		    GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    gif_status = EGifPutImageDesc(
		ft,
		0, 0,				/* Top & left */
		width, height,			/* Width & height */
		ctx->interlaced,
		NULL				/* No local colormap */
	    );
	    if(gif_status != GIF_OK)
	    {
		ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		status = -1;
		break;
	    }

	    /* Write image data */
	    if(ctx->interlaced)
	    {
		const int	po = 0,
				pm = height;
		int	i, y,
			lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
			y < height;
			y += gif_interlace_jumps[i]
		    )
		    {
			gif_status = EGifPutLine( 
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			);
			if(gif_status != GIF_OK)
			{
			    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((ctx->progress_cb != NULL) && ((lines_written % IMG_GIF_PROGRESS_RESOLUTION) == 0))
			{
			    if(!ctx->progress_cb(
			        ctx->progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*ctx->user_aborted = 1;
				status = -4;
				break;
			    }
			}
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		const int	po = 0,
				pm = height;
 		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    gif_status = EGifPutLine(
			ft,
			(GifPixelType *)cidx_ptr,
			cidx_bpl
		    );
		    if(gif_status != GIF_OK)
		    {
			ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % IMG_GIF_PROGRESS_RESOLUTION) == 0))
		    {
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *ctx->user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
		break;
	}

	/* Delete the GIF colormap */
	FreeMapObject(colormap);

	/* Delete the GIF color index image data */
	free(cidx);

	return(status);
}

/*
 *	Writes the image data to a GIF image file in black & white.
 */
static int ImgWriteGIFRGBAToBW(ImgGIFWriteContext *ctx)
{
	const int	width = ctx->data_width,
			height = ctx->data_height,
			bpp = ctx->data_bpp,
			bpl = ctx->data_bpl,
			nframes = ctx->nframes;
	int		status,
			gif_status,
			frame_num,
			cidx_bpl,
			color_resolution = 8,
			colormap_ncolors_set = 256,	/* Must be 256 or else fails */
			bg_color_num = 0,
			trans_color_num = -1;
	char user_input = 0;
	unsigned long delay_ms;
	const unsigned long *delay_ms_list = ctx->delay_ms_list;
	const u_int8_t	**rgba_list = ctx->data_list,
			*rgba_cur,
			*bg_color = ctx->bg_color;
	u_int8_t	disposal_method,
			alpha_threshold = ctx->alpha_threshold,
			bg_grey = (bg_color != NULL) ?
			    ImgConvertPixelRGBToGreyscale(bg_color) : 0x00;
	GifByteType *cidx;
	GifFileType *ft = ctx->gif_file;

	/* Createa a black & white GIF colormap */
	ColorMapObject *colormap = (ColorMapObject *)malloc(sizeof(ColorMapObject));
	if(colormap == NULL)
	{
	    imgio_last_save_error = "Memory allocation error";
	    return(-3);
	}
	colormap->ColorCount = colormap_ncolors_set;
	colormap->BitsPerPixel = 8;
	colormap->Colors = (GifColorType *)calloc(
	    colormap->ColorCount, sizeof(GifColorType)
	);
	/* Generate the black and white colors */
	if(colormap->Colors != NULL)
	{
	    /* Calculate the black and white background color */
	    const GifByteType bg_bw = (GifByteType)((bg_grey >= 0x80) ?
		0xFF : 0x00
	    );

	    /* Is there transparency? */
	    if(ctx->transparency)
	    {
		/* Generate 3 colors; black, white, and the rest
		 * are black (including color 2 for the transparent
		 * pixel)
		 */
		const int m = colormap->ColorCount;

		/* Set color 0 to black */
		int i = 0;
		GifColorType *c = &colormap->Colors[i];
		c->Red = c->Green = c->Blue = (GifByteType)0x00;
		if(c->Red == bg_bw)
		    bg_color_num = i;

		/* Set color 1 to white */
		i = 1;
		if(i < m)
		{
		    c = &colormap->Colors[i]; 
		    c->Red = c->Green = c->Blue = (GifByteType)0xFF;
		    if(c->Red == bg_bw)
			bg_color_num = i;
		}

		/* Set color 2 (transparent) to black */
		trans_color_num = i = 2;
		if(i < m)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}

		/* Set the rest of the colors to black */
		for(i++; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}
	    }
	    else
	    {
		const int m = colormap->ColorCount;

		/* Set color 0 to black */
		int i = 0;
		GifColorType *c = &colormap->Colors[i];
		c->Red = c->Green = c->Blue = (GifByteType)0x00;

		/* Set color 1 to white */
		i = 1;
		if(i < m)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0xFF;
		}

		/* Set the rest of the colors to black */
		for(i++; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}
	    }
	}

	/* Allocate the GIF color index image data */
	cidx_bpl = width * sizeof(GifByteType);
	cidx = (GifByteType *)malloc(cidx_bpl * height);
	if(cidx == NULL)
	{
	    const int error_code = errno;
	    FreeMapObject(colormap);
	    imgio_last_save_error = "Memory allocation error";
	    errno = error_code;
	    return(-3);
	}

	status = 0;

	/* Write each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*ctx->user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba_list[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame in milliseconds */
	    delay_ms = (delay_ms_list != NULL) ? delay_ms_list[frame_num] : 0l;

	    /* Copy/convert the RGBA image data to the GIF color
	     * index image data in black & white
	     */
	    if(ctx->transparency && (trans_color_num > -1))
	    {
	        GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (width * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
		{
		    /* Solid? */
		    if(rgba_ptr[3] >= alpha_threshold)
			*cidx_ptr++ = (GifByteType)((ImgConvertPixelRGBAToGreyscale(
			    rgba_ptr,
			    bg_grey
			) >= 0x80) ? 1 : 0);
		    else
		        *cidx_ptr++ = (GifByteType)trans_color_num;
		    rgba_ptr += bpp;
	        }
	    }
	    else
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (width * height);
		const u_int8_t *rgba_ptr = rgba_cur;
	        while(cidx_ptr < cidx_end)
	        {
		    *cidx_ptr++ = (GifByteType)((ImgConvertPixelRGBAToGreyscale(
			rgba_ptr,
			bg_grey
		    ) > 0x80) ? 1 : 0);
		    rgba_ptr += bpp;
	        }
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		gif_status = EGifPutScreenDesc(
		    ft,
		    width, height,
		    color_resolution,
		    bg_color_num,
		    colormap
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && ctx->looping)
		{
		    if(ImgWriteGIFLoopingBlock(ft) != 0)
		    {
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(ctx->transparency)
		disposal_method = IMG_GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = IMG_GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Need to write the graphic control? */
	    if(ctx->transparency || user_input || (delay_ms > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay_ms / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (ctx->transparency ? IMG_GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? IMG_GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
		};
		gif_status = EGifPutExtension(
		    ft,
		    GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		);
		if(gif_status != GIF_OK)
		{
		    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    gif_status = EGifPutImageDesc(
		ft,
		0, 0,				/* Top & left */
		width, height,			/* Width & height */
		ctx->interlaced,
		NULL				/* No local colormap */
	    );
	    if(gif_status != GIF_OK)
	    {
		ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		status = -1;
		break;
	    }

	    /* Write the image data */
	    if(ctx->interlaced)
	    {
		/* Interlaced */
		const int	po = 0,
				pm = height;
		int	i,
			y,
			lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
		        y < height;
		        y += gif_interlace_jumps[i]
		    )
		    {
			gif_status = EGifPutLine(
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			);
			if(gif_status != GIF_OK)
			{
			    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((ctx->progress_cb != NULL) && ((lines_written % IMG_GIF_PROGRESS_RESOLUTION) == 0))
			{
			    if(!ctx->progress_cb(
				ctx->progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*ctx->user_aborted = 1;
				status = -4;
				break;
			    }
			}
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		/* Progressive (non-interlaced) */
		const int	po = 0,
				pm = height;
		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    gif_status = EGifPutLine(
			ft,
			(GifPixelType *)cidx_ptr,
			cidx_bpl
		    );
		    if(gif_status != GIF_OK)
		    {
			ImgSaveSetError(EGIF_STR_ERROR(gif_status));
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % IMG_GIF_PROGRESS_RESOLUTION) == 0))
		    {
			if(!ctx->progress_cb(
			    ctx->progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *ctx->user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
		break;
	}

	/* Delete the GIF colormap */
	FreeMapObject(colormap);

	/* Delete the GIF color index image data */
	free(cidx);

	return(status);
}

/*
 *	Writes RGBA image data to a GIF image stream or buffer.
 */
static int ImgWriteGIFRGBA(
	const ImgGIFDataStorageType data_storage_type,
	void **bp, unsigned long *bp_len,
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : width * bpp;
	int		user_aborted = 0,
			gif_status;
	ImgGIFWriteContext *ctx;

#define CLEANUP_RETURN(_v_)		{	\
 const int error_code = errno;			\
						\
 /* Delete our GIF write context */		\
 if(ctx != NULL) {				\
  /* Close the GIF file */			\
  if(ctx->gif_file != NULL)			\
   (void)EGifCloseFile(ctx->gif_file);		\
						\
  free(ctx);					\
 }						\
						\
 errno = error_code;				\
						\
 return(_v_);					\
}

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((width <= 0) || (height <= 0) ||
	   (rgba_list == NULL) || (nframes <= 0)
	)
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Create our GIF write context */
	ctx = IMG_GIF_WRITE_CONTEXT(calloc(1, sizeof(ImgGIFWriteContext)));
	if(ctx == NULL)
	{
	    imgio_last_save_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->data_storage_type = data_storage_type;
	ctx->bp = bp;
	ctx->bp_len = bp_len;
	ctx->fp = fp;
	ctx->gif_width = width;
	ctx->gif_height = height;
	ctx->data_width = width;
	ctx->data_height = height;
	ctx->data_bpp = bpp;
	ctx->data_bpl = _bpl;
	ctx->nframes = nframes;
	ctx->data_list = rgba_list;
	ctx->delay_ms_list = delay_ms_list;
	ctx->bg_color = bg_color;
	ctx->comments = comments;
	ctx->interlaced = interlaced;
	ctx->color_type = color_type;
	ctx->transparency = transparency;
	ctx->alpha_threshold = alpha_threshold;
	ctx->looping = looping;
	ctx->progress_cb = progress_cb;
	ctx->progress_data = progress_data;
	ctx->user_aborted = &user_aborted;

	/* Report the initial progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted))
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		0, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[0] : NULL
	    ))
		*ctx->user_aborted = 1;
	}
	if(*ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}

	/* Check if the output is valid and get its statistics */
	switch(ctx->data_storage_type)
	{
	  case IMG_GIF_DATA_STORAGE_BUFFER:
	    if((ctx->bp != NULL) && (ctx->bp_len != NULL))
	    {
		ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;
	    }
	    else
	    {
		imgio_last_save_error = "Buffer not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;

	  case IMG_GIF_DATA_STORAGE_STREAM:
	    if(ctx->fp != NULL)
	    {
		FILE *fp = ctx->fp;
		struct stat stat_buf;
		if(fstat(fileno(fp), &stat_buf))
		{
		    ctx->io_buf_len = IMG_GIF_DEF_IO_SIZE;
		}
		else
		{
		    ctx->io_buf_len = MAX(
			stat_buf.st_blksize,
			IMG_GIF_MIN_IO_SIZE
		    );
		}
	    }
	    else
	    {
		imgio_last_save_error = "Stream not specified";
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	    }
	    break;
	}

	/* Set the GIF encode write callback and get the GIF encode
	 * file handle
	 */
	ctx->gif_file = EGifOpen(
	    ctx,
	    ImgWriteGIFWriteCB
	);
	if(ctx->gif_file == NULL)
	{
	    ImgSaveSetError(EGIF_STR_ERROR(GifLastError()));
	    CLEANUP_RETURN(-1);
	}

	/* Write the GIF image chunks by the color type */
	switch(ctx->color_type)
	{
	  case 2:				/* Color */
	    ctx->status = ImgWriteGIFRGBAToColor(ctx);
	    break;

	  case 1:				/* Greyscale */
	    ctx->status = ImgWriteGIFRGBAToGreyscale(ctx);
	    break;

	  case 0:				/* Black & White */
	    ctx->status = ImgWriteGIFRGBAToBW(ctx);
	    break;

	  default:
	    imgio_last_save_error = "Unsupported color type";
	    ctx->status = -2;
	    errno = EINVAL;
	    break;
	}

	/* Write the comments */
	if(!STRISEMPTY(ctx->comments) &&
	   (ctx->status == 0) && !(*ctx->user_aborted)
	)
	{
	    gif_status = EGifPutExtension(
		ctx->gif_file,
		COMMENT_EXT_FUNC_CODE,
		STRLEN(ctx->comments), ctx->comments
	    );
	    if(gif_status != GIF_OK)
	    {
		ImgSaveSetError(EGIF_STR_ERROR(gif_status));
		ctx->status = -1;
	    }
	}

	/* Report the final progress */
	if((ctx->progress_cb != NULL) && !(*ctx->user_aborted) &&
	   (ctx->status == 0)
	)
	{
	    if(!ctx->progress_cb(
		ctx->progress_data,
		ctx->data_height, ctx->data_height,
		ctx->data_width, ctx->data_height,
		ctx->data_bpl, ctx->data_bpp,
		(ctx->nframes > 0) ? ctx->data_list[ctx->nframes - 1] : NULL
	    ))
		*ctx->user_aborted = 1;
	}

	/* End the GIF encoding */
	gif_status = EGifCloseFile(ctx->gif_file);
	ctx->gif_file = NULL;
	if(gif_status != GIF_OK)
	{
	    ImgSaveSetError(EGIF_STR_ERROR(gif_status));
	    ctx->status = -1;
	}

	if(*ctx->user_aborted)
	{
	    imgio_last_save_error = "User aborted operation";
	    errno = EINTR;
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    const int status = ctx->status;
	    CLEANUP_RETURN(status);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the RGBA image data to a buffer as a GIF image.
 */
int ImgBufferWriteGIFRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteGIFRGBA(
	    IMG_GIF_DATA_STORAGE_BUFFER,
	    bp, bp_len,
	    NULL,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    comments,
	    interlaced,
	    color_type,
	    transparency,
	    alpha_threshold,
	    looping,
	    progress_cb, progress_data
	));
}

/*
 *	Writes the RGBA image data to a stream as a GIF image.
 */
int ImgStreamWriteGIFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	return(ImgWriteGIFRGBA(
	    IMG_GIF_DATA_STORAGE_STREAM,
	    NULL, NULL,
	    fp,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    comments,
	    interlaced,
	    color_type,
	    transparency,
	    alpha_threshold,
	    looping,
	    progress_cb, progress_data
	));
}

/*
 *	Saves the RGBA image data to a GIF image file.
 */
int ImgFileSaveGIFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,                 /* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	int		status,
			error_code;
	FILE *fp;

	if(STRISEMPTY(path))
	{
	    imgio_last_save_error = "Invalid value";
	    errno = EINVAL;
	    return(-2);
	}

	/* Open the GIF image file for writing */
	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    ImgSaveSetError(strerror(errno));
	    return(-1);
	}

	/* Write the stream */
	status = ImgStreamWriteGIFRGBA(
	    fp,
	    width, height,
	    bpl,
	    rgba_list,
	    delay_ms_list,
	    nframes,
	    bg_color,
	    comments,
	    interlaced,
	    color_type,
	    transparency,
	    alpha_threshold,
	    looping,
	    progress_cb, progress_data
	);
	error_code = errno;

	/* Close the file */
	(void)fclose(fp);

	errno = error_code;

	return(status);
}

#endif	/* HAVE_LIBGIF */
