/*
			Image Input/Output Wrapper

	This is a set of modules that act as a master wrapper for
	various format-specific image IO libraries.


	All function names start with "Img".


	Functions that perform IO have the following name format:

	"Img<Buffer|File><Read/Open|Write/Save><StorageFormat><ImageFormat>[Special]"


	Functions that perform library-specific operations have the
	following name format:

	"Img<LibraryName><WhatItDoes>"


	Return values:

	0	Success.
	-1	General error.
	-2	Invalid value.
	-3	Memory allocation error, out of disk space, or systems error.
	-4	User aborted.
	-5	User responded with "no".
	-6	Reenterant call.
	-7	Unsupported.

 */
#ifndef IMGIO_H
#define IMGIO_H

#include <stdio.h>
#include <sys/types.h>
#if defined(WIN32)
# include "../include/os.h"
#endif
#include "imgio_formats.h"


/*
 *	Progress Function Type:
 */
typedef int (*ImgProgressFunc)(
	void *progress_data,
	const int i, const int n,
	const int width, const int height,
	const int bpl, const int bpp,
	const u_int8_t *image_data
);


/* Get/Set Error Messages */
extern const char *ImgOpenGetError(void);
extern const char *ImgSaveGetError(void);
extern void ImgOpenSetError(const char *s);
extern void ImgSaveSetError(const char *s);

/* Calculate Thumb Size */
extern void ImgCalculateThumbSize(
	const int orig_width, const int orig_height,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn
);

/* Convert Pixels */
extern u_int8_t ImgConvertPixelRGBToGreyscale(const u_int8_t *rgb);
extern u_int8_t ImgConvertPixelRGBAToGreyscale(
	const u_int8_t *rgba,
	const u_int8_t bg_grey
);
extern void ImgConvertPixelRGBAToRGB(
	const u_int8_t *rgba,
	const u_int8_t *bg_rgb,
	u_int8_t *rgb
);

/* Check Type */
extern int ImgIsSupportedExtension(const char *ext);

/* Open Any Format */
extern int ImgFileOpenRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileOpenRGBAThumb(
	const char *path,
	int req_width, int req_height,		/* Requested thumb size */
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *orig_nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
);

/* Image Library (Imlib) */
#ifdef HAVE_IMLIB
extern int ImgImlibSetHandle(void *p);
extern u_int8_t *ImgImlibConvertImlibToRGBA(
       const u_int8_t *rgb, const u_int8_t *alpha,
	int width, int height,
	int rgb_bpl, int alpha_bpl,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenImlibRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileSaveImlibRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const float quality,			/* 0.0 to 1.0 */
	const int color_type,			/* 1 = color
						 * 0 = greyscale */
	const u_int8_t def_alpha_value,
	const int imlib_fallback,		/* 0 = none
						 * 1 = ImageMajick & NetPBM */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_IMLIB */

/* Image Library 2 (Imlib2) */
#ifdef HAVE_IMLIB2
extern int ImgFileOpenImlib2RGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
#endif	/* HAVE_IMLIB2 */

/* Compuserve Graphic Interchange Format (GIF) */
#ifdef HAVE_LIBGIF
extern void ImgGIFVersion(int *major, int *minor, int *release);
extern int ImgBufferIsGIF(const u_int8_t *data, const int len);
extern int ImgStreamIsGIF(FILE *fp);
extern int ImgFileIsGIF(const char *path);
extern int ImgBufferReadGIFRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadGIFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenGIFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferReadGIFRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadGIFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenGIFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferWriteGIFRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments, 
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,			/* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWriteGIFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments, 
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,			/* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveGIFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments, 
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	const int transparency,			/* 0 = No transparent colors
						 * 1 = Include transparent color */
	const u_int8_t alpha_threshold,
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBGIF */

/* Joint Photographic Experts Group (JPEG) */
#ifdef HAVE_LIBJPEG
extern void ImgJPEGVersion(int *major, int *minor, int *release);
extern int ImgBufferIsJPEG(const u_int8_t *data, const int len);
extern int ImgStreamIsJPEG(FILE *fp);
extern int ImgFileIsJPEG(const char *path);
extern int ImgBufferReadJPEGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadJPEGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenJPEGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferReadJPEGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadJPEGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenJPEGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	int *orig_width_rtn, int *orig_height_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferWriteJPEGGreyscale(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWriteJPEGGreyscale(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveJPEGGreyscale(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *grey,
	const int quality,			/* 0 to 100 */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgBufferWriteJPEGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,               /* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWriteJPEGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveJPEGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */ 
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBJPEG */

/* Multi-image Network Graphics (MNG) and JPEG Network Graphics (JNG) */
#ifdef HAVE_LIBMNG
extern void ImgMNGVersion(int *major, int *minor, int *release);
extern int ImgBufferIsJNG(const u_int8_t *data, const int len);
extern int ImgBufferIsMNG(const u_int8_t *data, const int len);
extern int ImgStreamIsJNG(FILE *fp);
extern int ImgStreamIsMNG(FILE *fp);
extern int ImgFileIsJNG(const char *path);
extern int ImgFileIsMNG(const char *path);
extern int ImgBufferReadMNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadMNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenMNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferReadMNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadMNGRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenMNGRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferWriteJNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWriteJNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveJNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int quality,			/* 0 to 100 */
	const int color_type,			/* 0 = Greyscale
						 * 1 = Color */
	const int transparency,			/* 0 = No alpha mask,
						 * 1 = Include alpha mask */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgBufferWriteMNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWriteMNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveMNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	const int looping,			/* 0 = Play once
						 * 1 = Repeating */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBMNG */

/* Portable Network Graphics (PNG) */
#ifdef HAVE_LIBPNG
extern void ImgPNGVersion(int *major, int *minor, int *release);
extern int ImgBufferIsPNG(const u_int8_t *data, const int len);
extern int ImgStreamIsPNG(FILE *fp);
extern int ImgFileIsPNG(const char *path);
extern int ImgBufferReadPNGRGBA(
	const void *bp, const unsigned long bp_len,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadPNGRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenPNGRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferReadPNGRGBAThumb(
	const void *bp, const unsigned long bp_len,
	int req_width, int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadPNGRGBAThumb(
	FILE *fp,
	int req_width, int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenPNGRGBAThumb(
	const char *path,
	int req_width, int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgBufferWritePNGRGBA(
	void **bp, unsigned long *bp_len,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgStreamWritePNGRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSavePNGRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const unsigned long modified_time_sec,	/* In seconds since EPOCH */
	const int compression_level,		/* 0 to 9 */
	const int interlaced,			/* 0 = Progressive
						 * 1 = Interlaced */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBPNG */

/* Targa (TGA) */
extern int ImgFileOpenTGARGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileSaveTGARGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *comments,
	const int color_type,			/* 0 = Greyscale
						 * 1 = RGB
						 * 2 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);

/* Tag Image File Format (TIFF) */
#ifdef HAVE_LIBTIFF
extern void ImgTIFFVersion(int *major, int *minor, int *release);
extern int ImgBufferIsTIFF(const u_int8_t *data, const int len);
extern int ImgStreamIsTIFF(FILE *fp);
extern int ImgFileIsTIFF(const char *path);
extern int ImgStreamReadTIFFRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenTIFFRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_ms_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamReadTIFFRGBAThumb(
	FILE *fp,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileOpenTIFFRGBAThumb(
	const char *path,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgStreamWriteTIFFRGBA(
	FILE *fp,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
extern int ImgFileSaveTIFFRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t **rgba_list,
	const unsigned long *delay_ms_list,
	const int nframes,
	const u_int8_t *bg_color,               /* 4 bytes RGBA */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	const int compression_type,		/* 0 = None
						 * 1 = GZIP
						 * 2 = JPEG
						 * 3 = CCITT RLE
						 * 4 = CCITT FAX3
						 * 5 = CCITT FAX4 */
	const float jpeg_quality,		/* 0.0 to 1.0 */
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBTIFF */

/* X Pixmap (XPM) */
#ifdef HAVE_LIBXPM
extern void ImgXPMVersion(int *major, int *minor, int *release);
extern int ImgFileOpenXPMRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
extern int ImgFileSaveXPMRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const int x, const int y,
	const int base_width, const int base_height,
	const char *c_id,
	const char *comments,
	const int color_type,			/* 0 = B&W
						 * 1 = Greyscale    
						 * 2 = Color */ 
	const int max_colors,			/* -1 for no limit */
	const u_int8_t threshold,
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_LIBXPM */


#endif	/* IMGIO_H */
