#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#if defined(HAVE_IMLIB)
# include <Imlib.h>
#elif defined(HAVE_IMLIB2)
# warning "HAVE_IMLIB2 is defined, but support for Imlib2 in this module is experimental"
# include <X11/Xlib.h>
# include <Imlib2.h>
#endif

#if defined(_WIN32)
# include "../include/string.h"
#endif

#include "../include/disk.h"

#include "libps.h"
#include "imgio.h"


/*
 *	IO Types:
 *
 *	Determines which library to use to open or save the image.
 */
typedef enum {
	IMG_IO_TYPE_ANY,			/* Let us decide */
	IMG_IO_TYPE_IMLIB,			/* Imlib or Imlib2 */
	IMG_IO_TYPE_GIF,
	IMG_IO_TYPE_JPEG,			/* JFIF, JPE, JPEG, and JPG */
	IMG_IO_TYPE_MNG,			/* MNG and JNG */
	IMG_IO_TYPE_PNG,
	IMG_IO_TYPE_PS,
	IMG_IO_TYPE_TGA,
	IMG_IO_TYPE_TIFF,
	IMG_IO_TYPE_XPM
} ImgIOType;


/* Path Parsing */
static const char *ImgGetExtensionFromPath(const char *path);
static const char *ImgGetNameFromPath(const char *path);

/* Get/set error message */
const char *ImgOpenGetError(void);
const char *ImgSaveGetError(void);
void ImgOpenSetError(const char *s);
void ImgSaveSetError(const char *s);

/* Calculate Thumb Size */
void ImgCalculateThumbSize(
	const int orig_width, const int orig_height,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn
);

/* Convert Pixels */
u_int8_t ImgConvertPixelRGBToGreyscale(const u_int8_t *rgb);
u_int8_t ImgConvertPixelRGBAToGreyscale(
	const u_int8_t *rgba,
	const u_int8_t bg_grey
);
void ImgConvertPixelRGBAToRGB(
	const u_int8_t *rgba,
	const u_int8_t *bg_rgb,
	u_int8_t *rgb
);

/* Check Type */
int ImgIsSupportedExtension(const char *ext);

/* Get ImgIOType */
static ImgIOType ImgGetIOTypeFromExtension(const char *ext);

/* Open Any Format */
int ImgFileOpenRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn, unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn, 
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH GMT */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
);
int ImgFileOpenRGBAThumb(
	const char *path,
	int req_width, int req_height,		/* Requested thumb size */
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *orig_nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH GMT */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
);

#ifdef HAVE_IMLIB
/* Image Library (Imlib) */
static void		*img_imlib_handle;
int ImgImlibSetHandle(void *p);
u_int8_t *ImgImlibConvertImlibToRGBA(
	const u_int8_t *rgb,
	const u_int8_t *alpha,
	const int width, const int height,
	const int rgb_bpl,
	const int alpha_bpl,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileOpenImlibRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
int ImgFileSaveImlibRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const float quality,			/* 0.0 to 1.0 */
	const int color_type,			/* 0 = Grayscale
						 * 1 = Color */
	const u_int8_t def_alpha_value,
	const int imlib_fallback,		/* 0 = none
						 * 1 = ImageMajick & NetPBM */
	ImgProgressFunc progress_cb, void *progress_data
);
#endif	/* HAVE_IMLIB */

#ifdef HAVE_IMLIB2
/* Image Library 2 (Imlib2) */
static ImgProgressFunc img_imlib2_progress_cb;
static void		*img_imlib2_progress_data;
static int ImgImlib2ProgressCB(
	Imlib_Image im, char percent,
	int update_x, int update_y,
	int update_w, int update_h
);
int ImgFileOpenImlib2RGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
);
#endif	/* HAVE_IMLIB2 */


/* Last error message pointer */
const char	*imgio_last_open_error = NULL,
		*imgio_last_save_error = NULL;

/* Error message buffer */
static char	imgio_error_buf[1024];


/*
 *	Progress update resolution (in lines per update):
 */
#define IMG_IO_PROGRESS_RESOLUTION	10


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)


/*
 *	Gets the extension from the path.
 *
 *	Returns the extension with the '.' character or NULL if no
 *	extension was found.
 */
static const char *ImgGetExtensionFromPath(const char *path)
{
	const char *name = ImgGetNameFromPath(path);
	if(name == NULL)
		return(NULL);

	return(strchr(name, '.'));
}

/*
 *	Gets the name from the path.
 *
 *	Returns the name from the path or path if the path did not
 *	have any directory deliminators.
 */
static const char *ImgGetNameFromPath(const char *path)
{
	const char *name;

	if(path == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	name = strrchr(path, DIR_DELIMINATOR);
	if(name != NULL)
		return(name + 1);
	else
		return(path);
}


/*
 *	Gets the last open error message.
 */
const char *ImgOpenGetError(void)
{
	return(imgio_last_open_error);
}
 
/*
 *	Gets the last save error message.
 */
const char *ImgSaveGetError(void)
{
	return(imgio_last_save_error);
}

/*
 *	Sets the last open error message.
 */
void ImgOpenSetError(const char *s)
{
	if(s == NULL)
	{
		*imgio_error_buf = '\0';
		imgio_last_open_error = NULL;
	}

	if(s != imgio_error_buf)
	{
		(void)strncpy(
			imgio_error_buf,
			s,
			sizeof(imgio_error_buf)
		);
		imgio_error_buf[sizeof(imgio_error_buf) - 1] = '\0';
		imgio_last_open_error = imgio_error_buf;
	}
}

/*
 *	Sets the last save error message.
 */
void ImgSaveSetError(const char *s)
{
	if(s == NULL)
	{
		*imgio_error_buf = '\0';
		imgio_last_save_error = NULL;
	}

	if(s != imgio_error_buf)
	{
		(void)strncpy(
			imgio_error_buf,
			s,
			sizeof(imgio_error_buf)
		);
		imgio_error_buf[sizeof(imgio_error_buf) - 1] = '\0';
		imgio_last_save_error = imgio_error_buf;
	}
}


/*
 *	Calculates the size needed to fit into a thumb.
 *
 *	The orig_width and orig_height specifies the original size.
 *
 *	The req_width and req_height specifies the maximum size limit
 *	for the thumb.
 *
 *	The width_rtn and height_rtn specifies the return values for
 *	the calculated thumb size.
 *
 *	The return size will always be positive.
 */
void ImgCalculateThumbSize(
	const int orig_width, const int orig_height,
	const int req_width, const int req_height,
	int *width_rtn, int *height_rtn
)
{
	if((width_rtn == NULL) || (height_rtn == NULL))
		return;

	*width_rtn = orig_width;
	*height_rtn = orig_height;

	/* Is the original size larger than the requested thumb size? */
	if((*width_rtn > req_width) && (orig_width > 0))
	{
		*width_rtn = req_width;
		*height_rtn = req_width * orig_height / orig_width;
	}
	if((*height_rtn > req_height) && (orig_height > 0))
	{
		*height_rtn = req_height;
		*width_rtn = req_height * orig_width / orig_height;
	}

	/* Return size must always be positive */
	if(*width_rtn < 1)
		*width_rtn = 1;
	if(*height_rtn < 1)
		*height_rtn = 1;
}


/*
 *	Converts the RGB pixel to a greyscale pixel.
 */
u_int8_t ImgConvertPixelRGBToGreyscale(const u_int8_t *rgb)
{
	return((u_int8_t)(
		(0.299f * rgb[0]) +
		(0.587f * rgb[1]) +
		(0.114f * rgb[2])
	));
}

/*
 *	Converts the RGBA pixel to a greyscale pixel.
 */
u_int8_t ImgConvertPixelRGBAToGreyscale(
	const u_int8_t *rgba,
	const u_int8_t bg_grey
)
{
	switch(rgba[3])
	{
	  case 0xFF:
		return(ImgConvertPixelRGBToGreyscale(rgba));
		break;
	  case 0x00:
		return(bg_grey);
		break;
	  default:
		{
			const float	rgba_coeff = (float)rgba[3] / (float)0xFF,
					bg_coeff = 1.0f - rgba_coeff;
			return((u_int8_t)(
				(ImgConvertPixelRGBToGreyscale(rgba) * rgba_coeff) +
				(bg_grey * bg_coeff)
			));
		}
		break;
	}
}

/*
 *	Converts the RGBA pixel to a RGB pixel.
 */
void ImgConvertPixelRGBAToRGB(
	const u_int8_t *rgba,
	const u_int8_t *bg_rgb,
	u_int8_t *rgb
)
{
	switch(rgba[3])
	{
	  case 0xFF:
		rgb[0] = rgba[0];
		rgb[1] = rgba[1];
		rgb[2] = rgba[2];
		break;
	  case 0x00:
		rgb[0] = bg_rgb[0];
		rgb[1] = bg_rgb[1];
		rgb[2] = bg_rgb[2];
		break;
	  default:
		{
			const float	src_coeff = (float)rgba[3] / (float)0xFF,
					bg_coeff = 1.0f - src_coeff;
			rgb[0] = (u_int8_t)(
				(rgba[0] * src_coeff) +
				(bg_rgb[0] * bg_coeff)
			);
			rgb[1] = (u_int8_t)(
				(rgba[1] * src_coeff) +
				(bg_rgb[1] * bg_coeff)
			);
			rgb[2] = (u_int8_t)(
				(rgba[2] * src_coeff) +
				(bg_rgb[2] * bg_coeff)
			);
		}
		break;
	}
}


/*
 *	Checks if the extension belongs to a supported image format.
 *
 *	The ext specifies the string describing the extension (includes
 *	the "." character).
 */
int ImgIsSupportedExtension(const char *ext)
{
	const char *ext_list[] = IMAGE_FORMAT_EXTENSIONS_LIST;
	int i;

	if(ext == NULL)
		return(0);

	for(i = 0; ext_list[i] != NULL; i++)
	{
		if(!strcasecmp(ext_list[i], ext))
			return(1);
	}

	return(0);
}


/*
 *	Gets the ImgIOType based on the extension.
 */
static ImgIOType ImgGetIOTypeFromExtension(const char *ext)
{
	if(STRISEMPTY(ext))
		return(IMG_IO_TYPE_IMLIB);

	if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_GIF))
		return(
#if defined(HAVE_LIBGIF)
			IMG_IO_TYPE_GIF
#else
			IMG_IO_TYPE_IMLIB
#endif
		);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_JFIF) ||
		!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_JPE) ||
		!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_JPEG) ||
		!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_JPG)
	)
		return(
#if defined(HAVE_LIBJPEG)
			IMG_IO_TYPE_JPEG
#else
			IMG_IO_TYPE_IMLIB
#endif
		);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_JNG) ||
		!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_MNG)
	)
		return(
#if defined(HAVE_LIBMNG)
			IMG_IO_TYPE_MNG
#else
			IMG_IO_TYPE_IMLIB
#endif
		);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_PNG))
		return(
#if defined(HAVE_LIBPNG)
			IMG_IO_TYPE_PNG
#else
			IMG_IO_TYPE_IMLIB
#endif
		);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_PS))
		return(IMG_IO_TYPE_PS);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_TGA))
		return(IMG_IO_TYPE_TGA);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_TIFF) ||
			!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_TIF)
	)
		return(IMG_IO_TYPE_TIFF);
	else if(!strcasecmp(ext, IMAGE_FORMAT_EXTENSION_XPM))
		return(
#if defined(HAVE_LIBXPM)
			IMG_IO_TYPE_XPM
#else
			IMG_IO_TYPE_IMLIB
#endif
		);
	else
		return(IMG_IO_TYPE_IMLIB);
}


/*
 *	Opens the image file to RGBA image data.
 */
int ImgFileOpenRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t ***rgba_list_rtn,
	unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn, 
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH GMT */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
			error_code,
			user_aborted = 0,
			width = 0, height = 0,
			bpl = 0,
			x = 0, y = 0,
			base_width = 0, base_height = 0;
	const char *ext;
	char		*creator = NULL,
			*title = NULL,
			*author = NULL,
			*comments = NULL;
	unsigned long modified_time_sec = 0l;
	u_int8_t *rgba = NULL;
	ImgIOType io_type;

	/* Reset the last open error message pointer */
	imgio_last_open_error = NULL;

	if((width_rtn == NULL) || (height_rtn == NULL) ||
	   (bpl_rtn == NULL) ||
	   (nframes_rtn == NULL) || (rgba_list_rtn == NULL) ||
	   (delay_list_rtn == NULL)
	)
	{
		imgio_last_open_error = "Invalid value";
		errno = EINVAL;
 	    return(-2);
	}

	/* Reset the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;
	*rgba_list_rtn = NULL;
	*delay_list_rtn = NULL;
	*nframes_rtn = 0;
	if(x_rtn != NULL)
		*x_rtn = x;
	if(y_rtn != NULL)
		*y_rtn = y;
	if(base_width_rtn != NULL)
		*base_width_rtn = base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = base_height;
	if(creator_rtn != NULL)
		*creator_rtn = creator;
	if(title_rtn != NULL)
		*title_rtn = title;
	if(author_rtn != NULL)
		*author_rtn = author;
	if(comments_rtn != NULL)
		*comments_rtn = comments;
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = modified_time_sec;

	if(STRISEMPTY(path))
	{
		imgio_last_open_error = "The path to the image file was not specified";
		errno = EINVAL;
		return(-2);
	}

	/* Get the extension */
	ext = ImgGetExtensionFromPath(path);
	if(ext == NULL)
		ext = "";

	/* Determine the image IO type from the extension */
	io_type = ImgGetIOTypeFromExtension(ext);

	/* Reset the status to -1 and open by the IO type */
	status = -1;
	error_code = 0;

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			progress_data,
			0, height,
			width, height,
			bpl, bpp,
			rgba
		))
			user_aborted = 1;
	}

/* Appends/transfers the RGBA image data to the RGBA images list */
#define APPEND_FRAME_RGBA(_rgba_)	{	\
 if((_rgba_) != NULL) {				\
  const int frame_num = *nframes_rtn;		\
  *nframes_rtn = frame_num + 1;			\
  *rgba_list_rtn = (u_int8_t **)realloc(	\
   *rgba_list_rtn,				\
   (*nframes_rtn) * sizeof(u_int8_t *)		\
  );						\
  if(*rgba_list_rtn != NULL) {			\
   (*rgba_list_rtn)[frame_num] = (_rgba_);	\
  } else {					\
   free(_rgba_);				\
   *nframes_rtn = 0;				\
  }						\
 }						\
}

	/* Begin opening the image by the IO type */
	switch(io_type)
	{
	  case IMG_IO_TYPE_ANY:
	  case IMG_IO_TYPE_IMLIB:		/* Imlib or Imlib2 */
#if defined(HAVE_IMLIB)
		status = ImgFileOpenImlibRGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
#elif defined(HAVE_IMLIB2)
		status = ImgFileOpenImlib2RGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
#endif
		break;

	  case IMG_IO_TYPE_GIF:
#if defined(HAVE_LIBGIF)
		status = ImgFileOpenGIFRGBA(
			path,
			&width, &height,
			&bpl,
			rgba_list_rtn,
			delay_list_rtn,
			nframes_rtn,
			bg_color,			/* 4 bytes RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
#endif
		break;

	  case IMG_IO_TYPE_JPEG:
#if defined(HAVE_LIBJPEG)
		status = ImgFileOpenJPEGRGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			progress_cb, progress_data,
			&user_aborted 
		);
		error_code = errno;
		base_width = width;
		base_height = height;
		APPEND_FRAME_RGBA(rgba);
#endif
		break;

	  case IMG_IO_TYPE_MNG:
#if defined(HAVE_LIBMNG)
		status = ImgFileOpenMNGRGBA(
			path,
			&width, &height,
			&bpl,
			rgba_list_rtn,
			delay_list_rtn,
			nframes_rtn,
			bg_color,			/* 4 bytes RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			&modified_time_sec,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
#endif
		break;

	  case IMG_IO_TYPE_PNG:
#if defined(HAVE_LIBPNG)
		status = ImgFileOpenPNGRGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			bg_color,			/* 4 byte RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			&modified_time_sec,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
#endif
		break;

	  case IMG_IO_TYPE_PS:
		status = PSReadFileRGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			bg_color,			/* 4 byte RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
		imgio_last_open_error = PSLastError();
		switch(status)
		{
		  case PSSuccess:
			status = 0;
			break;
		  case PSError:
			status = -1;
			break;
		  case PSBadValue:
			status = -2;
			break;
		  case PSErrorSystem:
			status = -3;
			break;
		  case PSAbort:
			status = -4;
			break;
		  default:
			status = -1;
			break;
		}
		break;

	  case IMG_IO_TYPE_TGA:
		status = ImgFileOpenTGARGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			bg_color,			/* 4 bytes RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
		break;

	  case IMG_IO_TYPE_TIFF:
#if defined(HAVE_LIBTIFF)
		status = ImgFileOpenTIFFRGBA(
			path,
			&width, &height,
			&bpl,
			rgba_list_rtn,
			delay_list_rtn,
			nframes_rtn,
			bg_color,			/* 4 bytes RGBA (will be modified) */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
#endif
		break;

	  case IMG_IO_TYPE_XPM:
#if defined(HAVE_LIBXPM)
	   status = ImgFileOpenXPMRGBA(
			path,
			&width, &height,
			&bpl,
			&rgba,
			bg_color,			/* 4 bytes RGBA (will be modified) */
			&x, &y, &base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted
		);
		error_code = errno;
		APPEND_FRAME_RGBA(rgba);
#endif
		break;
	}
#undef APPEND_FRAME_RGBA

	/* Set the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;

	if(x_rtn != NULL)
		*x_rtn = x;
	if(y_rtn != NULL)
		*y_rtn = y;
	if(base_width_rtn != NULL)
		*base_width_rtn = base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = base_height;

	if(creator_rtn != NULL)
		*creator_rtn = creator;
	else
		free(creator);
	if(title_rtn != NULL)
		*title_rtn = title;
	else
		free(title);
	if(author_rtn != NULL)
		*author_rtn = author;
	else
		free(author);
	if(comments_rtn != NULL)
		*comments_rtn = comments;
	else
		free(comments);

	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = modified_time_sec;

	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted && (status == 0))
	{
		const int nframes = *nframes_rtn;
		if(!progress_cb(
			progress_data,
			height, height,
			width, height,
			bpl, bpp,
			(nframes > 0) ?
				(*rgba_list_rtn)[nframes - 1] : NULL
		))
			user_aborted = 1;
	}

	if(user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		errno = EINTR;
		return(-4);
	}
	else
	{
		errno = error_code;
		return(status);
	}
}

/*
 *	Opens the image file to RGBA image data as a thumb.
 *
 *	The req_width and req_height specifies the desired thumb size.
 *	Not all image formats may be opened as a thumb and as such,
 *	the returned image data may still be larger than the requested
 *	thumb size.
 */
int ImgFileOpenRGBAThumb(
	const char *path,
	int req_width, int req_height,		/* Requested thumb size */
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes in RGBA format, will be modified */
	int *orig_width_rtn, int *orig_height_rtn,
	int *orig_nframes_rtn,
	unsigned long *play_time_ms_rtn,
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	unsigned long *modified_time_sec_rtn,	/* In seconds since EPOCH GMT */
	const u_int8_t def_alpha_value,
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
					user_aborted = 0,
					width = 0, height = 0,
					bpl = 0,
					orig_width = 0,
					orig_height = 0,
					orig_nframes = 0,
					x = 0, y = 0,
					base_width = 0, base_height = 0;
	unsigned long	play_time_ms = 0l,
					modified_time_sec = 0l;
	const char *ext;
	char		*creator = NULL,
					*title = NULL,
					*author = NULL,
					*comments = NULL;
	ImgIOType io_type;

	/* Reset the global open error message pointer */
	imgio_last_open_error = NULL;

	if((width_rtn == NULL) || (height_rtn == NULL) ||
	   (bpl_rtn == NULL) || (rgba_rtn == NULL)
	)
	{
		imgio_last_open_error =
			"Addresses for image return values not available";
 	    return(-2);
	}

	/* Reset the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;
	*rgba_rtn = NULL;
	if(orig_width_rtn != NULL)
		*orig_width_rtn = orig_width;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = orig_height;
	if(orig_nframes_rtn != NULL)
		*orig_nframes_rtn = orig_nframes;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = play_time_ms;
	if(x_rtn != NULL)
		*x_rtn = x;
	if(y_rtn != NULL)
		*y_rtn = y;
	if(base_width_rtn != NULL)
		*base_width_rtn = base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = base_height;
	if(creator_rtn != NULL)
		*creator_rtn = creator;
	if(title_rtn != NULL)
		*title_rtn = title;
	if(author_rtn != NULL)
		*author_rtn = author;
	if(comments_rtn != NULL)
		*comments_rtn = comments;
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = modified_time_sec;

	if(STRISEMPTY(path))
	{
		imgio_last_open_error = "The file name was not specified";
		return(-2);
	}

	/* Get the extension */
	ext = ImgGetExtensionFromPath(path);
	if(ext == NULL)
		ext = "";

	/* Determine the IO type based on the extension */
	io_type = ImgGetIOTypeFromExtension(ext);

	/* Reset the status to -1 and open by the IO type */
	status = -1;

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			progress_data,
			0, height,
			width, height,
			bpl, bpp,
			*rgba_rtn
		))
			user_aborted = 1;
	}

	/* Begin loading the image by the selected image IO type */
	switch(io_type)
	{
	  case IMG_IO_TYPE_ANY:
	  case IMG_IO_TYPE_IMLIB:		/* Imlib or Imlib2 */
	  case IMG_IO_TYPE_PS:
	  case IMG_IO_TYPE_TGA:
	  case IMG_IO_TYPE_XPM:
#if defined(HAVE_IMLIB)
		status = ImgFileOpenImlibRGBA(
			path,
			&width, &height,
			&bpl,
			rgba_rtn,
			progress_cb, progress_data,
			&user_aborted
		);
#elif defined(HAVE_IMLIB2)
		status = ImgFileOpenImlib2RGBA(
			path,
			&width, &height,
			&bpl,
			rgba_rtn,
			progress_cb, progress_data,
			&user_aborted
		);
#endif
		break;

	  case IMG_IO_TYPE_GIF:
#ifdef HAVE_LIBGIF
		status = ImgFileOpenGIFRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			rgba_rtn,
			bg_color,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title,
			&author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted 
		);
#endif
		break;

	  case IMG_IO_TYPE_JPEG:
#ifdef HAVE_LIBJPEG
		status = ImgFileOpenJPEGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			rgba_rtn,
			&orig_width, &orig_height,
			progress_cb, progress_data,
			&user_aborted 
		);
		orig_nframes = 1;
		base_width = width;
		base_height = height;
#endif
		break;

	  case IMG_IO_TYPE_MNG:
#ifdef HAVE_LIBMNG
		status = ImgFileOpenMNGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			rgba_rtn,
			bg_color,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title,
			&author, &comments,
			&modified_time_sec,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted 
		);
#endif
		break;

	  case IMG_IO_TYPE_PNG:
#ifdef HAVE_LIBPNG
		status = ImgFileOpenPNGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			rgba_rtn,
			bg_color,
			&orig_width, &orig_height,
			&x, &y,
			&base_width, &base_height,
			&creator, &title,
			&author, &comments,
			&modified_time_sec,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted 
		);
		orig_nframes = 1;
#endif

	  case IMG_IO_TYPE_TIFF:
#ifdef HAVE_LIBTIFF
		status = ImgFileOpenTIFFRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			rgba_rtn,
			bg_color,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title,
			&author, &comments,
			def_alpha_value,
			progress_cb, progress_data,
			&user_aborted 
		);
#endif
		break;
	}

	/* Set the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;

	if(orig_width_rtn != NULL)
		*orig_width_rtn = orig_width;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = orig_height;
	if(orig_nframes_rtn != NULL)
		*orig_nframes_rtn = orig_nframes;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = play_time_ms;

	if(x_rtn != NULL)
		*x_rtn = x;
	if(y_rtn != NULL)
		*y_rtn = y;
	if(base_width_rtn != NULL)
		*base_width_rtn = base_width;
	if(base_height_rtn != NULL)
		*base_height_rtn = base_height;

	if(creator_rtn != NULL)
		*creator_rtn = creator;
	else
		free(creator);
	if(title_rtn != NULL)
		*title_rtn = title;
	else
		free(title);
	if(author_rtn != NULL)
		*author_rtn = author;
	else
		free(author);
	if(comments_rtn != NULL)
		*comments_rtn = comments;
	else
		free(comments);

	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = modified_time_sec;

	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
		if(!progress_cb(
			progress_data,
			height, height,
			width, height,
			bpl, bpp,
			*rgba_rtn
		))
			user_aborted = 1;
	}

	if(user_aborted)
	{
		imgio_last_open_error = "User aborted operation";
		return(-4);
	}
	else
		return(status);
}


#ifdef HAVE_IMLIB
/*
 *	Sets the Imlib handle.
 *
 *	The p specifies the Imlib handle, if p is NULL then the Imlib
 *	handle is unset.
 *
 *	This function must be called first before any other function
 *	that uses Imlib.
 *
 *	This function always returns 0.
 */
int ImgImlibSetHandle(void *p)
{
	img_imlib_handle = p;
	return(0);
}

/*
 *	Converts the Imlib RGB and Alpha image data to RGBA.
 */
u_int8_t *ImgImlibConvertImlibToRGBA(
	const u_int8_t *rgb,
	const u_int8_t *alpha,
	const int width, const int height,
	const int rgb_bpl,
	const int alpha_bpl,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
) 
{
	int		rgba_bpp = 4,
					rgba_bpl,
					_rgb_bpl = rgb_bpl,
					_alpha_bpl = alpha_bpl;
	u_int8_t *rgba;

	if((rgb == NULL) || (width <= 0) || (height <= 0) ||
	   *user_aborted
	)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Calculate bytes per line for all buffers */
	if(_rgb_bpl <= 0)
		_rgb_bpl = width * 3;
	if(_alpha_bpl <= 0)
		_alpha_bpl = width * 1;
	rgba_bpl = width * rgba_bpp;

	/* Allocate the target RGBA image data */
	if(progress_cb != NULL)
		rgba = (u_int8_t *)calloc(
			rgba_bpl * height,
			sizeof(u_int8_t)
		);
	else
		rgba = (u_int8_t *)malloc(rgba_bpl * height * sizeof(u_int8_t));
	if(rgba == NULL)
		return(NULL);

	/* Alpha channel available? */
	if(alpha != NULL)
	{
		u_int8_t	*rgba_line,
					*rgba_line_end,
					*rgba_ptr,
					*rgba_end;
		const u_int8_t	*rgb_line,
							*rgb_ptr,
							*alpha_line,
							*alpha_ptr;

		/* Iterate through each line */
		for(rgba_line       = rgba,
			rgba_line_end   = rgba_line + (rgba_bpl * height),
			rgb_line        = rgb,
			alpha_line      = alpha;
			rgba_line < rgba_line_end;
			rgba_line       += rgba_bpl,
			rgb_line        += _rgb_bpl,
			alpha_line      += _alpha_bpl
		)
		{
			rgba_ptr        = rgba_line;
			rgba_end        = rgba_ptr + (width * 4);
			rgb_ptr         = rgb_line;
			alpha_ptr       = alpha_line;

			/* Call progress callback */
			if(progress_cb != NULL)
			{
				const int i = (int)(rgba_line - rgba);
				if((i % IMG_IO_PROGRESS_RESOLUTION) == 0)
				{
					if(!progress_cb(
						progress_data,
						i, (int)(rgba_line_end - rgba),
						width, height,
						rgba_bpl, rgba_bpp,
						rgba
					))
					{
						*user_aborted = 1;
						break;
					}
				}
			}

			/* Iterate through current line */
			while(rgba_ptr < rgba_end)
			{
				*rgba_ptr++ = *rgb_ptr++;
				*rgba_ptr++ = *rgb_ptr++;
				*rgba_ptr++ = *rgb_ptr++;
				*rgba_ptr++ = *alpha_ptr++;
			}
		}
	}
	else
	{
		u_int8_t	*rgba_line,
					*rgba_line_end,
					*rgba_ptr,
					*rgba_end;
		const u_int8_t	*rgb_line,
							*rgb_ptr;

		/* Iterate through each line */
		for(rgba_line       = rgba,
			rgba_line_end   = rgba_line + (rgba_bpl * height),
			rgb_line        = rgb;
			rgba_line < rgba_line_end;
			rgba_line       += rgba_bpl,
			rgb_line        += _rgb_bpl
		)
		{
			rgba_ptr        = rgba_line;
			rgba_end        = rgba_ptr + (width * 4);
			rgb_ptr         = rgb_line;

			/* Call progress callback */
			if(progress_cb != NULL)
			{
				const int i = (int)(rgba_line - rgba);
				if((i % IMG_IO_PROGRESS_RESOLUTION) == 0)
				{
				    if(!progress_cb(
						progress_data,
						i, (int)(rgba_line_end - rgba),
						width, height,
						rgba_bpl, rgba_bpp,
						rgba
					))
					{
						*user_aborted = 1;
						break;
					}
				}
			}

			/* Iterate through current line */
			while(rgba_ptr < rgba_end)
			{
				*rgba_ptr++ = rgb_ptr[0];
				*rgba_ptr++ = rgb_ptr[1];
				*rgba_ptr++ = rgb_ptr[2];
#ifdef USE_IMLIB_TRANSPIXEL_FIX
				*rgba_ptr++ = (
					(rgb_ptr[0] == 0xff) &&
					(rgb_ptr[1] == 0x00) &&
					(rgb_ptr[2] == 0xff)
				) ?
					0x00 : 0xff;
#else
				*rgba_ptr++ = 0xff;
#endif
				rgb_ptr += 3;
			}
		}
	}

	return(rgba);
}

/*
 *	Opens the image file using Imlib to RGBA image data.
 */
int ImgFileOpenImlibRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		width = 0, height = 0,
					bpl = 0;
	char *dpath;
	FILE *old_stderr;
	ImlibImage *imlib_image;
	u_int8_t *rgba = NULL;

	if(img_imlib_handle == NULL)
	{
		imgio_last_open_error = "Imlib was not initialized, did you forget to call ImgImlibSetHandle()?";
		return(-1);
	}

	/* Since Imlib prints error messages to stderr, we need to
	 * squelch them by temporarly redirecting stderr to
	 * /dev/null
	 */
	old_stderr = stderr;
	stderr = fopen("/dev/null", "wb");

	/* Open the image */
	dpath = strdup(path);
	imlib_image = Imlib_load_image(
		img_imlib_handle,
		dpath
	);
	free(dpath);

	/* Close /dev/null and restore stderr */
	if(stderr != NULL)
		(void)fclose(stderr);
	stderr = old_stderr;

	/* Image opened successfully? */
	if(imlib_image != NULL)
	{
		/* Need to realize changes */
		Imlib_changed_image(img_imlib_handle, imlib_image);

		/* Update the image information */
		width = imlib_image->rgb_width;
		height = imlib_image->rgb_height;
		bpl = width * bpp;

		/* Convert the Imlib image data to RGBA image data */
		rgba = ImgImlibConvertImlibToRGBA(
			(const u_int8_t *)imlib_image->rgb_data,
			(const u_int8_t *)imlib_image->alpha_data,
			width, height,
			-1, -1,
			progress_cb, progress_data,
			user_aborted
		);

		/* Unref the Imlib image, it is no longer needed */
		Imlib_destroy_image(img_imlib_handle, imlib_image);
	}

	/* Set the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;
	*rgba_rtn = rgba;

	if(rgba != NULL)
	{
		if(*user_aborted)
			return(-4);
		else
			return(0);
	}
	else
	{
		return(-1);
	}
}

/*
 *	Saves the RGBA image data using Imlib.
 */
int ImgFileSaveImlibRGBA(
	const char *path,
	const int width, const int height,
	const int bpl,
	const u_int8_t *rgba,
	const u_int8_t *bg_color,		/* 4 bytes in RGBA format */
	const char *comments,
	const float quality,			/* 0.0 to 1.0 */
	const int color_type,			/* 1 = color
											 * 0 = greyscale */
	const u_int8_t def_alpha_value,
	const int imlib_fallback,		/* 0 = none
											 * 1 = ImageMajick & NetPBM */
	ImgProgressFunc progress_cb, void *progress_data
)
{
	const int	bpp = 4;		/* RGBA */
	int		status,
					user_aborted = 0,
					_bpl = bpl;
	const char *ext;
	ImgIOType io_type = IMG_IO_TYPE_ANY;

	/* Reset the global last save error message pointer */
	imgio_last_save_error = NULL;

	if((rgba == NULL) || (width <= 0) || (height <= 0))
	{
		imgio_last_save_error = "Invalid value";
		return(-2);
	}
	if(STRISEMPTY(path))
	{
		imgio_last_save_error = "The file name was not specified";
		return(-2);
	}

	/* Automatically calculate bytes per line? */
	if(_bpl <= 0)
		_bpl = width * bpp;

	/* Get the extension */
	ext = ImgGetExtensionFromPath(path);
	if(ext == NULL)
		ext = "";

	/* Must have extension to determine format */
	if(STRISEMPTY(ext))
	{
		imgio_last_save_error = "Unable to determine format due to missing extension";
		return(-2);
	}

	/* Explicitly set io type to use Imlib or Imlib2 */
	io_type = IMG_IO_TYPE_IMLIB;


	/* Reset status to -1 and load by IO type */
	status = -1;

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			progress_data,
			0, 1,
			width, height,
			_bpl, bpp,
			rgba
		))
			user_aborted = 1;
	}

	/* Use Imlib or Imlib2? */
	if((io_type == IMG_IO_TYPE_IMLIB) && !user_aborted)
	{
#if defined(HAVE_IMLIB)
		if(img_imlib_handle != NULL)
		{
			const int	sbpp = bpp,
							sbpl = _bpl;
			int	x, y,
					bc,
					min_rgb_bpp,
					tbpl, tbpp;
			u_int8_t *tar_ptr, *tar_data_rgb, *tar_data_alpha;
			const u_int8_t	*src_data = rgba,
							*src_ptr;
			ImlibImage *imlib_image = NULL;
			ImlibSaveInfo *si;

			/* Allocate target RGB and alpha buffers */
			tbpp = 3;			/* RGB */
			tbpl = width * tbpp;
			tar_data_rgb = (u_int8_t *)malloc(tbpl * height);
			tar_data_alpha = (u_int8_t *)malloc(width * height);
			if((tar_data_rgb != NULL) && (tar_data_alpha != NULL))
			{
				/* Calculate smaller of rgb bytes per pixel */
				min_rgb_bpp = MIN(sbpp, tbpp);

				/* Copy source image to our target buffers */
				for(y = 0; y < height; y++)
				{
					if(progress_cb != NULL)
					{
						if(!progress_cb(
							progress_data,
							y, height,
							width, height,
							_bpl, bpp,
							rgba
						))
						{
							user_aborted = 1;
							break;
						}
					}

					for(x = 0; x < width; x++)
					{
						src_ptr = src_data + (y * sbpl) + (x * sbpp);

						/* Copy RGB */
						tar_ptr = tar_data_rgb + (y * tbpl) + (x * tbpp);
						for(bc = 0; bc < min_rgb_bpp; bc++)
							*tar_ptr++ = *src_ptr++;

						/* Copy alpha */
						tar_ptr = &tar_data_alpha[
							(y * width) + (x * 1)
						];
						/* More bytes left on source pixel pointer? */
						if(bc < sbpp)
						{
							*tar_ptr = *src_ptr++;
							bc++;
						}
						else
						{
							*tar_ptr = def_alpha_value;
						}
					}
				}	/* Copy source image to our target buffers */


				/* Create Imlib image from loaded data only if
				 * user did not abort
				 */
				if(!user_aborted)
					imlib_image = Imlib_create_image_from_data(
						img_imlib_handle,
						(unsigned char *)tar_data_rgb,
						(unsigned char *)tar_data_alpha,
						width, height
					);
				if(imlib_image != NULL)
				{
					/* Need to realize changes */
					Imlib_changed_image(img_imlib_handle, imlib_image);
				}
			}
			/* Delete target buffers used to create the Imlib
			 * image
			 */
			free(tar_data_rgb);
			tar_data_rgb = NULL;
			free(tar_data_alpha);
			tar_data_alpha = NULL;


			/* Allocate and set up save info */
			si = (ImlibSaveInfo *)calloc(1, sizeof(ImlibSaveInfo));
			if(si != NULL)
			{
				/* Quality for lossy formats (ie jpg), valid values
				 * from 0 = 0% to 256 = 100%.
				 */
				si->quality = CLIP(quality, 0.0f, 1.0f) * 256;

				/* Scaling for paper relative formats, valid values
				 * from 0 = 0% to 1024 = 100%.
				 */
				si->scaling = 1024;

				/* Bottom-left relative paper justification, valid
				 * values from 0 to 1024.
				 */
				si->xjustification = 0;
				si->yjustification = 1024;

				si->page_size = PAGE_SIZE_LETTER;

				/* Color type:
				 *
				 * 0 = Greyscale
				 * 1 = Color
				 */
				si->color = color_type;
			}

			/* Set Imlib fallback:
			 *	0 = No fallback
			 *	1 = Use ImageMagick and NetPBM.
			 */
			Imlib_set_fallback(img_imlib_handle, imlib_fallback);

			/* Was the Imlib image created successfully? */
			if(imlib_image != NULL)
			{
				char *dpath = STRDUP(path);
				/* Save Imlib image */
				status = Imlib_save_image(
					img_imlib_handle,
					imlib_image,
					dpath,
					si
				);
				free(dpath);

				/* Save successful? */
				if(status == 1)
				{
					/* Update status to our interpritation of success */
					status = 0;
				}
				else
				{
					/* Update status to our interpritation of error */
					status = -1;
				}

				/* Destroy Imlib image, it is no longer needed in
				 * the cache.
				 */
				Imlib_kill_image(img_imlib_handle, imlib_image);
				imlib_image = NULL;
			}
			/* At this point imlib_image has been deleted */

			/* Delete save info and any allocated members */
			free(si);
			si = NULL;
		}
#elif defined(HAVE_IMLIB2)






#endif
	}

	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
		if(!progress_cb(
			progress_data,
			height, height,
			width, height,
			_bpl, bpp,
			rgba
		))
			user_aborted = 1;
	}


	if(user_aborted)
	{
		imgio_last_save_error = "User aborted operation";
		return(-4);
	}
	else
	{
		return(status);
	}
}
#endif	/* HAVE_IMLIB */


#ifdef HAVE_IMLIB2
/*
 *	Imlib2 progress callback.
 */
static int ImgImlib2ProgressCB(
	Imlib_Image im, char percent,
	int update_x, int update_y,
	int update_w, int update_h
)
{
	if(img_imlib2_progress_cb != NULL)
		return(img_imlib2_progress_cb(
			img_imlib2_progress_data,
			percent,			/* [0, 100], so report 0% to
											 * 50% */
			200,				/* Limit to 50% */
			0, 0,
			0, 0,
			NULL
		));
	else
		return(0);
}

/*
 *	Opens the image file to RGBA image data using Imlib2.
 */
int ImgFileOpenImlib2RGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	ImgProgressFunc progress_cb, void *progress_data,
	int *user_aborted
)
{
	const int	bpp = 4;		/* RGBA */
	int		width = 0, height = 0,
					bpl = 0;
	u_int8_t *rgba = NULL;
	Imlib_Image *imlib_image;

	/* Set the pointer to the progress callback */
	img_imlib2_progress_cb = progress_cb;
	img_imlib2_progress_data = progress_data;
	imlib_context_set_progress_function(ImgImlib2ProgressCB);

	/* Load the Imlib image using Imlib2
	 *
	 * This places the image in cache if it was not already
	 * loaded or just increases the refcount for an image
	 * already loaded in the cache
	 */
	imlib_image = imlib_load_image(path);
	if(imlib_image != NULL)
	{
		const u_int8_t *src_data_rgba;

		/* Set loaded image into Imlib2's context */
		imlib_context_set_image(imlib_image);

		/* Get size of image */
		width = imlib_image_get_width();
		height = imlib_image_get_height();
		bpl = width * bpp;

		/* Get pointer to Imlib image's RGBA data */
		src_data_rgba = (const u_int8_t *)imlib_image_get_data_for_reading_only();

		/* Allocate the target RGBA image data */
		if(progress_cb != NULL)
			rgba = (u_int8_t *)calloc(
				bpl * height,
				sizeof(u_int8_t)
			);
		else
			rgba = (u_int8_t *)malloc(bpl * height * sizeof(u_int8_t));

		/* Source & target values valid? */
		if((src_data_rgba != NULL) && (rgba != NULL) &&
		   (width > 0) && (height > 0)
		)
		{
			const int tar_len = bpl * height;
			u_int8_t	*tar = rgba,
							*tar_end = tar + tar_len;
			const u_int8_t	*src = src_data_rgba;

			/* Copy/convert source image data to the target
			 * image data
			 */
			while(tar < tar_end)
			{
				/* Convert source BGRA pixel to target RGBA
				 * pixel
				 */
				*tar++ = src[2];
				*tar++ = src[1];
				*tar++ = src[0];
				*tar++ = src[3];
				src += bpp;

				/* Report progress */
				if(progress_cb != NULL)
				{
					const int i = (int)((tar - rgba) + tar_len);
					if((i % IMG_IO_PROGRESS_RESOLUTION) == 0)
					{
						if(!progress_cb(
						    progress_data,
							i, 2 * tar_len,
							width, height,
							bpl, bpp,
							rgba
						))
						{
							*user_aborted = 1;
							break;
						}
					}
				}
			}
		}

		/* Always delete Imlib2 images after rendering, this
		 * is efficient because the image* is cached and
		 * managed by Imlib2
		 *
		 * This will reduce the refcount of the image,
		 * when it is no longer needed then Imlib2 will
		 * actually delete it
		 */
		imlib_free_image();
	}

	/* Set the return values */
	*width_rtn = width;
	*height_rtn = height;
	*bpl_rtn = bpl;
	*rgba_rtn = rgba;

	if(rgba != NULL)
	{
		if(*user_aborted)
			return(-4);
		else
			return(0);
	}
	else
	{
		imgio_last_open_error =
			"Imlib2 was unable to open the image";
		return(-1);
	}
}
#endif	/* HAVE_IMLIB2 */
