#include <string.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "imgview.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_property_image.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_image_io.h"
#include "edv_date_format.h"
#include "edv_status_bar.h"
#include "image_browser.h"
#include "image_browser_view.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVImageBrowserViewProgressData	\
					EDVImageBrowserViewProgressData;
#define EDV_IMAGE_BROWSER_VIEW_PROGRESS_DATA(p)	\
					((EDVImageBrowserViewProgressData *)(p))


/* Progress Callback */
static gint edv_image_browser_view_progress_cb(
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const guint8 *image_data,
	const gulong i, const gulong m,
	gpointer data
);

void edv_image_browser_view_load(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gint thumb_num
);
void edv_image_browser_view_clear(EDVImageBrowser *imbr);


/*
 *	Progress Data;
 */
struct _EDVImageBrowserViewProgressData {

	EDVImageBrowser	*image_browser;

	imgview_image_struct	*img;		/* Current imgview image
						 * (NULL means not set yet) */
	u_int8_t	bg_color[4],
			last_bg_color[4];
	gboolean	override_image_bg_color;

	gchar		*name;
	gulong		file_size;

	gfloat		last_progress_coeff;
};


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Progress callback.
 */
static gint edv_image_browser_view_progress_cb(
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const guint8 *image_data,
	const gulong i, const gulong m,
	gpointer data
)
{
	gboolean image_changed;
	gfloat		progress_coeff,
					progress_coeff_change;
	const guint8 *rgba;
	imgview_struct *iv;
	imgview_image_struct *img;
	EDVImageBrowser *imbr;
	EDVImageBrowserViewProgressData *d = EDV_IMAGE_BROWSER_VIEW_PROGRESS_DATA(data);
	if(d == NULL)
		return(-2);

	if(bpp == 4)
		rgba = image_data;
	else
		rgba = NULL;

	imbr = d->image_browser;
	iv = imbr->imgview;

	if(imbr->stop_count > 0)
		return(-4);

	/* Calculate the progress coefficient, use -1.0 if it is
	 * not known
	 */
	if(m > 0l)
		progress_coeff = (gfloat)i / (gfloat)m;
	else
		progress_coeff = -1.0f;

	if((d->last_progress_coeff >= 0.0f) && (progress_coeff >= 0.0f))
		progress_coeff_change = progress_coeff - d->last_progress_coeff;
	else
		progress_coeff_change = 0.0f;

	img = ImgViewGetImage(iv);

/* Update the ImgView's background color if it has changed */
#define UPDATE_BG_COLOR	{		\
 if(!d->override_image_bg_color &&	\
    memcmp(d->last_bg_color, d->bg_color, sizeof(d->last_bg_color)) \
 ) {					\
  gint i;				\
  GdkColor c[5];			\
  for(i = 0; i < 5; i++) {		\
   GDK_COLOR_SET_BYTE(			\
    &(c[i]),				\
    d->bg_color[0],			\
    d->bg_color[1],			\
    d->bg_color[2]			\
   );					\
  }					\
  ImgViewSetViewBG(iv, c);		\
									\
  /* Record the new background color */	\
  (void)memcpy(				\
   d->last_bg_color,			\
   d->bg_color,				\
   sizeof(d->last_bg_color)		\
  );					\
 }					\
}


	/* If the ImgView image has not yet been set for this progress
	 * or the RGBA image data geometry has changed then set
	 * image_changed to TRUE
	 *
	 * ImgView image set for this progress?
	 */
	if(d->img != NULL)
	{
		if((img != NULL) && (rgba != NULL))
		{
			if((img->width != width) || (img->height != height))
				image_changed = TRUE;
			else
				image_changed = FALSE;
		}
		else
		{
			image_changed = FALSE;
		}
	}
	else
	{
		if(rgba != NULL)
			image_changed = TRUE;
		else
			image_changed = FALSE;
	}
	if(image_changed)
	{
		/* The RGBA image has specified for the first time or its
		 * geometry has changed, recreate the ImgView image and
		 * copy the specified RGBA image to it
		 */
		gchar *msg;
		gboolean need_zoom_to_fit = FALSE;

		/* Get the ImgView's view GtkWidget */
		GtkWidget *w = (GtkWidget *)ImgViewGetViewWidget(iv);
		if(w != NULL)
		{
			GtkAllocation *a = &w->allocation;
			if((width > a->width) ||
			   (height > a->height)
			)
				need_zoom_to_fit = TRUE;
		}

		/* Update the status message */
		msg = g_strdup_printf(
			"Displaying image \"%s\" (%ix%i %s %s)...",
			d->name,
			width, height,
			edv_str_size_delim(d->file_size),
			(d->file_size == 1l) ? "byte" : "bytes"
		);
		edv_status_bar_message(imbr->status_bar, msg, FALSE);
		g_free(msg);

		/* Create a new ImgView image */
		d->img = img = ImgViewImageNew(
			width, height,
			bpp, bpl
		);
		if(img != NULL)
		{
			/* Copy the specified RGBA image data to the new
			 * ImgView image
			 */
			ImgViewImageAppendFrame(
				img,
				g_memdup(
					rgba,
					bpl * height * sizeof(guint8)
				),
				0l				/* No delay */
			);

			/* Set/transfer the ImgView image to the ImgView */
			if(need_zoom_to_fit)
				ImgViewSetImageToFit(iv, img);
			else
				ImgViewSetImage(iv, img);

			/* Update the background color if it has changed */
			UPDATE_BG_COLOR
		}
	}
	else if(img != NULL)
	{
		const gfloat must_redraw_every_coeff_change = 0.2f;

		/* If the specified image geometry matches the current
		 * ImgView image's geometry then update/copy the
		 * specified image data to the ImgView's image
		 */
		if((progress_coeff_change >= must_redraw_every_coeff_change) &&
		   (rgba != NULL) && (img->width == width) &&
		   (img->height == height) && (img->bpp == bpp) &&
		   (img->bpl == bpl)
		)
		{
			/* Get the last frame of the ImgView's image */
			imgview_frame_struct *frame = ImgViewImageGetFrame(
				img,
				img->nframes - 1
			);
			if(frame != NULL)
			{
				/* Copy the specified RGBA image data to this frame
				 * and tell the ImgView to redraw
				 */
				(void)memcpy(
					frame->data,
					rgba,
					img->bpl * img->height * sizeof(guint8)
				);

				/* Update the background color if it has changed */
				UPDATE_BG_COLOR

				/* Queue the ImgView to redraw, which may not
				 * occure immediately if the CPU usage is too
				 * intense
				 */
				ImgViewQueueDrawView(iv);
			}
			d->last_progress_coeff = progress_coeff;
		}
	}
#undef UPDATE_BG_COLOR

	/* Update the progress bar */
	if(progress_coeff > 0.0f)
		edv_status_bar_progress(
			imbr->status_bar,
			progress_coeff,
			TRUE
		);

	return(0);
}


/*
 *	Loads an image to the ImgView.
 *
 *	The path specifies the path to the image file to be opened and
 *	displayed on the ImgView.
 *
 *	If thumb_num is not -1 then thumb_num specifies the thumb on
 *	the Thumbs List to be updated with the opened image file.
 *
 *	This function will automatically call
 *	edv_image_browser_view_clear() to clear the ImgView first.
 */
void edv_image_browser_view_load(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gint thumb_num
)
{
	gboolean	need_zoom_to_fit,
			image_geometries_match;
	const int	bpp = 4;		/* RGBA */
	gint		width, height,
			bpl, nframes;
	gulong		file_size = 0l,
			play_time_ms,
			modified_time_sec;
	gchar		*creator,
			*title,
			*author,
			*comments;
	const gchar	*name,
			*ext;
	GList		*rgba_list,
			*delay_list;
	GtkWidget	*w,
			*status_bar;
	CfgList *cfg_list;
	CfgColor *cfg_bg_color;
	imgview_image_struct *img;
	imgview_struct *iv;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVImageBrowserViewProgressData *d;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;
	status_bar = imbr->status_bar;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Clear the ImgView */
	edv_image_browser_view_clear(imbr);

	if(STRISEMPTY(path))
		return;

	name = g_basename(path);
	if(name != NULL)
		ext = edv_name_get_extension(name);
	else
		ext = edv_path_get_extension(path);

	/* Is the file's extension is not a supported image? */
	if(!edv_image_is_supported_extension(ext))
		return;

/* Note that passive loading will not work since calls to this function
 * are set busy and processing so that stopping is not available
 */
	edv_image_browser_set_passive_busy(imbr, TRUE);
	edv_status_bar_progress(status_bar, 0.0f, TRUE);
	imbr->stop_count = 0;

	obj = edv_vfs_object_stat(path);
	if(obj != NULL)
	{
		gchar *msg;

		file_size = obj->size;

		msg = g_strdup_printf(
			"Displaying image \"%s\" (%s %s)...",
			name,
			edv_str_size_delim(file_size),
			(file_size == 1l) ? "byte" : "bytes"
		);
		edv_status_bar_message(status_bar, msg, FALSE);
		g_free(msg);

		edv_vfs_object_delete(obj);
	}

	/* Allocate and set the progress callback data */
	d = EDV_IMAGE_BROWSER_VIEW_PROGRESS_DATA(g_malloc0(sizeof(EDVImageBrowserViewProgressData)));
	if(d == NULL)
	{
		edv_image_browser_set_passive_busy(imbr, FALSE);
		edv_status_bar_progress(status_bar, 0.0f, FALSE);
		return;
	}

	d->image_browser = imbr;
	d->name = g_strdup(name);
	d->file_size = file_size;

	/* Get the default background color from the configuration */
	cfg_bg_color = EDV_GET_COLOR(EDV_CFG_PARM_IMBR_BG_COLOR);
	if(cfg_bg_color != NULL)
	{
		d->bg_color[0] = (guint8)(0xFF * cfg_bg_color->r);
		d->bg_color[1] = (guint8)(0xFF * cfg_bg_color->g);
		d->bg_color[2] = (guint8)(0xFF * cfg_bg_color->b);
		d->bg_color[3] = 0xFF;
	}
	else
	{
		d->bg_color[0] = 0xFF;
		d->bg_color[1] = 0xFF;
		d->bg_color[2] = 0xFF;
		d->bg_color[3] = 0xFF;
	}

	/* Record the background color */
	(void)memcpy(
		d->last_bg_color,
		d->bg_color,
		sizeof(d->last_bg_color)
	);

	d->override_image_bg_color = EDV_GET_B(EDV_CFG_PARM_IMBR_OVERRIDE_IMAGE_BG_COLOR);

	/* Open the image in RGBA */
	rgba_list = edv_image_open_rgba_details(
		path,
		&width, &height,
		&bpl,
		&nframes,
		&delay_list,
		d->bg_color,
		&play_time_ms,
		&creator, &title, &author, &comments,
		&modified_time_sec,
		edv_image_browser_view_progress_cb, d
	);

	/* Failed to open the image? */
	if(rgba_list == NULL)
	{
		g_free(d->name);
		g_free(d);
		g_list_free(delay_list);
		g_free(creator);
		g_free(title);
		g_free(author);
		g_free(comments);
		edv_status_bar_progress(status_bar, 0.0f, FALSE);
		edv_image_browser_set_passive_busy(imbr, FALSE);
		return;
	}

	/* Update the Thumbs List thumb? */
	if(thumb_num > -1)
	{
		const gboolean no_enlarge = EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_NO_ENLARGE);
		const guint8 *rgba = (const guint8 *)g_list_nth_data(
			rgba_list,
			0
		);
		tlist_struct *tlist = imbr->tlist;
		EDVVFSObject *obj = EDV_VFS_OBJECT(TListGetThumbData(
			tlist,
			thumb_num
		));
		TListSetRGBA(
			tlist,
			thumb_num,
			width, height, bpl,
			GDK_RGB_DITHER_NORMAL,
			rgba,			/* First frame image data */
			no_enlarge
		);
		TListSetLoadState(
			tlist,
			thumb_num,
			TLIST_LOAD_STATE_LOADED
		);

		/* Update the image properties */
		if(obj != NULL)
		{
			gchar *modified_time_sec_str;
			if(modified_time_sec > 0l)
			{
				modified_time_sec_str = edv_date_string_format(
					modified_time_sec,
					EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT),
					EDV_GET_I(EDV_CFG_PARM_DATE_RELATIVITY)
				);
			}
			else
			{
				modified_time_sec_str = NULL;
			}
			obj->meta_data_list = edv_properties_list_image_set(
				obj->meta_data_list,
				width, height,
				nframes,
				play_time_ms,
				creator, author, title, comments,
				modified_time_sec_str
			);
			g_free(modified_time_sec_str);
		}
	}

	/* Set the background color */
	if(!d->override_image_bg_color &&
	   memcmp(d->last_bg_color, d->bg_color, sizeof(d->last_bg_color))
	)
	{
		GdkColor gdk_color[5];
		const gint m = sizeof(gdk_color) / sizeof(GdkColor);
		gint i;
		for(i = 0; i < m; i++)
		{
			GDK_COLOR_SET_BYTE(
				&gdk_color[i],
				d->bg_color[0],
				d->bg_color[1],
				d->bg_color[2]
			);
		}
		ImgViewSetViewBG(iv, gdk_color);
 	}

	/* Check if the image size is larger than the current size of
	 * the ImgView's view GtkWidget, in which case we will need
	 * to tell the ImgView to zoom to the new image
	 */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w != NULL)
	{
		const GtkAllocation *a = &w->allocation;
		if((width > a->width) ||
		   (height > a->height)
		)
			need_zoom_to_fit = TRUE;
		else
			need_zoom_to_fit = FALSE;
	}
	else
	{
		need_zoom_to_fit = FALSE;
	}

	/* Get the ImgView's current image and check if its geometry
	 * matches the opened image's geometry
	 *
	 * If and only if the geometries are the same then it
	 * implies the ImgView's image was created during the
	 * progress callback and we can simply update the current
	 * ImgView's image without having to recreate it
	 */
	img = ImgViewGetImage(iv);
	if(img != NULL)
	{
		if((img->width == width) && (img->height == height) &&
		   (img->bpp == bpp) && (img->bpl == bpl)
		)
			image_geometries_match = TRUE;
		else
			image_geometries_match = FALSE;
	}
	else
	{
		image_geometries_match = FALSE;
	}
	if(image_geometries_match)
	{
		/* Image geometries match, just update the ImgView's
		 * image
		 */
		img = ImgViewGetImage(iv);
		if(img != NULL)
		{
			const int len = bpl * height;
			GList	*glist,
				*glist_rgba = rgba_list,
				*glist_delay = delay_list;
			imgview_frame_struct *frame;

			/* Copy the frames to the existing frames on the
			 * ImgView's image
			 */
			for(glist = img->frames_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				if(glist_rgba == NULL)
					break;

				/* Get this frame and copy the opened image's
				 * frame data and delay over to it
				 */
				frame = IMGVIEW_FRAME(glist->data);
				if(frame != NULL)
				{
					if(frame->data != NULL)
						(void)memcpy(
							frame->data,
							glist_rgba->data,
							len * sizeof(guint8)
						);
					frame->delay = (gulong)((glist_delay != NULL) ?
						(guint)glist_delay->data : 0);
				}

				glist_rgba = g_list_next(glist_rgba);
				if(glist_delay != NULL)
					glist_delay = g_list_next(glist_delay);
			}

			/* Append/transfer the rest of the frames */
			while(glist_rgba != NULL)
			{
				/* Transfer this frame */
				(void)ImgViewImageAppendFrame(
					img,
					(guint8 *)glist_rgba->data,
					(gulong)((glist_delay != NULL) ?
						(guint)glist_delay->data : 0)
				);
				glist_rgba->data = NULL;

				glist_rgba = g_list_next(glist_rgba);
				if(glist_delay != NULL)
					glist_delay = g_list_next(glist_delay);
			}

			ImgViewQueueDrawView(iv);
		}
	}
	else
	{
		/* Image geometries do not match, create a new ImgView image
		 * and transfer it to the ImgView
		 */
		img = ImgViewImageNew(
			width, height,
			bpp, bpl
		);
		if(img != NULL)
		{
			/* Transfer each RGBA image data to the new ImgView image */
			GList	*glist_rgba = rgba_list,
					*glist_delay = delay_list;
			while(glist_rgba != NULL)
			{
				/* Transfer this frame */
				(void)ImgViewImageAppendFrame(
					img,
					(guint8 *)glist_rgba->data,
					(gulong)((glist_delay != NULL) ?
						(guint)glist_delay->data : 0)
				);
				glist_rgba->data = NULL;

				glist_rgba = g_list_next(glist_rgba);
				if(glist_delay != NULL)
					glist_delay = g_list_next(glist_delay);
			}

			/* Set/transfer the ImgView image to the ImgView */
			if(need_zoom_to_fit)
				ImgViewSetImageToFit(iv, img);
			else
				ImgViewSetImage(iv, img);
		}
	}

	/* Set the ImgView to play if there are multiple frames */
	if(nframes > 1)
		ImgViewPlay(iv);

	/* Delete the opened image data */
	g_list_foreach(
		rgba_list,
		(GFunc)g_free,
		NULL
	);
	g_list_free(rgba_list);
	g_list_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	/* Delete the progress callback data */
	g_free(d->name);
	g_free(d);

	edv_status_bar_progress(status_bar, 0.0f, FALSE);
	edv_status_bar_message(status_bar, NULL, FALSE);
	edv_image_browser_set_passive_busy(imbr, FALSE);
}

/*
 *	Clears the ImgView.
 */
void edv_image_browser_view_clear(EDVImageBrowser *imbr)
{
	CfgList *cfg_list;
	CfgColor *cfg_bg_color;
	imgview_struct *iv;
	EDVCore *core;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Delete the current ImgView image */
	ImgViewClear(iv);

	/* Restore the default background color from the configuration */
	cfg_bg_color = EDV_GET_COLOR(EDV_CFG_PARM_IMBR_BG_COLOR);
	if(cfg_bg_color != NULL)
	{
		GdkColor gdk_color[5];
		const gint m = sizeof(gdk_color) / sizeof(GdkColor);
		gint i;
		for(i = 0; i < m; i++)
		{
			GDK_COLOR_SET_COEFF(
				&gdk_color[i],
				cfg_bg_color->r,
				cfg_bg_color->g,
				cfg_bg_color->b
			);
		}
		ImgViewSetViewBG(iv, gdk_color);
	}
}
