#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "tool_bar.h"
#include "imgview.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_image_io.h"
#include "edv_recycle_obj.h"
#include "edv_device.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_devices_list.h"
#include "edv_device_mount.h"
#include "edv_mount_bar.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "obj_op_dlg.h"
#include "edv_confirm.h"
#include "image_browser.h"
#include "image_browser_cb.h"
#include "image_browser_op.h"
#include "image_browser_list.h"
#include "presentation_mode.h"
#include "edv_vfs_obj_create.h"
#include "edv_vfs_obj_op.h"
#include "edv_help.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


void edv_image_browser_op_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_image_browser_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_image_browser_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);

void edv_image_browser_op_close(EDVImageBrowser *imbr);
static void edv_image_browser_op_exit(EDVImageBrowser *imbr);

static void edv_image_browser_op_sync_disks(EDVImageBrowser *imbr);
static void edv_image_browser_op_run(EDVImageBrowser *imbr);
static void edv_image_browser_op_master_write_protect(EDVImageBrowser *imbr);
static void edv_image_browser_op_delete_method_recycle(EDVImageBrowser *imbr);
static void edv_image_browser_op_delete_method_purge(EDVImageBrowser *imbr);

static void edv_image_browser_new_object_nexus(
	EDVImageBrowser *imbr,
	const EDVObjectType type,
	GtkWidget *toplevel
);
static void edv_image_browser_op_new_object_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
);
static void edv_image_browser_op_new_file(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_directory(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_link(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_fifo(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_device_block(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_device_character(EDVImageBrowser *imbr);
static void edv_image_browser_op_new_socket(EDVImageBrowser *imbr);

static void edv_image_browser_op_open(EDVImageBrowser *imbr);
static void edv_image_browser_op_open_to(EDVImageBrowser *imbr, ToolBarItem *item);
static void edv_image_browser_op_open_with(EDVImageBrowser *imbr);

static void edv_image_browser_op_copy_path(EDVImageBrowser *imbr);
static void edv_image_browser_op_copy_url(EDVImageBrowser *imbr);
void edv_image_browser_op_paste(EDVImageBrowser *imbr);
void edv_image_browser_op_paste2(
	EDVImageBrowser *imbr,
	EDVVFSObject *obj
);

static void edv_image_browser_op_move(EDVImageBrowser *imbr);
static void edv_image_browser_op_copy(EDVImageBrowser *imbr);
static void edv_image_browser_op_link(EDVImageBrowser *imbr);
static void edv_image_browser_op_rename(EDVImageBrowser *imbr);
static void edv_image_browser_op_chmod(EDVImageBrowser *imbr);
static void edv_image_browser_op_chown(EDVImageBrowser *imbr);
static void edv_image_browser_op_chtime(EDVImageBrowser *imbr);
static void edv_image_browser_op_delete(EDVImageBrowser *imbr);
static void edv_image_browser_op_properties(EDVImageBrowser *imbr);

static void edv_image_browser_op_select_all(EDVImageBrowser *imbr);
static void edv_image_browser_op_unselect_all(EDVImageBrowser *imbr);
static void edv_image_browser_op_invert_selection(EDVImageBrowser *imbr);

static void edv_image_browser_op_goto_parent(EDVImageBrowser *imbr);
static void edv_image_browser_op_goto_home(EDVImageBrowser *imbr);
static void edv_image_browser_op_mount(EDVImageBrowser *imbr);
static void edv_image_browser_op_eject(EDVImageBrowser *imbr);

static void edv_image_browser_op_download(EDVImageBrowser *imbr);

void edv_image_browser_op_refresh(EDVImageBrowser *imbr);
void edv_image_browser_op_refresh_all(EDVImageBrowser *imbr);

static void edv_image_browser_op_stop(EDVImageBrowser *imbr);
static void edv_image_browser_op_continue(EDVImageBrowser *imbr);
static void edv_image_browser_op_presentation_mode(EDVImageBrowser *imbr);

static void edv_image_browser_previous_image(EDVImageBrowser *imbr);
static void edv_image_browser_next_image(EDVImageBrowser *imbr);

static void edv_image_browser_op_animation_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
);
#if 0
static void edv_image_browser_op_animation_play(EDVImageBrowser *imbr);
static void edv_image_browser_op_animation_pause(EDVImageBrowser *imbr);
#endif
static void edv_image_browser_op_animation_previous(EDVImageBrowser *imbr);
static void edv_image_browser_op_animation_next(EDVImageBrowser *imbr);

static void edv_image_browser_op_rotate_transform_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
);
static void edv_image_browser_op_rotate_cw90(EDVImageBrowser *imbr);
static void edv_image_browser_op_rotate_ccw90(EDVImageBrowser *imbr);
static void edv_image_browser_op_rotate_cw180(EDVImageBrowser *imbr);
static void edv_image_browser_op_mirror_horizontal(EDVImageBrowser *imbr);
static void edv_image_browser_op_mirror_vertical(EDVImageBrowser *imbr);

static void edv_image_browser_op_list_filter(EDVImageBrowser *imbr);

static void edv_image_browser_op_mime_types(EDVImageBrowser *imbr);
static void edv_image_browser_op_devices(EDVImageBrowser *imbr);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Operation ID callback nexus.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
void edv_image_browser_op_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data  
)
{
	gboolean iv_need_resume_play;
	gint dev_num;
	GtkWidget *toplevel;
	imgview_struct *iv;
	CfgList *cfg_list;
	EDVDevice *dev;
	EDVImageBrowser *imbr;
	EDVCore *core;
	EDVImageBrowserOp *op = EDV_IMAGE_BROWSER_OP(data);
	if(op == NULL)
		return;

	imbr = op->image_browser;
	if((imbr == NULL) || (op->flags & EDV_OPID_NO_OP))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return;

	imbr->freeze_count++;

	toplevel = imbr->toplevel;
	iv = imbr->imgview;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get last selected device (if any) */
	dev_num = imbr->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));

	/* Pause the ImgView if it was playing so that function calls
	 * here use less resources
	 */
	iv_need_resume_play = ImgViewIsPlaying(iv);
	if(iv_need_resume_play)
		ImgViewPause(iv);

	/* Handle by operation id code */
	switch(op->id)
	{
	  case EDV_IMAGE_BROWSER_OP_NONE:
	  case EDV_IMAGE_BROWSER_OP_SEPARATOR:
		break;

	  case EDV_IMAGE_BROWSER_OP_CLOSE:
		edv_image_browser_op_close(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_EXIT:
		edv_image_browser_op_exit(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_SYNC_DISKS:
		edv_image_browser_op_sync_disks(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_HISTORY:
		edv_map_history(
			core,
			-1,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	  case EDV_IMAGE_BROWSER_OP_RUN:
		edv_image_browser_op_run(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_RUN_TERMINAL:
		if(core != NULL)
		{
			gchar *working_dir = STRDUP(edv_image_browser_get_location(imbr));
			edv_run_terminal(
				core,
				NULL,
				working_dir,
				toplevel
			);
			g_free(working_dir);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_WRITE_PROTECT:
		edv_image_browser_op_master_write_protect(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_DELETE_METHOD_RECYCLE:
		edv_image_browser_op_delete_method_recycle(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_DELETE_METHOD_PURGE:
		edv_image_browser_op_delete_method_purge(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_NEW:
		edv_image_browser_op_new_object_menu(imbr, item);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_FILE:
		edv_image_browser_op_new_file(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_DIRECTORY:
		edv_image_browser_op_new_directory(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_LINK:
		edv_image_browser_op_new_link(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_FIFO:
		edv_image_browser_op_new_fifo(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_DEVICE_BLOCK:
		edv_image_browser_op_new_device_block(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_DEVICE_CHARACTER:
		edv_image_browser_op_new_device_character(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEW_SOCKET:
		edv_image_browser_op_new_socket(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_OPEN:
		edv_image_browser_op_open(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_OPEN_TO:
		edv_image_browser_op_open_to(imbr, item);
		break;
	  case EDV_IMAGE_BROWSER_OP_OPEN_WITH:
		edv_image_browser_op_open_with(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_COPY_PATH:
		edv_image_browser_op_copy_path(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_COPY_URL:
		edv_image_browser_op_copy_url(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_PASTE:
		edv_image_browser_op_paste(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_MOVE:
		edv_image_browser_op_move(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_COPY:
		edv_image_browser_op_copy(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_LINK:
		edv_image_browser_op_link(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_RENAME:
		edv_image_browser_op_rename(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_CHMOD:
		edv_image_browser_op_chmod(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_CHOWN:
		edv_image_browser_op_chown(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_CHTIME:
		edv_image_browser_op_chtime(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_DELETE:
		edv_image_browser_op_delete(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_PROPERTIES:
		edv_image_browser_op_properties(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_SELECT_ALL:
		edv_image_browser_op_select_all(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_UNSELECT_ALL:
		edv_image_browser_op_unselect_all(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_INVERT_SELECTION:
		edv_image_browser_op_invert_selection(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_FIND:
		edv_map_find_image_browser(core, imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_DOWNLOAD:
		edv_image_browser_op_download(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_REFRESH:
		edv_image_browser_op_refresh(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_REFRESH_ALL:
		edv_image_browser_op_refresh_all(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_GOTO_PARENT:
		edv_image_browser_op_goto_parent(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_GOTO_HOME:
		edv_image_browser_op_goto_home(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_STOP:
		edv_image_browser_op_stop(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_CONTINUE:
		edv_image_browser_op_continue(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_PRESENTATION_MODE:
		edv_image_browser_op_presentation_mode(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_PREVIOUS_IMAGE:
		edv_image_browser_previous_image(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_NEXT_IMAGE:
		edv_image_browser_next_image(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_ANIMATION:
		edv_image_browser_op_animation_menu(imbr, item);
		break;
	  case EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY:
		ImgViewPlay(iv);
		edv_image_browser_update_display(imbr);
		iv_need_resume_play = FALSE;
		break;
	  case EDV_IMAGE_BROWSER_OP_ANIMATION_PAUSE:
		edv_image_browser_update_display(imbr);
		iv_need_resume_play = FALSE;
		break;
	  case EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY_PAUSE:
		if(!iv_need_resume_play)
			ImgViewPlay(iv);
		edv_image_browser_update_display(imbr);
		iv_need_resume_play = FALSE;
		break;
	  case EDV_IMAGE_BROWSER_OP_ANIMATION_PREVIOUS_FRAME:
		edv_image_browser_op_animation_previous(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_ANIMATION_NEXT_FRAME:
		edv_image_browser_op_animation_next(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_ROTATE_TRANSFORM:
		edv_image_browser_op_rotate_transform_menu(imbr, item);
		break;
	  case EDV_IMAGE_BROWSER_OP_ROTATE_CW90:
		edv_image_browser_op_rotate_cw90(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_ROTATE_CCW90:
		edv_image_browser_op_rotate_ccw90(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_ROTATE_CW180:
		edv_image_browser_op_rotate_cw180(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_MIRROR_HORIZONTAL:
		edv_image_browser_op_mirror_horizontal(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_MIRROR_VERTICAL:
		edv_image_browser_op_mirror_vertical(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_SHOW_TOOL_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR, state);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_LOCATION_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR, state);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_MOUNT_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR, state);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_FIND_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_FIND_BAR
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_FIND_BAR, state);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_STATUS_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR, state);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_IMAGE_BROWSER_OP_LIST_FILTER:
		edv_image_browser_op_list_filter(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_SHOW_HIDDEN_OBJECTS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_OBJECT_HIDDEN
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_HIDDEN, state);
			edv_image_browser_op_refresh(imbr);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_NOACCESS_OBJECTS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_OBJECT_NOACCESS
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_NOACCESS, state);
			edv_image_browser_op_refresh(imbr);
		}
		break;
	  case EDV_IMAGE_BROWSER_OP_SHOW_NONIMAGE_OBJECTS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_IMBR_SHOW_OBJECT_NONIMAGE
			);
			EDV_SET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_NONIMAGE, state);
			edv_image_browser_op_refresh(imbr);
		}
		break;

	  case EDV_IMAGE_BROWSER_OP_MOUNT:
		edv_image_browser_op_mount(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_EJECT:
		edv_image_browser_op_eject(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_DEVICE_CHECK:
		edv_run_device_check(core, dev, toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_DEVICE_TOOLS:
		edv_run_device_tools(core, dev, toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_DEVICE_FORMAT:
		edv_run_device_format(core, dev, toplevel);
		break;

	  case EDV_IMAGE_BROWSER_OP_MIME_TYPES:
		edv_image_browser_op_mime_types(imbr);
		break;
	  case EDV_IMAGE_BROWSER_OP_DEVICES:
		edv_image_browser_op_devices(imbr);
		break;

	  case EDV_IMAGE_BROWSER_OP_NEW_BROWSER:
		edv_new_vfs_browser(
			core,
			edv_image_browser_get_location(imbr),
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_IMAGE_BROWSER_OP_NEW_IMBR:
		edv_new_image_browser(
			core,
			edv_image_browser_get_location(imbr),
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_IMAGE_BROWSER_OP_NEW_ARCHIVER:
		edv_new_archiver(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_IMAGE_BROWSER_OP_RECYCLE_BIN:
		edv_map_recycle_bin(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;


	  case EDV_IMAGE_BROWSER_OP_OPTIONS:
		edv_map_options(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_IMAGE_BROWSER_OP_CUSTOMIZE:
		edv_map_customize(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_IMAGE_BROWSER_OP_HELP_CONTENTS:
		edv_help(core, "Contents", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_FILE_BROWSER:
		edv_help(core, "File Browser", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_IMAGE_BROWSER:
		edv_help(core, "Image Browser", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_ARCHIVER:
		edv_help(core, "Archiver", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_RECYCLE_BIN:
		edv_help(core, "Recycle Bin", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_KEYS_LIST:
		edv_help(core, "Keys List", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_MIME_TYPES:
		edv_help(core, "MIME Types", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_DEVICES:
		edv_help(core, "Devices", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_COMMON_OPERATIONS:
		edv_help(core, "Common Operations", toplevel);
		break;
	  case EDV_IMAGE_BROWSER_OP_HELP_ABOUT:
		edv_map_about_dialog(
			core,
			core->geometry_flags, 
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	}

	if(iv_need_resume_play)
		ImgViewPlay(iv);

	imbr->freeze_count--;
}

/*
 *	Operation ID enter notify callback nexus.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
void edv_image_browser_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data  
)
{
	const gchar *tooltip;
	EDVImageBrowser *imbr;
	EDVImageBrowserOp *op = EDV_IMAGE_BROWSER_OP(data);
	if(op == NULL)
		return;

	imbr = op->image_browser;
	if(imbr == NULL)
		return;

	if(imbr->freeze_count > 0)
		return;

	tooltip = op->tooltip;
	if(!STRISEMPTY(tooltip))
		edv_status_bar_message(imbr->status_bar, tooltip, FALSE);
}

/*
 *	Operation ID leave notify callback nexus.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
void edv_image_browser_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
)
{
	EDVImageBrowser *imbr;
	EDVImageBrowserOp *op = EDV_IMAGE_BROWSER_OP(data);
	if(op == NULL)
		return;

	imbr = op->image_browser;
	if(imbr == NULL)
		return;

	if(imbr->freeze_count > 0)
		return;

	edv_status_bar_message(imbr->status_bar, NULL, FALSE);
}


/*
 *	Close.
 */
void edv_image_browser_op_close(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_sync_configuration(imbr);
	edv_image_browser_unmap(imbr);
}

/*
 *	Exit.
 */
static void edv_image_browser_op_exit(EDVImageBrowser *imbr)
{
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;

	edv_image_browser_sync_configuration(imbr);
	edv_image_browser_unmap(imbr);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	Sync Disks.
 */
static void edv_image_browser_op_sync_disks(EDVImageBrowser *imbr)
{
	GtkWidget *sb;
	EDVCore *core;

	if(imbr == NULL)
		return;

	sb = imbr->status_bar;
	core = imbr->core;

	edv_image_browser_set_busy(imbr, TRUE);
	edv_status_bar_message(
		sb,
		"Syncing disks...",
		TRUE
	);

	edv_sync_edv(imbr->core);

	edv_status_bar_message(
		sb,
		"Disk sync done",
		FALSE
	);
	edv_status_bar_progress(sb, 0.0f, FALSE);
	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Run.
 */
static void edv_image_browser_op_run(EDVImageBrowser *imbr)
{
	gchar *cmd;
	GList *glist;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	const EDVVFSObject *obj;
	EDVCore *core;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	/* Format command to contain the list of selected objects */
	cmd = STRDUP("");
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(
			tlist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		if(obj->path != NULL)
		{
			gchar *s = g_strconcat(
				cmd,
				obj->path,
				NULL
			);
			g_free(cmd);
			cmd = s;

			if(g_list_next(glist) != NULL)
			{
				gchar *s = g_strconcat(
					cmd,
					" ",
					NULL
				);
				g_free(cmd);
				cmd = s;
			}
		}
	}

	edv_map_run_dialog_command(
		core,
		cmd,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	g_free(cmd);
}

/*
 *	Master Write Protect.
 */
static void edv_image_browser_op_master_write_protect(EDVImageBrowser *imbr)
{
	gboolean state;
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get the current Master Write Protect state */
	state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Toggle the Master Write Protect */
	state = !state;

	/* Set the new Master Write Protect state */
	EDV_SET_B(EDV_CFG_PARM_WRITE_PROTECT, state);

	/* Notify about the Master Write Protect state change */
	edv_queue_emit_master_write_protect_changed(core);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Delete Method Recycle.
 */
static void edv_image_browser_op_delete_method_recycle(EDVImageBrowser *imbr)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	core = imbr->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_RECYCLE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Delete Method Purge.
 */
static void edv_image_browser_op_delete_method_purge(EDVImageBrowser *imbr)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	core = imbr->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_PURGE          
	);
	edv_queue_emit_delete_method_changed(core);

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	New Object Nexus.
 */
static void edv_image_browser_new_object_nexus(
	EDVImageBrowser *imbr,
	const EDVObjectType type,
	GtkWidget *toplevel
)
{
	gboolean yes_to_all = FALSE;
	gint status;
	const char	*obj_type_name,
			*cur_path,
			*error_msg;
	gchar *new_path = NULL;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	core = imbr->core;
	
#define CLEANUP_RETURN	{		\
 g_free(new_path);			\
					\
 return;				\
}

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		CLEANUP_RETURN;
	}

	/* Get current location as the path to create the new object
	 * at
	 */
	cur_path = edv_image_browser_get_location(imbr);
	if(cur_path == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		CLEANUP_RETURN;
	}

	/* Create new object by type */
	status = -1;
	obj_type_name = NULL;
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
	  case EDV_OBJECT_TYPE_ERROR:
	  case EDV_OBJECT_TYPE_FILE:
		obj_type_name = "file";
		status = EDVVFSObjectCreateFile(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
		obj_type_name = "directory";
		status = EDVVFSObjectCreateDirectory(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_LINK:
		obj_type_name = "symbolic link";
		status = EDVVFSObjectCreateLink(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		obj_type_name = "block device";
		status = EDVVFSObjectCreateDeviceBlock(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		obj_type_name = "character device";
		status = EDVVFSObjectCreateDeviceCharacter(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_FIFO:
		obj_type_name = "fifo pipe";
		status = EDVVFSObjectCreateFifo(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
		obj_type_name = "socket";
		status = EDVVFSObjectCreateSocket(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	}

	/* Unmap progress dialog since it may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Error creating new object? */
	if(status != 0)
	{
		error_msg = EDVVFSObjectCreateGetError(core);
		if(!STRISEMPTY(error_msg))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Create Failed",
				error_msg,
				NULL,
				toplevel
			);
		}

		edv_status_bar_message(imbr->status_bar, NULL, FALSE);
	}
	else
	{
		/* Successfully created new object */
		gchar *msg;
		EDVVFSObject *obj = edv_vfs_object_lstat(new_path);
		if(obj != NULL)
		{
			gint thumb_num;
			tlist_struct *tlist = imbr->tlist;

			/* Emit a disk object added signal to all of endeavour's
			 * resources.
			 */
			edv_emit_vfs_object_added(
				core,
				new_path,
				obj
			);

			/* Select new thumb (if any) who's disk object structure
			 * matches the newly added object path.
			 */
			thumb_num = edv_image_browser_list_find_by_path(imbr, new_path);
			if((thumb_num > -1) && (tlist != NULL))
			{
				TListFreeze(tlist);
				TListUnselectAll(tlist);
				TListSelectThumb(tlist, thumb_num);
				TListThaw(tlist);
			}

			edv_vfs_object_delete(obj);
		}

		msg = g_strdup_printf(
			"Created new %s",
			obj_type_name
		);
		edv_status_bar_message(imbr->status_bar, msg, FALSE);
		g_free(msg);
	}

	edv_image_browser_set_busy(imbr, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	New object submenu.
 */
static void edv_image_browser_op_new_object_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
)
{
	if(imbr == NULL)
		return;

	edv_map_menu_to_button(
		imbr->new_object_menu,
		ToolBarItemGetWidget(item)
	);
}

/*
 *	New File.
 */
static void edv_image_browser_op_new_file(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_FILE,
		imbr->toplevel
	);
}

/*
 *	New Directory.
 */
static void edv_image_browser_op_new_directory(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_DIRECTORY,
		imbr->toplevel
	);
}

/*
 *	New Link.
 */
static void edv_image_browser_op_new_link(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_LINK,
		imbr->toplevel
	);
}

/*
 *	New FIFO Pipe.
 */
static void edv_image_browser_op_new_fifo(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_FIFO,
		imbr->toplevel
	);
}

/*
 *	New Block Device.
 */
static void edv_image_browser_op_new_device_block(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_DEVICE_BLOCK,
		imbr->toplevel
	);
}

/*
 *	New Character Device.
 */
static void edv_image_browser_op_new_device_character(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_DEVICE_CHARACTER,
		imbr->toplevel
	);
}

/*
 *	New Socket.
 */
static void edv_image_browser_op_new_socket(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_new_object_nexus(
		imbr,
		EDV_OBJECT_TYPE_SOCKET,
		imbr->toplevel
	);
}


/*
 *	Open.
 */
static void edv_image_browser_op_open(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_list_open(
		imbr,
		-1,				/* All selected object(s) */
		0				/* Modifier keys */
	);
}

/*
 *	Open To.
 */
static void edv_image_browser_op_open_to(
	EDVImageBrowser *imbr,
	ToolBarItem *item
)
{
	if(imbr == NULL)
		return;

	edv_map_menu_to_button(
		imbr->open_to_menu,
		ToolBarItemGetWidget(item)
	);
}

/*
 *	Open With.
 */
static void edv_image_browser_op_open_with(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_list_open_with(
		imbr,
		-1				/* All selected object(s) */
	);
}


/*
 *	Copy Path.
 */
static void edv_image_browser_op_copy_path(EDVImageBrowser *imbr)
{
	GList *objs_list;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	edv_image_browser_sync_data(imbr);

	/* Get the list of selected objects */
	objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Copy the paths of the selected objects to the dde */
	edv_copy_paths_to_clipboard(
		imbr->core,
		objs_list,
		imbr->toplevel
	);

	/* Delete the list of selected objects */
	g_list_free(objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Copy URL.
 */
static void edv_image_browser_op_copy_url(EDVImageBrowser *imbr)
{
	GList *objs_list;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	edv_image_browser_sync_data(imbr);

	/* Get the list of selected objects */
	objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Copy the urls of the selected objects to the dde */
	edv_copy_urls_to_clipboard(
		imbr->core,
		objs_list,
		imbr->toplevel
	);

	/* Delete the list of selected objects */
	g_list_free(objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Paste to the current location.
 */
void edv_image_browser_op_paste(EDVImageBrowser *imbr)
{
	edv_image_browser_op_paste2(
		imbr,
		NULL
	);
}

/*
 *	Paste to target location.
 *
 *	If obj is not NULL then obj specifies the target location.
 */
void edv_image_browser_op_paste2(
	EDVImageBrowser *imbr,
	EDVVFSObject *obj
)
{
	gboolean yes_to_all = FALSE;
	gint status = -1;
	const gchar *protocol;
	gchar		*buf,
			*cur_location;
	GList		*glist,
			*url_list = NULL;
	GtkWidget *toplevel;
	URLStruct *url;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	edv_image_browser_sync_data(imbr);

	/* No target location specified? */
	if(obj == NULL)
	{
		tlist_struct *tlist = imbr->tlist;
		if(tlist->selection_end != NULL)
		{
			GList *glist = tlist->selection_end;
			const gint thumb_num = (gint)glist->data;
			obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
		}
	}
	/* The target location must lead to a directory */
	if(obj != NULL)
	{
		if(!edv_path_is_directory(obj->path))
			obj = NULL;
	}
	/* Get the target location */
	cur_location = (obj != NULL) ?
		STRDUP(obj->path) :
		STRDUP(edv_image_browser_get_location(imbr));
	if(cur_location == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the data from the clipboard as a string */
	buf = GUIDDEGetString(
		toplevel,
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME
	);
	if(buf == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Paste Failed",
"The clipboard is empty or does not contain the\n\
correct data type.",
			NULL,
			toplevel
		);
		g_free(cur_location);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Decode URL list string into a list of URLs */
	url_list = url_decode_string(buf);
	g_free(buf);
	buf = NULL;

	/* Nothing to paste? */
	if(url_list == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Paste Failed",
"There were no objects found in the clipboard's data.",
			NULL,
			toplevel
		);
		g_free(cur_location);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Confirm paste */
	url = URL(url_list->data);
	if(edv_confirm_copy(
		core,
		toplevel,
		(url != NULL) ? url->path : NULL, g_list_length(url_list),
		cur_location
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(url_list, (GFunc)url_delete, NULL);
		g_list_free(url_list);
		g_free(cur_location);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Paste each URL */
	for(glist = url_list; glist != NULL; glist = g_list_next(glist))
	{
		url = URL(glist->data);
		if(url == NULL)
			continue;

		protocol = url->protocol;
		if(STRISEMPTY(protocol))
			protocol = "file";

		/* File? */
		if(!g_strcasecmp(protocol, "file"))
		{
			const gchar *error_msg;
			GList *new_paths_list = NULL;

			/* Copy this object */
			status = edv_vfs_object_op_copy(
				core,
				url->path,
				cur_location,
				&new_paths_list,
				TRUE,		/* Archive */
				toplevel,
				TRUE,		/* Show progress */
				TRUE,		/* Interactive */
				&yes_to_all
			);

			/* Check for error */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Copy Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report the new objects? */
			if(new_paths_list != NULL)
			{
				const gchar *path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = new_paths_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					path = (const gchar *)glist->data;
					if(STRISEMPTY(path))
						continue;

					obj = edv_vfs_object_lstat(path);
					if(obj != NULL)
					{
						edv_emit_vfs_object_added(
							core,
							path,
							obj
						);
						edv_vfs_object_delete(obj);
					}
				}

				g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
				g_list_free(new_paths_list);
			}

			/* Skip handling of the rest of the objects on error
			 * (status != 0) and that the error was not a user
			 * response of no (status != -5)
			 */
			if((status != 0) && (status != -5))
				break;
		}
		/* Download? */
		else if(!g_strcasecmp(protocol, "http") ||
			!g_strcasecmp(protocol, "ftp") ||
			!g_strcasecmp(protocol, "https")
		)
		{
			edv_internet_download_object(
				core,
				url,
				cur_location,
				toplevel
			);
		}
	}

	/* Delete the URLs list */
	g_list_foreach(url_list, (GFunc)url_delete, NULL);
	g_list_free(url_list);

	g_free(cur_location);

	/* Unmap progress dialog, it may have been mapped if any
	 * operations occured in the above loop
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Move.
 */
static void edv_image_browser_op_move(EDVImageBrowser *imbr)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	EDVCore *core;
	edv_obj_op_dlg_struct *d;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Map the object operations dialog to move */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_MOVE,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Copy.
 */
static void edv_image_browser_op_copy(EDVImageBrowser *imbr)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	EDVCore *core;
	edv_obj_op_dlg_struct *d;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Map the object operations dialog to copy */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_COPY,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Link.
 */
static void edv_image_browser_op_link(EDVImageBrowser *imbr)
{
	GList		*glist,
			*sel_objs_list;
	GtkWidget *toplevel;
	EDVCore *core;
	edv_obj_op_dlg_struct *d;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Linking only allows one object in the list, so create
	 * a list with only one object
	 */
	glist = NULL;
	glist = g_list_append(glist, sel_objs_list->data);
	g_list_free(sel_objs_list);
	sel_objs_list = glist;

	/* Map the object operations dialog to link */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_LINK,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Rename.
 */
static void edv_image_browser_op_rename(EDVImageBrowser *imbr)
{
	gint thumb_num;
	GList *glist;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	glist = tlist->selection_end;
	thumb_num = (glist != NULL) ? (gint)glist->data : -1;
	if(thumb_num > -1)
		edv_image_browser_list_rename_query(imbr, thumb_num);
}

/*
 *	Change Permissions.
 */
static void edv_image_browser_op_chmod(EDVImageBrowser *imbr)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Map the object operations dialog to change permissions */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHMOD,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Change Ownership.
 */
static void edv_image_browser_op_chown(EDVImageBrowser *imbr)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Map the object operations dialog to change ownership */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHOWN,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Change Time Stamps.
 */
static void edv_image_browser_op_chtime(EDVImageBrowser *imbr)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_image_browser_sync_data(imbr);

	/* Get the selected objects list */
	sel_objs_list = edv_image_browser_get_selected_objects(imbr, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Map the object operations dialog to set time stamps */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHTIME,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_image_browser_get_location(imbr),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Delete.
 */
static void edv_image_browser_op_delete(EDVImageBrowser *imbr)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			response,
			npaths,
			objects_deleted = 0;
	const gchar	*path,
			*error_msg;
	GList		*glist,
			*paths_list,
			*index_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

#define CLEANUP_RETURN	{			\
 if(paths_list != NULL) {			\
  g_list_foreach(				\
   paths_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(paths_list);			\
 }						\
											\
 return;					\
}

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get the selected paths list */
	paths_list = edv_image_browser_get_selected_paths(imbr);

	/* No objects selected? */
	if(paths_list == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		CLEANUP_RETURN;
	}

	npaths = g_list_length(paths_list);

	/* Confirm delete */
	response = edv_confirm_delete(
		core, toplevel,
		(npaths == 1) ? (const gchar *)paths_list->data : NULL,
		npaths
	);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES_TO_ALL:
		yes_to_all = TRUE;
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_OK:
		break;

	  default:
		edv_image_browser_set_busy(imbr, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Delete each object */
	status = 0;
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		path = (const gchar *)glist->data;
		if(path == NULL)
			continue;

		/* Delete this object */
		index_list = NULL;
		status = edv_recycle_object(
			core, path,
			&index_list,
			toplevel,
			TRUE,				/* Show progress */
			TRUE,				/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_recycle_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Delete Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Report if the object has been removed */
		if(!edv_path_lexists(path))
			edv_emit_vfs_object_removed(
				core,
				path
			); 

		/* Notify about the recycled objects being added */
		if(index_list != NULL)
		{
			guint index;
			GList *glist;

			for(glist = index_list;
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				index = (guint)glist->data;
				if(index != 0)
					edv_emit_recycled_object_added(core, index);

				objects_deleted++;
			}
		}

		/* Delete the recycle objects index list */
		g_list_free(index_list);

		/* User aborted? */
		if(status == -4)
			break;
	}

	/* Update the status bar */
	if(TRUE)
	{
		gchar *s;
		if(status == -4)
			s = STRDUP(
				"Delete operation canceled"
			);
		else if(objects_deleted > 0)
			s = g_strdup_printf(
				"Deleted %i %s",
				objects_deleted,
				(objects_deleted == 1) ? "object" : "objects"
			);
		else
			s = g_strdup_printf(
				"Unable to delete %s",
				(npaths == 1) ? "object" : "objects"
			);
		edv_status_bar_message(imbr->status_bar, s, FALSE);
		g_free(s);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_image_browser_set_busy(imbr, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Properties.
 */
static void edv_image_browser_op_properties(EDVImageBrowser *imbr)
{
	GtkWidget *toplevel;
	EDVVFSObject *obj;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	edv_image_browser_sync_data(imbr);

	/* Get the selected object */
	obj = EDV_VFS_OBJECT(TListGetThumbData(
		imbr->tlist, imbr->tlist_selected_thumb
	));

	/* No selected object? */
	if(obj == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Create a new Properties Dialog to display the object */
	edv_new_properties_dialog_vfs(
		core,
		obj->path,
		NULL,				/* Default page */
		obj->meta_data_list,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Select All.
 */
static void edv_image_browser_op_select_all(EDVImageBrowser *imbr)
{
	EDVCore *core;
	tlist_struct *tlist;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	tlist = imbr->tlist;
	core = imbr->core;

	/* Select all thumbs */
	TListFreeze(tlist);
	TListSelectAll(tlist);
	TListThaw(tlist);

	/* Assume highest thumb index as the last selected thumb */
	imbr->tlist_selected_thumb = tlist->total_thumbs - 1;

	edv_status_bar_message(
		imbr->status_bar,
		"All objects selected",
		FALSE
	);
	edv_image_browser_update_display(imbr);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Unselect All.
 */
static void edv_image_browser_op_unselect_all(EDVImageBrowser *imbr)
{
	EDVCore *core;
	tlist_struct *tlist;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	tlist = imbr->tlist;
	core = imbr->core;

	/* Unselect all thumbs */
	TListFreeze(tlist);
	TListUnselectAll(tlist);
	TListThaw(tlist);

	/* Mark that no thumb is selected */
	imbr->tlist_selected_thumb = -1;

	edv_status_bar_message(
		imbr->status_bar,
		"All objects unselected",
		FALSE
	);
	edv_image_browser_update_display(imbr);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Invert Selection.
 */
static void edv_image_browser_op_invert_selection(EDVImageBrowser *imbr)
{
	gint		thumb,
			total_thumbs;
	GList		*glist,
			*selection;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	tlist = imbr->tlist;
	core = imbr->core;

	/* Get a copy of the selected thumbs list */
	selection = (tlist->selection != NULL) ?
		g_list_copy(tlist->selection) : NULL;

	/* Invert the selection */
	TListFreeze(tlist);
	for(thumb = 0, total_thumbs = tlist->total_thumbs;
		thumb < total_thumbs;
		thumb++
	)
	{
		/* Unselect all rows that are selected */
		for(glist = selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			if(thumb == (gint)glist->data)
			{
				TListUnselectThumb(
					tlist,
					thumb
				);
				break;
			}
		}
		/* If this row was not selected then select it */
		if(glist == NULL)
			TListSelectThumb(
				tlist,
				thumb
			);
	}
	TListThaw(tlist);

	g_list_free(selection);

	edv_status_bar_message(
		imbr->status_bar,
		"Selection inverted",
		FALSE
	);

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Go To Parent.
 */
static void edv_image_browser_op_goto_parent(EDVImageBrowser *imbr)
{
	gchar *parent_path;
	const gchar *current_location;
	GtkWidget *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Get the current location */
	current_location = edv_image_browser_get_location(imbr);
	if(STRISEMPTY(current_location))
	{
		edv_play_sound_beep(core);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Already at toplevel? */
	if(!strcmp((const char *)current_location, "/"))
	{
		edv_play_sound_beep(core);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Get copy of parent path of the current location */
	parent_path = g_dirname(current_location);
	if(parent_path == NULL)
	{
		edv_play_sound_beep(core);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Go to parent directory */
	GUIBlockInput(toplevel, TRUE);
	edv_image_browser_select_path(imbr, parent_path);
	g_free(parent_path);
	GUIBlockInput(toplevel, FALSE);

	edv_image_browser_update_display(imbr);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Go To Home.
 */
static void edv_image_browser_op_goto_home(EDVImageBrowser *imbr)
{
	gchar *home_path;
	GtkWidget *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Get home directory */
	home_path = STRDUP(core->home_path);
	if(STRISEMPTY(home_path))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A En Casa Gua Fallada",
"Incapaz de encontrar la gua de hogar, cercirese que el\n\
ambiente HOME variable se pone.",
			NULL,
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller au rpertoire d'accueil chou",
"Impossible de trouver le rpertoire d'accueil, s'assurer que\n\
la variable d''environnement HOME est dfinie.",
			NULL,
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Sie Zu Heim Verzeichnis Hat Versagt",
"Unfhig, das heim verzeichnis zu finden, vergewissert\n\
sich, da die umwelt vernderliche HOME gesetzt ist.",
			NULL,
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare All'Elenco Di Casa Fallito",
"Incapace per trovare l'elenco di casa, si assicura che\n\
l'ambiente HOME variabile  regolato.",
			NULL,
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te Huis Gids Verzuimde",
"Onbekwame de huis gids te vinden, vergewist zich ervan dat de\n\
omgeving, die veranderlijke HOME gezet is.",
			NULL,
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A Guia De Lar Fracassado",
"Incapaz de achar o guia de lar, assegura-se que o ambiente\n\
HOME varivel  posto.",
			NULL,
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til Hjem Sviktet Katalog",
"Maktesls finne hjemkatalogen, sjekker at miljet variabel\n\
HOME setter.",
			NULL,
#else
"Go To Home Directory Failed",
"Unable to find the home directory, make sure that the\n\
environment variable HOME is set.",
			NULL,
#endif
			toplevel
		);
		g_free(home_path);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Go to home directory */
	GUIBlockInput(toplevel, TRUE);
	edv_image_browser_select_path(imbr, home_path);
	g_free(home_path);
	GUIBlockInput(toplevel, FALSE);

	edv_image_browser_update_display(imbr);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Mount/Unmount.
 */
static void edv_image_browser_op_mount(EDVImageBrowser *imbr)
{
	gboolean original_mount_state;
	gint		status,
			dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Get number of last selected device index (if any) */
	dev_num = imbr->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);

	/* Update all the devices' statistics */
	edv_devices_list_update_statistics(core->devices_list);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Report mount signal to all of endeavour's resources */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Eject.
 */
static void edv_image_browser_op_eject(EDVImageBrowser *imbr)
{
	gboolean original_mount_state;
	gint		status,
			dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Get number of last selected device index (if any) */
	dev_num = imbr->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core, dev,
			TRUE, TRUE,
			toplevel
		);

	/* Eject */
	status = edv_device_eject(
		core, dev,
		TRUE, TRUE,
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	/* Eject error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Report eject (unmount) signal to all of endeavour's
		 * resources
		 */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Download.
 */
static void edv_image_browser_op_download(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);
	edv_internet_download_object(
		imbr->core,
		NULL,
		edv_image_browser_get_location(imbr),
		imbr->toplevel
	);
	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Refresh.
 */
void edv_image_browser_op_refresh(EDVImageBrowser *imbr)
{
	gchar *cur_path;
	GtkWidget	*toplevel,
			*sb;
	tlist_struct *tlist;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	sb = imbr->status_bar;

	edv_image_browser_set_busy(imbr, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Thumbs List */
	tlist = imbr->tlist;
	if(tlist != NULL)
	{
		GtkAdjustment *adj;

		/* Get the original scroll position */
		gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(tlist->hadjustment),
					last_y = GTK_ADJUSTMENT_GET_VALUE(tlist->vadjustment);

		TListFreeze(tlist);

		/* Reget listing and initialize values for the loading
		 * process
		 *
		 * The entire list should be fully obtained after this call
		 * so it is possible to scroll to the old position
		 */
		edv_image_browser_select_path(imbr, cur_path);

		/* Scroll back to the original location */
		adj = tlist->hadjustment;
		if(adj != NULL)
		{
			if(last_x > (adj->upper - adj->page_size))
				last_x = adj->upper - adj->page_size;
			if(last_x < adj->lower)
				last_x = adj->lower;
			gtk_adjustment_set_value(adj, last_x);
		}
		adj = tlist->vadjustment;
		if(adj != NULL)
		{
			if(last_y > (adj->upper - adj->page_size))
				last_y = adj->upper - adj->page_size;
			if(last_y < adj->lower)
				last_y = adj->lower;
			gtk_adjustment_set_value(adj, last_y);
		}

		TListThaw(tlist);
	}

	edv_image_browser_update_display(imbr);
	edv_status_bar_message(sb, "Refreshed contents listing", FALSE);

	g_free(cur_path);

	GUIBlockInput(toplevel, FALSE);
	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Refresh All.
 */
void edv_image_browser_op_refresh_all(EDVImageBrowser *imbr)
{
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	core = imbr->core;

	/* Refresh device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	/* Refresh Image Browser */
	edv_image_browser_op_refresh(imbr);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Stop.
 */
static void edv_image_browser_op_stop(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	imbr->stop_count++;
}

/*
 *	Continue.
 */
static void edv_image_browser_op_continue(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	/* Reset loading process timeout callback and initialize
	 * loading values
	 */
	edv_image_browser_queue_loading_process(imbr);

	edv_image_browser_update_display(imbr);
}

/*
 *	Presentation mode.
 */
static void edv_image_browser_op_presentation_mode(EDVImageBrowser *imbr)
{
	gboolean was_loading;
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	iv = imbr->imgview;

	/* Need to notify the loading process to stop before we enter
	 * presentation mode
	 */
	was_loading = (imbr->loading_tocb > 0) ? TRUE : FALSE;
	if(was_loading)
		imbr->stop_count++;

	/* Switch to the presentation mode and block until the
	 * presentation mode is exited
	 */
	EDVPresentationModeEnterFromImbr(
		imbr->core,
		imbr
	);

	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Go to the previous image.
 */
static void edv_image_browser_previous_image(EDVImageBrowser *imbr)
{
	gint		i,
			selected_thumb_num,
			goto_thumb_num;
	const gchar *ext;
	GList *glist;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return;

	tlist = imbr->tlist;
	glist = tlist->selection_end;
	selected_thumb_num = (glist != NULL) ?
		(gint)glist->data : 0;
	if(selected_thumb_num < 0)
		selected_thumb_num = 0;
	goto_thumb_num = -1;
	for(i = selected_thumb_num - 1; i >= 0; i--)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
		if(obj == NULL)
			continue;

		ext = edv_name_get_extension(obj->name);
		if(!edv_image_is_supported_extension(ext))
			continue;

		goto_thumb_num = i;
		break;
	}
	if(goto_thumb_num < 0)
	{
		for(i = tlist->total_thumbs - 1; i > selected_thumb_num; i--)
		{
			obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
			if(obj == NULL)
				continue;

			ext = edv_name_get_extension(obj->name);
			if(!edv_image_is_supported_extension(ext))
				continue;

			goto_thumb_num = i;
			break;
		}
	}

	if(goto_thumb_num > -1)
	{
		TListFreeze(tlist);
		TListUnselectAll(tlist);
		TListSelectThumb(tlist, goto_thumb_num);
		TListThaw(tlist);
	}
}

/*
 *	Go to the next image.
 */
static void edv_image_browser_next_image(EDVImageBrowser *imbr)
{
	gint		i,
			selected_thumb_num,
			goto_thumb_num;
	const gchar *ext;
	GList *glist;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return;

	tlist = imbr->tlist;
	glist = tlist->selection_end;
	selected_thumb_num = (glist != NULL) ?
		(gint)glist->data : 0;
	if(selected_thumb_num < 0)
		selected_thumb_num = 0;
	goto_thumb_num = -1;
	for(i = selected_thumb_num + 1; i < tlist->total_thumbs; i++)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
		if(obj == NULL)
			continue;

		ext = edv_name_get_extension(obj->name);
		if(!edv_image_is_supported_extension(ext))
			continue;

		goto_thumb_num = i;
		break;
	}
	if(goto_thumb_num < 0)
	{
		for(i = 0; i < selected_thumb_num; i++)
		{
			obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
			if(obj == NULL)
				continue;

			ext = edv_name_get_extension(obj->name);
			if(!edv_image_is_supported_extension(ext))
				continue;

			goto_thumb_num = i;
			break;
		}
	}

	if(goto_thumb_num > -1)
	{
		TListFreeze(tlist);
		TListUnselectAll(tlist);
		TListSelectThumb(tlist, goto_thumb_num);
		TListThaw(tlist);
	}
}


/*
 *	Animation submenu.
 */
static void edv_image_browser_op_animation_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
)
{
	if(imbr == NULL)
		return;

	edv_map_menu_to_button(
		imbr->animation_menu,
		ToolBarItemGetWidget(item)
	);
}

#if 0
/*
 *	Play animation.
 */
static void edv_image_browser_op_animation_play(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewPlay(iv);
}

/*
 *	Pause animation.
 */
static void edv_image_browser_op_animation_pause(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewPause(iv);
}
#endif

/*
 *	Previous frame.
 */
static void edv_image_browser_op_animation_previous(EDVImageBrowser *imbr)
{
	gint		frame_num,
			nframes;
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;
	nframes = ImgViewGetTotalFrames(iv);
	if(nframes <= 1)
		return;

	frame_num = ImgViewGetCurrentFrame(iv) - 1;
	if(frame_num < 0)
		frame_num = nframes - 1;

	ImgViewSeek(
		iv,
		frame_num
	);
}

/*
 *	Next frame.
 */
static void edv_image_browser_op_animation_next(EDVImageBrowser *imbr)
{
	gint		frame_num,
			nframes;
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;
	nframes = ImgViewGetTotalFrames(iv);
	if(nframes <= 1)
		return;

	frame_num = ImgViewGetCurrentFrame(iv) + 1;
	if(frame_num >= nframes)
		frame_num = 0;

	ImgViewSeek(
		iv,
		frame_num
	);
}


/*
 *	Rotate/transform menu.
 */
static void edv_image_browser_op_rotate_transform_menu(
	EDVImageBrowser *imbr,
	ToolBarItem *item
)
{
	if(imbr == NULL)
		return;

	edv_map_menu_to_button(
		imbr->rotate_transform_menu,
		ToolBarItemGetWidget(item)
	);
}

/*
 *	Rotate image clockwise 90 degrees.
 */
static void edv_image_browser_op_rotate_cw90(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewRotateCW90(iv);
}

/*
 *	Rotate image counter-clockwise 90 degrees.
 */
static void edv_image_browser_op_rotate_ccw90(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewRotateCCW90(iv);
}

/*
 *	Rotate image clockwise 180 degrees.
 */
static void edv_image_browser_op_rotate_cw180(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewRotateCW180(iv);
}


/*
 *	Mirror horizontal.
 */
static void edv_image_browser_op_mirror_horizontal(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewMirrorHorizontal(iv);
}

/*
 *	Mirror vertical.
 */
static void edv_image_browser_op_mirror_vertical(EDVImageBrowser *imbr)
{
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	iv = imbr->imgview;

	ImgViewMirrorVertical(iv);
}


/*
 *	List Filter.
 */
static void edv_image_browser_op_list_filter(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	if(edv_query_name_filter(
		imbr->core,
		&imbr->thumbs_list_filter,
		imbr->toplevel
	))
	{
		edv_image_browser_set_title(
			imbr,
			edv_image_browser_get_location(imbr)
		);
		edv_image_browser_op_refresh(imbr);
	}

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	MIME Types.
 */
static void edv_image_browser_op_mime_types(EDVImageBrowser *imbr)
{
	gint i;
	gchar *type_string = NULL;
	GList *glist;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	glist = tlist->selection_end;
	i = (glist != NULL) ? (gint)glist->data : -1;
	if(i > -1)
	{
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, i)
		);
		if(obj != NULL)
		{
			(void)edv_match_object_type_string(
				core->mime_types_list,
				obj->type,
				obj->path,
				obj->permissions,
				&type_string
			);
		}
	}
	else
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(edv_image_browser_get_location(imbr));
		if(obj != NULL)
		{
			(void)edv_match_object_type_string(
				core->mime_types_list,
				obj->type,
				obj->path,
				obj->permissions,
				&type_string
			);
			edv_vfs_object_delete(obj);
		}
	}

	edv_map_mime_types(
		core,
		type_string,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(type_string);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Devices.
 */
static void edv_image_browser_op_devices(EDVImageBrowser *imbr)
{
	gint i;
	gchar *path = NULL;
	GList *glist;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	glist = tlist->selection_end;
	i = (glist != NULL) ? (gint)glist->data : -1;
	if(i > -1)
	{
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, i)
		);
		if(obj != NULL)
			path = STRDUP(obj->path);
	}
	else
	{
		path = STRDUP(edv_image_browser_get_location(imbr));
	}

	edv_map_devices(
		core,
		path,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(path);

	edv_image_browser_set_busy(imbr, FALSE);
}
