#include <errno.h>
#if defined(HAVE_REGEX)
# include <regex.h>
#else
# include <fnmatch.h>
#endif
#include <gtk/gtk.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "fprompt.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_property_image.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_date_format.h"
#include "edv_image_io.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_confirm.h"
#include "edv_status_bar.h"
#include "image_browser.h"
#include "image_browser_cb.h"
#include "image_browser_list.h"
#include "edv_vfs_obj_op.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVImbrTListFPromptData	EDVImbrTListFPromptData;
#define EDV_IMAGE_BROWSER_TLIST_FPROMPT_DATA(p)	((EDVImbrTListFPromptData *)(p))


/* Image Loading */
static guint8 *edv_image_browser_list_openImageRGBAThumb(
	EDVImageBrowser *imbr,
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *orig_width_rtn, gint *orig_height_rtn,
	gint *orig_nframes_rtn,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,		/* In seconds since EPOCH */
	const gboolean no_enlarge
);

static void EDVImbrTListSetThumbValues(
	EDVImageBrowser *imbr,
	const gint thumb_num,
	EDVVFSObject *obj
);

static gint EDVImbrTListAppendObject(
	EDVImageBrowser *imbr,
	EDVVFSObject *obj
);

/* Finding */
gint edv_image_browser_list_find_by_index(
	EDVImageBrowser *imbr,
	const gulong device_index,
	const gulong index
);
gint edv_image_browser_list_find_by_path(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* Selection */
static GList *EDVImbrTListGetSelectedPaths(EDVImageBrowser *imbr);

/* Realize Listing */
void edv_image_browser_list_realize_listing(EDVImageBrowser *imbr);

/* Get Listing */
void edv_image_browser_list_get_listing(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gboolean show_progress
);
void edv_image_browser_list_clear(EDVImageBrowser *imbr);

/* Thumb Image Load Iteration */
gint edv_image_browser_list_load_iterate(
	EDVImageBrowser *imbr,
	const gboolean show_progress,
	const gboolean no_enlarge
);

/* Opening */
void edv_image_browser_list_open(
	EDVImageBrowser *imbr,
	const gint thumb_num,
	const guint state			/* Key modifiers */
);
void edv_image_browser_list_open_with(
	EDVImageBrowser *imbr,
	const gint thumb_num
);

/* Renaming */
static void EDVImbrTListFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
static void EDVImbrTListFPromptRenameCancelCB(gpointer data);
void edv_image_browser_list_rename_query(
	EDVImageBrowser *imbr,
	const gint thumb_num
);

/* Object Callbacks */
void edv_image_browser_list_vfs_object_added_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_image_browser_list_vfs_object_modified_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_image_browser_list_vfs_object_removed_cb(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* Mount Callbacks */
void edv_image_browser_list_device_mount_cb(
	EDVImageBrowser *imbr,
	EDVDevice *d,
	const gboolean mounted
);


struct _EDVImbrTListFPromptData {
	EDVImageBrowser	*imbr;
	gint		thumb_num;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/* 
 *	Opens the image data in RGBA suitable display as a thumb in the
 *	thumbs list.
 *
 *	The tlist specifies the thumbs list. If tlist is NULL then
 *	the image will not be resized even if resize_for_thumb is
 *	TRUE.
 *
 *	The path specifies the image to be opened.
 *
 *	The width and height specifies the image geometry return values.
 *
 *	The bpl specifies the bytes per line return value.
 *
 *	The nframes specifies the number of frames return value.
 *
 *	The delay_list specifies the delay GList return value. Each
 *	value in the GList will be a gulong.
 *
 *	If resize_for_thumb is TRUE then the image will be resized
 *	to the thumb size specified by the thumbs list.
 *
 *	If no_enlarge is TRUE then the image will not be enlarged
 *	if it is smalled than the thumb sized specified by the
 *	thumbs list.
 *
 *	The window specifies the reference GdkWindow.
 *
 *	Returns the RGBA image data.
 */
static guint8 *edv_image_browser_list_openImageRGBAThumb(
	EDVImageBrowser *imbr,
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *orig_width_rtn, gint *orig_height_rtn,
	gint *orig_nframes_rtn,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,		/* In seconds since EPOCH */
	const gboolean no_enlarge
)
{
	const gint	bpp = 4;		/* RGBA */
	gint		sbpl = 0,
					swidth = 0, sheight = 0;
	guint8		*rgba,
					orig_bg_color[4] = { 0xff, 0xff, 0xff, 0xff };
	EDVCore *core = imbr->core;
	CfgList *cfg_list = core->cfg_list;

	/* Get the configured total size of the thumb as the
	 * requested size, which may be larger than the exact size
	 * of the thumb to be displayed on the thumbs list, which is
	 * okay since we are resizing the thumb again later because
	 * edv_image_open_rgba_thumb() does not always give us a smaller
	 * sized thumb since it only opens images smaller if the
	 * library and image format supports it
	 */
	const gint	req_width = EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_WIDTH),
					req_height = EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_HEIGHT);

	/* Open the image to RGBA */
	rgba = edv_image_open_rgba_thumb(
		path,
		req_width, req_height,
		&swidth, &sheight,
		&sbpl,
		orig_bg_color,
		orig_width_rtn, orig_height_rtn,
		orig_nframes_rtn,
		play_time_ms_rtn,
		creator_rtn, title_rtn, author_rtn, comments_rtn,
		modified_time_sec_rtn,
		NULL, NULL
	);
	if(rgba != NULL)
	{
		gboolean need_resize;
		gint	twidth, theight,	/* Target/return size */
					tbpl;

		/* Calculate the exact size of the image for the thumb to
		 * be displayed on the thumbs list
		 */
		TListQueryThumbPixmapSize(
			imbr->tlist,
			swidth, sheight,
			&twidth, &theight
		);

		/* Check if we do not want to enlarge if the image is
		 * smaller than the thumb
		 */
		if(no_enlarge)
		{
			if(swidth < twidth)
				twidth = swidth;
			if(sheight < theight)
				theight = sheight;
		}

		/* Check if resizing is nessisary */
		if((swidth != twidth) || (sheight != theight))
			need_resize = TRUE;
		else
			need_resize = FALSE;

		/* Resize? */
		if(need_resize && (twidth > 0) && (theight > 0))
		{
			/* Allocate the exact sized RGBA image data for the
			 * thumb and copy/resize the opened RGBA image data to
			 * the exact sized RGBA image data for the thumb
			 *
			 * Then delete the opened RGBA image data and set the
			 * resized RGBA image data as the return RGBA image
			 * data
			 */
			guint8 *rgba_resized;

			tbpl = twidth * bpp;
			rgba_resized = (guint8 *)g_malloc(
				tbpl * theight * sizeof(guint8)
			);
			if(rgba_resized != NULL)
				GUIImageBufferResize(
					bpp,			/* RGBA */
					rgba,
					swidth, sheight, sbpl,
					rgba_resized,
					twidth, theight, tbpl,
					NULL, NULL
				);

			/* Delete the opened RGBA image data */
			g_free(rgba);

			/* Set the resized RGBA image data as the return RGBA
			 * image data
			 */
			rgba = rgba_resized;
		}
		else
		{
			/* Do not resize the opened RGBA image data, use the
			 * opened RGBA image data's values as the return
			 * values
			 */
			twidth = swidth;
			theight = sheight;
			tbpl = sbpl;
		}

		/* Update the return values */
		*width_rtn = twidth;
		*height_rtn = theight;
		*bpl_rtn = tbpl;
	}

	return(rgba);
}


/*
 *	Sets the Thumb's pixmap and text.
 *
 *	The obj specifies the object to derive the pixmap and text
 *	from.
 *
 *	The thumb_num specifies the thumb.
 */
static void EDVImbrTListSetThumbValues(
	EDVImageBrowser *imbr,
	const gint thumb_num,
	EDVVFSObject *obj
)
{
	const gchar	*name = obj->name,
					*ext;
	GdkColor	*text_color_fg = NULL,
					*text_color_bg = NULL;
	tlist_struct *tlist = imbr->tlist;
	EDVCore *core = imbr->core;
	CfgList *cfg_list = core->cfg_list;
	EDVIconSize icon_size = (EDVIconSize)EDV_GET_I(
		EDV_CFG_PARM_IMBR_THUMB_ICON_SIZE
	);

	if(name == NULL)
		name = "(null)";

	/* Get this object's extension */
	ext = edv_name_get_extension(name);

	TListFreeze(tlist);

	/* Clear all the attribute icons on this thumb */
	TListCleardAttributeIcons(
		tlist,
		thumb_num
	);

	/* Set the attribute icon placement */
	TListSetAttributeIconPlacement(
		tlist,
		thumb_num,
		GTK_CORNER_BOTTOM_RIGHT
	);

	/* Does this object's extension indicate that it is not a
	 * supported image format or is this object a dangling link?
	 */
	if(!edv_image_is_supported_extension(ext) ||
	   (EDV_VFS_OBJECT_IS_LINK(obj) && !EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj))
	)
	{
		/* Not a supported image format or a dangling link */
		EDVPixmap	*icon,
					*icon_inaccessable,
					*icon_hidden;

		/* Get the pixmap and mask for this object's icon to be
		 * used on the thumb
		 */
		(void)edv_match_object_icon(
			core->devices_list,
			core->mime_types_list,
			obj->type,
			obj->path,
			EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
			obj->permissions,
			icon_size,
			&icon,
			NULL,
			&icon_inaccessable,
			&icon_hidden
		);
		/* Use an alternate state icon based on the object's type */
		switch(obj->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
			break;
		  case EDV_OBJECT_TYPE_FILE:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_DIRECTORY:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			/* Directory not accessable? */
			if(!edv_is_object_accessable(core, obj))
			{
				if(edv_pixmap_is_loaded(icon_inaccessable))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_inaccessable);
				}
				if(text_color_fg == NULL)
				{
					const GdkColor cs = EDV_GDKCOLOR_NO_ACCESS_FG;
					text_color_fg = (GdkColor *)g_memdup(
						&cs,
						sizeof(GdkColor)
					);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_LINK:
			/* Link target exists? */
			if(EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj))
			{
				/* Hidden? */
				if(edv_is_object_hidden(obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
			}
			else
			{
				/* Dangling link */
				if(edv_pixmap_is_loaded(icon_inaccessable))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_inaccessable);
				}
				if(text_color_fg == NULL)
				{
					const GdkColor cs = EDV_GDKCOLOR_DANGLING_LINK_FG;
					text_color_fg = (GdkColor *)g_memdup(
						&cs,
						sizeof(GdkColor)
					);
				}
			}
			/* Infinately recursive? */
			if(EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(obj))
			{
				if(text_color_fg == NULL)
				{
					const GdkColor cs = EDV_GDKCOLOR_RECURSIVE_LINK_FG;
					text_color_fg = (GdkColor *)g_memdup(
						&cs,
						sizeof(GdkColor)
					);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_FIFO:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_SOCKET:
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			break;
		  case EDV_OBJECT_TYPE_ERROR:
			break;
 	    }

		/* Parent directory? */
		if(!strcmp((const char *)name, ".."))
		{
			if(edv_pixmap_is_loaded(imbr->directory_parent_icon))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(imbr->directory_parent_icon);
			}
		}

		/* Set this Thumb's pixmap */
		if(edv_pixmap_is_loaded(icon))
			TListSetPixmap(
				tlist,
				thumb_num,
				icon->pixmap, icon->mask
			);

		/* Mark this thumb as loaded so that no image gets
		 * loaded over it during any incidental calls to the
		 * loading process
		 */
		TListSetLoadState(
			tlist,
			thumb_num,
			TLIST_LOAD_STATE_LOADED
		);

		(void)edv_pixmap_unref(icon);
		(void)edv_pixmap_unref(icon_inaccessable);
		(void)edv_pixmap_unref(icon_hidden);
	}
	else
	{
		/* This object name's extension refers to a supported
		 * image
		 */
		EDVPixmap *icon = NULL;

		/* Check if a placeholding icon should be displayed on
		 * this thumb
		 *
		 * Is this object a file or a link? */
		if(EDV_VFS_OBJECT_IS_FILE(obj) || EDV_VFS_OBJECT_IS_LINK(obj))
		{
			/* If this thumb is still marked as loading then set
			 * this thumb's pixmap as the image pending icon
			 */
			switch(TListGetLoadState(tlist, thumb_num))
			{
			  case TLIST_LOAD_STATE_NOT_LOADED:
				icon = edv_pixmap_ref(imbr->file_image_pending_icon);
				break;
			  case TLIST_LOAD_STATE_LOADED:
				break;
			  case TLIST_LOAD_STATE_FAILED:
				icon = edv_pixmap_ref(imbr->file_image_bad_icon);
				break;
			}
		}

		/* Should we set a placeholding icon for this thumb? */
		if(edv_pixmap_is_loaded(icon))
			TListSetPixmap(
				tlist,
				thumb_num,
				icon->pixmap, icon->mask
			);

		/* If this is a link to an image then append the link icon
		 * to the thumb
		 */
		if(EDV_VFS_OBJECT_IS_LINK(obj))
		{
			EDVMIMEType *m = edv_mime_types_list_match_type(
				core->mime_types_list,
				NULL,
				EDV_MIME_TYPE_TYPE_INODE_LINK,
				FALSE
			);
			if(m != NULL)
			{
				EDVPixmap *link_icon;
				edv_mime_type_realize(m, FALSE);
				link_icon = edv_pixmap_ref(m->small_icon[EDV_MIME_TYPE_ICON_STATE_STANDARD]);
				if(edv_pixmap_is_loaded(link_icon))
					TListAppendAttributeIcon(
						tlist,
						thumb_num,
						link_icon->pixmap, link_icon->mask
					);
				(void)edv_pixmap_unref(link_icon);
			}
		}

		/* If this image has multiple frames then append the
		 * multiple frames icon to the thumb
		 */
		if(obj->meta_data_list != NULL)
		{
			const gint nframes = edv_properties_list_get_i(
				obj->meta_data_list,
				"Image Frames"
			);
			if(nframes > 1)
			{
				EDVPixmap *multiple_frames_icon = edv_pixmap_ref(imbr->multiple_frames_icon);
				if(edv_pixmap_is_loaded(multiple_frames_icon))
					TListAppendAttributeIcon(
						tlist,
						thumb_num,
						multiple_frames_icon->pixmap, multiple_frames_icon->mask
					);
				(void)edv_pixmap_unref(multiple_frames_icon);
			}
		}

		(void)edv_pixmap_unref(icon);
	}

	/* Set the thumb's text */
	TListSetText(
		tlist,
		thumb_num,
		name
	);
	TListSetTextColor(
		tlist,
		thumb_num,
		text_color_fg,			/* Foreground color */
		text_color_bg			/* Background color */
	);

	TListThaw(tlist);

	g_free(text_color_fg);
	g_free(text_color_bg);
}

/*
 *	Appends the object to the Thumbs List.
 *
 *	The tlist specifies the Thumbs List.
 *
 *	The obj specifies the object to add to the Thumbs List. It
 *	will be transfered to the list so it should not be referenced
 *	again after this call.
 *
 *	Returns the new thumb index or -1 on error.
 *
 *	All inputs assumed valid.
 */
static gint EDVImbrTListAppendObject(
	EDVImageBrowser *imbr,
	EDVVFSObject *obj
)
{
	gint thumb_num;
	tlist_struct *tlist = imbr->tlist;

	TListFreeze(tlist);

	/* Append the new Thumb */
	thumb_num = TListAppend(tlist, "");
	if(thumb_num < 0)
	{
		TListThaw(tlist);
		edv_vfs_object_delete(obj);
		return(-1);
	}

	/* Set the new Thumb's values */
	TListSetLoadState(
		tlist,
		thumb_num,
		TLIST_LOAD_STATE_NOT_LOADED
	);
	TListSetThumbDataFull(
		tlist,
		thumb_num,
		obj, (GtkDestroyNotify)edv_vfs_object_delete
	);

	/* Set the new Thumb's values */
	EDVImbrTListSetThumbValues(
		imbr,
		thumb_num,
		obj
	);

	TListThaw(tlist);

	return(thumb_num);
}

/*
 *	Finds the thumb who's object matches the location index.
 *
 *	The device_index specifies the index of the device that the
 *	object resides on.
 *
 *	The index specifies the index on the device that the object
 *	resides at.
 *
 *	Returns the matched thumb or negative on error.
 */
gint edv_image_browser_list_find_by_index(
	EDVImageBrowser *imbr, 
	const gulong device_index, const gulong index
)
{
	gint i, n;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return(-2);

	tlist = imbr->tlist;

	n = tlist->total_thumbs;
	for(i = 0; i < n; i++)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
		if(obj == NULL)
			continue;

		/* Location indices match? */
		if((obj->device_index == device_index) &&
		   (obj->index == index)
		)
			return(i);
	}

	return(-1);
}

/*
 *	Finds the thumb who's object matches the path.
 *
 *	The path specifies the full path of the object.
 *
 *	Returns the matched thumb or negative on error.
 */
gint edv_image_browser_list_find_by_path(
	EDVImageBrowser *imbr, const gchar *path
)
{
	gint i, n;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if((imbr == NULL) || (path == NULL))
		return(-2);

	tlist = imbr->tlist;

	n = tlist->total_thumbs;
	for(i = 0; i < n; i++)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, i));
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		/* Full paths match? */
		if(!strcmp((const char *)obj->path, (const char *)path))
			return(i);
	}

	return(-1);
}


/*
 *	Returns a list of full paths describing the selected items.
 *
 *	The calling function must delete the list.
 */
static GList *EDVImbrTListGetSelectedPaths(EDVImageBrowser *imbr)
{
	GList		*glist,
			*paths_list;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return(NULL);

	tlist = imbr->tlist;

	/* Create the selected paths list */
	paths_list = NULL;
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(TListGetThumbData(
			tlist, (gint)glist->data
		));
		if(obj == NULL)
			continue;

		if(STRISEMPTY(obj->path))
			continue;

		paths_list = g_list_append(
			paths_list,
			STRDUP(obj->path)
		);
	}

	return(paths_list);
}


/*
 *	Updates all the existing thumbs on the Thumbs List by getting
 *	each thumb's object and updating the thumb's values with it.
 */
void edv_image_browser_list_realize_listing(EDVImageBrowser *imbr)
{
	gint i, n;
	tlist_thumb_struct *thumb;
	tlist_struct *tlist;
	EDVVFSObject *obj;
	EDVCore *core;

	if(imbr == NULL)
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	TListFreeze(tlist);

	/* Realize each Thumb */
	n = tlist->total_thumbs;
	for(i = 0; i < n; i++)
	{
		thumb = tlist->thumb[i];
		if(thumb == NULL)
			continue;

		obj = EDV_VFS_OBJECT(thumb->data);
		if(obj == NULL)
			continue;

		/* Update this Thumb's values */
		EDVImbrTListSetThumbValues(
			imbr,
			i,
			obj
		);
	}

	TListThaw(tlist);
}

/*
 *	Deletes all Thumbs in the Thumbs List and gets a new listing
 *	of objects (does not load each Thumb's image) from the location
 *	specified by path.
 */
void edv_image_browser_list_get_listing(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gboolean show_progress
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gboolean	hide_object_hidden,
					hide_object_noaccess,
					hide_object_nonimages;
	const gchar *filter;
	GList *names_list;
	GtkWidget *sb;
	CfgList *cfg_list;
	tlist_struct *tlist;
	EDVCore *core;

	if((imbr == NULL) || STRISEMPTY(path))
		return;

	tlist = imbr->tlist;
	sb = imbr->status_bar;
	core = imbr->core;
	cfg_list = core->cfg_list;

	hide_object_hidden = !EDV_GET_B(
		EDV_CFG_PARM_IMBR_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDV_GET_B(
		EDV_CFG_PARM_IMBR_SHOW_OBJECT_NOACCESS
	);
	hide_object_nonimages = !EDV_GET_B(
		EDV_CFG_PARM_IMBR_SHOW_OBJECT_NONIMAGE
	);

	filter = imbr->thumbs_list_filter;

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
		FALSE : strcmp(filter, "*")
	)
	{
		regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
		if(regcomp(
			regex_filter,
			filter,
#ifdef REG_EXTENDED
			REG_EXTENDED |		/* Use POSIX extended regex */
#endif
			REG_NOSUB		/* Do not report subpattern matches */
		))
		{
			g_free(regex_filter);
			regex_filter = NULL;
		}
	}
	else
	{
		regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter) ?
		TRUE : !strcmp(filter, "*")
	)
		filter = NULL;
#endif

	/* Report the initial progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
"Loading directory contents...",
			FALSE
		);
		edv_status_bar_progress(sb, 0.0f, TRUE);
	}

	/* Clear the Thumbs List */
	TListFreeze(tlist);
	TListClear(tlist);
	TListThaw(tlist);

	/* If showing progress then reset the progress value, otherwise
	 * freeze the Thumbs List since its contents don't need to
	 * be displayed during this update when progress is not shown
	 */
	if(show_progress)
		edv_status_bar_progress(sb, 0.0f, TRUE);
	else
		TListFreeze(tlist);

	/* Add the parent directory */
	if(TRUE)
	{
		gchar *full_path = g_dirname(path);
		EDVVFSObject *obj = edv_vfs_object_lstat(full_path);
		if(obj != NULL)
		{
			/* Set the object's name as the previous directory
			 * notation
			 */
			edv_vfs_object_set_path2(
				obj,
				"..",
				full_path
			);

			/* Append/transfer this object to the listing */
			(void)EDVImbrTListAppendObject(
				imbr,
				obj
			);
		}
		g_free(full_path);
	}

	/* Get the listing of objects at the specified location */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint	nobjs_loaded = 0,
					last_progress_percent = -1;
		const gchar	*name,
					*ext;
		gchar *full_path;
		GList *glist;
		EDVVFSObject *obj;

		/* Report the number of objects being loaded? */
		if(show_progress && (nobjs > 0))
		{
/* Do not print how many objects because some objects may be
 * filtered out?
 */
			gchar *msg = g_strdup_printf(
"Loading directory contents (%i %s)...",
				nobjs,
				(nobjs == 1) ? "object" : "objects"
			);
			edv_status_bar_message(sb, msg, TRUE);
			g_free(msg);
		}

#define UPDATE_PROGRESS	{				\
 if(show_progress && (nobjs > 0)) {			\
  const gint progress_percent = nobjs_loaded *		\
   100 / nobjs;						\
  if(progress_percent > last_progress_percent) {	\
   edv_status_bar_progress(				\
    sb,							\
    (gfloat)progress_percent / 100.0f,			\
    TRUE						\
   );							\
   last_progress_percent = progress_percent;		\
  }							\
 }							\
}

		/* Load the directories */
		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(glist->data);
				glist->data = NULL;
				continue;
			}

			/* Get this object's destination stats */
			obj = edv_vfs_object_stat(full_path);
			if(obj == NULL)
			{
				/* Failed to get the statistics of this object
				 * but do not add it as an error object on
				 * this directory iteration, it will be added
				 * and counted later in the second iteration
				 */
				g_free(full_path);
				continue;
			}

			/* Skip if the destination is not a directory */
			if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_vfs_object_delete(obj);
				g_free(full_path);
				continue;
			}

			edv_vfs_object_delete(obj);

			/* Get this directory's local statistics */
			obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				/* Unable to get the directory's local statistics
				 * so add it as an error object
				 */
				const gint error_code = (gint)errno;
				obj = edv_vfs_object_new_error(
					full_path,
					error_code,
					g_strerror(error_code)
				);
				if(obj != NULL)
					(void)EDVImbrTListAppendObject(imbr, obj);

				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(full_path);
				g_free(glist->data);
				glist->data = NULL;
				continue;
			}

			/* Skip hidden or no access objects? */
			if((hide_object_hidden ?
				edv_is_object_hidden(obj) : FALSE) ||
			   (hide_object_noaccess ?
				!edv_is_object_accessable(core, obj) : FALSE)
			)
			{
				edv_vfs_object_delete(obj);
			}
			else
			{
				/* Append/transfer this object to the listing */
				(void)EDVImbrTListAppendObject(
					imbr,
					obj
				);
			}

			/* Count this object as being loaded */
			nobjs_loaded++;
			UPDATE_PROGRESS
			g_free(full_path);
			g_free(glist->data);
			glist->data = NULL;
		}

		/* Load the other objects
		 *
		 * All the names in the names list should be deleted in
		 * this iteration
		 */
		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			ext = edv_name_get_extension(name);

			/* Filter check */
#if defined(HAVE_REGEX)
			if(regex_filter != NULL)
			{
				if(regexec(
					regex_filter,
					name,
					0, NULL,
					0
				) == REG_NOMATCH)
				{
					/* Count this object as being loaded */
					nobjs_loaded++;
					UPDATE_PROGRESS
					g_free(glist->data);
					continue;
				}
			}
#else
			if(filter != NULL)
			{
				if(fnmatch(filter, name, 0) == FNM_NOMATCH)
				{
					/* Count this object as being loaded */
					nobjs_loaded++;
					UPDATE_PROGRESS
					g_free(glist->data);
					continue;
				}
			}
#endif

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(glist->data);
				continue;
			}

			/* Get this object's local statistics */
			obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				/* Unable to get the object's local statistics
				 * so add it as an error object
				 */
				const gint error_code = (gint)errno;
				obj = edv_vfs_object_new_error(
					full_path,
					error_code,
					g_strerror(error_code)
				);
				if(obj != NULL)
					(void)EDVImbrTListAppendObject(
						imbr,
						obj
					);

				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(full_path);
				g_free(glist->data);
				continue;
			}

			/* Skip hidden or non-image objects? */
			if((hide_object_hidden ?
				edv_is_object_hidden(obj) : FALSE) ||
/*
			   (hide_object_noaccess ?
				!edv_is_object_accessable(core, obj) : FALSE) ||
 */
			   (hide_object_nonimages ?
				!edv_image_is_supported_extension(ext) : FALSE)
			)
			{
				edv_vfs_object_delete(obj);
			}
			else
			{
				/* Append/transfer this object to the listing */
				(void)EDVImbrTListAppendObject(
					imbr,
					obj
				);
			}

			/* Count this object as being loaded */
			nobjs_loaded++;
			UPDATE_PROGRESS
			g_free(full_path);
			g_free(glist->data);
		}

#undef UPDATE_PROGRESS

		/* Delete the directory entries list, each entry should
		 * already be deleted
		 */
		g_list_free(names_list);
	}

#ifdef HAVE_REGEX
	if(regex_filter != NULL)
	{
		regfree(regex_filter);
		g_free(regex_filter);
	}
#endif

	/* If showing progress then reset the progress value, otherwise
	 * thaw the Thumbs List since it was frozen when progress
	 * was not being shown
	 */
	if(show_progress)
		edv_status_bar_progress(sb, 0.0f, TRUE);
	else
		TListThaw(tlist);
}

/*
 *	Deletes all Thumbs in the Thumbs List.
 */
void edv_image_browser_list_clear(EDVImageBrowser *imbr)
{
	tlist_struct *tlist;

	if(imbr == NULL)
		return;

	tlist = imbr->tlist;
	TListFreeze(tlist);
	TListClear(tlist);
	TListThaw(tlist);
}


/*
 *	Loads the Image Browser Thumbs List's next unloaded thumb.
 *
 *	The next unloaded thumb will be the next thumb from the
 *	current scroll position. If there are no unloaded thumbs there
 *	then a second pass will check if there is an unloaded thumb
 *	starting from the top.
 *
 *	Returns:
 *
 *	0	All thumbs loaded.
 *	1	Thumb loaded successfully or failed to load, either
 *		way, continue loading subsequent thumbs.
 *	-1	General error.
 *	-2	Ambiguous, corrupt image, or unsupported format.
 *	-3	Systems error.
 */
gint edv_image_browser_list_load_iterate(
	EDVImageBrowser *imbr,
	const gboolean show_progress,
	const gboolean no_enlarge
)
{
	gint		thumb_num,
					starting_thumb_num,
					thumb_to_load_num = -1,
					thumbs_already_loaded = 0;
	GtkWidget	*toplevel,
					*sb;
	CfgList *cfg_list;
	tlist_thumb_struct	*thumb,
							*thumb_to_load = NULL;
	tlist_struct *tlist;
	EDVVFSObject *obj;
	EDVCore *core;

#define RESET_PROGRESS	{			\
 if(show_progress) {				\
  edv_status_bar_progress(sb, 0.0f, FALSE);	\
  edv_status_bar_message(sb, NULL, FALSE);		\
 }						\
}

	if(imbr == NULL)
		return(-1);

	imbr->freeze_count++;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	sb = imbr->status_bar;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Count the number of thumbs already loaded */
	if(show_progress)
	{
		for(thumb_num = 0;
			thumb_num < tlist->total_thumbs;
			thumb_num++
		)
		{
			thumb = tlist->thumb[thumb_num];
			if(thumb == NULL)
			{
				thumbs_already_loaded++;	/* Count NULL's as loaded */
				continue;
			}

			/* Is this thumb loaded or failed to load? */
			if(thumb->load_state != TLIST_LOAD_STATE_NOT_LOADED)
				thumbs_already_loaded++;
		}
	}

	/* Calculate the first starting thumb at the current scroll
	 * position
	 */
	(void)TListGetSelection(
		tlist,
		0, 0,
		&starting_thumb_num,
		NULL, NULL
	);

	/* Look for the thumb to load starting from the scroll
	 * position to the end of the list
	 */
	if(starting_thumb_num > -1)
	{
		for(thumb_num = starting_thumb_num;
			thumb_num < tlist->total_thumbs;
			thumb_num++
		)
		{
			thumb = tlist->thumb[thumb_num];
			if(thumb == NULL)
				continue;

			/* Is this thumb not loaded? */
			if(thumb->load_state == TLIST_LOAD_STATE_NOT_LOADED)
			{
				/* Select this thumb to load */
				thumb_to_load = thumb;
				thumb_to_load_num = thumb_num;
				break;
			}
		}
	}
	/* If no thumbs need to be loaded starting from the
	 * starting_thumb_num then start loading from thumb 0
	 */
	if((thumb_to_load == NULL) &&
	   (starting_thumb_num <= tlist->total_thumbs)
	)
	{
		for(thumb_num = 0; thumb_num < starting_thumb_num; thumb_num++)
		{
			thumb = tlist->thumb[thumb_num];
			if(thumb == NULL)
				continue;

			/* Is this thumb not loaded? */
			if(thumb->load_state == TLIST_LOAD_STATE_NOT_LOADED)
			{
				/* Select this thumb to load */
				thumb_to_load = thumb;
				thumb_to_load_num = thumb_num;
				break;
			}
		}
	}

	/* No more thumbs need to be loaded? */
	if((thumb_to_load == NULL) || (thumb_to_load_num < 0))
	{
		if(show_progress)
		{
			edv_status_bar_message(
				sb,
"Loading done",
				FALSE
			);
			edv_status_bar_progress(sb, 0.0f, FALSE);
		}
		imbr->freeze_count--;
		return(0);				/* All thumbs loaded */
	}

	/* Get the object from the thumb data */
	obj = EDV_VFS_OBJECT(thumb_to_load->data);
	if(obj == NULL)
	{
		/* Mark that this thumb has failed to load and set the
		 * "bad image" icon on it
		 */
		TListFreeze(tlist);
		TListSetLoadState(
			tlist,
			thumb_to_load_num,
			TLIST_LOAD_STATE_FAILED
		);
		EDVImbrTListSetThumbValues(
			imbr,
			thumb_to_load_num,
			obj
		);
		TListThaw(tlist);
		imbr->freeze_count--;
		return(1);				/* Thumb failed to load but
											 * keep loading subsequent
											 * thumbs */
	}
	if(STRISEMPTY(obj->path))
	{
		/* Mark that this thumb has failed to load and set the
		 * "bad image" icon on it
		 */
		TListFreeze(tlist);
		TListSetLoadState(
			tlist,
			thumb_to_load_num,
			TLIST_LOAD_STATE_FAILED
		);
		EDVImbrTListSetThumbValues(
			imbr,
			thumb_to_load_num,
			obj
		);
		TListThaw(tlist);
		imbr->freeze_count--;
		return(1);				/* Thumb failed to load but
											 * keep loading subsequent
											 * thumbs */
	}

	/* Load the image for this thumb */
	if(TRUE)
	{
		gint	width, height,
					bpl,
					orig_width, orig_height,
					orig_nframes;
		gulong	play_time_ms,
					modified_time_sec;
		const gchar	*name = obj->name,
					*full_path = obj->path;
		gchar	*creator,
					*title,
					*author,
					*comments,
					*modified_time_sec_str;
		guint8 *rgba;

		if(show_progress)
		{
			const gulong size = obj->size;
			const gfloat progress = (tlist->total_thumbs > 0) ?
				((gfloat)thumbs_already_loaded / (gfloat)tlist->total_thumbs) : 0.0f;
			gchar *msg = g_strdup_printf(
"Loading \"%s\" (%s %s)...",
				name,
				edv_str_size_delim(size),
				(size == 1l) ? "byte" : "bytes"
			);
			edv_status_bar_message(sb, msg, FALSE);
			g_free(msg);

			edv_status_bar_progress(sb, progress, TRUE);
		}

/* Check if the image is cached */

		/* Open the image in RGBA and resize it to fit the thumb */
		rgba = edv_image_browser_list_openImageRGBAThumb(
			imbr,
			full_path,
			&width, &height,
			&bpl,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&creator, &title, &author, &comments,
			&modified_time_sec,
			no_enlarge
		);
		if(rgba == NULL)
		{
			/* Mark that this thumb has failed to load and set the
			 * "bad image" icon on it
			 */
			TListFreeze(tlist);
			TListSetLoadState(
				tlist,
				thumb_to_load_num,
				TLIST_LOAD_STATE_FAILED
			);
			EDVImbrTListSetThumbValues(
				imbr,
				thumb_to_load_num,
				obj
			);
			TListThaw(tlist);
			g_free(creator);
			g_free(title);
			g_free(author);
			g_free(comments);
			imbr->freeze_count--;
			return(1);			/* Thumb failed to load but
											 * keep loading subsequent
											 * thumbs */
		}

		/* Update the object's image property values */
		if(modified_time_sec > 0l)
		{
			modified_time_sec_str = edv_date_string_format(
				modified_time_sec,
				EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT),
				EDV_GET_I(EDV_CFG_PARM_DATE_RELATIVITY)
			);
		}
		else
		{
			modified_time_sec_str = NULL;
		}
		obj->meta_data_list = edv_properties_list_image_set(
			obj->meta_data_list,
			orig_width, orig_height,
			orig_nframes,
			play_time_ms,
			creator, author, title, comments,
			modified_time_sec_str
		);
		g_free(modified_time_sec_str);


		TListFreeze(tlist);

		/* Mark that this thumb has been loaded successfully */
		TListSetLoadState(
			tlist,
			thumb_to_load_num,
			TLIST_LOAD_STATE_LOADED
		);

		/* Set the image to the thumb */
		TListSetRGBA(
			tlist,
			thumb_to_load_num,
			width, height,
			bpl,
			GDK_RGB_DITHER_NORMAL,
			rgba,
			no_enlarge
		);

		/* Update the thumb's values after loading the image */
		EDVImbrTListSetThumbValues(
			imbr,
			thumb_to_load_num,
			obj
		);

		TListThaw(tlist);

		/* Delete the opened image data */
		g_free(rgba);

		g_free(creator);
		g_free(title);
		g_free(author);
		g_free(comments);

		imbr->freeze_count--;

		return(1);				/* Thumb loaded, keep loading
											 * subsequent thumbs */
	}

	imbr->freeze_count--;

	return(0);				/* All thumbs loaded */
}


/*
 *	Opens the object.
 *
 *	If thumb_num is -1 then the selected object(s) will be opened.
 *
 *	The state specifies the current key modifiers.
 */
void edv_image_browser_list_open(
	EDVImageBrowser *imbr, const gint thumb_num,
	const guint state			/* Key modifiers */
)
{
	gint npaths;
	const gchar *path;
	GList		*glist,
			*paths_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Use selected objects? */
	if(thumb_num < 0)
	{
		paths_list = EDVImbrTListGetSelectedPaths(imbr);
	}
	else
	{
		tlist_struct *tlist = imbr->tlist;
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, thumb_num)
		);
		if(obj == NULL)
			return;

		if(STRISEMPTY(obj->path))
			return;

		paths_list = NULL;
		paths_list = g_list_append(
			paths_list,
			g_strdup(obj->path)
		);
	}

	/* Count the number of paths in the list and get the path to
	 * the last path in the list
	 */
	npaths = 0;
	path = NULL;
	glist = paths_list;
	while(glist != NULL)
	{
		npaths++;
		path = (const gchar *)glist->data;
		glist = g_list_next(glist);
	}

	/* Check if there is only one object selected and that it
	 * refers to a directory
	 */
	if(npaths == 1)
	{
		gchar *path = edv_path_evaluate(
			NULL,				/* No parent */
			(const gchar *)paths_list->data
		);
		if(edv_path_is_directory(path))
		{
			/* Change directory and return */
			edv_image_browser_goto_directory_cb(imbr, path);
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
			g_free(path);
			return;
		}
		g_free(path);
	}

	/* Confirm open */
	if(edv_confirm_open(
		core,
		toplevel,
		path,
		npaths
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		return;
	}

	/* Open */
	if(paths_list != NULL)
	{
		const gchar *command_name = NULL;

		edv_image_browser_set_busy(imbr, TRUE);

		if(state & GDK_CONTROL_MASK)
			command_name = "edit";
		else if(state & GDK_SHIFT_MASK)
			command_name = "edit";

		(void)edv_open(
			core,
			paths_list,
			command_name,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);

		edv_image_browser_set_busy(imbr, FALSE);
	}
}

/*
 *	Maps the Open With list to open the object.
 *
 *	If thumb_num is -1 then the selected object(s) will be opened.
 */
void edv_image_browser_list_open_with(
	EDVImageBrowser *imbr, const gint thumb_num
)
{
	gint npaths;
	const gchar *path;
	GList		*glist,
					*paths_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Use selected objects? */
	if(thumb_num < 0)
	{
		/* Get list of selected object paths */
		paths_list = EDVImbrTListGetSelectedPaths(imbr);
	}
	else
	{
		tlist_struct *tlist = imbr->tlist;
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, thumb_num)
		);
		if(obj == NULL)
			return;

		if(STRISEMPTY(obj->path))
			return;

		paths_list = NULL;
		paths_list = g_list_append(
			paths_list,
			g_strdup(obj->path)
		);
	}

	/* Count the number of paths in the list and get the path to
	 * the last path in the list
	 */
	npaths = 0;
	path = NULL;
	glist = paths_list;
	while(glist != NULL)
	{
		npaths++;
		path = (const gchar *)glist->data;
		glist = g_list_next(glist);
	}

	/* Confirm open */
	if(edv_confirm_open(
		core,
		toplevel,
		path,
		npaths
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		return;
	}

	/* Open With */
	if(paths_list != NULL)
	{
		(void)edv_open_with(
			core,
			paths_list,
			NULL,			/* Default command */
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
	}
}


/*
 *	FPrompt apply callback, set in edv_image_browser_list_rename_query().
 */
static void EDVImbrTListFPromptRenameApplyCB(
	gpointer data, const gchar *value
)
{
	gint thumb_num;
	EDVImageBrowser *imbr;
	EDVImbrTListFPromptData *d = EDV_IMAGE_BROWSER_TLIST_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	imbr = d->imbr;
	thumb_num = d->thumb_num;

	/* Inputs valid? */
	if((imbr != NULL) && (thumb_num > -1) && (value != NULL))
	{
		GtkWidget	*toplevel = imbr->toplevel,
					*sb = imbr->status_bar;
		tlist_struct *tlist = imbr->tlist;
		EDVCore *core = imbr->core;

		/* Get the object from the selected thumb */
		EDVVFSObject *obj = EDV_VFS_OBJECT(TListGetThumbData(
			tlist,
			thumb_num
		));

		/* Check if the selected object is valid */
		if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
		{
			gboolean yes_to_all = FALSE;
			const gchar *error_msg;
			gchar *old_full_path = STRDUP(obj->path);
			GList *modified_paths_list;

			/* Rename */
			edv_vfs_object_op_rename(
				core,
				old_full_path, value,
				&modified_paths_list,
				toplevel,
				FALSE,			/* Do not show progress */
				TRUE,			/* Interactive */
				&yes_to_all
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Rename Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report the modified objects */
			if(modified_paths_list != NULL)
			{
				const gchar *modified_path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = modified_paths_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					modified_path = (const gchar *)glist->data;
					if(modified_path == NULL)
						continue;

					obj = edv_vfs_object_lstat(modified_path);
					if(obj != NULL)
					{
						gchar *msg = g_strdup_printf(
							"Object \"%s\" renamed to \"%s\"",
							g_basename(old_full_path),
							g_basename(modified_path)
						);
						edv_status_bar_message(sb, msg, FALSE);
						g_free(msg);

						edv_emit_vfs_object_modified(
							core,
							old_full_path,
							modified_path,
							obj
						);
						edv_vfs_object_delete(obj);
					}
				}

				if(modified_paths_list != NULL)
				{
					g_list_foreach(
						modified_paths_list, (GFunc)g_free, NULL
					);
					g_list_free(modified_paths_list);
				}
			}
			else
			{
				/* Did not get the modified object path so this
				 * implies failure
				 */
				edv_status_bar_message(
					sb,
					"Rename object failed",
					FALSE
				);
			}

			g_free(old_full_path);
		}
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void EDVImbrTListFPromptRenameCancelCB(gpointer data)
{
	EDVImbrTListFPromptData *d = EDV_IMAGE_BROWSER_TLIST_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	g_free(d);
}

/*
 *	Prompt the user to rename an object.
 *
 *	The thumb_num specifies the thumb index.
 */
void edv_image_browser_list_rename_query(
	EDVImageBrowser *imbr,
	const gint thumb_num
)
{
	gint cx, cy, px, py, pwidth, pheight;
	GtkWidget *toplevel, *w;
	tlist_struct *tlist;
	EDVVFSObject *obj;
	EDVCore *core;

	if((imbr == NULL) || (thumb_num < 0) || FPromptIsQuery())
		return;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	edv_image_browser_sync_data(imbr);

	/* Make sure given thumb index is in bounds */
	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return;

	pwidth = tlist->thumb_width;
	pheight = 20;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	w = TListGetListWidget(tlist);
	if(w != NULL)
		gdk_window_get_deskrelative_origin(
			w->window,
			&px, &py
		);
	if(TListGetThumbPosition(
		tlist,
		thumb_num,
		&cx, &cy
	))
		px += cx;
	if(TListGetThumbLabelGeometry(
		tlist,
		thumb_num,
		&cx, &cy,
		NULL, NULL
	))
		py += cy - 4;

	/* Get object from thumb data */
	obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
	if(obj == NULL)
		return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(obj->name != NULL)
	{
		const gchar *name = obj->name;
		if(!strcmp(name, ".") || !strcmp(name, "..") ||
		   !strcmp(name, "/")
		)
			return;
	}

	if(TRUE)
	{
		gchar *value = STRDUP(obj->name);
		EDVImbrTListFPromptData *d = EDV_IMAGE_BROWSER_TLIST_FPROMPT_DATA(
			g_malloc(sizeof(EDVImbrTListFPromptData))
		);
		if(d != NULL)
		{
			d->imbr = imbr;
			d->thumb_num = thumb_num;
		}

		/* Map floating prompt to change values */
		FPromptSetTransientFor(toplevel);
		FPromptSetPosition(px, py);
		FPromptMapQuery(
			NULL,			/* No label */
			value,			/* Initial value */
			NULL,			/* No tooltip message */
			FPROMPT_MAP_TO_POSITION,	/* Map code */
			pwidth, -1,		/* Width and height */
			0,			/* Flags */
			d,			/* Callback data */
			NULL,			/* No browse callback */
			EDVImbrTListFPromptRenameApplyCB,
			EDVImbrTListFPromptRenameCancelCB
		);

		g_free(value);
	}
}


/*
 *	Object added callback.
 */
void edv_image_browser_list_vfs_object_added_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gint thumb_num;
	gchar		*cur_path,
					*parent_path;
	CfgList *cfg_list;
	tlist_struct *tlist;
	EDVCore *core;

	if((imbr == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	if(cur_path == NULL)
		return;

	tlist = imbr->tlist;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Is the added path the same as the current location? */
	if(!strcmp((const char *)path, (const char *)cur_path))
	{
		/* Reget the listing */
		edv_image_browser_list_get_listing(
			imbr,
			path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
		g_free(cur_path);
		return;
	}

	/* Update the object who's path matches the added path */
	thumb_num = edv_image_browser_list_find_by_path(imbr, path);
	if(thumb_num > -1)
	{
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, thumb_num)
		);
		if(tar_obj != NULL)
		{
			edv_vfs_object_set_object(tar_obj, obj);

			/* Set this thumb's load state to not loaded so that
			 * EDVImbrTListSetThumbValues() will set its pixmap
			 * back to the image pending pixmap as needed,
			 * EDVImbrTListSetThumbValues() will then set the load
			 * as appropriate
			 */
			TListSetLoadState(
				tlist,
				thumb_num,
				TLIST_LOAD_STATE_NOT_LOADED
			);
			EDVImbrTListSetThumbValues(
				imbr,
				thumb_num,
				tar_obj
			);

			/* Need to (re)queue loading process so that the
			 * modified thumb gets its image loaded
			 */
			edv_image_browser_queue_loading_process(imbr);
		}
		g_free(cur_path);
		return;
	}

	/* Get the parent path of the added path */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
		/* Is the parent directory of the added object the same as
		 * the current location?
		 */
		if(!strcmp((const char *)parent_path, (const char *)cur_path))
		{
			/* Add this object to the list */
			const gint new_thumb_num = EDVImbrTListAppendObject(
				imbr,
				edv_vfs_object_copy(obj)
			);
			if(new_thumb_num > -1)
			{
				/* Need to (re)queue loading process so that
				 * the added thumb gets its image loaded
				 */
				edv_image_browser_queue_loading_process(imbr);
			}
		}
		g_free(parent_path);
	}

	g_free(cur_path);
}

/*
 *	Object modified callback.
 */
void edv_image_browser_list_vfs_object_modified_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gint thumb_num;
	gchar *cur_path;
	CfgList *cfg_list;
	tlist_struct *tlist;
	EDVCore *core;

	if((imbr == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	if(cur_path == NULL)
		return;

	tlist = imbr->tlist;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Is the new modified path the same as the current location? */
	if(!strcmp((const char *)new_path, (const char *)cur_path))
	{
		/* Reget the listing */
		edv_image_browser_list_get_listing(
			imbr,
			new_path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
		g_free(cur_path);
		return;
	}

	/* Update the object who's path matches the modified path */
	thumb_num = edv_image_browser_list_find_by_path(imbr, path);
	if(thumb_num > -1)
	{
		/* Update this object */
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			TListGetThumbData(tlist, thumb_num)
		);
		if(tar_obj != NULL)
		{
			edv_vfs_object_set_object(tar_obj, obj);

			/* Set this thumb's load state to not loaded so that
			 * EDVImbrTListSetThumbValues() will set its pixmap
			 * back to the image pending pixmap as needed,
			 * EDVImbrTListSetThumbValues() will then set the load
			 * as appropriate
			 */
			TListSetLoadState(
				tlist,
				thumb_num,
				TLIST_LOAD_STATE_NOT_LOADED
			);
			EDVImbrTListSetThumbValues(
				imbr,
				thumb_num,
				tar_obj
			);

			/* Need to (re)queue loading process so that the
			 * modified thumb gets its image loaded
			 */
			edv_image_browser_queue_loading_process(imbr);
		}
	}

	g_free(cur_path);
}

/*
 *	Object removed callback.
 */
void edv_image_browser_list_vfs_object_removed_cb(
	EDVImageBrowser *imbr,
	const gchar *path
)
{
	gchar *cur_path;
	tlist_struct *tlist;

	if((imbr == NULL) || STRISEMPTY(path))
		return;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	if(cur_path == NULL)
		return;

	tlist = imbr->tlist;

	/* Is the removed path the same as the current location? */
	if(!strcmp((const char *)path, (const char *)cur_path))
	{
		/* Clear the thumbs list */
		TListFreeze(tlist);
		TListClear(tlist);
		TListThaw(tlist);
	}
	else
	{
		/* Remove all the thumbs who's object's path matches the
		 * removed path
		 */
		gint thumb_num = edv_image_browser_list_find_by_path(imbr, path);
		if(thumb_num > -1)
		{
			TListFreeze(tlist);
			do {
				TListRemove(tlist, thumb_num);
				thumb_num = edv_image_browser_list_find_by_path(imbr, path);
			} while(thumb_num > -1);
			TListThaw(tlist);
		}
	}

	g_free(cur_path);
}


/*
 *	Device mount/unmount callback.
 */
void edv_image_browser_list_device_mount_cb(
	EDVImageBrowser *imbr,
	EDVDevice *d,
	const gboolean mounted
)
{
	gchar		*cur_path,
					*mount_path;
	GtkWidget *toplevel;
	EDVCore *core;

	if((imbr == NULL) || (d == NULL))
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	mount_path = STRDUP(d->mount_path);
	if((cur_path == NULL) || (mount_path == NULL))
	{
		g_free(cur_path);
		g_free(mount_path);
		return;
	}

	edv_path_simplify(cur_path);
	edv_path_simplify(mount_path);

	/* Check if the current location is within the mount path */
	if(strpfx(cur_path, mount_path))
	{
		/* Switch to the mount location, reget the listing, and
		 * restart the loading procedure
		 */
		edv_image_browser_set_busy(imbr, TRUE);
		GUIBlockInput(toplevel, TRUE);
		edv_image_browser_select_path(imbr, mount_path);
		GUIBlockInput(toplevel, FALSE);
		edv_image_browser_set_busy(imbr, FALSE);
	}

	g_free(cur_path);
	g_free(mount_path);
}
