#include <gtk/gtk.h>

#include "url.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "image_browser.h"
#include "image_browser_dnd.h"
#include "edv_dnd.h"

#include "config.h"


/* Drag Data Received Nexus */
static gint edv_image_brwoser_drag_data_receieved_nexus(
	EDVImageBrowser *imbr,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVVFSObject *tar_obj
);

/* Location Bar Callbacks */
gint edv_image_browser_location_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* DND Drag Icon */
void edv_image_browser_list_dnd_set_drag_icon(EDVImageBrowser *imbr);

/* DND Callbacks */
void edv_image_browser_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
void edv_image_browser_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
void edv_image_browser_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void edv_image_browser_view_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Drag data received nextus.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	The tar_obj specifies the target location. If tar_obj is NULL
 *	then the current location will be used as the target location.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_image_brwoser_drag_data_receieved_nexus(
	EDVImageBrowser *imbr,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVVFSObject *tar_obj
)
{
	gint status;
	EDVCore *core = imbr->core;
	gchar *tar_path = (tar_obj != NULL) ? STRDUP(tar_obj->path) : NULL;
	if(tar_path == NULL)
		tar_path = STRDUP(edv_image_browser_get_location(imbr));
	if(tar_path == NULL)
		return(-3);

	edv_image_browser_set_busy(imbr, TRUE);

	/* Check if the object is an archive, and if so, then add to
	 * the archive isntead
	 */
	if(edv_check_archive_format_supported(core, g_basename(tar_path)))
	{
		gchar *password = NULL;

		/* Process the DND operation to the VFS */
		status = edv_dnd_vfs_to_archive(
			core,
			dc,
			info,			/* Source location type */
			selection_data,		/* Source objects */
			tar_path,		/* Target archive path */
			&password,		/* May be modified */
			TRUE,			/* Verbose */
			imbr->toplevel,
			imbr->status_bar
		);

		g_free(password);
	}
	else
	{
		/* Process a DND operation to the VFS */
		status = edv_dnd_any_to_vfs(
			core,
			dc,
			info,			/* Source location type */
			selection_data,		/* Source objects */
			tar_path,		/* Target location */
			TRUE,			/* Verbose */
			imbr->toplevel,
			imbr->status_bar
		);
	}

	g_free(tar_path);

	edv_image_browser_set_busy(imbr, FALSE);

	return(status);
}


/*
 *	Location Bar icon "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
gint edv_image_browser_location_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	GtkWidget *sb;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (crossing == NULL) || (imbr == NULL))
		return(status);

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return(status);

	sb = imbr->status_bar;

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
		if(imbr->location_icon_pm != NULL)
		{
			GdkBitmap *mask;
			GdkPixmap *pixmap;
			GtkWidget *w = imbr->location_icon_pm;
			gtk_pixmap_get(GTK_PIXMAP(w), &pixmap, &mask);
			if(pixmap != NULL)
				GUIDNDSetDragIcon(
					widget,
					pixmap, mask,
					GUI_DND_DRAG_ICON_DEF_HOT_X, GUI_DND_DRAG_ICON_DEF_HOT_Y
				);
		}
		edv_status_bar_message(
			sb,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, eine verbindung zu diesem ort zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a questa posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze plaats te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a esta localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til denne plasseringen"
#else
"Drag this to create a link to this location"
#endif
			, FALSE
		);
		status = TRUE;
		break;

	  case GDK_LEAVE_NOTIFY:
		edv_status_bar_message(sb, NULL, FALSE);
		status = TRUE;
		break;
	}

	return(status);
}


/*
 *	Sets the DND drag icon.
 */
void edv_image_browser_list_dnd_set_drag_icon(EDVImageBrowser *imbr)
{
	const gint nitems_visible = GUI_DND_DRAG_ICON_DEF_NITEMS_VISIBLE;
	gint i;
	GList		*glist,
			*cells_list;
	GdkFont *font;
	GtkStyle *style;
	GtkWidget *w;
	GtkCellPixText *cell_pixtext;
	tlist_thumb_struct *thumb;
	tlist_struct *tlist;

	if(imbr == NULL)
		return;

	tlist = imbr->tlist;
	w = TListGetListWidget(tlist);
	style = gtk_widget_get_style(w);
	font = style->font;

	/* Create a GtkCells list from the selected objects */
	cells_list = NULL;
	for(glist = tlist->selection, i = 0;
	    (glist != NULL) && (i < nitems_visible);
	    glist = g_list_next(glist), i++
	)
	{
		thumb = TListGetThumb(
			tlist,
			(gint)glist->data
		);
		if(thumb == NULL)
			continue;

		cell_pixtext = (GtkCellPixText *)g_malloc0(
			sizeof(GtkCellPixText)
		);
		if(cell_pixtext != NULL)
		{
			cell_pixtext->type = GTK_CELL_PIXTEXT;
#if 0
/* Does not work */
			if(thumb->text != NULL)
			{
				const gint max_width = MAX(
					(tlist->thumb_width -
						(2 * tlist->thumb_border)),
					1
				);
				gint	lbearing, rbearing, swidth = 0,
					ascent, descent;
				gdk_string_extents(
					font, thumb->text,
					&lbearing, &rbearing, &swidth, &ascent, &descent
				);
				if(swidth >= max_width)
				{
					const gint len = strlen(thumb->text);
					/* coeff will be in the range of (0.0 to 1.0] */
					const gfloat coeff = (gfloat)max_width / (gfloat)swidth;
					cell_pixtext->text = thumb->text + (gint)(coeff * len);
				}
				else
					cell_pixtext->text = thumb->text;
			}
#else
			cell_pixtext->text = thumb->text;
#endif
			cell_pixtext->spacing = EDV_LIST_PIXMAP_TEXT_SPACING;
			cell_pixtext->pixmap = thumb->pixmap;
			cell_pixtext->mask = thumb->mask;
			cells_list = g_list_append(
				cells_list,
				cell_pixtext
			);
		}
	}
	if(cells_list != NULL)
	{
		/* Set the DND drag icon from the GtkCells list */
		GUIDNDSetDragIconFromCellsList(
			w,
			cells_list,
			g_list_length(tlist->selection),
			GTK_ORIENTATION_HORIZONTAL,	/* Layout orientation */
			GTK_ORIENTATION_VERTICAL,	/* Icon and text orientation */
			MAX((2 * tlist->thumb_border), 0)	/* Item spacing */
		);

		/* Delete the GtkCells list */
		g_list_foreach(
			cells_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(cells_list);
	}
}


/*
 *	Thumbs List "drag_data_get" signal callback.
 */
void edv_image_browser_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint thumb_num;
	EDVCore *core;
	GList *glist, *urls_list = NULL;
	URLStruct *url;
	tlist_struct *tlist;
	EDVVFSObject *obj;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (dc == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	edv_image_browser_sync_data(imbr);

	/* Generate a list of URLs from the selected thumbs */
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		thumb_num = (gint)glist->data;
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
		if(obj == NULL)
			continue;

		url = url_new();
		url->path = STRDUP(obj->path);
		urls_list = g_list_append(urls_list, url);
	}

	/* Encode DDE buffer from the URL list */
	if(urls_list != NULL)
	{
		gint buf_len;
		guint8 *buf = url_encode(urls_list, &buf_len);
		if(buf != NULL)
		{
			/* Send out DND data buffer */ 
			gtk_selection_data_set(
				selection_data,
				GDK_SELECTION_TYPE_STRING,
				8,			/* Bits Per Character */
				buf,		/* Data */
				buf_len		/* Length */
			);
			data_sent = TRUE;
			g_free(buf);
		}
	}

	/* Delete the URLs list */
	if(urls_list != NULL)
	{
		g_list_foreach(urls_list, (GFunc)url_delete, NULL);
		g_list_free(urls_list);
	}

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
		const gchar *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,			/* Bits Per Character */
			s,			/* Data */
			STRLEN(s)		/* Length */
		);
		data_sent = TRUE;
	}
}

/*
 *	Thumbs List "drag_data_received" signal callback.
 */
void edv_image_browser_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	gint thumb_num = -1;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((dc == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	core = imbr->core;

	edv_image_browser_sync_data(imbr);

	/* Find the thumb that this drop occured on, it is okay if no
	 * thumb is matched in which case thumb_num is -1 and obj will
	 * be NULL, in which case the current location will be used
	 */
	if(!TListGetSelection(
		tlist,
		x, y,
		&thumb_num,
		NULL, NULL
	))
		thumb_num = -1;

	obj = EDV_VFS_OBJECT(TListGetThumbData(
		tlist,
		thumb_num
	));

	/* Perform the drag operation */
	(void)edv_image_brwoser_drag_data_receieved_nexus(
		imbr,
		dc,
		info,
		selection_data,
		obj
	);
}

/*
 *	Thumbs List "drag_data_delete" signal callback.
 */
void edv_image_browser_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}

/*
 *	ImgView "drag_data_received" signal callback.
 */
void edv_image_browser_view_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	GtkWidget *toplevel;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((dc == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	edv_image_browser_sync_data(imbr);

	/* Perform the drag operation */
	(void)edv_image_brwoser_drag_data_receieved_nexus(
		imbr,
		dc,
		info,
		selection_data,
		NULL				/* Use the current location */
	);
}
