#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_device.h"
#include "edv_device_mount.h"
#include "edv_devices_list.h"
#include "edv_utils_gtk.h"
#include "edv_list_seek.h"
#include "edv_status_bar.h"
#include "image_browser.h"
#include "image_browser_cb.h"
#include "image_browser_op.h"
#include "image_browser_list.h"
#include "image_browser_view.h"
#include "image_browser_dnd.h"
#include "edv_vfs_obj_create.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_endeavour_image_browser_48x48.xpm"


/* GTK+ Signal Callbacks */
void edv_image_browser_realize_cb(GtkWidget *widget, gpointer data);
gint edv_image_browser_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint edv_image_browser_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint edv_image_browser_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void edv_image_browser_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void edv_image_browser_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void edv_image_browser_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data);

void edv_image_browser_list_realize_cb(GtkWidget *widget, gpointer data);
void edv_image_browser_list_select_cb(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);
void edv_image_browser_list_unselect_cb(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);

void edv_image_browser_combo_activate_cb(GtkWidget *widget, gpointer data);

gint edv_image_browser_load_thumbs_passive_timeout_cb(gpointer data);

/* Menu Item Callbacks */
void edv_image_browser_menu_item_cb(GtkWidget *widget, gpointer data);
gint edv_image_browser_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_image_browser_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
void edv_image_browser_menu_item_command_object_op_cb(GtkWidget *widget, gpointer data);
void edv_image_browser_menu_item_command_new_object_cb(GtkWidget *widget, gpointer data);
gint edv_image_browser_menu_item_command_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_image_browser_menu_item_command_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* Paths List Get Callback */
GList *edv_image_browser_get_selected_paths_cb(gpointer data);

/* Go To Directory Callback */
void edv_image_browser_goto_directory_cb(gpointer data, const gchar *path);

/* Mount Bar Callbacks */
void edv_image_browser_mount_bar_mount_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);
void edv_image_browser_mount_bar_eject_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);
void edv_image_browser_mount_bar_goto_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);

/* Find Bar Callbacks */
const gchar *edv_image_browser_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_image_browser_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_image_browser_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
);
void edv_image_browser_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
);

/* Status Bar Callbacks */
void edv_image_browser_status_message_cb(
	GtkWidget *widget,
	const gchar *message, 
	gpointer data
);
void edv_image_browser_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress, 
	gpointer data
);

/* Window Created/Deleted Callbacks */
void edv_image_browser_window_created_cb(
	EDVImageBrowser *imbr,
	const EDVWindowType win_type,
	gpointer win
);
void edv_image_browser_window_deleted_cb(
	EDVImageBrowser *imbr,
	const EDVWindowType win_type,
	gpointer win
);

/* Write Protect Changed Callback */
void edv_image_browser_master_write_protect_changed_cb(
	EDVImageBrowser *imbr,
	const gboolean state
);

/* Delete Method Changed Callback */
void edv_image_browser_delete_method_changed_cb(
	EDVImageBrowser *imbr,
	const EDVDeleteMethod delete_method
);

/* Object Callbacks */
void edv_image_browser_vfs_object_added_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_image_browser_vfs_object_modified_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_image_browser_vfs_object_removed_cb(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* Mount/Unmount Callback */
void edv_image_browser_device_mount_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d,
	const gboolean mounted
);

/* Recycled Object Callbacks */
void edv_image_browser_recycled_object_added_cb(
	EDVImageBrowser *imbr,
	const guint index
);
void edv_image_browser_recycled_object_modified_cb(
	EDVImageBrowser *imbr,
	const guint index
);
void edv_image_browser_recycled_object_removed_cb(
	EDVImageBrowser *imbr,
	const guint index
);

/* Reconfigured Callback */
void edv_image_browser_reconfigured_cb(EDVImageBrowser *imbr);

/* MIME Type Callbacks */
void edv_image_browser_mime_type_added_cb(
	EDVImageBrowser *imbr,
	const gint mt_num, EDVMIMEType *m
);
void edv_image_browser_mime_type_modified_cb(
	EDVImageBrowser *imbr,
	const gint mt_num, EDVMIMEType *m
);
void edv_image_browser_mime_type_removed_cb(
	EDVImageBrowser *imbr,
	const gint mt_num
);

/* Device Callbacks */
void edv_image_browser_device_added_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d
);
void edv_image_browser_device_modified_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d
);
void edv_image_browser_device_removed_cb(
	EDVImageBrowser *imbr,
	const gint dev_num
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
void edv_image_browser_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (imbr == NULL))
		return;

	window = widget->window;
	if(window != NULL)
	{
		GdkGeometry geo;
		geo.min_width = 320;
		geo.min_height = 240;
		geo.base_width = 0;
		geo.base_height = 0;
		geo.width_inc = 1;
		geo.height_inc = 1;
		gdk_window_set_geometry_hints(
			window,
			&geo,
			GDK_HINT_MIN_SIZE | GDK_HINT_BASE_SIZE |
				GDK_HINT_RESIZE_INC
		);
	   gdk_window_set_hints(
			window,
			0, 0,				/* Position */
			geo.min_width, geo.min_height,	/* Min size */
			geo.max_width, geo.max_height,	/* Max size */
			GDK_HINT_MIN_SIZE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_endeavour_image_browser_48x48_xpm
		);
	}

	imbr->flags |= EDV_IMAGE_BROWSER_REALIZED;
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint edv_image_browser_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return(FALSE);

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
	{
		imbr->stop_count++;
		return(TRUE);
	}

	edv_image_browser_op_close(imbr);

	return(TRUE);
}

/*
 *	Any GtkWidget "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint edv_image_browser_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gboolean is_press;
	gint		etype,
					status = FALSE;
	guint		keyval,
					state;
	GtkWidget	*w,
					*focus_widget;
	CfgList *cfg_list;
	tlist_struct *tlist;
 	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (key == NULL) || (imbr == NULL))
		return(status);

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return(status);

	w = imbr->toplevel;
	focus_widget = GTK_WINDOW(w)->focus_widget;
	tlist = imbr->tlist;
	core = imbr->core;
	cfg_list = core->cfg_list;

	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

#define STOP_SIGNAL_EMISSION(_w_)	{	\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
		!GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
		EDVImageBrowserOpID op = (EDVImageBrowserOpID)edv_match_accel_key_op_id(
			cfg_list, EDV_CFG_PARM_IMBR_ACCELERATOR_KEYS,
			keyval, state
		);
		if(op > 0)
		{
			if(is_press)
			{
				EDVImageBrowserOp *opid = edv_image_browser_op_match_by_id(
					imbr, op
				);
				if(opid != NULL)
					edv_image_browser_op_cb(NULL, -1, opid);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			return(status);
		}
	}

	/* Check which widget this signal is for */

	/* Thumbs List */
	if(widget == TListGetListWidget(tlist))
	{
		switch(keyval)
		{
		  default:
			/* For all other alphanumeric character keys and while
			 * no modifier keys are held, attempt to seek to the
			 * item who's name starts with the letter of the key
			 * that was pressed
			 */
			if(EDV_GET_B(EDV_CFG_PARM_LISTS_KEYBOARD_SCROLL_TO_KEY_NAME) &&
				isalnum((int)keyval) && is_press)
			{
				const gboolean backwards = (state & GDK_MOD1_MASK) ? TRUE : FALSE;
				const gint start_thumb_num = (backwards) ?
					(tlist->total_thumbs - 1) : 0;

				TListFreeze(tlist);
				edv_tlist_seek_character(
					tlist,
					start_thumb_num,
					backwards,
					0,			/* Text index */
					keyval			/* Character */
				);
				TListThaw(tlist);
				STOP_SIGNAL_EMISSION(widget);
				status = TRUE;
			}
			break;
		}
	}

#undef STOP_SIGNAL_EMISSION

	return(status);
}

/*
 *	Any GtkWidget "button_press_event" signal callback.
 */
gint edv_image_browser_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint		status = FALSE,
					etype;
	guint state;
	CfgList *cfg_list;
	tlist_struct *tlist;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (button == NULL) || (imbr == NULL))
		return(status);

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
		return(status);

	etype = (gint)button->type;
	tlist = imbr->tlist;
	core = imbr->core;
	cfg_list = core->cfg_list;
	state = button->state;

	/* Check which widget this event is for
	 *
	 * Thumbs List
	 */
	if(widget == TListGetListWidget(tlist))
	{
		gint thumb_num = -1;

		switch(button->button)
		{
		  case GDK_BUTTON2:
			switch((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2))
			{
			  case EDV_LISTS_POINTER_OP_BUTTON2_NONE:
			  case EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY:
				/* Generic list callback handles this */
				break;

			  case EDV_LISTS_POINTER_OP_BUTTON2_RENAME:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Rename */
					TListGetSelection(
						tlist,
						button->x, button->y,
						&thumb_num,
						NULL, NULL
					);
					if(thumb_num > -1)
						edv_image_browser_list_rename_query(
							imbr,
							thumb_num
						);
				}
				status = TRUE;
				break;

			  case EDV_LISTS_POINTER_OP_BUTTON2_PASTE:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Paste Objects */
					TListGetSelection(
						tlist,
						button->x, button->y,
						&thumb_num,
						NULL, NULL
					);
					edv_image_browser_op_paste2(
						imbr,
						EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num))
					);
				}
				status = TRUE;
				break;
			}
			break;

		  case GDK_BUTTON3:			/* Right Click GtkMenu */
			if(etype == GDK_BUTTON_PRESS)
			{
				/* Select before mapping the menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS))
				{
					TListGetSelection(
						tlist,
						button->x, button->y,
						&thumb_num,
						NULL, NULL
					);
					if(thumb_num > -1)
					{
						gint single_selected_thumb;

						if(g_list_length(tlist->selection) == 1)
							single_selected_thumb = (gint)tlist->selection->data;
						else
							single_selected_thumb = -1;

						TListFreeze(tlist);
						if(!(state & GDK_CONTROL_MASK) &&
						   !(state & GDK_SHIFT_MASK) &&
						   (thumb_num != single_selected_thumb)
						)
							TListUnselectAll(tlist);
						tlist->focus_thumb = thumb_num;
						TListSelectThumb(tlist, thumb_num);
						TListThaw(tlist);
					}
				}
				/* Update all the menus and then map the
				 * right-click menu
				 */
				edv_image_browser_update_display(imbr);
				gtk_menu_popup(
					GTK_MENU(imbr->tlist_menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
				break;
			}
			status = TRUE;
			break;
		}
	}
	/* ImgView */
	else if(widget == (GtkWidget *)ImgViewGetViewWidget(imbr->imgview))
	{
		gint thumb_num;

		switch((gint)button->type)
		{
		  case GDK_2BUTTON_PRESS:		/* Double click */
			switch(button->button)
			{
			  case GDK_BUTTON1:		/* Open */
				thumb_num = imbr->tlist_selected_thumb;
				if(thumb_num > -1)
				{
					/* Open */
					edv_image_browser_list_open(
						imbr,
						thumb_num,
						state		/* Modifier keys */
					);
				}
				status = TRUE;
				break;
			}
			break;
		}
	}

	return(status);
}

/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void edv_image_browser_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((handle_box == NULL) || (imbr == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *	GtkHandleBox "child_detached" signal callback.
 */
void edv_image_browser_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((handle_box == NULL) || (imbr == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Location Bar Icon GtkFixed "realize" signal callback.
 */
void edv_image_browser_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (imbr == NULL))
		return;

	window = widget->window;
	core = imbr->core;

	if(window != NULL)
	{
		GdkCursor *cursor = edv_get_cursor(core, EDV_CURSOR_CODE_HAND);
		gdk_window_set_cursor(window, cursor);
	}
}


/*
 *	Thumbs list "realize" signal callback.
 */
void edv_image_browser_list_realize_cb(GtkWidget *widget, gpointer data)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (imbr == NULL))
		return;

}

/*
 *	Thumbs list select thumb callback.
 */
void edv_image_browser_list_select_cb(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	GtkWidget *toplevel;
	tlist_thumb_struct *thumb;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((tlist == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	thumb = TListGetThumb(tlist, thumb_num);

	if(tlist == imbr->tlist)
	{
		/* Change in selection? */
		if(imbr->tlist_selected_thumb != thumb_num)
		{
			const gint nselected = g_list_length(tlist->selection);
			EDVVFSObject *obj = EDV_VFS_OBJECT(
				TListGetThumbData(tlist, thumb_num)
			);

			/* If the selected thumb is not fully visible then
			 * scroll to it
			 */
			if(TListIsThumbVisible(tlist, thumb_num) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(
					tlist,
					thumb_num,
					0.5f
				);

			/* Record the current selected thumb */
			imbr->tlist_selected_thumb = thumb_num;

			/* Record the device who's mount path matches the
			 * selected object's path
			 */
			if(obj != NULL)
				edv_devices_list_match_mount_path(
					core->devices_list,
					&imbr->selected_dev_num,
					obj->path
				);
			else
				imbr->selected_dev_num = -1;

			/* Update the DND drag icon */
			edv_image_browser_list_dnd_set_drag_icon(imbr);

			/* Update the title and the status bar */
			if((obj != NULL) ? !STRISEMPTY(obj->name) : FALSE)
			{
				gchar *s, *size_str = NULL;
				const gchar	*type_str = NULL,
							*full_path = obj->path;
				EDVMIMEType *m = edv_mime_types_list_match_path(
					core->mime_types_list,
					obj->path
				);

				/* If only one thumb is selected then display it
				 * on the Image Browser's ImgView
				 */
				if((nselected == 1) &&
				   (thumb != NULL)
				)
				{
					gboolean need_reload_thumb = FALSE;
					if(thumb->load_state != TLIST_LOAD_STATE_LOADED)
					{
						need_reload_thumb = TRUE;
					}
					else if(!STRISEMPTY(full_path))
					{
						EDVVFSObject *obj2 = edv_vfs_object_stat(full_path);
						if(obj2 != NULL)
						{
							if(obj->modify_time != obj2->modify_time)
							{
								obj->modify_time = obj2->modify_time;
								need_reload_thumb = TRUE;
							}
							if(obj->size != obj2->size)
							{
								obj->size = obj2->size;
								need_reload_thumb = TRUE;
							}
							if((obj->device_index != obj2->device_index) ||
							   (obj->index != obj2->index)
							)
							{
								obj->device_index = obj2->device_index;
								obj->index = obj2->index;
								need_reload_thumb = TRUE;
							}

							edv_vfs_object_delete(obj2);
						}
					}

					edv_image_browser_set_busy(imbr, TRUE);
					GUIBlockInput(toplevel, TRUE);
					imbr->flags |= EDV_IMAGE_BROWSER_PROCESSING;

					edv_image_browser_view_load(
						imbr,
						full_path,
						need_reload_thumb ? thumb_num : -1
					);

					imbr->flags &= ~EDV_IMAGE_BROWSER_PROCESSING;
					GUIBlockInput(toplevel, FALSE);
					edv_image_browser_set_busy(imbr, FALSE);
				}

				/* Update the open to menus */
				edv_image_browser_open_to_menu_regenerate(imbr, m);

				/* Get object type string and size string */
				switch(obj->type)
				{
				  case EDV_OBJECT_TYPE_UNKNOWN:
					type_str = "Object";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FILE:
					type_str = "File";
					size_str = g_strdup_printf(
						" (%s %s)",
						edv_str_size_delim(obj->size),
						(obj->size == 1l) ? "byte" : "bytes"
					);
					break;
				  case EDV_OBJECT_TYPE_DIRECTORY:
					type_str = "Directory";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_LINK:
					type_str = "Link";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
					type_str = "Block device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
					type_str = "Character device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FIFO:
					type_str = "FIFO pipe";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_SOCKET:
					type_str = "Socket";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_ERROR:
					type_str = "Error";
					size_str = NULL;
					break;
				}

				/* Set the status bar message */
				if(nselected > 1)
				{
					gulong total_size = 0l;
					EDVVFSObject *obj;
					GList *glist = tlist->selection;
					while(glist != NULL)
					{
						obj = EDV_VFS_OBJECT(TListGetThumbData(
							tlist, (gint)glist->data
						));
						if(obj != NULL)
							total_size += obj->size;
						glist = g_list_next(glist);
					}
					g_free(size_str);
					size_str = g_strdup_printf(
						"%s %s",
						edv_str_size_delim(total_size),
						(total_size == 1l) ? "byte" : "bytes"
					);
					s = g_strdup_printf(
						"%s objects selected (totaling %s)",
						edv_str_size_delim(nselected),
						size_str
					);
				}
				else if(!strcmp((const char *)obj->name, ".."))
					s = g_strdup_printf(
						"Parent directory selected"
					);
				else
					s = g_strdup_printf(
						"%s \"%s\" selected%s",
						type_str, obj->name,
						(size_str != NULL) ? size_str : ""
					);
				edv_status_bar_message(imbr->status_bar, s, FALSE);
				g_free(s);
				g_free(size_str);
			}

			edv_image_browser_update_display(imbr);
		}

		/* Double click? */
		if(button != NULL)
		{
			if(button->type == GDK_2BUTTON_PRESS)
			{
				if(thumb_num > -1)
				{
					/* Open */
					edv_image_browser_list_open(
						imbr,
						thumb_num,
						button->state	/* Modifier keys */
					);
				}
			}
		}
	}
}

/*
 *	Image Browser thumbs list unselect thumb callback.
 */
void edv_image_browser_list_unselect_cb(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((tlist == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;

	if(tlist == imbr->tlist)
	{
		gchar *s;
		const gint nselected = g_list_length(tlist->selection);
		GList *glist = tlist->selection_end;
		const gint thumb_num = (glist != NULL) ? (gint)glist->data : -1;
		EDVMIMEType *m = NULL;
		if(thumb_num > -1)
		{
			/* Last selected thumb changed? */
			if(imbr->tlist_selected_thumb != thumb_num)
			{
				EDVVFSObject *obj = EDV_VFS_OBJECT(
					TListGetThumbData(tlist, thumb_num)
				);
				if(obj != NULL)
				{
					const gchar *path = obj->path;

					m = edv_mime_types_list_match_path(
						core->mime_types_list,
						path
					);

					/* Update last selected thumb and device */
					imbr->tlist_selected_thumb = thumb_num;
					edv_devices_list_match_mount_path(
						core->devices_list,
						&imbr->selected_dev_num,
						path
					);
				}
			}

			/* Only one thumb remained selected? */
			if(nselected == 1)
			{
/* TODO */
			}
		}
		else
		{
			/* Mark that no thumb and device are selected */
			imbr->tlist_selected_thumb = -1;
			imbr->selected_dev_num = -1;
		}

		/* Update the open to menus */
		edv_image_browser_open_to_menu_regenerate(imbr, m);

		/* Update the status bar message */
		if(nselected > 0)
		{
			gchar *size_str;
			gulong total_size = 0l;
			EDVVFSObject *obj;
			GList *glist = tlist->selection;
			while(glist != NULL)
			{
				obj = EDV_VFS_OBJECT(TListGetThumbData(
					tlist, (gint)glist->data
				));
				if(obj != NULL)
					total_size += obj->size;
				glist = g_list_next(glist);
			}
			size_str = g_strdup_printf(
				"%s %s",
				edv_str_size_delim(total_size),
				(total_size == 1l) ? "byte" : "bytes"
			);
			s = g_strdup_printf(
				"%s %s selected (totaling %s)",
				edv_str_size_delim(nselected),
				(nselected == 1) ? "object" : "objects",
				size_str
			);
			g_free(size_str);
		}
		else
			s = STRDUP("No objects selected");
		edv_status_bar_message(
			imbr->status_bar, s, FALSE
		);
		g_free(s);

		edv_image_browser_update_display(imbr);
	}
}

/*
 *	Image Browser Location Combo activate callback.
 */
void edv_image_browser_combo_activate_cb(GtkWidget *widget, gpointer data)
{
	gchar *path;
	GtkWidget *toplevel;
	GtkCombo *combo;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((widget == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	combo = GTK_COMBO(widget);
	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Evaluate/copy the location combo's current path */
	path = edv_path_evaluate(
		NULL,                               /* No parent reference */
		gtk_entry_get_text(GTK_ENTRY(combo->entry))
	);
	if(path == NULL)
	{
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Make sure that the path exists and leads to a directory */
	obj = edv_vfs_object_stat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Location Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		g_free(path);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Not a directory? */
	if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		gchar *msg = g_strdup_printf(
"Not a directory:\n\
\n\
    %s",
			path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Location Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		edv_vfs_object_delete(obj);
		g_free(path);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	edv_vfs_object_delete(obj);

	GUIBlockInput(toplevel, TRUE);

	/* Update the location combo, this is really just to record
	 * the new value as history.
	 *
	 * The combo will be updated again (without recording history)
	 * if the new path is valid and selected in the callbacks
	 * triggered by edv_image_browser_select_path() farther below
	 */
	edv_image_browser_set_location(imbr, path, TRUE);

	/* Reset the loading procedure values and get new directory
	 * listing, then initialize loading procedure values for the
	 * loading of images from the directory specified by the path
	 */
	edv_image_browser_select_path(imbr, path);

	edv_image_browser_update_display(imbr);

	g_free(path);

	GUIBlockInput(toplevel, FALSE);
	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Load Thumbs Passive timeout callback.
 */
gint edv_image_browser_load_thumbs_passive_timeout_cb(gpointer data)
{
	GtkWidget *sb;
	CfgList *cfg_list;
	EDVRunlevel prev_runlevel;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return(FALSE);

	/* If there are any pending events then reschedual a call
	 * to this function using the lowest priority
	 */
	if(gtk_events_pending() > 0)
	{
		imbr->loading_tocb = gtk_timeout_add(
			edv_get_interval_from_load_images_priority(
				EDV_LISTS_LOAD_IMAGES_PRIORITY_LOWEST
			),
			edv_image_browser_load_thumbs_passive_timeout_cb, imbr
		);
		return(FALSE);
	}

	core = imbr->core;
	cfg_list = core->cfg_list;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) || (imbr->freeze_count > 0))
	{
		imbr->loading_tocb = gtk_timeout_add(
			edv_get_interval_from_load_images_priority(
				(EDVListsLoadImagesPriority)EDV_GET_I(
					EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY
				)
			),
			edv_image_browser_load_thumbs_passive_timeout_cb, imbr
		);
		return(FALSE);
	}

	/* If there is currently an operation being performed then
	 * reschedual a call to this function using the lowest
	 * priority
	 */
	if(core->op_level > 0)
	{
		imbr->loading_tocb = gtk_timeout_add(
			edv_get_interval_from_load_images_priority(
				EDV_LISTS_LOAD_IMAGES_PRIORITY_LOWEST
			),
			edv_image_browser_load_thumbs_passive_timeout_cb, imbr
		);
		return(FALSE);
	}

	imbr->freeze_count++;

#define RETURN_FINAL(_imbr_) {				\
							\
 /* Reset loading timeout id */				\
 (_imbr_)->loading_tocb = 0;				\
							\
 /* Restore the previous runlevel */			\
 core->runlevel = prev_runlevel;			\
							\
 edv_image_browser_set_passive_busy((_imbr_), FALSE);	\
											\
 return(FALSE);					\
}

#define RETURN_RESCHEDUAL(_imbr_) {		\
 /* Set the new loading timeout callback */	\
 (_imbr_)->loading_tocb = gtk_timeout_add(	\
  edv_get_interval_from_load_images_priority( \
   (EDVListsLoadImagesPriority)EDV_GET_I(	\
    EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY	\
   )						\
  ),						\
  edv_image_browser_load_thumbs_passive_timeout_cb, imbr		\
 );						\
											\
 /* Restore the previous runlevel */		\
 core->runlevel = prev_runlevel;		\
											\
 /* Return FALSE since we reschedualed a	\
  * timeout callback to this function		\
  */						\
 return(FALSE);					\
}

	sb = imbr->status_bar;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Update the core's runlevel to background */
	if(core->runlevel < EDV_RUNLEVEL_BACKGROUND)
	{
		prev_runlevel = core->runlevel;
		core->runlevel = EDV_RUNLEVEL_BACKGROUND;
	}
	else
	{
		prev_runlevel = core->runlevel;
	}

	/* Loading interrupted? */
	if(imbr->stop_count > 0)
	{
		imbr->stop_count = 0;
		imbr->loading_tocb = 0;

		edv_status_bar_message(sb, "Loading interrupted", FALSE);
		edv_status_bar_progress(sb, 0.0f, FALSE);
		edv_image_browser_update_display(imbr);

		imbr->freeze_count--;

		RETURN_FINAL(imbr);
	}

	/* Load the next thumb */
	switch(edv_image_browser_list_load_iterate(
		imbr,
		EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
		EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_NO_ENLARGE)
	))
	{
	  case 0:	/* All thumbs loaded */
		imbr->loading_tocb = 0;
		edv_image_browser_update_display(imbr);
		imbr->freeze_count--;
		RETURN_FINAL(imbr);
		break;

	  case 1:	/* Thumb loaded successfully or failed to load,
					 * either way, continue loading subsequent thumbs */
		imbr->freeze_count--;
		RETURN_RESCHEDUAL(imbr);
		break;

	  case -3:	/* System error */
		imbr->freeze_count--;
		RETURN_FINAL(imbr);
		break;

	  default:	/* Other error */
		/* It is safe to continue again after this error since
		 * this thumb that failed to load will not be attempted
		 * to be loaded again
		 */
		imbr->freeze_count--;
		RETURN_RESCHEDUAL(imbr);
		break;
	}
#undef RETURN_FINAL
#undef RETURN_RESCHEDUAL
}


/*
 *	Menu item "activate" signal callback.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
void edv_image_browser_menu_item_cb(GtkWidget *widget, gpointer data)
{
	edv_image_browser_op_cb(NULL, -1, data);
}

/*
 *	Menu item "enter_notify_event" signal callback.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
gint edv_image_browser_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_image_browser_op_enter_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item "leave_notify_event" signal callback.
 *
 *	The data must be a EDVImageBrowserOp *.
 */
gint edv_image_browser_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_image_browser_op_leave_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu Item command object op "activate" signal callback.
 */
void edv_image_browser_menu_item_command_object_op_cb(GtkWidget *widget, gpointer data)
{
	const gchar *cmd_src;
	gchar *cmd;
	GtkWidget *toplevel;
	GList *paths_list;
	CfgList *cfg_list;
	EDVImageBrowser *imbr;
	EDVCore *core;
	EDVImageBrowserCommand *d = EDV_IMAGE_BROWSER_COMMAND(data);
	if(d == NULL)
		return;

	imbr = d->image_browser;
	toplevel = imbr->toplevel;
	core = imbr->core;
	cfg_list = core->cfg_list;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	/* Get the command for this menu item */
	cmd_src = d->command;
	if(STRISEMPTY(cmd_src))
	{
		gchar *msg = g_strdup_printf(
"The custom menu item:\n\
\n\
    \"%s\"\n\
\n\
Does not have a defined command.",
			d->label
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Open Failed",
			msg,
"To set the command for this custom menu item,\n\
go to Settings->Customize...->Global->Object Ops Menu.",
			toplevel
		);
		g_free(msg);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Format the command */
	cmd = STRDUP(cmd_src);

	/* Prepend the run in terminal command? */
	if(d->flags & CFG_MENU_ITEM_RUN_IN_TERMINAL)
	{
		const gchar *cmd_terminal_run = EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(!STRISEMPTY(cmd_terminal_run))
		{
			/* Format the command for run in terminal */
			gchar *s = g_strconcat(
				cmd_terminal_run,
				" ",
				cmd,
				NULL
			);
			g_free(cmd);
			cmd = s;
		}
		else
		{
			edv_play_sound_warning(core);
			edv_message_warning(
				"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
				NULL,
				toplevel
			);
			g_free(cmd);
			edv_image_browser_set_busy(imbr, FALSE);
			return;
		}
	}

	/* Get the selected paths list */
	paths_list = edv_image_browser_get_selected_paths(imbr);
	if(paths_list != NULL)
	{
		/* Open the selected objects */
		(void)edv_open_command(
			core,
			paths_list,
			cmd,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
	}

	/* Delete the formatted command */
	g_free(cmd);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Menu Item command new object "activate" signal callback.
 */
void edv_image_browser_menu_item_command_new_object_cb(GtkWidget *widget, gpointer data)
{
	gboolean yes_to_all = FALSE;
	gint status;
	const gchar	*cmd_src,
					*error_msg;
	gchar		*cmd,
					*new_name,
					*new_path;
	GList *new_paths_list;
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVImageBrowser *imbr;
	EDVCore *core;
	EDVImageBrowserCommand *d = EDV_IMAGE_BROWSER_COMMAND(data);
	if(d == NULL)
		return;

	imbr = d->image_browser;
	toplevel = imbr->toplevel;
	core = imbr->core;
	cfg_list = core->cfg_list;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	/* Get the command for this menu item */
	cmd_src = d->command;
	if(STRISEMPTY(cmd_src))
	{
		gchar *msg = g_strdup_printf(
"The custom menu item:\n\
\n\
    %s\n\
\n\
Does not have a defined command.",
			d->label
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Create New Object Failed",
			msg,
"To set the command for this custom menu item,\n\
go to Settings->Customize...->Global->New Objects Menu.",
			toplevel
		);
		g_free(msg);
		edv_image_browser_set_busy(imbr, FALSE);
		return;
	}

	/* Format the suggested name for the new object */
	if(!STRISEMPTY(d->ext_data))
	{
		const gchar *ext = d->ext_data;
		gchar *s, *new_ext;

		if(*ext != '.')
			new_ext = g_strconcat(".", ext, NULL);
		else
			new_ext = STRDUP(ext);
		s = (gchar *)strpbrk((char *)new_ext, " \t,;");
		if(s != NULL)
			*s = '\0';

		new_name = g_strconcat(
			"new",
			new_ext,
			NULL
		);

		g_free(new_ext);
	}
	else
	{
		new_name = STRDUP("new");
	}

	/* Prompt the user for the name of the new file and create the
	 * new file
	 */
	status = EDVVFSObjectCreateFileQueryName(
		core,
		edv_image_browser_get_location(imbr),	/* Parent path */
		new_name,				/* Suggested name */
		&new_path,
		toplevel,
		TRUE,				/* Show progress */
		&yes_to_all
	);

	g_free(new_name);

	/* Check for errors */
	error_msg = EDVVFSObjectCreateGetError(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Create New Object Failed",
			error_msg,   
			NULL,
			toplevel
		);
	}

	/* Report the new object being added and create the new paths
	 * list
	 */
	new_paths_list = NULL;
	if(new_path != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(new_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_added(
				core,
				new_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}

		new_paths_list = g_list_append(
			new_paths_list,
			STRDUP(new_path)
		);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Format the command */
	cmd = STRDUP(cmd_src);

	/* Prepend the run in terminal command? */
	if(d->flags & CFG_MENU_ITEM_RUN_IN_TERMINAL)
	{
		const gchar *cmd_terminal_run = EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(!STRISEMPTY(cmd_terminal_run))
		{
			/* Format the command for run in terminal */
			gchar *s = g_strconcat(
				cmd_terminal_run,
				" ",
				cmd,
				NULL
			);
			g_free(cmd);
			cmd = s;
		}
		else
		{
			edv_play_sound_warning(core);
			edv_message_warning(
				"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
				NULL,
				toplevel
			);
			if(new_paths_list != NULL)
			{
				g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
				g_list_free(new_paths_list);
			}
			g_free(cmd);
			g_free(new_path);
			edv_image_browser_set_busy(imbr, FALSE);
			return;
		}
	}

	/* Open the new object */
	if(new_paths_list != NULL)
	{
		(void)edv_open_command(
			core,
			new_paths_list,
			cmd,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			new_paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(new_paths_list);
	}

	/* Delete the formatted command */
	g_free(cmd);

	/* Delete the new path */
	g_free(new_path);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Menu item command "leave_enter_event" signal callback.
 */
gint edv_image_browser_menu_item_command_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVImageBrowser *imbr;
	EDVImageBrowserCommand *d = EDV_IMAGE_BROWSER_COMMAND(data);
	if(d == NULL)
		return(FALSE);

	imbr = d->image_browser;

	edv_status_bar_message(
		imbr->status_bar,
		STRISEMPTY(d->description) ?
			d->command : d->description,
		FALSE
	);

	return(TRUE);
}

/*
 *	Menu item command "leave_notify_event" signal callback.
 */
gint edv_image_browser_menu_item_command_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVImageBrowser *imbr;
	EDVImageBrowserCommand *d = EDV_IMAGE_BROWSER_COMMAND(data);
	if(d == NULL)
		return(FALSE);

	imbr = d->image_browser;

	edv_status_bar_message(
		imbr->status_bar,
		NULL,
		FALSE
	);

	return(TRUE);
}


/*
 *	Open paths List Get callback.
 */
GList *edv_image_browser_get_selected_paths_cb(gpointer data)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return(NULL);

	return(edv_image_browser_get_selected_paths(imbr));
}

/*
 *	Go to directory callback.
 */
void edv_image_browser_goto_directory_cb(gpointer data, const gchar *path)
{
	GtkWidget *toplevel;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((imbr == NULL) || (path == NULL))
		return;

	toplevel = imbr->toplevel;

	edv_image_browser_set_busy(imbr, TRUE);
	GUIBlockInput(toplevel, TRUE);

	edv_image_browser_select_path(imbr, path);
	edv_image_browser_update_display(imbr);

	GUIBlockInput(toplevel, FALSE);
	edv_image_browser_set_busy(imbr, FALSE);
}

/*
 *	Mount Bar mount callback.
 */
void edv_image_browser_mount_bar_mount_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gboolean original_mount_state;
	gint status;
	GtkWidget *toplevel;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((dev == NULL) || (imbr == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	toplevel = imbr->toplevel;
	core = imbr->core;

	/* Record the original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Mount/unmount the device */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);

	/* Update all the devices' statistics */
	edv_devices_list_update_statistics(core->devices_list);

	edv_image_browser_set_busy(imbr, FALSE);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Notify about this device being mounted/unmounted */
		edv_emit_device_mount(
			core,
			dev_num, dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		);
	}
}

/*
 *	Mount Bar eject callback.
 */
void edv_image_browser_mount_bar_eject_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gboolean original_mount_state;
	gint status;
	GtkWidget *toplevel;
	EDVCore *core;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((imbr == NULL) || (dev == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;
	if(core == NULL)
		return;

	edv_image_browser_set_busy(imbr, TRUE);

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core, dev,
			TRUE, TRUE,
			toplevel
		);

	/* Eject */
	status = edv_device_eject(
		core, dev,
		TRUE, TRUE,
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	edv_image_browser_set_busy(imbr, FALSE);

	/* Mount error? */
	if(status)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Report eject (unmount) signal to all of endeavour's
		 * resources
		 */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}
}

/*
 *	Mount Bar go to mount path callback.
 */
extern void edv_image_browser_mount_bar_goto_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gchar *path;
	GtkWidget *toplevel;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((imbr == NULL) || (dev == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	toplevel = imbr->toplevel;

	/* Get copy of mount path */
	path = STRDUP(dev->mount_path);

	edv_image_browser_set_busy(imbr, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Go to mount path */
	edv_image_browser_select_path(imbr, path);
	edv_image_browser_update_display(imbr);

	g_free(path);

	GUIBlockInput(toplevel, FALSE);
	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Find Bar get current location callback.
 */
const gchar *edv_image_browser_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
)
{
	return(edv_image_browser_get_location(EDV_IMBR(data)));
}

/*
 *	Find Bar start find callback.
 */
void edv_image_browser_find_bar_start_cb(GtkWidget *bar, gpointer data)
{
	tlist_struct *tlist;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return;

	tlist = imbr->tlist;

	edv_image_browser_set_busy(imbr, TRUE);

	TListFreeze(tlist);
	TListUnselectAll(tlist);
	TListThaw(tlist);

	edv_image_browser_update_display(imbr);

	imbr->freeze_count++;
}

/*
 *	Find Bar end find callback.
 */
void edv_image_browser_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return;

	imbr->freeze_count--;
	edv_image_browser_set_busy(imbr, FALSE);
}


/*
 *	Find Bar match callback.
 */
void edv_image_browser_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
)
{
	gint thumb_num;
	tlist_struct *tlist;
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if((path == NULL) || (imbr == NULL))
		return;

	tlist = imbr->tlist;

	thumb_num = edv_image_browser_list_find_by_path(imbr, path);
	if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
	{
		TListFreeze(tlist);
		TListSelectThumb(tlist, thumb_num);
		TListThaw(tlist);
	}
}


/*
 *	Status message callback.
 */
void edv_image_browser_status_message_cb(
	GtkWidget *widget,
	const gchar *message,
	gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return;

	edv_status_bar_message(imbr->status_bar, message, FALSE);
}

/*
 *	Status progress callback.
 */
void edv_image_browser_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress,
	gpointer data
)
{
	EDVImageBrowser *imbr = EDV_IMBR(data);
	if(imbr == NULL)
		return;

	edv_status_bar_progress(imbr->status_bar, progress, FALSE);
}


/*
 *	Window created callback.
 */
void edv_image_browser_window_created_cb(
	EDVImageBrowser *imbr,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(imbr == NULL)
		return;

	if(imbr == EDV_IMBR(win))
		return;

	edv_image_browser_update_display(imbr);
}

/*
 *	Window deleted callback.
 */
void edv_image_browser_window_deleted_cb(
	EDVImageBrowser *imbr,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(imbr == NULL)
		return;

	if(imbr == EDV_IMBR(win))
		return;

	edv_image_browser_update_display(imbr);
}


/*
 *	Master Write Protect changed callback.
 */
void edv_image_browser_master_write_protect_changed_cb(
	EDVImageBrowser *imbr,
	const gboolean state
)
{
	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_update_display(imbr);
}

/*
 *	Delete Method changed callback.
 */
void edv_image_browser_delete_method_changed_cb(
	EDVImageBrowser *imbr,
	const EDVDeleteMethod delete_method
)
{
	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	edv_image_browser_update_display(imbr);
}

/*
 *	Object added callback.
 */
void edv_image_browser_vfs_object_added_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	EDVVFSObject *obj
)
{
	if((imbr == NULL) || STRISEMPTY(path))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	/* Thumbs List */
	edv_image_browser_list_vfs_object_added_cb(
		imbr,
		path,
		obj
	);

	/* Mount Bar */
	edv_mount_bar_update_display(imbr->mount_bar);

	/* Need to update the widgets because a new loading process
	 * may have been queued
	 */
	edv_image_browser_update_display(imbr);
}

/*
 *	Object modified callback.
 */
void edv_image_browser_vfs_object_modified_cb(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gchar *cur_path;

	if((imbr == NULL) || STRISEMPTY(path))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	if(cur_path == NULL)
		return;

	/* Is the modified path the current location? */
	if(!strcmp((const char *)path, (const char *)cur_path))
	{
		/* Path changed? */
		if(path != new_path)
		{
			if(strcmp((const char *)path, (const char *)new_path))
			{
				edv_image_browser_set_title(
					imbr,
					new_path
				);
				edv_image_browser_set_location(
					imbr,
					new_path,
					FALSE
				);
				edv_image_browser_update_location_icon(
					imbr,
					new_path
				);
			}
		}
	}

	/* Thumbs List */
	edv_image_browser_list_vfs_object_modified_cb(
		imbr,
		path,
		new_path,
		obj
	);

	/* Mount Bar */
	edv_mount_bar_update_display(imbr->mount_bar);

	/* Need to update the widgets because a new loading process
	 * may have been queued
	 */
	edv_image_browser_update_display(imbr);

	g_free(cur_path);
}

/*
 *	Object removed callback.
 */
void edv_image_browser_vfs_object_removed_cb(
	EDVImageBrowser *imbr,
	const gchar *path
)
{
	gchar *cur_path;
	CfgList *cfg_list;
	EDVCore *core;

	if((imbr == NULL) || STRISEMPTY(path))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	cur_path = STRDUP(edv_image_browser_get_location(imbr));
	if(cur_path == NULL)
		return;

	/* Thumbs List */
	edv_image_browser_list_vfs_object_removed_cb(imbr, path);

	/* Mount Bar */
	edv_mount_bar_update_display(imbr->mount_bar);

	/* Is the removed path the current location or one of its
	 * parents?
	 */
	if(strpfx((const char *)cur_path, (const char *)path))
	{
		/* Change the location to the parent of the removed path */
		gchar *parent_path = g_dirname(path);
		if(parent_path != NULL)
		{
			edv_image_browser_set_title(
				imbr,
				parent_path
			);
			edv_image_browser_set_location(
				imbr,
				parent_path,
				FALSE
			);
			edv_image_browser_update_location_icon(
				imbr,
				parent_path
			);
			edv_image_browser_list_get_listing(
				imbr,
				parent_path,
				EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
			);
			g_free(parent_path);
		}
	}

/*	edv_image_browser_update_display(imbr); */

	g_free(cur_path);
}


/*
 *	Device mount/unmount callback.
 */
void edv_image_browser_device_mount_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d,
	const gboolean mounted
)
{
	if((imbr == NULL) || (d == NULL))
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	/* Update the location icon */
	edv_image_browser_update_location_icon(
		imbr,
		edv_image_browser_get_location(imbr)
	);

	/* Thumbs List */
	edv_image_browser_list_device_mount_cb(
		imbr,
		d,
		mounted
	);
  
	edv_image_browser_update_display(imbr);
}


/*
 *	Recycled object added callback.
 */
void edv_image_browser_recycled_object_added_cb(
	EDVImageBrowser *imbr, const guint index
)
{
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != imbr->last_nrecycle_bin_items)
		edv_image_browser_update_display(imbr);
}

/*
 *	Recycled object modified callback.
 */
void edv_image_browser_recycled_object_modified_cb(
	EDVImageBrowser *imbr, const guint index
)
{
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != imbr->last_nrecycle_bin_items)
		edv_image_browser_update_display(imbr);
}

/*
 *	Recycled object removed callback.
 */
void edv_image_browser_recycled_object_removed_cb(
	EDVImageBrowser *imbr, const guint index
)
{
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != imbr->last_nrecycle_bin_items)
		edv_image_browser_update_display(imbr);
}


/*
 *	Reconfigured callback.
 */
void edv_image_browser_reconfigured_cb(EDVImageBrowser *imbr)
{
	gchar *current_location_path;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;
	tlist_struct *tlist;
	imgview_struct *iv;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfigure
	 */
	imbr->last_nrecycle_bin_items = -1;
	imbr->last_write_protect_state = -1;

	/* Get the current location */
	current_location_path = STRDUP(edv_image_browser_get_location(imbr));

	/* Begin updating */

	/* Reload the icons */
	edv_image_browser_reload_icons(imbr);

	/* Update the title */
	edv_image_browser_set_title(imbr, current_location_path);

	/* Recreate the Tool Bar */
	edv_image_browser_tool_bar_regenerate(imbr);

	/* Recreate the File menu items */
	edv_image_browser_file_menu_items_regenerate(imbr);

	/* Show/hide Tool Bar */
	w = imbr->tool_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR))
		{
			imbr->flags |= EDV_IMAGE_BROWSER_TOOL_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			imbr->flags &= ~EDV_IMAGE_BROWSER_TOOL_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide Location Bar */
	w = imbr->location_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR))
		{
			imbr->flags |= EDV_IMAGE_BROWSER_LOCATION_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			imbr->flags &= ~EDV_IMAGE_BROWSER_LOCATION_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide Mount Bar */
	w = imbr->mount_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR))
		{
			imbr->flags |= EDV_IMAGE_BROWSER_MOUNT_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			imbr->flags &= ~EDV_IMAGE_BROWSER_MOUNT_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide Find Bar */
	w = imbr->find_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_FIND_BAR))
		{
			imbr->flags |= EDV_IMAGE_BROWSER_FIND_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			imbr->flags &= ~EDV_IMAGE_BROWSER_FIND_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide Status Bar */
	w = imbr->status_bar;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR))
		{
			imbr->flags |= EDV_IMAGE_BROWSER_STATUS_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			imbr->flags &= ~EDV_IMAGE_BROWSER_STATUS_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Thumbs List */
	tlist = imbr->tlist;
	if(tlist != NULL)
	{
		GtkOrientation orientation;

		TListFreeze(tlist);

		TListThumbGeometry(
			tlist,
			EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_WIDTH),
			EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_HEIGHT),
			EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_BORDER)
		);
		TListDoubleBuffer(
			tlist,
			EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER)
		);
		orientation = (EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_LIST_HORIZONTAL)) ?
			GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL;
		TListOrientation(tlist, orientation);
		TListShowThumbFrames(
			tlist,
			EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_FRAMES)
		);
		TListShowThumbLabels(
			tlist,
			EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_LABELS)
		);
		TListShowTextTips(
			tlist,
			EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS)
		);
		TListEnableListDragScroll(
			tlist,
			((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2) == EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY) ? TRUE : FALSE
		);

		/* Update geometry of thumb size */
		TListResize(tlist, -1, -1);

		/* Realize listing */
		edv_image_browser_list_realize_listing(imbr);

		TListThaw(tlist);
	}

	/* Image Viewer */
	iv = imbr->imgview;
	if(iv != NULL)
	{
		CfgColor *cfg_bg_color;

		/* Image quality */
		switch(EDV_GET_I(EDV_CFG_PARM_IMAGE_QUALITY))
		{
		  case 0:
			iv->quality = 0;
			break;
		  case 1:
			iv->quality = 1;
			break;
		  case 2:
			iv->quality = 2;
			break;
		  default:
			iv->quality = 1;
			break;
		}

		/* Background color */
		cfg_bg_color = EDV_GET_COLOR(EDV_CFG_PARM_IMBR_BG_COLOR);
		if(cfg_bg_color != NULL)
		{
			GdkColor gdk_color[5];
			const gint m = sizeof(gdk_color) / sizeof(GdkColor);
			gint i;
			for(i = 0; i < m; i++)
			{
				GDK_COLOR_SET_COEFF(
					&gdk_color[i],
					cfg_bg_color->r,
					cfg_bg_color->g,
					cfg_bg_color->b
				)
			}
			ImgViewSetViewBG(iv, gdk_color);
		}

		/* Redraw image viewer and update menus */
		ImgViewDraw(iv);
		ImgViewUpdateMenus(iv);
	}

	/* Recreate the New Object Submenu */
	edv_image_browser_new_object_menu_items_regenerate(imbr);

	/* Recreate the Open To menus */
	if(imbr->tlist_selected_thumb > -1)
	{
		const gint thumb_num = imbr->tlist_selected_thumb;
		EDVVFSObject *obj = EDV_VFS_OBJECT(TListGetThumbData(
			imbr->tlist, thumb_num
		));
		const gchar *path = (obj != NULL) ? obj->path : NULL;
		EDVMIMEType *m = edv_mime_types_list_match_path(
			core->mime_types_list,
			path
		);
		edv_image_browser_open_to_menu_regenerate(imbr, m);
	}

	/* Recreate the Accelkey labels */
	edv_image_browser_accelkeys_regenerate(imbr);

	/* Update RC styles */
	w = imbr->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	tlist = imbr->tlist;
	w = TListGetListWidget(tlist);
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = ImgViewGetToplevelWidget(imbr->imgview);
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = (GtkWidget *)ImgViewGetMenuWidget(imbr->imgview);
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(
			w, standard_rcstyle
		);
	w = imbr->tlist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);


	edv_image_browser_update_display(imbr);

	/* Notify GTK about possible size changes (such as the tool
	 * bar resizing)
	 */
	gtk_widget_queue_resize(imbr->toplevel);

	g_free(current_location_path);
}


/*
 *	MIME Type added callback.
 */
void edv_image_browser_mime_type_added_cb(
	EDVImageBrowser *imbr,
	const gint mt_num, EDVMIMEType *m
)
{
	/* Treat a MIME Type added the same as it would be for a MIME
	 * Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	edv_image_browser_mime_type_modified_cb(imbr, mt_num, m);
}

/*
 *	MIME Type modified callback.
 */
void edv_image_browser_mime_type_modified_cb(
	EDVImageBrowser *imbr,
	const gint mt_num, EDVMIMEType *m
)
{
	gchar *current_location_path;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	/* Get current location */
	current_location_path = STRDUP(edv_image_browser_get_location(imbr));

	/* Update the location icon */
	edv_image_browser_update_location_icon(
		imbr,
		current_location_path
	);

	/* Realize the listing */
	TListFreeze(tlist);
	edv_image_browser_list_realize_listing(imbr);
	TListThaw(tlist);

	g_free(current_location_path);
}

/*
 *	MIME Type removed callback.
 */
void edv_image_browser_mime_type_removed_cb(
	EDVImageBrowser *imbr,
	const gint mt_num
)
{
	gchar *current_location_path;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_image_browser_get_location(imbr));

	/* Update the location icon */
	edv_image_browser_update_location_icon(
		imbr,
		current_location_path
	);

	/* Realize the listing */
	TListFreeze(tlist);
	edv_image_browser_list_realize_listing(imbr);
	TListThaw(tlist);

	g_free(current_location_path);
}


/*
 *	Device added callback.
 */
void edv_image_browser_device_added_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d
)
{
	/* Treat a device added the same as it would be for a device
	 * modified, forward signal to the device modified callback
	 */
	edv_image_browser_device_modified_cb(imbr, dev_num, d);
}

/*
 *	Device modified callback.
 */
void edv_image_browser_device_modified_cb(
	EDVImageBrowser *imbr,
	const gint dev_num, EDVDevice *d
)
{
	gchar *current_location_path;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_image_browser_get_location(imbr));

	/* Update the location icon */
	edv_image_browser_update_location_icon(
		imbr,
		current_location_path
	);

	/* Realize the listing */
	TListFreeze(tlist);
	edv_image_browser_list_realize_listing(imbr);
	TListThaw(tlist);

	edv_image_browser_update_display(imbr);

	g_free(current_location_path);
}

/*
 *	Device removed callback.
 */
void edv_image_browser_device_removed_cb(
	EDVImageBrowser *imbr,
	const gint dev_num
)
{
	gchar *current_location_path;
	tlist_struct *tlist;
	EDVCore *core;

	if(imbr == NULL)
		return;

	if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr))
		return;

	tlist = imbr->tlist;
	core = imbr->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_image_browser_get_location(imbr));

	/* Check if the Mount Bar is currently referencing this device,
	 * if it is then its selected device needs to be set to -1
	 */
	if(edv_mount_bar_current_device(imbr->mount_bar) == dev_num)
		edv_mount_bar_select(
			imbr->mount_bar,
			-1
		);

	/* Update the location icon */
	edv_image_browser_update_location_icon(
		imbr,
		current_location_path
	);

	/* Realize the listing */
	TListFreeze(tlist);
	edv_image_browser_list_realize_listing(imbr);
	TListThaw(tlist);

	edv_image_browser_update_display(imbr);

	g_free(current_location_path);
}
