#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "tool_bar.h"
#include "cfg_gtk.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_device.h"
#include "edv_utils_gtk.h"
#include "edv_obj_info_match.h"
#include "edv_list_cb.h"
#include "edv_mount_bar.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "image_browser.h"
#include "image_browser_cb.h"
#include "image_browser_op.h"
#include "image_browser_list.h"
#include "image_browser_view.h"
#include "image_browser_dnd.h"
#include "edv_cb.h"
#include "edv_open_to_menu.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_rename_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_go_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_monitor2_20x20.xpm"
#include "images/icon_prev_20x20.xpm"
#include "images/icon_next_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_download_file_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_devices_list_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_options_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_write_protect_off_20x20.xpm"
#include "images/icon_write_protect_on_20x20.xpm"
#include "images/icon_fsck_20x20.xpm"
#include "images/icon_tools_20x20.xpm"
#include "images/icon_floppy_20x20.xpm"

#include "images/icon_open_20x20.xpm"
#include "images/icon_file_20x20.xpm"
#include "images/icon_file_hidden_20x20.xpm"
#include "images/icon_file_noaccess_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"
#include "images/icon_folder_home_20x20.xpm"
#include "images/icon_folder_parent_20x20.xpm"
#include "images/icon_link_20x20.xpm"
#include "images/icon_pipe_20x20.xpm"
#include "images/icon_device_block_20x20.xpm"
#include "images/icon_device_character_20x20.xpm"
#include "images/icon_socket_20x20.xpm"

#include "images/icon_purge_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/icon_endeavour_file_browser_20x20.xpm"
#include "images/icon_endeavour_image_browser_20x20.xpm"
#include "images/icon_endeavour_archiver_20x20.xpm"

#include "images/icon_play_20x20.xpm"
#include "images/icon_pause_20x20.xpm"
#include "images/icon_previous_track_20x20.xpm"
#include "images/icon_next_track_20x20.xpm"
#include "images/icon_rotate_cw_20x20.xpm"
#include "images/icon_rotate_ccw_20x20.xpm"
#include "images/icon_mirror_horizontal_20x20.xpm"
#include "images/icon_mirror_vertical_20x20.xpm"


/* Utilities */
EDVImageBrowserOp *edv_image_browser_op_match_by_id(
	EDVImageBrowser *imbr,
	const EDVImageBrowserOpID id
);

/* EDVImageBrowserCommand */
static EDVImageBrowserCommand *edv_image_browser_command_new(void);
static void edv_image_browser_command_delete(EDVImageBrowserCommand *cmd);

/* Title */
void edv_image_browser_set_title(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* Location */
const gchar *edv_image_browser_get_location(EDVImageBrowser *imbr);
void edv_image_browser_set_location(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gboolean record_history
);
void edv_image_browser_update_location_icon(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* Get selected object & paths */
GList *edv_image_browser_get_selected_objects(
	EDVImageBrowser *imbr,
	const gboolean copy
);
GList *edv_image_browser_get_selected_paths(EDVImageBrowser *imbr);

/* Regenerate Styles, Menus, and Tool Bars */
void edv_image_browser_accelkeys_regenerate(EDVImageBrowser *imbr);
void edv_image_browser_file_menu_items_regenerate(EDVImageBrowser *imbr);
void edv_image_browser_new_object_menu_items_regenerate(EDVImageBrowser *imbr);
void edv_image_browser_open_to_menu_regenerate(
	EDVImageBrowser *imbr,
	EDVMIMEType *m
);
void edv_image_browser_tool_bar_regenerate(EDVImageBrowser *imbr);
void edv_image_browser_list_menu_regenerate(EDVImageBrowser *imbr);

/* Sync Configuration */
void edv_image_browser_sync_configuration(EDVImageBrowser *imbr);

/* Load Control */
void edv_image_browser_queue_loading_process(EDVImageBrowser *imbr);

/* Select Path */
void edv_image_browser_select_path(
	EDVImageBrowser *imbr,
	const gchar *path
);

/* EDVImageBrowserOp */
static EDVImageBrowserOp *edv_image_browser_op_new(void);
static void edv_image_browser_op_delete(EDVImageBrowserOp *op);
GList *edv_image_browser_ops_list_new(EDVImageBrowser *imbr);
void edv_image_browser_ops_list_delete(GList *glist);

/* Accel Keys */
CfgAccelkeyList *edv_image_browser_accelkeys_new(void);

/* Reload Icons */
void edv_image_browser_reload_icons(EDVImageBrowser *imbr);

/* EDVImageBrowser */
static void EDVImbrCreateMenuBar(
	EDVImageBrowser *imbr,
	GtkWidget *parent
);
EDVImageBrowser *edv_image_browser_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_image_browser_sync_data(EDVImageBrowser *imbr);
void edv_image_browser_update_display(EDVImageBrowser *imbr);
void edv_image_browser_set_busy(
	EDVImageBrowser *imbr,
	const gboolean busy
);
void edv_image_browser_set_passive_busy(
	EDVImageBrowser *imbr,
	const gboolean busy
);
gboolean edv_image_browser_is_mapped(EDVImageBrowser *imbr);
void edv_image_browser_map(EDVImageBrowser *imbr);
void edv_image_browser_unmap(EDVImageBrowser *imbr);
void edv_image_browser_delete(EDVImageBrowser *imbr);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the Image Browser Operation ID that matches the given
 *	operation.
 */
EDVImageBrowserOp *edv_image_browser_op_match_by_id(
	EDVImageBrowser *imbr,
	const EDVImageBrowserOpID id
)
{
	GList *glist;
	EDVImageBrowserOp *op;

	if(imbr == NULL)
		return(NULL);

	for(glist = imbr->op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		op = EDV_IMAGE_BROWSER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->id == id)
			return(op);
	}

	return(NULL);
}


/*
 *	Creates a new EDVImageBrowserCommand.
 */
static EDVImageBrowserCommand *edv_image_browser_command_new(void)
{
	return(EDV_IMAGE_BROWSER_COMMAND(g_malloc0(sizeof(EDVImageBrowserCommand))));
}

/*
 *	Deletes the EDVImageBrowserCommand.
 */
static void edv_image_browser_command_delete(EDVImageBrowserCommand *cmd)
{
	if(cmd == NULL)
		return;

	g_free(cmd->label);
	g_free(cmd->command);
	g_free(cmd->description);
	g_free(cmd->ext_data);
	g_free(cmd);
}


/*
 *	Sets the Image Browser's title.
 *
 *	The given path is optional, it can be either the current
 *	selected location or the last selected object.
 */
void edv_image_browser_set_title(
	EDVImageBrowser *imbr,
	const gchar *path
)
{
	gboolean        show_program_name,
			show_user_name,
			show_location;
	const gchar	*prog_name = "Image Browser",
			*user_name,
			*filter;
	gchar *title;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	w = imbr->toplevel;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get the values to show in the title */
	if(!STRISEMPTY(core->title))
		prog_name = core->title;

	user_name = core->effective_user_id_host_str;
	filter = imbr->thumbs_list_filter;

	show_program_name = EDV_GET_B(EDV_CFG_PARM_IMBR_TITLE_WITH_PROGRAM);
	if(show_program_name && STRISEMPTY(prog_name))
		show_program_name = FALSE;

	show_user_name = EDV_GET_B(EDV_CFG_PARM_IMBR_TITLE_WITH_USER);
	if(show_user_name && STRISEMPTY(user_name))
		show_user_name = FALSE;

	show_location = EDV_GET_B(EDV_CFG_PARM_IMBR_TITLE_WITH_LOCATION);
	if(show_location && STRISEMPTY(path))
		show_location = FALSE;

	/* Format the new title string */
	title = g_strdup("");
	if(EDV_GET_B(EDV_CFG_PARM_IMBR_TITLE_REVERSE))
	{
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					G_DIR_SEPARATOR_S,
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_location) ? " - " : "",
				user_name,
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_user_name || show_location) ? " - " : "",
				prog_name,
				NULL
			);
			g_free(title);
			title = s;
		}
	}
	else
	{
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				prog_name,
				(show_user_name || show_location) ? ": " : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				user_name,
				(show_location) ? "" : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					G_DIR_SEPARATOR_S,
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
	}

	/* Set the new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Returns the current location.
 */
const gchar *edv_image_browser_get_location(EDVImageBrowser *imbr)
{
	GtkCombo *combo;

	if(imbr == NULL)
		return(NULL);

	combo = GTK_COMBO(imbr->location_combo);

	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the location.
 *
 *	If record_history is TRUE then the new location will be
 *	recorded on the location history.
 */
void edv_image_browser_set_location(
	EDVImageBrowser *imbr,
	const gchar *path,
	const gboolean record_history
)
{
	const gchar *s;
	gchar *new_path;
	GtkEntry *entry;
	GtkCombo *combo = (GtkCombo *)((imbr != NULL) ?
		imbr->location_combo : NULL
	);
	if((combo == NULL) || STRISEMPTY(path))
		return;

	entry = GTK_ENTRY(combo->entry);

	/* Get the new path */
	new_path = STRDUP(path);

	/* Simplify the new path */
	edv_path_simplify(new_path);

	/* No change? */
	s = gtk_entry_get_text(entry);
	if((s != NULL) ? !strcmp(s, new_path) : FALSE)
	{
		g_free(new_path);
		return;
	}

	/* Record new location on the location history? */
	if(record_history)
		GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set new location */
	gtk_entry_set_text(entry, new_path);
	gtk_entry_set_position(entry, -1);

	g_free(new_path);
}

/*
 *	Updates the Image Browser Location Bar's icon.
 *
 *	If the given path is NULL then the icon will be set as unknown.
 */
void edv_image_browser_update_location_icon(
	EDVImageBrowser *imbr,
	const gchar *path
)
{
	GtkWidget	*w,
			*parent;
	EDVPixmap	*icon,
			*icon_inaccessable,
			*icon_hidden;
	EDVVFSObject *obj;
	EDVCore *core;

	if(imbr == NULL)
		return;

	parent = imbr->location_icon_fixed;
	core = imbr->core;

	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		obj = edv_vfs_object_new();
		if(obj != NULL)
		{
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			edv_vfs_object_set_path(obj, path);
		}
	}
	if(obj == NULL)
		return;

	/* Get the pixmap and mask for the location icon */
	(void)edv_match_object_icon(
		core->devices_list,
		core->mime_types_list,
		obj->type,
		obj->path,
		EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		NULL,
		&icon_inaccessable,
		&icon_hidden
	);
	/* Hidden? */
	if(edv_is_object_hidden(obj))
	{
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}
	/* No access? */
	if(!edv_is_object_accessable(core, obj))
	{
		if(edv_pixmap_is_loaded(icon_inaccessable))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_inaccessable);
		}
	}

	edv_vfs_object_delete(obj);

	/* Create or set the location icon */
	w = imbr->location_icon_pm;
	if(w != NULL)
	{
		edv_pixmap_set_gtk_pixmap(icon, w);
	}
	else
	{
		imbr->location_icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
			gtk_widget_show(w);
		}
	}

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}


/*
 *	Recreates the Image Browser's Tool Bar.
 */
void edv_image_browser_tool_bar_regenerate(EDVImageBrowser *imbr)
{
	const gint border_minor = 2;
	GList		*glist,
			*items_list;
	GtkReliefStyle tb_relief;
	GtkWidget	*w,
			*parent;
	CfgIntList *intlist;
	CfgList *cfg_list;
	ToolBarButtonDecals tb_button_decals;
	ToolBarItem *item;
	EDVImageBrowserOp *op;
	EDVCore *core;

	if(imbr == NULL)
		return;

	parent = imbr->tool_bar_handle;
	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_button_decals = (ToolBarButtonDecals)EDV_GET_I(
		EDV_CFG_PARM_IMBR_TOOL_BAR_BUTTON_DECALS
	);
	tb_relief = (GtkReliefStyle)EDV_GET_I(
		EDV_CFG_PARM_IMBR_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INT_LIST(EDV_CFG_PARM_IMBR_TOOL_BAR);
	if(intlist == NULL)
		return;

	/* Delete the existing Tool Bar */
	GTK_WIDGET_DESTROY(imbr->tool_bar);
	imbr->tool_bar = NULL;

	/* Create the ToolBarItems list */
	items_list = NULL;
	for(glist = intlist->list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		/* Get the OPID */
		op = edv_image_browser_op_match_by_id( 
			imbr,
			(EDVImageBrowserOpID)glist->data
		);
		if(op == NULL)
			continue;

		/* Exclude this OPID from Tool Bars? */
		if(op->flags & EDV_OPID_EX_TOOL_BAR)
			continue;

		/* Create a new ToolBarItem from this OPID */
		item = ToolBarItemNew(
			op->tb_item_type,
			op->button_name,
			op->button_icon_data,
			op->tooltip,
			op->id,
			edv_image_browser_op_cb, op,
			edv_image_browser_op_enter_cb, op,
			edv_image_browser_op_leave_cb, op
		);
		if(item == NULL)
			continue;

		/* Append the new ToolBarItem to the list */
		items_list = g_list_append(
			items_list,
			item
		);
	}

	/* Create the Tool Bar */
	imbr->tool_bar = w = ToolBarNew(
		GTK_ORIENTATION_HORIZONTAL,
		tb_button_decals,
		tb_relief,
		GTK_POLICY_AUTOMATIC,
		items_list
	);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_TOOL_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(w);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
		g_list_foreach(
			items_list,
			(GFunc)ToolBarItemDelete,
			NULL
		);
		g_list_free(items_list);
	}
}


/*
 *	Gets a list of the selected objects.
 *
 *	If copy is TRUE then each object in the returned list will
 *	be a copy of the original object.
 *
 *	Returns a GList of EDVVFSObject * objects. If copy
 *	was TRUE then the calling function must delete each object
 *	in the list and the list itself, otherwise the calling
 *	function need only delete the returned list.
 */
GList *edv_image_browser_get_selected_objects(
	EDVImageBrowser *imbr,
	const gboolean copy
)
{
	gint thumb_num;
	GList *glist, *objs_list;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return(NULL);

	objs_list = NULL;

	tlist = imbr->tlist;
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		thumb_num = (gint)glist->data;
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
		if(obj == NULL)
			continue;

		objs_list = g_list_append(
			objs_list,
			(copy) ? edv_vfs_object_copy(obj) : obj
		);
	}
	 
	return(objs_list);
}

/*
 *	Gets a list of selected object paths.
 *
 *	Returns a GList of gchar * paths. The calling function must
 *	delete each path and the list.
 */
GList *edv_image_browser_get_selected_paths(EDVImageBrowser *imbr)
{
	gint thumb_num;
	GList *glist, *paths_list;
	tlist_struct *tlist;
	EDVVFSObject *obj;

	if(imbr == NULL)
		return(NULL);

	paths_list = NULL;

	tlist = imbr->tlist;
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		thumb_num = (gint)glist->data;
		obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		paths_list = g_list_append(
			paths_list,
			STRDUP(obj->path)
		);
	}
	 
	return(paths_list);
}


/*
 *	Resets the accelkeys displayed on the widgets.
 */
void edv_image_browser_accelkeys_regenerate(EDVImageBrowser *imbr)
{
	GtkAccelGroup *accelgrp = NULL;
	CfgAccelkeyList *ak_list;
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;
			   
	core = imbr->core;
	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_IMBR_ACCELERATOR_KEYS);
	if(ak_list == NULL)
		return;

#define UPDATE_ACCELKEY(_w_,_id_)	{	\
 if(((_w_) != NULL) && ((_id_) > 0)) {		\
  GList *glist;					\
  CfgAccelkey *ak;				\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->op_id == (gint)(_id_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}
			   
	UPDATE_ACCELKEY(
		imbr->open_mi,
		EDV_IMAGE_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
		imbr->open_to_mi,
		EDV_IMAGE_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
		imbr->open_with_mi,
		EDV_IMAGE_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
		imbr->new_file_mi,
		EDV_IMAGE_BROWSER_OP_NEW_FILE
	);
	UPDATE_ACCELKEY(
		imbr->new_directory_mi,
		EDV_IMAGE_BROWSER_OP_NEW_DIRECTORY
	);
	UPDATE_ACCELKEY(
		imbr->new_link_mi,
		EDV_IMAGE_BROWSER_OP_NEW_LINK
	);
	UPDATE_ACCELKEY(
		imbr->new_fifo_mi,
		EDV_IMAGE_BROWSER_OP_NEW_FIFO
	);
	UPDATE_ACCELKEY(
		imbr->new_device_block_mi,
		EDV_IMAGE_BROWSER_OP_NEW_DEVICE_BLOCK
	);
	UPDATE_ACCELKEY(
		imbr->new_device_character_mi,
		EDV_IMAGE_BROWSER_OP_NEW_DEVICE_CHARACTER
	);
	UPDATE_ACCELKEY(
		imbr->new_socket_mi,
		EDV_IMAGE_BROWSER_OP_NEW_SOCKET
	);
	UPDATE_ACCELKEY(
		imbr->sync_disks_mi,
		EDV_IMAGE_BROWSER_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
		imbr->history_mi,
		EDV_IMAGE_BROWSER_OP_HISTORY
	);
	UPDATE_ACCELKEY(
		imbr->run_mi,
		EDV_IMAGE_BROWSER_OP_RUN
	);
	UPDATE_ACCELKEY(
		imbr->close_mi,
		EDV_IMAGE_BROWSER_OP_CLOSE
	);

	UPDATE_ACCELKEY(
		imbr->move_mi,
		EDV_IMAGE_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
		imbr->copy_mi,
		EDV_IMAGE_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
		imbr->link_mi,
		EDV_IMAGE_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
		imbr->rename_mi,
		EDV_IMAGE_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
		imbr->chmod_mi,
		EDV_IMAGE_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		imbr->chown_mi,
		EDV_IMAGE_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		imbr->chtime_mi,
		EDV_IMAGE_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		imbr->delete_mi,
		EDV_IMAGE_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		imbr->copy_path_mi,
		EDV_IMAGE_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
		imbr->copy_url_mi,
		EDV_IMAGE_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
		imbr->paste_mi,
		EDV_IMAGE_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
		imbr->select_all_mi,
		EDV_IMAGE_BROWSER_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
		imbr->unselect_all_mi,
		EDV_IMAGE_BROWSER_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
		imbr->invert_selection_mi,
		EDV_IMAGE_BROWSER_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
		imbr->find_mi,
		EDV_IMAGE_BROWSER_OP_FIND
	);
	UPDATE_ACCELKEY(
		imbr->properties_mi,
		EDV_IMAGE_BROWSER_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
		imbr->mount_mi,
		EDV_IMAGE_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
		imbr->eject_mi,
		EDV_IMAGE_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
		imbr->device_check_mi,
		EDV_IMAGE_BROWSER_OP_DEVICE_CHECK
	);
	UPDATE_ACCELKEY(
		imbr->device_tools_mi,
		EDV_IMAGE_BROWSER_OP_DEVICE_TOOLS
	);
	UPDATE_ACCELKEY(
		imbr->device_format_mi,
		EDV_IMAGE_BROWSER_OP_DEVICE_FORMAT
	);
	UPDATE_ACCELKEY(
		imbr->devices_mi,
		EDV_IMAGE_BROWSER_OP_DEVICES
	);

	UPDATE_ACCELKEY(
		imbr->stop_mi,
		EDV_IMAGE_BROWSER_OP_STOP
	);
	UPDATE_ACCELKEY(
		imbr->continue_mi,
		EDV_IMAGE_BROWSER_OP_CONTINUE
	);

	UPDATE_ACCELKEY(
		imbr->presentation_mode_mi,
		EDV_IMAGE_BROWSER_OP_PRESENTATION_MODE
	);

	UPDATE_ACCELKEY(
		imbr->animation_play_mi,
		EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY
	);
	UPDATE_ACCELKEY(
		imbr->animation_pause_mi,
		EDV_IMAGE_BROWSER_OP_ANIMATION_PAUSE
	);
	UPDATE_ACCELKEY(
		imbr->animation_previous_frame_mi,
		EDV_IMAGE_BROWSER_OP_ANIMATION_PREVIOUS_FRAME
	);
	UPDATE_ACCELKEY(
		imbr->animation_next_frame_mi,
		EDV_IMAGE_BROWSER_OP_ANIMATION_NEXT_FRAME
	);

	UPDATE_ACCELKEY(
		imbr->rotate_transform_cw90_mi,
		EDV_IMAGE_BROWSER_OP_ROTATE_CW90
	);
	UPDATE_ACCELKEY(
		imbr->rotate_transform_ccw90_mi,
		EDV_IMAGE_BROWSER_OP_ROTATE_CCW90
	);
	UPDATE_ACCELKEY(
		imbr->rotate_transform_cw180_mi,
		EDV_IMAGE_BROWSER_OP_ROTATE_CW180
	);

	UPDATE_ACCELKEY(
		imbr->rotate_transform_mirror_horizontal_mi,
		EDV_IMAGE_BROWSER_OP_MIRROR_HORIZONTAL
	);
	UPDATE_ACCELKEY(
		imbr->rotate_transform_mirror_vertical_mi,
		EDV_IMAGE_BROWSER_OP_MIRROR_VERTICAL
	);

	UPDATE_ACCELKEY(
		imbr->refresh_mi,
		EDV_IMAGE_BROWSER_OP_REFRESH
	);
	UPDATE_ACCELKEY(
		imbr->refresh_all_mi,
		EDV_IMAGE_BROWSER_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
		imbr->show_tool_bar_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
		imbr->show_location_bar_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_LOCATION_BAR
	);
	UPDATE_ACCELKEY(
		imbr->show_mount_bar_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_MOUNT_BAR
	);
	UPDATE_ACCELKEY(
		imbr->show_find_bar_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
		imbr->show_status_bar_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
		imbr->thumbs_list_filter_mi,
		EDV_IMAGE_BROWSER_OP_LIST_FILTER
	);
	UPDATE_ACCELKEY(
		imbr->show_hidden_objects_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_HIDDEN_OBJECTS
	);
	UPDATE_ACCELKEY(
		imbr->show_noaccess_objects_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_NOACCESS_OBJECTS
	);
	UPDATE_ACCELKEY(
		imbr->show_nonimage_objects_micheck,
		EDV_IMAGE_BROWSER_OP_SHOW_NONIMAGE_OBJECTS
	);
	UPDATE_ACCELKEY(
		imbr->mime_types_mi,
		EDV_IMAGE_BROWSER_OP_MIME_TYPES
	);


	UPDATE_ACCELKEY(
		imbr->write_protect_mi,
		EDV_IMAGE_BROWSER_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(
		imbr->delete_method_recycle_micheck,
		EDV_IMAGE_BROWSER_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
		imbr->delete_method_purge_micheck,
		EDV_IMAGE_BROWSER_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
		imbr->options_mi,
		EDV_IMAGE_BROWSER_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
		imbr->customize_mi,
		EDV_IMAGE_BROWSER_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
		imbr->window_new_browser_mi,
		EDV_IMAGE_BROWSER_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
		imbr->window_new_imbr_mi,
		EDV_IMAGE_BROWSER_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
		imbr->window_new_archiver_mi,
		EDV_IMAGE_BROWSER_OP_NEW_ARCHIVER
	);
	UPDATE_ACCELKEY(
		imbr->window_recycle_bin_mi,
		EDV_IMAGE_BROWSER_OP_RECYCLE_BIN
	);

	UPDATE_ACCELKEY(
		imbr->help_contents_mi,
		EDV_IMAGE_BROWSER_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
		imbr->help_image_browser_mi,
		EDV_IMAGE_BROWSER_OP_HELP_IMAGE_BROWSER
	);
	UPDATE_ACCELKEY(
		imbr->help_keys_list_mi,
		EDV_IMAGE_BROWSER_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
		imbr->help_mime_types_mi,
		EDV_IMAGE_BROWSER_OP_HELP_MIME_TYPES
	);
	UPDATE_ACCELKEY(
		imbr->help_devices_mi,
		EDV_IMAGE_BROWSER_OP_HELP_DEVICES
	);
	UPDATE_ACCELKEY(
		imbr->help_common_operations_mi,
		EDV_IMAGE_BROWSER_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
		imbr->help_about_mi,
		EDV_IMAGE_BROWSER_OP_HELP_ABOUT
	);

	UPDATE_ACCELKEY(
		imbr->tlist_open_mi,
		EDV_IMAGE_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
		imbr->tlist_open_to_mi,
		EDV_IMAGE_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
		imbr->tlist_open_with_mi,
		EDV_IMAGE_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
		imbr->tlist_mount_mi,
		EDV_IMAGE_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
		imbr->tlist_eject_mi,
		EDV_IMAGE_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
		imbr->tlist_move_mi,
		EDV_IMAGE_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
		imbr->tlist_copy_mi,
		EDV_IMAGE_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
		imbr->tlist_link_mi,
		EDV_IMAGE_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
		imbr->tlist_rename_mi,
		EDV_IMAGE_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
		imbr->tlist_chmod_mi,
		EDV_IMAGE_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		imbr->tlist_chown_mi,
		EDV_IMAGE_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		imbr->tlist_chtime_mi,
		EDV_IMAGE_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		imbr->tlist_delete_mi,
		EDV_IMAGE_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		imbr->tlist_copy_path_mi,
		EDV_IMAGE_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
		imbr->tlist_copy_url_mi,
		EDV_IMAGE_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
		imbr->tlist_paste_mi,
		EDV_IMAGE_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
		imbr->tlist_properties_mi,
		EDV_IMAGE_BROWSER_OP_PROPERTIES
	);


#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the File menu's items.
 */
void edv_image_browser_file_menu_items_regenerate(EDVImageBrowser *imbr)
{
	edv_pixmap_data *icon_data = NULL;  
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu, *submenu;
	GtkMenuShell *menushell;
	CfgList *cfg_list;
	CfgMenu *cfg_menu;
	EDVImageBrowserOpID id;
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get the File menu */
	menu = imbr->file_menu;
	if(menu == NULL)
		return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
		g_list_free(glist);
	}

#define ADD_MENU_ITEM_LABEL	{		\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_image_browser_menu_item_cb, op		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CMD_OBJ_OP(_item_) {	\
 if((_item_) != NULL) {				\
  EDVImageBrowserCommand *cmd;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  cmd = edv_image_browser_command_new();	\
  if(cmd != NULL) {				\
   cmd->image_browser = imbr;			\
   cmd->label = STRDUP((_item_)->label);	\
   cmd->command = STRDUP((_item_)->command);	\
   cmd->description = STRDUP((_item_)->description);\
   cmd->ext_data = STRDUP((_item_)->ext_data);	\
   cmd->flags = (_item_)->flags;		\
						\
   if((_item_)->flags & CFG_MENU_ITEM_IS_SEPARATOR) { \
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_SEPARATOR,		\
     NULL,					\
     NULL,					\
     NULL,					\
     0, 0,					\
     NULL, NULL					\
    );						\
   } else {					\
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_LABEL,			\
     accelgrp,					\
     NULL,					\
     (_item_)->label,				\
     0, 0,					\
     edv_image_browser_menu_item_command_object_op_cb, cmd		\
    );						\
    GUISetMenuItemCrossingCB(			\
     w,						\
     (gpointer)edv_image_browser_menu_item_command_enter_cb, cmd,	\
     (gpointer)edv_image_browser_menu_item_command_leave_cb, cmd	\
    );						\
    if(pixmap != NULL)				\
     GUIMenuItemSetPixmap2(			\
      w,					\
      pixmap, mask				\
     );						\
   }						\
						\
   gtk_object_set_data_full(			\
    GTK_OBJECT(w), EDV_IMAGE_BROWSER_COMMAND_DATA_KEY,	\
    cmd, (GtkDestroyNotify)edv_image_browser_command_delete	\
   );						\
  }						\
						\
  (void)GDK_PIXMAP_UNREF(pixmap);		\
  (void)GDK_BITMAP_UNREF(mask);			\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   NULL,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	id = EDV_IMAGE_BROWSER_OP_OPEN;
	ADD_MENU_ITEM_LABEL
	imbr->open_mi = w;

	id = EDV_IMAGE_BROWSER_OP_OPEN_TO;
	ADD_MENU_ITEM_SUBMENU_LABEL
	imbr->open_to_mi = w;
	imbr->open_to_menu = NULL;

	id = EDV_IMAGE_BROWSER_OP_OPEN_WITH;
	ADD_MENU_ITEM_LABEL
	imbr->open_with_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	/* Customized Object Ops menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_OBJECT_OPS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
		GList *glist;
		CfgMenuItem *item;

		for(glist = cfg_menu->list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			item = CFG_MENU_ITEM(glist->data);
			ADD_MENU_ITEM_CMD_OBJ_OP(item);
		}

		ADD_MENU_ITEM_SEPARATOR
	}

	id = EDV_IMAGE_BROWSER_OP_NEW;
	ADD_MENU_ITEM_SUBMENU_LABEL
	imbr->new_object_mi = w;
	imbr->new_object_menu = submenu = GUIMenuCreate();
	GUIMenuItemSetSubMenu(w, submenu);

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_SYNC_DISKS;
	ADD_MENU_ITEM_LABEL
	imbr->sync_disks_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_HISTORY;
	ADD_MENU_ITEM_LABEL
	imbr->history_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_RUN;
	ADD_MENU_ITEM_LABEL
	imbr->run_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_CLOSE;
	ADD_MENU_ITEM_LABEL
	imbr->close_mi = w;


#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CMD_OBJ_OP
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Recreates the Image Browser's New Object Submenu.
 */
void edv_image_browser_new_object_menu_items_regenerate(EDVImageBrowser *imbr)
{
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;
	GtkMenuShell *menushell;
	CfgList *cfg_list;
	CfgMenu *cfg_menu;
	edv_pixmap_data *icon_data = NULL;
	EDVImageBrowserOpID id;
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Get the New Object submenu */
	menu = imbr->new_object_menu;
	if(menu == NULL)
		return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
		g_list_free(glist);
	}

#define ADD_MENU_ITEM_LABEL	{		\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_image_browser_menu_item_cb, op		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CMD_NEW_OBJ(_item_) {	\
 if((_item_) != NULL) {				\
  EDVImageBrowserCommand *cmd;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  cmd = edv_image_browser_command_new();	\
  if(cmd != NULL) {				\
   cmd->image_browser = imbr;			\
   cmd->label = STRDUP((_item_)->label);	\
   cmd->command = STRDUP((_item_)->command);	\
   cmd->description = STRDUP((_item_)->description);\
   cmd->ext_data = STRDUP((_item_)->ext_data);	\
   cmd->flags = (_item_)->flags;		\
						\
   if((_item_)->flags & CFG_MENU_ITEM_IS_SEPARATOR) { \
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_SEPARATOR,		\
     NULL,					\
     NULL,					\
     NULL,					\
     0, 0,					\
     NULL, NULL					\
    );						\
   } else {					\
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_LABEL,			\
     accelgrp,					\
     NULL,					\
     (_item_)->label,				\
     0, 0,					\
     edv_image_browser_menu_item_command_new_object_cb, cmd		\
    );						\
    GUISetMenuItemCrossingCB(			\
     w,						\
     (gpointer)edv_image_browser_menu_item_command_enter_cb, cmd,	\
     (gpointer)edv_image_browser_menu_item_command_leave_cb, cmd	\
    );						\
    if(pixmap != NULL)				\
     GUIMenuItemSetPixmap2(			\
      w,					\
      pixmap, mask				\
     );						\
   }						\
						\
   gtk_object_set_data_full(			\
    GTK_OBJECT(w), EDV_IMAGE_BROWSER_COMMAND_DATA_KEY,	\
    cmd, (GtkDestroyNotify)edv_image_browser_command_delete	\
   );						\
  }						\
						\
  (void)GDK_PIXMAP_UNREF(pixmap);		\
  (void)GDK_BITMAP_UNREF(mask);			\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	id = EDV_IMAGE_BROWSER_OP_NEW_FILE;
	ADD_MENU_ITEM_LABEL
	imbr->new_file_mi = w;

	id = EDV_IMAGE_BROWSER_OP_NEW_DIRECTORY;
	ADD_MENU_ITEM_LABEL
	imbr->new_directory_mi = w;

	id = EDV_IMAGE_BROWSER_OP_NEW_LINK;
	ADD_MENU_ITEM_LABEL
	imbr->new_link_mi = w;

	/* Customized New Object menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_NEW_OBJECTS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
		GList *glist;
		CfgMenuItem *item;

		ADD_MENU_ITEM_SEPARATOR

		for(glist = cfg_menu->list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			item = CFG_MENU_ITEM(glist->data);
			ADD_MENU_ITEM_CMD_NEW_OBJ(item);
		}

		ADD_MENU_ITEM_SEPARATOR
	}

	id = EDV_IMAGE_BROWSER_OP_NEW_FIFO;
	ADD_MENU_ITEM_LABEL
	imbr->new_fifo_mi = w;

	id = EDV_IMAGE_BROWSER_OP_NEW_DEVICE_BLOCK;
	ADD_MENU_ITEM_LABEL
	imbr->new_device_block_mi = w;

	id = EDV_IMAGE_BROWSER_OP_NEW_DEVICE_CHARACTER;
	ADD_MENU_ITEM_LABEL
	imbr->new_device_character_mi = w;

	id = EDV_IMAGE_BROWSER_OP_NEW_SOCKET;
	ADD_MENU_ITEM_LABEL
	imbr->new_socket_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CMD_NEW_OBJ
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Regenerates the Open To menu.
 */
void edv_image_browser_open_to_menu_regenerate(
	EDVImageBrowser *imbr,
	EDVMIMEType *m
)
{
	GtkWidget *menu, *toplevel;
	EDVCore *core;

	if(imbr == NULL)
		return;

	toplevel = imbr->toplevel;
	core = imbr->core;

	menu = edv_open_to_menu_new_from_mime_type(
		core, m,
		edv_image_browser_get_selected_paths_cb, imbr,
		edv_image_browser_goto_directory_cb, imbr,
		toplevel,
		TRUE			/* Verbose */
	);
	GUIMenuItemSetSubMenu(imbr->open_to_mi, menu);
	imbr->open_to_menu = menu;

	menu = edv_open_to_menu_new_from_mime_type(
		core, m,
		edv_image_browser_get_selected_paths_cb, imbr,
		edv_image_browser_goto_directory_cb, imbr,
		toplevel,
		TRUE			/* Verbose */
	);
	GUIMenuItemSetSubMenu(imbr->tlist_open_to_mi, menu);
	imbr->tlist_open_to_menu = menu;
}

/*
 *	Recreates the Image Browser Thumbs List right-click menu.
 */
void edv_image_browser_list_menu_regenerate(EDVImageBrowser *imbr)
{
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu;
	edv_pixmap_data *icon_data = NULL;
	EDVImageBrowserOpID id;

	if(imbr == NULL)
		return;

	/* Recreate menu */
	GTK_WIDGET_DESTROY(imbr->tlist_menu);
	imbr->tlist_menu = menu = GUIMenuCreate();

#define ADD_MENU_ITEM_LABEL	{		\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_image_browser_menu_item_cb, op		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   NULL,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	id = EDV_IMAGE_BROWSER_OP_OPEN;
	ADD_MENU_ITEM_LABEL
	GUISetMenuItemSetDefault(w, TRUE);
	imbr->tlist_open_mi = w;

	id = EDV_IMAGE_BROWSER_OP_OPEN_TO;
	ADD_MENU_ITEM_SUBMENU_LABEL
	imbr->tlist_open_to_mi = w;

	id = EDV_IMAGE_BROWSER_OP_OPEN_WITH;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_open_with_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_MOUNT;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_mount_mi = w;

	id = EDV_IMAGE_BROWSER_OP_EJECT;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_eject_mi = w;

	ADD_MENU_ITEM_SEPARATOR
	imbr->tlist_mount_misep = w;

	id = EDV_IMAGE_BROWSER_OP_MOVE;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_move_mi = w;

	id = EDV_IMAGE_BROWSER_OP_COPY;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_copy_mi = w;

	id = EDV_IMAGE_BROWSER_OP_LINK;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_link_mi = w;

	id = EDV_IMAGE_BROWSER_OP_RENAME;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_rename_mi = w;

	id = EDV_IMAGE_BROWSER_OP_CHMOD;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_chmod_mi = w;

	id = EDV_IMAGE_BROWSER_OP_CHOWN;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_chown_mi = w;

	id = EDV_IMAGE_BROWSER_OP_CHTIME;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_chtime_mi = w;

	id = EDV_IMAGE_BROWSER_OP_DELETE;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_delete_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_COPY_PATH;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_copy_path_mi = w;

	id = EDV_IMAGE_BROWSER_OP_COPY_URL;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_copy_url_mi = w;

	id = EDV_IMAGE_BROWSER_OP_PASTE;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_paste_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_IMAGE_BROWSER_OP_PROPERTIES;
	ADD_MENU_ITEM_LABEL
	imbr->tlist_properties_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}


/*
 *	Sets the Image Browser's values to the configuration.
 */
void edv_image_browser_sync_configuration(EDVImageBrowser *imbr)
{
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Location history */
	w = imbr->location_combo;
	if(w != NULL)
		edv_save_text_file_glist(
			EDV_GET_S(EDV_CFG_PARM_FILE_IMBR_LOCATION_HISTORY),
			GUIComboGetList(w)
		);

	/* Toplevel GtkWindow position and size */
	CfgGtkGetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_IMBR_X,
		EDV_CFG_PARM_IMBR_Y,
		EDV_CFG_PARM_IMBR_WIDTH,
		EDV_CFG_PARM_IMBR_HEIGHT,
		imbr->toplevel
	);

	/* Main GtkPaned position */
	CfgGtkGetGtkPaned(
		cfg_list,
		EDV_CFG_PARM_IMBR_PANED_POSITION,
		imbr->main_paned
	);

	/* Set the new startup directory as the last directory of
	 * this window's location
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECORD_START_UP_DIR))
		EDV_SET_S(
			EDV_CFG_PARM_DIR_START_UP,
			edv_image_browser_get_location(imbr)
		);

	/* Thumbs list filter */
	if(STRISEMPTY(imbr->thumbs_list_filter))
		EDV_SET_S(
			EDV_CFG_PARM_IMBR_THUMBS_LIST_FILTER,
			"*"
		);
	else
		EDV_SET_S(
			EDV_CFG_PARM_IMBR_THUMBS_LIST_FILTER,
			imbr->thumbs_list_filter
		);
}

/*
 *	Resets the loading process timeout callback and initializes
 *	the loading values.
 */
void edv_image_browser_queue_loading_process(EDVImageBrowser *imbr)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	/* Remove the previous loading timeout callback */
	imbr->loading_tocb = GTK_TIMEOUT_REMOVE(imbr->loading_tocb);

	/* Remove any passive busy levels */
	while(imbr->passive_busy_count > 0)
		edv_image_browser_set_passive_busy(imbr, FALSE);

	/* Reset the stop count */
	imbr->stop_count = 0;

	/* Set the new loading timeout callback */
	imbr->loading_tocb = gtk_timeout_add(
		edv_get_interval_from_load_images_priority(
			(EDVListsLoadImagesPriority)EDV_GET_I(
				EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY
			)
		),
		edv_image_browser_load_thumbs_passive_timeout_cb, imbr
	);

	edv_image_browser_set_passive_busy(imbr, TRUE);
}


/*
 *	Clears the Image Browser's Thumbs List and resets all loading
 *	procedure values, then loads the listing for the new directory
 *	specified by path.
 */
void edv_image_browser_select_path(
	EDVImageBrowser *imbr,
	const gchar *path
)
{
	gchar *dpath;
	CfgList *cfg_list;
	EDVCore *core;

	if((imbr == NULL) || STRISEMPTY(path))
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;

	dpath = g_strdup(path);
	if(dpath == NULL)
		return;

	/* Remove the previous loading timeout callback */
	imbr->loading_tocb = GTK_TIMEOUT_REMOVE(imbr->loading_tocb);

	/* Remove any passive busy levels */
	while(imbr->passive_busy_count > 0)
		edv_image_browser_set_passive_busy(imbr, FALSE);

	/* Clear the image view */
	edv_image_browser_view_clear(imbr);

	/* Update Title & Location Bar */
	edv_image_browser_set_title(imbr, dpath);
	edv_image_browser_set_location(imbr, dpath, FALSE);
	edv_image_browser_update_location_icon(imbr, dpath);

	/* Get the new listing  */
	edv_image_browser_list_get_listing(
		imbr,
		dpath,
		EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
	);

	/* Set the loading progress timeout callback and reset
	 * the loading progress values
	 */
	edv_image_browser_queue_loading_process(imbr);

	g_free(dpath);
}


/*
 *	Creates a new EDVImageBrowserOp.
 */
static EDVImageBrowserOp *edv_image_browser_op_new(void)
{
	return(EDV_IMAGE_BROWSER_OP(g_malloc0(sizeof(EDVImageBrowserOp))));
}

/*
 *	Deletes the EDVImageBrowserOp.
 */
static void edv_image_browser_op_delete(EDVImageBrowserOp *op)
{
	if(op == NULL)
		return;

	g_free(op->button_name);
	g_free(op->menu_name);
	g_free(op->tooltip);
	g_free(op);
}

/*
 *	Creates a new EDVImageBrowserOps list.
 */
GList *edv_image_browser_ops_list_new(EDVImageBrowser *imbr)
{
	guint		accel_key,
			accel_mods;
	const gchar	*button_name,
			*menu_name,
			*tooltip;
	guint8		**button_icon_data,
			**menu_icon_data;
	GList *op_ids_list = NULL;
	ToolBarItemType tb_item_type;
	EDVOpIDFlags flags;
	EDVImageBrowserOpID id;
	void (*func_cb)(ToolBarItem *, gint, gpointer);
	void (*enter_func_cb)(ToolBarItem *, gint, gpointer);

#define APPEND_OPID	{			\
 EDVImageBrowserOp *op = edv_image_browser_op_new(); \
 if(op != NULL) {				\
  op->id = id;					\
  op->tb_item_type = tb_item_type;		\
  op->flags = flags;				\
  op->func_cb = func_cb;			\
  op->enter_func_cb = enter_func_cb;		\
  op->button_name = STRDUP(button_name);	\
  op->menu_name = STRDUP(menu_name);		\
  op->tooltip = STRDUP(tooltip);		\
  op->accel_key = accel_key;			\
  op->accel_mods = accel_mods;			\
  op->button_icon_data = button_icon_data;	\
  op->menu_icon_data = menu_icon_data;		\
  op->image_browser = imbr;			\
  op_ids_list = g_list_append(op_ids_list, op); \
 }						\
}

	func_cb = edv_image_browser_op_cb;
	enter_func_cb = edv_image_browser_op_enter_cb;


	id = EDV_IMAGE_BROWSER_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOL_BAR_ITEM_SEPARATOR;
	button_icon_data = (edv_pixmap_data *)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_CLOSE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Fermer";
	menu_name = "Fermer";
	tooltip = "Fermer cette fentre";
#endif
	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_EXIT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Quitter";
	menu_name = "Quitter";
	tooltip = "Ferme toutes fentres de l'application et sortir";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_SYNC_DISKS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Synchronise";
	menu_name = "Synchronise Les Disques";
	tooltip = "Effectuer des transferts de donnes en attente pour le disque";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HISTORY;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Historique";
	menu_name = "Historique...";
	tooltip = "Historique des commandes effectues dans cette application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_RUN;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Excuter";
	menu_name = "Excuter...";
	tooltip = "Excute un programme";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_RUN_TERMINAL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Ouvrir un terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_WRITE_PROTECT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_write_protect_on_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Protect";
	menu_name = "Master Write Protect";
	tooltip = "Turn the master write protect on/off";
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_NEW;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New";
	menu_name = "New";
	tooltip = "Create a new object";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Nuevo";
	menu_name = "Nuevo";
	tooltip = "Crea un objeto nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Nouveau";
	menu_name = "Nouveau";
	tooltip = "Crer un nouvel objet";
#endif
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_FILE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New File";
	menu_name = "File";
	tooltip = "Create a new file";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Archivo";
	menu_name = "El Archivo";
	tooltip = "Crea un objeto nuevo del archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Fichier";
	menu_name = "Fichier";
	tooltip = "Crer un nouvel objet de fichier";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_DIRECTORY;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_folder_closed_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New Dir";
	menu_name = "Directory";
	tooltip = "Create a new directory";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Carpeta";
	menu_name = "Carpeta";
	tooltip = "Crea una nueva carpeta";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Rpertoire";
	menu_name = "Rpertoire";
	tooltip = "Crer un nouvel objet Rpertoire";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_LINK;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_link_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New Lnk";
	menu_name = "Link";
	tooltip = "Create a new symbolic link";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Enlace";
	menu_name = "Enlace";
	tooltip = "Crea un enlace simblico nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Lien";
	menu_name = "Lien";
	tooltip = "Cre un nouveau lien symbolique";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_FIFO;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_pipe_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New FIFO";
	menu_name = "FIFO Pipe";
	tooltip = "Create a new FIFO pipe";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FIFO";
	menu_name = "FIFO Tubo";
	tooltip = "Crea un objeto nuevo de tubo de FIFO";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Tube FIFO";
	menu_name = "Tube nomm";
	tooltip = "Crer un nouvel objet Tube nomm FIFO";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_DEVICE_BLOCK;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_device_block_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New BDev";
	menu_name = "Block Device";
	tooltip = "Create a new block (buffered) device node";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "BDispositivo";
	menu_name = "El Dispositivo Del Bloque";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de bloque (buffered)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Prif Bloc";
	menu_name = "Priphrique mode blocs";
	tooltip = "Crer un nouveau priphrique en mode d'accs blocs";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_DEVICE_CHARACTER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_device_character_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New CDev";
	menu_name = "Character Device";
	tooltip = "Create a new character (unbuffered) device node";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "CDispositivo";
	menu_name = "El Dispositivo Del Carcter";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de carcter (unbuffered)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Prif carac";
	menu_name = "Priphrique mode Caractres";
	tooltip = "Crer un nouveau priphrique en mode d'accs caractres";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_SOCKET;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_socket_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "New Sock";
	menu_name = "Socket";
	tooltip = "Create a new socket";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Enchufe";
	menu_name = "El Enchufe";
	tooltip = "Crea un objeto nuevo del enchufe";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Prise Rs";
	menu_name = "Prise rseau";
	tooltip = "Crer un nouvel objet Prise de connexion rseau";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_OPEN;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Open";
	menu_name = "Open";
	tooltip = "Open the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Abrir";
	menu_name = "Abrir";
	tooltip = "Abre archivo(s) escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Ouvrir";
	menu_name = "Ouvrir";
	tooltip = "Ouvrir l'objet choisi";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_OPEN_TO;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Open To";
	menu_name = "Open To";
	tooltip = "Open the selected object(s) using a specific method";
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_OPEN_WITH;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Open With";
	menu_name = "Open With...";
	tooltip = "Open the selected object(s) using a specific application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Abrir con";
	menu_name = "Abrir con...";
	tooltip = "Abrir archivo escogido usando un mtodo especfico";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Ouvrir Avec";
	menu_name = "Ouvrir Avec...";
	tooltip = "Ouvrir l'objet choisi en utilisant une mthode spcifique";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_SHIFT_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_COPY_PATH;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "CopyPath";
	menu_name = "Copy Path";
	tooltip = "Copy the location of the selected object(s) as path(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "CopiSende";
	menu_name = "Copie Sendero";
	tooltip = "Copia la ubicacin del objeto(s) escogido como sendero(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "CopiRep";
	menu_name = "Copier Le rpertoire";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme un rpertoire)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_COPY_URL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "CopyURL";
	menu_name = "Copy URL";
	tooltip = "Copy the location of the selected object(s) as url(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "CopiURL";
	menu_name = "Copie URL";
	tooltip = "Copie la ubicacin del objeto(s) escogido como url(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "CopiURL";
	menu_name = "Copier URL";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme url(s)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_PASTE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_paste_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Coller";
	menu_name = "Coller";
	tooltip = "Coller object(s) depuis la soure(s)";
#else
	button_name = "Paste";
	menu_name = "Paste";
	tooltip = "Paste object(s) from coppied location source(s)";
#endif
	accel_key = GDK_v;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_MOVE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_move_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Move";
	menu_name = "Move...";
	tooltip = "Move the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Mueva";
	menu_name = "Mueva...";
	tooltip = "Mueva objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Dplacer";
	menu_name = "Dplacer...";
	tooltip = "Dplacer l'objet choisi(s)";
#endif
	accel_key = GDK_F7;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_COPY;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_copy_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Copy";
	menu_name = "Copy...";
	tooltip = "Copy the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Copia";
	menu_name = "Copia...";
	tooltip = "Copia objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Copier";
	menu_name = "Copier...";
	tooltip = "Copier l'objet choisi(s)";
#endif
	accel_key = GDK_F8;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_LINK;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_link_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Link";
	menu_name = "Link...";
	tooltip = "Link the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Enlace";
	menu_name = "Enlace...";
	tooltip = "Enlace objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Lier";
	menu_name = "Lier...";
	tooltip = "Lier l'objet(s) choisi";
#endif
	accel_key = GDK_F6;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_RENAME;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_rename_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Rename";
	menu_name = "Rename";
	tooltip = "Rename the selected object";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Renombrar";
	menu_name = "Renombrar";
	tooltip = "Renombrar archivo escogido";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Renommer";
	menu_name = "Renommer";
	tooltip = "Renomme l'objet choisi";
#endif
	accel_key = GDK_F2;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_CHMOD;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_chmod_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "ChMod";
	menu_name = "Change Permissions...";
	tooltip = "Change the permissions of the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "ChMod";
	menu_name = "Cambie permisos...";
	tooltip = "Cambiar los permisos de los objetos escogidos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "ChMod";
	menu_name = "Changer Mode...";
	tooltip = "Changer les permissions de l'objet(s) choisi";
#endif
	accel_key = GDK_F9;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_CHOWN;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_owned_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Change the ownership of the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "ChOwn";
	menu_name = "Changer propritaire...";
	tooltip = "Changer le propritaire de l'objet(s)";
#endif
	accel_key = GDK_F10;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_CHTIME;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_time_stamp_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "ChTime";
	menu_name = "Change Time Stamps...";
	tooltip = "Change the time stamps of the selected object(s)";
	accel_key = GDK_F11;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DELETE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Borre";
	menu_name = "Borre";
	tooltip = "Borre objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Supprimer";
	menu_name = "Supprimer";
	tooltip = "Supprime l'objet(s) choisi";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_PROPERTIES;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_properties_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "Modify the selected object's properties";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Props";
	menu_name = "Modifier Proprits";
	tooltip = "Modifier les proprits de l'objet";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_SELECT_ALL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "SelAll";
	menu_name = "Choisir Tous";
	tooltip = "Choisir tous objets";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_UNSELECT_ALL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "UnSelAll";
	menu_name = "Choisir aucun";
	tooltip = "Aucun objet choisi";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_INVERT_SELECTION;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip = "Inverser les objets slectionns";

#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_FIND;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_search_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Find";
	menu_name = "Recherche";
	tooltip = "Rechercher un objet suivant les attributs spcifis ";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_DOWNLOAD;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_download_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download an object";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download del objeto";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Descendre";
	menu_name = "Descendre";
	tooltip = "Descendre l'objet (download)";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_REFRESH;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the thumbs list";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque la carpeta escogida para listar";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit le rpertoire slectionn";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_REFRESH_ALL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_GOTO_PARENT;	
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_folder_parent_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Parent";
	menu_name = "Go To Parent";
	tooltip = "Go to the parent directory";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Carpeta padre";
	menu_name = "Ir a carpeta padre";
	tooltip = "Para ir a carpeta padre";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Parent";
	menu_name = "Aller Au Parent";
	tooltip = "Aller au rpertoire parent";
#endif
	accel_key = GDK_BackSpace;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_GOTO_HOME;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_folder_home_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Accueil";
	menu_name = "Aller  l'accueil";
	tooltip = "Aller au rpertoire d'accueil (HOME)";
#else
	button_name = "Home";
	menu_name = "Go To Home";
	tooltip = "Go to the home directory";
#endif
	accel_key = GDK_grave;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_STOP;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_stop_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Parada";
	menu_name = "La Parada Cargando";
	tooltip = "Para el proceso que carga";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Arrt";
	menu_name = "Arrt du chargement";
	tooltip = "Arrte le procd de chargement (download)";
#else
	button_name = "Stop";
	menu_name = "Stop Loading";
	tooltip = "Stop the loading process";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_CONTINUE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_go_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Contine";
	menu_name = "Siga Carga";
	tooltip = "Contine el proceso que carga";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Poursuivre";
	menu_name = "Poursuivre le Chargement";
	tooltip = "Poursuivre le procd de chargement";
#else
	button_name = "Continue";
	menu_name = "Continue Loading";
	tooltip = "Continue the loading process";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_PRESENTATION_MODE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_monitor2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Diaporama";
	menu_name = "Prsentation en mode Diaporama";
	tooltip = "Commuter en mode prsentation Diaporama (plein cran)";
#else
	button_name = "Present";
	menu_name = "Presentation Mode";
	tooltip = "Switch to presentation mode (fullscreen slideshow)";
#endif
	accel_key = GDK_F11;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_PREVIOUS_IMAGE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_prev_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Prev";
	menu_name = "Previous Image";
	tooltip = "Go to the previous image";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEXT_IMAGE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_next_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Next";
	menu_name = "Next Image";
	tooltip = "Go to the next image";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_ANIMATION;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_play_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Anim";
	menu_name = "Animation";
	tooltip = "Control the animation";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_play_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Play";
	menu_name = "Play";
	tooltip = "Play the animation";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ANIMATION_PAUSE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_pause_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Pause";
	menu_name = "Pause";
	tooltip = "Pause the animation";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY_PAUSE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_play_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Play/Pause";
	menu_name = "Play/Pause";
	tooltip = "Play/pause the animation";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ANIMATION_PREVIOUS_FRAME;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_previous_track_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Prev";
	menu_name = "Previous Frame";
	tooltip = "Go to the previous frame";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ANIMATION_NEXT_FRAME;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_next_track_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Next";
	menu_name = "Next Frame";
	tooltip = "Go to the next frame";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_ROTATE_TRANSFORM;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_rotate_cw_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "RotTrans";
	menu_name = "Rotate/Transform";
	tooltip = "Rotate or transform the image";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ROTATE_CW90;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_rotate_cw_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "CW90";
	menu_name = "90 Clockwise";
	tooltip = "Rotate the image 90 clockwise";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ROTATE_CCW90;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_rotate_ccw_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "CCW90";
	menu_name = "90 Counter-Clockwise";
	tooltip = "Rotate the image 90 counter-clockwise";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_ROTATE_CW180;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_rotate_cw_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "CW180";
	menu_name = "180 Clockwise";
	tooltip = "Rotate the image 180 clockwise";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_MIRROR_HORIZONTAL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_mirror_horizontal_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "MirrorH";
	menu_name = "Mirror Horizontal";
	tooltip = "Flip the image along its Y axis";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_MIRROR_VERTICAL;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_mirror_vertical_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "MirrorV";
	menu_name = "Mirror Vertical";
	tooltip = "Flip the image along its X axis";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_SHOW_TOOL_BAR;
	flags = EDV_OPID_EX_TOOL_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Outils";
	menu_name = "Barre d'outils";
	tooltip = "Monter/cacher la barre d'outil";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_LOCATION_BAR;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "Show/hide location bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La exposicin/la barra de la ubicacin del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Adresse";
	menu_name = "Barre d'adresse";
	tooltip = "Monter/cacher la barre d'emplacement";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_MOUNT_BAR;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "Show/hide mount bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "La Exposicin/la barra del monte del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Montage";
	menu_name = "Barre de montage";
	tooltip = "Monter/cacher la barre de montage";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_FIND_BAR;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Recherche";
	menu_name = "Barre de recherche";
	tooltip = "Monter/cacher la barre de recherche";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_STATUS_BAR;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Etat";
	menu_name = "Barre d'tat";
	tooltip = "Monter/cacher la barre d'tat";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_LIST_FILTER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Filter...";
	tooltip = "Set the name filter for listing objects";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_file_hidden_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "HidObj";
	menu_name = "Objets Cachs";
	tooltip = "Monter/cacher objets cachs(.)";
#else
	button_name = "HidObj";
	menu_name = "Hidden Objects";
	tooltip = "Show/hide hidden objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_file_noaccess_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "NoAccObj";
	menu_name = "Objet  accs rserv";
	tooltip = "Monter/cacher les objets  accs rserv";
#else
	button_name = "NoAccObj";
	menu_name = "No Access Objects";
	tooltip = "Show/hide no access objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_SHOW_NONIMAGE_OBJECTS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_file_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "NonImgObj";
	menu_name = "Autres qu'Image";
	tooltip = "Monter/cacher les objets non-image";
#else
	button_name = "NonImgObj";
	menu_name = "Non-Image Objects";
	tooltip = "Show/hide non-image objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_MOUNT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Mount";
	menu_name = "Mount";
	tooltip = "Mount/unmount selected mount point directory";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Montar";
	menu_name = "Montar";
	tooltip = "Montar/desmontar carpeta de punto de montaje seleccionada";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Monter";
	menu_name = "Monter...";
	tooltip = "Monter/dmonter le composant sur le rpertoire de montage";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_EJECT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_eject_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Eject";
	menu_name = "Eject";
	tooltip = "Eject media from device";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Expulse";
	menu_name = "Expulse";
	tooltip = "Expulse medios del Dispositivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Ejecter";
	menu_name = "Ejecter";
	tooltip = "Ejecter le support du composant";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DEVICE_CHECK;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_fsck_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "FSCK";
	menu_name = "File System Check...";
	tooltip = "Check the device's file system";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FSCK";
	menu_name = "Archive Cheque De Sistema...";
	tooltip = "Verifique el sistema del archivo de Dispositivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "FSCK";
	menu_name = "Vrifier le systme de fichier...";
	tooltip = "Vrifier le systme de fichier du composant";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DEVICE_TOOLS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_tools_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "DevTool";
	menu_name = "Device Tools...";
	tooltip = "Run the device's tools program";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "DevTool";
	menu_name = "Herramienta Equipar Dispositivo  ...";
	tooltip = "Corra el programa de herramientas de Dispositivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Panneau";
	menu_name = "Panneau du composant...";
	tooltip = "Ouvrir le panneau de commande du composant";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DEVICE_FORMAT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_floppy_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Format";
	menu_name = "Format Media...";
	tooltip = "Format the media in the device";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Format";
	menu_name = "Formatear Medios...";
	tooltip = "Formatear los medios en el Dispositivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Formatter";
	menu_name = "Formatter le composant...";
	tooltip = "Formatter le support dans le composant";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_MIME_TYPES;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Voir/modifier MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_DEVICES;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_devices_list_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Devices";
	menu_name = "Devices...";
	tooltip = "View/modify file system device references";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Dispositivos";
	menu_name = "Los Dispositivos...";
	tooltip = "El Panorama/modifica las referencias de Dispositivo de sistema de archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Composant";
	menu_name = "Composant...";
	tooltip = "Voir/modifier la configuration du systme de fichier";
#endif
	accel_key = GDK_d;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_NEW_BROWSER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_file_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Dossier";
	menu_name = "Nouveau navigateur de Fichier";
	tooltip = "Crer une nouvelle fentre de navigation Dossier";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_IMBR;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Galerie";
	menu_name = "Nouveau navigateur de Galerie d'image";
	tooltip = "Crer un nouvelle fentre de navigation de galerie d'image";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_NEW_ARCHIVER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Archive";
	menu_name = "Nouvelle Archive";
	tooltip = "Crer une nouvelle fentre d'archivage";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_RECYCLE_BIN;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "RecBin";
	menu_name = "Recycle Bin";
	tooltip = "Open Recycle Bin";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "RecBin";
	menu_name = "El Cajn De Reciclaje";
	tooltip = "Abre Cajn de Reciclaje (basura)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Corbeille";
	menu_name = "Corbeille";
	tooltip = "Corbeille(la poubelle)";
#endif
	accel_key = GDK_4;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_OPTIONS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_options_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Options";
	menu_name = "Options";
	tooltip = "Modifier les options de l'application";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_CUSTOMIZE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Prfrences";
	menu_name = "Prfrences...";
	tooltip = "Rgler les prfrences pour cette application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_IMAGE_BROWSER_OP_HELP_CONTENTS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Aide";
	menu_name = "Contenu de l'aide";
	tooltip = "Index des pages d'aide";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc";
	menu_name = "Tematy pomocy";
	tooltip = "Spis treci wszystkich stron pomocy";
#else
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_FILE_BROWSER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_IMAGE_BROWSER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_ARCHIVER;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_RECYCLE_BIN;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Aide";
	menu_name = "Corbeille";
	tooltip = "Aide sur l'utilisation de la Corbeille";
#else
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_KEYS_LIST;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Clavier";
	menu_name = "Liste des touches raccourci clavier";
	tooltip = "Liste des oprations principales disponibles avec les touches clavier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocKlaw";
	menu_name = "Lista klawiszy";
	tooltip = "Lista skrtw klawiaturowych";
#else
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_MIME_TYPES;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Ayude a establecer MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Aide sur les types MIME";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc MIME";
	menu_name = "Typy MIME";
	tooltip = "Pomoc w ustawianiu typw MIME";
#else
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Help on setting up MIME Types";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_DEVICES;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpDev";
	menu_name = "Los Dispositivos";
	tooltip = "La Ayuda a establecer Dispositivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Composant";
	menu_name = "Composant";
	tooltip = "Aide sur les composants";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomUrzdzenia";
	menu_name = "Urzdzenia";
	tooltip = "Pomoc w ustawianiu urzdze";
#else
	button_name = "HelpDev";
	menu_name = "Devices";
	tooltip = "Help on setting up devices";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_COMMON_OPERATIONS;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpOp";
	menu_name = "Las Operaciones Comunes";
	tooltip = "La Ayuda en las operaciones ms comunes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Oprations";
	menu_name = "Oprations classiques";
	tooltip = "Aide sur les opration";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomOperacje";
	menu_name = "Najczstsze operacje";
	tooltip = "Pomoc dotyczca najczstszych operacji";
#else
	button_name = "HelpOp";
	menu_name = "Common Operations";
	tooltip = "Help on the most common operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_IMAGE_BROWSER_OP_HELP_ABOUT;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
	menu_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "A propos";
	menu_name = "A propos";
	tooltip = "A propos de cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "O programie";
	menu_name = "O programie";
	tooltip = "Informacje o programie";
#else
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	return(op_ids_list);
#undef APPEND_OPID
}

/*
 *	Deletes the EDVImageBrowserOps list.
 */
void edv_image_browser_ops_list_delete(GList *op_ids_list)
{
	if(op_ids_list == NULL)
		return;

	g_list_foreach(
		op_ids_list,
		(GFunc)edv_image_browser_op_delete,
		NULL
	);
	g_list_free(op_ids_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
CfgAccelkeyList *edv_image_browser_accelkeys_new(void)
{
	GList		*glist,
			*op_ids_list,
			*ak_glist = NULL;
	CfgAccelkeyList *ak_list;
	EDVImageBrowserOp *op;

	op_ids_list = edv_image_browser_ops_list_new(NULL);
	for(glist = op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		op = EDV_IMAGE_BROWSER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->flags & EDV_OPID_NO_OP)
			continue;

		ak_glist = g_list_append(
			ak_glist,
			CFGAccelkeyNew(
				(gint)op->id,
				op->accel_key,
				op->accel_mods
			)
		);
	}
	edv_image_browser_ops_list_delete(op_ids_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Reloads the icons.
 */
void edv_image_browser_reload_icons(EDVImageBrowser *imbr)
{
	const gchar *filename;
	gchar *icons_path;
	CfgList *cfg_list;
	EDVIconSize icon_size;
	EDVCore *core;

	if(imbr == NULL)
		return;

	core = imbr->core;
	cfg_list = core->cfg_list;
	icon_size = (EDVIconSize)EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_ICON_SIZE);

	/* Get the icons directory */
	icons_path = g_strconcat(
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
		EDV_NAME_ICONS_SUBDIR,
		NULL
	);

#define RELOAD_ICON(_icon_,_path_)	{		\
 gchar *path;						\
							\
 (_icon_) = edv_pixmap_unref(_icon_);			\
							\
 path = g_strconcat(					\
  icons_path,						\
  G_DIR_SEPARATOR_S,					\
  (_path_),						\
  NULL							\
 );							\
 if(path != NULL) {					\
  (_icon_) = edv_pixmap_new_from_file(path);		\
  g_free(path);						\
 }							\
}

	/* Parent Directory */
	filename = NULL;
	switch(icon_size)
	{
	  case EDV_ICON_SIZE_48:
		filename = "icon_folder_parent_48x48.xpm";
		break;
	  case EDV_ICON_SIZE_32:
		filename = "icon_folder_parent_32x32.xpm";
		break;
	  case EDV_ICON_SIZE_20:
		filename = "icon_folder_parent_20x20.xpm";
		break;
	  case EDV_ICON_SIZE_16:
		break;
	}
	RELOAD_ICON(
		imbr->directory_parent_icon,
		filename
	);

	/* Bad Image */
	filename = NULL;
	switch(icon_size)
	{
	  case EDV_ICON_SIZE_48:
		filename = "icon_file_image_bad_48x48.xpm";
		break;
	  case EDV_ICON_SIZE_32:
		filename = "icon_file_image_bad_32x32.xpm";
		break;
	  case EDV_ICON_SIZE_20:
		filename = "icon_file_image_bad_20x20.xpm";
		break;
	  case EDV_ICON_SIZE_16:
		break;
	}
	RELOAD_ICON(
		imbr->file_image_bad_icon,
		filename
	);

	/* Pending Image */
	filename = NULL;
	switch(icon_size)
	{
	    case EDV_ICON_SIZE_48:
		filename = "icon_file_image_pending_48x48.xpm";
		break;
	    case EDV_ICON_SIZE_32:
		filename = "icon_file_image_pending_32x32.xpm";
		break;
	    case EDV_ICON_SIZE_20:
		filename = "icon_file_image_pending_20x20.xpm";
		break;
	    case EDV_ICON_SIZE_16:
		break;
	}
	RELOAD_ICON(
		imbr->file_image_pending_icon,
		filename
	);

	/* Multiple Frames Icon */
	RELOAD_ICON(
		imbr->multiple_frames_icon,
		"icon_video_20x20.xpm"
	);

#undef RELOAD_ICON

	g_free(icons_path);
}


/*
 *	Creates the menu bar.
 */
static void EDVImbrCreateMenuBar(
	EDVImageBrowser *imbr,
	GtkWidget *parent
)
{
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu,
			*submenu,
			*menu_bar;
	edv_pixmap_data *icon_data = NULL;
	EDVImageBrowserOpID id;

	/* Create the GtkMenuBar */
	imbr->menu_bar = menu_bar = w = GUIMenuBarCreate(NULL);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_MENU_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

#define ADD_MENU_ITEM_LABEL	{		\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_image_browser_menu_item_cb, op		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CHECK	{		\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_CHECK,			\
   accelgrp,					\
   NULL,					\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_image_browser_menu_item_cb, op		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVImageBrowserOp *op =			\
  edv_image_browser_op_match_by_id(imbr, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, NULL,	\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_image_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_image_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	/* Create menus */

	/* File Menu */
	imbr->file_menu = menu = GUIMenuCreateTearOff();
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#else
"File"
#endif
		, GTK_JUSTIFY_LEFT
	);
	edv_image_browser_file_menu_items_regenerate(imbr);

	/* Edit Menu */
	imbr->edit_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_MOVE;
		ADD_MENU_ITEM_LABEL
		imbr->move_mi = w;

		id = EDV_IMAGE_BROWSER_OP_COPY;
		ADD_MENU_ITEM_LABEL
		imbr->copy_mi = w;

		id = EDV_IMAGE_BROWSER_OP_LINK;
		ADD_MENU_ITEM_LABEL
		imbr->link_mi = w;

		id = EDV_IMAGE_BROWSER_OP_RENAME;
		ADD_MENU_ITEM_LABEL
		imbr->rename_mi = w;

		id = EDV_IMAGE_BROWSER_OP_CHMOD;
		ADD_MENU_ITEM_LABEL
		imbr->chmod_mi = w;

		id = EDV_IMAGE_BROWSER_OP_CHOWN;
		ADD_MENU_ITEM_LABEL
		imbr->chown_mi = w;

		id = EDV_IMAGE_BROWSER_OP_CHTIME;
		ADD_MENU_ITEM_LABEL
		imbr->chtime_mi = w;

		id = EDV_IMAGE_BROWSER_OP_DELETE;
		ADD_MENU_ITEM_LABEL
		imbr->delete_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_COPY_PATH;
		ADD_MENU_ITEM_LABEL
		imbr->copy_path_mi = w;

		id = EDV_IMAGE_BROWSER_OP_COPY_URL;
		ADD_MENU_ITEM_LABEL
		imbr->copy_url_mi = w;

		id = EDV_IMAGE_BROWSER_OP_PASTE;
		ADD_MENU_ITEM_LABEL    
		imbr->paste_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_SELECT_ALL;
		ADD_MENU_ITEM_LABEL
		imbr->select_all_mi = w;

		id = EDV_IMAGE_BROWSER_OP_UNSELECT_ALL;
		ADD_MENU_ITEM_LABEL
		imbr->unselect_all_mi = w;

		id = EDV_IMAGE_BROWSER_OP_INVERT_SELECTION;
		ADD_MENU_ITEM_LABEL
		imbr->invert_selection_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_FIND;
		ADD_MENU_ITEM_LABEL
		imbr->find_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_PROPERTIES;
		ADD_MENU_ITEM_LABEL
		imbr->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* View Menu */
	imbr->view_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_STOP;
		ADD_MENU_ITEM_LABEL
		imbr->stop_mi = w;

		id = EDV_IMAGE_BROWSER_OP_CONTINUE;
		ADD_MENU_ITEM_LABEL
		imbr->continue_mi = w;

		id = EDV_IMAGE_BROWSER_OP_PRESENTATION_MODE;
		ADD_MENU_ITEM_LABEL
		imbr->presentation_mode_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_ANIMATION;
		ADD_MENU_ITEM_SUBMENU_LABEL
		imbr->animation_mi = w;
		imbr->animation_menu = submenu = GUIMenuCreate();
		if(submenu != NULL)
		{
			GtkWidget	*w,
							*menu = submenu;

			id = EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY;
			ADD_MENU_ITEM_LABEL
			imbr->animation_play_mi = w;

			id = EDV_IMAGE_BROWSER_OP_ANIMATION_PAUSE;
			ADD_MENU_ITEM_LABEL
			imbr->animation_pause_mi = w;

			id = EDV_IMAGE_BROWSER_OP_ANIMATION_PREVIOUS_FRAME;
			ADD_MENU_ITEM_LABEL
			imbr->animation_previous_frame_mi = w;

			id = EDV_IMAGE_BROWSER_OP_ANIMATION_NEXT_FRAME;
			ADD_MENU_ITEM_LABEL
			imbr->animation_next_frame_mi = w;
		}
		GUIMenuItemSetSubMenu(w, submenu);

		id = EDV_IMAGE_BROWSER_OP_ROTATE_TRANSFORM;
		ADD_MENU_ITEM_SUBMENU_LABEL
		imbr->rotate_transform_mi = w;
		imbr->rotate_transform_menu = submenu = GUIMenuCreate();
		if(submenu != NULL)
		{
			GtkWidget	*w,
							*menu = submenu;

			id = EDV_IMAGE_BROWSER_OP_ROTATE_CW90;
			ADD_MENU_ITEM_LABEL
			imbr->rotate_transform_cw90_mi = w;

			id = EDV_IMAGE_BROWSER_OP_ROTATE_CCW90;
			ADD_MENU_ITEM_LABEL
			imbr->rotate_transform_ccw90_mi = w;

			id = EDV_IMAGE_BROWSER_OP_ROTATE_CW180;
			ADD_MENU_ITEM_LABEL
			imbr->rotate_transform_cw180_mi = w;

			id = EDV_IMAGE_BROWSER_OP_MIRROR_HORIZONTAL;
			ADD_MENU_ITEM_LABEL
			imbr->rotate_transform_mirror_horizontal_mi = w;

			id = EDV_IMAGE_BROWSER_OP_MIRROR_VERTICAL;
			ADD_MENU_ITEM_LABEL
			imbr->rotate_transform_mirror_vertical_mi = w;
		}
		GUIMenuItemSetSubMenu(w, submenu);

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_REFRESH;
		ADD_MENU_ITEM_LABEL
		imbr->refresh_mi = w;

		id = EDV_IMAGE_BROWSER_OP_REFRESH_ALL;
		ADD_MENU_ITEM_LABEL
		imbr->refresh_all_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_SHOW_TOOL_BAR;
		ADD_MENU_ITEM_CHECK
		imbr->show_tool_bar_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_LOCATION_BAR;
		ADD_MENU_ITEM_CHECK
		imbr->show_location_bar_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_MOUNT_BAR;
		ADD_MENU_ITEM_CHECK
		imbr->show_mount_bar_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_FIND_BAR;
		ADD_MENU_ITEM_CHECK
		imbr->show_find_bar_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_STATUS_BAR;
		ADD_MENU_ITEM_CHECK
		imbr->show_status_bar_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
		ADD_MENU_ITEM_CHECK
		imbr->show_hidden_objects_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
		ADD_MENU_ITEM_CHECK
		imbr->show_noaccess_objects_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_SHOW_NONIMAGE_OBJECTS;
		ADD_MENU_ITEM_CHECK
		imbr->show_nonimage_objects_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_LIST_FILTER;
		ADD_MENU_ITEM_LABEL
		imbr->thumbs_list_filter_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		imbr->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Voir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#else
"View"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Device Menu */
	imbr->device_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_MOUNT;
		ADD_MENU_ITEM_LABEL
		imbr->mount_mi = w;

		id = EDV_IMAGE_BROWSER_OP_EJECT;
		ADD_MENU_ITEM_LABEL
		imbr->eject_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_DEVICE_CHECK;
		ADD_MENU_ITEM_LABEL
		imbr->device_check_mi = w;

		id = EDV_IMAGE_BROWSER_OP_DEVICE_TOOLS;
		ADD_MENU_ITEM_LABEL
		imbr->device_tools_mi = w;

		id = EDV_IMAGE_BROWSER_OP_DEVICE_FORMAT;
		ADD_MENU_ITEM_LABEL
		imbr->device_format_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_DEVICES;
		ADD_MENU_ITEM_LABEL
		imbr->devices_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning"
#else
"Device"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Settings Menu */
	imbr->settings_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_WRITE_PROTECT;
		ADD_MENU_ITEM_LABEL
		imbr->write_protect_mi = w;

		id = EDV_IMAGE_BROWSER_OP_DELETE_METHOD_RECYCLE;
		ADD_MENU_ITEM_CHECK
		imbr->delete_method_recycle_micheck = w;

		id = EDV_IMAGE_BROWSER_OP_DELETE_METHOD_PURGE;
		ADD_MENU_ITEM_CHECK
		imbr->delete_method_purge_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_OPTIONS;
		ADD_MENU_ITEM_LABEL
		imbr->options_mi = w;

		id = EDV_IMAGE_BROWSER_OP_CUSTOMIZE;
		ADD_MENU_ITEM_LABEL
		imbr->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rglages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#else
"Settings"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Windows Menu */
	imbr->windows_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_NEW_BROWSER;
		ADD_MENU_ITEM_LABEL
		imbr->window_new_browser_mi = w;

		id = EDV_IMAGE_BROWSER_OP_NEW_IMBR;
		ADD_MENU_ITEM_LABEL
		imbr->window_new_imbr_mi = w;

		id = EDV_IMAGE_BROWSER_OP_NEW_ARCHIVER;
		ADD_MENU_ITEM_LABEL
		imbr->window_new_archiver_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_RECYCLE_BIN;
		ADD_MENU_ITEM_LABEL
		imbr->window_recycle_bin_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fentres"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#else
"Windows"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Help Menu */
	imbr->help_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_IMAGE_BROWSER_OP_HELP_CONTENTS;
		ADD_MENU_ITEM_LABEL
		imbr->help_contents_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_HELP_IMAGE_BROWSER;
		ADD_MENU_ITEM_LABEL
		imbr->help_image_browser_mi = w;

		id = EDV_IMAGE_BROWSER_OP_HELP_KEYS_LIST;
		ADD_MENU_ITEM_LABEL
		imbr->help_keys_list_mi = w;

		id = EDV_IMAGE_BROWSER_OP_HELP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		imbr->help_mime_types_mi = w;

		id = EDV_IMAGE_BROWSER_OP_HELP_DEVICES;
		ADD_MENU_ITEM_LABEL
		imbr->help_devices_mi = w;

		id = EDV_IMAGE_BROWSER_OP_HELP_COMMON_OPERATIONS;
		ADD_MENU_ITEM_LABEL
		imbr->help_common_operations_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_IMAGE_BROWSER_OP_HELP_ABOUT;
		ADD_MENU_ITEM_LABEL
		imbr->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#else
"Help"
#endif
		, GTK_JUSTIFY_RIGHT
	);

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Creates a new Image Browser.
 */
EDVImageBrowser *edv_image_browser_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gchar *wm_name = NULL, *wm_class;
	gpointer combo_rtn;
	GList *glist;
	GtkOrientation orientation;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*toplevel;
	GtkCombo *combo;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT},
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	CfgList *cfg_list;
	tlist_struct *tlist;
	imgview_struct *iv;
	EDVImageBrowser *imbr;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	/* Create the Image Browser */
	imbr = EDV_IMBR(g_malloc0(sizeof(EDVImageBrowser)));
	if(imbr == NULL)
		return(NULL);

	gtk_widget_push_visual(gdk_rgb_get_visual());
	gtk_widget_push_colormap(gdk_rgb_get_cmap());
	imbr->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
	imbr->core = core;
#if 0
	imbr->flags = 0;
	imbr->freeze_count = 0;
	imbr->busy_count = 0;
	imbr->passive_busy_count = 0;
	imbr->stop_count = 0;
	imbr->loading_tocb = 0;
#endif
	imbr->op_ids_list = edv_image_browser_ops_list_new(imbr);

	if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR))
		imbr->flags |= EDV_IMAGE_BROWSER_TOOL_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR))
		imbr->flags |= EDV_IMAGE_BROWSER_LOCATION_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR))
		imbr->flags |= EDV_IMAGE_BROWSER_MOUNT_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_FIND_BAR))
		imbr->flags |= EDV_IMAGE_BROWSER_FIND_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR))
		imbr->flags |= EDV_IMAGE_BROWSER_STATUS_BAR_MAPPED;

	imbr->thumbs_list_filter = STRDUP(EDV_GET_S(
		EDV_CFG_PARM_IMBR_THUMBS_LIST_FILTER
	));
	imbr->tlist_selected_thumb = -1;
	imbr->selected_dev_num = -1;
	imbr->last_nrecycle_bin_items = -1;
	imbr->last_write_protect_state = -1;

	imbr->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w), EDV_IMAGE_BROWSER_WM_CLASS_WINDOW_NAME, PROG_NAME
		);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_IMAGE_BROWSER_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);

		if((geometry_flags & GDK_GEOMETRY_WIDTH) || (geometry_flags & GDK_GEOMETRY_HEIGHT))
			gtk_widget_set_usize(w, geometry->width, geometry->height);
		else
			CfgGtkSetGtkWindow(
				cfg_list,
				NULL, NULL,
				EDV_CFG_PARM_IMBR_WIDTH,
				EDV_CFG_PARM_IMBR_HEIGHT,
				w
			);
	}
	else
	{
		CfgGtkSetGtkWindow(
			cfg_list,
			EDV_CFG_PARM_IMBR_X,
			EDV_CFG_PARM_IMBR_Y,
			EDV_CFG_PARM_IMBR_WIDTH,
			EDV_CFG_PARM_IMBR_HEIGHT,
			w
		);
	}
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_image_browser_realize_cb), imbr
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_image_browser_delete_event_cb), imbr
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_image_browser_key_event_cb), imbr
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_image_browser_key_event_cb), imbr
	);
	parent = w;


	/* Main GtkVBox */
	imbr->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Bars GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Menu Bar GtkHandleBox */
	imbr->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_attached_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_detached_cb), imbr
	);
	gtk_widget_show(w);
	/* Create menu bar */
	EDVImbrCreateMenuBar(imbr, w);


	/* GtkHandleBox for the Tool Bar */
	imbr->tool_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_attached_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_detached_cb), imbr
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(imbr->flags & EDV_IMAGE_BROWSER_TOOL_BAR_MAPPED)
		gtk_widget_show(w);
	/* Build the Tool Bar later */
/*	imbr->tool_bar = NULL; */


	/* GtkHandleBox for the Location Bar */
	imbr->location_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_attached_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_detached_cb), imbr
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(imbr->flags & EDV_IMAGE_BROWSER_LOCATION_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Location Bar GtkHBox */
	imbr->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;


	/* Location Icon and Combo GtkHBox */
	w = gtk_hbox_new(FALSE, 0);	/* Don't need spacing between icon
					 * and combo because combo has it
					 * already
					 */
	gtk_widget_set_name(w, EDV_WIDGET_NAME_LOCATION_BAR);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Vbox to orient location icon fixed parent */
	w = gtk_vbox_new(TRUE, 0);
	gtk_widget_set_usize(w, 20, 20);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Location icon fixed parent */
	imbr->location_icon_fixed = w = gtk_fixed_new();
	gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_image_browser_location_bar_icon_realize_cb), imbr
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_image_browser_location_bar_crossing_cb), imbr
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_image_browser_location_bar_crossing_cb), imbr
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
"Drag this to create a link to this location"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Arrastre esto para crear un enlace en esta ubicacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Traner ceci pour crer un lien  cet emplacement"
#endif
	);
	gtk_widget_show(w);

	/* Reset the location icon pixmap */
	imbr->location_icon_pm = NULL;

	/* Location GtkCombo */
	glist = edv_open_text_file_glist(
		EDV_GET_S(EDV_CFG_PARM_FILE_IMBR_LOCATION_HISTORY),
		-1,				/* No lines limit */
		TRUE				/* Strip CRs */
	);
	w = GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
		"Location:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"La Ubicacin:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Emplacement:",
#endif
		"/",
		glist,
		20,				/* Maximum items */
		&combo_rtn,
		imbr,
		edv_image_browser_combo_activate_cb,
		NULL
	);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)g_free, NULL);
		g_list_free(glist);
		glist = NULL;
	}
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	imbr->location_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	GUIDNDSetSrc(
		imbr->location_icon_fixed,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Allowed actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		edv_location_bar_icon_drag_data_get_cb,
		NULL,
		NULL,
		combo
	);
	w = combo->entry;
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* GtkHandleBox for the Mount Bar */
	imbr->mount_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_attached_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_detached_cb), imbr
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(imbr->flags & EDV_IMAGE_BROWSER_MOUNT_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Mount Bar */
	imbr->mount_bar = w = edv_mount_bar_new(
		core,
		edv_image_browser_mount_bar_mount_cb,
		imbr,
		edv_image_browser_mount_bar_eject_cb,
		imbr,
		edv_image_browser_mount_bar_goto_cb,
		imbr,
		edv_image_browser_status_message_cb,
		imbr
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* GtkHandleBox for the Find Bar */
	imbr->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_attached_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_image_browser_handle_child_detached_cb), imbr
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(imbr->flags & EDV_IMAGE_BROWSER_FIND_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Find Bar */
	imbr->find_bar = w = edv_find_bar_new(
		core,
		EDV_LOCATION_TYPE_VFS,
		edv_image_browser_find_bar_location_cb,
		imbr,
		edv_image_browser_find_bar_start_cb,
		imbr,
		edv_image_browser_find_bar_end_cb,
		imbr,
		edv_image_browser_find_bar_match_cb,
		imbr,
		edv_image_browser_status_message_cb,
		imbr,
		edv_image_browser_status_progress_cb,
		imbr
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* Main GtkHPaned */
	imbr->main_paned = w = gtk_hpaned_new();
	CfgGtkSetGtkPaned(
		cfg_list,
		EDV_CFG_PARM_IMBR_PANED_POSITION,
		w
	);
	gtk_paned_set_handle_size(
		GTK_PANED(w), EDV_PANED_HANDLE_SIZE
	);
	gtk_paned_set_gutter_size(
		GTK_PANED(w), EDV_PANED_GUTTER_SIZE
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Thumbs List */
	orientation = (EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_LIST_HORIZONTAL)) ?
		GTK_ORIENTATION_HORIZONTAL : GTK_ORIENTATION_VERTICAL;
	imbr->tlist = tlist = TListNew(
		orientation,
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_WIDTH),
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_HEIGHT),
		EDV_GET_I(EDV_CFG_PARM_IMBR_THUMB_BORDER),
		edv_image_browser_list_select_cb,
		imbr,
		edv_image_browser_list_unselect_cb,
		imbr
	);
	w = TListGetToplevelWidget(tlist);
	gtk_paned_add1(GTK_PANED(parent2), w);
	TListDoubleBuffer(
		tlist,
		EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER)
	);
	TListSelectionMode(
		tlist,
		GTK_SELECTION_EXTENDED
	);
	TListShowThumbFrames(
		tlist,
		EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_FRAMES)
	);
	TListShowThumbLabels(
		tlist,
		EDV_GET_B(EDV_CFG_PARM_IMBR_THUMB_SHOW_LABELS)
	);
	TListShowTextTips(
		tlist,
		EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS)
	);
	TListEnableListDragScroll(
		tlist,
		((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2) == EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY) ? TRUE : FALSE
	);

	/* Thumbs List List GtkWidget */
	w = TListGetListWidget(tlist);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_IMAGE_BROWSER_THUMBS_LIST);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_image_browser_list_realize_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_image_browser_key_event_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_image_browser_key_event_cb), imbr
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_image_browser_button_event_cb), imbr
	);
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,		/* Allowed actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		edv_image_browser_list_drag_data_get_cb,
		edv_image_browser_list_drag_data_delete_cb,
		NULL,
		imbr
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		edv_image_browser_list_drag_data_received_cb,
		imbr,
		FALSE				/* Do not highlight */
	);
	TListMap(tlist);


	/* Image Viewer */
	imbr->imgview = iv = ImgViewNew(
		TRUE,				/* Show Tool Bar */
		TRUE,				/* Show Values */
		FALSE,				/* Hide Status Bar */
		FALSE,				/* No WM Icon */
		EDV_GET_I(EDV_CFG_PARM_IMAGE_QUALITY),	/* Quality (0 to 2) */
		FALSE,				/* Toplevel is not a GtkWindow */
		&w				/* Toplevel return */
	);
	gtk_paned_add2(GTK_PANED(parent2), w);
	if(iv != NULL)
	{
		/* Background color */
		CfgColor *color = EDV_GET_COLOR(EDV_CFG_PARM_IMBR_BG_COLOR);
		if(color != NULL)
		{
			GdkColor gdk_color[5];
			const gint m = sizeof(gdk_color) / sizeof(GdkColor);
			gint i;
			CfgColor *sc = color;
			for(i = 0; i < m; i++)
			{
				GDK_COLOR_SET_COEFF(
					&gdk_color[i],
					sc->r, sc->g, sc->b
				);
			}
			ImgViewSetViewBG(iv, gdk_color);
		}
		ImgViewSetValuesShowable(iv, TRUE);
		ImgViewSetToolBarShowable(iv, FALSE);
		ImgViewSetToolBarMenuShowable(iv, FALSE);
		ImgViewSetStatusBarShowable(iv, FALSE);
		ImgViewMap(iv);
	}

	/* Set up the image view's GtkWidget */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w != NULL)
	{
		gtk_widget_set_name(w, EDV_WIDGET_NAME_IMAGE_BROWSER_IMAGE_VIEWER);
		gtk_signal_connect_after(
			GTK_OBJECT(w), "button_press_event",
			GTK_SIGNAL_FUNC(edv_image_browser_button_event_cb), imbr
		);
		GUIDNDSetTar(
			w,
			dnd_tar_types,
			sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY | GDK_ACTION_MOVE |
				GDK_ACTION_LINK,/* Actions */
			GDK_ACTION_MOVE,	/* Default action if same */
			GDK_ACTION_MOVE,	/* Default action */
			edv_image_browser_view_drag_data_received_cb,
			imbr,
			FALSE			/* Highlight */
		);
	}


	/* Tool Bar */
	edv_image_browser_tool_bar_regenerate(imbr);

	/* Thumbs List Right Click GtkMenu */
	edv_image_browser_list_menu_regenerate(imbr);


	/* Status Bar */
	imbr->status_bar = w = edv_status_bar_new(core);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(imbr->flags & EDV_IMAGE_BROWSER_STATUS_BAR_MAPPED)
		gtk_widget_show(w);


	/* Create the New Objects submenu */
	edv_image_browser_new_object_menu_items_regenerate(imbr);

	/* Create the Open To submenu */
	edv_image_browser_open_to_menu_regenerate(imbr, NULL);


	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			imbr->toplevel,
			standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			TListGetListWidget(imbr->tlist),
			lists_rcstyle
		);
	w = ImgViewGetToplevelWidget(imbr->imgview);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			w,
			lists_rcstyle
		);
	w = (GtkWidget *)ImgViewGetMenuWidget(imbr->imgview);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			w,
			standard_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			imbr->tlist_menu,
			standard_rcstyle
		);

	/* Load the icons */
	edv_image_browser_reload_icons(imbr);

	/* Set the initial Accel Keys */
	edv_image_browser_accelkeys_regenerate(imbr);

	imbr->freeze_count--;

	return(imbr);
}

/*
 *	Syncronizes the Image Browser's widget values with other values.
 */
void edv_image_browser_sync_data(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the image
 * browser
 */

}


/*
 *	Updates the Image Browser's widgets to reflect current 
 *	values.
 */
void edv_image_browser_update_display(EDVImageBrowser *imbr)
{
	gboolean	write_protect,
					write_protect_changed = FALSE,
					loading_in_progress,
					sensitive,
					state;
	gint		i,
					selected_dev_num,
					tlist_selected_thumb;
	guint8		**icon_data,
					**recbin_icon_data;
	GtkWidget	*w,
					*tb;
	CfgList *cfg_list;
	imgview_struct *iv;
	EDVDeleteMethod delete_method;
	EDVDevice *selected_dev_ptr;
	EDVCore *core;

	if(imbr == NULL)
		return;

	imbr->freeze_count++;

	tb = imbr->tool_bar;
	iv = imbr->imgview;
	core = imbr->core;
	cfg_list = core->cfg_list;

#define MENU_ITEM_SET_TOGGLE(_w_,_state_)		\
 GUIMenuItemSetCheck(					\
  (_w_),						\
  (_state_),						\
  FALSE							\
 )
#define TOOL_BAR_ITEM_SET_SENSITIVE(_op_,_sensitive_)	\
 ToolBarSetItemSensitive(				\
  tb,							\
  (_op_),						\
  (_sensitive_)						\
 )
#define TOOL_BAR_ITEM_SET_TOGGLE(_op_,_state_)		\
 ToolBarSetItemToggle(					\
  tb,							\
  (_op_),						\
  (_state_)						\
 )
#define TOOL_BAR_ITEM_UPDATE(_op_,_text_,_icon_data_,_tip_) \
 ToolBarUpdateItem(					\
  tb,							\
  (_op_),						\
  (_text_),						\
  (_icon_data_),					\
  (_tip_)						\
 )

	/* Get the load progress state */
	loading_in_progress = (imbr->loading_tocb > 0) ? TRUE : FALSE;

	/* Get global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(imbr->last_write_protect_state < 0)
	{
		write_protect_changed = TRUE;
	}
	else
	{
		if((write_protect && !imbr->last_write_protect_state) ||
		   (!write_protect && imbr->last_write_protect_state)
		)
			write_protect_changed = TRUE;
	}
	imbr->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get last selected thumb */
	tlist_selected_thumb = imbr->tlist_selected_thumb;

	/* Get last selected device (if any) */
	selected_dev_num = imbr->selected_dev_num;
	selected_dev_ptr = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)selected_dev_num
	));

	/* Get the Recycle Bin icon data */
	icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		core->last_nrecycle_bin_items,
		NULL
	);
	imbr->last_nrecycle_bin_items = core->last_nrecycle_bin_items;
	recbin_icon_data = icon_data;


	/* Update the Mount Bar */
	edv_mount_bar_update_display(imbr->mount_bar);

	/* Update the Find Bar */
	edv_find_bar_update_display(imbr->find_bar);


	/* Update the GtkMenus, GtkMenuItems, and ToolBar */

	/* File Menu */
	/* Open & Open With */
	sensitive = (tlist_selected_thumb > -1) ? TRUE : FALSE;
	/* Open */
	GTK_WIDGET_SET_SENSITIVE(imbr->open_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_OPEN, sensitive);
	/* Open To */
	GTK_WIDGET_SET_SENSITIVE(imbr->open_to_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_OPEN_TO, sensitive);
	/* Open With */
	GTK_WIDGET_SET_SENSITIVE(imbr->open_with_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_OPEN_WITH, sensitive);

	/* New Object Submenu Item */
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(imbr->new_object_mi, sensitive);
	/* New file */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_file_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_FILE, sensitive);
	/* New Directory */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_directory_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_DIRECTORY, sensitive);
	/* New Link */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_link_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_LINK, sensitive);
	/* New FIFO */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_fifo_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_FIFO, sensitive);
	/* New Block Device */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_device_block_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_DEVICE_BLOCK, sensitive);
	/* New Character Device */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_device_character_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_DEVICE_CHARACTER, sensitive);
	/* New Socket */
	GTK_WIDGET_SET_SENSITIVE(imbr->new_socket_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_SOCKET, sensitive);

	/* Go To Parent */
	sensitive = TRUE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_GOTO_PARENT, sensitive);


	/* Edit Menu */
	sensitive = (tlist_selected_thumb > -1) ? TRUE : FALSE;
	/* Move */
	GTK_WIDGET_SET_SENSITIVE(imbr->move_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_MOVE, sensitive);
	/* Copy */
	GTK_WIDGET_SET_SENSITIVE(imbr->copy_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_COPY, sensitive);
	/* Link */
	GTK_WIDGET_SET_SENSITIVE(imbr->link_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_LINK, sensitive);
	/* Rename */
	GTK_WIDGET_SET_SENSITIVE(imbr->rename_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_RENAME, sensitive);
	/* Change Permissions */
	GTK_WIDGET_SET_SENSITIVE(imbr->chmod_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_CHMOD, sensitive);
	/* Change Ownership */
	GTK_WIDGET_SET_SENSITIVE(imbr->chown_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_CHOWN, sensitive);
	/* Set Time Stamps */
	GTK_WIDGET_SET_SENSITIVE(imbr->chtime_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_CHTIME, sensitive);
	/* Delete */
	GTK_WIDGET_SET_SENSITIVE(imbr->delete_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_DELETE, sensitive);
	/* Copy path */
	GTK_WIDGET_SET_SENSITIVE(imbr->copy_path_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_COPY_PATH, sensitive);
	/* Copy URL */
	GTK_WIDGET_SET_SENSITIVE(imbr->copy_url_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_COPY_URL, sensitive);
	/* Properties */
	GTK_WIDGET_SET_SENSITIVE(imbr->properties_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_PROPERTIES, sensitive);

	/* View Menu */
	/* Stop loading */
	sensitive = loading_in_progress;
	GTK_WIDGET_SET_SENSITIVE(imbr->stop_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_STOP, sensitive);
	/* Continue loading */
	sensitive = !loading_in_progress;
	GTK_WIDGET_SET_SENSITIVE(imbr->continue_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_CONTINUE, sensitive);
	/* Presentation Mode */
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(imbr->presentation_mode_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_PRESENTATION_MODE, sensitive);
	/* Previous Image */
	sensitive = TRUE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_PREVIOUS_IMAGE, sensitive);
	/* Next Image */
	sensitive = TRUE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEXT_IMAGE, sensitive);
	/* Animation Menu */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_menu, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION, sensitive);
	/* Play */
	sensitive = (ImgViewIsLoaded(iv) && !ImgViewIsPlaying(iv)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_play_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY, sensitive);
	/* Pause */
	sensitive = (ImgViewIsLoaded(iv) && ImgViewIsPlaying(iv)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_pause_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION_PAUSE, sensitive);
	/* Play/Pause */
	sensitive = (ImgViewIsLoaded(iv)) ? TRUE : FALSE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY_PAUSE, sensitive);
	if(sensitive)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(ImgViewIsPlaying(iv))
		{
			text = "Pause";
			icon_data = (edv_pixmap_data *)icon_pause_20x20_xpm;
		}
		else
		{
			text = "Play";
			icon_data = (edv_pixmap_data *)icon_play_20x20_xpm;
		}
		TOOL_BAR_ITEM_UPDATE(
			EDV_IMAGE_BROWSER_OP_ANIMATION_PLAY_PAUSE,
			text,
			icon_data,
			NULL
		);
	}
	/* Previous Frame */
	sensitive = (ImgViewIsLoaded(iv) && !ImgViewIsPlaying(iv) &&
		(ImgViewGetTotalFrames(iv) > 1)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_previous_frame_mi, sensitive)
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION_PREVIOUS_FRAME, sensitive);
	/* Next Frame */
	GTK_WIDGET_SET_SENSITIVE(imbr->animation_next_frame_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ANIMATION_NEXT_FRAME, sensitive);
	/* Rotate/Transform Menu */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_menu, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ROTATE_TRANSFORM, sensitive);
	/* Rotate Clockwise 90 */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_cw90_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ROTATE_CW90, sensitive);
	/* Rotate Counter-Clockwise 90 */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_ccw90_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ROTATE_CCW90, sensitive);
	/* Rotate Clockwise 180 */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_cw180_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_ROTATE_CW180, sensitive);
	/* Mirror Horizontal */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_mirror_horizontal_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_MIRROR_HORIZONTAL, sensitive);
	/* Mirror Vertical */
	sensitive = ImgViewIsLoaded(iv);
	GTK_WIDGET_SET_SENSITIVE(imbr->rotate_transform_mirror_vertical_mi, sensitive)
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_MIRROR_VERTICAL, sensitive);

	/* Show Tool Bar */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_TOOL_BAR);
	MENU_ITEM_SET_TOGGLE(imbr->show_tool_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_TOOL_BAR, state);
	/* Show Location Bar */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_LOCATION_BAR);
	MENU_ITEM_SET_TOGGLE(imbr->show_location_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_LOCATION_BAR, state);
	/* Show Mount Bar */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_MOUNT_BAR);
	MENU_ITEM_SET_TOGGLE(imbr->show_mount_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_MOUNT_BAR, state);
	/* Show Find Bar */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_FIND_BAR);
	MENU_ITEM_SET_TOGGLE(imbr->show_find_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_FIND_BAR, state);
	/* Show Status Bar */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_STATUS_BAR);
	MENU_ITEM_SET_TOGGLE(imbr->show_status_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_STATUS_BAR, state);
	/* Show Hidden Objects */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_HIDDEN);
	MENU_ITEM_SET_TOGGLE(imbr->show_hidden_objects_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_HIDDEN_OBJECTS, state);
	/* Show No Access Objects */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_NOACCESS);
	MENU_ITEM_SET_TOGGLE(imbr->show_noaccess_objects_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_NOACCESS_OBJECTS, state);
	/* Show Non-Image Objects */
	state = EDV_GET_B(EDV_CFG_PARM_IMBR_SHOW_OBJECT_NONIMAGE);
	MENU_ITEM_SET_TOGGLE(imbr->show_nonimage_objects_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_SHOW_NONIMAGE_OBJECTS, state);

	/* Device Menu */
	w = imbr->mount_mi;
	sensitive = (selected_dev_ptr != NULL) ?
		!EDV_DEVICE_IS_NO_UNMOUNT(selected_dev_ptr) : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_MOUNT, sensitive);
	if(w != NULL)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(selected_dev_ptr != NULL)
		{
			if(EDV_DEVICE_IS_MOUNTED(selected_dev_ptr))
			{
				text = "Unmount";
				icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
			}
			else  
			{
				text = "Mount";
				icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
			}
		}
		else
		{
			text = "Mount/Unmount";
			icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
		}
		GUIMenuItemSetPixmap(w, icon_data);
		GUIMenuItemSetLabel(w, text);
		TOOL_BAR_ITEM_UPDATE(
			EDV_IMAGE_BROWSER_OP_MOUNT,
			text,
			icon_data,
			NULL
		);
	}

	/* Eject */
	if(selected_dev_ptr != NULL)
		sensitive = !STRISEMPTY(selected_dev_ptr->command_eject);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->eject_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_EJECT, sensitive);


	/* Device check, tools, and format commands */
	if(selected_dev_ptr != NULL)
		sensitive = !STRISEMPTY(selected_dev_ptr->command_check);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->device_check_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_DEVICE_CHECK, sensitive);

	if(selected_dev_ptr != NULL)
		sensitive = !STRISEMPTY(selected_dev_ptr->command_tools);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->device_tools_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_DEVICE_TOOLS, sensitive);

	if(selected_dev_ptr != NULL)
		sensitive = !STRISEMPTY(selected_dev_ptr->command_format);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->device_format_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_DEVICE_FORMAT, sensitive);


	/* Settings Menu */
	if(write_protect_changed)
	{
		w = imbr->write_protect_mi;
		GUIMenuItemSetPixmap(
			w,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm)
		);
		GUIMenuItemSetLabel(
			w,
			write_protect ?
				"Master Write Protect: On" :
				"Master Write Protect: Off"
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_IMAGE_BROWSER_OP_WRITE_PROTECT,
			NULL,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm),
			NULL
		);
	}

	/* Delete Method: Recycle */
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(imbr->delete_method_recycle_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_DELETE_METHOD_RECYCLE, state);

	/* Delete Method: Purge */
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ?
		TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(imbr->delete_method_purge_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_IMAGE_BROWSER_OP_DELETE_METHOD_PURGE, state);


	/* Windows
	 *
	 * New Browser
	 */
	w = imbr->window_new_browser_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_BROWSERS);
	sensitive = ((core->total_browsers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_BROWSER, sensitive);

	/* New Image Browser */
	w = imbr->window_new_imbr_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_IMBRS);
	sensitive = ((core->total_imbrs < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_IMBR, sensitive);

	/* New Archiver */
	w = imbr->window_new_archiver_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_ARCHIVERS);
	sensitive = ((core->total_archivers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_IMAGE_BROWSER_OP_NEW_ARCHIVER, sensitive);

	/* Recycle Bin */
	if(recbin_icon_data != NULL)
	{
		GUIMenuItemSetPixmap(
			imbr->window_recycle_bin_mi, recbin_icon_data
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_IMAGE_BROWSER_OP_RECYCLE_BIN,
			NULL, recbin_icon_data, NULL
		);
	}


	/* TList Right-Click Menu */
	sensitive = (tlist_selected_thumb > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_open_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_open_to_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_open_with_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_move_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_copy_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_link_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_rename_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_chmod_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_chown_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_delete_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_copy_path_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_copy_url_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(imbr->tlist_properties_mi, sensitive);

	w = imbr->tlist_mount_mi;
	if(selected_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}
	sensitive = (selected_dev_ptr != NULL) ?
		!EDV_DEVICE_IS_NO_UNMOUNT(selected_dev_ptr) : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	if(selected_dev_ptr != NULL)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(EDV_DEVICE_IS_MOUNTED(selected_dev_ptr))
		{
			text = "Unmount";
			icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
		}
		else
		{
			text = "Mount";
			icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
		}
		GUIMenuItemSetPixmap(w, icon_data);
		GUIMenuItemSetLabel(w, text);
	}

	w = imbr->tlist_mount_misep;
	if(selected_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}

	/* Eject */
	w = imbr->tlist_eject_mi;
	if(selected_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}
	if(selected_dev_ptr != NULL)
		sensitive = !STRISEMPTY(selected_dev_ptr->command_eject);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);


	/* Status Bar */
	edv_status_bar_update_display(imbr->status_bar);

#undef MENU_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_SET_SENSITIVE
#undef TOOL_BAR_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_UPDATE

	imbr->freeze_count--;
}

/*
 *	Sets the EDVImageBrowser busy or ready.
 */
void edv_image_browser_set_busy(
	EDVImageBrowser *imbr,
	const gboolean busy
)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(imbr == NULL)
		return;

	w = imbr->toplevel;
	core = imbr->core;

	if(busy)
	{
		imbr->busy_count++;
		if(imbr->busy_count > 1)
			return;

		cur = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
		imbr->busy_count--;
		if(imbr->busy_count < 0)
			imbr->busy_count = 0;
		if(imbr->busy_count > 0)
			return;
		cur = (imbr->passive_busy_count > 0) ?
			edv_get_cursor(core, EDV_CURSOR_CODE_PASSIVE_BUSY) :
			NULL;
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Sets the EDVImageBrowser passive busy or ready.
 */
void edv_image_browser_set_passive_busy(
	EDVImageBrowser *imbr,
	const gboolean busy
)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(imbr == NULL)
		return; 

	w = imbr->toplevel;
	core = imbr->core;

	if(busy)
	{
		imbr->passive_busy_count++;
		if((imbr->passive_busy_count > 1) || (imbr->busy_count > 0))
			return;
		cur = edv_get_cursor(core, EDV_CURSOR_CODE_PASSIVE_BUSY);
	}
	else
	{
		imbr->passive_busy_count--;
		if(imbr->passive_busy_count < 0)
			imbr->passive_busy_count = 0;
		if((imbr->passive_busy_count > 0) || (imbr->busy_count > 0))
			return;
		cur = NULL;
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Checks if the EDVImageBrowser is mapped.
 */
gboolean edv_image_browser_is_mapped(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(imbr->toplevel));
}

/*
 *	Maps the EDVImageBrowser.
 */
void edv_image_browser_map(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	gtk_widget_show_raise(imbr->toplevel);
	imbr->flags |= EDV_IMAGE_BROWSER_MAPPED;
}

/*
 *	Unmaps the EDVImageBrowser.
 */
void edv_image_browser_unmap(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	gtk_widget_hide(imbr->toplevel);
	imbr->flags &= ~EDV_IMAGE_BROWSER_MAPPED;
}

/*
 *	Deletes the EDVImageBrowser.
 */
void edv_image_browser_delete(EDVImageBrowser *imbr)
{
	if(imbr == NULL)
		return;

	/* Clear the Thumbs List */
	edv_image_browser_list_clear(imbr);

	/* Remove the timeout and idle callbacks */
	imbr->loading_tocb = GTK_TIMEOUT_REMOVE(imbr->loading_tocb);

	/* Reset the last selected item references */
	imbr->tlist_selected_thumb = -1;


	edv_image_browser_unmap(imbr);

	imbr->freeze_count++;

	TListDelete(imbr->tlist);
	ImgViewDelete(imbr->imgview);

	GTK_WIDGET_DESTROY(imbr->tlist_open_to_menu);
	GTK_WIDGET_DESTROY(imbr->tlist_menu);

	GTK_WIDGET_DESTROY(imbr->location_icon_pm);
	GTK_WIDGET_DESTROY(imbr->location_icon_fixed);
	GTK_WIDGET_DESTROY(imbr->location_combo);
	GTK_WIDGET_DESTROY(imbr->location_bar_hbox);

	GTK_WIDGET_DESTROY(imbr->open_to_menu);
	GTK_WIDGET_DESTROY(imbr->new_object_menu);
	GTK_WIDGET_DESTROY(imbr->animation_menu);
	GTK_WIDGET_DESTROY(imbr->rotate_transform_menu);
	GTK_WIDGET_DESTROY(imbr->menu_bar);

	GTK_WIDGET_DESTROY(imbr->toplevel);

	/* Icon EDVPixmaps */
	(void)edv_pixmap_unref(imbr->directory_parent_icon);
	(void)edv_pixmap_unref(imbr->file_image_pending_icon);
	(void)edv_pixmap_unref(imbr->file_image_bad_icon);
	(void)edv_pixmap_unref(imbr->multiple_frames_icon);

	/* EDVImageBrowserOps List */
	edv_image_browser_ops_list_delete(imbr->op_ids_list);

	g_free(imbr->thumbs_list_filter);

	imbr->freeze_count--;

	g_free(imbr);
}
