#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "tlist.h"
#include "fb.h"
#include "icon_sel_dlg.h"
#include "config.h"

#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"

#include "images/icon_folder_parent_32x32.xpm"
#include "images/icon_folder_closed_32x32.xpm"


typedef struct _IconSelDlg		IconSelDlg;
#define ICON_SEL_DLG(p)			((IconSelDlg *)(p))
#define ICON_SEL_DLG_KEY		"/IconSelDlg"


/* Callbacks */
static void IconSelDlgDataDestroyCB(gpointer data);
static gint IconSelDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void IconSelDlgLocationEntryActivateCB(GtkWidget *widget, gpointer data);
static void IconSelDlgLocationBrowseCB(GtkWidget *widget, gpointer data);
static void IconSelDlgTListSelectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);
static void IconSelDlgTListUnselectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);
static gint IconSelTListLoadTOCB(gpointer data);
static void IconSelDlgOKCB(GtkWidget *widget, gpointer data);
static void IconSelDlgCancelCB(GtkWidget *widget, gpointer data);

/* Get/Set Location */
const gchar *IconSelDlgCurrentLocation(GtkWidget *w);
void IconSelDlgChangeLocation(
	GtkWidget *w,
	const gchar *path
);

/* Thumb Image Loading */
gboolean IconSelDlgIsLoading(GtkWidget *w);
void IconSelDlgQueueLoading(GtkWidget *w);
void IconSelDlgStopLoading(GtkWidget *w);

/* Query */
gboolean IconSelDlgIsQuery(GtkWidget *w);
GList *IconSelDlgQuery(
	GtkWidget *w,
	const gchar *title,
	const gchar *path,
	const gint req_icon_width, const gint req_icon_height,
	GtkWidget *ref_toplevel
);

/* Icon Selector Dialog */
GtkWidget *IconSelDlgNew(
	const icon_sel_dlg_flags flags,
	const GtkOrientation list_orientation,
	const gint list_height,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	const gchar *default_path,
	const gulong load_int
);
void IconSelDlgUpdateWidgets(GtkWidget *w);
void IconSelDlgMap(GtkWidget *w);
void IconSelDlgUnmap(GtkWidget *w);
GtkWidget *IconSelDlgRef(GtkWidget *w);
GtkWidget *IconSelDlgUnref(GtkWidget *w);


#define ICON_SEL_DLG_WIDTH		500
#define ICON_SEL_DLG_HEIGHT		-1


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Icon Selector Dialog:
 */
struct _IconSelDlg {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count,
			ref_count,
			main_level;

	icon_sel_dlg_flags	flags;

	gint		req_icon_width,	/* Requested icon size */
			req_icon_height;

	gulong		load_int;	/* Load interval in milliseconds */
	guint		load_toid;

	gchar		*default_path;	/* Default location to use when no
					 * path is specified and the tlist
					 * is empty */

	GtkWidget	*main_vbox,
			*location_entry,
			*location_browse_btn,
			*ok_btn,
			*cancel_btn;

	tlist_struct	*tlist;

	/* Last drag position */
	gint		drag_last_x,
			drag_last_y;

	/* Selected paths list, a GList of gchar * paths */
	GList		*selected_paths_list;

	GdkPixmap	*dir_pixmap;
	GdkBitmap	*dir_mask;
	GdkPixmap	*parent_dir_pixmap;
	GdkBitmap	*parent_dir_mask;
};


/*
 *	Toplevel GtkWindow data "destroy" signal callback.
 */
static void IconSelDlgDataDestroyCB(gpointer data)
{
	IconSelDlg *d = ICON_SEL_DLG(data);
	if(d == NULL)
	    return;

	if(d->ref_count > 0)
	{
	    g_printerr(
"IconSelDlgDataDestroyCB(): \
Warning: Destroying IconSelDlg %p with %i reference counts remaining.\n\
Was IconSelDlgUnref() not used properly to unref this IconSelDlg?\n",
		d->toplevel,
		d->ref_count
	    );
	}

	d->freeze_count++;

	/* Stop and active loading */
	if(d->load_toid != 0)
	{
	    g_printerr(
"IconSelDlgDataDestroyCB(): \
Warning: Destroying IconSelDlg %p while it is still loading.\n\
Was IconSelDlgStopLoading() not called?\n",
		d->toplevel
	    );
	    d->load_toid = GTK_TIMEOUT_REMOVE(d->load_toid);
	}

	/* Break out of any remaining GTK main levels */
	if(d->main_level > 0)
	{
	    g_printerr(
"IconSelDlgDataDestroyCB(): \
Warning: Destroying IconSelDlg %p with %i main levels remaining.\n\
Is this IconSelDlg being destroyed during a query?\n",
		d->toplevel,
		d->main_level
	    );
	    while(d->main_level > 0)
	    {
		d->main_level--;
		gtk_main_quit();
	    }
	    g_printerr(
"Leaving IconSelDlgDataDestroyCB() at a lower main level, any further\n\
references to this IconSelDlg will cause a segmentation fault.\n"
	    );
	}

	gtk_accel_group_unref(d->accelgrp);

	if(d->selected_paths_list != NULL)
	{
	    g_list_foreach(
		d->selected_paths_list,
		(GFunc)g_free,
		NULL
	    );
	    g_list_free(d->selected_paths_list);
	}

	GDK_PIXMAP_UNREF(d->parent_dir_pixmap);
	GDK_BITMAP_UNREF(d->parent_dir_mask);
	GDK_PIXMAP_UNREF(d->dir_pixmap);
	GDK_BITMAP_UNREF(d->dir_mask);

	g_free(d->default_path);

	d->freeze_count--;

	g_free(d);
}

/*
 *	Dialog toplevel GtkWindow "delete_event" signal callback.
 */
static gint IconSelDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	IconSelDlgCancelCB(widget, data);
	return(TRUE);
}

/*
 *	Location GtkEntry "activate" signal callback.
 */
static void IconSelDlgLocationEntryActivateCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	IconSelDlg *d = ICON_SEL_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	/* Set the new location and start loading the thumb images */
	IconSelDlgChangeLocation(
	    toplevel,
	    gtk_entry_get_text(GTK_ENTRY(d->location_entry))
	);
	IconSelDlgQueueLoading(toplevel);
}

/*
 *	Browse callback.
 */
static void IconSelDlgLocationBrowseCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint		npaths = 0,
			nftypes = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	IconSelDlg *d = ICON_SEL_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the origin directory */
	FileBrowserSetTransientFor(toplevel);
	d->freeze_count++;
	response = FileBrowserGetResponse(
	    "Set Location",
	    "Set", "Cancel",
	    gtk_entry_get_text(GTK_ENTRY(d->location_entry)),
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	d->freeze_count--;
	FileBrowserSetTransientFor(NULL);

	if(response && (npaths > 0))
	{
	    gchar *path = STRDUP(paths_list[npaths - 1]);
	    if(path != NULL)
	    {
		/* Change to this directory and start loading the
		 * thumb images
		 */
		IconSelDlgChangeLocation(
		    toplevel,
		    path
		);
		IconSelDlgQueueLoading(toplevel);
		g_free(path);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Thumbs List select callback.
 */
static void IconSelDlgTListSelectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	GtkWidget *toplevel;
	IconSelDlg *d = ICON_SEL_DLG(data);
	if((tlist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	/* If the selected thumb is not fully visible then scroll to it */
	if(TListIsThumbVisible(tlist, thumb_num) !=
	    GTK_VISIBILITY_FULL
	)
	    TListMoveTo(
		tlist,
		thumb_num,
		0.5f
	    );

	/* Double click? */
	if(button != NULL)
	{
	    if(button->type == GDK_2BUTTON_PRESS)
	    {
		const char *path = (const gchar *)TListGetThumbData(
		    tlist,
		    thumb_num
		);
		if(path != NULL)
		{
		    struct stat stat_buf;
		    gchar *dpath = g_strdup(path);

		    if(stat((const char *)dpath, &stat_buf))
			memset(&stat_buf, 0x00, sizeof(struct stat));

		    if(S_ISDIR(stat_buf.st_mode))
		    {
			/* Change to this directory and start loading
			 * the thumb images
			 */
			IconSelDlgChangeLocation(
			    toplevel,
			    dpath
			);
			IconSelDlgQueueLoading(toplevel);
		    }
		    else
		    {
			/* OK */
			IconSelDlgOKCB(tlist->toplevel, data);
		    }

		    g_free(dpath);
		}
	    }
	}
}

/*
 *	Thumbs List unselect callback.
 */
static void IconSelDlgTListUnselectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	IconSelDlg *d = ICON_SEL_DLG(data);
	if((tlist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

}

/*
 *	Thumbs List load timeout callback.
 */
static gint IconSelTListLoadTOCB(gpointer data)
{
	gint i, starting_thumb_num, thumb_to_load_num = -1;
	const gchar *path;
	tlist_thumb_struct *thumb, *thumb_to_load = NULL;
	tlist_struct *tlist;
	IconSelDlg *d = ICON_SEL_DLG(data);
	if(d == NULL)
	    return(FALSE);

	if(gtk_events_pending() > 0)
	{
	    /* Set the new loading timeout callback with a longer delay */
	    d->load_toid = gtk_timeout_add(
		MAX(d->load_int, ICON_SEL_DLG_LOAD_INT_LONG),
		IconSelTListLoadTOCB, d
	    );
	    return(FALSE);
	}

	if(d->freeze_count > 0)
	{
	    /* Set the new loading timeout callback with a longer delay */
	    d->load_toid = gtk_timeout_add(
		(d->load_int != 0) ? d->load_int : ICON_SEL_DLG_DEF_LOAD_INT,
		IconSelTListLoadTOCB, d
	    );
	    return(FALSE);
	}

	d->freeze_count++;

	tlist = d->tlist;

#if 0
	/* Loading interrupted? */
	if(d->stop_count > 0)
	{
	    d->stop_count = 0;
	    d->load_toid = 0;
	    d->freeze_count--;
	    return(FALSE);
	}
#endif

	/* Calculate the first starting thumb at the current scroll
	 * position
	 */
	TListGetSelection(
	    tlist,
	    0, 0,
	    &starting_thumb_num,
	    NULL, NULL
	);

	/* Look for the thumb to load starting from the scroll
	 * position to the end of the list
	 */
	if(starting_thumb_num > -1)
	{
	    for(i = starting_thumb_num;
		i < tlist->total_thumbs;
		i++
	    )
	    {
		thumb = tlist->thumb[i];
		if(thumb == NULL)
		    continue;

		if(thumb->load_state == TLIST_LOAD_STATE_NOT_LOADED)
		{
		    thumb_to_load = thumb;
		    thumb_to_load_num = i;
		    break;
		}
	    }
	}
	/* If no thumbs needed to be loaded starting from
	 * starting_thumb_num then start loading from thumb 0
	 */
	if((thumb_to_load == NULL) &&
	   (starting_thumb_num <= tlist->total_thumbs)
	)
	{
	    for(i = 0; i < starting_thumb_num; i++)
	    {
		thumb = tlist->thumb[i];
		if(thumb == NULL)
		    continue;

		if(thumb->load_state == TLIST_LOAD_STATE_NOT_LOADED)
		{
		    thumb_to_load = thumb;
		    thumb_to_load_num = i;
		    break;
		}
	    }
	}

	/* No more thumbs need to be loaded? */
	if((thumb_to_load == NULL) || (thumb_to_load_num < 0))
	{
/*	    d->stop_count = 0; */
	    d->load_toid = 0;
	    d->freeze_count--;
	    return(FALSE);
	}

	thumb = thumb_to_load;
	if(thumb == NULL)
	{
/*	    d->stop_count = 0; */
	    d->load_toid = 0;
	    d->freeze_count--;
	    return(FALSE);
	}

	TListFreeze(tlist);

	/* Load this thumb */
	path = (const gchar *)thumb->data;
	if(path != NULL)
	{
	    GtkWidget *w = TListGetListWidget(tlist);
	    GdkWindow *window = (w != NULL) ? w->window : NULL;
	    if(window != NULL)
	    {
		GdkBitmap *mask;
		GdkPixmap *pixmap = gdk_pixmap_create_from_xpm(
		    window,
		    &mask,
		    NULL,
		    path
		);
		if(pixmap != NULL)
		{
		    TListSetPixmap(
			tlist,
			thumb_to_load_num,
			pixmap, mask
		    );
		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);
		    TListSetLoadState(
			tlist,
			thumb_to_load_num,
			TLIST_LOAD_STATE_LOADED
		    );
		}
		else
		{
		    TListSetLoadState(
			tlist,
			thumb_to_load_num,
			TLIST_LOAD_STATE_FAILED
		    );
		}
	    }
	    else
	    {
		TListSetLoadState(
		    tlist,
		    thumb_to_load_num,
		    TLIST_LOAD_STATE_FAILED
		);
	    }
	}

	TListThaw(tlist);

	d->freeze_count--;

	d->load_toid = gtk_timeout_add(
	    (d->load_int != 0) ? d->load_int : ICON_SEL_DLG_DEF_LOAD_INT,
	    IconSelTListLoadTOCB, d
	);

	return(FALSE);
}

/*
 *	Dialog OK callback.
 */
static void IconSelDlgOKCB(GtkWidget *widget, gpointer data)
{
	const gchar *path;
	GList *glist;
	tlist_struct *tlist;
	IconSelDlg *d = ICON_SEL_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	tlist = d->tlist;

	/* Set the response as OK */
	d->flags |= ICON_SEL_DLG_RESPONSE_OK;

	/* Delete the current selected paths list */
	if(d->selected_paths_list != NULL)
	{
	    g_list_foreach(
		d->selected_paths_list,
		(GFunc)g_free,
		NULL
	    );
	    g_list_free(d->selected_paths_list);
	    d->selected_paths_list = NULL;
	}

	/* Set the new selected paths list from the selected thumbs */
	for(glist = tlist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    path = (const gchar *)TListGetThumbData(
		tlist,
		(gint)glist->data
	    );
	    if(STRISEMPTY(path))
		continue;

	    d->selected_paths_list = g_list_append(
		d->selected_paths_list,
		STRDUP(path)
	    );
	}

	/* Break out of one main level */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}

/*
 *	Dialog Cancel callback.
 */
static void IconSelDlgCancelCB(GtkWidget *widget, gpointer data)
{
	IconSelDlg *d = ICON_SEL_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	/* Set the response as not OK */
	d->flags &= ~ICON_SEL_DLG_RESPONSE_OK;

	/* Break out of one main level */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}


/*
 *	Gets the current location.
 */
const gchar *IconSelDlgCurrentLocation(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(NULL);

	return(gtk_entry_get_text(GTK_ENTRY(d->location_entry)));
}

/*
 *	Change location.
 *
 *	Deletes all the existing thumbs and loads all the thumbs found
 *	in the new location.
 *
 *	If the thumbs images were being passively loaded prior to this
 *	call then that loading will be stopped. Passive loading of
 *	each thumb's image will not resume.
 *
 *	The path specifies the full path to the new location, which
 *	must be a directory.
 */
void IconSelDlgChangeLocation(
	GtkWidget *w,
	const gchar *path
)
{
	gint nnames;
	gchar **names_list;
	tlist_struct *tlist;
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if((d == NULL) || STRISEMPTY(path))
	    return;

	tlist = d->tlist;

	d->freeze_count++;

	/* Stop the previous loading process */
	d->load_toid = GTK_TIMEOUT_REMOVE(d->load_toid);

	TListFreeze(tlist);

	/* Delete all the existing thumbs */
	TListClear(tlist);

	/* Get the list of objects at the specified location */
	names_list = GetDirEntNames2(
	    (const char *)path,
	    (int *)&nnames
	);
	if(names_list != NULL)
	{
	    struct stat stat_buf;
	    gint i, thumb_num;
	    const gchar *ext;
	    gchar *name, *child_path;

	    names_list = StringQSort(
		names_list,
		nnames
	    );

#ifdef S_ISDIR
	    /* Get the directories */
	    for(i = 0; i < nnames; i++)
	    {
 		name = names_list[i];
		if(name == NULL)
		    continue;

		/* Parent directory? */
		if(!strcmp((const char *)name, ".."))
		{
		    thumb_num = TListAppend(
			tlist,
			name
		    );
		    if(thumb_num > -1)
		    {
			TListSetThumbDataFull(
			    tlist,
			    thumb_num,
			    g_dirname(path),		/* Parent path */
			    (GtkDestroyNotify)g_free
			);
			TListSetPixmap(
			    tlist,
			    thumb_num,
			    d->parent_dir_pixmap,
			    d->parent_dir_mask
			);
			TListSetLoadState(
			    tlist,
			    thumb_num,
			    TLIST_LOAD_STATE_LOADED
			);
		    }
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Skip special notations */
		if(!strcmp((const char *)name, "."))
		{
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Get the full path to this object */
		child_path = g_strconcat(
		    path,
		    G_DIR_SEPARATOR_S,
		    name,
		    NULL
		);
		if(child_path == NULL)
		{
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Get this object's statistics */
		if(stat((const char *)child_path, &stat_buf))
		{
		    g_free(child_path);
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Not a directory? */
		if(!S_ISDIR(stat_buf.st_mode))
		{
		    /* Skip this object but leave it in the list */
		    g_free(child_path);
		    continue;
		}

		/* Add this directory to the Thumbs List */
		thumb_num = TListAppend(
		    tlist,
		    name
		);
		if(thumb_num > -1)
		{
		    TListSetThumbDataFull(
			tlist,
			thumb_num,
			STRDUP(child_path),
			(GtkDestroyNotify)g_free
		    );
		    TListSetPixmap(
			tlist,
			thumb_num,
			d->dir_pixmap,
			d->dir_mask
		    );
		    TListSetLoadState(
			tlist,
			thumb_num,
			TLIST_LOAD_STATE_LOADED
		    );
		}

		/* Remove the name of this directory from the list */
		g_free(child_path);
		g_free(name);
		names_list[i] = NULL;
	    }
#endif	/* S_ISDIR */

	    /* Get the objects */
	    for(i = 0; i < nnames; i++)
	    {
 		name = names_list[i];
		if(name == NULL)
		    continue;

		/* Get the full path to this object */
		child_path = g_strconcat(
		    path,
		    G_DIR_SEPARATOR_S,
		    name,
		    NULL
		);
		if(child_path == NULL)
		{
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Get this object's statistics */
		if(stat((const char *)child_path, &stat_buf))
		{
		    g_free(child_path);
		    g_free(name);
		    names_list[i] = NULL;
		    continue;
		}

		/* Get the extension */
		ext = (const gchar *)strchr(
		    (const char *)name,
		    '.'
		);
		if(ext != NULL)
		{
		    /* Handle by the extension */
		    if(!strcmp((const char *)ext, ".xpm"))
		    {
			thumb_num = TListAppend(
			    tlist,
			    name
			);
			if(thumb_num > -1)
			{
			    TListSetThumbDataFull(
				tlist,
				thumb_num,
				STRDUP(child_path),
				(GtkDestroyNotify)g_free
			    );
			    TListSetLoadState(
				tlist,
				thumb_num,
				TLIST_LOAD_STATE_NOT_LOADED
			    );
			}
		    }
		}

		/* Remove the name of this object from the list */
		g_free(child_path);
		g_free(name);
		names_list[i] = NULL;
	    }

	    /* Delete the names list */
	    g_free(names_list);
	}

	TListThaw(tlist);

	/* Set the new location on the location GtkEntry */
	if(d->location_entry != NULL)
	{
	    gchar *dpath = STRDUP(path);
	    gtk_entry_set_text(
		GTK_ENTRY(d->location_entry),
		dpath
	    );
	    g_free(dpath);
	}

	d->freeze_count--;
}


/*
 *	Checks if the thumb image loading is in progress.
 */
gboolean IconSelDlgIsLoading(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(FALSE);

	return((d->load_toid != 0) ? TRUE : FALSE);
}

/*
 *	Queues the loading of the thumb's images.
 *
 *	If the icons are already being loaded then nothing will be
 *	done.
 */
void IconSelDlgQueueLoading(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return;

	if(d->load_toid == 0)
	    d->load_toid = gtk_timeout_add(
		(d->load_int != 0) ? d->load_int : ICON_SEL_DLG_DEF_LOAD_INT,
		IconSelTListLoadTOCB, d
	    );
}

/*
 *	Stops the loading of the thumb's images.
 */
void IconSelDlgStopLoading(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return;

	d->load_toid = GTK_TIMEOUT_REMOVE(d->load_toid);
}


/*
 *	Checks if the Icon Selector Dialog is making a query.
 */
gboolean IconSelDlgIsQuery(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(FALSE);

	return((d->main_level > 0) ? TRUE : FALSE);
}

/*
 *	Query the user to select icons by creating and mapping a new
 *	Icon Selector Dialog and waiting for the user response.
 *
 *	The title specifies the title to display on the Icon Selector
 *	Dialog.
 *
 *	The path specifies the string describing the full path to the
 *	directory or icon file. If path is NULL then the last location
 *	will be used. If there was no previous location set (thumbs
 *	list is empty) then the default_path will be used.
 *
 *	The req_icon_width and req_icon_height specifies the prefered
 *	icon size.
 *
 *	The load_int specifies the image loading interval in
 *	milliseconds. If laod_int is 0 then the default interval is
 *	used.
 *
 *	The ref_toplevel specifies the reference toplevel GtkWidget.
 *
 *	Returns a GList of gchar * paths to the selected icons. The
 *	calling function must delete the returned list and each string.
 *	If the user cancels then NULL is returned.
 */
GList *IconSelDlgQuery(
	GtkWidget *w,
	const gchar *title,
	const gchar *path,
	const gint req_icon_width, const gint req_icon_height,
	GtkWidget *ref_toplevel
)
{
	GList *selected_paths_list;
	GtkWidget *toplevel;
	tlist_struct *tlist;
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(NULL);

	/* Already querying? */
	if(d->main_level > 0)
	    return(NULL);

	toplevel = w;
	tlist = d->tlist;

	/* Ref the dialog */
	toplevel = IconSelDlgRef(toplevel);

	/* Set the new requested icon size */
	d->req_icon_width = req_icon_width;
	d->req_icon_height = req_icon_height;

	/* Set the new title? */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(toplevel),
		title
	    );

	/* Set the new location? */
	if(!STRISEMPTY(path))
	{
	    struct stat stat_buf;
	    gchar *parent;
	    const gchar	*select_thumb_path,
			*cur_location = IconSelDlgCurrentLocation(toplevel);
	    if(stat((const char *)path, &stat_buf))
	    {
		parent = g_dirname(path);
		select_thumb_path = NULL;
	    }
	    else if(S_ISDIR(stat_buf.st_mode))
	    {
		parent = STRDUP(path);
		select_thumb_path = NULL;
	    }
	    else
	    {
		parent = g_dirname(path);
		select_thumb_path = path;
	    }

	    /* New location is different from current location? */
	    if((parent != NULL) && (cur_location != NULL))
	    {
		if(strcmp((const char *)cur_location, (const char *)parent))
		{
		    /* Set the new location and load the thumbs from
		     * that location
		     */
		    IconSelDlgChangeLocation(toplevel, parent);
		}
	    }

	    /* Select an existing thumb? */
	    if(select_thumb_path != NULL)
	    {
		const gint nthumbs = tlist->total_thumbs;
		gint i;
		const gchar *thumb_path;
		tlist_thumb_struct *thumb;

		for(i = 0; i < nthumbs; i++)
		{
		    thumb = tlist->thumb[i];
		    if(thumb == NULL)
			continue;

		    thumb_path = (const gchar *)thumb->data;
		    if(thumb_path == NULL)
			continue;

		    if(!strcmp(thumb_path, select_thumb_path))
		    {
			TListFreeze(tlist);
			TListUnselectAll(tlist);
			TListSelectThumb(tlist, i);
			TListQueueMoveTo(tlist, i, 0.5f);
			TListThaw(tlist);
			break;
		    }
		}
	    }

	    g_free(parent);
	}
	else
	{
	    /* Use existing location, check if the current location
	     * is not set
	     */
	    const gchar *cur_location = IconSelDlgCurrentLocation(toplevel);
	    if(STRISEMPTY(cur_location))
	    {
		/* No current location set, so use the default location */
		if(!STRISEMPTY(d->default_path))
		{
		    /* Set the default location as the current location
		     * and load the thumbs from that location
		     */
		    IconSelDlgChangeLocation(toplevel, d->default_path);
		}
	    }
	}

	/* Set the reference toplevel? */
	if(ref_toplevel != NULL)
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel),
		GTK_WINDOW(ref_toplevel)
	    );
	else
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel),
		NULL
	    );

	/* Map the dialog */
	IconSelDlgMap(toplevel);

	/* Start the loading of the thumb images */
	IconSelDlgQueueLoading(toplevel);

	/* Wait for the user response */
	d->main_level++;
	gtk_main();

	/* Did the user click on OK? */
	if(d->flags & ICON_SEL_DLG_RESPONSE_OK)
	{
	    /* Transfer the selected paths list to the return list */
	    selected_paths_list = d->selected_paths_list;
	    d->selected_paths_list = NULL;
	}
	else
	{
	    selected_paths_list = NULL;
	}

	/* Stop the loading of the thumb images */
	IconSelDlgStopLoading(toplevel);

	/* Unmap the dialog */
	IconSelDlgUnmap(toplevel);

	/* Unref the dialog */
	(void)IconSelDlgUnref(toplevel);

	return(selected_paths_list);
}


/*
 *	Creates a new Icon Selector Dialog.
 *
 *	The flags specifies the Icon Selector Dialog flags.
 *
 *	The list_orientation specifies the orientation of the list,
 *	either GTK_ORIENTATION_HORIZONTAL or GTK_ORIENTATION_VERTICAL.
 *
 *	The list_height specifies the height of the list in pixels. If
 *	list_height is -1 then the default height is used.
 *
 *	The thumb_width and thumb_height specifies the size of each
 *	thumb in pixels.
 *
 *	The thumb_border specifies the border of the thumb in pixels.
 *
 *	Returns the new Icon Selector Dialog.
 */
GtkWidget *IconSelDlgNew(
	const icon_sel_dlg_flags flags,
	const GtkOrientation list_orientation,
	const gint list_height,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	const gchar *default_path,
	const gulong load_int
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	tlist_struct *tlist;
	IconSelDlg *d = ICON_SEL_DLG(g_malloc0(sizeof(IconSelDlg)));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
/*
	d->freeze_count = 0;
 */
	d->ref_count = 1;
/*
	d->main_level = 0;
 */
	d->flags = flags;
	d->flags &= ~ICON_SEL_DLG_MAPPED;
	d->flags &= ~ICON_SEL_DLG_REALIZED;
/*
	d->req_icon_width = 0;
	d->req_icon_height = 0;
 */
	d->default_path = STRDUP(default_path);
/*
	d->load_toid = 0;
 */
	d->load_int = (load_int != 0l) ? load_int : ICON_SEL_DLG_DEF_LOAD_INT;
/*
	d->selected_paths_list = NULL;
 */

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_object_set_data_full(
	    GTK_OBJECT(w), ICON_SEL_DLG_KEY,
	    d, IconSelDlgDataDestroyCB
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_usize(
	    w,
	    ICON_SEL_DLG_WIDTH, ICON_SEL_DLG_HEIGHT
	);
	gtk_window_set_title(GTK_WINDOW(w), ICON_SEL_DLG_DEF_TITLE);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(IconSelDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Location GtkLabel */
	w = gtk_label_new("Location:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Location GtkEntry */
	d->location_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(IconSelDlgLocationEntryActivateCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	/* Location Browse GtkButton */
	d->location_browse_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(IconSelDlgLocationBrowseCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Browse"
	);
	gtk_widget_show(w);


	/* Thumbs List */
	d->tlist = tlist = TListNew(
	    list_orientation,
	    thumb_width, thumb_height,
	    thumb_border,
	    IconSelDlgTListSelectCB,
	    d,
	    IconSelDlgTListUnselectCB,
	    d
	);
	w = tlist->toplevel;
	gtk_widget_set_usize(
	    w,
	    -1,
	    (list_height > 0) ? list_height : ICON_SEL_DLG_DEF_LIST_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	TListDoubleBuffer(
	    tlist,
	    (flags & ICON_SEL_DLG_DOUBLE_BUFFER) ? TRUE : FALSE
	);
	TListSelectionMode(tlist, GTK_SELECTION_EXTENDED);
	TListShowThumbFrames(tlist, FALSE);
	TListShowThumbLabels(
	    tlist,
	    (flags & ICON_SEL_DLG_SHOW_ICON_PATHS) ? TRUE : FALSE
	);
	TListShowTextTips(
	    tlist,
	    (flags & ICON_SEL_DLG_SHOW_TEXTTIPS) ? TRUE : FALSE
	);
	TListEnableListDragScroll(
	    tlist,
	    (flags & ICON_SEL_DLG_ENABLE_DRAG_SCROLL) ? TRUE : FALSE
	);
	TListMap(tlist);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK GtkButton */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
	    "Select",
	    NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(IconSelDlgOKCB), d
	);
	gtk_widget_show(w);
	  
	/* Cancel GtkButton */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
	    "Cancel",
	    NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(IconSelDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w); 


	/* Load the icons */
	d->parent_dir_pixmap = gdk_pixmap_create_from_xpm_d(
	    window,
	    &d->parent_dir_mask,
	    NULL,
	    (gchar **)icon_folder_parent_32x32_xpm
	);
	d->dir_pixmap = gdk_pixmap_create_from_xpm_d(
	    window,
	    &d->dir_mask,
	    NULL,
	    (gchar **)icon_folder_closed_32x32_xpm
	);


	d->freeze_count--;

	return(d->toplevel);
}

/*
 *	Updates the widgets.
 */
void IconSelDlgUpdateWidgets(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return;


}

/*
 *	Maps the Icon Selector Dialog.
 */
void IconSelDlgMap(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);
	d->flags |= ICON_SEL_DLG_MAPPED;

	TListGrabFocus(d->tlist);
}

/*
 *	Unmaps the Icon Selector Dialog.
 */
void IconSelDlgUnmap(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
	d->flags &= ~ICON_SEL_DLG_MAPPED;
}

/*
 *	Adds a reference count to the Icon Selector Dialog.
 *
 *	Returns the Icon Selector Dialog.
 */
GtkWidget *IconSelDlgRef(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(NULL);

	d->ref_count++;

	return(w);
}

/*
 *	Removes a reference count from the Icon Selector Dialog.
 *
 *	If the reference counts reach 0 then the Icon Selector Dialog
 *	is destroyed.
 *
 *	This function always returns NULL.
 */
GtkWidget *IconSelDlgUnref(GtkWidget *w)
{
	IconSelDlg *d = ICON_SEL_DLG(GTK_OBJECT_GET_DATA(
	    w,
	    ICON_SEL_DLG_KEY
	));
	if(d == NULL)
	    return(NULL);

	d->ref_count--;

	/* All reference counts removed? */
	if(d->ref_count <= 0)
	{
	    /* Delete the Icon Selector Dialog */

	    d->load_toid = GTK_TIMEOUT_REMOVE(d->load_toid);

	    IconSelDlgUnmap(d->toplevel);

	    d->freeze_count++;

	    TListClear(d->tlist);
	    TListDelete(d->tlist);

	    d->freeze_count--;

	    /* Destroy the toplevel GtkWindow and call the "destroy"
	     * callback
	     */
	    gtk_widget_destroy(d->toplevel);
	}

	return(NULL);
}
