#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "icon_sel_dlg.h"

#include "icon_picker.h"


typedef struct _IconPicker		IconPicker;
#define ICON_PICKER(p)			((IconPicker *)(p))
#define ICON_PICKER_KEY			"IconPicker"


/* Callbacks */
static void icon_picker_destroy_cb(gpointer data);
static gint icon_picker_display_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void icon_picker_display_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data,
	guint info,
	guint t,
	gpointer data
);
static void icon_picker_display_drag_leave_cb(
	GtkWidget *widget, GdkDragContext *dc, guint t,
	gpointer data
);
static gboolean icon_picker_display_drag_motion_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	guint t,
	gpointer data
);
static void icon_picker_display_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data,
	guint info,
	guint t,
	gpointer data
);

static void icon_picker_display_realize_cb(GtkWidget *widget, gpointer data);

static gint icon_picker_icon_label_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void icon_picker_icon_label_realize_cb(GtkWidget *widget, gpointer data);

static void icon_picker_previous_cb(GtkWidget *widget, gpointer data);
static void icon_picker_next_cb(GtkWidget *widget, gpointer data);
static void icon_picker_change_cb(GtkWidget *widget, gpointer data);
static void icon_picker_clear_cb(GtkWidget *widget, gpointer data);

/* Icon */
IconPickerIcon *icon_picker_icon_new(void);
IconPickerIcon *icon_picker_icon_copy(IconPickerIcon *icon);
void icon_picker_icon_delete(IconPickerIcon *icon);

/* Draw */
static void icon_picker_draw_icon_display(IconPicker *ip);
static void icon_picker_draw_icon_label(IconPicker *ip);

/* Switch To Icon */
void icon_picker_switch_to(
	GtkWidget *w,
	const gint i
);

/* Change/Clear Icon */
gint icon_picker_change_icon(
	GtkWidget *w,
	const gchar *path,
	const gboolean verbose,
	const gboolean interactive
);
gboolean icon_picker_change_icon_query(GtkWidget *w);
gboolean icon_picker_clear_icon(GtkWidget *w);

/* Update Tip */
static void icon_picker_update_tooltip(IconPicker *ip);

/* Icon Picker */
GtkWidget *icon_picker_new(
	const gchar *title,
	const IconPickerFlags flags,
	const IconPickerIconPosition icon_position,
	const gint req_icon_width, const gint req_icon_height,
	const gint display_width, const gint display_height,
	const gint label_width,
	GtkWidget *icon_sel_dlg
);
void icon_picker_set_has_changes(
	GtkWidget *w,
	const gboolean has_changes
);
void icon_picker_set_switched_icon_cb(
	GtkWidget *w,
	void (*cb)(
		GtkWidget *w,
		const gint,
		IconPickerIcon *,
		gpointer
	),
	gpointer data 
);
void icon_picker_set_changed_cb(
	GtkWidget *w,
	void (*cb)(
		GtkWidget *w,
		gpointer
	),
	gpointer data 
);
void icon_picker_set_icon(
	GtkWidget *w,
	const gint i,
	IconPickerIcon *icon
);
void icon_picker_set_icons_list(
	GtkWidget *w,
	GList *icons_list                       /* Transfered */
);
IconPickerIcon *icon_picker_get_icon(
	GtkWidget *w,
	const gint i
);
IconPickerIcon *icon_picker_get_current_icon(GtkWidget *w);
GList *icon_picker_get_icons_list(GtkWidget *w);
void icon_picker_clear(GtkWidget *w);
void icon_picker_queue_draw(GtkWidget *w);
static void icon_picker_update_widgets(IconPicker *ip);


#define ICON_PICKER_ARROW_BUTTON_WIDTH	16
#define ICON_PICKER_ARROW_BUTTON_HEIGHT	16

#define ICON_PICKER_BUTTON_WIDTH	60
#define ICON_PICKER_BUTTON_HEIGHT	-1


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Icon Picker:
 */
struct _IconPicker {

	GtkWidget	*toplevel;		/* GtkBox */
	gint		freeze_count;
	GdkGC		*gc;

	IconPickerFlags	flags;
	IconPickerIconPosition	icon_position;

	gint		req_icon_width,		/* Prefered icon size */
			req_icon_height;

	GtkWidget       *title_label,
			*display_da,
			*switch_buttons_toplevel,
			*left_btn,
			*right_btn,
			*icon_label_da,
			*change_btn,
			*clear_btn,
			*menu,
			*next_mi,
			*prev_mi,
			*change_mi,
			*clear_mi;

	GdkPixmap       *display_pixmap;	/* Back buffer for display_da */

	/* Icons List, GList of IconPickerIcon * icons */
	GList		*icons_list;
	gint		nicons,
			current_icon_num;

	/* Switched Icon Callback, called whenever a new icon is
	 * displayed
	 */
	void		(*switched_icon_cb)(
			GtkWidget *,		/* Icon Picker */
			const gint,		/* Icon index */
			IconPickerIcon *,	/* Icon */
			gpointer		/* Data */
	);
	gpointer	switched_icon_data;

	/* Changed Callback, called whenever an icon has been changed */
	void		(*changed_cb)(
			GtkWidget *,		/* Icon Picker */
			gpointer		/* Data */
	);
	gpointer	changed_data;

	/* Icon Selector Dialog */
	GtkWidget	*icon_sel_dlg;
};


/*
 *	Toplevel GtkWidget "destroy" signal callback.
 */
static void icon_picker_destroy_cb(gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if(ip == NULL)
		return;

	ip->freeze_count++;

	/* Unref the Icon Selector Dialog */
	ip->icon_sel_dlg = IconSelDlgUnref(ip->icon_sel_dlg);

	/* Delete all the icons */
	if(ip->icons_list != NULL)
	{
		g_list_foreach(
			ip->icons_list,
			(GFunc)icon_picker_icon_delete,
			NULL
		);
		g_list_free(ip->icons_list);
	}

	/* Destroy the Right Click GtkMenu */
	gtk_widget_destroy(ip->menu);

	(void)GDK_GC_UNREF(ip->gc);
	(void)GDK_PIXMAP_UNREF(ip->display_pixmap);

	ip->freeze_count--;

	g_free(ip);
}

/*
 *	Display GtkDrawingArea event signal callback.
 */
static gint icon_picker_display_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gboolean key_press;
	gint status = FALSE;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (event == NULL) || (ip == NULL))
		return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
		configure = (GdkEventConfigure *)event;

		/* Recreate the display GdkPixmap */
		GDK_PIXMAP_UNREF(ip->display_pixmap);
		if(ip->flags & ICON_PICKER_DOUBLE_BUFFER)
			ip->display_pixmap = gdk_pixmap_new(
				configure->window,
				configure->width, configure->height,
				-1
			);
		else
			ip->display_pixmap = NULL;

		status = TRUE;
		break;

	  case GDK_EXPOSE:
		/* Draw the display */
		icon_picker_draw_icon_display(ip);
		status = TRUE;
		break;

	  case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
		}
		else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
		}
		status = TRUE;
		break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
		if(ip->freeze_count > 0)
			return(status);
		key = (GdkEventKey *)event;
		key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
#define STOP_KEY_SIGNAL_EMIT	{		\
 if(widget != NULL)				\
  gtk_signal_emit_stop_by_name(			\
   GTK_OBJECT(widget),				\
   key_press ?					\
    "key_press_event" : "key_release_event"	\
  );						\
}
		switch(key->keyval)
		{
		  case GDK_Left:
		  case GDK_KP_Left:
			if(key_press)
			{
				/* Previous icon */
				ip->current_icon_num--;
				if(ip->current_icon_num < 0)
					ip->current_icon_num = 0;
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_Right:
		  case GDK_KP_Right:
			if(key_press)
			{
				/* Next icon */
				ip->current_icon_num++;
				if(ip->current_icon_num >= ip->nicons)
					ip->current_icon_num = MAX((ip->nicons - 1), 0);
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_Home:
		  case GDK_KP_Home:
			if(key_press)
			{
				/* First icon */
				ip->current_icon_num = 0;
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_End:
		  case GDK_KP_End:
			if(key_press)
			{
				/* Last icon */
				ip->current_icon_num = MAX((ip->nicons - 1), 0);
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_Return:
		  case GDK_KP_Enter:
		  case GDK_3270_Enter:
		  case GDK_ISO_Enter:
		  case GDK_space:
		  case GDK_Insert:
			if(key_press)
			{
				/* Query the user to change the icon */
				(void)icon_picker_change_icon_query(ip->toplevel);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_Delete:
			if(key_press)
			{
				/* Clear icon */
				(void)icon_picker_clear_icon(ip->toplevel);
			}
			STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		}
#undef STOP_KEY_SIGNAL_EMIT
		break;

	  case GDK_BUTTON_PRESS:
		if(ip->freeze_count > 0)
			return(status);
		button = (GdkEventButton *)event;
		if(!GTK_WIDGET_HAS_FOCUS(widget))
			gtk_widget_grab_focus(widget);
		switch(button->button)
		{
		  case GDK_BUTTON1:
			/* Set the DND drag icon */
		    if(ip->flags & ICON_PICKER_PROVIDE_DND_DRAG)
		    {
				IconPickerIcon *icon = icon_picker_get_current_icon(ip->toplevel);
				if(icon != NULL)
					GUIDNDSetDragIcon(
						widget,
						icon->pixmap, icon->mask,
						GUI_DND_DRAG_ICON_DEF_HOT_X, GUI_DND_DRAG_ICON_DEF_HOT_Y
					);
				else
					GUIDNDClearDragIcon(widget);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON3:
			/* Map the right click GtkMenu */
			if(ip->menu != NULL)
			{
				gtk_menu_popup(
					GTK_MENU(ip->menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
				status = TRUE;
			}
			break;

		  case GDK_BUTTON4:
			if(TRUE)
			{
				/* Previous icon */
				ip->current_icon_num--;
				if(ip->current_icon_num < 0)
					ip->current_icon_num = 0;
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			if(TRUE)
			{
				/* Next icon */
				ip->current_icon_num++;
				if(ip->current_icon_num >= ip->nicons)
					ip->current_icon_num = MAX((ip->nicons - 1), 0);
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			status = TRUE;
			break;
		}
		ip->flags |= ICON_PICKER_BUTTON_PRESSED;
		break;

	  case GDK_BUTTON_RELEASE:
		if(ip->freeze_count > 0)
			return(status);
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON1:
			if(ip->flags & ICON_PICKER_BUTTON_PRESSED)
			{
				gint width, height;
				gdk_window_get_size(widget->window, &width, &height);
				if((button->x >= 0) && (button->x < width) &&
				   (button->y >= 0) && (button->y < height)
				)
				{
					/* Need to remove the button press flag before
					 * querying to change the icon or else this case
					 * will get triggered again
					 */
					ip->flags &= ~ICON_PICKER_BUTTON_PRESSED;

					/* Query the user to change the icon */
					(void)icon_picker_change_icon_query(ip->toplevel);
				}
			}
			status = TRUE;
			break;
		}
		ip->flags &= ~ICON_PICKER_BUTTON_PRESSED;
		break;
	}

	return(status);
}

/*
 *	Display GtkDrawingArea "drag_data_get" signal callback.
 */
static void icon_picker_display_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	IconPickerIcon *icon;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (selection_data == NULL) || (ip == NULL))
		return;

	if(ip->freeze_count > 0)
		return;

	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon != NULL)
	{
		if(!STRISEMPTY(icon->path))
		{
			gchar *s = g_strconcat(
				"file://",
				icon->path,
				NULL
			);
			gtk_selection_data_set(
				selection_data,
				GDK_SELECTION_TYPE_STRING,
				8,				/* Bits Per Character */
				s,				/* Data */
				STRLEN(s)			/* Length */
			);
			g_free(s);
			data_sent = TRUE;
		}
	}

	if(!data_sent)
	{
		const gchar *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,				/* Bits Per Character */
			s,				/* Data */
			STRLEN(s)			/* Length */
		);
		data_sent = TRUE;
	}
}

/*
 *	Display GtkDrawingArea "drag_leave" signal callback.
 */
static void icon_picker_display_drag_leave_cb(
	GtkWidget *widget, GdkDragContext *dc, guint t,
	gpointer data
)
{
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (dc == NULL) || (ip == NULL))
		return;

	if(!GTK_WIDGET_SENSITIVE(widget))
		return;

	/* Mark that the drag has left */
	if(ip->flags & ICON_PICKER_DRAG_OVER)
	{
		ip->flags &= ~ICON_PICKER_DRAG_OVER;
		gtk_widget_queue_draw(widget);
	}
}

/*
 *	Display GtkDrawingArea "drag_motion" signal callback.
 */
static gboolean icon_picker_display_drag_motion_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	guint t,
	gpointer data
)
{
	gchar *target_name;
	GList *glist;
	GdkDragAction	allowed_actions,
					default_action,
					action;
	GtkTargetFlags target_flags;
	GtkWidget *src_w;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (dc == NULL) || (ip == NULL))
		return(FALSE);

	if(!GTK_WIDGET_SENSITIVE(widget))
		return(FALSE);

	/* Get the GtkWidget that the drag originated from */
	src_w = gtk_drag_get_source_widget(dc);


	/* Set the target flags to denote if this is the same
	 * GtkWidget or not
	 */
	target_flags = ((src_w == widget) ? GTK_TARGET_SAME_WIDGET : 0);

	/* Iterate through the list of targets on the drag context
	 * and check if we will accept one of them
	 */
	for(glist = dc->targets;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		/* Get the next target name on the drag context */
		target_name = gdk_atom_name((GdkAtom)glist->data);
		if(target_name != NULL)
		{
			/* Check if this target name is in the list of
			 * accepted targets on the GtkWidget
			 */
			if(GUIDNDIsTargetAccepted(
				widget,
				target_name,
				target_flags
			))
			{
				g_free(target_name);
				break;
			}

			g_free(target_name);
		}
	}
	/* Drag not accepted? */
	if(glist == NULL)
		return(FALSE);

	/* Determine the appropriate drag action */
	allowed_actions = GDK_ACTION_COPY;
	default_action = GDK_ACTION_COPY;
	action = GUIDNDDetermineDragAction(
		dc->actions,                        /* Requested actions */
		allowed_actions,
		default_action
	);


	/* Mark that the drag is occuring over this widget */
	if((action != 0) &&
	   !(ip->flags & ICON_PICKER_DRAG_OVER)
	)
	{
		ip->flags |= ICON_PICKER_DRAG_OVER;
		gtk_widget_queue_draw(widget);
	}
	else if((action == 0) &&
			(ip->flags & ICON_PICKER_DRAG_OVER)
	)
	{
		ip->flags &= ~ICON_PICKER_DRAG_OVER;
		gtk_widget_queue_draw(widget);
	}

	return(FALSE);
}


/*
 *	Display GtkDrawingArea "drag_data_received" signal callback.
 */
static void icon_picker_display_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	IconPickerIcon *icon;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (selection_data == NULL) || (ip == NULL))
		return;

	if((selection_data->length <= 0) ||
	   (selection_data->data == NULL)
	)
		return;

	/* Ignore if the data came from the same GtkWidget as this one */
	if(gtk_drag_get_source_widget(dc) == widget)
		return;

	if(ip->freeze_count > 0)
		return;

	/* Changing of icons not allowed? */
	if(ip->flags & ICON_PICKER_NO_CHANGE_ICON)
		return;

	/* Get the current icon */
	icon = icon_picker_get_current_icon(ip->toplevel);

	/* Handle by the target type */
	if((info == 0) ||
	   (info == 1) ||
	   (info == 2)
	)
	{
		const gchar *url_pfx = "file://";
		const gchar *path = (const gchar *)selection_data->data;

		if(!g_strncasecmp(
			path,
			url_pfx,
			STRLEN(url_pfx)
		))
			path += STRLEN(url_pfx);

		path = (const gchar *)strchr(
			(const char *)path,
			G_DIR_SEPARATOR
		);

		if(path != NULL)
		{
			/* Set the new icon, set the has changes flag, call
			 * the changed and switched icon callbacks, and update
			 * the tooltips
			 */
			(void)icon_picker_change_icon(
				ip->toplevel,
				path,
				TRUE,			/* Verbose */
				TRUE			/* Interactive */
			);
		}
	}
}


/*
 *	Display GtkDrawingArea "realized" signal callback.
 */
static void icon_picker_display_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (ip == NULL))
		return;

	window = widget->window;
	if(window == NULL)
		return;

	/* Create the GC as needed */
	if(ip->gc == NULL)
		ip->gc = gdk_gc_new(window);

	/* Mark the Icon Picker as realized */
	ip->flags |= ICON_PICKER_REALIZED;
}


/*
 *	Icon Label GtkDrawingArea event signal callback.
 */
static gint icon_picker_icon_label_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (event == NULL) || (ip == NULL))
		return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
		status = TRUE;
		break;

	  case GDK_EXPOSE:
		/* Draw the icon label */
		icon_picker_draw_icon_label(ip);
		status = TRUE;
		break;

	  case GDK_BUTTON_PRESS:
		if(ip->freeze_count > 0)
			return(status);
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON3:
			/* Map the right click GtkMenu */
			if(ip->menu != NULL)
			{
				gtk_menu_popup(
					GTK_MENU(ip->menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
				status = TRUE;
			}
			break;

		  case GDK_BUTTON4:
			if(TRUE)
			{
				/* Previous icon */
				ip->current_icon_num--;
				if(ip->current_icon_num < 0)
					ip->current_icon_num = 0;
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			if(TRUE)
			{
				/* Next icon */
				ip->current_icon_num++;
				if(ip->current_icon_num >= ip->nicons)
					ip->current_icon_num = MAX((ip->nicons - 1), 0);
				icon_picker_switch_to(ip->toplevel, ip->current_icon_num);
			}
			status = TRUE;
			break;
		}
		ip->flags |= ICON_PICKER_BUTTON_PRESSED;
		break;

	  case GDK_BUTTON_RELEASE:
		if(ip->freeze_count > 0)
			return(status);
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON1:
			if(ip->flags & ICON_PICKER_BUTTON_PRESSED)
			{
				gint width, height;
				gdk_window_get_size(widget->window, &width, &height);
				if((button->x >= 0) && (button->x < width) &&
				   (button->y >= 0) && (button->y < height)
				)
				{
					/* Need to remove the button press flag before
					 * querying to change the icon or else this case
					 * will get triggered again
					 */
					ip->flags &= ~ICON_PICKER_BUTTON_PRESSED;

					/* Query the user to change the icon */
					(void)icon_picker_change_icon_query(ip->toplevel);
				}
			}
			status = TRUE;
			break;
		}
		ip->flags &= ~ICON_PICKER_BUTTON_PRESSED;
		break;

	}

	return(status);
}

/*
 *	Current Icon Label GtkDrawingArea "realized" signal callback.
 */
static void icon_picker_icon_label_realize_cb(GtkWidget *widget, gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if((widget == NULL) || (ip == NULL))
		return;


}


/*
 *	Previous callback.
 */
static void icon_picker_previous_cb(GtkWidget *widget, gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if(ip == NULL)
		return;

	if(ip->freeze_count > 0)
		return;

	/* Switch to the previous icon */
	ip->current_icon_num--;
	if(ip->current_icon_num < 0)
		ip->current_icon_num = 0;
	icon_picker_switch_to(
		ip->toplevel,
		ip->current_icon_num
	);
}

/*
 *	Next callback.
 */
static void icon_picker_next_cb(GtkWidget *widget, gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if(ip == NULL)
		return;

	if(ip->freeze_count > 0)
		return;

	/* Switch to the next icon */
	ip->current_icon_num++;
	if(ip->current_icon_num >= ip->nicons)
		ip->current_icon_num = MAX((ip->nicons - 1), 0);
	icon_picker_switch_to(
		ip->toplevel,
		ip->current_icon_num
	);
}

/*
 *	Change icon callback.
 */
static void icon_picker_change_cb(GtkWidget *widget, gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if(ip == NULL)
		return;

	if(ip->freeze_count > 0)
		return;

	/* Query the user to change the icon */
	(void)icon_picker_change_icon_query(ip->toplevel);
}

/*
 *	Clear callback.
 */
static void icon_picker_clear_cb(GtkWidget *widget, gpointer data)
{
	IconPicker *ip = ICON_PICKER(data);
	if(ip == NULL)
		return;

	if(ip->freeze_count > 0)
		return;

	/* Clear the icon */
	(void)icon_picker_clear_icon(ip->toplevel);
}


/*
 *	Creates a new Icon.
 */
IconPickerIcon *icon_picker_icon_new(void)
{
	return(ICON_PICKER_ICON(g_malloc0(sizeof(IconPickerIcon))));
}

/*
 *	Coppies the icon.
 *
 *	Members data and destroy_cb will not be coppied.
 */
IconPickerIcon *icon_picker_icon_copy(IconPickerIcon *icon)
{
	IconPickerIcon	*tar,
							*src = icon;
	if(src == NULL)
		return(NULL);

	tar = icon_picker_icon_new();
	if(tar == NULL)
		return(NULL);

	tar->label = STRDUP(src->label);
	tar->path = STRDUP(src->path);
	tar->pixmap = GDK_PIXMAP_REF(src->pixmap);
	tar->mask = GDK_PIXMAP_REF(src->mask);
	tar->width = src->width;
	tar->height = src->height;
/*
	tar->data = NULL;
	tar->destroy_cb = NULL;
 */

	return(tar);
}

/*
 *	Deletes the Icon.
 */
void icon_picker_icon_delete(IconPickerIcon *icon)
{
	if(icon == NULL)
		return;

	/* Notify about this icon being destroyed */
	if(icon->destroy_cb != NULL)
		icon->destroy_cb(icon->data);

	g_free(icon->label);
	g_free(icon->path);
	GDK_PIXMAP_UNREF(icon->pixmap);
	GDK_BITMAP_UNREF(icon->mask);
	g_free(icon);
}


/*
 *	Draws the display.
 */
static void icon_picker_draw_icon_display(IconPicker *ip)
{
	gint width, height;
	GdkDrawable *drawable;
	GdkWindow *window;
	GdkGC *gc;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w;
	IconPickerFlags flags;
	IconPickerIconPosition icon_position;
	IconPickerIcon *icon;

	if(ip == NULL)
		return;

	flags = ip->flags;
	icon_position = ip->icon_position;
	w = ip->display_da;
	state = GTK_WIDGET_STATE(w);
	window = w->window;
	gc = ip->gc;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (gc == NULL) || (style == NULL))
		return;

	drawable = (GdkDrawable *)ip->display_pixmap;
	if(drawable == NULL)
		drawable = (GdkDrawable *)window;

	gdk_window_get_size(drawable, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	/* Draw the background */
	gdk_draw_rectangle(
		drawable,
		style->base_gc[state],
		TRUE,				/* Fill */
		0, 0, width, height
	);

	/* Get the current icon */
	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon != NULL)
	{
		if(icon->pixmap != NULL)
		{
			const gint border = 3;
			gint icon_width, icon_height;
			GdkBitmap *mask = icon->mask;
			GdkPixmap *pixmap = icon->pixmap;
			gdk_window_get_size(pixmap, &icon_width, &icon_height);
			if((icon_width > 0) && (icon_height > 0))
			{
				gint x = 0, y = 0;

				switch(icon_position)
				{
				  case ICON_PICKER_ICON_POSITION_CENTER:
					x = (width - icon_width) / 2;
					y = (height - icon_height) / 2;
					break;
				  case ICON_PICKER_ICON_POSITION_UPPER_LEFT:
					x = border;
					y = border;
					break;
				  case ICON_PICKER_ICON_POSITION_TOP:
					x = (width - icon_width) / 2;
					y = border;
					break;
				  case ICON_PICKER_ICON_POSITION_UPPER_RIGHT:
					x = width - icon_width - border;
					y = border;
					break;
				  case ICON_PICKER_ICON_POSITION_RIGHT:
					x = width - icon_width - border;
					y = (height - icon_height) / 2;
					break;
				  case ICON_PICKER_ICON_POSITION_LOWER_RIGHT:
					x = width - icon_width - border;
					y = height - icon_height - border;
					break;
				  case ICON_PICKER_ICON_POSITION_BOTTOM:
					x = (width - icon_width) / 2;
					y = height - icon_height - border;
					break;
				  case ICON_PICKER_ICON_POSITION_LOWER_LEFT:
					x = border;
					y = height - icon_height - border;
					break;
				  case ICON_PICKER_ICON_POSITION_LEFT:
					x = border;
					y = (height - icon_height) / 2;
					break;
				}
				gdk_gc_set_clip_mask(gc, mask);
				if(flags & ICON_PICKER_SHOW_ICON_SHADOW)
				{
					/* Draw the shadow */
					const gint	shadow_x = x + 3,
									shadow_y = y + 3;
					gdk_gc_set_clip_origin(gc, shadow_x, shadow_y);
					gdk_gc_set_foreground(gc, &style->dark[state]);
				    gdk_draw_rectangle(
				        drawable,
						gc,
				        TRUE,		/* Fill */
						shadow_x, shadow_y,
						icon_width, icon_height
				    );
				}
				/* Draw the icon */
				gdk_gc_set_clip_origin(gc, x, y);
				gdk_draw_pixmap(
					drawable,
					gc,
					pixmap,
					0, 0,
					x, y,
					icon_width, icon_height
				);
				gdk_gc_set_clip_mask(gc, NULL);
			}
		}
	}

	/* Draw the focus if widget is focused */
	if(GTK_WIDGET_SENSITIVE(w) &&
	   (GTK_WIDGET_HAS_FOCUS(w) || (flags & ICON_PICKER_DRAG_OVER))
	)
		gtk_paint_focus(
			style,
			(GdkWindow *)drawable,
			NULL,				/* Entire area */
			w,
			NULL,				/* No detail */
			0, 0,
			width - 1, height - 1
		);

	/* If the drawable is not the window then send the drawable to
	 * the window
	 */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			window,
			gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Draws the icon label.
 */
static void icon_picker_draw_icon_label(IconPicker *ip)
{
	gint width, height;
	GdkDrawable *drawable;
	GdkWindow *window;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w;
	IconPickerIcon *icon;

	if(ip == NULL)
		return;

	w = ip->icon_label_da;
	state = GTK_WIDGET_STATE(w);
	window = w->window;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
		return;

	drawable = (GdkDrawable *)window;

	gdk_window_get_size(drawable, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	/* Draw the background */
	gdk_draw_rectangle(
		drawable,
		style->base_gc[state],
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Get the current icon */
	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon != NULL)
	{
		if((icon->label != NULL) && (style->font != NULL))
		{
			gint lbearing, rbearing, swidth, ascent, descent;
			const gchar *label = icon->label;
			GdkFont *font = style->font;
			const gint font_height = font->ascent + font->descent;

			gdk_string_extents(
				font,
				label,
				&lbearing, &rbearing, &swidth, &ascent, &descent
			);

			gdk_draw_string(
				drawable,
				font,
				style->text_gc[state],
				((width - swidth) / 2) - lbearing,
				((height - font_height) / 2) + font->ascent,
				label
			);
		}
	}

	/* Draw focus if widget is focused */
	if(GTK_WIDGET_SENSITIVE(w) && GTK_WIDGET_HAS_FOCUS(w))
		gtk_paint_focus(
			style,
			drawable,
			NULL,				/* Entire area */
			w,
			NULL,				/* No detail */
			0, 0,
			width - 1, height - 1
		);

	/* If the drawable is not the window then send the drawable to
	 * the window
	 */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}


/*
 *	Switches to the icon, updates the displayed icon, updates the
 *	widgets, and calls the switched icon callback..
 *
 *	The i specifies the icon to switch to.
 */
void icon_picker_switch_to(
	GtkWidget *w,
	const gint i
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	/* Set the current icon */
	ip->current_icon_num = i;

	/* Update the displayed icon */
	icon_picker_update_widgets(ip);

	/* Update the tooltip */
	icon_picker_update_tooltip(ip);

	/* Notify about the icon being switched */
	if(ip->switched_icon_cb != NULL)
		ip->switched_icon_cb(
			ip->toplevel,
			i,
			icon_picker_get_current_icon(ip->toplevel),
			ip->switched_icon_data
		);
}


/*
 *	Sets the current icon, sets the has changes flag, calls the
 *	changed and switched icon callbacks, and updates the tooltip.
 *
 *	The path describes the full path to the new icon image file.
 *	If the icon could not be opened from the icon image file then
 *	nothing will be done.
 *
 *	If verbose is TRUE then any warnings or errors will be
 *	displayed.
 *
 *	If interactive is TRUE then the user will be queried for any
 *	confirmation.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint icon_picker_change_icon(
	GtkWidget *w,
	const gchar *path,
	const gboolean verbose,
	const gboolean interactive
)
{
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkWindow *window;
	IconPickerIcon *icon;
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if((ip == NULL) || STRISEMPTY(path))
		return(-2);

	w = ip->display_da;
	window = w->window;
	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon == NULL)
		return(-1);

	mask = NULL;
	pixmap = NULL;
	if(TRUE)
	{
		/* Load the new pixmap and mask */
		pixmap = gdk_pixmap_create_from_xpm(
			window,
			&mask,
			NULL,				/* No background color */
			path
		);
	}
	if(pixmap != NULL)
	{
		gint width, height;

		/* Get the new icon's size */
		gdk_window_get_size(
			(GdkWindow *)pixmap,
			&width, &height
		);

		/* Does the new icon size not match the prefered icon size? */
		if((width != ip->req_icon_width) ||
		   (height != ip->req_icon_height)
		)
		{
			/* Icon too small and not allowed? */
			if(((width < ip->req_icon_width) ||
				(height < ip->req_icon_height)) &&
			   !(ip->flags & ICON_PICKER_ACCEPT_SMALLER_SIZE)
			)
			{
				if(verbose)
				{
					gchar *msg = g_strdup_printf(
"The new icon \"%s\"\n\
has a size of %ix%i, which is smaller than\n\
the recommended icon size of %ix%i, and is\n\
therefore not allowed.",
						g_basename(path),
						width, height,
						ip->req_icon_width, ip->req_icon_height
					);
					CDialogSetTransientFor(gtk_widget_get_toplevel(ip->toplevel));
					CDialogGetResponse(
						"Invalid Icon Size",
						msg,
						NULL,
						CDIALOG_ICON_WARNING,
						CDIALOG_BTNFLAG_OK,
						CDIALOG_BTNFLAG_OK
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
				}
				GDK_PIXMAP_UNREF(pixmap);
				GDK_BITMAP_UNREF(mask);
				return(-2);
			}

			/* Icon too large and not allowed? */
			if(((width > ip->req_icon_width) ||
				(height > ip->req_icon_height)) &&
			   !(ip->flags & ICON_PICKER_ACCEPT_LARGER_SIZE)
			)
			{
				if(verbose)
				{
					gchar *msg = g_strdup_printf(
"The new icon \"%s\"\n\
has a size of %ix%i, which is larger than\n\
the recommended icon size of %ix%i, and is\n\
therefore not allowed.",
						g_basename(path),
						width, height,
						ip->req_icon_width, ip->req_icon_height
					);
					CDialogSetTransientFor(gtk_widget_get_toplevel(ip->toplevel));
					CDialogGetResponse(
						"Invalid Icon Size",
						msg,
						NULL,
						CDIALOG_ICON_WARNING,
						CDIALOG_BTNFLAG_OK,
						CDIALOG_BTNFLAG_OK
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
				}
				GDK_PIXMAP_UNREF(pixmap);
				GDK_BITMAP_UNREF(mask);
				return(-2);
			}

			/* Warn the user? */
			if((ip->flags & ICON_PICKER_WARN_DIFFERENT_SIZE) &&
			   interactive
			)
			{
				gint response;
				gchar *msg = g_strdup_printf(
"The new icon \"%s\"\n\
has a size of %ix%i, which does not match\n\
the recommended icon size of %ix%i.\n\
\n\
Are you sure you want to use this icon?",
					g_basename(path),
					width, height,
					ip->req_icon_width, ip->req_icon_height
				);
				CDialogSetTransientFor(gtk_widget_get_toplevel(ip->toplevel));
				response = CDialogGetResponse(
					"Icon Size Warning",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_NO
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
				switch(response)
				{
				  case CDIALOG_RESPONSE_YES:
				  case CDIALOG_RESPONSE_YES_TO_ALL:
				  case CDIALOG_RESPONSE_OK:
					break;

				  default:
					GDK_PIXMAP_UNREF(pixmap);
					GDK_BITMAP_UNREF(mask);
					return(-5);		/* User responded with "no" */
					break;
				}
			}
		}

		/* Set the new icon */
		GDK_PIXMAP_UNREF(icon->pixmap);
		GDK_BITMAP_UNREF(icon->mask);
		icon->pixmap = pixmap;		/* Already has a ref count */
		icon->mask = mask;			/* Already has a ref count */
		icon->width = width;
		icon->height = height;

		/* Set the new icon file path */
		g_free(icon->path);
		icon->path = STRDUP(path);

		/* Set the has changes flag */
		if(!(ip->flags & ICON_PICKER_HAS_CHANGES))
			ip->flags |= ICON_PICKER_HAS_CHANGES;

		/* Update the displayed icon */
		icon_picker_update_widgets(ip);

		/* Update the tooltip */
		icon_picker_update_tooltip(ip);

		/* Notify about the icon being changed */
		if(ip->changed_cb != NULL)
			ip->changed_cb(
				ip->toplevel,
				ip->changed_data
			);

		/* Notify about the icon being switched whenever an
		 * icon has been changed
		 */
		if(ip->switched_icon_cb != NULL)
			ip->switched_icon_cb(
				ip->toplevel,
				ip->current_icon_num,
				icon,
				ip->switched_icon_data
			);

		return(0);
	}
	else
	{
		if(verbose)
		{
			gchar *msg = g_strdup_printf(
"Unable to open the icon file:\n\
\n\
    %s\n\
\n\
This file might not be a valid icon file.",
				path
			);
			CDialogSetTransientFor(gtk_widget_get_toplevel(ip->toplevel));
			CDialogGetResponse(
				"Change Icon Failed",
				msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
		}
		return(-1);
	}
}

/*
 *	Queries the user to change the icon.
 *
 *	Returns TRUE if the icon was changed or FALSE if it was not.
 */
gboolean icon_picker_change_icon_query(GtkWidget *w)
{
#if 1
	/* Use the Icon Selector Dialog */
	gboolean response = FALSE;
	const gchar *title, *label, *path;
	gchar *isd_title;
	GList *paths_list;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *toplevel;
	IconPickerFlags flags;
	IconPickerIcon *icon;
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return(FALSE);

	/* Is there already a query? */
	if(IconSelDlgIsQuery(ip->icon_sel_dlg))
		return(FALSE);

	/* Changing of icons not allowed? */
	flags = ip->flags;
	if(flags & ICON_PICKER_NO_CHANGE_ICON)
		return(FALSE);

	w = ip->display_da;
	window = w->window;
	style = gtk_widget_get_style(w);
	toplevel = gtk_widget_get_toplevel(ip->toplevel);
	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon == NULL)
		return(FALSE);

	/* Get the icon's label and the initial location */
	label = icon->label;
	path = icon->path;

	/* Set the Icon Selector Dialog title */
	title = GTK_LABEL(ip->title_label)->label;
	if(STRISEMPTY(title))
	{
		isd_title = STRDUP("Select Icon");
	}
	else
	{
		if(STRISEMPTY(label))
			isd_title = g_strdup_printf(
				"Select %s%s",
				title,
				(strcasestr((const char *)title, "icon") != NULL) ? "" : " Icon"
			);
		else
			isd_title = g_strdup_printf(
				"Select %s %s%s",
				title,
				label,
				(strcasestr((const char *)title, "icon") != NULL) ? "" : " Icon"
			);
	}

	/* Query the user for an icon */
	ip->freeze_count++;
	paths_list = IconSelDlgQuery(
		ip->icon_sel_dlg,
		isd_title,				/* Title */
		path,				/* Initial location */
		ip->req_icon_width, ip->req_icon_height,
		toplevel
	);
	ip->freeze_count--;

	g_free(isd_title);

	/* Got user response? */
	if(paths_list != NULL)
	{
		const gchar *new_path = (const gchar *)g_list_nth_data(
			paths_list,
			0
		);
		if(new_path != NULL)
		{
			/* Set the new icon, set the has changes flag, call
			 * the changed and switched icon callbacks, and update
			 * the tooltips
			 */
			(void)icon_picker_change_icon(
				ip->toplevel,
				new_path,
				TRUE,			/* Verbose */
				TRUE			/* Interactive */
			);
		}

		/* Delete the paths list */
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);

		response = TRUE;
	}

	return(response);
#else
	/* Use the File Selector Dialog */
	gboolean response;
	gint		nftypes = 0,
					npaths = 0;
	const gchar *path;
	gchar		*parent,
					**paths_list = NULL;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
					*ftype_rtn = NULL;
	IconPickerIcon *icon;
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if((ip == NULL) || FileBrowserIsQuery())
		return(FALSE);

	/* Changing of icons not allowed? */
	if(ip->flags & ICON_PICKER_NO_CHANGE_ICON)
		return(FALSE);

	w = ip->display_da;
	window = w->window;
	style = gtk_widget_get_style(w);
	toplevel = gtk_widget_get_toplevel(ip->toplevel);
	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon == NULL)
		return(FALSE);

	path = icon->path;
	parent = (path != NULL) ? g_dirname(path) : NULL;

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		".xpm", "X Pixmap"
	);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for an icon */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Icono Selecto",
"Selecto",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne Privilgie",
"Privilgi",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Erlesenes Abbild",
"Erlesen",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere L'Icona",
"Scegliere",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitgezochte Beeld",
"Uitgezocht",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione Icone",
"Selecione",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvalgt Ikon",
"Utvalgt",
"Kanseller",
#else
"Select Icon",
"Select",
"Cancel",
#endif
		parent,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		gchar *path = STRDUP(paths_list[npaths - 1]);
		if(path != NULL)
		{
			gchar *s = FileBrowserTypeCompleteExtension(
				path,
				ftype_rtn,
				FALSE			/* Do not replace if specified */
			);
			if(s != NULL)
			{
				g_free(path);
				path = s;
			}

			/* Set the new icon, set the has changes flag, call
			 * the changed and switched icon callbacks, and update
			 * the tooltips
			 */
			(void)icon_picker_change_icon(
				ip->toplevel,
				path,
				TRUE,			/* Verbose */
				TRUE			/* Interactive */
			);

			g_free(path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(parent);

	return(response);
#endif
}

/*
 *	Clears the icon.
 */
gboolean icon_picker_clear_icon(GtkWidget *w)
{
	IconPickerIcon *icon;
	IconPickerFlags flags;
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return(FALSE);

	flags = ip->flags;

	/* Clearing of icons not allowed? */
	if((flags & ICON_PICKER_NO_CHANGE_ICON) ||
	   !(flags & ICON_PICKER_CAN_CLEAR_ICON)
	)
		return(FALSE);

	icon = icon_picker_get_current_icon(ip->toplevel);
	if(icon == NULL)
		return(FALSE);

	/* Already cleared? */
	if((icon->pixmap == NULL) && STRISEMPTY(icon->path))
		return(FALSE);

	/* Remove the pixmap, mask, and path */
	GDK_PIXMAP_UNREF(icon->pixmap);
	icon->pixmap = NULL;
	GDK_BITMAP_UNREF(icon->mask);
	icon->mask = NULL;
	icon->width = 0;
	icon->height = 0;
	g_free(icon->path);
	icon->path = NULL;

	/* Set the has changes flag */
	if(!(flags & ICON_PICKER_HAS_CHANGES))
		ip->flags |= ICON_PICKER_HAS_CHANGES;

	/* Update the displayed icon */
	icon_picker_update_widgets(ip);

	/* Update the tooltip */
	icon_picker_update_tooltip(ip);

	/* Notify about the icon being changed */
	if(ip->changed_cb != NULL)
		ip->changed_cb(
			ip->toplevel,
			ip->changed_data
		);

	/* Notify about the icon being switched whenever an
	 * icon has been changed
	 */
	if(ip->switched_icon_cb != NULL)
		ip->switched_icon_cb(
			ip->toplevel,
			ip->current_icon_num,
			icon,
			ip->switched_icon_data
		);

	return(TRUE);
}


/*
 *	Updates the Icon Picker's tooltip.
 */
static void icon_picker_update_tooltip(IconPicker *ip)
{
	if(ip == NULL)
		return;

	if(!(ip->flags & ICON_PICKER_NO_CHANGE_ICON))
	{
		gchar *msg;
		const IconPickerFlags flags = ip->flags;
		IconPickerIcon *icon = icon_picker_get_current_icon(ip->toplevel);
		if((icon != NULL) ? !STRISEMPTY(icon->path) : FALSE)
		{
			msg = g_strdup_printf(
"%s %ix%i\n\
%s",
				g_basename(icon->path),
				icon->width, icon->height,
				(flags & ICON_PICKER_ACCEPT_DND_DROP) ?
"Click here to select a new icon or drag an image file here to set a new icon." :
"Click here to select a new icon."
			);
		}
		else if(flags & ICON_PICKER_ACCEPT_DND_DROP)
		{
			msg = STRDUP(
"Click here to select a new icon or drag an image file here to set a new icon."
			);
		}
		else
		{
			msg = STRDUP(
"Click here to select a new icon."
			);
		}
		GUISetWidgetTip(ip->display_da, msg);
		g_free(msg);
	}
}


/*
 *	Creates a new Icon Picker.
 */
GtkWidget *icon_picker_new(
	const gchar *title,
	const IconPickerFlags flags,
	const IconPickerIconPosition icon_position,
	const gint req_icon_width, const gint req_icon_height,
	const gint display_width, const gint display_height,
	const gint label_width,
	GtkWidget *icon_sel_dlg
)
{
	GtkWidget	*w,
					*menu,
					*parent, *parent2, *parent3;
	IconPicker *ip;

	/* Create the Icon Picker */
	ip = ICON_PICKER(g_malloc0(sizeof(IconPicker)));
	if(ip == NULL)
		return(NULL);

	ip->freeze_count = 0;
	ip->gc = NULL;
	ip->flags = flags;
	ip->icon_position = icon_position;
	ip->req_icon_width = req_icon_width;
	ip->req_icon_height = req_icon_height;
	ip->display_pixmap = NULL;
	ip->icon_sel_dlg = IconSelDlgRef(icon_sel_dlg);

	ip->freeze_count++;

	/* Toplevel GtkVBox */
	ip->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_object_set_data_full(
		GTK_OBJECT(w), ICON_PICKER_KEY,
		ip, icon_picker_destroy_cb
	);
	parent = w;

	/* Title Label GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Title GtkLabel */
	ip->title_label = w = gtk_label_new(
		(title != NULL) ? title : ""
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(title != NULL)
		gtk_widget_show(w);


	/* Display GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Display GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Display GtkDrawingArea */
	ip->display_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, display_width, display_height);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(icon_picker_display_realize_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(icon_picker_display_event_cb), ip
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	if(flags & ICON_PICKER_ACCEPT_DND_DROP)
	{
		const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	0},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	1},
{GUI_TARGET_NAME_STRING,	0,	2}
		};
		gtk_signal_connect_after(
			GTK_OBJECT(w), "drag_leave",
			GTK_SIGNAL_FUNC(icon_picker_display_drag_leave_cb), ip
		);
		gtk_signal_connect_after(
			GTK_OBJECT(w), "drag_motion",
			GTK_SIGNAL_FUNC(icon_picker_display_drag_motion_cb), ip
		);
		GUIDNDSetTar(
			w,
			dnd_tar_types,
			sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY,		/* Actions */
			GDK_ACTION_COPY,		/* Default action if same */
			GDK_ACTION_COPY,		/* Default action */
			icon_picker_display_drag_data_received_cb, ip,
			FALSE				/* Do not highlight */
		);
	}
	if(flags & ICON_PICKER_PROVIDE_DND_DRAG)
	{
		const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	0},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	1},
{GUI_TARGET_NAME_STRING,	0,	2}
		};
		GUIDNDSetSrc(
			w,
			dnd_src_types,
			sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY,		/* Actions */
			GDK_BUTTON1_MASK,		/* Buttons */
			NULL,
			icon_picker_display_drag_data_get_cb,
			NULL,
			NULL,
			ip
		);
	}
	gtk_widget_show(w);


	/* Previous & Next Buttons and Label GtkHBox */
	ip->switch_buttons_toplevel = w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
/*	gtk_widget_show(w); */
	parent2 = w;

	/* Previous & Next Buttons and Label GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	if(label_width <= 0)
		gtk_widget_set_usize(w, display_width, -1);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Left GtkButton */
	ip->left_btn = w = GUIButtonArrow(
		GTK_ARROW_LEFT,
		ICON_PICKER_ARROW_BUTTON_WIDTH, ICON_PICKER_ARROW_BUTTON_HEIGHT
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(icon_picker_previous_cb), ip
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Icon Label GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	if(label_width > 0)
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	else
		gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icon Label GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icon Label GtkDrawingArea */
	ip->icon_label_da = w = gtk_drawing_area_new();
	if(label_width > 0)
		gtk_widget_set_usize(w, label_width, -1);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(icon_picker_icon_label_realize_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(icon_picker_icon_label_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(icon_picker_icon_label_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(icon_picker_icon_label_event_cb), ip
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(icon_picker_icon_label_event_cb), ip
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);

	/* Right GtkButton */
	ip->right_btn = w = GUIButtonArrow(
		GTK_ARROW_RIGHT,
		ICON_PICKER_ARROW_BUTTON_WIDTH, ICON_PICKER_ARROW_BUTTON_HEIGHT
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(icon_picker_next_cb), ip
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Change Icon GtkButton */
	ip->change_btn = w = gtk_button_new_with_label("Change");
	gtk_widget_set_usize(
		w,
		ICON_PICKER_BUTTON_WIDTH, ICON_PICKER_BUTTON_HEIGHT
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(icon_picker_change_cb), ip
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(!(flags & ICON_PICKER_NO_CHANGE_ICON))
		gtk_widget_show(w);

	/* Clear Icon GtkButton */
	ip->clear_btn = w = gtk_button_new_with_label("Clear");
	gtk_widget_set_usize(
		w,
		ICON_PICKER_BUTTON_WIDTH, ICON_PICKER_BUTTON_HEIGHT
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(icon_picker_clear_cb), ip
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(flags & ICON_PICKER_CAN_CLEAR_ICON)
		gtk_widget_show(w);

	/* Right click GtkMenu */
	ip->menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
		guint accel_key, accel_mods;
		gpointer data = ip;
		guint8 **icon;
		const gchar *label;
		GtkAccelGroup *accelgrp = NULL;
		void (*func_cb)(GtkWidget *w, gpointer) = NULL;

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		/* Next */
		icon = NULL;
		label = "Next";
		accel_key = 0;
		accel_mods = 0;
		func_cb = icon_picker_next_cb;
		ADD_MENU_ITEM_LABEL
		ip->next_mi = w;

		/* Previous */
		icon = NULL;
		label = "Previous";
		accel_key = 0;
		accel_mods = 0;
		func_cb = icon_picker_previous_cb;
		ADD_MENU_ITEM_LABEL
		ip->prev_mi = w;

		if(!(flags & ICON_PICKER_NO_CHANGE_ICON) ||
		   (flags & ICON_PICKER_CAN_CLEAR_ICON)
		)
		{
			ADD_MENU_ITEM_SEPARATOR
		}

		if(!(flags & ICON_PICKER_NO_CHANGE_ICON))
		{
			/* Change */
			icon = NULL;
			label = "Change...";
			accel_key = 0;
			accel_mods = 0;
			func_cb = icon_picker_change_cb;
			ADD_MENU_ITEM_LABEL
			ip->change_mi = w;
		}

		if(flags & ICON_PICKER_CAN_CLEAR_ICON)
		{
			/* Clear */
			icon = NULL;
			label = "Clear";
			accel_key = 0;
			accel_mods = 0;
			func_cb = icon_picker_clear_cb;
			ADD_MENU_ITEM_LABEL
			ip->clear_mi = w;
		}

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}

	icon_picker_update_widgets(ip);

	ip->freeze_count--;

	return(ip->toplevel);
}

/*
 *	Sets the has changes flag.
 */
void icon_picker_set_has_changes(
	GtkWidget *w,
	const gboolean has_changes
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	if(has_changes)
		ip->flags |= ICON_PICKER_HAS_CHANGES;
	else
		ip->flags &= ~ICON_PICKER_HAS_CHANGES;
}

/*
 *	Sets the switched icon callback.
 */
void icon_picker_set_switched_icon_cb(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *w,
			const gint,
			IconPickerIcon *,
			gpointer
	),
	gpointer data 
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	ip->switched_icon_cb = cb;
	ip->switched_icon_data = data;
}

/*
 *	Sets the changed callback.
 */
void icon_picker_set_changed_cb(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *w,
			gpointer
	),
	gpointer data 
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	ip->changed_cb = cb;
	ip->changed_data = data;
}

/*
 *	Sets the icon.
 *
 *	The i specifies the icon's index.
 *
 *	The icon specifies the icon who's values will be used to
 *	set the icon's values. The specified icon will not be modified
 *	or deleted.
 */
void icon_picker_set_icon(
	GtkWidget *w,
	const gint i,
	IconPickerIcon *icon
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	IconPickerIcon *tar_icon = icon_picker_get_icon(ip->toplevel, i);
	if(tar_icon == NULL)
		return;

	g_free(tar_icon->label);
	tar_icon->label = STRDUP(icon->label);

	g_free(tar_icon->path);
	tar_icon->path = STRDUP(icon->path);

	(void)GDK_PIXMAP_UNREF(tar_icon->pixmap);
	tar_icon->pixmap = GDK_PIXMAP_REF(icon->pixmap);

	(void)GDK_BITMAP_UNREF(tar_icon->mask);
	tar_icon->mask = GDK_BITMAP_REF(icon->mask);

	/* Update the displayed icon if this is the current icon */
	if(i == ip->current_icon_num)
	{
		icon_picker_update_widgets(ip);

		/* Notify about the icon being switched whenever an icon
		 * has been set
		 */
		if(ip->switched_icon_cb != NULL)
			ip->switched_icon_cb(
				ip->toplevel,
				i,
				tar_icon,
				ip->switched_icon_data
			);
	}
}

/*
 *	Sets the Icon Picker's icons list.
 */
void icon_picker_set_icons_list(
	GtkWidget *w,
	GList *icons_list                       /* Transfered */
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	/* Delete the existing icons list */
	icon_picker_clear(ip->toplevel);

	/* Set the new icons list */
	ip->icons_list = icons_list;

	/* Update the total number of icons */
	ip->nicons = g_list_length(icons_list);

	/* Show or hide the switch icon buttons depending on the number
	 * of icons in the list
	 */
	if(ip->nicons > 1)
		gtk_widget_show(ip->switch_buttons_toplevel);
	else
		gtk_widget_hide(ip->switch_buttons_toplevel);
	w = gtk_widget_get_toplevel(ip->toplevel);
	if(w != NULL)
		gtk_widget_queue_resize(w);

	/* Set the current icon, update the displayed icon, widgets
	 * and tooltip, and call the switched icon callback
	 */
	icon_picker_switch_to(ip->toplevel, 0);
}

/*
 *	Gets the Icon.
 *
 *	The i specifies the icon's index.
 */
IconPickerIcon *icon_picker_get_icon(
	GtkWidget *w,
	const gint i
)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return(NULL);

	return(ICON_PICKER_ICON(g_list_nth_data(
		ip->icons_list,
		(guint)i
	)));
}

/*
 *	Gets the current Icon.
 */
IconPickerIcon *icon_picker_get_current_icon(GtkWidget *w)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return(NULL);

	return(icon_picker_get_icon(
		ip->toplevel,
		ip->current_icon_num
	));
}

/*
 *	Gets the Icon Picker's icons list.
 */
GList *icon_picker_get_icons_list(GtkWidget *w)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return(NULL);

	return(ip->icons_list);
}

/*
 *	Deletes the Icons List.
 */
void icon_picker_clear(GtkWidget *w)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	if(ip->icons_list != NULL)
	{
		g_list_foreach(
			ip->icons_list,
			(GFunc)icon_picker_icon_delete,
			NULL
		);
		g_list_free(ip->icons_list);
		ip->icons_list = NULL;
	}

	ip->nicons = 0;

	gtk_widget_hide(ip->switch_buttons_toplevel);
	w = gtk_widget_get_toplevel(ip->toplevel);
	if(w != NULL)
		gtk_widget_queue_resize(w);

	/* Set the current icon, update the displayed icon, widgets
	 * and tooltip, and call the switched icon callback
	 */
	icon_picker_switch_to(ip->toplevel, 0);
}

/*
 *	Queues the Icon Picker to draw.
 */
void icon_picker_queue_draw(GtkWidget *w)
{
	IconPicker *ip = ICON_PICKER(GTK_OBJECT_GET_DATA(
		w,
		ICON_PICKER_KEY
	));
	if(ip == NULL)
		return;

	gtk_widget_queue_draw(ip->display_da);
	gtk_widget_queue_draw(ip->icon_label_da);
}

/*
 *	Updates the widgets.
 */
static void icon_picker_update_widgets(IconPicker *ip)
{
	gboolean sensitive;
	gint		current_icon_num,
			nicons;
	IconPickerIcon *icon;

	if(ip == NULL)
		return;

	ip->freeze_count++;

	icon_picker_queue_draw(ip->toplevel);

	current_icon_num = ip->current_icon_num;
	nicons = ip->nicons;
	icon = icon_picker_get_current_icon(ip->toplevel);

	sensitive = ((nicons > 1) && (current_icon_num > 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(ip->left_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(ip->prev_mi, sensitive);

	sensitive = ((nicons > 1) && (current_icon_num < (nicons - 1))) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(ip->right_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(ip->next_mi, sensitive);

	if(icon != NULL)
	{
		sensitive = ((icon->path != NULL) || (icon->pixmap != NULL)) ? TRUE : FALSE;
		GTK_WIDGET_SET_SENSITIVE(ip->clear_btn, sensitive);
		GTK_WIDGET_SET_SENSITIVE(ip->clear_mi, sensitive);
	}

	ip->freeze_count--;
}
