#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../guiutils.h"
#include "../fprompt.h"
#include "../cdialog.h"
#include "../fb.h"
#include "../pdialog.h"
#include "../ascii_chart_dlg.h"
#include "../hview.h"

#include "../libendeavour2-base/endeavour2.h"

#include "config.h"


#include "../images/icon_edit_48x48.xpm"
#include "../images/icon_new_20x20.xpm"
#include "../images/icon_open_20x20.xpm"
#include "../images/icon_save_20x20.xpm"
#include "../images/icon_save_as_20x20.xpm"
#include "../images/icon_close_20x20.xpm"
#include "../images/icon_cut_20x20.xpm"
#include "../images/icon_copy_20x20.xpm"
#include "../images/icon_paste_20x20.xpm"
#include "../images/icon_add_20x20.xpm"
#include "../images/icon_cancel_20x20.xpm"
#include "../images/icon_search_20x20.xpm"
#include "../images/icon_calculator_20x20.xpm"
#include "../images/icon_about_20x20.xpm"


/*
 *	Hex Editor:
 */
typedef struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;

	GtkWidget	*main_vbox,
					*menu_bar_handle,
					*cut_mi,
					*copy_mi,
					*paste_mi,
					*delete_mi,
					*edit_mode_hex_mi,
					*edit_mode_ascii_mi,
					*select_all_mi,
					*unselect_all_mi,
					*find_mi,
					*find_next_mi;

	hview_struct	*hv;

	gchar		*path;

	guint8		*last_find_needle;
	gint		last_find_needle_len;
	gboolean	last_find_case_sensitive;

	ASCIIChartDlg	*ac_dlg;

	EDVContext *ctx;

} hedit_struct;
#define HEDIT(p)	((hedit_struct *)(p))


/* Callbacks */
static void hedit_signal_cb(int s);

static void hedit_file_selector_created_cb(
	const gchar *path, gpointer data
);
static void hedit_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
);
static void hedit_file_selector_deleted_cb(
	const gchar *path, gpointer data
);

static void hedit_ascii_chart_dlg_insert_cb(
	ASCIIChartDlg *d, const gchar c, gpointer data
);

static gint hedit_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data);
static gint hedit_key_event_cb(GtkWidget *widget, GdkEventKey *key, gpointer data);

static void hedit_hview_changed_cb(hview_struct *hv, gpointer data);
static void hedit_hview_select_cb(
	hview_struct *hv, int start_pos, int end_pos, gpointer data
);
static void hedit_hview_edit_mode_changed_cb(
	hview_struct *hv, hview_edit_mode edit_mode, gpointer data
);

static void hedit_dnd_drag_received_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);

static void hedit_new_cb(GtkWidget *widget, gpointer data);
static void hedit_open_cb(GtkWidget *widget, gpointer data);
static void hedit_save_cb(GtkWidget *widget, gpointer data);
static void hedit_save_as_cb(GtkWidget *widget, gpointer data);
static void hedit_close_cb(GtkWidget *widget, gpointer data);
static void hedit_cut_cb(GtkWidget *widget, gpointer data);
static void hedit_copy_cb(GtkWidget *widget, gpointer data);
static void hedit_paste_cb(GtkWidget *widget, gpointer data);
static void hedit_insert_cb(GtkWidget *widget, gpointer data);
static void hedit_delete_cb(GtkWidget *widget, gpointer data);
static void hedit_select_all_cb(GtkWidget *widget, gpointer data);
static void hedit_unselect_all_cb(GtkWidget *widget, gpointer data);
static void hedit_find_cb(GtkWidget *widget, gpointer data);
static void hedit_find_next_cb(GtkWidget *widget, gpointer data);
static void hedit_edit_mode_hex_cb(GtkWidget *widget, gpointer data);
static void hedit_edit_moed_ascii_cb(GtkWidget *widget, gpointer data);
static void hedit_calculate_selected_cb(GtkWidget *widget, gpointer data);
static void hedit_ascii_chart_cb(GtkWidget *widget, gpointer data);
static void hedit_about_cb(GtkWidget *widget, gpointer data);

static gint hedit_hview_open_progress_cb(
	hview_struct *hview, gulong i, gulong m, gpointer data
);
static gint hedit_hview_save_progress_cb(
	hview_struct *hview, gulong i, gulong m, gpointer data
);

static void hedit_build_menu_bar(
	hedit_struct *he,
	GtkWidget *parent
);
static void hedit_update_display(hedit_struct *he);


#define HEDIT_WIDTH		640
#define HEDIT_HEIGHT		480


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define DO_CHECK_CHANGES_AND_QUERY_USER(_he_,_hv_) {	\
													\
 HViewSetBusy((_hv_), TRUE);				\
													\
 /* Check if current data has modifications and		\
  * if so, query user to save first			\
  */							\
 if((_hv_)->flags & HVIEW_FLAG_HAS_CHANGES) {		\
  gint response;					\
													\
  edv_play_sound_question((_he_)->ctx);			\
  CDialogSetTransientFor((_he_)->toplevel);		\
  response = CDialogGetResponse(			\
"Save Changes?",					\
"The data contains changes that have not been\n\
saved, do you want to save those changes?", \
   NULL,						\
   CDIALOG_ICON_WARNING,				\
   CDIALOG_BTNFLAG_YES |				\
   CDIALOG_BTNFLAG_NO |					\
   CDIALOG_BTNFLAG_CANCEL,				\
   CDIALOG_BTNFLAG_YES					\
  );							\
  CDialogSetTransientFor(NULL);				\
  switch(response) {					\
   case CDIALOG_RESPONSE_YES_TO_ALL:			\
   case CDIALOG_RESPONSE_YES:				\
    hedit_save_cb(NULL, data);				\
    /* Check if the changes were not saved? */		\
    if((_hv_)->flags & HVIEW_FLAG_HAS_CHANGES) {	\
     HViewSetBusy((_hv_), FALSE);			\
     return;						\
    }							\
    break;						\
													\
   case CDIALOG_RESPONSE_NO:				\
    break;						\
													\
   default:						\
    HViewSetBusy((_hv_), FALSE);			\
    return;						\
    break;						\
  }							\
 }							\
													\
 HViewSetBusy((_hv_), FALSE);				\
}


/*
 *	UNIX signal handler.
 */
static void hedit_signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
		exit(1);
		break;
	}
}


/*
 *      File selector created callback.
 */
static void hedit_file_selector_created_cb(
	const gchar *path, gpointer data
)
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_added(ctx, path);
	edv_context_flush(ctx);
}

/*
 *      File selector changed callback.
 */
static void hedit_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
)
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_modified(ctx, old_path, new_path);
	edv_context_flush(ctx);
}

/*
 *      File selector deleted callback.
 */
static void hedit_file_selector_deleted_cb(
	const gchar *path, gpointer data
)
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_removed(ctx, path);
	edv_context_flush(ctx);
}


/*
 *	ASCII Chart Dialog insert callback.
 */
static void hedit_ascii_chart_dlg_insert_cb(
	ASCIIChartDlg *d, const gchar c, gpointer data
)
{
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if((d == NULL) || (he == NULL))
		return;

	if(he->freeze_count > 0)
		return;

	he->freeze_count++;

	hv = he->hv;

	HViewSetBusy(hv, TRUE);

	/* Selected? */
	if(HVIEW_IS_BUF_SELECTED(hv))
	{
		HViewSetSelected(hv, (guint8)c);
	}
	else
	{
		gint pos;

		/* Insert a new byte */
		HViewInsert(hv);

		/* Set the value of the new byte to the specified character */
		pos = hv->buf_pos;
		if(HVIEW_IS_BUF_POS_VALID(hv, pos))
			hv->buf[pos] = (guint8)c;
	}
	HViewQueueDraw(hv);

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);

	he->freeze_count--;
}


/*
 *	Hex Editor toplevel GtkWindow "delete_event" callback.
 */
static gint hedit_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint status = FALSE;
	hedit_struct *he = HEDIT(data);
	if((widget == NULL) || (he == NULL))
		return(status);

	if(he->freeze_count > 0)
		return(status);

	hedit_close_cb(widget, data);
	status = TRUE;

	return(status);
}

/*
 *	Hex Editor toplevel GtkWindow "key_press_event" or
 *	"key_release_event" signal callback.
 */
static gint hedit_key_event_cb(GtkWidget *widget, GdkEventKey *key, gpointer data)
{
	gint status = FALSE;
	gboolean press;
	hedit_struct *he = HEDIT(data);
	if((widget == NULL) || (key == NULL) || (he == NULL))
		return(status);

	if(he->freeze_count > 0)
		return(status);

	press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;

	switch(key->keyval)
	{
	  case GDK_F3:
		if(press)
			hedit_open_cb(widget, data);
		status = TRUE;
		break;

	}

	return(status);
}

/*
 *	Hex View changed callback.
 */
static void hedit_hview_changed_cb(hview_struct *hv, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	hedit_update_display(he);
}

/*
 *	Hex View select callback.
 */
static void hedit_hview_select_cb(
	hview_struct *hv, int start_pos, int end_pos, gpointer data
)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	hedit_update_display(he);
}

/*
 *	Hex View edit mode changed callback.
 */
static void hedit_hview_edit_mode_changed_cb(
	hview_struct *hv, hview_edit_mode edit_mode, gpointer data  
)
{
	hedit_struct *he = HEDIT(data); 
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	hedit_update_display(he);
}

/*
 *	Hex View "drag_data_received" callback.
 *
 *	Opens the object specified by the url in the drag data.
 */
static void hedit_dnd_drag_received_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	GtkWidget *w, *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if((widget == NULL) || (dc == NULL) || (he == NULL))
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	DO_CHECK_CHANGES_AND_QUERY_USER(he, hv);

	HViewSetBusy(hv, TRUE);

	/* Get hv's view drawing area widget */
	w = hv->view_da;

	/* View widget the same as the callback widget? */
	if(w == widget)
	{
		/* Check if the DND target type is one that we support */
		if((info == HEDIT_DND_TYPE_INFO_TEXT_PLAIN) ||
		   (info == HEDIT_DND_TYPE_INFO_TEXT_URI_LIST) ||
		   (info == HEDIT_DND_TYPE_INFO_STRING)
		)
		{
			const gchar	*s,
							*url = (const gchar *)selection_data->data;
			gchar *path;


			/* The DND data should be a URL string, we need to parse
			 * it and get the absolute path portion of that URL
			 * string
			 */
			s = (const gchar *)strstr((const char *)url, "://");
			if(s != NULL)
				s += STRLEN("://");
			else
				s = url;

			s = (const gchar *)strchr((const char *)s, '/');
			path = STRDUP(s);

			/* Got absolute path to object to be loaded? */
			if(path != NULL)
			{
				if(path != he->path)
				{
					g_free(he->path);
					he->path = STRDUP(path);
				}
				HViewOpenFileProgress(
					hv,
					he->path,
					hedit_hview_open_progress_cb, he
				);

				g_free(path);
			}
		}
	}

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	New callback.
 */
static void hedit_new_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	DO_CHECK_CHANGES_AND_QUERY_USER(he, hv);

	HViewSetBusy(hv, TRUE);

	HViewClear(hv);

	hv->flags &= ~HVIEW_FLAG_HAS_CHANGES;

	g_free(he->path);
	he->path = NULL;

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	Open callback.
 */
static void hedit_open_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if((he == NULL) || FileBrowserIsQuery())
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	DO_CHECK_CHANGES_AND_QUERY_USER(he, hv);

	HViewSetBusy(hv, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for the file to open */
	he->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Open File",
		"Open", "Cancel",
		he->path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	he->freeze_count--;

	/* Got user response? */
	if(response)
	{
		gchar *path = (npaths > 0) ? STRDUP(paths_list[npaths - 1]) : NULL;
		if(path != NULL)
		{
			struct stat stat_buf;
			gboolean file_exists;

			/* Check if the file exists */
			if(stat((const char *)path, &stat_buf))
			{
				const gint error_code = (gint)errno;
				if(error_code == ENOENT)
					file_exists = FALSE;
				else
					file_exists = TRUE;
			}
			else
			{
				file_exists = TRUE;
			}
			/* Does the file exist? */
			if(file_exists)
			{
				/* File exists, open it */
				if(path != he->path)
				{
					g_free(he->path);
					he->path = g_strdup(path);
				}
				HViewOpenFileProgress(
					hv,
					he->path,
					hedit_hview_open_progress_cb, he
				);
			}
			else
			{
				/* File does not exist, prompt the user to create
				 * a new file
				 */
				gint response;
				gchar *msg = g_strdup_printf(
"The file \"%s\" does not exist,\n\
do you want to create it?",
					g_basename(path)
				);
				edv_play_sound_question(he->ctx);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Confirm Create",
					msg,
					NULL,
					CDIALOG_ICON_QUESTION,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_NO
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_YES)
				{
					/* Create the new file */
					if(edv_touch(
						path,
						(gulong)-1,		/* Current time */
						TRUE		/* Create */
					) == 0)
					{
						/* Set the new file name */
						if(path != he->path)
						{
							g_free(he->path);
							he->path = g_strdup(path);
						}

						/* Notify about the file being created */
						edv_notify_queue_vfs_object_added(
							he->ctx,
							he->path
						);
						edv_context_sync(he->ctx);

						/* Open the new file */
						HViewOpenFileProgress(
							hv,
							he->path,
							hedit_hview_open_progress_cb, he
						);
					}
					else
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
							g_strerror(error_code),
							path
						);
						edv_play_sound_error(he->ctx);
						CDialogSetTransientFor(toplevel);
						CDialogGetResponse(
							"Create Failed",
							msg,
							NULL,
							CDIALOG_ICON_ERROR,
							CDIALOG_BTNFLAG_OK,
							CDIALOG_BTNFLAG_OK
						);
						g_free(msg);
						CDialogSetTransientFor(NULL);
					}
				}
			}	/* Does the file exist? */

			g_free(path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	Save callback.
 */
static void hedit_save_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	HViewSetBusy(hv, TRUE);

	/* If there was no path to save to then call the save as
	 * callback so user can be prompted for a file name
	 */
	if(he->path == NULL)
	{
		hedit_save_as_cb(widget, data);
		HViewSetBusy(hv, FALSE);
		return;
	}

	/* Save the data to file */
	if(HViewSaveFileProgress(
		hv, he->path,
		hedit_hview_save_progress_cb, he
	))
	{
		/* Error occured while saving the file */
		gchar *msg = g_strdup_printf(
"An error occured while saving the file:\n\
\n\
    %s",
			he->path
		);
		edv_play_sound_error(he->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
			"Save Error",
			msg,
			NULL,
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
	}
	else
	{
		/* File was saved successfully, notify about the file
		 * being saved
		 */
		edv_notify_queue_vfs_object_modified(
			he->ctx, he->path, NULL
		);
		edv_context_sync(he->ctx);

		hv->flags &= ~HVIEW_FLAG_HAS_CHANGES;
	}

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	Save as callback.
 */
static void hedit_save_as_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if((he == NULL) || FileBrowserIsQuery())
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	HViewSetBusy(hv, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for the file to save */
	he->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Save File As",
		"Save", "Cancel",
		he->path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	he->freeze_count--;

	/* Got user response? */
	if(response)
	{
		gchar *new_path = (npaths > 0) ? STRDUP(paths_list[npaths - 1]) : NULL;
		if(new_path != NULL)
		{
			struct stat stat_buf;
			gboolean	file_exists,
							allow_save = TRUE;

			/* Check if the file exists */
			if(stat((const char *)new_path, &stat_buf))
			{
				const gint error_code = (gint)errno;
				if(error_code == ENOENT)
					file_exists = FALSE;
				else
					file_exists = TRUE;
			}
			else
			{
				file_exists = TRUE;
			}
			if(file_exists)
			{
				gint response;
				gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
					new_path
				);
				edv_play_sound_warning(he->ctx);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Confirm Overwrite",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_YES
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
				switch(response)
				{
				  case CDIALOG_RESPONSE_YES_TO_ALL:
				  case CDIALOG_RESPONSE_YES:
					break;
				  default:
					allow_save = FALSE;
					break;
				}
			}

			/* File not exist or user permits ovewrite? */
			if(allow_save)
			{
				/* Update the path */
				if(new_path != he->path)
				{
					g_free(he->path);
					he->path = g_strdup(new_path);
				}

				/* Save the data to the new file name */
				if(HViewSaveFileProgress(
					hv,
					he->path,
					hedit_hview_save_progress_cb, he
				))
				{
					/* Error occured while saving file */
					gchar *msg = g_strdup_printf(
"An error occured while saving the file:\n\
\n\
    %s",
						he->path
					);
					edv_play_sound_error(he->ctx);
					CDialogSetTransientFor(toplevel);
					CDialogGetResponse(
						"Save Error",
						msg,
						NULL,
						CDIALOG_ICON_ERROR,
						CDIALOG_BTNFLAG_OK,
						CDIALOG_BTNFLAG_OK
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
				}
				else
				{
					/* File was saved successfully, notify about
					 * the file being saved
					 */
					if(file_exists)
						edv_notify_queue_vfs_object_modified(
							he->ctx,
							he->path,
							NULL
						);
					else
						edv_notify_queue_vfs_object_added(
							he->ctx,
							he->path
						);
					edv_context_sync(he->ctx);

					hv->flags &= ~HVIEW_FLAG_HAS_CHANGES;
				}
			}

			g_free(new_path);
		}
	}	/* Got user response? */

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset the file selector due to possible file related change */
	FileBrowserReset();

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	Close callback.
 */
static void hedit_close_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	DO_CHECK_CHANGES_AND_QUERY_USER(he, hv);

	gtk_main_quit();
}

/*
 *	Cut callback.
 */
static void hedit_cut_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewCut(he->hv);
}

/*
 *      Copy callback.
 */
static void hedit_copy_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewCopy(he->hv);
}

/*
 *      Paste callback.
 */
static void hedit_paste_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewPaste(he->hv);
}

/*
 *      Insert callback.
 */
static void hedit_insert_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewInsert(he->hv);
}

/*
 *      Delete callback.
 */
static void hedit_delete_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewDeleteSelected(he->hv);
}

/*
 *	Select All callback.
 */
static void hedit_select_all_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewSelectAll(he->hv);
}

/*
 *	Unselect All callback.
 */
static void hedit_unselect_all_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	HViewUnselectAll(he->hv);
}

/*
 *	Find callback.
 */
static void hedit_find_cb(GtkWidget *widget, gpointer data)
{
	gchar **strv, *last_needle = NULL;
	gint strc;
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if((he == NULL) || PDialogIsQuery())
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	HViewSetBusy(hv, TRUE);

	PDialogDeleteAllPrompts();
	if((he->last_find_needle != NULL) &&
	   (he->last_find_needle_len > 0)
	)
	{
		last_needle = (gchar *)g_malloc(
			(he->last_find_needle_len + 1) * sizeof(gchar)
		);
		memcpy(
			last_needle, he->last_find_needle, 
			he->last_find_needle_len * sizeof(gchar)
		);
		last_needle[he->last_find_needle_len] = '\0';
	}
 	PDialogAddPrompt(NULL, "Find:", last_needle);
	g_free(last_needle);
	last_needle = NULL;
	PDialogAddPromptToggle(NULL, "Case Sensitive", he->last_find_case_sensitive);
	PDialogSetSize(320, -1);
	PDialogSetTransientFor(toplevel);
	strv = PDialogGetResponse(
		"Find", NULL, NULL,
		PDIALOG_ICON_SEARCH,
		"Find", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	);
	PDialogSetTransientFor(NULL);

	if((strv != NULL) && (strc >= 2))
	{
		gint i;
		const gchar *new_needle = strv[0];
		const gboolean case_sensitive = ATOI(strv[1]) ? TRUE : FALSE;

		if(!STRISEMPTY(new_needle))
		{
			/* Need to convert the user given string from ASCII to
			 * hex and update the last find needle
			 */
			g_free(he->last_find_needle);
			he->last_find_needle = (guint8 *)STRDUP(new_needle);
			he->last_find_needle_len = STRLEN(new_needle);
			he->last_find_case_sensitive = case_sensitive;

			/* Do search */
			i = HViewFind(
				hv,
				he->last_find_needle, he->last_find_needle_len,
				hv->buf_pos + 1,	/* Start index */
				he->last_find_case_sensitive
			);
			/* Got match? */
			if(i > -1)
			{
				/* Scroll to position */
				HViewSetPosition(hv, i);
				HViewSelect(hv, i, i + he->last_find_needle_len - 1);
				if(HViewIsBufferPositionVisible(hv, i) !=
					GTK_VISIBILITY_FULL
				)
					HViewScrollTo(hv, i, 0.5f);
			}
			else
			{
				/* No match, then start search again from the
				 * beginning.
				 */
				i = HViewFind(
					hv,
					he->last_find_needle, he->last_find_needle_len,
					0,		/* Start index */
					he->last_find_case_sensitive
				);
				/* Got match the second time? */
				if(i > -1)
				{
					/* Scroll to position */
					HViewSetPosition(hv, i);
					HViewSelect(hv, i, i + he->last_find_needle_len - 1);
					if(HViewIsBufferPositionVisible(hv, i) !=
						GTK_VISIBILITY_FULL
					)
						HViewScrollTo(hv, i, 0.5f);
				}
				else
				{
					/* No match */
					edv_play_sound_info(he->ctx);
					CDialogSetTransientFor(toplevel);
					CDialogGetResponse(
						"Find",
"The string that you entered was not found.",
						NULL,
						CDIALOG_ICON_INFO,
						CDIALOG_BTNFLAG_OK,
						CDIALOG_BTNFLAG_OK
					);
					CDialogSetTransientFor(NULL);
				}	/* Got match the second time? */
			}	/* Got match? */
		}
	}

	PDialogDeleteAllPrompts();

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *      Find next callback.
 */
static void hedit_find_next_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	if(he->last_find_needle == NULL)
		return;

	HViewSetBusy(hv, TRUE);

	if(TRUE)
	{
		/* Do search */
		gint i = HViewFind(
			hv,
			he->last_find_needle, he->last_find_needle_len,
			hv->buf_pos + 1,	/* Start index */
			he->last_find_case_sensitive
		);
		/* Got match? */
		if(i > -1)
		{
			/* Scroll to position */
			HViewSetPosition(hv, i);
			HViewSelect(hv, i, i + he->last_find_needle_len - 1);
			if(HViewIsBufferPositionVisible(hv, i) !=
				GTK_VISIBILITY_FULL
			)
				HViewScrollTo(hv, i, 0.5f);
		}
		else
		{
			/* No match, then start search again from the
			 * beginning.
			 */
			i = HViewFind(
				hv,
				he->last_find_needle, he->last_find_needle_len,
				0,		/* Start index */
				he->last_find_case_sensitive
			);
			/* Got match the second time? */
			if(i > -1)
			{
				/* Scroll to position */
				HViewSetPosition(hv, i);
				HViewSelect(hv, i, i + he->last_find_needle_len - 1);
				if(HViewIsBufferPositionVisible(hv, i) !=
					GTK_VISIBILITY_FULL
				)
					HViewScrollTo(hv, i, 0.5f);
			}
			else
			{
				/* No match */
				edv_play_sound_info(he->ctx);
				CDialogSetTransientFor(toplevel);
				CDialogGetResponse(
					"Find",
"The string that you entered was not found.",
					NULL,
					CDIALOG_ICON_INFO,
					CDIALOG_BTNFLAG_OK,
					CDIALOG_BTNFLAG_OK
				);
				CDialogSetTransientFor(NULL);
			}	/* Got match the second time? */
		}		/* Got match? */
	}

	hedit_update_display(he);
	HViewSetBusy(hv, FALSE);
}

/*
 *	Edit mode hex callback.
 */
static void hedit_edit_mode_hex_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	HViewSetEditMode(hv, HVIEW_EDIT_MODE_HEX);
	hedit_update_display(he);
}

/*
 *	Edit mode ASCII callback.
 */
static void hedit_edit_moed_ascii_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->freeze_count > 0)
		return;

	toplevel = he->toplevel;
	hv = he->hv;

	HViewSetEditMode(hv, HVIEW_EDIT_MODE_ASCII);
	hedit_update_display(he);
}

/*
 *	Calculate selected callback.
 */
static void hedit_calculate_selected_cb(GtkWidget *widget, gpointer data)
{
	gint pos;
	guint8 i8 = 0, i8s = 0;
	guint16 i16 = 0, i16s = 0;
	guint32 i24 = 0, i24s = 0;
	guint32 i32 = 0, i32s = 0;
	guint64 i64 = 0, i64s = 0;
	const guint8 *buf;
	GtkWidget *toplevel;
	hview_struct *hv;
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	toplevel = he->toplevel;
	hv = he->hv;
	if(hv == NULL)
		return;

	buf = hv->buf;
	if(buf == NULL)
		return;

	if(CDialogIsQuery())
		return;

	/* Check if nothing is selected */
	if(HVIEW_IS_BUF_SELECTED(hv))
	{
		gint	i = hv->buf_sel_start,
					n = hv->buf_sel_end;
		pos = i;
		if((i >= 0) && (i < hv->buf_len) &&
		   (n >= 0) && (n < hv->buf_len)
		)
		{
			gint len;

			if(i > n)
			{
				gint t = i;
				i = n;
				n = t;
			}

			len = n - i + 1;

			if(len >= sizeof(guint8))
				i8 = *(guint8 *)(&buf[i]);
			if(len >= sizeof(guint16))
				i16 = *(guint16 *)(&buf[i]);
			if(len >= sizeof(guint32))
				i32 = *(guint32 *)(&buf[i]);
			if(len >= sizeof(guint64))
				i64 = *(guint64 *)(&buf[i]);

			i24 = (i32 & 0x00ffffff);
		}
	}
	else
	{
		gint i = hv->buf_pos;
		pos = i;
		if((i >= 0) && (i < hv->buf_len))
		{
			if((i + sizeof(guint8)) <= hv->buf_len)
				i8 = *(guint8 *)(&buf[i]);
			if((i + sizeof(guint16)) <= hv->buf_len)
				i16 = *(guint16 *)(&buf[i]);
			if((i + sizeof(guint32)) <= hv->buf_len)
				i32 = *(guint32 *)(&buf[i]);
			if((i + sizeof(guint64)) <= hv->buf_len)
				i64 = *(guint64 *)(&buf[i]);

			i24 = (i32 & 0x00ffffff);
		}
	}

	/* Calculate swapped values */
	if(TRUE)
	{
		i8s = i8;

		i16s |= ((i16 & 0x00ff) << 8);
		i16s |= ((i16 & 0xff00) >> 8);

		i24s |= ((i24 & 0x000000ff) << 16);
		i24s |= ((i24 & 0x0000ff00) >> 0);
		i24s |= ((i24 & 0x00ff0000) >> 16);

		i32s |= ((i32 & 0x000000ff) << 24);
		i32s |= ((i32 & 0x0000ff00) << 8);
		i32s |= ((i32 & 0x00ff0000) >> 8);
		i32s |= ((i32 & 0xff000000) >> 24);

		i64s |= ((i64 & 0x00000000000000ffLL) << 56);
		i64s |= ((i64 & 0x000000000000ff00LL) << 40);
		i64s |= ((i64 & 0x0000000000ff0000LL) << 24);
		i64s |= ((i64 & 0x00000000ff000000LL) << 8);
		i64s |= ((i64 & 0x000000ff00000000LL) >> 8);
		i64s |= ((i64 & 0x0000ff0000000000LL) >> 24);
		i64s |= ((i64 & 0x00ff000000000000LL) >> 40);
		i64s |= ((i64 & 0xff00000000000000LL) >> 56);
	}


	if(TRUE)
	{
		gchar *msg = g_strdup_printf(
"Position: %i\n\
8 Bit Value: %i  Unsigned: %i  0x%.2X\n\
16 Bit Value: %i  Unsigned: %i  0x%.4X\n\
24 Bit Value: %i  Unsigned: %i  0x%.6X\n\
32 Bit Value: %i  Unsigned: %i  0x%.8X\n\
64 Bit Value: %ld  Unsigned: %ld  0x%.16LX\n\
\n\
Swapped Values:\n\
8 Bit Value: %i  Unsigned: %i  0x%.2X\n\
16 Bit Value: %i  Unsigned: %i  0x%.4X\n\
24 Bit Value: %i  Unsigned: %i  0x%.6X\n\
32 Bit Value: %i  Unsigned: %i  0x%.8X\n\
64 Bit Value: %ld  Unsigned: %ld  0x%.16LX",
			pos,
			(gint8)((gint32)i8 - (0xff / 2) - 1), i8, i8,
			(gint16)((gint32)i16 - (0xffff / 2) - 1), i16, i16,
			(gint32)((gint64)i24 - (0xffffff / 2) - 1), i24, i24,
			(gint32)((gint64)i32 - (0xffffffff / 2) - 1), i32, i32,
			(unsigned long)((gint64)i64 - (0xffffffffffffffffLL / 2) - 1),
			(unsigned long)i64,
			(long long unsigned int)i64,
			(gint8)((gint32)i8s - (0xff / 2) - 1), i8s, i8s,
			(gint16)((gint32)i16s - (0xffff / 2) - 1), i16s, i16s,
			(gint32)((gint64)i24s - (0xffffff / 2) - 1), i24s, i24s,
			(gint32)((gint64)i32s - (0xffffffff / 2) - 1), i32s, i32s,
			(unsigned long)((gint64)i64s - (0xffffffffffffffffLL / 2) - 1),
			(unsigned long)i64s,
			(long long unsigned int)i64s
		);
		edv_play_sound_info(he->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
			"Results",
			msg,
			NULL,
			CDIALOG_ICON_INFO,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
	}


	hedit_update_display(he);
}

/*
 *	ASCII Chart callback.
 */
static void hedit_ascii_chart_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	if(he->ac_dlg == NULL)
	{
		ASCIIChartDlg *d;
		he->ac_dlg = d = ascii_chart_dlg_new(he->toplevel);
		ascii_chart_dlg_set_insert_cb(
			d,
			hedit_ascii_chart_dlg_insert_cb,
			he
		);
	}

	ascii_chart_dlg_map(he->ac_dlg);
}

/*
 *	About callback.
 */
static void hedit_about_cb(GtkWidget *widget, gpointer data)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)
		return;

	edv_window_about_dialog_map(he->ctx, NULL);
	edv_context_sync(he->ctx);
}


/*
 *	HView open file progress callback.
 */
static gint hedit_hview_open_progress_cb(
	hview_struct *hview, gulong i, gulong m, gpointer data
)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)                 
		return(1);


	return(0);
}

/*
 *	HView save file progress callback.
 */
static gint hedit_hview_save_progress_cb(
	hview_struct *hview, gulong i, gulong m, gpointer data
)
{
	hedit_struct *he = HEDIT(data);
	if(he == NULL)                 
		return(1);


	return(0);
}


/*
 *	Builds the menu bar.
 */
static void hedit_build_menu_bar(hedit_struct *he, GtkWidget *parent)
{
	guint8 **icon;
	const gchar *label;
	guint accel_key, accel_mods;
	GtkAccelGroup *accelgrp;
	GtkWidget *menu_bar, *menu, *w;
	gpointer mclient_data = he;
	void (*func_cb)(GtkWidget *w, gpointer);

	if((he == NULL) || (parent == NULL))
		return;

	/* Get the keyboard accelerator group */
	accelgrp = he->accelgrp;

	/* Create the menu bar */
	menu_bar = GUIMenuBarCreate(NULL);
	if(GTK_IS_BOX(parent))
		gtk_box_pack_start(GTK_BOX(parent), menu_bar, FALSE, FALSE, 0);
	else
		gtk_container_add(GTK_CONTAINER(parent), menu_bar);
	gtk_widget_show(menu_bar);

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, mclient_data				\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, mclient_data				\
 );						\
}
#define ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	/* Create each menu */

	/* File menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		icon = (guint8 **)icon_new_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Nuevo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Nouveau"
#elif defined(PROG_LANGUAGE_GERMAN)
"Neu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Nuovo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Nieuw"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Novo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ny"
#else
"New"
#endif
		;
		accel_key = GDK_n;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_new_cb;
		ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_open_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Abierto..."
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir..."
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen..."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto..."
#elif defined(PROG_LANGUAGE_DUTCH)
"Open..."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto..."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen..."
#else
"Open..."
#endif
		;
		accel_key = GDK_o;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_open_cb;
		ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_save_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Salve"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister"
#elif defined(PROG_LANGUAGE_GERMAN)
"Sparen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Red"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt"
#else
"Save"
#endif
		;
		accel_key = GDK_s;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_save_cb;
		ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_save_as_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Salve Como..."
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister sous..."
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer Als..."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare Come..."
#elif defined(PROG_LANGUAGE_DUTCH)
"Behalve Als..."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe Como..."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt As..."
#else
"Save As..."
#endif
		;
		accel_key = GDK_a;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_save_as_cb;
		ADD_MENU_ITEM_LABEL

		ADD_MENU_SEP

		icon = (guint8 **)icon_close_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
			"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
			"Proche"
#elif defined(PROG_LANGUAGE_POLISH)
			"Zamknij"
#else
			"Close"
#endif
		;
		accel_key = GDK_w;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_close_cb;
		ADD_MENU_ITEM_LABEL

	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#else
"File"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Edit menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		icon = (guint8 **)icon_cut_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"El Corte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Couper"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schnitt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Taglio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Snee"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Corte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Snitt"
#else
"Cut"
#endif
		;
		accel_key = GDK_x;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_cut_cb;
		ADD_MENU_ITEM_LABEL
		he->cut_mi = w;

		icon = (guint8 **)icon_copy_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Copia"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Copia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Cpia"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopi"
#else
"Copy"
#endif
		;
		accel_key = GDK_c;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_copy_cb;
		ADD_MENU_ITEM_LABEL
		he->copy_mi = w;

		icon = (guint8 **)icon_paste_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Pasta"
#elif defined(PROG_LANGUAGE_FRENCH)
"Coller"
#elif defined(PROG_LANGUAGE_GERMAN)
"Paste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Pasta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plakmiddel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Pasta"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Masse"
#else
"Paste"
#endif
		;
		accel_key = GDK_v;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_paste_cb;
		ADD_MENU_ITEM_LABEL
		he->paste_mi = w;

		ADD_MENU_SEP

		icon = (guint8 **)icon_add_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Adicin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Inserer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Einsatz"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Inserzione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tussenvoegsel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Insira"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innsetning"
#else
"Insert"
#endif
		;
		accel_key = GDK_Insert;
		accel_mods = 0;
		func_cb = hedit_insert_cb;
		ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_cancel_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Borre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Couper"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk"
#else
"Delete"
#endif
		;
		accel_key = GDK_Delete;
		accel_mods = 0;
		func_cb = hedit_delete_cb;
		ADD_MENU_ITEM_LABEL
		he->delete_mi = w;

		ADD_MENU_SEP

		icon = NULL;
		label = "Hex Edit Mode";
		accel_key = GDK_F7;
		accel_mods = 0;
		func_cb = hedit_edit_mode_hex_cb;
		ADD_MENU_ITEM_CHECK
		he->edit_mode_hex_mi = w;

		icon = NULL;
		label = "ASCII Edit Mode";
		accel_key = GDK_F8;
		accel_mods = 0;
		func_cb = hedit_edit_moed_ascii_cb;
		ADD_MENU_ITEM_CHECK
		he->edit_mode_ascii_mi = w;

		ADD_MENU_SEP

		icon = NULL;
		label = "Select All";
		accel_key = 0;
		accel_mods = 0;
		func_cb = hedit_select_all_cb;
		ADD_MENU_ITEM_LABEL
		he->select_all_mi = w;

		icon = NULL;
		label = "Unselect All";
		accel_key = 0;
		accel_mods = 0;
		func_cb = hedit_unselect_all_cb;
		ADD_MENU_ITEM_LABEL
		he->unselect_all_mi = w;

		ADD_MENU_SEP

		icon = (guint8 **)icon_search_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"El Hallazgo..."
#elif defined(PROG_LANGUAGE_FRENCH)
"Rechercher..."
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund..."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare..."
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst..."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache..."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn..."
#else
"Find..."
#endif
		;
		accel_key = GDK_f;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = hedit_find_cb;
		ADD_MENU_ITEM_LABEL
		he->find_mi = w;

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Encuentre Luego"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rechercher suivant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Finden Sie Nchst"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vind Volgend"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache Logo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Finn Next"
#else
"Find Next"
#endif
		;
		accel_key = GDK_F6;
		accel_mods = 0;
		func_cb = hedit_find_next_cb;
		ADD_MENU_ITEM_LABEL
		he->find_next_mi = w;

	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Tools menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		icon = (guint8 **)icon_calculator_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Calcule Escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Calculer Choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kalkulieren Sie Ausgewhlt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Calcolare Scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Reken Geselecterenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Calcule Selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kalkuler Valgt Ut"
#else
"Calculate Selected"
#endif
		;
		accel_key = GDK_F9;
		accel_mods = 0;
		func_cb = hedit_calculate_selected_cb;
		ADD_MENU_ITEM_LABEL

		icon = NULL;
		label = "ASCII Chart...";
		accel_key = GDK_F10;
		accel_mods = 0;
		func_cb = hedit_ascii_chart_cb;
		ADD_MENU_ITEM_LABEL
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Instrumentos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Outils"
#elif defined(PROG_LANGUAGE_GERMAN)
"Werkzeuge"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Attrezzi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Werktuigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ferramentas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Redskaper"
#else
"Tools"
#endif
		, GTK_JUSTIFY_LEFT
	);



	/* Help menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		icon = (guint8 **)icon_about_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Acerca De"
#elif defined(PROG_LANGUAGE_FRENCH)
"A propos"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ungefhr"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Circa"
#elif defined(PROG_LANGUAGE_DUTCH)
"Over"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Sobre"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Omtrent"
#else
"About"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = hedit_about_cb;
		ADD_MENU_ITEM_LABEL
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#else
"Help"
#endif
		, GTK_JUSTIFY_RIGHT
	);





#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_SEP
}

/*
 *	Updates Hex Editor's widgets to reflect current values.
 */
static void hedit_update_display(hedit_struct *he)
{
	gboolean sensitive, has_changes;
	GtkWidget *w;
	hview_struct *hv;

	if(he == NULL)
		return;

	he->freeze_count++;

	hv = he->hv;
	has_changes = HVIEW_HAS_CHANGES(hv);

	w = he->toplevel;
	if(w != NULL)
	{
		const gchar *name;
		gchar *s;

		if(he->path != NULL)
		{
			name = (const gchar *)strrchr(
				(const char *)he->path, G_DIR_SEPARATOR
			);
			if(name != NULL)
				name++;
			else
				name = he->path;
		}
		else
		{
			name = "Untitled";
		}

		s = g_strconcat(
			PROG_NAME_FULL,
			": ",
			name,
			(has_changes) ? " (*)" : "",
			NULL
		);
		gtk_window_set_title(GTK_WINDOW(w), s);
		g_free(s);
	}

	sensitive = ((hv->buf_sel_start >= 0) && (hv->buf_sel_end >= 0)) ?
		TRUE : FALSE;
	gtk_widget_set_sensitive(he->cut_mi, sensitive);
	gtk_widget_set_sensitive(he->copy_mi, sensitive);
	sensitive = TRUE;
	gtk_widget_set_sensitive(he->paste_mi, sensitive);

	GTK_CHECK_MENU_ITEM_SET_ACTIVE(
		he->edit_mode_hex_mi,
		(hv->edit_mode == HVIEW_EDIT_MODE_HEX) ? TRUE : FALSE
	);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(
		he->edit_mode_ascii_mi,
		(hv->edit_mode == HVIEW_EDIT_MODE_ASCII) ? TRUE : FALSE
	);

	he->freeze_count--;
}


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	gint		cursor_pos = 0;
	const gchar	*arg,
					*view_font_name = NULL,
					*path = NULL;
	GdkWindow *window;
	GtkWidget *w;
	hedit_struct *he;
	hview_struct *hv;
	EDVContext *ctx;

#define CLEANUP_RETURN(_v_)	{	\
									\
 return(_v_);				\
}

	/* Set up signal callbacks */
#ifdef SIGINT
	signal(SIGINT, hedit_signal_cb);
#endif
#ifdef SIGTERM
	signal(SIGTERM, hedit_signal_cb);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, hedit_signal_cb);
#endif
#ifdef SIGSTOP
	signal(SIGSTOP, hedit_signal_cb);
#endif
#ifdef SIGCONT
	signal(SIGCONT, hedit_signal_cb);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, hedit_signal_cb);
#endif

	/* Handle for arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			g_print("%s", PROG_HELP_MESG);
			CLEANUP_RETURN(0);
		}
		/* Version? */
		else if(!g_strcasecmp(arg, "--version") ||
				!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s %s\n%s",
				PROG_NAME_FULL, PROG_VERSION, PROG_COPYRIGHT
			);
			CLEANUP_RETURN(0);
		}
		/* Address */
		else if(!g_strcasecmp(arg, "--address") ||
				!g_strcasecmp(arg, "-address") ||
				!g_strcasecmp(arg, "--a") ||
				!g_strcasecmp(arg, "-a")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			{
				cursor_pos = ATOI(arg);
			}
			else
			{
				g_printerr(
					"%s: Requires argument\n",
					argv[i - 1]
				);
				CLEANUP_RETURN(2);
			}
		}
		/* View Font */
		else if(!g_strcasecmp(arg, "--view-font-name") ||
				!g_strcasecmp(arg, "-view-font-name") ||
				!g_strcasecmp(arg, "--view-font") ||
				!g_strcasecmp(arg, "-view-font")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			{
				view_font_name = arg;
			}
			else
			{
				g_printerr(
					"%s: Requires argument\n",
					argv[i - 1]
				);
				CLEANUP_RETURN(2);
			}
		}
		/* Skip these arguments so that gtk_window_apply_args()
		 * handles them
		 */
		else if(gtk_is_window_arg(arg))
		{
			i++;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;
/*		gchar c; */

			while(*v != '\0')
			{
#if 0
				c = *v;
				if(c == 's')
				{

				}
				else
				{
					g_printerr(
"-%c: Unsupported argument.\n",
						c
					);
					CLEANUP_RETURN(2);
				}
#endif
				v++;
			}
		}
		/* Non-option argument? */
		else if((*arg != '-') && (*arg != '+'))
		{
			path = arg;
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(2);
		}
	}

	/* Initialize GTK+ as needed */
	if(!initialized_gtk)
	{
		if(!gtk_init_check(&argc, &argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			CLEANUP_RETURN(1);
		}
		initialized_gtk = TRUE;

		/* Initialize the GDK RGB buffers system */
		gdk_rgb_init();
	}

	/* Initialize the dialogs */
	FPromptInit();
	CDialogInit();
	FileBrowserInit();
	PDialogInit();

	/* Initialize the Endeavour context */
	ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Set the file selector callbacks */
	FileBrowserSetObjectCreatedCB(
		hedit_file_selector_created_cb, ctx
	);
	FileBrowserSetObjectModifiedCB(
		hedit_file_selector_modified_cb, ctx
	);
	FileBrowserSetObjectDeletedCB(
		hedit_file_selector_deleted_cb, ctx
	);


	/* Begin creating the HEdit window */
	he = HEDIT(g_malloc0(sizeof(hedit_struct)));
	he->toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	he->accelgrp = gtk_accel_group_new();
	he->freeze_count = 0;
	he->ac_dlg = NULL;
	he->ctx = ctx;
	he->path = NULL;
	he->last_find_needle = NULL;
	he->last_find_needle_len = 0;
	he->last_find_case_sensitive = FALSE;

	he->freeze_count++;

	/* Toplevel GtkWindow */
	w = he->toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w), "Editor", PROG_NAME
	);
	gtk_widget_set_name(w, HEDIT_WIDGET_NAME_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	gtk_widget_set_usize(w, HEDIT_WIDTH, HEDIT_HEIGHT);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(hedit_delete_event_cb), he
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(hedit_key_event_cb), he
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(hedit_key_event_cb), he
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
		GdkGeometry geo;

		geo.min_width = 100;
		geo.min_height = 70;

		geo.base_width = 0;
		geo.base_height = 0;

		geo.width_inc = 1;
		geo.height_inc = 1;

		gdk_window_set_geometry_hints(
			window, &geo,
			GDK_HINT_MIN_SIZE |
			GDK_HINT_BASE_SIZE |
			GDK_HINT_RESIZE_INC
		);

		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_RESIZEH | GDK_DECOR_MINIMIZE |
		    GDK_DECOR_MAXIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_RESIZE |
			GDK_FUNC_MINIMIZE | GDK_FUNC_MAXIMIZE |
			GDK_FUNC_CLOSE
		);

		GUISetWMIcon(window, (guint8 **)icon_edit_48x48_xpm);
	}
	gtk_window_add_accel_group(GTK_WINDOW(w), he->accelgrp);
	gtk_window_apply_args(GTK_WINDOW(w), argc, argv);

	he->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(he->toplevel), w);
	gtk_widget_show(w);

	he->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(he->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	hedit_build_menu_bar(he, he->menu_bar_handle);

	he->hv = hv = HViewNew(he->main_vbox);
	gtk_widget_set_name(hv->view_da, HEDIT_WIDGET_NAME_HEX_VIEWER);
	if(view_font_name != NULL)
		HViewSetViewFont(hv, view_font_name);
	HViewSetChangedCB(hv, hedit_hview_changed_cb, he);
	HViewSetSelectCB(hv, hedit_hview_select_cb, he);
	HViewSetEditModeChangedCB(hv, hedit_hview_edit_mode_changed_cb, he);
	HViewMap(hv);

	/* Set up the HexView for DND */
	w = hv->view_da;
	if(w != NULL)
	{
		const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	HEDIT_DND_TYPE_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	HEDIT_DND_TYPE_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	HEDIT_DND_TYPE_INFO_STRING}
		};
		GUIDNDSetTar(
			w,
			dnd_tar_types,
			sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY,		/* Drag actions */
			GDK_ACTION_MOVE,                /* Default action if same */
			GDK_ACTION_COPY,                /* Default action */
			hedit_dnd_drag_received_cb,
			he,
			TRUE				/* Highlight */
		);
	}

	hedit_update_display(he);

	he->freeze_count--;

	gtk_widget_show_raise(he->toplevel);
	gtk_widget_grab_focus(hv->view_da);

	/* Open file at startup? */
	if(!STRISEMPTY(path))
	{
		struct stat stat_buf;
		gboolean file_exists;
		GtkWidget *toplevel = he->toplevel;

		HViewSetBusy(hv, TRUE);

		/* Check if the file exists */
		if(stat((const char *)path, &stat_buf))
		{
			const gint error_code = (gint)errno;
			if(error_code == ENOENT)
				file_exists = FALSE;
			else
				file_exists = TRUE;
		}
		else
		{
			file_exists = TRUE;
		}
		/* Does the file exist? */
		if(file_exists)
		{
			/* File exists, open it */
			if(path != he->path)
			{
				g_free(he->path);
				he->path = g_strdup(path);
			}
			HViewOpenFileProgress(
				hv,
				he->path,
				hedit_hview_open_progress_cb, he
			);
		}
		else
		{
			/* File does not exist, prompt the user to create
			 * a new file
			 */
			gint response;
			gchar *msg = g_strdup_printf(
"The file \"%s\" does not exist,\n\
do you want to create it?",
				g_basename(path)
			);
			edv_play_sound_question(he->ctx);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
				"Confirm Create",
				msg,
				NULL,
				CDIALOG_ICON_QUESTION,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_NO
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
			if(response == CDIALOG_RESPONSE_YES)
			{
				/* Create the new file */
				if(edv_touch(
					path,
					(gulong)-1,		/* Current time */
					TRUE			/* Create */
				) == 0)
				{
					/* Set the new file name */
					if(path != he->path)
					{
						g_free(he->path);
						he->path = g_strdup(path);
					}

					/* Notify about the file being created */
					edv_notify_queue_vfs_object_added(
						he->ctx,
						he->path
					);
					edv_context_sync(he->ctx);

					/* Open the new file */
					HViewOpenFileProgress(
						hv,
						he->path,
						hedit_hview_open_progress_cb, he
					);
				}
				else
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
						g_strerror(error_code),
						path
					);
					edv_play_sound_error(he->ctx);
					CDialogSetTransientFor(toplevel);
					CDialogGetResponse(
						"Create Failed",
						msg,
						NULL,
						CDIALOG_ICON_ERROR,
						CDIALOG_BTNFLAG_OK,
						CDIALOG_BTNFLAG_OK
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
				}
			}
		}	/* Does the file exist? */

		/* Set the cursor position at startup? */
		if(cursor_pos > 0)
		{
			if(cursor_pos >= hv->buf_len)
				cursor_pos = MAX((hv->buf_len - 1), 0);
			HViewSetPosition(hv, cursor_pos);
			HViewScrollTo(hv, cursor_pos, 0.5f);
		}
		else if(cursor_pos == -1)
		{
			cursor_pos = MAX((hv->buf_len - 1), 0);
			HViewSetPosition(hv, cursor_pos);
			HViewScrollTo(hv, cursor_pos, 0.5f);
		}

		hedit_update_display(he);

		HViewSetBusy(hv, FALSE);
	}


	/* Push the first GTK main level */
	gtk_main();


	/* Delete the HEdit window */
	ascii_chart_dlg_delete(he->ac_dlg);
	he->ac_dlg = NULL;

	gtk_widget_hide(he->toplevel);

	he->freeze_count++;
	HViewClear(hv);
	HViewDelete(hv);
	gtk_widget_destroy(he->toplevel);
	gtk_accel_group_unref(he->accelgrp);

	g_free(he->path);
	g_free(he->last_find_needle);

	he->freeze_count--;

	g_free(he);
	he = NULL;


	/* Delete the Endeavour context */
	edv_context_sync(ctx);
	edv_context_delete(ctx);
	ctx = NULL;

	/* Shutdown dialogs */
	PDialogShutdown();
	FileBrowserShutdown();
	CDialogShutdown();
	FPromptShutdown();

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
