#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#if defined(_WIN32)
# include <gdk/gdkwin32.h>
# define HAVE_WIN32
#else
# include <gdk/gdkx.h>
# define HAVE_X
#endif

#include "guirgbimg.h"


static void GUIImagePixelRGBABlend(
	const gint bpp, guint8 *tar, const guint8 *src
);
static void GUIImagePixelGreyscaleAlphaBlend(
	const gint bpp, guint8 *tar, const guint8 *src
);
static void GUIImagePixelAnySet(
	const gint bpp, guint8 *tar, const guint8 *src
);

void GUIImageBufferCopyArea(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl,
	const gint tar_x, const gint tar_y,
	const gboolean blend,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);
void GUIImageBufferResize(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);
void GUIImageBufferRotateCW90(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
);
void GUIImageBufferRotateCCW90(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
);
void GUIImageBufferRotateCW180(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
);
void GUIImageBufferMirrorH(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data, 
	const gint tar_width, const gint tar_height, const gint tar_bpl
);
void GUIImageBufferMirrorV(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
);

void GUIImageBufferFlattenWithBG(
	const gint bpp,
	guint8 *data,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);

guint8 *GUIGetRGBImage(
	GdkDrawable *drawable,
	const GdkRectangle *area,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn
);
guint8 *GUIGetRGBAImage(
	GdkDrawable *drawable,
	GdkBitmap *mask,
	const GdkRectangle *area,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn  
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void GUIImagePixelRGBABlend(
	const gint bpp, guint8 *tar, const guint8 *src
)
{
	const guint32 src_alpha = src[3];
	if(src_alpha == 0xFF)
	{
		*(guint32 *)tar = *(guint32 *)src;
	}
	else if(src_alpha > 0x00)
	{
		const guint32 tar_alpha = 0xFF - src_alpha;
		tar[0] = (guint8)(
			((guint32)tar[0] * tar_alpha / 0xFF) +
			((guint32)src[0] * src_alpha / 0xFF)
		);
		tar[1] = (guint8)(
			((guint32)tar[1] * tar_alpha / 0xFF) +
			((guint32)src[1] * src_alpha / 0xFF)
		);
		tar[2] = (guint8)(
			((guint32)tar[2] * tar_alpha / 0xFF) +
			((guint32)src[2] * src_alpha / 0xFF)
		);
		tar[3] = (guint8)MIN(
			(guint32)tar[3] + src_alpha,
			0xFF
		);
	}
}

static void GUIImagePixelGreyscaleAlphaBlend(
	const gint bpp, guint8 *tar, const guint8 *src
)
{
	const guint32 src_alpha = src[1];
	if(src_alpha == 0xFF)
	{
		*(guint16 *)tar = *(guint16 *)src;
	}
	else if(src_alpha > 0x00)
	{
		const guint32 tar_alpha = 0xFF - src_alpha;
		tar[0] = (guint8)(
			((guint32)tar[0] * tar_alpha / 0xFF) +
			((guint32)src[0] * src_alpha / 0xFF)
		);
		tar[1] = (guint8)MIN(
			(guint32)tar[1] + src_alpha,
			0xFF
		);
	}
}

static void GUIImagePixelAnySet(
	const gint bpp, guint8 *tar, const guint8 *src
)
{
	(void)memcpy(tar, src, bpp);
}




/*
 *	Coppies an area of the source image to the target image.
 *
 *	The bpp specifies the bytes per pixel.
 *
 *	The src_data, src_width, and src_height specifies the
 *	source image.
 *
 *	The src_bpl specifies the number of bytes per line on the
 *	source image.
 *
 *	The tar_data, tar_width, and tar_height specifies the
 *	source image.
 *
 *	The tar_bpl specifies the number of bytes per line on the
 *	source image.
 *
 *	The tar_x and tar_y specifies the coordinates on the target
 *	image to copy the source image to.
 *
 *	If blend is TRUE then if bpp is 4 or 2 (if there is an alpha
 *	channel) then the source image will be blended on to the
 *	target image, otherwise the source image's pixel is set to
 *	the target image's pixel.
 */
void GUIImageBufferCopyArea(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl,
	const gint tar_x, const gint tar_y,
	const gboolean blend,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	gint	x, y, src_y,
			_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8 *src_ptr, *src_end;
	void (*pixel_op_func)(const gint, guint8 *, const guint8 *);

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	if(progress_cb != NULL)
	{
		if(progress_cb(0l, (gulong)src_height, progress_data))
			return;
	}

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Determine the pixel operation function */
	switch(bpp)
	{
	    case 4:
		if(blend)
			pixel_op_func = GUIImagePixelRGBABlend;
		else
			pixel_op_func = GUIImagePixelAnySet;
		break;
	    case 2:
		if(blend)
			pixel_op_func = GUIImagePixelGreyscaleAlphaBlend;
		else
			pixel_op_func = GUIImagePixelAnySet;
		break;
	    default:
		pixel_op_func = GUIImagePixelAnySet;
		break;
	}

	/* Iterate through source image */
	for(y = tar_y, src_y = 0;
	    src_y < src_height;
	    y++, src_y++
	)
	{
		if(y < 0)
			continue;
		if(y >= tar_height)
			break;

		if((progress_cb != NULL) && ((y % 10) == 0))
		{
			if(progress_cb(
				(gulong)src_y, (gulong)src_height,
				progress_data
			))
				return;
		}

		x = tar_x;
		src_ptr = src_data + (src_y * _src_bpl);
		src_end = src_ptr + (src_width * bpp);
		while(src_ptr < src_end)
		{
			/* Current target coordinates fall in bounds on the
			 * target image?
			 */
			if((x >= 0) && (x < tar_width))
			{
				/* Target coordinates are in bounds, set this
				 * pixel
				 */
				pixel_op_func(
					bpp,
					tar_data + (y * _tar_bpl) + (x * bpp),
					src_ptr
				);
				src_ptr += bpp;
			}
			else
			{
				/* Target coordinates out of bounds, skip this
				 * pixel
				 */
 		    src_ptr += bpp;
			}

			x++;
		}
	}

	if(progress_cb != NULL)
	{
		if(progress_cb((gulong)src_height, (gulong)src_height, progress_data))
			return;
	}
}


/*
 *	Copy/resize the source image data to the target image data of
 *	a different size.
 */
void GUIImageBufferResize(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	gint		_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8 *src_ptr8;
	guint8 *tar_ptr8;
	const guint16 *src_ptr16;
	guint16 *tar_ptr16;
	const guint32 *src_ptr32;
	guint32 *tar_ptr32;

	/* Current positions */
	gint		tar_x_col, tar_y_row,
			src_x_col, src_y_row;	/* In units of 256 */

	/* Increments, in units of 256 */
	gint		src_dh_x_col_inc, src_dh_y_row_inc,
			src_dh_width, src_dh_height;


	if((tar_width <= 0) || (tar_height <= 0) ||
	   (src_width <= 0) || (src_height <= 0) ||
	   (tar_data == NULL) || (src_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	if(progress_cb != NULL)
	{
		if(progress_cb(0l, (gulong)tar_height, progress_data))
			return;
	}

	/* Calculate the bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* 8 bits */
	if(bpp == 1)
	{
		src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
		src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

		src_dh_width = src_width * 256;
		src_dh_height = src_height * 256;

		tar_x_col = 0;
		tar_y_row = 0;
		src_x_col = 0 * 256;
		src_y_row = 0 * 256;

		/* Begin copying source buffer to target buffer */
		while((tar_y_row < tar_height) &&
		      (src_y_row < src_dh_height)
		)
		{
			if((progress_cb != NULL) &&
			   ((tar_y_row % 10) == 0)
			)
			{
				if(progress_cb((gulong)tar_y_row, (gulong)tar_height, progress_data))
					return;
			}

			/* Get the pointers to the current pixel in the data */
			src_ptr8 = (const guint8 *)(src_data +
				((src_y_row >> 8) * _src_bpl) +
				((src_x_col >> 8) * bpp)
			);
			tar_ptr8 = (guint8 *)(tar_data +
				(tar_y_row * _tar_bpl) +
				(tar_x_col * bpp)
			);

			/* Set the pixel */
			*tar_ptr8 = *src_ptr8;

			/* Increment the column positions */
			tar_x_col += 1;
			src_x_col += src_dh_x_col_inc;

			/* Go to next line? */
			if((tar_x_col >= tar_width) ||
			   (src_x_col >= src_dh_width)
			)
			{
				tar_x_col = 0;
				src_x_col = 0;

				tar_y_row += 1;
				src_y_row += src_dh_y_row_inc;
			}
		}
	}
	/* 16 bits */
	else if(bpp == 2)
	{
		src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
		src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

		src_dh_width = src_width * 256;
		src_dh_height = src_height * 256;

		tar_x_col = 0;
		tar_y_row = 0;
		src_x_col = 0 * 256;
		src_y_row = 0 * 256;

		/* Begin copying source buffer to target buffer */
		while((tar_y_row < tar_height) &&
		      (src_y_row < src_dh_height)
		)
		{
			if((progress_cb != NULL) &&
			   ((tar_y_row % 10) == 0)
			)
			{
				if(progress_cb((gulong)tar_y_row, (gulong)tar_height, progress_data))
					return;
			}

			/* Get the pointers to the current pixel in the data */
			src_ptr16 = (const guint16 *)(src_data +
				((src_y_row >> 8) * _src_bpl) +
				((src_x_col >> 8) * bpp)
			);
			tar_ptr16 = (guint16 *)(tar_data +
				(tar_y_row * _tar_bpl) +
				(tar_x_col * bpp)
			);

			/* Set the pixel */
			*tar_ptr16 = *src_ptr16;

			/* Increment the column positions */
			tar_x_col += 1;
			src_x_col += src_dh_x_col_inc;

			/* Go to next line? */
			if((tar_x_col >= tar_width) ||
			   (src_x_col >= src_dh_width)
			)
			{
				tar_x_col = 0;
				src_x_col = 0;

				tar_y_row += 1;
				src_y_row += src_dh_y_row_inc;
			}
		}
	}
	/* 24 bits */
	else if(bpp == 3)
	{
		src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
		src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

		src_dh_width = src_width * 256;
		src_dh_height = src_height * 256;

		tar_x_col = 0;
		tar_y_row = 0;
		src_x_col = 0 * 256;
		src_y_row = 0 * 256;

		/* Begin copying source buffer to target buffer */
		while((tar_y_row < tar_height) &&
		      (src_y_row < src_dh_height)
		)
		{
			if((progress_cb != NULL) &&
			   ((tar_y_row % 10) == 0)
			)
			{
				if(progress_cb((gulong)tar_y_row, (gulong)tar_height, progress_data))
					return;
			}

			/* Get the pointers to the current pixel in the data */
			src_ptr8 = (const guint8 *)(src_data +
				((src_y_row >> 8) * _src_bpl) +
				((src_x_col >> 8) * bpp)
			);
			tar_ptr8 = (guint8 *)(tar_data +
				(tar_y_row * _tar_bpl) +
				(tar_x_col * bpp)
			);

			/* Set the pixel */
			*tar_ptr8++ = *src_ptr8++;
			*tar_ptr8++ = *src_ptr8++;
			*tar_ptr8 = *src_ptr8;

			/* Increment the column positions */
			tar_x_col += 1;
			src_x_col += src_dh_x_col_inc;

			/* Go to next line? */
			if((tar_x_col >= tar_width) ||
			   (src_x_col >= src_dh_width)
			)
			{
				tar_x_col = 0;
				src_x_col = 0;

				tar_y_row += 1;
				src_y_row += src_dh_y_row_inc;
			}
		}
	}
	/* 32 bits */
	else if(bpp == 4)
	{
		src_dh_x_col_inc = (gint)(src_width * 256) / tar_width;
		src_dh_y_row_inc = (gint)(src_height * 256) / tar_height;

		src_dh_width = src_width * 256;
		src_dh_height = src_height * 256;

		tar_x_col = 0;
		tar_y_row = 0;
		src_x_col = 0 * 256;
		src_y_row = 0 * 256;

		/* Begin copying source buffer to target buffer */
		while((tar_y_row < tar_height) &&
		      (src_y_row < src_dh_height)
		)
		{
			if((progress_cb != NULL) &&
			   ((tar_y_row % 10) == 0)
			)
			{
				if(progress_cb((gulong)tar_y_row, (gulong)tar_height, progress_data))
					return;
			}

			/* Get the pointers to the current pixel in the data */
			src_ptr32 = (const guint32 *)(src_data +
				((src_y_row >> 8) * _src_bpl) +
				((src_x_col >> 8) * bpp)
			);
			tar_ptr32 = (guint32 *)(tar_data +
				(tar_y_row * _tar_bpl) +
				(tar_x_col * bpp)
			);

			/* Set the pixel */
			*tar_ptr32 = *src_ptr32;

			/* Increment the column positions */
			tar_x_col += 1;
			src_x_col += src_dh_x_col_inc;

			/* Go to next line? */
			if((tar_x_col >= tar_width) ||
			   (src_x_col >= src_dh_width)
			)
			{
				tar_x_col = 0;
				src_x_col = 0;

				tar_y_row += 1;
				src_y_row += src_dh_y_row_inc;
			}
		}
	}

	if(progress_cb != NULL)
	{
		if(progress_cb((gulong)tar_height, (gulong)tar_height, progress_data))
			return;
	}
}

/*
 *	Copy/rotates the source image data to the target image data
 *	clockwise 90 degrees.
 */
void GUIImageBufferRotateCW90(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
)
{
	gint		tx, ty,
			_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8	*src_line,
			*src_end,
			*src_ptr;

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	/* Rotated geometry not correct? */
	if((src_width != tar_height) || (src_height != tar_width))
		return;

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Copy/rotate clockwise 90 degrees */
	for(src_line = src_data,
	    src_end = src_line + (src_height * _src_bpl),
	    tx = tar_width - 1;
	    src_line < src_end;
	    src_line += _src_bpl,
	    tx--
	)
	{
		for(src_ptr = src_line,
		    ty = 0;
		    ty < tar_height;
		    src_ptr += bpp,
		    ty++
		)
			(void)memcpy(
				tar_data + (ty * _tar_bpl) + (tx * bpp),
				src_ptr,
				bpp
			);
	}
}

/*
 *	Copy/rotates the source image data to the target image data
 *	counter-clockwise 90 degrees.
 */
void GUIImageBufferRotateCCW90(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
)
{
	gint		tx, ty,
			_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8	*src_line,
			*src_end,
			*src_ptr;

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	/* Rotated geometry not correct? */
	if((src_width != tar_height) || (src_height != tar_width))
		return;

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Copy/rotate counter-clockwise 90 degrees */
	for(src_line = src_data,
	    src_end = src_line + (src_height * _src_bpl),
	    tx = 0;
	    src_line < src_end;
	    src_line += _src_bpl,
	    tx++
	)
	{
		for(src_ptr = src_line,
		    ty = tar_height - 1;
		    ty >= 0;
		    src_ptr += bpp,
		    ty--
		)
			(void)memcpy(
				tar_data + (ty * _tar_bpl) + (tx * bpp),
				src_ptr,
				bpp
			);
	}
}

/*
 *	Copy/rotates the source image data to the target image data
 *	clockwise 180 degrees.
 */
void GUIImageBufferRotateCW180(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
) 
{
	gint		_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8	*src_line,
			*src_end,
			*src_ptr;
	guint8		*tar_line,
			*tar_ptr;

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	/* Rotated geometry not correct? */
	if((src_width != tar_width) || (src_height != tar_height))
		return;

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Copy/rotate clockwise 180 degrees */
	for(src_line = src_data,
	    src_end = src_line + (src_height * _src_bpl),
	    tar_line = tar_data + ((tar_height - 1) * _tar_bpl);
	    src_line < src_end;
	    src_line += _src_bpl,
	    tar_line -= _tar_bpl
	)
	{
		for(src_ptr = src_line,
		    tar_ptr = tar_line + ((tar_width - 1) * bpp);
		    tar_ptr >= tar_line;
		    src_ptr += bpp,
		    tar_ptr -= bpp
		)
			(void)memcpy(
				tar_ptr,
				src_ptr,
				bpp
			);
	}
}

/*
 *	Copy/mirrors the source image data to the target image data
 *	horizontally.
 */
void GUIImageBufferMirrorH(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
)
{
	gint		_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8	*src_line,
			*src_ptr,
			*src_end;
	guint8		*tar_line,
			*tar_ptr;

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	/* Mirror geometry not correct? */
	if((src_width != tar_width) || (src_height != tar_height))
		return;

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Copy/mirror horizontally */
	src_line = src_data;
	src_end = src_line + (src_height * _src_bpl);
	tar_line = tar_data;
	while(src_line < src_end)
	{
		src_ptr = src_line;
		tar_ptr = tar_line + ((tar_width - 1) * bpp);
		while(tar_ptr >= tar_line)
		{
			(void)memcpy(
				tar_ptr,
				src_ptr,
				bpp
			);
			src_ptr += bpp;
			tar_ptr -= bpp;
		}
		src_line += _src_bpl;
		tar_line += _tar_bpl;
	}
}

/*
 *	Copy/mirrors the source image data to the target image data
 *	vertically.
 */
void GUIImageBufferMirrorV(
	const gint bpp,
	const guint8 *src_data,
	const gint src_width, const gint src_height, const gint src_bpl,
	guint8 *tar_data,
	const gint tar_width, const gint tar_height, const gint tar_bpl
)
{
	gint		min_bpl,
			_src_bpl = src_bpl,
			_tar_bpl = tar_bpl;
	const guint8 *src_line;
	guint8 *tar_line;

	if((src_width <= 0) || (src_height <= 0) ||
	   (tar_width <= 0) || (tar_height <= 0) ||
	   (src_data == NULL) || (tar_data == NULL) ||
	   (bpp <= 0)
	)
		return;

	/* Mirror geometry not correct? */
	if((src_width != tar_width) || (src_height != tar_height))
		return;

	/* Calculate bytes per line values as needed */
	if(_src_bpl <= 0)
		_src_bpl = src_width * bpp;
	if(_tar_bpl <= 0)
		_tar_bpl = tar_width * bpp;

	/* Copy/mirror vertically */
	min_bpl = MIN(_src_bpl, _tar_bpl);
	src_line = src_data;
	tar_line = tar_data + ((tar_height - 1)	* _tar_bpl);
	while(tar_line >= tar_data)
	{
		(void)memcpy(
			tar_line,
			src_line,
			min_bpl
		);
		src_line += _src_bpl;
		tar_line -= _tar_bpl;
	}
}

/*
 *	Combines the image data with the background color.
 *
 *	The bpp specifies the bytes per pixel and this value must
 *	be either 2 (Greyscale Alpha) or 4 (RGBA).
 */
void GUIImageBufferFlattenWithBG(
	const gint bpp,
	guint8 *data,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	gint _bpl = bpl;
	guint		tar_alpha,
			bg_alpha;

	if((data == NULL) || (bg_color == NULL))
		return;

	if(progress_cb != NULL)
	{
		if(progress_cb(0l, (gulong)height, progress_data))
			return;
	}

	/* Calculate bytes per line value as needed */
	if(_bpl <= 0)
		_bpl = width * bpp;

	/* Greyscale Alpha */
	if(bpp == 2)
	{
		gint y;
		guint8	*line,
			*ptr,
			*end;
		for(y = 0,
		    line = data;
		    y < height;
		    y++,
		    line += _bpl
		)
		{
			if((progress_cb != NULL) && ((y % 10) == 0))
			{
				if(progress_cb(
					(gulong)y, (gulong)height,
					progress_data
				))
					return;
			}

			ptr = line;
			end = ptr + (width * bpp);
			while(ptr < end)
			{
				tar_alpha = (guint)ptr[3];
				if(tar_alpha == 0xFF)
				{
					ptr += bpp;
				}
				else if(tar_alpha > 0x00)
				{
					bg_alpha = 0xFF - tar_alpha;
					ptr[0] = (guint8)(
						((guint)(ptr[0]) * tar_alpha / 0xFF) +
						((guint)(bg_color[0]) * bg_alpha / 0xFF)
					);
					ptr[1] = 0xFF;
					ptr += bpp;
				}
				else
				{
					*(guint16 *)ptr = *(guint16 *)bg_color;
					ptr += bpp;
				}
			}
		}
	}
	/* RGBA */
	else if(bpp == 4)
	{
		gint y;
		guint8	*line,
			*ptr,
			*end;
		for(y = 0,
		    line = data;
		    y < height;
		    y++,
		    line += _bpl
		)
		{
			if((progress_cb != NULL) && ((y % 10) == 0))
			{
				if(progress_cb(
					(gulong)y, (gulong)height,
					progress_data
				))
					return;
			}

			ptr = line;
			end = ptr + (width * bpp);
			while(ptr < end)
			{
				tar_alpha = (guint)ptr[3];
				if(tar_alpha == 0xFF)
				{
					ptr += bpp;
				}
				else if(tar_alpha > 0x00)
				{
					bg_alpha = 0xFF - tar_alpha;
					ptr[0] = (guint8)(
						((guint)(ptr[0]) * tar_alpha / 0xFF) +
						((guint)(bg_color[0]) * bg_alpha / 0xFF)
					);
					ptr[1] = (guint8)(
						((guint)(ptr[1]) * tar_alpha / 0xFF) +
						((guint)(bg_color[1]) * bg_alpha / 0xFF)
					);
					ptr[2] = (guint8)(
						((guint)(ptr[2]) * tar_alpha / 0xFF) +
						((guint)(bg_color[2]) * bg_alpha / 0xFF)
					);
					ptr[3] = 0xFF;
					ptr += bpp;
				}
				else
				{
					*(guint32 *)ptr = *(guint32 *)bg_color;
					ptr += bpp;
				}
			}
		}
	}

	if(progress_cb != NULL)
	{
		if(progress_cb(
			(gulong)height, (gulong)height,
			progress_data
		))
			return;
	}
}


/*
 *	Gets the RGB image data from a GdkDrawable.
 *
 *	The drawable specifies the GdkDrawable.
 *
 *	The area specifies the area within the GdkDrawable. If area
 *	is NULL then the entire GdkDrawable is obtained.
 *
 *	The width_rtn, height_rtn, and bpl_rtn specifies the
 *	return values for the image data.
 *
 *	Returns a dynamically allocated copy of the image data or NULL
 *	on error.
 */
guint8 *GUIGetRGBImage(
	GdkDrawable *drawable, 
	const GdkRectangle *area,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn
)
{
#if defined(HAVE_X)
	const gint	tar_bpp = 3;		/* RGB */
	gint		width, height,
			tar_bpl;
	const guint8 *src_data;
	guint8 *tar_data;
	XImage *ximg;
	GdkRectangle larea;
	GdkWindowPrivate *private;

	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;

	if(drawable == NULL)
		return(NULL);

	private = (GdkWindowPrivate *)drawable;

	/* If no area was specified then set the area to cover the
	 * entire Drawable
	 */
	if(area == NULL)
	{
		area = &larea;
		larea.x = 0;
		larea.y = 0;
		larea.width = private->width;
		larea.height = private->height;
	}

	if((area->width <= 0) || (area->height <= 0))
		return(NULL);

	/* Get an XImage from the GdkDrawable's Window */
	ximg = XGetImage(
		private->xdisplay,			/* Display */
		private->xwindow,			/* Window */
		(int)area->x, (int)area->y,
		(unsigned int)area->width, (unsigned int)area->height,
		AllPlanes,
		ZPixmap
	);
	if(ximg == NULL)
		return(NULL);

	if(ximg->data == NULL)
	{
		XDestroyImage(ximg);
		return(NULL);
	}

	/* Get the values from the XImage */
	width = ximg->width;
	height = ximg->height;
	src_data = (const guint8 *)ximg->data;

	/* Allocate the target RGB image data */
	tar_bpl = width * tar_bpp;
	tar_data = (guint8 *)g_malloc(
		tar_bpl * height * sizeof(guint8)
	);
	if(tar_data == NULL)
	{
		XDestroyImage(ximg);
		return(NULL);
	}

#define GET_COLOR_MASKS(_drawable_,_ximg_)	{	\
 r_mask = (_ximg_)->red_mask;				\
 g_mask = (_ximg_)->green_mask;				\
 b_mask = (_ximg_)->blue_mask;				\
							\
 /* In some cases when the XImage was obtained from	\
  * a Pixmap, the color masks are not set even		\
  * when the Visual of the Pixmap's Screen is		\
  * TrueColor or DirectColor				\
  */							\
 if((r_mask == 0) || (g_mask == 0) || (b_mask == 0)) {	\
  GdkWindowPrivate *private = (GdkWindowPrivate *)(_drawable_); \
  Window xroot;						\
  int xx, xy;						\
  unsigned int xwidth, xheight, xborder, xdepth;	\
  XGetGeometry(						\
   private->xdisplay,					\
   private->xwindow,					\
   &xroot,						\
   &xx, &xy,						\
   &xwidth, &xheight,					\
   &xborder,						\
   &xdepth						\
  );							\
  if(xroot != None) {					\
   Visual *xvisual;					\
   XWindowAttributes xwattr;				\
   XGetWindowAttributes(				\
    private->xdisplay,					\
    xroot,						\
    &xwattr						\
   );							\
   xvisual = xwattr.visual;				\
   if(xvisual != NULL) {				\
    r_mask = xvisual->red_mask;				\
    g_mask = xvisual->green_mask;			\
    b_mask = xvisual->blue_mask;			\
   }							\
  }							\
 }							\
							\
}

	/* Copy/convert the XImage data to the target RGBA image data
	 * by the XImage's bit depth and bits per pixel
	 *
	 * The bits per pixel is the actual number of bits allocated
	 * for each pixel.
	 *
	 * The bit depth is the number of bits defined in each pixel.
	 *
	 * 1 bit depth and 1 bits per pixel (Bitmap)
	 */
	if((ximg->depth == 1) && (ximg->bits_per_pixel == 1))
	{
		const gint	src_bpp = 1,	/* Bitmap consuming 1 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_row;
		guint8 *tar_ptr;

		for(y = 0; y < height; y++)
		{
			src_row = (const guint8 *)(
				src_data + (y * src_bpl) + src_xoffset_bytes
			);
			tar_ptr = tar_data + (y * tar_bpl);
			for(x = 0; x < width; x++)
			{
				if(src_row[x / 8] & (1 << (x % 8)))
				{
					*tar_ptr++ = 0xFF;
					*tar_ptr++ = 0xFF;
					*tar_ptr++ = 0xFF;
				}
				else
				{
					*tar_ptr++ = 0x00;
					*tar_ptr++ = 0x00;
					*tar_ptr++ = 0x00;
				}
			}
		}
	}
	/* Any bit depth and 8 bits per pixel */
	else if(ximg->bits_per_pixel == 8)
	{
		const gint	src_bpp = 1,	/* 1 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			for(y = 0; y < height; y++)
			{
				src_ptr = (const guint8 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);
				for(x = 0; x < width; x++)
				{
					v = (unsigned long)(*src_ptr);
					src_ptr++;
					*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
					*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
					*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
				}
			}
		}
		else
		{
			GdkColormap *colormap = (gdk_window_get_type(drawable) != GDK_WINDOW_PIXMAP) ?
				gdk_window_get_colormap(drawable) :
				gdk_colormap_get_system();

			/* GDK_VISUAL_GRAYSCALE, GDK_VISUAL_STATIC_COLOR, or
			 * GDK_VISUAL_PSEUDO_COLOR
			 */
			if((colormap != NULL) ?
			   ((colormap->colors != NULL) && (colormap->size > 1)) : FALSE
			)
			{
				GdkColor *c;
				const guint32 color_index_highest = colormap->size - 1;
				guint8 color_index;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						color_index = (guint8)MIN(
							(guint32)*src_ptr,
							color_index_highest
						);
						c = &colormap->colors[color_index];
						src_ptr++;
						*tar_ptr++ = (guint8)(c->red >> 8);
						*tar_ptr++ = (guint8)(c->green >> 8);
						*tar_ptr++ = (guint8)(c->blue >> 8);
					}
				}
			}
			/* GDK_VISUAL_STATIC_GRAY */
			else
			{
				guint8 grey8;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						grey8 = *src_ptr++;
						*tar_ptr++ = grey8;
						*tar_ptr++ = grey8;
						*tar_ptr++ = grey8;
					}
				}
			}
		}
	}
	/* Any bit depth and 16 bits per pixel */
	else if(ximg->bits_per_pixel == 16)
	{
		const gint	src_bpp = 2,	/* 2 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8 *src_data = (const guint8 *)ximg->data;
		const guint16 *src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			for(y = 0; y < height; y++)
			{
				src_ptr = (const guint16 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);
				for(x = 0; x < width; x++)
				{
					v = (unsigned long)(*src_ptr);
					src_ptr++;
					*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
					*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
					*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
				}
			}
		}
		else
		{
			GdkColormap *colormap = (gdk_window_get_type(drawable) != GDK_WINDOW_PIXMAP) ?
				gdk_window_get_colormap(drawable) :
				gdk_colormap_get_system();

			/* GDK_VISUAL_GRAYSCALE, GDK_VISUAL_STATIC_COLOR, or
			 * GDK_VISUAL_PSEUDO_COLOR
			 */
			if((colormap != NULL) ?
			   ((colormap->colors != NULL) && (colormap->size > 1)) : FALSE
			)
			{
				GdkColor *c;
				const guint32 color_index_highest = colormap->size - 1;
				guint16 color_index;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint16 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						color_index = (guint16)MIN(
							(guint32)*src_ptr,
							color_index_highest
						);
						c = &colormap->colors[color_index];
						src_ptr++;
						*tar_ptr++ = (guint8)(c->red >> 8);
						*tar_ptr++ = (guint8)(c->green >> 8);
						*tar_ptr++ = (guint8)(c->blue >> 8);
					}
				}
			}
			/* GDK_VISUAL_STATIC_GRAY */
			else
			{
				guint8 grey8;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint16 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						grey8 = (guint8)((*src_ptr) >> 8);
						src_ptr++;
						*tar_ptr++ = grey8;
						*tar_ptr++ = grey8;
						*tar_ptr++ = grey8;
					}
				}
			}
		}
	}
	/* Any bit depth and 24 bits per pixel */
	else if(ximg->bits_per_pixel == 24)
	{
		const gint	src_bpp = 3,	/* 3 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			for(y = 0; y < height; y++)
			{
				src_ptr = (const guint8 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);
				for(x = 0; x < width; x++)
				{
					v = (unsigned long)(src_ptr[0] << 0) |
						(unsigned long)(src_ptr[1] << 8) | 
						(unsigned long)(src_ptr[2] << 16);
					src_ptr += src_bpp;
					*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
					*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
					*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
				}
			}
		}
	}
	/* Any bit depth and 32 bits per pixel */
	else if(ximg->bits_per_pixel == 32)
	{
		const gint	src_bpp = 4,	/* 4 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8 *src_data = (const guint8 *)ximg->data;
		const guint32 *src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			for(y = 0; y < height; y++)
			{
				src_ptr = (const guint32 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);
				for(x = 0; x < width; x++)
				{
					v = (unsigned long)(*src_ptr);
					src_ptr++;
					*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
					*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
					*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
				}
			}
		}
	}
#undef GET_COLOR_MASKS

	XDestroyImage(ximg);

	/* Set the return values */
	if(width_rtn != NULL)
		*width_rtn = width;
	if(height_rtn != NULL)
		*height_rtn = height;
	if(bpl_rtn != NULL)
		*bpl_rtn = tar_bpl;

	return(tar_data);
#else
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;
	return(NULL);
#endif
}

/*
 *	Gets the RGBA image data from a GdkDrawable.
 *
 *	The drawable specifies the GdkDrawable.
 *
 *	If mask is not NULL then the mask specifies the GdkBitmap mask
 *	in which the alpha channel values will be obtained from,
 *	otherwise the alpha channel values will be fully opaque.
 *
 *	The area specifies the area within the GdkDrawable. If area
 *	is NULL then the entire GdkDrawable is obtained.
 *
 *	The width_rtn, height_rtn, and bpl_rtn specifies the
 *	return values for the image data.
 *
 *	Returns a dynamically allocated copy of the image data or NULL
 *	on error.
 */
guint8 *GUIGetRGBAImage(
	GdkDrawable *drawable,
	GdkBitmap *mask,
	const GdkRectangle *area,
	gint *width_rtn, gint *height_rtn, gint *bpl_rtn  
)
{
#if defined(HAVE_X)
	const gint	tar_bpp = 4;		/* RGBA */
	gint		width, height,
			tar_bpl;
	guint8 *tar_data;
	XImage		*ximg,
			*mask_ximg;
	GdkRectangle larea;
	GdkWindowPrivate *private;

	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;

	if(drawable == NULL)
		return(NULL);

	private = (GdkWindowPrivate *)drawable;

	/* If no area was specified then set the area to cover the
	 * entire Drawable
	 */
	if(area == NULL)
	{
		area = &larea;
		larea.x = 0;
		larea.y = 0;
		larea.width = private->width;
		larea.height = private->height;
	}

	if((area->width <= 0) || (area->height <= 0))
		return(NULL);

	/* Get an XImage from the GdkDrawable's Window */
	ximg = XGetImage(
		private->xdisplay,		/* Display */
		private->xwindow,		/* Drawable */
		(int)area->x, (int)area->y,
		(unsigned int)area->width, (unsigned int)area->height,
		AllPlanes,
		ZPixmap
	);
	if(ximg == NULL)
		return(NULL);

	if(ximg->data == NULL)
	{
		XDestroyImage(ximg);
		return(NULL);
	}

	/* Get the size of the GdkDrawable's Window from the XImage */
	width = ximg->width;
	height = ximg->height;

	/* Allocate the target RGBA image data */
	tar_bpl = width * tar_bpp;
	tar_data = (guint8 *)g_malloc(
		tar_bpl * height * sizeof(guint8)
	);
	if(tar_data == NULL)
	{
		XDestroyImage(ximg);
		return(NULL);
	}

	/* Get the mask's XImage data */
	if(mask != NULL)
	{
		/* Get an XImage from the GdkBitmap mask's Window */
		GdkWindowPrivate *private = (GdkWindowPrivate *)mask;
		mask_ximg = XGetImage(
			private->xdisplay,	/* Display */
			private->xwindow,	/* Drawable */
			(int)area->x, (int)area->y,
			(unsigned int)area->width, (unsigned int)area->height,
			AllPlanes,
			ZPixmap
		);
		if(mask_ximg != NULL)
		{
			/* Check if the mask's depth and bits per pixel are
			 * useable to us, we require that its bit depth be 1
			 * and its bits per pixel to be 1
			 */
			if((mask_ximg->data == NULL) || (mask_ximg->depth != 1) ||
			   (mask_ximg->bits_per_pixel != 1)
			)
			{
				XDestroyImage(mask_ximg);
				mask_ximg = NULL;
			}
		}
	}
	else
	{
		mask_ximg = NULL;
	}

/* Gets the color masks r_mask, g_mask, and b_mask, first obtaining
 * them from the XImage, if the XImage did not have the color masks
 * set then it obtains the color masks from the Visual of the
 * specified Drawable, this additional check is needed if the
 * Drawable is a Pixmap in which case the XImage will not contain
 * the color masks
 */
#define GET_COLOR_MASKS(_drawable_,_ximg_)	{	\
 r_mask = (_ximg_)->red_mask;				\
 g_mask = (_ximg_)->green_mask;				\
 b_mask = (_ximg_)->blue_mask;				\
							\
 /* In some cases when the XImage was obtained from	\
  * a Pixmap, the color masks are not set even		\
  * when the Visual of the Pixmap's Screen is		\
  * TrueColor or DirectColor				\
  */							\
 if((r_mask == 0) || (g_mask == 0) || (b_mask == 0)) {	\
  GdkWindowPrivate *private = (GdkWindowPrivate *)(_drawable_); \
  Window xroot;						\
  int xx, xy;						\
  unsigned int xwidth, xheight, xborder, xdepth;	\
  XGetGeometry(						\
   private->xdisplay,					\
   private->xwindow,					\
   &xroot,						\
   &xx, &xy,						\
   &xwidth, &xheight,					\
   &xborder,						\
   &xdepth						\
  );							\
  if(xroot != None) {					\
   Visual *xvisual;					\
   XWindowAttributes xwattr;				\
   XGetWindowAttributes(				\
    private->xdisplay,					\
    xroot,						\
    &xwattr						\
   );							\
   xvisual = xwattr.visual;				\
   if(xvisual != NULL) {				\
    r_mask = xvisual->red_mask;				\
    g_mask = xvisual->green_mask;			\
    b_mask = xvisual->blue_mask;			\
   }							\
  }							\
 }							\
							\
}

	/* Copy/convert the XImage data to the target RGBA image data
	 * by the XImage's bit depth and bits per pixel
	 *
	 * The bits per pixel is the actual number of bits allocated
	 * for each pixel.
	 *
	 * The bit depth is the number of bits defined in each pixel.
	 *
	 * 1 bit depth and 1 bits per pixel (Bitmap)
	 */
	if((ximg->depth == 1) && (ximg->bits_per_pixel == 1))
	{
		const gint	src_bpp = 1,		/* Bitmap consuming 1 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_row;
		guint8 *tar_ptr;
		if(mask_ximg != NULL)
		{
			const gint	mask_bpp = 1,
					mask_bpl = mask_ximg->bytes_per_line,
					mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
			const guint8	*mask_data = (const guint8 *)mask_ximg->data,
					*mask_row;
			for(y = 0; y < height; y++)
			{
				src_row = (const guint8 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				mask_row = (const guint8 *)(
					mask_data + (y * mask_bpl) + mask_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);
				for(x = 0; x < width; x++)
				{
					if(src_row[x / 8] & (1 << (x % 8)))
					{
						*tar_ptr++ = 0xFF;
						*tar_ptr++ = 0xFF;
						*tar_ptr++ = 0xFF;
					}
					else
					{
						*tar_ptr++ = 0x00;
						*tar_ptr++ = 0x00;
						*tar_ptr++ = 0x00;
					}
					if(mask_row[x / 8] & (1 << (x % 8)))
						*tar_ptr++ = 0xFF;
					else
						*tar_ptr++ = 0x00;
				}
			}
		}
		else
		{
			for(y = 0; y < height; y++)
			{
				src_row = (const guint8 *)(
					src_data + (y * src_bpl) + src_xoffset_bytes
				);
				tar_ptr = tar_data + (y * tar_bpl);

				for(x = 0; x < width; x++)
				{
					if(src_row[x / 8] & (1 << (x % 8)))
					{
						*tar_ptr++ = 0xFF;
						*tar_ptr++ = 0xFF;
						*tar_ptr++ = 0xFF;
					}
					else
					{
						*tar_ptr++ = 0x00;
						*tar_ptr++ = 0x00;
						*tar_ptr++ = 0x00;
					}
					*tar_ptr++ = 0xFF;
				}
			}
		}
	}
	/* Any bit depth and 8 bits per pixel */
	else if(ximg->bits_per_pixel == 8)
	{
		const gint	src_bpp = 1,		/* 1 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			if(mask_ximg != NULL)
			{
				const gint	mask_bpp = 1,
						mask_bpl = mask_ximg->bytes_per_line,
						mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
				const guint8	*mask_data = (const guint8 *)mask_ximg->data,
						*mask_row;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					mask_row = (const guint8 *)(
						mask_data + (y * mask_bpl) + mask_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						if(mask_row[x / 8] & (1 << (x % 8)))
							*tar_ptr++ = 0xFF;
						else
							*tar_ptr++ = 0x00;
					}
				}
			}
			else
			{
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						*tar_ptr++ = 0xFF;
					}
				}
			}
		}
		else
		{
			GdkColormap *colormap = (gdk_window_get_type(drawable) != GDK_WINDOW_PIXMAP) ?
				gdk_window_get_colormap(drawable) :
				gdk_colormap_get_system();

			/* GDK_VISUAL_GRAYSCALE, GDK_VISUAL_STATIC_COLOR, or
			 * GDK_VISUAL_PSEUDO_COLOR
			 */
			if((colormap != NULL) ?
			   ((colormap->colors != NULL) && (colormap->size > 1)) : FALSE
			)
			{
				GdkColor *c;
				const guint32 color_index_highest = colormap->size - 1;
				guint8 color_index;
				if(mask_ximg != NULL)
				{
					const gint	mask_bpp = 1,
							mask_bpl = mask_ximg->bytes_per_line,
							mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
					const guint8	*mask_data = (const guint8 *)mask_ximg->data,
							*mask_row;
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint8 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						mask_row = (const guint8 *)(
							mask_data + (y * mask_bpl) + mask_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							color_index = (guint8)MIN(
								(guint32)*src_ptr,
								color_index_highest
							);
							c = &colormap->colors[color_index];
							src_ptr++;
							*tar_ptr++ = (guint8)(c->red >> 8);
							*tar_ptr++ = (guint8)(c->green >> 8);
							*tar_ptr++ = (guint8)(c->blue >> 8);
							if(mask_row[x / 8] & (1 << (x % 8)))
								*tar_ptr++ = 0xFF;
							else
								*tar_ptr++ = 0x00;
						}
					}
				}
				else
				{
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint8 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							color_index = (guint8)MIN(
								(guint32)*src_ptr,
								color_index_highest
							);
							c = &colormap->colors[color_index];
							src_ptr++;
							*tar_ptr++ = (guint8)(c->red >> 8);
							*tar_ptr++ = (guint8)(c->green >> 8);
							*tar_ptr++ = (guint8)(c->blue >> 8);
							*tar_ptr++ = 0xFF;
						}
					}
				}
			}
			/* GDK_VISUAL_STATIC_GRAY */
			else
			{
				guint8 grey8;
				if(mask_ximg != NULL)
				{
					const gint	mask_bpp = 1,
							mask_bpl = mask_ximg->bytes_per_line,
							mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
					const guint8	*mask_data = (const guint8 *)mask_ximg->data,
							*mask_row;
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint8 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						mask_row = (const guint8 *)(
							mask_data + (y * mask_bpl) + mask_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							grey8 = *src_ptr++;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							if(mask_row[x / 8] & (1 << (x % 8)))
								*tar_ptr++ = 0xFF;
							else
								*tar_ptr++ = 0x00;
						}
					}
				}
				else
				{
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint8 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							grey8 = *src_ptr++;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = 0xFF;
						}
					}
				}
			}
		}
	}
	/* Any bit depth and 16 bits per pixel */
	else if(ximg->bits_per_pixel == 16)
	{
		const gint	src_bpp = 2,	/* 2 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8 *src_data = (const guint8 *)ximg->data;
		const guint16 *src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			if(mask_ximg != NULL)
			{
				const gint	mask_bpp = 1,
						mask_bpl = mask_ximg->bytes_per_line,
						mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
				const guint8	*mask_data = (const guint8 *)mask_ximg->data,
						*mask_row;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint16 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					mask_row = (const guint8 *)(
						mask_data + (y * mask_bpl) + mask_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						if(mask_row[x / 8] & (1 << (x % 8)))
							*tar_ptr++ = 0xFF;
						else
							*tar_ptr++ = 0x00;
					}
				}
			}
			else
			{
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint16 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						*tar_ptr++ = 0xFF;
					}
				}
			}
		}
		else
		{
			GdkColormap *colormap = (gdk_window_get_type(drawable) != GDK_WINDOW_PIXMAP) ?
				gdk_window_get_colormap(drawable) :
				gdk_colormap_get_system();

			/* GDK_VISUAL_GRAYSCALE, GDK_VISUAL_STATIC_COLOR, or
			 * GDK_VISUAL_PSEUDO_COLOR
			 */
			if((colormap != NULL) ?
			   ((colormap->colors != NULL) && (colormap->size > 1)) : FALSE
			)
			{
				GdkColor *c;
				const guint32 color_index_highest = colormap->size - 1;
				guint16 color_index;
				if(mask_ximg != NULL)
				{
					const gint	mask_bpp = 1,
							mask_bpl = mask_ximg->bytes_per_line,
							mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
					const guint8	*mask_data = (const guint8 *)mask_ximg->data,
							*mask_row;
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint16 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						mask_row = (const guint8 *)(
							mask_data + (y * mask_bpl) + mask_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							color_index = (guint16)MIN(
								(guint32)*src_ptr,
								color_index_highest
							);
							c = &colormap->colors[color_index];
							src_ptr++;
							*tar_ptr++ = (guint8)(c->red >> 8);
							*tar_ptr++ = (guint8)(c->green >> 8);
							*tar_ptr++ = (guint8)(c->blue >> 8);
							if(mask_row[x / 8] & (1 << (x % 8)))
								*tar_ptr++ = 0xFF;
							else
								*tar_ptr++ = 0x00;
						}
					}
				}
				else
				{
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint16 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							color_index = (guint16)MIN(
								(guint32)*src_ptr,
								color_index_highest
							);
							c = &colormap->colors[color_index];
							src_ptr++;
							*tar_ptr++ = (guint8)(c->red >> 8);
							*tar_ptr++ = (guint8)(c->green >> 8);
							*tar_ptr++ = (guint8)(c->blue >> 8);
							*tar_ptr++ = 0xFF;
						}
					}
				}
			}
			/* GDK_VISUAL_STATIC_GRAY */
			else
			{
				guint8 grey8;
				if(mask_ximg != NULL)
				{
					const gint	mask_bpp = 1,
							mask_bpl = mask_ximg->bytes_per_line,
							mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
					const guint8	*mask_data = (const guint8 *)mask_ximg->data,
							*mask_row;
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint16 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						mask_row = (const guint8 *)(
							mask_data + (y * mask_bpl) + mask_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							grey8 = (guint8)((*src_ptr) >> 8);
							src_ptr++;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							if(mask_row[x / 8] & (1 << (x % 8)))
								*tar_ptr++ = 0xFF;
							else
								*tar_ptr++ = 0x00;
						}
					}
				}
				else
				{
					for(y = 0; y < height; y++)
					{
						src_ptr = (const guint16 *)(
							src_data + (y * src_bpl) + src_xoffset_bytes
						);
						tar_ptr = tar_data + (y * tar_bpl);
						for(x = 0; x < width; x++)
						{
							grey8 = (guint8)((*src_ptr) >> 8);
							src_ptr++;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = grey8;
							*tar_ptr++ = 0xFF;
						}
					}
				}
			}
		}
	}
	/* Any bit depth and 24 bits per pixel */
	else if(ximg->bits_per_pixel == 24)
	{
		const gint	src_bpp = 3,		/* 3 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8	*src_data = (const guint8 *)ximg->data,
				*src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			if(mask_ximg != NULL)
			{
				const gint	mask_bpp = 1,
						mask_bpl = mask_ximg->bytes_per_line,
						mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
				const guint8	*mask_data = (const guint8 *)mask_ximg->data,
						*mask_row;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					mask_row = (const guint8 *)(
						mask_data + (y * mask_bpl) + mask_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(src_ptr[0] << 0) |
							(unsigned long)(src_ptr[1] << 8) | 
							(unsigned long)(src_ptr[2] << 16);
						src_ptr += src_bpp;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						if(mask_row[x / 8] & (1 << (x % 8)))
							*tar_ptr++ = 0xFF;
						else
							*tar_ptr++ = 0x00;
					}
				}
			}
			else
			{
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint8 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(src_ptr[0] << 0) |
							(unsigned long)(src_ptr[1] << 8) | 
							(unsigned long)(src_ptr[2] << 16);
						src_ptr += src_bpp;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						*tar_ptr++ = 0xFF;
					}
				}
			}
		}
	}
	/* Any bit depth and 32 bits per pixel */
	else if(ximg->bits_per_pixel == 32)
	{
		const gint	src_bpp = 4,		/* 4 BPP */
				src_bpl = ximg->bytes_per_line,
				src_xoffset_bytes = ximg->xoffset * src_bpp;
		unsigned long	r_mask, g_mask, b_mask;
		gint	x, y;
		const guint8 *src_data = (const guint8 *)ximg->data;
		const guint32 *src_ptr;
		guint8 *tar_ptr;

		GET_COLOR_MASKS(drawable, ximg);

		/* GDK_VISUAL_TRUE_COLOR or GDK_VISUAL_DIRECT_COLOR */
		if((r_mask != 0l) && (g_mask != 0l) && (b_mask != 0l))
		{
			unsigned long v;
			if(mask_ximg != NULL)
			{
				const gint	mask_bpp = 1,
						mask_bpl = mask_ximg->bytes_per_line,
						mask_xoffset_bytes = mask_ximg->xoffset * mask_bpp;
				const guint8	*mask_data = (const guint8 *)mask_ximg->data,
						*mask_row;
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint32 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					mask_row = (const guint8 *)(
						mask_data + (y * mask_bpl) + mask_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						if(mask_row[x / 8] & (1 << (x % 8)))
							*tar_ptr++ = 0xFF;
						else
							*tar_ptr++ = 0x00;
					}
				}
			}
			else
			{
				for(y = 0; y < height; y++)
				{
					src_ptr = (const guint32 *)(
						src_data + (y * src_bpl) + src_xoffset_bytes
					);
					tar_ptr = tar_data + (y * tar_bpl);
					for(x = 0; x < width; x++)
					{
						v = (unsigned long)(*src_ptr);
						src_ptr++;
						*tar_ptr++ = (v & r_mask) * 0xFF / r_mask;
						*tar_ptr++ = (v & g_mask) * 0xFF / g_mask;
						*tar_ptr++ = (v & b_mask) * 0xFF / b_mask;
						*tar_ptr++ = 0xFF;
					}
				}
			}
		}
	}

#undef GET_COLOR_MASKS

	XDestroyImage(ximg);
	if(mask_ximg != NULL)
		XDestroyImage(mask_ximg);

	/* Set the return values */
	if(width_rtn != NULL)
		*width_rtn = width;
	if(height_rtn != NULL)
		*height_rtn = height;
	if(bpl_rtn != NULL)
		*bpl_rtn = tar_bpl;

	return(tar_data);
#else
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;
	return(NULL);
#endif
}
