#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "fsd.h"

#include "images/icon_fonts_20x20.xpm"


typedef struct _FontSelectionDialog	FontSelectionDialog;
#define FONT_SELECTION_DIALOG(p)	((FontSelectionDialog *)(p))

typedef struct _FontPrompt		FontPrompt;
#define FONT_PROMPT(p)			((FontPrompt *)(p))
#define FONT_PROMPT_DATA_KEY		"/FontPrompt"


/*
 *	Flags:
 */
typedef enum {
	FSD_MAPPED			= (1 << 0),
	FSD_REALIZED			= (1 << 1),
	FSD_RESPONSE_OK			= (1 << 8)
} FSDFlags;


/* Font Selection Dialog */
static gint FSDDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void FSDRealizeCB(GtkWidget *widget, gpointer data);
static void FSDOKCB(GtkWidget *widget, gpointer data);
static void FSDCancelCB(GtkWidget *widget, gpointer data);

gint FSDInit(void);
void FSDSetStyle(GtkRcStyle *rc_style);
void FSDSetTransientFor(GtkWidget *w);
gboolean FSDIsQuery(void);
void FSDBreakQuery(void);
gboolean FSDGetResponse(
	const gchar *title,
	const gchar *ok_label,
	const gchar *cancel_label,
	const gchar *start_font_name,
	gchar **font_name_rtn
);
void FSDMap(void);
void FSDUnmap(void);
void FSDShutdown(void);


/* Font Prompt */
static void FSDPromptDestroyCB(gpointer data);
static void FSDPromptSelectCB(GtkWidget *widget, gpointer data);
GtkWidget *FSDPromptNew(
	const gchar *label,
	const gint label_width,
	const gint entry_width,
	void (*func_cb)(GtkWidget *, gpointer),
	gpointer data
);
GtkWidget *FSDPromptNewSimple(
	const gchar *label,
	const gint label_width,
	const gint entry_width
);
GtkWidget *FSDPromptGetEntry(GtkWidget *w);
GtkWidget *FSDPromptGetButton(GtkWidget *w);
const gchar *FSDPromptGetFontName(GtkWidget *w);
void FSDPromptSetFontName(
	GtkWidget *w,
	const gchar *font_name
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Font Selection Dialog:
 */
struct _FontSelectionDialog {

	GtkWidget	*fsd;		/* GtkFontSelectionDialog */
	gint		freeze_count,
			main_level;
	FSDFlags        flags;

	gchar		*response_font_name;

};

static FontSelectionDialog	*font_selection_dialog = NULL;


/*
 *	Font Selection Dialog Prompt:
 */
struct _FontPrompt {

	GtkWidget	*toplevel;
	gint		freeze_count;

	GtkWidget	*label,
			*entry,
			*button;

	gchar		*label_text;

};


/*
 *	FontSelectionDialog "delete_event" signal callback.
 */
static gint FSDDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	FSDCancelCB(widget, data);
	return(TRUE);
}

/*
 *	FontSelectionDialog "realize" signal callback.
 */
static void FSDRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	FontSelectionDialog *fsdd = FONT_SELECTION_DIALOG(data);
	if((widget == NULL) || (fsdd == NULL))
	    return;

	window = widget->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}

	/* Mark the Font Selection Dialog as realized */
	fsdd->flags |= FSD_REALIZED;
}

/*
 *	OK callback.
 */
static void FSDOKCB(GtkWidget *widget, gpointer data)
{
	GtkFontSelectionDialog *fsd;
	FontSelectionDialog *fsdd = FONT_SELECTION_DIALOG(data);
	if(fsdd == NULL)
	    return;

	fsd = GTK_FONT_SELECTION_DIALOG(fsdd->fsd);

	/* Mark that the user clicked on OK */
	fsdd->flags |= FSD_RESPONSE_OK;

#if 0
	/* Check if the selected font was loaded properly */
	font_loaded = (gtk_font_selection_dialog_get_font(fsd) != NULL) ?
	    TRUE : FALSE;
#endif

	/* Get the selected font name */
	g_free(fsdd->response_font_name);
	fsdd->response_font_name = gtk_font_selection_dialog_get_font_name(fsd);

	/* Unmap the dialog */
	FSDUnmap();

	/* Break out of one main loop */
	if(fsdd->main_level > 0)
	{
	    gtk_main_quit();
	    fsdd->main_level--;
	}
}

/*
 *	Cancel callback.
 */
static void FSDCancelCB(GtkWidget *widget, gpointer data)
{
	FontSelectionDialog *fsdd = FONT_SELECTION_DIALOG(data);
	if(fsdd == NULL)
	    return;

	/* Mark that the user clicked on cancel */
	fsdd->flags &= ~FSD_RESPONSE_OK;

	/* Unmap the dialog */
	FSDUnmap();

	/* Break out of one main loop */
	if(fsdd->main_level > 0)
	{
	    gtk_main_quit();
	    fsdd->main_level--;
	}
}           


/*
 *	Initializes Font Selection Dialog.
 */
int FSDInit(void)
{
	GtkWidget *w;
	GtkFontSelectionDialog *fsd;
	FontSelectionDialog *fsdd;

        /* Already initialized? */
        if(font_selection_dialog != NULL)
            return(0);

        /* Create a new Font Selection Dialog */
        font_selection_dialog = fsdd = FONT_SELECTION_DIALOG(g_malloc0(
	    sizeof(FontSelectionDialog)
	));
        if(fsdd == NULL)
            return(-3);

/*
	fsdd->freeze_count = 0;
	fsdd->main_level = 0;
	fsdd->flags = 0;
	fsdd->response_font_name = NULL;
 */

	fsdd->freeze_count++;

	/* GtkFontSelectionDialog */
	fsdd->fsd = w = gtk_font_selection_dialog_new("Select Font");
	fsd = GTK_FONT_SELECTION_DIALOG(w);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(FSDDeleteEventCB), fsdd
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(FSDRealizeCB), fsdd
	);

	/* Get GtkFontSelection widget */
	w = fsd->fontsel;

/* Connect some signals to the GtkFontSelection widget? */

	/* OK GtkButton */
	gtk_signal_connect(
	    GTK_OBJECT(fsd->ok_button), "clicked",
	    GTK_SIGNAL_FUNC(FSDOKCB), fsdd
	);
	gtk_widget_show(fsd->ok_button);

	/* Hide the Apply GtkButton */
	if(fsd->apply_button != NULL)
	    gtk_widget_hide(fsd->apply_button);

	/* Cancel GtkButton */
	gtk_signal_connect(
	    GTK_OBJECT(fsd->cancel_button), "clicked",
	    GTK_SIGNAL_FUNC(FSDCancelCB), fsdd
	);
	gtk_widget_show(fsd->cancel_button);

	fsdd->freeze_count--;

	return(0);
}

/*
 *	Sets the Font Selection Dialog's style.
 */
void FSDSetStyle(GtkRcStyle *rc_style)
{
	GtkWidget *w;
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	w = fsdd->fsd;
	if(w != NULL)
	{
	    if(rc_style != NULL)
	    {
		gtk_widget_modify_style(w, rc_style);
	    }
	    else
	    {
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style_recursive(w, rc_style);
		GTK_RC_STYLE_UNREF(rc_style)
	    }
	}
}

/*
 *	Sets the Font Selection Dialog to be a transient for the
 *	specified GtkWindow.
 *
 *	If w is NULL then transient for will be unset.
 */
void FSDSetTransientFor(GtkWidget *w)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	if(fsdd->fsd != NULL)
	{
	    if(w != NULL)
	    {
		/* Given widget if not NULL, must be a window */
		if(!GTK_IS_WINDOW(GTK_OBJECT(w)))
		    return;

		gtk_window_set_modal(
		    GTK_WINDOW(fsdd->fsd),
		    TRUE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(fsdd->fsd),
		    GTK_WINDOW(w)
		);
	    }
	    else
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(fsdd->fsd),
		    FALSE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(fsdd->fsd),
		    NULL
		);
	    }
	}
}

/*
 *	Returns TRUE if currently blocking for query.
 */
gboolean FSDIsQuery(void)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return(FALSE);

	return((fsdd->main_level > 0) ? TRUE : FALSE);
}

/*
 *	Ends query if any and returns a not available response.
 */
void FSDBreakQuery(void)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	/* Report that the user canceled */
	fsdd->flags &= ~FSD_RESPONSE_OK;

	/* Break out of an additional blocking loops */
	while(fsdd->main_level > 0)
	{
	    gtk_main_quit();
	    fsdd->main_level--;
	}
	fsdd->main_level = 0;
}

/*
 *	Maps the Font Selection Dialog and sets up the inital values.
 *
 *	Returns TRUE if a font was selected or FALSE if user canceled.
 *
 *	For most values that are set NULL, the value is left unchanged.
 *	All given values are coppied.
 *
 *	All returned pointer values should be considered statically
 *	allocated, do not deallocate them.
 */
gboolean FSDGetResponse(
	const gchar *title,
	const gchar *ok_label,
	const gchar *cancel_label,
	const gchar *start_font_name,
	gchar **font_name_rtn
)
{
	GtkWidget *w;
	FontSelectionDialog *fsdd = font_selection_dialog;

	if(font_name_rtn != NULL)
	    *font_name_rtn = NULL;

	if(fsdd == NULL)
	    return(FALSE);

	/* Already waiting for a query? */
	if(fsdd->main_level > 0)
	    return(FALSE);

	fsdd->flags &= ~FSD_RESPONSE_OK;
	g_free(fsdd->response_font_name);
	fsdd->response_font_name = NULL;

	/* Get the GtkFontSelectionDialog */
	w = fsdd->fsd;

	/* Set the title */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(w),
		title
	    );

	/* Set the initial font name */
	if(start_font_name != NULL)
	    gtk_font_selection_dialog_set_font_name(
		GTK_FONT_SELECTION_DIALOG(w),
		start_font_name
	    );

	/* Map the dialog */
	FSDMap();

	/* Wait for the user's response */
	fsdd->main_level++;
	gtk_main();

	/* Unmap the dialog */
	FSDUnmap();

	/* Break out of an additional blocking loops */
	while(fsdd->main_level > 0)
	{
	    gtk_main_quit();
	    fsdd->main_level--;
	}
	fsdd->main_level = 0;


	/* Begin setting returns */

	/* Font name return */
	if(font_name_rtn != NULL)
	    *font_name_rtn = fsdd->response_font_name;

	return((fsdd->flags & FSD_RESPONSE_OK) ? TRUE : FALSE);
}


/*
 *	Maps the Font Selection Dialog.
 */
void FSDMap(void)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	gtk_widget_show_raise(fsdd->fsd);
	fsdd->flags |= FSD_MAPPED;
}

/*
 *	Unmaps the Font Selection Dialog.
 */
void FSDUnmap(void)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	gtk_widget_hide(fsdd->fsd);
	fsdd->flags &= ~FSD_MAPPED;
}

/*
 *	Shuts down the Font Selection Dialog.
 */
void FSDShutdown(void)
{
	FontSelectionDialog *fsdd = font_selection_dialog;
	if(fsdd == NULL)
	    return;

	fsdd->flags &= ~FSD_RESPONSE_OK;

	/* Break out of any remaining main levels */
	while(fsdd->main_level > 0)
	{
	    gtk_main_quit();
	    fsdd->main_level--;
	}
	fsdd->main_level = 0;

	FSDUnmap();

	/* Mark the Font Selection Dialog as shut down */
	font_selection_dialog = NULL;

	fsdd->freeze_count++;

	/* Destroy the GtkFontSelectionDialog */
	GTK_WIDGET_DESTROY(fsdd->fsd);

	g_free(fsdd->response_font_name);

	fsdd->freeze_count--;

	g_free(fsdd);
}


/*
 *	FSD Prompt GtkObject data "destroy" signal callback.
 */
static void FSDPromptDestroyCB(gpointer data)
{
	FontPrompt *p = FONT_PROMPT(data);
	if(p == NULL)
	    return;

	p->freeze_count++;

	g_free(p->label_text);

	p->freeze_count--;

	g_free(p);
}

/*
 *	FSD Prompt select callback.
 */
static void FSDPromptSelectCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gchar *cur_font_name, *font_name;
	GtkWidget *w, *toplevel;
	FontPrompt *p = FONT_PROMPT(data);
	if((p == NULL) || FSDIsQuery())
	    return;

	if(p->freeze_count > 0)
	    return;

	w = p->entry;
	toplevel = gtk_widget_get_toplevel(p->toplevel);

	cur_font_name = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));

	FSDSetTransientFor(toplevel);
	response = FSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El Tipo De Letra",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Jeu De Caractres Privilgi",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Schriftart Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere Font",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer Lettertype",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Fonte",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Font",
	    "Velg", "Kanseler",
#else
	    "Select Font",
	    "Select", "Cancel",
#endif
	    cur_font_name,
	    &font_name
	);
	FSDSetTransientFor(NULL);
	if(response)
	{
	    if(!STRISEMPTY(font_name))
	    {
		gtk_entry_set_text(GTK_ENTRY(w), font_name);
		gtk_entry_set_position(GTK_ENTRY(w), -1);
	    }
	}

	g_free(cur_font_name);
}

/*
 *	Creates a new FSD Prompt.
 */
GtkWidget *FSDPromptNew(
	const gchar *label,
	const gint label_width,
	const gint entry_width,
	void (*func_cb)(GtkWidget *, gpointer),
	gpointer data
)
{
	const gint border_minor = 2;
	GtkWidget *w, *parent, *parent2;
	FontPrompt *p = FONT_PROMPT(g_malloc0(sizeof(FontPrompt)));

/*
	p->freeze_count = 0;
 */

	p->freeze_count++;

	/* Toplevel GtkBox */
	p->toplevel = w = gtk_hbox_new(FALSE, border_minor);
	gtk_object_set_data_full(
	    GTK_OBJECT(w), FONT_PROMPT_DATA_KEY,
	    p, FSDPromptDestroyCB
	);
	parent = w;

	/* GtkLabel */
	if(!STRISEMPTY(label))
	{
	    w = gtk_alignment_new(
		1.0f, 0.5f,
		0.0f, 0.0f
	    );
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* GtkLabel */
	    p->label = w = gtk_label_new(label);
	    p->label_text = STRDUP(label);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_container_add(GTK_CONTAINER(parent2), w);
	    gtk_widget_show(w);
	}

	/* GtkEntry */
	p->entry = w = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	if(entry_width > 0)
	{
	    gtk_widget_set_usize(w, entry_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	}
	else
	{
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	}
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_READ_ONLY);
	gtk_widget_show(w);

	/* GtkButton */
	p->button = w = GUIButtonPixmap(
	    (guint8 **)icon_fonts_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"El clic para escoger el tipo de letra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le dclic pour choisir le jeu de caractres"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klicken, schriftart auszuwhlen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo scatto di scegliere il font"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klik lettertype te selecteren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O estalido selecionar fonte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klikk velge ut font"
#else
"Click to select font"
#endif
	);
	if(func_cb != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(func_cb), data
	    );
	gtk_widget_show(w);

	p->freeze_count--;

	return(p->toplevel);
}

/*
 *	Creates a new FSD Prompt with the default select signal
 *	connected to the default handler.
 */
GtkWidget *FSDPromptNewSimple(
	const gchar *label,
	const gint label_width,
	const gint entry_width
)
{
	GtkWidget *w = FSDPromptNew(
	    label,
	    label_width,
	    entry_width,
	    NULL, NULL
	);
	FontPrompt *p = FONT_PROMPT(GTK_OBJECT_GET_DATA(
	    w,
	    FONT_PROMPT_DATA_KEY
	));
	if(p->button != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(p->button), "clicked",
		GTK_SIGNAL_FUNC(FSDPromptSelectCB), p
	    );
	return(w);
}

/*
 *	Gets the FSD Prompt's GtkEntry.
 */
GtkWidget *FSDPromptGetEntry(GtkWidget *w)
{
	FontPrompt *p = FONT_PROMPT(GTK_OBJECT_GET_DATA(
	    w,
	    FONT_PROMPT_DATA_KEY
	));
	if(p == NULL)
	    return(NULL);

	return(p->entry);
}

/*
 *	Gets the FSD Prompt's GtkButton.
 */
GtkWidget *FSDPromptGetButton(GtkWidget *w)
{
	FontPrompt *p = FONT_PROMPT(GTK_OBJECT_GET_DATA(
	    w,
	    FONT_PROMPT_DATA_KEY
	));
	if(p == NULL)
	    return(NULL);

	return(p->button);
}

/*
 *	Gets the FSD Prompt's currently selected font name.
 */
const gchar *FSDPromptGetFontName(GtkWidget *w)
{
	FontPrompt *p = FONT_PROMPT(GTK_OBJECT_GET_DATA(
	    w,
	    FONT_PROMPT_DATA_KEY
	));
	if(p == NULL)
	    return(NULL);

	return(gtk_entry_get_text(GTK_ENTRY(p->entry)));
}

/*
 *	Sets the FSD Prompts font name.
 */
void FSDPromptSetFontName(
	GtkWidget *w,
	const gchar *font_name
)
{
	GtkEntry *entry = (GtkEntry *)FSDPromptGetEntry(w);
	if(entry == NULL)
	    return;

	gtk_entry_set_text(
	    entry,
	    (font_name != NULL) ? font_name : ""
	);
	gtk_entry_set_position(entry, -1);
}
