#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../cdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "fsck_win.h"
#include "fsck_win_cb.h"
#include "config.h"


static void fsck_signal_cb(int s);
static gint fsck_timeout_cb(gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


static void fsck_signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
		exit(1);
		break;
	}
}

static gint fsck_timeout_cb(gpointer data)
{
	FSCKWin *fm = FSCK_WIN(data);
	if(fm == NULL)
	{
		gtk_main_quit();
		return(FALSE);
	}

	if(!(fm->flags & FSCK_WIN_MAPPED))
	{
		gtk_main_quit();
		return(FALSE);
	}

	return(TRUE);
}


int main(int argc, char *argv[])
{
	gboolean	initialized_gtk = FALSE,
			auto_start = FALSE;
	gint i;
	const gchar	*arg,
			*startup_device = NULL;
	GList *devices_list;
	EDVContext *ctx = NULL;
	FSCKWin *fm;

#define CLEANUP_RETURN(_v_)	{	\
					\
 edv_context_delete(ctx);		\
					\
 return(_v_);				\
}

	/* Set up the signal callbacks */
	signal(SIGINT, fsck_signal_cb);
	signal(SIGTERM, fsck_signal_cb);
	signal(SIGKILL, fsck_signal_cb);
	signal(SIGSEGV, fsck_signal_cb);
	signal(SIGSTOP, fsck_signal_cb);
	signal(SIGCONT, fsck_signal_cb);
	signal(SIGPIPE, fsck_signal_cb);

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			printf("%s", PROG_HELP_MESG);
			CLEANUP_RETURN(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
				!g_strcasecmp(arg, "-version")
		)
		{
			printf(
				"%s %s\n%s",
				PROG_NAME, PROG_VERSION, PROG_COPYRIGHT
			);
			CLEANUP_RETURN(0);
		}
		/* Auto start */
		else if(!g_strcasecmp(arg, "--start") ||
				!g_strcasecmp(arg, "-start") ||
				!strcmp(arg, "--s") ||
				!strcmp(arg, "-s")
		)
		{
			auto_start = TRUE;
		}
		/* Skip these arguments so that gtk_window_apply_args()
		 * handles them
		 */
		else if(gtk_is_window_arg(arg))
		{
			i++;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;
			gchar c;

			while(*v != '\0')
			{
				c = *v;
				if(c == 's')
				{
					auto_start = TRUE;
				}
				else
				{
					g_printerr(
"-%c: Unsupported argument.\n",
						c
					);
					CLEANUP_RETURN(2);
				}
				v++;
			}
		}
		/* Non-option argument? */
		else if((*arg != '-') && (*arg != '+'))
		{
			startup_device = arg;
		}
		else
		{   
			g_printerr(
"%s: Unsupported argument.\n",
				arg        
			);
			CLEANUP_RETURN(2);
		}
	}



	/* Initialize GTK+ as needed */
	if(!initialized_gtk)
	{
		if(!gtk_init_check(&argc, &argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			CLEANUP_RETURN(1);
		}
		initialized_gtk = TRUE;
	}


	/* Initialize the GDK RGB buffers system */
	gdk_rgb_init();


	/* Initialize the dialogs */
	CDialogInit();

	/* Initialize the Endeavour context */
	ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Get the Devices List */
	devices_list = edv_devices_list(ctx);

	/* Update the device mount states and stats */
	edv_devices_list_update_mount_states(ctx);
	edv_devices_list_update_stats(ctx);

	/* Check if no devices have been loaded, suggesting that
	 * either no device references exist in the devices.ini or
	 * that the file does not exist
	 */
	if(devices_list == NULL)
	{
		edv_play_sound_warning(ctx);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
			"No Devices Found",
"No devices were found.\n\
\n\
If you do have devices then you should run\n\
Endeavour Mark II to configure the device\n\
references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	}


	/* Create first and only fsck manager window */
	fm = fsck_win_new(ctx);
	if(fm != NULL)
	{
		GList *glist, *devices_list2 = NULL;

		gtk_window_apply_args(GTK_WINDOW(fm->toplevel), argc, argv);
		fsck_win_map(fm);

		fsck_win_set_busy(fm, TRUE);

		/* Copy and transfer the Devices List to the FSCK Manager */
		devices_list2 = NULL;
		for(glist = devices_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
			devices_list2 = g_list_append(
				devices_list2,
				edv_device_copy(EDV_DEVICE(glist->data))
			);
		fsck_win_set_devices(fm, devices_list2);
		devices_list2 = NULL;

		fsck_win_set_busy(fm, FALSE);

		/* Select startup device? */
		if(startup_device != NULL)
		{
			gboolean device_selected = FALSE;
			gint row, column;
			gchar *cell_text;
			guint8 spacing;
			GdkPixmap *pixmap;
			GdkBitmap *mask;
			GtkCList *clist = (GtkCList *)fm->devices_clist;
			if(clist != NULL)
			{
				/* Search column 1, the device path column, for
				 * a cell who's value matches startup_device.
				 */
				column = 1;
				for(row = 0; row < clist->rows; row++)
				{
					cell_text = NULL;
					switch((gint)gtk_clist_get_cell_type(clist, row, column))
					{
					  case GTK_CELL_TEXT:
						gtk_clist_get_text(clist, row, column, &cell_text);
						break;

					  case GTK_CELL_PIXTEXT:
						gtk_clist_get_pixtext(
							clist, row, column, &cell_text,
							&spacing, &pixmap, &mask
						);
						break;
					}
					/* Got match? */
					if((cell_text != NULL) ?
						!strcmp(startup_device, cell_text) : FALSE
					)
					{
						gtk_clist_select_row(clist, row, column);
						device_selected = TRUE;
						break;
					}
				}
			}

			/* Auto start and a device was selected? */
			if(auto_start && device_selected)
			{
				fsck_win_start_cb(fm->start_btn, fm);
			}
		}


		gtk_timeout_add(
			1000,		/* ms */
			(GtkFunction)fsck_timeout_cb,
			fm
		);
		gtk_main();

		fsck_win_delete(fm);
	}

	edv_context_sync(ctx);

	/* Shutdown the dialogs */
	CDialogShutdown();

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
