#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

#include "../guiutils.h"
#include "../animicon.h"

#include "../libendeavour2-base/endeavour2.h"

#include "fsck_win.h"
#include "fsck_win_cb.h"

#include "config.h"

#include "../images/icon_run_20x20.xpm"
#include "../images/icon_stop_20x20.xpm"
#include "../images/icon_clear_20x20.xpm"
#include "../images/icon_close_20x20.xpm"

#include "../animations/disk_icon_00_48x48.xpm"
#include "../animations/disk_icon_01_48x48.xpm"
#include "../animations/disk_icon_02_48x48.xpm"


void fsck_win_set_devices(
	FSCKWin *fm,
	GList *devices_list
);
void fsck_win_append_message(
	FSCKWin *fm,
	GdkFont *font,
	GdkColor *color_fg,
	GdkColor *color_bg,
	const gchar *text,
	const gboolean allow_auto_scroll
);

FSCKWin *fsck_win_new(EDVContext *ctx);
void fsck_win_update_display(FSCKWin *fm);
void fsck_win_set_busy(
	FSCKWin *fm,
	const gboolean busy
);
void fsck_win_set_progress(
	FSCKWin *fm,
	const gfloat value,
	const gboolean allow_gtk_iteration
);
void fsck_win_map(FSCKWin *fm);
void fsck_win_unmap(FSCKWin *fm);
void fsck_win_delete(FSCKWin *fm);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)

#define KILL(p,s)	(((p) > 0) ? kill((p), (s)) : -1)


#define FSCK_WIN_WIDTH		640
#define FSCK_WIN_HEIGHT		480

#define EDV_STATUS_BAR_HEIGHT		26


/*
 *	Loads the given list of devices to the FSCK Manager's
 *	devices listing and devices clist.
 *
 *	The devices_list specifies a GList of EDVDevice *
 *	Devices. This list should not be referenced again after this
 *	call.
 */
void fsck_win_set_devices(
	FSCKWin *fm,
	GList *devices_list
)
{
	gint		i,
					row, column,
					ncolumns,
					dev_num;
	gchar **strv;
	const gchar	*path,
					*cell_text;
	GList *glist;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GtkWidget *w;
	GtkCList *clist;
	EDVDevice *dev;

#define CLEANUP_RETURN	{			\
 if(devices_list != NULL) {			\
  g_list_foreach(				\
   devices_list,				\
   (GFunc)edv_device_delete,			\
   NULL						\
  );						\
  g_list_free(devices_list);			\
  devices_list = NULL;				\
 }						\
											\
 return;					\
}

	if(fm == NULL)
	{
		CLEANUP_RETURN;
	}

	w = fm->devices_clist;
	window = w->window;
	clist = GTK_CLIST(w);
	ncolumns = MAX(clist->columns, 1);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	gtk_clist_freeze(clist);

	/* Clear the GtkCList and delete the Devices List */
	gtk_clist_clear(clist);
	if(fm->devices_list != NULL)
	{
		g_list_foreach(
			fm->devices_list,
			(GFunc)edv_device_delete,
			NULL
		);
		g_list_free(fm->devices_list);
		fm->devices_list = NULL;
	}

	/* Transfer the specified Devices List */
	fm->devices_list = devices_list;
	devices_list = NULL;

	/* Update the clist's listing of devices */
	for(glist = fm->devices_list, dev_num = 0;
		glist != NULL;
		glist = g_list_next(glist), dev_num++
	)
	{
		dev = EDV_DEVICE(glist->data);
		if(dev == NULL)
			continue;

		/* Hide this device from general listing? */
		if(EDV_DEVICE_IS_UNLISTED(dev))
			continue;

		/* Append a new clist row */
		row = gtk_clist_append(clist, strv);

		/* Load the icon */
		path = edv_device_get_best_icon_path(
			dev,
			EDV_ICON_SIZE_20,
			EDV_DEVICE_ICON_STATE_STANDARD,
			FALSE,                          /* Do not allow smaller size */
			TRUE                            /* Allow standard state */
		);
		if(path != NULL)
		{
			pixmap = gdk_pixmap_create_from_xpm(
				window,
				&mask,
				NULL,
				path
			);
		}
		else
		{
			pixmap = NULL;
			mask = NULL;
		}

		/* Set name cell */
		column = 0;
		cell_text = dev->name;
		if((cell_text == NULL) && (dev->device_path != NULL))
		{
			cell_text = strrchr(
				(const char *)dev->device_path,
				G_DIR_SEPARATOR
			);
			if(cell_text != NULL)
				cell_text++;
			else
				cell_text = dev->device_path;
		}
		if(cell_text == NULL)
			cell_text = "(null)";

		if(pixmap != NULL)
			gtk_clist_set_pixtext(
				clist,
				row, column,
				cell_text,
				EDV_LIST_PIXMAP_TEXT_SPACING,
				pixmap, mask
			);
		else
			gtk_clist_set_text(
				clist,
				row, column,
				cell_text
			);

		/* Set device path cell */
		column = 1;
		cell_text = dev->device_path;
		if(cell_text == NULL)
			cell_text = "(null)";

		gtk_clist_set_text(
			clist,
			row, column,
			cell_text
		);


		/* Set mount path cell */
		column = 2;
		cell_text = dev->mount_path;
		if(cell_text == NULL)
			cell_text = "(null)";

		gtk_clist_set_text(
			clist,
			row, column,
			cell_text
		);


		/* Set fstype cell */
		column = 3;
		gtk_clist_set_text(
			clist, row, column,
			(dev->fs_type_name != NULL) ? dev->fs_type_name : ""
		);


		/* Set row data of this new row to be a gint indicating
		 * the device listing index of the device that this row
		 * is displaying.
		 */
		gtk_clist_set_row_data(
			clist,
			row,
			(gpointer)dev_num
		);
	}

	g_free(strv);

	gtk_clist_thaw(clist);

	fsck_win_update_display(fm);

	CLEANUP_RETURN
#undef CLEANUP_RETURN
}

/*
 *	Appends the given text to the output text on the fsck
 *	manager window.
 */
void fsck_win_append_message(
	FSCKWin *fm,
	GdkFont *font,
	GdkColor *color_fg,
	GdkColor *color_bg,
	const gchar *text,
	const gboolean allow_auto_scroll
)
{
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text_w;

	if((fm == NULL) || (text == NULL))
		return;

	w = fm->output_text;
	editable = GTK_EDITABLE(w);
	text_w = GTK_TEXT(w);

	gtk_text_insert(
		text_w,
		font,
		color_fg, color_bg,
		text, -1
	);

	if(allow_auto_scroll)
	{
		GtkAdjustment *adj = text_w->vadj;

		/* Need to scroll down to show text not visible? */
		if(((adj->upper - adj->lower) > adj->page_size) &&
		   ((adj->value + adj->page_size) < adj->upper)
		)
		{
			adj->value = adj->upper - adj->page_size;
			if(adj->value < adj->lower)
				adj->value = adj->lower;
			gtk_signal_emit_by_name(
				GTK_OBJECT(adj), "value_changed"
			);
		}
	}
}


/*
 *	Creates a new FSCK Manager.
 */
FSCKWin *fsck_win_new(EDVContext *ctx)
{
	const gint	border_major = 5,
					border_minor = 2;
	gchar *heading[4];
	GList *glist;
	GtkAdjustment *adj;
	GtkWidget	*w,
					*parent, *parent2, *parent3, *parent4,
					*parent5,
					*toplevel;
	GtkAccelGroup *accelgrp;
	GtkEditable *editable;
	GtkText *text;
	GtkCList *clist;
	FSCKWin *fm;

	if(ctx == NULL)
		return(NULL);

	fm = FSCK_WIN(g_malloc0(sizeof(FSCKWin)));
	if(fm == NULL)
		return(NULL);

	fm->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	fm->accelgrp = accelgrp = gtk_accel_group_new();
/*
	fm->busy_count = 0;
	fm->freeze_count = 0;
 */
	fm->ctx = ctx;
	fm->busy_cur = gdk_cursor_new(GDK_WATCH);
	fm->text_font = gdk_font_load(
		EDV_GDK_FONT_NAME_FIXED_12
	);
/*
	fm->queued_device_to_check = NULL;
	fm->fsck_toid = 0;
	fm->fsck_pid = 0;
	fm->fsck_cstdout = NULL;
	fm->fsck_cstderr = NULL;
	fm->stop_count = 0;
 */
	fm->progress_pos_last = -1.0f;
/*
	fm->devices_list = NULL;
 */

	fm->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(                        
		GTK_WINDOW(w), "fsck", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
	gtk_widget_set_usize(w, FSCK_WIN_WIDTH, FSCK_WIN_HEIGHT);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(fsck_win_realize_cb), fm
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(fsck_win_delete_event_cb), fm
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox for the Devices GtkCList and parameters */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkScrolledWindow for the Devices GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Devices GtkCList */
#ifdef PROG_LANGUAGE_ENGLISH
	heading[0] = "Name";
	heading[1] = "Device Path";
	heading[2] = "Mount Path";
	heading[3] = "Filesystem";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	heading[0] = "El Nombre";
	heading[1] = "El Sendero Del Artefacto";
	heading[2] = "Monte Sendero";
	heading[3] = "Archive Sistema";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	heading[0] = "Nom";
	heading[1] = "Composant";
	heading[2] = "Rpertoire de montage";
	heading[3] = "Systme de fichier";
#endif
	fm->devices_clist = w = gtk_clist_new_with_titles(4, heading);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 150);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(fsck_win_button_press_event_cb), fm
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(fsck_win_select_row_cb), fm
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(fsck_win_unselect_row_cb), fm
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_column_width(clist, 0, 130);
	gtk_clist_set_column_width(clist, 1, 130);
	gtk_clist_set_column_width(clist, 2, 130);
	gtk_clist_set_column_width(clist, 3, 50);
	gtk_widget_show(w);

	/* Right Click GtkMenu */
	if(TRUE)
	{
		guint	accel_key,
					accel_mods;
		const gchar *label;
		guint8 **icon;
		GtkWidget *menu = GUIMenuCreate();
		gpointer data = fm;
		void (*func_cb)(GtkWidget *w, gpointer);

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR		{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

		icon = (guint8 **)icon_run_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
		label = "Start";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		label = "Comienzo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		label = "Dmarrer";
#endif
		accel_key = 0;
		accel_mods = 0;
		func_cb = fsck_win_start_cb;
		ADD_MENU_ITEM_LABEL
		fm->devices_start_mi = w;

		icon = (guint8 **)icon_stop_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
		label = "Stop";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		label = "Parada";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		label = "Arrt";
#endif
		accel_key = 0;
		accel_mods = 0;
		func_cb = fsck_win_stop_cb;
		ADD_MENU_ITEM_LABEL
		fm->devices_stop_mi = w;


		fm->devices_menu = menu;
#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}


	/* Parameters GtkFrame */
	w = gtk_frame_new("Parameters");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Check for bad blocks */
	fm->bad_blocks_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
		"Verifique Bloques Malos"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Vrifier Mauvais Blocs"
#else
		"Check Bad Blocks"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"El cheque para bloques malos (esto puede aumentar el tiempo que verifica)"
#elif defined(PROG_LANGUAGE_FRENCH)
		"La vrifications des mauvais blocs augemente le temps de la vrification"
#else
		"Check this to check for bad blocks (this may increase the checking time)"
#endif
	);
	gtk_widget_show(w);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;

	/* Verbose */
	fm->verbose_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
		"Detallado"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dtails"
#else
		"Verbose"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"Sea detallado"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Plus d'information sur le compte rendu"
#else
		"Check this to display more verbose messages during the check"
#endif
	);
	gtk_widget_show(w);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Animation Icon GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Animation Icon */
	fm->animation_icon = w = AnimIconNew(
		48, 48
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	glist = NULL;
	glist = g_list_append(
		glist,
		disk_icon_00_48x48_xpm
	);
	glist = g_list_append(
		glist,
		disk_icon_01_48x48_xpm
	);
	glist = g_list_append(
		glist,
		disk_icon_02_48x48_xpm
	);
	AnimIconSetFromXPMDataList(
		w,
		glist,
		50l
	);
	g_list_free(glist);
	AnimIconSetRepeating(w, TRUE);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;


	/* Start GtkButton */
	fm->start_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_run_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
		"Comienzo"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dmarrer"
#else
		"Start"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF,
		GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"El comienzo verificando el artefacto escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dmarrer le programme de vrification"
#else
		"Start checking the selected device"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(fsck_win_start_cb), fm
	);
	gtk_accel_group_add(
		accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_show(w);

	/* Stop button */
	fm->stop_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_stop_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
		"Stop",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Parada",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Arrt",
#endif
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF,
		GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Stop the current checking process"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Pare la corriente el proceso que verifica"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Arrter le procd de vrification en cours"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(fsck_win_stop_cb), fm
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_t, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);                          
	GUIButtonLabelUnderline(w, GDK_t);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for output text and buttons */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Frame for Results GtkText */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Resultados"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Rsultats"
#else
		"Results"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent3 = w;

	/* Table for text and scroll bar widgets */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent3 = w;

	fm->output_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(fsck_win_text_event_cb), fm
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(fsck_win_text_event_cb), fm
	);
	text->default_tab_width = 8;
	gtk_text_set_editable(text, FALSE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
		GTK_TABLE(parent3), w,
		0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_READ_ONLY);
	gtk_widget_show(w);

	w = gtk_vscrollbar_new(text->vadj);
	gtk_table_attach(
		GTK_TABLE(parent3), w,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_widget_show(w);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkVBox for the Clear GtkButton */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Clear GtkButton */
	fm->clear_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_clear_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
		"Clear",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Claro",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Effacer",
#endif
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF,
		GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Clear results"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Los resultados claros"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Effacer le compte-rendu"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(fsck_win_clear_cb), fm
	);
	gtk_accel_group_add(
		accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	gtk_widget_show(w);

	/* Vbox for close button */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Close button */
	fm->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
		"Close",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Cierre",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Quitter",
#endif
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF,
		GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Close this window"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Cierre esta ventana"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Fermer cette fentre"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(fsck_win_close_cb), fm
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);                          
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);



	/* Status bar frame */
	w = gtk_frame_new(NULL);
	gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Table in main frame */
	w = gtk_table_new(1, 1, FALSE);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Progress bar */
	adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 0, 0);
	fm->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(w, -1, 20);
	gtk_progress_bar_set_orientation(
		GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
		GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
		GTK_PROGRESS(w), FALSE
	);
	gtk_progress_set_show_text(GTK_PROGRESS(w), FALSE);
	gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 0.0);
	gtk_table_attach(
		GTK_TABLE(parent2), w,
		0, 1, 0, 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		GTK_SHRINK,
		border_minor, border_minor
	);
	gtk_widget_show(w);


	fsck_win_update_display(fm);

	fm->freeze_count--;

	return(fm);
}

/*
 *	Updates widgets to reflect local data values.
 */
void fsck_win_update_display(FSCKWin *fm)
{
	gboolean sensitive;
	GtkWidget *w;
	GtkCList *clist;

	if(fm == NULL)
		return;

#define DO_SET_SENSITIVE	{		\
 if(w != NULL)					\
  gtk_widget_set_sensitive(w, sensitive);	\
}

	sensitive = (fm->fsck_toid > 0) ? FALSE : TRUE;
	w = fm->start_btn;
	DO_SET_SENSITIVE
	w = fm->devices_start_mi;
	DO_SET_SENSITIVE

	sensitive = (fm->fsck_toid > 0) ? TRUE : FALSE;
	w = fm->stop_btn;
	DO_SET_SENSITIVE
	w = fm->devices_stop_mi;
	DO_SET_SENSITIVE


	w = fm->devices_clist;
	clist = (GtkCList *)w;







#undef DO_SET_SENSITIVE
}

/*
 *      Marks the FSCK Manager as busy or ready.
 */
void fsck_win_set_busy(
	FSCKWin *fm,
	const gboolean busy
)
{
	GdkCursor *cur;
	GtkWidget *w;

	if(fm == NULL)
		return;

	w = fm->toplevel;
	if(w != NULL)
	{
		if(busy)
		{
			/* Increase busy count */
			fm->busy_count++;

			/* If already busy then don't change anything */
			if(fm->busy_count > 1)
				return;

			cur = fm->busy_cur;
		}
		else
		{
			/* Reduce busy count */
			fm->busy_count--;
			if(fm->busy_count < 0)
				fm->busy_count = 0;

			/* If still busy do not change anything */
			if(fm->busy_count > 0)
				return;

			cur = NULL;	/* Use default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cur);
			gdk_flush();
		}
	}
}

/*
 *	Updates the progress bar on the FSCK Manager.
 */
void fsck_win_set_progress(
	FSCKWin *fm,
	const gfloat value,
	const gboolean allow_gtk_iteration
)
{
	gfloat _value = value;
	GtkWidget *w;
	GtkProgress *progress;

	if(fm == NULL)
		return;

	w = fm->progress_bar;
	if(w == NULL)
		return;
	progress = GTK_PROGRESS(w);

	/* Negative? Implies just do activity */
	if(_value < 0.0f)
	{
		GtkAdjustment *adj = progress->adjustment;

		/* Get new value based on unknown position */
		_value = gtk_progress_get_value(progress) + 1.0;
		if(_value > adj->upper)
			_value = adj->lower;

		/* Update progress widget for `activity mode' (unknown limit)
		 * and set new value.
		 */
		gtk_progress_set_activity_mode(progress, TRUE);
		gtk_progress_set_show_text(progress, FALSE);
		gtk_progress_set_value(progress, _value);

		/* Reset last progress position to -1.0, indicating no last
		 * position since in unknown time limit we don't keep track of
		 * the last position.
		 */
		fm->progress_pos_last = -1.0;

		/* Process any pending GTK events */
		if(allow_gtk_iteration)
			gtk_events_process();

		return;
	}

	/* Clip value to 0.0, 1.0 */
	if(_value > 1.0)
		_value = 1.0;

	/* Reset last progress position if it is greater than the given
	 * value, implying the progress wraped back to the beginning.
	 */
	if(fm->progress_pos_last > _value)
		fm->progress_pos_last = -1.0;

	/* Check if percent did not sufficiently change (use 0.01
	 * increments).
	 */
	if((fm->progress_pos_last > 0.0) &&
	   ((_value - fm->progress_pos_last) < 0.01)
	)
		return;

	/* Update progress */
	gtk_progress_set_activity_mode(progress, FALSE);
	gtk_progress_set_format_string(
		progress, "%p%%"
	);
	gtk_progress_set_show_text(
		progress, (_value > 0.0) ? TRUE : FALSE
	);
	gtk_progress_bar_update(GTK_PROGRESS_BAR(w), _value);

	/* Record last position */
	fm->progress_pos_last = _value;

	/* Process any pending GTK events */
	if(allow_gtk_iteration)
		gtk_events_process();
}

/*
 *	Maps the FSCK Manager.
 */
void fsck_win_map(FSCKWin *fm)
{
	if(fm == NULL)
		return;

	gtk_widget_show_raise(fm->toplevel);
	fm->flags |= FSCK_WIN_MAPPED;

	gtk_widget_grab_focus(fm->devices_clist);
}

/*
 *      Unmaps the FSCK Manager.
 */
void fsck_win_unmap(FSCKWin *fm)
{
	if(fm == NULL)                                     
		return;

	gtk_widget_hide(fm->toplevel);
	fm->flags &= ~FSCK_WIN_MAPPED;
}

/*
 *	Deletes the FSCK Manager.
 */
void fsck_win_delete(FSCKWin *fm)
{
	GdkColormap *colormap;

	if(fm == NULL)
		return;

	g_list_free(fm->queued_device_to_check);
	fm->queued_device_to_check = NULL;

	fm->fsck_toid = GTK_TIMEOUT_REMOVE(fm->fsck_toid);

	KILL(fm->fsck_pid, SIGTERM);
	fm->fsck_pid = 0;

	(void)FCLOSE(fm->fsck_cstdout);
	fm->fsck_cstdout = NULL;

	(void)FCLOSE(fm->fsck_cstderr);
	fm->fsck_cstderr = NULL;


	GTK_WIDGET_DESTROY(fm->devices_menu);
	GTK_WIDGET_DESTROY(fm->toplevel);

	GTK_ACCEL_GROUP_UNREF(fm->accelgrp);

	colormap = fm->colormap;
	if(colormap != NULL)
	{
		GDK_COLORMAP_FREE_COLOR(colormap, &fm->error_color);
		GDK_COLORMAP_UNREF(colormap);
	}

	GDK_CURSOR_DESTROY(fm->busy_cur);

	(void)GDK_FONT_UNREF(fm->text_font);

	if(fm->devices_list != NULL)
	{
		g_list_foreach(
			fm->devices_list,
			(GFunc)edv_device_delete,
			NULL
		);
		g_list_free(fm->devices_list);
		fm->devices_list = NULL;
	}

	g_free(fm);
}

