#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>

#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../cdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "ff_win.h"
#include "config.h"


static void ff_signal_cb(int s);
static gint ff_timeout_cb(gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	UNIX signal callback.
 */
static void ff_signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
		exit(1);
		break;
	}
}

/*
 *	Main timeout callback.
 */
static gint ff_timeout_cb(gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return(FALSE);

	if(win->flags & FF_WIN_MAPPED)
	{
		return(TRUE);
	}
	else
	{
		gtk_main_quit();
		return(FALSE);
	}
}


int main(int argc, char *argv[])
{
	gboolean	initialized_gtk = FALSE,
			auto_start = FALSE,
			quick_format = FALSE,
			verify = TRUE,
			verbose = TRUE;
	gint i;
	const gchar	*arg,
			*startup_device = NULL,
			*capacity = NULL,
			*filesystem = NULL,
			*volume_label = NULL;
	GList *devices_list;
	FFWinProgram program = FF_WIN_PROGRAM_NONE;
	FFWin *win;
	EDVContext *ctx = NULL;

	/* Set up signal callbacks */
	signal(SIGINT, ff_signal_cb);
	signal(SIGTERM, ff_signal_cb);
	signal(SIGKILL, ff_signal_cb);
	signal(SIGSEGV, ff_signal_cb);
	signal(SIGSTOP, ff_signal_cb);
	signal(SIGCONT, ff_signal_cb);
	signal(SIGPIPE, ff_signal_cb);

#define CLEANUP_RETURN(_v_)	{		\
						\
 edv_context_delete(ctx);			\
						\
 return(_v_);					\
}

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			g_print("%s", PROG_HELP_MESG);
			CLEANUP_RETURN(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
				!g_strcasecmp(arg, "-version")
		)
		{
			g_print("%s %s\n%s", PROG_NAME, PROG_VERSION, PROG_COPYRIGHT);
			CLEANUP_RETURN(0);
		}
		/* Auto start */
		else if(!g_strcasecmp(arg, "--start") ||
				!g_strcasecmp(arg, "-start") ||
				!strcmp(arg, "--s") ||
				!strcmp(arg, "-s")
		)
		{
			auto_start = TRUE;
		}
		/* Quick Format */
		else if(!g_strcasecmp(arg, "--quick-format") ||
				!g_strcasecmp(arg, "-quick-format") ||
				!g_strcasecmp(arg, "--quick_format") ||
				!g_strcasecmp(arg, "-quick_format")
		)
		{
			quick_format = TRUE;
		}
		/* Program */
		else if(!g_strcasecmp(arg, "--program") ||
				!g_strcasecmp(arg, "-program") ||
				!g_strcasecmp(arg, "--p") ||
				!g_strcasecmp(arg, "-p")
		)
		{
			i++;
			if(i < argc)
			{
				arg = argv[i];
				if(!g_strcasecmp(arg, "fdformat"))
					program = FF_WIN_PROGRAM_FDFORMAT;
				else if(!g_strcasecmp(arg, "superformat"))
					program = FF_WIN_PROGRAM_SUPERFORMAT;
				else
					program = FF_WIN_PROGRAM_NONE;
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					arg
				);
				CLEANUP_RETURN(2);
			}
		}
		/* Capacity */
		else if(!g_strcasecmp(arg, "--capacity") ||
				!g_strcasecmp(arg, "-capacity")
		)
		{
			i++;
			if(i < argc)
			{
				capacity = arg = argv[i];
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					arg
				);
				CLEANUP_RETURN(2);
			}
		}
		/* Filesystem */
		else if(!g_strcasecmp(arg, "--filesystem") ||
				!g_strcasecmp(arg, "-filesystem")
		)
		{
			i++;
			if(i < argc)
			{
				filesystem = arg = argv[i];
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					arg
				);
				CLEANUP_RETURN(2);
			}
		}
		/* No Verify */
		else if(!g_strcasecmp(arg, "--no-verify") ||
				!g_strcasecmp(arg, "-no-verify") ||
				!g_strcasecmp(arg, "--no_verify") ||
				!g_strcasecmp(arg, "-no_verify")
		)
		{
			verify = FALSE;
		}
		/* Volume Label */
		else if(!g_strcasecmp(arg, "--volume-label") ||
				!g_strcasecmp(arg, "-volume-label") ||
				!g_strcasecmp(arg, "--volume_label") ||
				!g_strcasecmp(arg, "-volume_label") ||
				!g_strcasecmp(arg, "--label") ||
				!g_strcasecmp(arg, "-label")
		)
		{
			i++;
			if(i < argc)
			{
				volume_label = arg = argv[i];
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					arg
				);
				CLEANUP_RETURN(2);
			}
		}
		/* No Verbose */
		else if(!g_strcasecmp(arg, "--no-verbose") ||
				!g_strcasecmp(arg, "-no-verbose") ||
				!g_strcasecmp(arg, "--no_verbose") ||
				!g_strcasecmp(arg, "-no_verbose")
		)
		{
			verbose = FALSE;
		}
		/* Skip these arguments so that gtk_window_apply_args()
		 * handles them
		 */
		else if(gtk_is_window_arg(arg))
		{
			i++;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;
			gchar c;

			while(*v != '\0')
			{
				c = *v;
				if(c == 's')
				{
					auto_start = TRUE;
				}
				else
				{
					g_printerr(
"-%c: Unsupported argument.\n",
						c  
					);
					CLEANUP_RETURN(2);
				}
				v++;
			}
		}
		/* Non-option argument? */
		else if((*arg != '-') && (*arg != '+'))
		{
			startup_device = arg;
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(2);
		}
	}


	/* Initialize the Endeavour context */
	ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Initialize GTK+ */
	if(!initialized_gtk)
	{
		if(!gtk_init_check(&argc, &argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			CLEANUP_RETURN(1);
		}
		initialized_gtk = TRUE;
	}

	/* Initialize the GDK RGB buffers system */
	gdk_rgb_init();

	/* Initialize the dialogs */
	CDialogInit();


	/* Get the Devices List */
	devices_list = edv_devices_list(ctx);

	/* Update the device mount states and stats */
	edv_devices_list_update_mount_states(ctx);
	edv_devices_list_update_stats(ctx);

	/* Check if no devices have been loaded, suggesting that
	 * either no device references exist in the devices.ini or
	 * that the file does not exist
	 */
	if(devices_list == NULL)
	{
		edv_play_sound_warning(ctx);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
			"No Devices Found",
"No devices were found.\n\
\n\
If you do have devices then you should run\n\
Endeavour Mark II to configure the device\n\
references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	}

	/* No format program specified? */
	if(program == FF_WIN_PROGRAM_NONE)
	{
		/* Autodetect the format program */
		gchar *prog = edv_which(PROG_NAME_SUPERFORMAT);
		if(prog != NULL)
		{
			program = FF_WIN_PROGRAM_SUPERFORMAT;
			g_free(prog);
		}
		else
		{
			prog = edv_which(PROG_NAME_FDFORMAT);
			if(prog != NULL)
			{
				program = FF_WIN_PROGRAM_FDFORMAT;
				g_free(prog);
			}
		}
		/* No format programs found? */
		if(program == FF_WIN_PROGRAM_NONE)
		{
			/* Default to FDFormat even though it does not exist,
			 * a warning will be printed later when the user
			 * attempts to format
			 */
			program = FF_WIN_PROGRAM_FDFORMAT;
		}
	}

	/* Create the FFWin */
	win = ff_win_new(ctx);
	if(win != NULL)
	{
		gboolean device_selected = FALSE;

		gtk_window_apply_args(
			GTK_WINDOW(win->toplevel),
			argc, argv
		);

		ff_win_map(win);

		ff_win_set_busy(win, TRUE);

		/* Set the list of devices, the devices_list will not be
		 * modified or deleted
		 */
		ff_win_set_devices_list(
			win,
			devices_list
		);

		/* Select the startup device */
		if(startup_device != NULL)
			device_selected = ff_win_select_device_by_device_path(
				win,
				startup_device
			);

		/* Set the initial values */
		ff_win_set_quick_format(
			win,
			quick_format
		);
		ff_win_set_program(
			win,
			program
		);
		if(capacity != NULL)
			ff_win_set_capacity(
				win,
				capacity
			);
		if(filesystem != NULL)
			FFWinSelectFilesystem(
				win,
				filesystem
			);
		ff_win_set_verify(
			win,
			verify
		);
		if(volume_label != NULL)
			ff_win_set_volume_label(
				win,
				volume_label
			);
		ff_win_set_verbose(
			win,
			verbose
		);

		/* Auto start and a device was selected? */
		if(auto_start && device_selected)
			ff_win_start(win);

		ff_win_set_busy(win, FALSE);


		gtk_timeout_add(
			1000l,
			ff_timeout_cb, win
		);
		gtk_main();

		ff_win_delete(win);
	}

	/* Shutdown the dialogs */
	CDialogShutdown();

	edv_context_sync(ctx);

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
