#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../libendeavour2-base/endeavour2.h"

#include "../guiutils.h"
#include "../animicon.h"
#include "../pulist.h"
#include "../cdialog.h"

#include "ff_win.h"
#include "ff_win_cb.h"
#include "config.h"

#include "../images/icon_floppy_48x48.xpm"


void ff_win_realize_cb(GtkWidget *widget, gpointer data);
gint ff_win_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint ff_win_button_event(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
gint ff_win_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint ff_win_timeout_cb(gpointer data);

void ff_win_changed_cb(GtkWidget *widget, gpointer data);
void ff_win_popup_list_box_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
);

void ff_win_start_cb(GtkWidget *widget, gpointer data);
void ff_win_stop_cb(GtkWidget *widget, gpointer data);
void ff_win_clear_cb(GtkWidget *widget, gpointer data);
void ff_win_close_cb(GtkWidget *widget, gpointer data);

void ff_win_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void ff_win_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)

#define TERMINATE(p)	(((p) > 0) ? kill((p), SIGTERM) : -1)


/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
void ff_win_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	FFWin *win = FF_WIN(data);
	if((widget == NULL) || (win == NULL))
		return;

	window = widget->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_floppy_48x48_xpm
		);
	}

	if(win->colormap == NULL)
	{
		GdkColor *c;
		GdkColormap *colormap = GDK_COLORMAP_REF(gtk_widget_get_colormap(widget));

		c = (GdkColor *)g_malloc(sizeof(GdkColor));
		c->red	= (guint16)(1.0f * (guint16)-1);
		c->green	= (guint16)(0.0f * (guint16)-1);
		c->blue	= (guint16)(0.0f * (guint16)-1);
		GDK_COLORMAP_ALLOC_COLOR(colormap, c);

		win->error_color = c;

		win->colormap = colormap;
	}

	win->flags |= FF_WIN_REALIZED;
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint ff_win_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	ff_win_close_cb(widget, data);
	return(TRUE);
}

/*
 *	Devices GtkCList "button_press_event" or "button_release_event"
 *	signal callback.
 */
gint ff_win_button_event(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint		status = FALSE,
					row, column;
	GtkCList *clist;
	FFWin *win = FF_WIN(data);
	if((widget == NULL) || (button == NULL) || (win == NULL))
		return(status);

	clist = GTK_CLIST(widget);

	/* Find row and column based on given coordinates */
	if(!gtk_clist_get_selection_info(
		clist,
		button->x, button->y,
		&row, &column
	))
	{
		row = -1;
		column = 0;
	}

	/* Handle by event type */
	switch((gint)button->type)
	{
	  case GDK_BUTTON_PRESS:
		/* Handle by button number */
		switch(button->button)
		{
		  case GDK_BUTTON3:
			gtk_clist_freeze(clist);
			if(!(button->state & GDK_CONTROL_MASK) &&
			   !(button->state & GDK_SHIFT_MASK)
			)
				gtk_clist_unselect_all(clist);
			clist->focus_row = row;
			gtk_clist_select_row(clist, row, column);
			gtk_clist_thaw(clist);

			/* Get right click menu widget and map it */
			gtk_menu_popup(
				GTK_MENU(win->devices_menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);

			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		break;
	}

	return(status);
}

/*
 *	GtkText event signal callback.
 */
gint ff_win_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	GdkEventConfigure *configure;
	GdkEventButton *button;
	GtkText *text;
	FFWin *win = FF_WIN(data);
	if((widget == NULL) || (event == NULL) || (win == NULL))
		return(status);

	etype = (gint)event->type;
	text = GTK_TEXT(widget);

	switch(etype)
	{
	  case GDK_CONFIGURE:
		configure = (GdkEventConfigure *)event;
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Scroll up */ 
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value - inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			GTK_TEXT(widget)->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			/* Scroll down */ 
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value + inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		  case GDK_BUTTON5:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		}
		break;
	}

	return(status);
}


/*
 *	Format timeout callback.
 *
 *	Called while a format or mkfs process is running, the process is
 *	monitored and the appropriate action is taken.
 */
gint ff_win_timeout_cb(gpointer data)
{
	FILE *fp;
	gint pid;
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return(FALSE);

/* Resets the stop count, resets the progress bar, sets format_toid
 * to 0, closes the stdout and stderr streams and removes their
 * respective temporary files, removes the current device from the
 * queued_devices_list list, and starts on to the next stage or
 * formats the next device
 */
#define END_PROCESS_CLEANUP	{			\
 win->stop_count = 0;					\
 ff_win_status_progress(win, 0.0f, FALSE);			\
							\
 win->format_toid = 0;					\
							\
 TERMINATE(win->format_pid);				\
 win->format_pid = 0;					\
							\
 (void)FCLOSE(win->format_cstdout);			\
 win->format_cstdout = NULL;				\
 (void)FCLOSE(win->format_cstderr);			\
 win->format_cstderr = NULL;				\
							\
 AnimIconPause(win->animation_icon);			\
 AnimIconSeek(win->animation_icon, 0);			\
							\
 /* Is there another stage to perform? */		\
 if(win->format_stage < FF_WIN_STAGE_HIGHEST) {	\
  /* Get the list of devices to format */		\
  GList *glist = win->queued_devices_list;		\
  if(glist != NULL) {					\
   /* Get the current device */				\
   const gint row = (gint)glist->data;			\
   EDVDevice *dev = EDV_DEVICE(gtk_clist_get_row_data( \
    GTK_CLIST(win->devices_clist),			\
    row							\
   ));							\
							\
   /* Start the next format stage */			\
   win->format_stage++;					\
   ff_win_start_stage(					\
    win,						\
    dev							\
   );							\
  }							\
 }							\
 /* Already at/past the highest stage, check if	there	\
  * is another queued device to format			\
  */							\
 else if(win->queued_devices_list != NULL) {		\
  /* Remove the last queued device from the list */	\
  GList *glist = win->queued_devices_list,		\
	*next_glist = g_list_next(glist);		\
  win->queued_devices_list = (next_glist != NULL) ?	\
   g_list_copy(next_glist) : NULL;			\
  g_list_free(glist);					\
							\
  /* Check if there is another device to format */	\
  glist = win->queued_devices_list;			\
  if(glist != NULL) {					\
   /* There is another device to format */		\
   const gint row = (gint)glist->data;			\
   EDVDevice *dev = EDV_DEVICE(gtk_clist_get_row_data( \
    GTK_CLIST(win->devices_clist),			\
    row							\
   ));							\
   /* Start formatting the next device */		\
   win->format_stage = FF_WIN_STAGE_FORMAT;		\
   ff_win_start_stage(					\
    win,						\
    dev							\
   );							\
  } else {						\
   /* Done, no more devices to format, append the	\
    * "done" message, play the completed sound and	\
    * raise the FFWin				\
    */							\
   ff_win_append_message(					\
    win,						\
    win->text_font,					\
    NULL, NULL,						\
    "Format done.\n",					\
    TRUE						\
   );							\
   edv_play_sound_completed(win->ctx);			\
   gtk_widget_show_raise(win->toplevel);		\
  }							\
 }							\
							\
 ff_win_update_display(win);				\
													\
}

	/* Need to stop? */
	if(win->stop_count > 0)
	{
		/* First remove all queued devices and set the format
		 * stage to the highest, then END_PROCESS_CLEANUP so that
		 * no other queued devices are processed and everything
		 * is properly stopped
		 */
		g_list_free(win->queued_devices_list);
		win->queued_devices_list = NULL;
		win->format_stage = FF_WIN_STAGE_HIGHEST;

edv_play_sound_warning(win->ctx);
ff_win_append_message(
 win,
 win->text_font,
 win->error_color, NULL,
 "\n*** Format Interrupted! ***\n",
 TRUE
);

		/* Now process end */
		END_PROCESS_CLEANUP
		return(FALSE);
	}

	/* Process no longer running? */
	pid = win->format_pid;
	if(!edv_pid_exists(pid))
		win->format_pid = pid = 0;

	/* Read from stdout output file */
	fp = win->format_cstdout;
	if(fp != NULL)
	{
		gchar *s;
		for(s = edv_stream_read_str(fp, FALSE);
		    s != NULL;
		    g_free(s), s = edv_stream_read_str(fp, FALSE)
		)
		{
			ff_win_append_message(
				win,
				win->text_font,
				NULL, NULL,
				s,
				TRUE
			);
		}
		g_free(s);
	}

	/* Read from stderr output file */
	fp = win->format_cstderr;
	if(fp != NULL)
	{
		gchar *s;
		for(s = edv_stream_read_str(fp, FALSE);
		    s != NULL;
		    g_free(s), s = edv_stream_read_str(fp, FALSE)
		)
		{
			ff_win_append_message(
				win,
				win->text_font,
				NULL, NULL,
				s,
				TRUE
			);
		}
		g_free(s);
	}

	/* Process marked as no longer running? */
	if(pid == 0)
	{
		/* Go on to next format stage, next queued device, or
		 * finish
		 */
		END_PROCESS_CLEANUP

		return(FALSE);
	}

	/* Update the progress value */
	ff_win_status_progress(win, -1.0f, FALSE);

	return(TRUE);
#undef END_PROCESS_CLEANUP
}


/*
 *	Any GtkWidget "changed" or "toggled" signal callback.
 */
void ff_win_changed_cb(GtkWidget *widget, gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	ff_win_update_display(win);
}

/*
 *	Any PopupListBox changed signal callback.
 */
void ff_win_popup_list_box_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	ff_win_update_display(win);
}


/*
 *	Start callback.
 */
void ff_win_start_cb(GtkWidget *widget, gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	/* Still formatting? */
	if((win->format_toid != 0) || (win->format_pid > 0))
		return;

	ff_win_start(win);
}

/*
 *	Stop callback.
 */
void ff_win_stop_cb(GtkWidget *widget, gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	win->stop_count++;
}

/*
 *	Clear callback.
 */
void ff_win_clear_cb(GtkWidget *widget, gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	ff_win_clear_messages(win);
}

/*
 *	Close button callback.
 */
void ff_win_close_cb(GtkWidget *widget, gpointer data)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	/* Still formatting? */
	if((win->format_toid != 0) || (win->format_pid > 0))
		return;

	ff_win_unmap(win);
}


/*
 *	Devices GtkCList "select_row" signal callback.
 */
void ff_win_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVDevice *dev;
	FFWin *win = FF_WIN(data);
	if((clist == NULL) || (win == NULL))
		return;

	if(win->freeze_count > 0)
		return;

	dev = EDV_DEVICE(gtk_clist_get_row_data(
		clist,
		row
	));
	if(dev != NULL)
	{
		const gchar *fs_type_name = dev->fs_type_name;
		GtkWidget	*w = win->filesystem_pulistbox,
					*pulist = PUListBoxGetPUList(w);
		if(!STRISEMPTY(fs_type_name) && (pulist != NULL))
		{
			const gint n = PUListGetTotalItems(pulist);
			gint i;
			const gchar *s;
			for(i = 0; i < n; i++)
			{
				s = (const gchar *)PUListGetItemData(
					pulist,
					i
				);
				if(s == NULL)
					continue;

				if(!strcmp(s, fs_type_name))
				{
					PUListBoxSelect(w, i);
					break;
				}
			}
		}
	}

	if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	)
		gtk_clist_moveto(
			clist,
			row, column,			/* Row, column */
			0.5f, 0.0f			/* Row, column */
		);

	if(event != NULL)
	{
		if(event->type == GDK_2BUTTON_PRESS)
		{
			GdkEventButton *button = (GdkEventButton *)event;
			switch(button->button)
			{
			  case GDK_BUTTON1:
				ff_win_start(win);
				break;
			}
		}
	}

	ff_win_update_display(win);
}

/*
 *	Devices GtkCList "unselect_row" signal callback.
 */
void ff_win_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	FFWin *win = FF_WIN(data);
	if(win == NULL)
		return;

	if(win->freeze_count > 0)
		return;

	ff_win_update_display(win);
}
