#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

#include "../guiutils.h"
#include "../animicon.h"
#include "../pulist.h"
#include "../cdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "ff_win.h"
#include "ff_win_cb.h"

#include "config.h"

#include "../images/icon_run_20x20.xpm"
#include "../images/icon_stop_20x20.xpm"
#include "../images/icon_clear_20x20.xpm"
#include "../images/icon_close_20x20.xpm"

#include "../animations/disk_icon_00_48x48.xpm"
#include "../animations/disk_icon_01_48x48.xpm"
#include "../animations/disk_icon_02_48x48.xpm"


/* Get Supported Filesystems */
static GList *FFWinGetSupportedFilesystems(
	const gchar *sbin_path
);

/* FFWinDeviceParameter */
FFWinDeviceParameter *ff_win_win_device_parameter_new(void);
FFWinDeviceParameter *ff_win_win_device_parameter_copy(FFWinDeviceParameter *p);
void ff_win_win_device_parameter_delete(FFWinDeviceParameter *p);

/* Messages */
void ff_win_append_message(
	FFWin *win,
	GdkFont *font,
	GdkColor *color_fg, GdkColor *color_bg,
	const gchar *msg,
	const gboolean auto_scroll
);
void ff_win_clear_messages(FFWin *win);

/* Start */
void ff_win_start(FFWin *win);
void ff_win_start_stage(
	FFWin *win,
	EDVDevice *dev
);

/* FFWin */
FFWin *ff_win_new(EDVContext *ctx);
void ff_win_set_devices_list(
	FFWin *win,
	GList *devices_list
);
gboolean ff_win_select_device_by_device_path(
	FFWin *win,
	const gchar *device_path
);
void ff_win_set_quick_format(
	FFWin *win,
	const gboolean quick_format
);
void ff_win_set_program(
	FFWin *win,
	const FFWinProgram program
);
void ff_win_set_capacity(
	FFWin *win,
	const gchar *capacity
);
void FFWinSelectFilesystem(
	FFWin *win,
	const gchar *filesystem
);
void ff_win_set_verify(
	FFWin *win,
	const gboolean verify
);
void ff_win_set_volume_label(
	FFWin *win,
	const gchar *label
);
void ff_win_set_verbose(
	FFWin *win,
	const gboolean verbose
);
void ff_win_update_display(FFWin *win);
void ff_win_set_busy(FFWin *win, const gboolean busy);
void ff_win_status_progress(
	FFWin *win,
	const gfloat v,
	const gboolean allow_gtk_iteration
);
void ff_win_map(FFWin *win);
void ff_win_unmap(FFWin *win);
void ff_win_delete(FFWin *win);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)

#define INTERRUPT(p)	(((p) > 0) ? kill((p), SIGINT) : -1)
#define TERMINATE(p)	(((p) > 0) ? kill((p), SIGTERM) : -1)


#define FF_WIN_WIDTH		640
#define FF_WIN_HEIGHT		480

#define EDV_STATUS_BAR_HEIGHT		26


/*
 *	Gets a list of filesystem names from the available mkfs.*
 *	programs currently installed on the system.
 *
 *	Returns a GList of gchar * strings describing the name of
 *	the filesystem.
 */
static GList *FFWinGetSupportedFilesystems(
	const gchar *sbin_path
)
{
	const gchar	*bin_prefix = "mkfs.",
					*name,
					*fs_type_name;
	GList *names_list;
	EDVDirectory *dp = edv_directory_open(
		sbin_path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(dp == NULL)
		return(NULL);

	names_list = NULL;

	for(name = edv_directory_next(dp);
		name != NULL;
		name = edv_directory_next(dp)
	)
	{
		if(strncmp((const char *)name, (const char *)bin_prefix, strlen(bin_prefix)))
			continue;

		fs_type_name = (const gchar *)strchr(
			(const char *)name,
			'.'
		);
		if(fs_type_name == NULL)
			continue;

		fs_type_name++;
		names_list = g_list_append(
			names_list,
			g_strdup(fs_type_name)
		);
	}

	edv_directory_close(dp);

	return(names_list);
}

/*
 *	Creates a new  FFWinDeviceParameter.
 */
FFWinDeviceParameter *ff_win_win_device_parameter_new(void)
{
	return(FF_WIN_DEVICE_PARAMETER(g_malloc0(sizeof(FFWinDeviceParameter))));
}

/*
 *	Coppies the FFWinDeviceParameter.
 */
FFWinDeviceParameter *ff_win_win_device_parameter_copy(FFWinDeviceParameter *p)
{
	FFWinDeviceParameter	*p_src = p,
									*p_tar;
	if(p_src == NULL)
		return(NULL);

	p_tar = ff_win_win_device_parameter_new();
	if(p_tar == NULL)
		return(NULL);

	p_tar->name = STRDUP(p_src->name);
	p_tar->parameters = STRDUP(p_src->parameters);
	p_tar->size = p_src->size;
	p_tar->cylindars = p_src->cylindars;
	p_tar->sectors = p_src->sectors;
	p_tar->heads = p_src->heads;

	return(p_tar);
}

/*
 *	Deletes the FFWinDeviceParameter.
 */
void ff_win_win_device_parameter_delete(FFWinDeviceParameter *p)
{
	if(p == NULL)
		return;

	g_free(p->name);
	g_free(p->parameters);
	g_free(p);
}


/*
 *	Append message.
 */
void ff_win_append_message(
	FFWin *win,
	GdkFont *font,
	GdkColor *color_fg, GdkColor *color_bg,
	const gchar *msg,
	const gboolean auto_scroll
)
{
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text;

	if((win == NULL) || (msg == NULL))
		return;

	w = win->results_text;
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);

	gtk_text_insert(
		text,
		font,
		color_fg, color_bg,
		msg, -1
	);

	if(auto_scroll)
	{
		GtkAdjustment *adj = text->vadj;
		GTK_ADJUSTMENT_SET_VALUE(
			adj,
			adj->upper
		);
	}
}

/*
 *	Clears all the messages.
 */
void ff_win_clear_messages(FFWin *win)
{
	GtkWidget *w;
	GtkEditable *editable;

	if(win == NULL)
		return;

	w = win->results_text;
	editable = GTK_EDITABLE(w);

	gtk_editable_delete_text(editable, 0, -1);
}


/*
 *	Start formatting.
 *
 *	Queues the list of devices to be formatted.
 *
 *	Sets the initial stage and calls ff_win_start_stage() to
 *	start formatting.
 */
void ff_win_start(FFWin *win)
{
	gint		i,
					nselected,
					response;
	gchar		*s,
					*msg;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;

	if(win == NULL)
		return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->devices_clist);

	/* Process already running? */
	if((win->format_toid != 0) || (win->format_pid != 0))
		return;

	/* Nothing selected? */
	if(clist->selection == NULL)
		return;

	glist = clist->selection;
	dev = EDV_DEVICE(gtk_clist_get_row_data(
		clist,
		(gint)glist->data
	));
	if(dev == NULL)
		return;

	/* Query user for one last confirmation about formatting
	 * media and loosing any existing data in the media
	 */
	nselected = g_list_length(clist->selection);
	if(nselected > 1)
		msg = g_strdup_printf(
"Format %i medias?",
			nselected
		);
	else
		msg = g_strdup_printf(
"Format the media in the device \"%s\"?",
			STRISEMPTY(dev->name) ?
				dev->device_path : dev->name
		);
	s = g_strconcat(
		msg,
		"\n\n",
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear los medios borrarn todos datos existentes\n\
en los medios, usted est seguro que usted quiere\n\
continuar formatear?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le formattage dtruira toutes les donnes\n\
existantes sur le disque, tes vous sr de vouloir continuer ?"
#else
"Formatting the media will erase all existing data\n\
on the media, are you sure you want to continue\n\
formatting?"
#endif
		,
		NULL
	);
	if(s != NULL)
	{
		g_free(msg);
		msg = s;
	}
	edv_play_sound_question(win->ctx);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
		"Confirme Formato"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Confirmer Le Formattage"
#else
		"Confirm Format"
#endif
		,
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	if(response != CDIALOG_RESPONSE_YES)
		return;

	/* Update the mount states of each device */
	for(i = 0; i < clist->rows; i++)
		edv_device_update_mount_state(EDV_DEVICE(gtk_clist_get_row_data(
			clist,
			i
		)));

	/* Clear the messages */
	ff_win_clear_messages(win);

	/* Clear the previous list of queued devices */
	if(win->queued_devices_list != NULL)
	{
		g_list_free(win->queued_devices_list);
		win->queued_devices_list = NULL;
	}

	/* Copy the list of selected rows on the Devices GtkCList as
	 * the new list of queued devices
	 */
	win->queued_devices_list = g_list_copy(clist->selection);

	/* Start formatting the first queued device */
	glist = win->queued_devices_list;
	if(glist != NULL)
	{
		const gint row = (gint)glist->data;
		EDVDevice *dev = EDV_DEVICE(gtk_clist_get_row_data(
			clist,
			row
		));

		/* Set the initial stage and start formatting */
		win->format_stage = FF_WIN_STAGE_FORMAT;
		ff_win_start_stage(
			win,
			dev
		);
	}
}

/*
 *	Start the current stage.
 *
 *	Sets up the values to start formatting, calls the format
 *	program, and sets the timeout callback.
 *
 *	The win->format_stage must be set prior to each call to this
 *	function.
 */
void ff_win_start_stage(
	FFWin *win,
	EDVDevice *dev
)
{
	gboolean	quick_format,
					verify,
					verbose;
	guint		pid,
					toid;
	const gchar	*fs_type,
					*volume_label = NULL,
					*shell_cmd,
					*shell_args;
	gchar		*cmd,
					*shell_prog,
					*startup_message;
	GtkWidget *toplevel;
	FFWinProgram program;
	FFWinDeviceParameter *device_parameter;

	if((win == NULL) || (dev == NULL))
		return;

	toplevel = win->toplevel;

	/* Get the format parameters */
	quick_format = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->quick_format_check);
	program = (FFWinProgram)PUListBoxGetSelectedData(win->program_pulistbox);
	device_parameter = FF_WIN_DEVICE_PARAMETER(PUListBoxGetSelectedData(win->capacity_pulistbox));
	fs_type = (const gchar *)PUListBoxGetSelectedData(win->filesystem_pulistbox);
	volume_label = gtk_entry_get_text(GTK_ENTRY(win->volume_label_entry));
	verify = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->verify_check);
	verbose = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->verbose_check);

	/* Convert the fs_type for aliases */
	if(!strcmp((const char *)fs_type, EDV_FS_TYPE_VFAT_NAME))
		fs_type = EDV_FS_TYPE_MSDOS_NAME;

	/* Skip the format stage if quick format is specified */
	if(quick_format)
	{
		if(win->format_stage == FF_WIN_STAGE_FORMAT)
			win->format_stage++;
	}

	/* Check if this device is already mounted, in which case we
	 * cannot format it
	 */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El artefacto `%s' se monta actualmente.\n\
\n\
Usted debe unmount el artefacto primero antes de format.",
#elif defined(PROG_LANGUAGE_FRENCH)
"`%s' d'appareil est actuellement mont.\n\
\n\
Vous devez dmont l'appareil avant de pouvoir le formatter.",
#else
"Device `%s' is currently mounted.\n\
\n\
You must unmount the device before formatting.",
#endif
			dev->device_path
		);
		edv_play_sound_warning(win->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
			"El Format Fall",
#elif defined(PROG_LANGUAGE_FRENCH)
			"Le Format a Echou",
#else
			"Format Failed",
#endif
			msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);
		return;
	}


	/* Check if there is already an format process running */
	pid = win->format_pid;
	if(pid != 0)
	{
		if(edv_pid_exists(pid))
		{
			edv_play_sound_error(win->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
				"Internal Error",
"Starting format while a format process is\n\
already running.",
				NULL,
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			return;
		}
	}

	/* Format the command, program, and startup message  based on
	 * the current stage
	 */
	cmd = NULL;
	startup_message = NULL;
	switch(win->format_stage)
	{
		gchar	*name,
					*dev_path_params,
					*format_prog,
					*mkfs_prog;

	  case FF_WIN_STAGE_FORMAT:
		/* Format the device's path with parameters */
		if(device_parameter != NULL)
		{
			dev_path_params = g_strconcat(
				dev->device_path,
				device_parameter->parameters,
				NULL
			);
		}
		else
		{
			dev_path_params = g_strdup(dev->device_path);
		}

		/* Get the format program and the format the format command */
		name = NULL;
		format_prog = NULL;
		switch(program)
		{
		  case FF_WIN_PROGRAM_NONE:
			break;
		  case FF_WIN_PROGRAM_FDFORMAT:
			name = PROG_NAME_FDFORMAT;
			format_prog = edv_which(name);
			if(format_prog != NULL)
			{
				cmd = g_strdup_printf(
"\"%s\"%s \"%s\"",
					format_prog,
					verify ? "" : " -n",
					dev_path_params
				);
			}
			break;
		  case FF_WIN_PROGRAM_SUPERFORMAT:
			name = PROG_NAME_SUPERFORMAT;
			format_prog = edv_which(name);
			if(format_prog != NULL)
			{
				cmd = g_strdup_printf(
"\"%s\" \"%s\"%s",
					format_prog,
					dev_path_params,
					verify ? "" : " --noverify"
				);
			}
			break;
		}

		/* No format programs installed? */
		if(name == NULL)
		{
			edv_play_sound_warning(win->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
				"No Format Program Specified",
"No format program was specified.",
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(cmd);
			g_free(dev_path_params);
			g_free(format_prog);
			return;
		}
		/* Unable to find the format program? */
		if((format_prog == NULL) || (cmd == NULL))
		{
			gchar *msg = g_strdup_printf(
"Unable to find the format program:\n\
\n\
    %s",
				name
			);
			edv_play_sound_warning(win->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
				"Unable To Find Format Program",
				msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(msg);
			g_free(cmd);
			g_free(dev_path_params);
			g_free(format_prog);
			return;
		}
		startup_message = g_strdup_printf(
"-------------------------------------------------\n\
Formatting the media in the device \"%s\"...\n",
			STRISEMPTY(dev->name) ?
				dev->device_path : dev->name
		);
		g_free(dev_path_params);
		g_free(format_prog);
		break;

	  case FF_WIN_STAGE_MKFS:
		/* Format the device's path with parameters */
		if(device_parameter != NULL)
		{
			dev_path_params = g_strconcat(
				dev->device_path,
				device_parameter->parameters,
				NULL
			);
		}
		else
		{
			dev_path_params = g_strdup(dev->device_path);
		}

		/* Determine the MKFS program appropriate for the selected
		 * filesystem
		 */
		name = g_strconcat(
			PROG_NAME_MKFS,
			".",
			fs_type,
			NULL
		);
		/* Check if this MKFS program exists */
		mkfs_prog = edv_which(name);
		if(mkfs_prog == NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to find the MKFS program:\n\
\n\
    %s",
				name
			);
			edv_play_sound_warning(win->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
				"Unable To Find MKFS Program",
				msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(msg);
			g_free(dev_path_params);
			g_free(name);
			g_free(mkfs_prog);
			return;
		}

		g_free(name);

		/* Format the MKFS command */
		cmd = g_strdup_printf(
			"\"%s\" \"%s\" %s",
			mkfs_prog,
			dev_path_params,
			verbose ? " -v" : ""
		);
		/* Append the volume label argument? */
		if(!STRISEMPTY(volume_label) &&
		   !strcmp(fs_type, EDV_FS_TYPE_MSDOS_NAME)
		)
		{
			gchar	*s,
					*dvolume_label = g_strdup(volume_label);
			/* No more than 11 characters */
			if(STRLEN(dvolume_label) > 11)
				dvolume_label[11] = '\0';
			s = g_strconcat(
				cmd,
				" -n \"",
				dvolume_label,
				"\"",
				NULL
			);
			g_free(dvolume_label);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}
		}
		startup_message = g_strdup_printf(
"-------------------------------------------------\n\
Creating the \"%s\" filesystem...\n",
			fs_type
		);
		g_free(dev_path_params);
		g_free(mkfs_prog);
		break;
	}
	/* Unable to format the command? */
	if(cmd == NULL)
	{
		edv_play_sound_error(win->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
			"Internal Error",
"Invalid format stage specified as the starting\n\
format stage.",
			NULL,
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(cmd);
		g_free(startup_message);
		return;
	}

	/* Print the startup message */
	ff_win_append_message(
		win,
		win->text_font,
		NULL, NULL,
		startup_message,
		TRUE
	);
	g_free(startup_message);
	startup_message = NULL;

	/* Get the shell program and arguments */
	shell_cmd = edv_get_s(win->ctx, EDV_CFG_PARM_PROG_SHELL);
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Close the streams */
	(void)FCLOSE(win->format_cstdout);
	win->format_cstdout = NULL;
	(void)FCLOSE(win->format_cstderr);
	win->format_cstderr = NULL;

	/* Execute the format or mkfs command */
	win->format_pid = pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		NULL,
		&win->format_cstdout,
		&win->format_cstderr
	);

	g_free(shell_prog);

	if(pid < 0)
	{
		/* Execute failed */
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de ejecutar la orden:\n\
\n\
    %s",
#elif defined(PROG_LANGUAGE_FRENCH)
"Impossible d'excuter la commande:\n\
\n\
    %s",
#else
"Unable to execute the command:\n\
\n\
    %s",
#endif
			cmd
		);
		edv_play_sound_warning(win->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
			"El Format Fall",
#elif defined(PROG_LANGUAGE_FRENCH)
			"Le Formattage a Echou",
#else
			"Format Failed",
#endif
			msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);

		(void)FCLOSE(win->format_cstdout);
		win->format_cstdout = NULL;
		(void)FCLOSE(win->format_cstderr);
		win->format_cstderr = NULL;

		g_free(cmd);

		return;
	}

	/* Start playing the animation icon */
	if(!AnimIconIsPlaying(win->animation_icon))
		AnimIconPlay(win->animation_icon);

	/* Schedual the timeout callback to monitor this process */
	if(win->format_toid == 0)
		win->format_toid = toid = gtk_timeout_add(
			50l,				/* Milliseconds */
			ff_win_timeout_cb, win
		);

	ff_win_update_display(win);

	g_free(cmd);
}


/*
 *	Creates a new FFWin.
 */
FFWin *ff_win_new(EDVContext *ctx)
{
	const gint	border_major = 5,
					border_minor = 2;
	gint i;
	gchar *heading[4];
	GList *glist;
	GtkAdjustment *adj;
	GtkWidget	*w,
					*parent, *parent2, *parent3, *parent4,
					*parent5, *parent6,
					*toplevel,
					*sb,
					*pulist;
	GtkAccelGroup *accelgrp;
	GtkEditable *editable;
	GtkText *text;
	GtkCList *clist;
	FFWin *win = FF_WIN(
		g_malloc0(sizeof(FFWin))
	);
	if(win == NULL)
		return(NULL);

	win->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	win->accelgrp = accelgrp = gtk_accel_group_new();
/*
	win->busy_count = 0;
	win->freeze_count = 0;
	win->flags = 0;
 */
	win->ctx = ctx;

	win->busy_cur = gdk_cursor_new(GDK_WATCH);

	win->text_font = gdk_font_load(
		EDV_GDK_FONT_NAME_FIXED_12
	);

/*	win->queued_devices_list = NULL; */
	win->format_stage = FF_WIN_STAGE_FORMAT;
/*
	win->format_toid = 0;
	win->format_pid = 0;
	win->format_cstdout = NULL;
	win->format_stderr = NULL;
	win->stop_count = 0;
 */
	win->last_progress_value = -1.0f;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w), "format", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);           
	gtk_widget_set_usize(w, FF_WIN_WIDTH, FF_WIN_HEIGHT);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(ff_win_realize_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(ff_win_delete_event_cb), win
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Hbox for devices clist, start and stop buttons */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox for the Devices GtkCList and the format parameters */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkScrolledWindow for the Devices GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Devices GtkCList */
#if defined(PROG_LANGUAGE_SPANISH)
	heading[0] = "El Nombre";
	heading[1] = "El Sendero Del Artefacto";
	heading[2] = "Monte Sendero";
	heading[3] = "Archive Sistema";
#elif defined(PROG_LANGUAGE_FRENCH)
	heading[0] = "Nom";
	heading[1] = "Composant";
	heading[2] = "Rpertoire de montage";
	heading[3] = "Systme de fichier";
#else
	heading[0] = "Name";
	heading[1] = "Device Path";
	heading[2] = "Mount Path";
	heading[3] = "Filesystem";
#endif
	win->devices_clist = w = gtk_clist_new_with_titles(4, heading);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 150);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(ff_win_button_event), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(ff_win_select_row_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(ff_win_unselect_row_cb), win
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_width(clist, 0, 130);
	gtk_clist_set_column_width(clist, 1, 130);
	gtk_clist_set_column_width(clist, 2, 130);
	gtk_clist_set_column_width(clist, 3, 50);
	gtk_widget_show(w);


	/* Right Click GtkMenu */
	if(TRUE)
	{
#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR		{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

		guint	accel_key,
					accel_mods;
		const gchar *label;
		guint8 **icon;
		GtkWidget *menu = GUIMenuCreate();
		gpointer data = win;
		void (*func_cb)(GtkWidget *w, gpointer);

		icon = (guint8 **)icon_run_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
			"Comienzo"
#elif defined(PROG_LANGUAGE_FRENCH)
			"Dmarrer"
#else
			"Start"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = ff_win_start_cb;
		ADD_MENU_ITEM_LABEL
		win->devices_start_mi = w;

		icon = (guint8 **)icon_stop_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
			"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
			"Arrt"
#else
			"Stop"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = ff_win_stop_cb;
		ADD_MENU_ITEM_LABEL
		win->devices_stop_mi = w;


		win->devices_menu = menu;
#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}


	/* Frame for format parameters */
	w = gtk_frame_new("Parameters");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Parameters Row 1 GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Quick Format GtkCheckButton */
	win->quick_format_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
		"El Formato Rpido"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Formattage Rapide"
#else
		"Quick Format"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(ff_win_changed_cb), win
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto saltarse la etapa de formato y slo crear sistema del archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Sauter l'tape de formattage et cre seulement le systme de fichier"
#else
"Check this to skip the format stage and only create the filesystem.\
 Useful when you only need to erase all the data on a media that\
 has already been formatted."
#endif
	);
	gtk_widget_show(w);

	/* Program GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Program GtkLabel */
	w = gtk_label_new("Program:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Program PopupListBox */
	win->program_pulistbox = w = PUListBoxNew(
		100, -1
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	PUListBoxSetChangedCB(
		w,
		ff_win_popup_list_box_changed_cb, win
	);
	PUListBoxSetTip(
		w,
"Select the format program"
	);
	gtk_widget_show(w);
	pulist = PUListBoxGetPUList(w);
#define ADD_ITEM(_label_,_program_)	{	\
 i = PUListAddItem(pulist, (_label_));		\
 PUListSetItemData(				\
  pulist,					\
  i,						\
  (gpointer)(_program_)				\
 );						\
}
	ADD_ITEM(
		"none",
		FF_WIN_PROGRAM_NONE
	);
	ADD_ITEM(
		PROG_NAME_FDFORMAT,
		FF_WIN_PROGRAM_FDFORMAT
	);
	ADD_ITEM(
		PROG_NAME_SUPERFORMAT,
		FF_WIN_PROGRAM_SUPERFORMAT
	);
#undef ADD_ITEM
	PUListBoxSetLinesVisible(
		w,
		MIN((i + 1), 10)
	);

	/* Capacity GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Capacity GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Capacidad"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Capacit"
#else
		"Capacity"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Capacity PopupListBox */
	win->capacity_pulistbox = w = PUListBoxNew(
		-1, -1
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
	PUListBoxSetChangedCB(
		w,
		ff_win_popup_list_box_changed_cb, win
	);
	PUListBoxSetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Seleccione la capacidad de los medios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir la capacite des medias"
#else
"Select the capacity of the media"
#endif
	);
	gtk_widget_show(w);
	pulist = PUListBoxGetPUList(w);
	if(pulist != NULL)
	{
		gint	i = 0,
					p_num;
		FFWinDeviceParameter	*p,
									parameters_list[] = DEVICE_PARAMETERS_LIST;
		for(p_num = 0; parameters_list[p_num].name != NULL; p_num++)
		{
			p = &parameters_list[p_num];
			i = PUListAddItem(
				pulist,
				p->name
			);
			if(i < 0)
				break;

			PUListSetItemDataFull(
				pulist,
				i,
				ff_win_win_device_parameter_copy(p),
				(GtkDestroyNotify)ff_win_win_device_parameter_delete
			);
		}

		PUListBoxSetLinesVisible(
			w,
			MIN((i + 1), 10)
		);
	}

	/* Parameters Row 2 GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;


	/* Filesystem GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	/* Filesystem GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Archive El Tipo De Sistema:"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Type de Systme de fichier:"
#else PROG_LANGUAGE_ENGLISH
		"Filesystem:"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Filesystem PopupListBox */
	win->filesystem_pulistbox = w = PUListBoxNew(
		100, -1  
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	PUListBoxSetChangedCB(
		w,
		ff_win_popup_list_box_changed_cb, win
	);
	PUListBoxSetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Escoja el tipo de sistema de archivo para ser creado en los medios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir le type du systme de fichier pour ce support"
#else
"Select the type of filesystem to be created on the media"
#endif
	);
	gtk_widget_show(w);
	pulist = PUListBoxGetPUList(w);
	if(pulist != NULL)
	{
		GList *filesystem_names_list = FFWinGetSupportedFilesystems("/sbin");
		if(filesystem_names_list != NULL)
		{
			gint i = 0;
			const gchar *name;
			GList *glist;

			for(glist = filesystem_names_list;
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				name = (const gchar *)glist->data;
				if(STRISEMPTY(name))
					continue;

				i = PUListAddItem(pulist, name);
				if(i < 0)
					break;

				PUListSetItemDataFull(
					pulist,
					i,
					g_strdup(name),
					(GtkDestroyNotify)g_free
				);

				/* Default to msdos */
				if(!strcmp((const char *)name, "msdos"))
					PUListBoxSelect(w, i);
			}

			g_list_foreach(
				filesystem_names_list,
				(GFunc)g_free,
				NULL
			);
			g_list_free(filesystem_names_list);

			PUListBoxSetLinesVisible(
				w,
				MIN((i + 1), 10)
			);
		}
	}


	/* Verify GtkCheckButton */
	win->verify_check = w = gtk_check_button_new_with_label(
		"Verify"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(ff_win_changed_cb), win
	);
	GUISetWidgetTip(
		w,
"Check this to verify the format. This may lengthen the time it\
 takes to format."
	);
	gtk_widget_show(w);


	/* Volume Label GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Volume Label GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Etiqueta"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Nom"
#else
		"Label"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Volume Label GtkEntry */
	win->volume_label_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre la etiqueta del volumen para ser colocada en los medios despus de formato"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom du volume qui sera enregist sue le support"
#else
"Enter the volume label (up to 11 characters) or leave this blank for no volume label."
#endif
	);
	gtk_widget_show(w);


	/* Verbose GtkCheckButton */
	win->verbose_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
		"Detallado"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Bavard"
#else
		"Verbose"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(ff_win_changed_cb), win
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique este despliegue los mensajes adicionales durante formato"
#elif defined(PROG_LANGUAGE_FRENCH)
"Donner les messages dtaills pendant le formattage"
#else
"Check this display additional detailed messages during the format"
#endif
	);
	gtk_widget_show(w);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkHBox for the Animation Icon */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Animation Icon GtkDrawingArea */
	win->animation_icon = w = AnimIconNew(
		48, 48
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	glist = NULL;
	glist = g_list_append(
		glist,
		disk_icon_00_48x48_xpm
	);
	glist = g_list_append(
		glist,
		disk_icon_01_48x48_xpm
	);
	glist = g_list_append(
		glist,
		disk_icon_02_48x48_xpm
	);
	AnimIconSetFromXPMDataList(
		w,
		glist,
		50l
	);
	g_list_free(glist);
	AnimIconSetRepeating(w, TRUE);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;


	/* Start GtkButton */
	win->start_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_run_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
		"Comienzo"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dmarrage"
#else
		"Start"
#endif
		,
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"El comienzo formatting la media el artefacto escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dmarrage du programme de formattage"
#else
		"Start formatting the media in the selected device"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(ff_win_start_cb), win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Return, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_s, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_show(w);

	/* Stop GtkButton */
	win->stop_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_stop_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
		"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Arrt"
#else
		"Stop"
#endif
		,
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"Pare la corriente el proceso que formatting"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Arrter la commande de formattage"
#else
		"Stop the current formatting process"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(ff_win_stop_cb), win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_t, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_t);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for output text and buttons */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Results GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Resultados"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Compte-rendu"
#else
		"Results"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent3 = w;

	/* Results GtkTable */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Results GtkText */
	win->results_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(ff_win_text_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(ff_win_text_event_cb), win
	);
	text->default_tab_width = 8;
	gtk_text_set_editable(text, FALSE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
		GTK_TABLE(parent3),
		w,
		0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_READ_ONLY);
	gtk_widget_show(w);

	sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	gtk_table_attach(
		GTK_TABLE(parent3),
		sb,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_widget_show(sb);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Clear button GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Clear GtkButton */
	win->clear_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_clear_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
		"Clear",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Claro",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Effacer",
#endif
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Clear results"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Los resultados claros"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Effacer le compte-rendu"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(ff_win_clear_cb), win
	);
	gtk_accel_group_add(
		accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	gtk_widget_show(w);

	/* Vbox for close button */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Close button */
	win->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
		"Close",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Cierre",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Quitter",
#endif
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
		w,
#ifdef PROG_LANGUAGE_ENGLISH
		"Close this window"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Cierre esta ventana"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Fermer cette fentre"
#endif
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(ff_win_close_cb), win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	/* Status Bar GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Status Bar Main GtkTable */
	w = gtk_table_new(1, 1, FALSE);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkProgressBar */
	adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 0, 0);
	win->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(w, -1, 20);
	gtk_progress_bar_set_orientation(
		GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
		GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
		GTK_PROGRESS(w), FALSE
	);
	gtk_progress_set_show_text(GTK_PROGRESS(w), FALSE);
	gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 0.0);
	gtk_table_attach(
		GTK_TABLE(parent2), w,
		0, 1, 0, 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		GTK_SHRINK,
		border_minor, border_minor
	);
	gtk_widget_show(w);


	ff_win_update_display(win);

	return(win);
}

/*
 *	Sets the devices list.
 *
 *	The devices_list specifies a GList of EDVDevice *
 *	Devices. This list will not be modified or deleted by this
 *	function.
 */
void ff_win_set_devices_list(
	FFWin *win,
	GList *devices_list
)
{
	gint		i,
					row, column,
					ncolumns;
	const gchar	*path,
					*text;
	gchar **strv;
	GList *glist;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GtkWidget *w;
	GtkCList *clist;
	EDVDevice *dev;

	if(win == NULL)
		return;

	w = win->devices_clist;
	window = w->window;
	clist = GTK_CLIST(w);
	ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Clear the Devices GtkCList */
	gtk_clist_clear(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Copy the list of devices to the Devices GtkCList */
	for(glist = devices_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		dev = EDV_DEVICE(glist->data);
		if(dev == NULL)
			continue;

		/* Hide this device from general listings? */
		if(EDV_DEVICE_IS_UNLISTED(dev))
			continue;

		/* Append a new clist row */
		row = gtk_clist_append(
			clist,
			strv
		);

		/* Load the icon */
		path = edv_device_get_best_icon_path(
			dev,
			EDV_ICON_SIZE_20,
			EDV_DEVICE_ICON_STATE_STANDARD,
			FALSE,				/* Do not allow smaller size */
			TRUE				/* Allow standard state */
		);
		if(path != NULL)
		{
			pixmap = gdk_pixmap_create_from_xpm(
				window,
				&mask,
				NULL,
				path
			);
		}
		else
		{
			pixmap = NULL;
			mask = NULL;
		}

		/* Name */
		column = 0;
		if(column < ncolumns)
		{
			text = dev->name;
			if(text == NULL)
				text = dev->device_path;
			if(text == NULL)
				text = dev->mount_path;
			if(text != NULL)
				text = g_basename(text);
			else
				text = "";

			if(pixmap != NULL)
				gtk_clist_set_pixtext(
					clist,
					row, column,
					text,
					EDV_LIST_PIXMAP_TEXT_SPACING,
					pixmap, mask
				);
			else
				gtk_clist_set_text(
					clist,
					row, column,
					text
				);
		}

		/* Device Path */
		column = 1;
		if(column < ncolumns)
		{
			text = dev->device_path;
			if(text == NULL)
				text = "";

			gtk_clist_set_text(
				clist,
				row, column,
				text
			);
		}

		/* Mount Path */
		column = 2;
		if(column < ncolumns)
		{
			text = dev->mount_path;
			if(text == NULL)
				text = "";

			gtk_clist_set_text(
				clist,
				row, column,
				text
			);
		}

		/* Filesystem */
		column = 3;
		if(column < ncolumns)
		{
			text = dev->fs_type_name;
			if(text == NULL)
				text = "";

			gtk_clist_set_text(
				clist,
				row, column,
				text
			);
		}

		/* Set the row data as the Device */
		gtk_clist_set_row_data_full(
			clist,
			row,
			edv_device_copy(dev), (GtkDestroyNotify)edv_device_delete
		);
	}

	g_free(strv);

	gtk_clist_thaw(clist);

	ff_win_update_display(win);
}

/*
 *	Selects the device by device path.
 */
gboolean ff_win_select_device_by_device_path(
	FFWin *win,
	const gchar *device_path
)
{
	gboolean device_selected;
	gint		i,
					nrows;
	GtkCList *clist;
	EDVDevice *dev;

	if((win == NULL) || (device_path == NULL))
		return(FALSE);

	clist = GTK_CLIST(win->devices_clist);
	nrows = clist->rows;

	device_selected = FALSE;
	for(i = 0; i < nrows; i++)
	{
		dev = EDV_DEVICE(gtk_clist_get_row_data(
			clist,
			i
		));
		if(dev == NULL)
			continue;

		if(STRISEMPTY(dev->device_path))
			continue;

		if(strcmp(device_path, dev->device_path))
			continue;

		gtk_clist_select_row(
			clist,
			i, 0
		);
		ff_win_update_display(win);
		device_selected = TRUE;
		break;
	}

	return(device_selected);
}

/*
 *	Sets the quick format option.
 */
void ff_win_set_quick_format(
	FFWin *win,
	const gboolean quick_format
)
{
	if(win == NULL)
		return;

	GTK_TOGGLE_BUTTON_SET_ACTIVE(
		win->quick_format_check,
		quick_format
	);
	ff_win_update_display(win);
}

/*
 *	Selects the program.
 */
void ff_win_set_program(
	FFWin *win,
	const FFWinProgram program
)
{
	gint		i,
					nitems;
	GtkWidget *pulist;

	if(win == NULL)
		return;

	pulist = PUListBoxGetPUList(win->program_pulistbox);
	if(pulist == NULL)
		return;

	nitems = PUListGetTotalItems(pulist);
	for(i = 0; i < nitems; i++)
	{
		if(program != (FFWinProgram)PUListGetItemData(pulist, i))
			continue;

		PUListBoxSelect(
			win->program_pulistbox,
			i
		);
		ff_win_update_display(win);
		break;
	}
}

/*
 *	Selects the capacity.
 */
void ff_win_set_capacity(
	FFWin *win,
	const gchar *capacity
)
{
	gint		i,
					nitems;
	GtkWidget *pulist;
	FFWinDeviceParameter *device_parameter;

	if((win == NULL) || (capacity == NULL))
		return;

	pulist = PUListBoxGetPUList(win->capacity_pulistbox);
	if(pulist == NULL)
		return;

	nitems = PUListGetTotalItems(pulist);
	for(i = 0; i < nitems; i++)
	{
		device_parameter = FF_WIN_DEVICE_PARAMETER(PUListGetItemData(
			pulist,
			i
		));
		if(device_parameter == NULL)
			continue;

		if(STRISEMPTY(device_parameter->parameters))
			continue;

		if(strcmp(capacity, device_parameter->parameters))
			continue;

		PUListBoxSelect(
			win->capacity_pulistbox,
			i
		);
		ff_win_update_display(win);
		break;
	}
}

/*
 *	Selects the filesystem.
 */
void FFWinSelectFilesystem(
	FFWin *win,
	const gchar *filesystem
)
{
	gint		i,
					nitems;
	const gchar *filesystem2;
	GtkWidget *pulist;

	if((win == NULL) || (filesystem == NULL))
		return;

	pulist = PUListBoxGetPUList(win->filesystem_pulistbox);
	if(pulist == NULL)
		return;

	nitems = PUListGetTotalItems(pulist);
	for(i = 0; i < nitems; i++)
	{
		filesystem2 = (const gchar *)PUListGetItemData(
			pulist,
			i
		);
		if(filesystem2 == NULL)
			continue;

		if(strcmp(filesystem, filesystem2))
			continue;

		PUListBoxSelect(
			win->filesystem_pulistbox,
			i
		);
		ff_win_update_display(win);
		break;
	}
}

/*
 *	Sets the verify option.
 */
void ff_win_set_verify(
	FFWin *win,
	const gboolean verify
)
{
	if(win == NULL)
		return;

	GTK_TOGGLE_BUTTON_SET_ACTIVE(
		win->verify_check,
		verify
	);
	ff_win_update_display(win);
}

/*
 *	Sets the volume label.
 */
void ff_win_set_volume_label(
	FFWin *win,
	const gchar *label
)
{
	if(win == NULL)
		return;

	gtk_entry_set_text(
		GTK_ENTRY(win->volume_label_entry),
		(label != NULL) ? label : ""
	);
	ff_win_update_display(win);
}

/*
 *	Sets the verbose option.
 */
void ff_win_set_verbose(
	FFWin *win,
	const gboolean verbose
)
{
	if(win == NULL)
		return;

	GTK_TOGGLE_BUTTON_SET_ACTIVE(
		win->verbose_check,
		verbose
	);
	ff_win_update_display(win);
}

/*
 *	Updates widgets to reflect local data values.
 */
void ff_win_update_display(FFWin *win)
{
	gboolean sensitive;
	gint row;
	GList *glist;
	GtkWidget *w;
	GtkCList *clist;

	if(win == NULL)
		return;

	win->freeze_count++;

	w = win->devices_clist;
	clist = GTK_CLIST(w);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Program */
	sensitive = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->quick_format_check) ? FALSE : TRUE;
	gtk_widget_set_sensitive(win->program_pulistbox, sensitive);

	/* Start */
	sensitive = ((win->format_toid == 0) && (row > -1)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(win->start_btn, sensitive);
	gtk_widget_set_sensitive(win->devices_start_mi, sensitive);

	/* Stop */
	sensitive = (win->format_toid > 0) ? TRUE : FALSE;
	gtk_widget_set_sensitive(win->stop_btn, sensitive);
	gtk_widget_set_sensitive(win->devices_stop_mi, sensitive);

	/* Close */
	sensitive = (win->format_toid == 0) ? TRUE : FALSE;
	gtk_widget_set_sensitive(win->close_btn, sensitive);

	win->freeze_count--;
}

/*
 *	Sets the FFWin as busy or ready.
 */
void ff_win_set_busy(FFWin *win, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;

	if(win == NULL)
		return;

	w = win->toplevel;
	if(w != NULL)
	{
		if(busy)
		{
			/* Increase busy count */
			win->busy_count++;

			/* If already busy then don't change anything */
			if(win->busy_count > 1)
				return;

			cur = win->busy_cur;
		}
		else
		{
			/* Reduce busy count */
			win->busy_count--;
			if(win->busy_count < 0)
				win->busy_count = 0;

			/* If still busy do not change anything */
			if(win->busy_count > 0)
				return;

			cur = NULL;	/* Use default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cur);
			gdk_flush();
		}
	}
}

/*
 *	Sets the progress value.
 */
void ff_win_status_progress(
	FFWin *win,
	const gfloat v,
	const gboolean allow_gtk_iteration
)
{
	gfloat _v = v;
	GtkAdjustment *adj;
	GtkWidget *w;
	GtkProgress *progress;
	GtkProgressBar *progress_bar;

	if(win == NULL)
		return;

	w = win->progress_bar;
	progress = GTK_PROGRESS(w);
	progress_bar = GTK_PROGRESS_BAR(w);

	adj = progress->adjustment;

	/* Do activity? */
	if(_v < 0.0f)
	{
		/* Get the new value based on the unknown position */
		_v = gtk_progress_get_value(progress) + 1.0f;
		if(_v > adj->upper)
			_v = adj->lower;

		/* Do activity */
		gtk_progress_set_activity_mode(progress, TRUE);
		gtk_progress_set_show_text(progress, FALSE);
		gtk_progress_set_value(progress, _v);

		/* Reset the last progress position */
		win->last_progress_value = -1.0f;
	}
	else
	{
		/* Clip the value to [0.0, 1.0] */
		if(_v > 1.0f)
			_v = 1.0f;

		/* Reset the last progress position if it is greater than
		 * the new value, implying that the progress value has
		 * wraped back to the beginning
		 */
		if(win->last_progress_value > _v)
			win->last_progress_value = -1.0f;

		/* Check if percent did not sufficiently change (use 0.009
		 * increments)
		 */
		if((win->last_progress_value > 0.0f) &&
		   ((_v - win->last_progress_value) < 0.009f)
		)
			return;

		/* Update the progress */
		gtk_progress_set_activity_mode(
			progress,
			FALSE
		);
		gtk_progress_set_format_string(
			progress,
			"%p%%"
		);
		gtk_progress_set_show_text(
			progress,
			(_v > 0.0f) ? TRUE : FALSE
		);
		gtk_progress_bar_update(
			progress_bar,
			_v
		);

		/* Record the last progress value */
		win->last_progress_value = _v;
	}

	/* Manage any pending GTK events */
	if(allow_gtk_iteration)
		gtk_events_process();
}

/*
 *	Maps the FFWin.
 */
void ff_win_map(FFWin *win)
{
	if(win == NULL)
		return;

	gtk_widget_show_raise(win->toplevel);
	gtk_widget_grab_focus(win->devices_clist);
	win->flags |= FF_WIN_MAPPED;
}

/*
 *	Unmaps the FFWin.
 */
void ff_win_unmap(FFWin *win)
{
	if(win == NULL)
		return;

	gtk_widget_hide(win->toplevel);
	win->flags &= ~FF_WIN_MAPPED;
}

/*
 *	Deletes the FFWin.
 */
void ff_win_delete(FFWin *win)
{
	GdkColormap *colormap;
	GtkCList *clist;

	if(win == NULL)
		return;

	win->freeze_count++;

	if(win->queued_devices_list != NULL)
	{
		g_list_free(win->queued_devices_list);
		win->queued_devices_list = NULL;
	}

	win->format_toid = GTK_TIMEOUT_REMOVE(win->format_toid);

	TERMINATE(win->format_pid);
	win->format_pid = 0;

	(void)FCLOSE(win->format_cstdout);
	win->format_cstdout = NULL;
	(void)FCLOSE(win->format_cstderr);
	win->format_cstderr = NULL;

	clist = GTK_CLIST(win->devices_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	GTK_WIDGET_DESTROY(win->devices_menu);
	GTK_WIDGET_DESTROY(win->toplevel);

	GTK_ACCEL_GROUP_UNREF(win->accelgrp);

	colormap = win->colormap;
	if(colormap != NULL)
	{
#define DELETE_COLOR(_c_)	{		\
 if((_c_) != NULL) {				\
  GDK_COLORMAP_FREE_COLOR(colormap, (_c_));	\
  g_free(_c_);					\
 }						\
}
		DELETE_COLOR(win->error_color);
#undef DELETE_COLOR
		GDK_COLORMAP_UNREF(colormap);
	}

	GDK_CURSOR_DESTROY(win->busy_cur);

	(void)GDK_FONT_UNREF(win->text_font);

	win->freeze_count--;

	g_free(win);
}
