#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "pulist.h"
#include "cfg_gtk.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_date_format.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_cursor.h"
#include "edv_list_cb.h"
#include "edv_status_bar.h"
#include "find_win.h"
#include "find_win_cb.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_search_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_goto_20x20.xpm"
#include "images/icon_copy_20x20.xpm"


/* Search Value */
const gchar *edv_find_win_current_search(EDVFindWin *fw);
void edv_find_win_set_search(
	EDVFindWin *fw,
	const gchar *s,
	const gboolean record_history
);

/* Location Type */
EDVLocationType edv_find_win_current_location_type(EDVFindWin *fw);
void edv_find_win_set_location_type(
	EDVFindWin *fw,
	const EDVLocationType location_type
);

/* Current Location */
const gchar *edv_find_win_current_location(EDVFindWin *fw);
void edv_find_win_set_location(
	EDVFindWin *fw,
	const gchar *path,
	const gboolean record_history
);

/* Find By */
EDVFindWinFindBy edv_find_win_current_find_by(EDVFindWin *fw); 
void edv_find_win_set_find_by(
	EDVFindWin *fw,
	const EDVFindWinFindBy find_by
);

/* Find Using */
EDVFindWinFindUsing edv_find_win_current_find_using(EDVFindWin *fw); 
void edv_find_win_set_find_using(
	EDVFindWin *fw,
	const EDVFindWinFindUsing find_using
);

/* Reference Window */
void edv_find_win_set_reference_window(
	EDVFindWin *fw,
	const gint browser_num,
	const gint imbr_num,
	const gint recbin_num,
	const gint archiver_num
);

/* List */
void edv_find_win_list_reset_columns(
	EDVFindWin *fw,
	const EDVFindWinFindBy find_by
);
void edv_find_win_list_append(
	EDVFindWin *fw,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt
);
void edv_find_win_list_clear(EDVFindWin *fw);

/* Sync Configuration */
void edv_find_win_sync_configuration(EDVFindWin *fw);

/* Find Window Object */
EDVFindWinObject *edv_find_win_object_new(void);
void edv_find_win_object_delete(EDVFindWinObject *obj);

/* Find Window */
EDVFindWin *edv_find_win_new(EDVCore *core);
void edv_find_win_update_display(EDVFindWin *fw);
void edv_find_win_set_busy(EDVFindWin *fw, const gboolean busy);
gboolean edv_find_win_is_mapped(EDVFindWin *fw);
void edv_find_win_map(EDVFindWin *fw);
void edv_find_win_unmap(EDVFindWin *fw);
void edv_find_win_delete(EDVFindWin *fw);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the current search string.
 */
const gchar *edv_find_win_current_search(EDVFindWin *fw)
{
	GtkCombo *combo;

	if(fw == NULL)
		return(NULL);

	combo = GTK_COMBO(fw->search_combo);

	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the search string.
 *
 *	The s specifies the new search string.
 *
 *	If record_history is TRUE then the new search string will be
 *	recorded on the search history.
 */
void edv_find_win_set_search(
	EDVFindWin *fw,
	const gchar *s,
	const gboolean record_history
)
{
	gchar *new_s;
	GtkCombo *combo;

	if((fw == NULL) || STRISEMPTY(s))
		return;

	combo = GTK_COMBO(fw->search_combo);

	/* Make a copy of the new search string */
	new_s = STRDUP(s);
	if(new_s == NULL)
		return;

#if 0
/* This does not work out if we need to record history */
	/* Check for no change in value */
	s = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if((s != NULL) ? !strcmp(s, new_s) : FALSE)
	{
		g_free(new_s);
		return;
	}
#endif

	/* Record history? */
	if(record_history)
		GUIComboAddItem(GTK_WIDGET(combo), new_s);

	/* Set the new search string */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);

	g_free(new_s);
}


/*
 *	Gets the current location type.
 */
EDVLocationType edv_find_win_current_location_type(EDVFindWin *fw)
{
	if(fw == NULL)
		return(EDV_LOCATION_TYPE_VFS);

	return(fw->location_type);

}

/*
 *	Sets the location type.
 */
void edv_find_win_set_location_type(
	EDVFindWin *fw,
	const EDVLocationType location_type
)
{
	if(fw == NULL)
		return;

	if(fw->location_type != location_type)
	{
		fw->location_type = location_type;
		edv_find_win_update_display(fw);
	}
}


/*
 *	Gets the current location.
 */
const gchar *edv_find_win_current_location(EDVFindWin *fw)
{
	GtkCombo *combo;

	if(fw == NULL)
		return(NULL);

	combo = GTK_COMBO(fw->location_combo);

	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the location.
 *
 *	The path specifies the new location.
 *
 *	If record_history is TRUE then the new location will be
 *	recorded in the locations history.
 */
void edv_find_win_set_location(
	EDVFindWin *fw,
	const gchar *path,
	const gboolean record_history
)
{
	gchar *new_path;
	GtkCombo *combo;

	if((fw == NULL) || STRISEMPTY(path))
		return;

	combo = GTK_COMBO(fw->location_combo);

	/* Make a copy of the specified path */
	new_path = STRDUP(path);
	if(new_path == NULL)
		return;

	/* Simplify the new path */
	edv_path_simplify(new_path);

#if 0
/* This does not work out if we need to record history */
	/* Check for no change in value */
	path = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if((path != NULL) ? !strcmp(path, new_path) : FALSE)
	{
		g_free(new_path);
		return;
	}
#endif

	/* Record history? */
	if(record_history)
		GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set the new location */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_path);

	g_free(new_path);
}


/*
 *	Gets the current find by method.
 */
EDVFindWinFindBy edv_find_win_current_find_by(EDVFindWin *fw)
{
	if(fw == NULL)
		return(EDV_FIND_WIN_FIND_BY_NAME);

	return((EDVFindWinFindBy)PUListBoxGetSelectedData(
		fw->find_by_pulistbox
	));
}

/*
 *	Sets the find by method.
 */
void edv_find_win_set_find_by(
	EDVFindWin *fw,
	const EDVFindWinFindBy find_by
)
{
	gint i, n;
	GtkWidget *pulist;

	if(fw == NULL)
		return;

	fw->freeze_count++;

	pulist = PUListBoxGetPUList(fw->find_by_pulistbox);
	n = PUListGetTotalItems(pulist);

	PUListUnselectAll(pulist);

	/* Select the find by item on the find by method popup list */
	for(i = 0; i < n; i++)
	{
		if((EDVFindWinFindBy)PUListGetItemData(pulist, i) == find_by)
		{
			PUListSelect(pulist, i);
			break;
		}
	}

	edv_find_win_update_display(fw);

	fw->freeze_count--;
}


/*
 *	Gets the current find using method.
 */
EDVFindWinFindUsing edv_find_win_current_find_using(EDVFindWin *fw)
{
	if(fw == NULL)
		return(EDV_FIND_WIN_FIND_USING_LITERAL);

	return((EDVFindWinFindUsing)PUListBoxGetSelectedData(
		fw->find_using_pulistbox
	));
}

/*
 *	Sets the find using method.
 */
void edv_find_win_set_find_using(
	EDVFindWin *fw,
	const EDVFindWinFindUsing find_using
)
{
	gint i, n;
	GtkWidget *pulist;

	if(fw == NULL)
		return;

	fw->freeze_count++;

	pulist = PUListBoxGetPUList(fw->find_using_pulistbox);
	n = PUListGetTotalItems(pulist);

	PUListUnselectAll(pulist);

	/* Select the find using item on the find using method popup list */
	for(i = 0; i < n; i++)
	{
		if((EDVFindWinFindUsing)PUListGetItemData(pulist, i) == find_using)
		{
			PUListSelect(pulist, i);
			break;
		}
	}

	edv_find_win_update_display(fw);

	fw->freeze_count--;
}


/*
 *	Sets the reference window and location type.
 *
 *	Only one of the window indices may be non-negative.
 */
void edv_find_win_set_reference_window(
	EDVFindWin *fw,
	const gint browser_num,
	const gint imbr_num,
	const gint recbin_num,
	const gint archiver_num
)
{
	EDVLocationType location_type;

	if(fw == NULL)
		return;

	/* Clear the list */
	edv_find_win_list_clear(fw);

	fw->browser_num = browser_num;
	fw->imbr_num = imbr_num;
	fw->recbin_num = recbin_num;
	fw->archiver_num = archiver_num;

	/* Set the location type based on which reference window was
	 * specified
	 */
	if(browser_num > -1)
		location_type = EDV_LOCATION_TYPE_VFS;
	else if(imbr_num > -1)
		location_type = EDV_LOCATION_TYPE_VFS;
	else if(recbin_num > -1)
		location_type = EDV_LOCATION_TYPE_RECYCLE_BIN;
	else if(archiver_num > -1)
		location_type = EDV_LOCATION_TYPE_ARCHIVE;
	else
		location_type = EDV_LOCATION_TYPE_VFS;
	fw->location_type = location_type;

	/* Update the Results GtkCList's title and DND source types
	 * due to change in the location type
	 */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		gtk_window_set_title(
			GTK_WINDOW(fw->toplevel),
			"Find Objects"
		);
		if(fw->results_clist != NULL)
		{
			const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,    0,      EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST, 0,      EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,        0,      EDV_DND_INFO_STRING}
			};
			GUIDNDSetSrcTypes(
				fw->results_clist,
				dnd_src_types,
				sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
				GDK_ACTION_COPY | GDK_ACTION_MOVE |
					GDK_ACTION_LINK,
				GDK_BUTTON1_MASK
			);
		}
		break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		gtk_window_set_title(
			GTK_WINDOW(fw->toplevel),
			"Find Recycled Objects"
		);
		if(fw->results_clist != NULL)
		{
			const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT}
			};
			GUIDNDSetSrcTypes(
				fw->results_clist,
				dnd_src_types,
				sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
				GDK_ACTION_MOVE,
				GDK_BUTTON1_MASK
			);
		}
		break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
		gtk_window_set_title(
			GTK_WINDOW(fw->toplevel),
			"Find Archive Objects"
		);
		if(fw->results_clist != NULL)
		{
			const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_ARCHIVE_OBJECT, 0,      EDV_DND_INFO_ARCHIVE_OBJECT}
			};
			GUIDNDSetSrcTypes(
				fw->results_clist,
				dnd_src_types,
				sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
				GDK_ACTION_COPY,
				GDK_BUTTON1_MASK
			);
		}
		break;
	}
}


/*
 *	Resets the Results List column attributes.
 *
 *	The find_by specifies the find by method.
 */
void edv_find_win_list_reset_columns(
	EDVFindWin *fw,
	const EDVFindWinFindBy find_by
)
{
	gint i, ncolumns;
	gchar **title;
	gint *justify;
	GtkRcStyle *lists_rcstyle;
	GtkWidget *w;
	GtkCList *clist;
	EDVCore *core;

	if(fw == NULL)
		return;

	w = fw->results_clist;
	clist = GTK_CLIST(w);
	ncolumns = MAX(clist->columns, 1);
	core = fw->core;
	lists_rcstyle = core->lists_rcstyle;

	/* Allocate the column property values */
	title = (gchar **)g_malloc0(ncolumns * sizeof(gchar *));
	justify = (gint *)g_malloc0(ncolumns * sizeof(gint));
	if((title == NULL) || (justify == NULL))
	{
		g_free(title);
		g_free(justify);
		return;
	}

	/* Begin resetting the columns */

	gtk_clist_freeze(clist);

	/* Update the column settings */
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_active(clist);

	/* Set up each column */
	for(i = 0; i < ncolumns; i++)
	{
		/* Set column attributes by the find operation code */
		switch(find_by)
		{
		  case EDV_FIND_WIN_FIND_BY_NAME:
			switch(i)
			{
			  case 0:
				title[i] = "Name";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 1:
				title[i] = "Size";
				justify[i] = GTK_JUSTIFY_RIGHT;
				break;
			  case 2:
				title[i] = "Date Modified";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 3:
				title[i] = "Location";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  default:
				title[i] = NULL;
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			}
			break;
		  case EDV_FIND_WIN_FIND_BY_CONTENT:
			switch(i)
			{
			  case 0:
				title[i] = "Name";	/* And location */
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 1:
				title[i] = "Size";
				justify[i] = GTK_JUSTIFY_RIGHT;
				break;
			  case 2:
				title[i] = "Line";
				justify[i] = GTK_JUSTIFY_RIGHT;
				break;
			  case 3:
				title[i] = "Excerpt";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 4:
				title[i] = "Date Modified";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 5:
				title[i] = "Location";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  default:
				title[i] = NULL;
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			}
			break;
		  case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
			switch(i)
			{
			  case 0:
				title[i] = "Name";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 1:
				title[i] = "Size";
				justify[i] = GTK_JUSTIFY_RIGHT;
				break;
			  case 2:
				title[i] = "Date Modified";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 3:
				title[i] = "Location";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  default:
				title[i] = NULL;
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			}
			break;
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
			switch(i)
			{
			  case 0:
				title[i] = "Name";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 1:
				title[i] = "Size";
				justify[i] = GTK_JUSTIFY_RIGHT;
				break;
			  case 2:
				title[i] = "Date Modified";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  case 3:
				title[i] = "Location";
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			  default:
				title[i] = NULL;
				justify[i] = GTK_JUSTIFY_LEFT;
				break;
			}
			break;
		}

		/* Is this column's properties defined? */
		if(title[i] != NULL)
		{
			/* Show this column and set its attributes */
			gtk_clist_set_column_visibility(
				clist, i, TRUE
			);
			gtk_clist_set_column_resizeable(
				clist, i, TRUE
			);
			gtk_clist_set_column_auto_resize(
				clist, i, FALSE
			);
			gtk_clist_set_column_title(
				clist, i, title[i]
			);
			gtk_clist_set_column_justification(
				clist, i, justify[i]
			);
		}
		else
		{
			/* Hide this column */
			gtk_clist_set_column_visibility(
				clist, i, FALSE
			);
		}
	}

	gtk_clist_thaw(clist);

	/* Set the RC style */
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, lists_rcstyle);

	g_free(title);
	g_free(justify);
}

/*
 *	Appends an item to the Results List.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * properties.
 *
 *	The line_index specifies the line index of the excerpt.
 *
 *	The excerpt specifies the content excerpt or NULL if this
 *	is not applicable.
 */
void edv_find_win_list_append(
	EDVFindWin *fw,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt
)
{
	gint ncolumns;
	const gchar *format;
	gulong block_size;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVDateRelativity relativity;
	EDVSizeFormat size_format;
	EDVFindWinFindBy find_by;
	EDVCore *core;
	if(fw == NULL)
		return;

	clist = GTK_CLIST(fw->results_clist);
	ncolumns = MAX(clist->columns, 1);
	core = fw->core;
	cfg_list = core->cfg_list;

	/* Get the current find by method */
	find_by = edv_find_win_current_find_by(fw);

	/* Get the configuration */
	relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	size_format = (EDVSizeFormat)EDV_GET_I(
		EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);

	/* Begin appending the item */
	if(TRUE)
	{
		gint i, row;

		/* Allocate the row cell values */
		gchar **strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
		for(i = 0; i < ncolumns; i++)
			strv[i] = "";

		gtk_clist_freeze(clist);

		/* Append a new row */
		row = gtk_clist_append(clist, strv);
		if(row > -1)
		{
			/* Create a new EDVFindWinObject */
			EDVFindWinObject *obj = edv_find_win_object_new();
			if(obj != NULL)
			{
				const gchar	*name = edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_NAME
				),
							*path = edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_PATH
				);
				const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
					properties_list,
					EDV_PROP_NAME_TYPE
				);
				const EDVPermissionFlags permissions = (EDVPermissionFlags)edv_properties_list_get_i(
					properties_list,
					EDV_PROP_NAME_PERMISSIONS
				);
				EDVPixmap	*icon,
							*icon_hidden;

				if(path == NULL)
					path = name;

				/* Set the new EDVFindWinObject's values */
				obj->location_type = fw->location_type;
				obj->properties_list = edv_properties_list_copy(properties_list);
				obj->line_index = line_index;
				obj->excerpt = STRDUP(excerpt);
				obj->find_window = fw;

				/* Get the pixmap and mask for the object's icon */
				(void)edv_match_object_icon(
					core->devices_list,
					core->mime_types_list,
					type,
					path,
					TRUE,			/* Assume link valid */
					permissions,
					EDV_ICON_SIZE_20,
					&icon,
					NULL,
					NULL,
					&icon_hidden
				);
				/* Hidden? */
				if(edv_path_is_hidden(path))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}

				/* Begin setting the cell values */

				/* Set the name, which is always the first column
				 * regardless of the find by method
				 */
				i = 0;
				if(i < ncolumns)
				{
					if(edv_pixmap_is_loaded(icon))
						gtk_clist_set_pixtext(
							clist,
							row, i,
							(name != NULL) ? name : "(null)",
							EDV_LIST_PIXMAP_TEXT_SPACING,
							icon->pixmap, icon->mask
						);
					else
						gtk_clist_set_text(
							clist,
							row, i,
							(name != NULL) ? name : "(null)"
						);
				}

				(void)edv_pixmap_unref(icon);
				(void)edv_pixmap_unref(icon_hidden);

				/* Set the subsequent cells by the find by method */
				switch(find_by)
				{
				  case EDV_FIND_WIN_FIND_BY_NAME:
					/* Size */
					i = 1;
					if(i < ncolumns)
					{
						const gulong size = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_SIZE
						);
						const gchar *s = edv_str_size_format(
							size,
							size_format,
							block_size,
							',',
							TRUE
						);
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
					}
					/* Modify Time */
					i = 2;
					if(i < ncolumns)
					{
						const gulong modify_time = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_MODIFY_TIME
						);
						gchar *s = (modify_time > 0l) ?
							edv_date_string_format(
								modify_time,
								format, relativity
							) : NULL;
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
						g_free(s);
					}
					/* Location */
					i = 3;
					if(i < ncolumns)
					{
						gchar *location = (path != NULL) ? g_dirname(path) : NULL;
						EDVVFSObject *loc_obj = edv_vfs_object_lstat(location);
						if(loc_obj != NULL)
						{
							/* Get the pixmap and mask for the
							 * location's icon
							 */
							EDVPixmap	*icon,
											*icon_hidden;
							(void)edv_match_object_icon(
								core->devices_list,
								core->mime_types_list,
								loc_obj->type,
								loc_obj->path,
								EDV_VFS_OBJECT_LINK_TARGET_EXISTS(loc_obj),
								loc_obj->permissions,
								EDV_ICON_SIZE_20,
								&icon,
								NULL,
								NULL,
								&icon_hidden
							);
							/* Hidden? */
							if(edv_path_is_hidden(loc_obj->path))
							{
								if(edv_pixmap_is_loaded(icon_hidden))
								{
									(void)edv_pixmap_unref(icon);
									icon = edv_pixmap_ref(icon_hidden);
								}
							}
							if(edv_pixmap_is_loaded(icon))
								gtk_clist_set_pixtext(
									clist,
									row, i,
									location,
									EDV_LIST_PIXMAP_TEXT_SPACING,
									icon->pixmap, icon->mask
								);
							else
								gtk_clist_set_text(
									clist,
									row, i,
									location
								);

							(void)edv_pixmap_unref(icon);
							(void)edv_pixmap_unref(icon_hidden);

							edv_vfs_object_delete(loc_obj);
						}
						g_free(location);
					}
					break;

				  case EDV_FIND_WIN_FIND_BY_CONTENT:
					/* Size */
					i = 1;
					if(i < ncolumns)
					{
						const gulong size = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_SIZE
						);
						const gchar *s = edv_str_size_format(
							size,
							size_format,
							block_size,
							',',
							TRUE
						);
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
					}
					/* Line */
					i = 2;
					if(i < ncolumns)
					{
						gchar *s;
						if(line_index > -1)
							s = g_strdup_printf(
								"%i",
								line_index + 1
							);
						else
							s = g_strdup("");
						gtk_clist_set_text(
							clist,
							row, i,
							s
						);
						g_free(s);
					}
					/* Excerpt */
					i = 3;
					if(i < ncolumns)
					{
						gtk_clist_set_text(
							clist,
							row, i,
							(excerpt != NULL) ? excerpt : ""
						);
					}
					/* Modify Time */
					i = 4;
					if(i < ncolumns)
					{
						const gulong modify_time = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_MODIFY_TIME
						);
						gchar *s = (modify_time > 0l) ?
							edv_date_string_format(
								modify_time,
								format, relativity
							) : NULL;
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
						g_free(s);
					}
					/* Location */
					i = 5;
					if(i < ncolumns)
					{
						gchar *location = (path != NULL) ? g_dirname(path) : NULL;
						EDVVFSObject *loc_obj = edv_vfs_object_lstat(location);
						if(loc_obj != NULL)
						{
							/* Get the pixmap and mask for the
							 * location's icon
							 */
							EDVPixmap	*icon,
											*icon_hidden;
							(void)edv_match_object_icon(
								core->devices_list,
								core->mime_types_list,
								loc_obj->type,
								loc_obj->path,
								EDV_VFS_OBJECT_LINK_TARGET_EXISTS(loc_obj),
								loc_obj->permissions,
								EDV_ICON_SIZE_20,
								&icon,
								NULL,
								NULL,
								&icon_hidden
							);
							/* Hidden? */
							if(edv_path_is_hidden(loc_obj->path))
							{
								if(edv_pixmap_is_loaded(icon_hidden))
								{
									(void)edv_pixmap_unref(icon);
									icon = edv_pixmap_ref(icon_hidden);
								}
							}
							if(edv_pixmap_is_loaded(icon))
								gtk_clist_set_pixtext(
									clist,
									row, i,
									location,
									EDV_LIST_PIXMAP_TEXT_SPACING,
									icon->pixmap, icon->mask
								);
							else
								gtk_clist_set_text(
									clist,
									row, i,
									location
								);

							(void)edv_pixmap_unref(icon);
							(void)edv_pixmap_unref(icon_hidden);

							edv_vfs_object_delete(loc_obj);
						}
						g_free(location);
					}
					break;

				  case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
				  case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
				  case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
				  case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
					/* Size */
					i = 1;
					if(i < ncolumns)
					{
						const gulong size = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_SIZE
						);
						const gchar *s = edv_str_size_format(
							size,
							size_format,
							block_size,
							',',
							TRUE
						);
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
					}
					/* Modify Time */
					i = 2;
					if(i < ncolumns)
					{
						const gulong modify_time = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_MODIFY_TIME
						);
						gchar *s = (modify_time > 0l) ?
							edv_date_string_format(
								modify_time,
								format, relativity
							) : NULL;
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
						g_free(s);
					}
					/* Location */
					i = 3;
					if(i < ncolumns)
					{
						gchar *location = (path != NULL) ? g_dirname(path) : NULL;
						EDVVFSObject *loc_obj = edv_vfs_object_lstat(location);
						if(loc_obj != NULL)
						{
							/* Get the pixmap and mask for the
							 * location's icon
							 */
							EDVPixmap	*icon,
											*icon_hidden;
							(void)edv_match_object_icon(
								core->devices_list,
								core->mime_types_list,
								loc_obj->type,
								loc_obj->path,
								EDV_VFS_OBJECT_LINK_TARGET_EXISTS(loc_obj),
								loc_obj->permissions,
								EDV_ICON_SIZE_20,
								&icon,
								NULL,
								NULL,
								&icon_hidden
							);
							/* Hidden? */
							if(edv_path_is_hidden(loc_obj->path))
							{
								if(edv_pixmap_is_loaded(icon_hidden))
								{
									(void)edv_pixmap_unref(icon);
									icon = edv_pixmap_ref(icon_hidden);
								}
							}
							if(edv_pixmap_is_loaded(icon))
								gtk_clist_set_pixtext(
									clist,
									row, i,
									location,
									EDV_LIST_PIXMAP_TEXT_SPACING,
									icon->pixmap, icon->mask
								);
							else
								gtk_clist_set_text(
									clist,
									row, i,
									location
								);

							(void)edv_pixmap_unref(icon);
							(void)edv_pixmap_unref(icon_hidden);

							edv_vfs_object_delete(loc_obj);
						}
						g_free(location);
					}
					break;

				  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
				  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
				  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
				  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
					/* Size */
					i = 1;
					if(i < ncolumns)
					{
						const gulong size = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_SIZE
						);
						const gchar *s = edv_str_size_format(
							size,
							size_format,
							block_size,
							',',
							TRUE
						);
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
					}
					/* Modify Time */
					i = 2;
					if(i < ncolumns)
					{
						const gulong modify_time = edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_MODIFY_TIME
						);
						gchar *s = (modify_time > 0l) ?
							edv_date_string_format(
								modify_time,
								format, relativity
							) : NULL;
						gtk_clist_set_text(
							clist,
							row, i,
							(s != NULL) ? s : ""
						);
						g_free(s);
					}
					/* Location */
					i = 3;
					if(i < ncolumns)
					{
						gchar *location = (path != NULL) ? g_dirname(path) : NULL;
						EDVVFSObject *loc_obj = edv_vfs_object_lstat(location);
						if(loc_obj != NULL)
						{
							/* Get the pixmap and mask for the
							 * location's icon
							 */
							EDVPixmap	*icon,
											*icon_hidden;
							(void)edv_match_object_icon(
								core->devices_list,
								core->mime_types_list,
								loc_obj->type,
								loc_obj->path,
								EDV_VFS_OBJECT_LINK_TARGET_EXISTS(loc_obj),
								loc_obj->permissions,
								EDV_ICON_SIZE_20,
								&icon,
								NULL,
								NULL,
								&icon_hidden
							);
							/* Hidden? */
							if(edv_path_is_hidden(loc_obj->path))
							{
								if(edv_pixmap_is_loaded(icon_hidden))
								{
									(void)edv_pixmap_unref(icon);
									icon = edv_pixmap_ref(icon_hidden);
								}
							}
							if(edv_pixmap_is_loaded(icon))
								gtk_clist_set_pixtext(
									clist,
									row, i,
									location,
									EDV_LIST_PIXMAP_TEXT_SPACING,
									icon->pixmap, icon->mask
								);
							else
								gtk_clist_set_text(
									clist,
									row, i,
									location
								);

							(void)edv_pixmap_unref(icon);
							(void)edv_pixmap_unref(icon_hidden);

							edv_vfs_object_delete(loc_obj);
						}
						g_free(location);
					}
					break;
				}

				/* Set the new row's data as the object */
				gtk_clist_set_row_data_full(
					clist,
					row,
					obj, (GtkDestroyNotify)edv_find_win_object_delete
				);
/*		    obj = NULL; */
			}
		}

		gtk_clist_thaw(clist);

		gtk_clist_columns_autosize(clist);

		/* Delete the row's cell values */
		g_free(strv);
	}

}

/*
 *	Clears the results clist.
 */
void edv_find_win_list_clear(EDVFindWin *fw)
{
	GtkCList *clist;

	if(fw == NULL)
		return;

	clist = GTK_CLIST(fw->results_clist);

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}



/*
 *	Sets values on the given find window to the configuration list on
 *	the core structure of the given find window.
 */
void edv_find_win_sync_configuration(EDVFindWin *fw)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(fw == NULL)
		return;

	core = fw->core;
	cfg_list = core->cfg_list;

	/* Toplevel GtkWindow position and size */
	CfgGtkGetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_X,
		EDV_CFG_PARM_FIND_WIN_Y,
		EDV_CFG_PARM_FIND_WIN_WIDTH,
		EDV_CFG_PARM_FIND_WIN_HEIGHT,
		fw->toplevel
	);

	/* Find By Method */
	EDV_SET_I(
		EDV_CFG_PARM_FIND_WIN_FIND_BY,
		(gint)edv_find_win_current_find_by(fw)
	);

	/* Search String */
	CfgGtkGetGtkEditable(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_SEARCH_STRING,
		fw->search_combo
	);

	/* Find Using Method */
	EDV_SET_I(
		EDV_CFG_PARM_FIND_WIN_FIND_USING,
		(gint)edv_find_win_current_find_using(fw)
	);

	/* Case Sensitive */
	CfgGtkGetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_CASE_SENSITIVE,
		fw->case_sensitive_check
	);

	/* Recursive */
	CfgGtkGetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_RECURSIVE,
		fw->recursive_check
	);
}


/*
 *	Creates a new EDVFindWinObject.
 */
EDVFindWinObject *edv_find_win_object_new(void)
{
	return(EDV_FIND_WIN_OBJECT(g_malloc0(sizeof(EDVFindWinObject))));
}

/*
 *	Deletes the EDVFindWinObject.
 */
void edv_find_win_object_delete(EDVFindWinObject *obj)
{
	if(obj == NULL)
		return;

	(void)edv_properties_list_delete(obj->properties_list);
	g_free(obj->excerpt);
	g_free(obj);
}


/*
 *	Creates a new Find Window.
 */
EDVFindWin *edv_find_win_new(EDVCore *core)
{
	const gint	border_major = 5,
					border_minor = 2;
	gint i;
	GList *glist;
	GtkRcStyle	*standard_rcstyle,
					*lists_rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
					*parent, *parent2, *parent3, *parent4,
					*parent5,
					*toplevel,
					*pulist;
	gpointer combo_rtn;
	GtkCombo *combo;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVFindWinFindBy find_by;
	EDVFindWinFindUsing find_using;
	EDVFindWin *fw;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	find_by = (EDVFindWinFindBy)EDV_GET_I(
		EDV_CFG_PARM_FIND_WIN_FIND_BY
	);
	find_using = (EDVFindWinFindUsing)EDV_GET_I(
		EDV_CFG_PARM_FIND_WIN_FIND_USING
	);

	/* Create the Find Window */
	fw = EDV_FIND_WIN(g_malloc0(sizeof(EDVFindWin)));
	if(fw == NULL)
		return(NULL);

	fw->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	fw->accelgrp = accelgrp = gtk_accel_group_new();
/*
	fw->processing = FALSE;
	fw->freeze_count = 0;
	fw->busy_count = 0;
	fw->stop_count = 0;
 */
	fw->core = core;
/*	fw->flags = 0; */

	fw->browser_num = -1;
	fw->imbr_num = -1;
	fw->recbin_num = -1;
	fw->archiver_num = -1;

	fw->location_type = EDV_LOCATION_TYPE_VFS;
	fw->last_write_protect_state = -1;

	fw->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	CfgGtkSetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_X,
		EDV_CFG_PARM_FIND_WIN_Y,
		EDV_CFG_PARM_FIND_WIN_WIDTH,
		EDV_CFG_PARM_FIND_WIN_HEIGHT,
		w
	);
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		EDV_FIND_WM_CLASS_WINDOW_NAME,
		PROG_NAME
	);
	gtk_window_set_title(GTK_WINDOW(w), "Find Objects");
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_find_win_realize_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_find_win_delete_event_cb), fw
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main GtkVBox */
	fw->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Upper section hbox to separate two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_set_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left panel vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Begin creating search criteria widgets */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkHBox for find the find by method and search string */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkHBox for the find by method */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Buscar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
		":"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Find By Popup List Box */
	fw->find_by_pulistbox = w = PUListBoxNew(
		170, -1
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	PUListBoxSetChangedCB(
		w,
		edv_find_win_find_by_changed_cb, fw
	);
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
		PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
#define ADD_ITEM(_label_,_find_by_)	{		\
 i = PUListAddItem(pulist, (_label_));			\
 if(i > -1) {						\
  PUListSetItemData(pulist, i, (gpointer)(_find_by_));	\
  if(find_by == (_find_by_))				\
   PUListSelect(pulist, i);				\
 }							\
}
	ADD_ITEM("Name", EDV_FIND_WIN_FIND_BY_NAME);
	ADD_ITEM("Content", EDV_FIND_WIN_FIND_BY_CONTENT);
	ADD_ITEM("Size Equal To", EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO);
	ADD_ITEM("Size Not Equal To", EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO);
	ADD_ITEM("Size Less Than", EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN);
	ADD_ITEM("Size Greater Than", EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN);
	ADD_ITEM("Date Modified On", EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO);
	ADD_ITEM("Date Modified Not On", EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO);
	ADD_ITEM("Date Modified Before", EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN);
	ADD_ITEM("Date Modified After", EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN);
#undef ADD_ITEM
	if(i > -1)
		PUListBoxSetLinesVisible(w, MIN((i + 1), 10));
	gtk_widget_show(w);


	/* Search string GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
#if 0
#if defined(PROG_LANGUAGE_SPANISH)
"Palabra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Egaler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anpassend"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Uguagliare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Passend"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Combinar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passende"
#else
"Matching"
#endif
		":",
#endif
		NULL,				/* No label */
		"",					/* Initial value */
		glist,				/* List */
		20,					/* Max items */
		&combo_rtn,
		fw,
		edv_find_win_combo_activate_cb,
		NULL
	);
	fw->search_combo = (GtkWidget *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = fw->search_combo;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	CfgGtkSetGtkEditable(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_SEARCH_STRING,
		w
	);

	w = combo->entry;
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Location GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Empezar En"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commencer A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anfangen An"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cominciare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beginnen Aan"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comear Em"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Start P"
#else
"Starting At"
#endif
		":",
		"",
		glist,
		25,		/* Maximum items */
		&combo_rtn,
		fw,
		edv_find_win_combo_activate_cb,
		NULL
	);
	fw->location_combo = (GtkWidget *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = fw->location_combo;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Browse GtkButton */
	fw->browse_location_btn = w = GUIButtonPixmap(
		(guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_browse_location_cb), fw
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Parcourir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);


	/* Options GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;


	/* Find Using GtkHBox */
	fw->find_using_box = w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new("Using:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Find Using Popup List Box */
	fw->find_using_pulistbox = w = PUListBoxNew(
		150, -1
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
#if 0
	PUListBoxSetChangedCB(
		w,
		EDVFindWinFindUsingChangedCB, fw
	);
#endif
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
		PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
#define ADD_ITEM(_label_,_find_using_)	{		\
 i = PUListAddItem(pulist, (_label_));			\
 if(i > -1) {						\
  PUListSetItemData(pulist, i, (gpointer)(_find_using_));	\
  if(find_using == (_find_using_))			\
   PUListSelect(pulist, i);				\
 }							\
}
	ADD_ITEM("Literal Matching", EDV_FIND_WIN_FIND_USING_LITERAL);
	ADD_ITEM("Regular Expression", EDV_FIND_WIN_FIND_USING_REGEX);
#undef ADD_ITEM
	if(i > -1)
		PUListBoxSetLinesVisible(w, MIN((i + 1), 10));
	gtk_widget_show(w);

	/* Case Sensitive GtkCheckButton */
	fw->case_sensitive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Sensible May/min."
#elif defined(PROG_LANGUAGE_FRENCH)
"Reconnatre Sensible"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fall Empfindlich"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Imballare Sensibile"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit Gevoelig In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Caso Sensvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfelle Sensitiv"
#else
"Case Sensitive"
#endif
	);
	CfgGtkSetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_CASE_SENSITIVE,
		w
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto para discriminar Mayusculas y minusculas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour galer des ficelles reconnaissent sensiblement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, schnren fall empfindlich anzupassen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per uguagliare le cordicelle imballano\
 sensibilmente"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om bij koorden geval gevoelig te passen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto combinar caso de barbantes sensivelmente"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette passe snortilfelle sensitivt"
#else
"Check this to match strings case sensitively"
#endif
	);
	gtk_widget_show(w);

	/* Recursive GtkCheckButton */
	fw->recursive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Recursivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recursive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Rekursiv"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ricorsivo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Recursief"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Recursive"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rekursiv"
#else
"Recursive"
#endif
	);
	CfgGtkSetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_FIND_WIN_RECURSIVE,
		w
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto buscar recursivamente en sub guas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour chercher recursively dans sous les annuaires"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, rekursiv in u-boot verzeichnisse zu suchen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per ricercare ricorsivamente in sotto gli elenchi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om recursief in sub gidzen te zoeken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto procurar recursively em guias de submarino"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette gjennomske rekursivt inn i sub kataloger"
#else
"Check this to search recursively into sub directories"
#endif
	);
	gtk_widget_show(w);


	/* GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* GtkScrolledWindow for the Results GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Results clist */
	fw->results_clist = w = gtk_clist_new(
		EDV_FIND_WIN_RESULTS_LIST_NCOLUMNS
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_find_win_button_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "click_column",
		GTK_SIGNAL_FUNC(edv_find_win_click_column_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_find_win_select_row_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_find_win_unselect_row_cb), fw
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_column_titles_hide(clist);
	if(w != NULL)
	{
		const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
		};
		GUIDNDSetSrc(
			w,
			dnd_src_types,
			sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY | GDK_ACTION_MOVE |
				GDK_ACTION_LINK,		/* Actions */
			GDK_BUTTON1_MASK,		/* Buttons */
			NULL,
			edv_find_win_drag_data_get_cb,
			edv_find_win_drag_data_delete_cb,
			NULL,
			fw
		);
	}
	gtk_widget_show(w);


	/* Right-Click GtkMenu */
	fw->results_clist_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
		guint accel_key, accel_mods;
		guint8 **icon;
		const gchar *label;
		gpointer data = fw;
		GtkWidget *menu = w;
		void (*func_cb)(GtkWidget *, gpointer);

#define ADD_MENU_ITEM_LABEL	{	\
 w = GUIMenuItemCreate(			\
  menu,					\
  GUI_MENU_ITEM_TYPE_LABEL,		\
  accelgrp,				\
  icon, label,				\
  accel_key, accel_mods,		\
  func_cb, data				\
 );					\
}
#define ADD_MENU_ITEM_SEPARATOR	{	\
 w = GUIMenuItemCreate(			\
  menu,					\
  GUI_MENU_ITEM_TYPE_SEPARATOR,		\
  NULL,					\
  NULL, NULL,				\
  0, 0,					\
  NULL, NULL				\
 );					\
}
		icon = (guint8 **)icon_open_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen"
#else
"Open"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_find_win_open_cb;
		ADD_MENU_ITEM_LABEL
		fw->results_clist_open_mi = w;

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir Con"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Avec"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Mit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Con"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Met"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Com"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Med"
#else
"Open With"
#endif
		"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_find_win_open_with_cb;
		ADD_MENU_ITEM_LABEL
		fw->results_clist_open_with_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		icon = (guint8 **)icon_goto_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til"
#else
"GoTo"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_find_win_goto_cb;
		ADD_MENU_ITEM_LABEL
		fw->results_clist_goto_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		icon = (guint8 **)icon_copy_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Copie Sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier Le Sentier"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj ciek"
#else
"Copy Path"
#endif
		;
		accel_key = GDK_c;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = edv_find_win_copy_path_cb;
		ADD_MENU_ITEM_LABEL
		fw->results_clist_copy_path_mi = w;

		icon = (guint8 **)icon_copy_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Copie URL"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier URL"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj URL"
#else
"Copy URL"
#endif
		;
		accel_key = GDK_c;
		accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
		func_cb = edv_find_win_copy_url_cb;
		ADD_MENU_ITEM_LABEL
		fw->results_clist_copy_url_mi = w;

#undef ADD_MENU_ITEM_SEPARATOR
#undef ADD_MENU_ITEM_LABEL
	}

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Open GtkButton */
	fw->open_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_open_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen"
#else
"Open"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_open_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt"
#else
"Open selected object"
#endif
	);
	gtk_widget_show(w);

	/* Open With GtkButton */
	fw->open_with_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_open_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir Con"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Avec"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Mit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Con"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Met"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Com"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Med"
#else
"Open With"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_open_with_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_w, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_w);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido utilizando un mtodo especfico"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi utilisant une mthode spcifique"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt, eine spezifische methode zu benutzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto usando uno specifico metodo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp een specifieke methode te gebruiken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto usando um mtodo especfico"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt bruke en spesifikk metode"
#else
"Open selected object using a specific method"
#endif
	);
	gtk_widget_show(w);

	/* Goto GtkButton */
	fw->goto_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_goto_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til"
#else
"GoTo"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_goto_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_g, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_g);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya a escogido se opone"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller  l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen sie zu ausgewhltem objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare all'oggetto scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga te geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V a objeto selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra til valgt ut objekt"
#else
"Go to selected object"
#endif
	);
	gtk_widget_show(w);

	/* Copy Path GtkButton */
	fw->copy_path_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_copy_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Copie Sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier Le Sentier"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj ciek"
#else
"Copy Path"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_copy_path_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Copia la ubicacin del objeto(s) escogido como sendero(s)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier l'emplacement de l'objet choisi(s) comme le sentier(s)"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako ciek (cieki)"
#else
"Copy the location of the selected object(s) as path(s)"
#endif
	);
	gtk_widget_show(w);

	/* Copy URL GtkButton */
	fw->copy_url_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_copy_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Copie URL"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier URL"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj URL"
#else
"Copy URL"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_copy_url_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK | GDK_SHIFT_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Copie la ubicacin del objeto(s) escogido como url(s)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier l'emplacement de l'objet choisi(s) comme url(s)"
#elif defined(PROG_LANGUAGE_POLISH)
"Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako URL"
#else
"Copy the location of the selected object(s) as url(s)"
#endif
	);
	gtk_widget_show(w);


	/* Right Column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Buttons Set GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Search GtkButton */
	fw->search_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_search_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Bsqueda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Suche"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ricerca"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zoektocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Busca"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Leting"
#else
"Search"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_search_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Empiece la bsqueda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commencer la recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fangen sie die suche an"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cominciare la ricerca"
#elif defined(PROG_LANGUAGE_DUTCH)
"Begin de zoektocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comece a busca"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Start letingen"
#else
"Start the search"
#endif
	);
	gtk_widget_show(w);

	/* Stop GtkButton */
	fw->stop_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_stop_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrt"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermata"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Parada"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans"
#else
"Stop"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_stop_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_t, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_t);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Pare el procedimiento actual del hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrter la procdure actuelle de dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halten sie das jetzige fund verfahren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermare la corrente trova la procedura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stop de huidig vondst procedure"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pare a corrente achar procedimento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans den nvrendee funnprosedyre"
#else
"Stop the current find procedure"
#endif
	);
	gtk_widget_show(w);

	/* Clear GtkButton */
	fw->clear_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_clear_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Clair"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_clear_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"La cuerda clara de la bsqueda y la lista de resultados"
#elif defined(PROG_LANGUAGE_FRENCH)
"La ficelle clairs de recherche et la liste de rsultats"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klare suche schnur und ergebnisse fhren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'elenco di cordicella di ricerca e risultati chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder zoektocht koord en resultaten sommen op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Barbante clara de busca e lista de resultados"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar letingssnor og resultater lister opp"
#else
"Clear search string and results list"
#endif
	);
	gtk_widget_show(w);


	/* Buttons set GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Close GtkButton */
	fw->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_win_close_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_win_crossing_event_cb), fw
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre esta ventana"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer cette fentre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schlieen sie dieses fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino questa finestra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit deze raam"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo esta janela"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stenge dette vinduet"
#else
"Close this window"
#endif
	);
	gtk_widget_show(w);


	/* Status Bar */
	fw->status_bar = w = edv_status_bar_new(core);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			fw->toplevel,
			standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			fw->results_clist,
			lists_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			fw->results_clist_menu,
			standard_rcstyle
		);


	fw->freeze_count--;

	/* Set initial values */
	edv_find_win_find_by_changed_cb(
		fw->find_by_pulistbox,
		PUListBoxGetSelected(fw->find_by_pulistbox),
		fw
	);

	return(fw);
}

/*
 *	Updates the Find Window's widgets to reflect current values.
 */
void edv_find_win_update_display(EDVFindWin *fw)
{
	gboolean	write_protect,
					write_protect_changed = FALSE,
					sensitive,
					processing;
	gint sel_row;
	GList *glist;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVLocationType location_type;
	EDVFindWinFindBy find_by;
	EDVFindWinFindUsing find_using;
	EDVCore *core;

	if(fw == NULL)
		return;

	processing = fw->processing;
	location_type = fw->location_type;
	find_by = edv_find_win_current_find_by(fw);
	find_using = edv_find_win_current_find_using(fw);
	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;
	cfg_list = core->cfg_list;

	/* Get the last selected row on the results GtkCList */
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Get the global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(fw->last_write_protect_state < 0)
	{
		write_protect_changed = TRUE;
	}
	else
	{
		if((write_protect && !fw->last_write_protect_state) ||
		   (!write_protect && fw->last_write_protect_state)
		)
			write_protect_changed = TRUE;
	}
	fw->last_write_protect_state = write_protect ? 1 : 0;

	/* Find By */
	sensitive = !processing;
	gtk_widget_set_sensitive(fw->find_by_pulistbox, sensitive);

	/* Search String */
	sensitive = !processing;
	gtk_widget_set_sensitive(fw->search_combo, sensitive);

	/* Location */
	if(processing)
	{
		sensitive = FALSE;
	}
	else
	{
		switch(location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			sensitive = TRUE;
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			sensitive = FALSE;
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			sensitive = TRUE;
			break;
		}
	}
	gtk_widget_set_sensitive(fw->location_combo, sensitive);
	gtk_widget_set_sensitive(fw->browse_location_btn, sensitive);

	/* Find Using */
	if(processing)
	{
		sensitive = FALSE;
	}
	else
	{
		switch(find_by)
		{
		  case EDV_FIND_WIN_FIND_BY_CONTENT:
			sensitive = TRUE;
			break;
		  case EDV_FIND_WIN_FIND_BY_NAME:
		  case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
			sensitive = FALSE;
			break;
		}
	}
	gtk_widget_set_sensitive(fw->find_using_box, sensitive);

	/* Case Sensitive */
	if(processing)
	{
		sensitive = FALSE;
	}
	else
	{
		switch(find_by)
		{
		  case EDV_FIND_WIN_FIND_BY_NAME:
		  case EDV_FIND_WIN_FIND_BY_CONTENT:
			sensitive = TRUE;
			break;
		  case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
		  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
			sensitive = FALSE;
			break;
		}
	}
	gtk_widget_set_sensitive(fw->case_sensitive_check, sensitive);

	/* Recursive */
	if(processing)
	{
		sensitive = FALSE;
	}
	else
	{
		switch(location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			sensitive = TRUE;
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			sensitive = FALSE;
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			sensitive = FALSE;
			break;
		}
	}
	gtk_widget_set_sensitive(fw->recursive_check, sensitive);

	/* Stop */
	sensitive = processing;
	gtk_widget_set_sensitive(fw->stop_btn, sensitive);

	/* Search */
	sensitive = !processing;
	gtk_widget_set_sensitive(fw->search_btn, sensitive);
	gtk_widget_set_sensitive(fw->clear_btn, sensitive);
	gtk_widget_set_sensitive(fw->close_btn, sensitive);

	/* Open */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		sensitive = (!processing && (sel_row > -1)) ? TRUE : FALSE;
		break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
		sensitive = FALSE;
		break;
	}
	gtk_widget_set_sensitive(fw->open_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_open_mi, sensitive);
	gtk_widget_set_sensitive(fw->open_with_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_open_with_mi, sensitive);

	/* Goto */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
		sensitive = (!processing && (sel_row > -1)) ? TRUE : FALSE;
		break;
	}
	gtk_widget_set_sensitive(fw->goto_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_goto_mi, sensitive);

	/* Copy Path & Copy URL */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
		sensitive = (sel_row > -1) ? TRUE : FALSE;
		break;
	}
	gtk_widget_set_sensitive(fw->copy_path_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_copy_path_mi, sensitive);

	gtk_widget_set_sensitive(fw->copy_url_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_copy_url_mi, sensitive);

	/* Status Bar */
	edv_status_bar_update_display(fw->status_bar);
}

/*
 *	Sets the Find Window as busy or ready.
 */
void edv_find_win_set_busy(EDVFindWin *fw, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(fw == NULL)
		return;

	w = fw->toplevel;
	core = fw->core;

	if(w != NULL)
	{
		if(busy)
		{
			/* Increase busy count */
			fw->busy_count++;

			/* If already busy then don't change anything */
			if(fw->busy_count > 1)
				return;

			cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
		}
		else
		{
			/* Reduce busy count */
			fw->busy_count--;
			if(fw->busy_count < 0)
				fw->busy_count = 0;

			/* If still busy do not change anything */
			if(fw->busy_count > 0)
				return;

			cursor = NULL;  /* Use default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cursor);
			gdk_flush();
		}
	}
}

/*
 *	Checks if the Find Window is mapped.
 */
gboolean edv_find_win_is_mapped(EDVFindWin *fw)
{
	if(fw == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(fw->toplevel));
}

/*
 *	Maps the Find Window.
 */
void edv_find_win_map(EDVFindWin *fw)
{
	GtkWidget *w;

	if(fw == NULL)
		return;

	gtk_widget_show_raise(fw->toplevel);

	w = fw->search_combo;
	w = GTK_COMBO(w)->entry;
	gtk_widget_grab_focus(w);

	fw->flags |= EDV_FIND_WIN_MAPPED;
}

/*
 *	Unmaps the Find Window.
 */
void edv_find_win_unmap(EDVFindWin *fw)
{
	if(fw == NULL)
		return;

	gtk_widget_hide(fw->toplevel);

	fw->flags &= ~EDV_FIND_WIN_MAPPED;
}

/*
 *	Deletes the Find Window.
 */
void edv_find_win_delete(EDVFindWin *fw)
{
	if(fw == NULL)
		return;

	edv_find_win_unmap(fw);

	fw->freeze_count++;

	gtk_widget_destroy(fw->results_clist_menu);
	gtk_widget_destroy(fw->toplevel);
	gtk_accel_group_unref(fw->accelgrp);

	fw->freeze_count--;

	g_free(fw);
}
