#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>				/* mknod() */
#include <sys/types.h>				/* mknod() */
#include <sys/stat.h>				/* mknod() */
#include <gtk/gtk.h>
#include <unistd.h>				/* mknod() */

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_link.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_vfs_obj_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_replace_file_32x32.xpm"
#include "images/icon_chmod_32x32.xpm"
#include "images/icon_owned_32x32.xpm"
#include "images/icon_time_stamp_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No", "Ignore", or response was
 *		not available.
 *	-6	An operation is already in progress.
 *	-99	Retry (internal to this module).
 */


/* Error Message */
const gchar *edv_vfs_object_op_get_error(EDVCore *core);
static void edv_vfs_object_op_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Checks */
static gboolean edv_vfs_object_op_objects_same(
	EDVVFSObject *src_obj,
	EDVVFSObject *tar_obj
);

/* Confirmation */
static gint edv_vfs_object_op_confirm_overwrite(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *tar_lobj,
	GtkWidget *toplevel
);

/* Counting */
static gint edv_vfs_object_op_count_objects(
	const gchar *path,
	const gboolean recursive,
	const gboolean archive
);

/* Create Directories Recursive */
static gint edv_vfs_object_op_rmkdir(
	EDVCore *core,
	const gchar *path,
	GList **new_paths_list_rtn,
	const gboolean interactive,
	gboolean *yes_to_all,
	GtkWidget *toplevel
);

/* Deleting */
static gint edv_vfs_object_op_unlink(
	EDVCore *core,
	const gchar *path
);
static gint edv_vfs_object_op_rmdir(
	EDVCore *core,
	const gchar *path
);

/* Object Copying */
static gint edv_vfs_object_op_copy_file(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	const gboolean originally_move
);
static gint edv_vfs_object_op_copy_link(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
);
static gint edv_vfs_object_op_copy_fifo(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
);
static gint edv_vfs_object_op_copy_device_block(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
);
static gint edv_vfs_object_op_copy_device_character(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
);
static gint edv_vfs_object_op_copy_socket(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
);
static gint edv_vfs_object_op_copy_nexus(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	const gboolean originally_move
);

/* Object Moving */
static gint edv_vfs_object_op_move_objects_local(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	gint *status,
	gboolean *need_copy
);

/* Object Copying */
static gint edv_vfs_object_op_copy_objects_local(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	gint *status,
	const gboolean originally_move
);

/* Directory Copying */
static gint edv_vfs_object_op_copy_directory_iterate(
	EDVCore *core,
	const gchar *src_dir,
	const gchar *tar_dir,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gint *status,
	gboolean *yes_to_all,
	const gboolean originally_move
);

/* Copy & Move */
static gint edv_vfs_object_op_copy_or_move(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_copy
);
gint edv_vfs_object_op_copy(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_vfs_object_op_move(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Link */
static gchar *edv_vfs_object_op_generate_new_link_path(
	const gchar *path,
	const gchar *dest_name
);
gint edv_vfs_object_op_link(
	EDVCore *core,
	const gchar *path,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_vfs_object_op_relink(
	EDVCore *core,
	const gchar *path,
	const gchar *new_dest_value,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Rename */
gint edv_vfs_object_op_rename(
	EDVCore *core,
	const gchar *path,
	const gchar *new_name,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Permissions */
static gint edv_vfs_object_op_chmod_iterate(
	EDVCore *core,
	const gchar *path,
	const mode_t m,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
);
gint edv_vfs_object_op_chmod(
	EDVCore *core,
	GList *paths_list,
	const EDVPermissionFlags permissions,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
);

/* Change Ownership */
static gint edv_vfs_object_op_chown_iterate(
	EDVCore *core,
	const gchar *path,
	const gint owner_id,
	const gint group_id,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
);
gint edv_vfs_object_op_chown(
	EDVCore *core,
	GList *paths_list,
	const gint owner_id,
	const gint group_id,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
);

/* Change Time Stamps */
static gint edv_vfs_object_op_chtime_iterate(
	EDVCore *core,
	const gchar *path,
	const gulong atime,
	const gulong mtime,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
);
gint edv_vfs_object_op_chtime(
	EDVCore *core,
	GList *paths_list,
	const gulong atime,
	const gulong mtime,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_vfs_object_op_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message specified by msg to the last error
 *	message buffer and sets last_error to point to that buffer.
 */
static void edv_vfs_object_op_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Checks if the objects are the same by the stat device index
 *	and index on the device.
 */
static gboolean edv_vfs_object_op_objects_same(
	EDVVFSObject *src_obj,
	EDVVFSObject *tar_obj
)
{
	if((src_obj == NULL) || (tar_obj == NULL))
		return(FALSE);

	if((src_obj->device_index == tar_obj->device_index) &&
	   (src_obj->index == tar_obj->index)
	)
		return(TRUE);
	else
		return(FALSE);
}


/*
 *	Queries the user to confirm replace.
 *
 *	The src_path specifies the full path to the replacement
 *	object.
 *
 *	The tar_path specifies the full path to the object that is to
 *	be replaced.
 *
 *	The src_lobj specifies the local statistics of the replacement
 *	object.
 *
 *	The tar_lobj specifies the local statistics of the object that
 *	is to be replaced.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	Returns one of CDIALOG_RESPONSE_*.
 */
static gint edv_vfs_object_op_confirm_overwrite(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *tar_lobj,
	GtkWidget *toplevel
)
{
	gint response;
	const gchar *format;
	gchar		*msg,
					*src_date, *tar_date,
					*src_size_s, *tar_size_s;
	gulong		src_size,
					tar_size;
	CfgList *cfg_list;
	EDVDateRelativity relativity;

	if(core == NULL)
		return(CDIALOG_RESPONSE_NOT_AVAILABLE);

	cfg_list = core->cfg_list;

	/* Get the date relativity and format */
	relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

	/* Get the date strings of the source and target objects */
	src_date = edv_date_string_format(
		(src_lobj != NULL) ? src_lobj->modify_time : 0l,
		format, relativity
	);
	tar_date = edv_date_string_format(
		(tar_lobj != NULL) ? tar_lobj->modify_time : 0l,
		format, relativity
	);

	/* Get the size strings of the source and target objects */
	src_size = (src_lobj != NULL) ? src_lobj->size : 0l;
	src_size_s = STRDUP(edv_str_size_delim(src_size));

	tar_size = (tar_lobj != NULL) ? tar_lobj->size : 0l;
	tar_size_s = STRDUP(edv_str_size_delim(tar_size));

	/* Format the message */
	msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Reemplace:\n\
\n\
    %s (%s %s) %s\n\
\n\
La Anchura:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_FRENCH)
"Remplacer:\n\
\n\
    %s (%s %s) %s\n\
\n\
Largeur:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ersetzen:\n\
\n\
    %s (%s %s) %s\n\
\n\
Breite:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sostituire:\n\
\n\
    %s (%s %s) %s\n\
\n\
La Larghezza:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vervang:\n\
\n\
    %s (%s %s) %s\n\
\n\
Breedte:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Reponha:\n\
\n\
    %s (%s %s) %s\n\
\n\
A Largura:\n\
\n\
    %s (%s %s) %s"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Erstatt:\n\
\n\
    %s (%s %s) %s\n\
\n\
Bredde:\n\
\n\
    %s (%s %s) %s"
#else
"Replace:\n\
\n\
    %s (%s %s) %s\n\
\n\
With:\n\
\n\
    %s (%s %s) %s"
#endif
		,
		tar_path,
		tar_size_s,
		(tar_size == 1l) ? "byte" : "bytes",
		tar_date,
		src_path,
		src_size_s,
		(src_size == 1l) ? "byte" : "bytes",
		src_date
	);

	g_free(src_date);
	g_free(tar_date);
	g_free(src_size_s);
	g_free(tar_size_s);

	/* Query the user to confirm replace */
	edv_play_sound_warning(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Superposer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Replace"
#endif
		,
		msg,
		NULL,
		(guint8 **)icon_replace_file_32x32_xpm,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
			CDIALOG_BTNFLAG_NO | CDIALOG_BTNFLAG_CANCEL,
		CDIALOG_BTNFLAG_NO
	);
	g_free(msg);
	CDialogSetTransientFor(NULL);

	return(response);
}


/*
 *	Counts the number of objects.
 *
 *	The path specifies the object to count.
 *
 *	If recursive is TRUE and path is a directory then all the
 *	objects in that directory will be counted as well.
 *
 *	If archive is TRUE and path refers to a link that leads to a
 *	directory then only the link will be counted and none of
 *	the objects in the destination directory will be counted
 *	regardless of if recursive is TRUE or FALSE.
 */
static gint edv_vfs_object_op_count_objects(
	const gchar *path,
	const gboolean recursive,
	const gboolean archive
)
{
	gint nobjs = 0;
	EDVVFSObject *obj;

	if(archive)
		obj = edv_vfs_object_lstat(path);
	else
		obj = edv_vfs_object_stat(path);
	if(obj == NULL)
		return(nobjs);

	/* Count this object */
	nobjs++;

	/* Directory? */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		/* Count all the objects in this directory */
		EDVDirectory *dp = edv_directory_open(
			path,
			FALSE,				/* Unsorted */
			FALSE				/* Exclude notations */
		);
		if(dp != NULL)
		{
			const gchar *name;
			gchar *child_path;

			for(name = edv_directory_next(dp);
				name != NULL;
				name = edv_directory_next(dp)
			)
			{
				child_path = g_strconcat(
					path,
					G_DIR_SEPARATOR_S,
					name,
					NULL
				);
				nobjs += edv_vfs_object_op_count_objects(
					child_path,
					recursive,
					archive
				);
				g_free(child_path);
			}

			edv_directory_close(dp);
		}
	}

	edv_vfs_object_delete(obj);

	return(nobjs);
}


/*
 *	Creates the directory and parent directories as needed.
 *
 *	The path specifies the directory to create. If it does not
 *	exist and/or its parent directories do not exist then
 *	it/they will be created.
 *
 *	If interactive is TRUE then the user will be prompted to
 *	create the directories if needed.
 *
 *	Returns:
 *
 *	0	Success or path refers to an existing directory.
 *	-1	General error.
 *	-2	Invalid value or path is not a directory.
 *	-3	Systems error.
 *	-4	User aborted.
 */
static gint edv_vfs_object_op_rmkdir(
	EDVCore *core,
	const gchar *path,
	GList **new_paths_list_rtn,
	const gboolean interactive,
	gboolean *yes_to_all,
	GtkWidget *toplevel
)
{
	gint		status,
			error_code;
	GList *new_directories_list;
	EDVVFSObject *obj;

	if(path == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the destination's statistics */
	obj = edv_vfs_object_stat(path);
	if(obj != NULL)
	{
		/* Directory already exists? */
		if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
		{
			/* The directory already exists */
			edv_vfs_object_delete(obj);
			return(0);
		}
		else
		{
			/* Object exists but is not a directory */
			edv_vfs_object_delete(obj);
			errno = EINVAL;
			return(-2);			/* Not a directory */
		}

		/* Never reached */
		edv_vfs_object_delete(obj);
	}
	else
	{
		/* Exists but other error other than it does not exist? */
		error_code = (gint)errno;
#ifdef ENOENT
		if(error_code != ENOENT)
			return(-1);
#endif
	}

	/* Query the user to create the directories? */
	if(interactive && !(*yes_to_all))
	{
		gint response;
		gchar *msg = g_strdup_printf(
"Create directory:\n\
\n\
    %s\n\
\n\
Including any parent directories as needed.",
			path
		);

		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
			"Confirm Create Directory",
			msg,
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
			CDIALOG_BTNFLAG_CANCEL,
			CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);

		switch(response)
		{
		  case CDIALOG_RESPONSE_YES_TO_ALL:
			*yes_to_all = TRUE;
		  case CDIALOG_RESPONSE_YES:
		  case CDIALOG_RESPONSE_OK:
			break;

		  default:
			return(-4);			/* User aborted */
			break;
		}
	}

	/* Create the new directory */
	status = edv_directory_create(
		path,
		TRUE,				/* Create parents */
		&new_directories_list
	);

	error_code = (gint)errno;

	if(new_directories_list != NULL)
	{
		if(new_paths_list_rtn != NULL)
		{
			const gchar *s;
			GList *glist;

			for(glist = new_directories_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				s = (const gchar *)glist->data;
				if(s == NULL)
					continue;

				*new_paths_list_rtn = g_list_append(
					*new_paths_list_rtn,
					g_strdup(s)
				);
			}
		}

		g_list_foreach(
			new_directories_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(new_directories_list);
	}

	errno = (int)error_code;

	return(status);
}


/*
 *	Unlinks the object.
 *
 *	The path specifies the full path to the object to unlink.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error. Can also return
 *	0 if the object did not exist.
 */
static gint edv_vfs_object_op_unlink(
	EDVCore *core,
	const gchar *path
)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

#define SET_LAST_ERROR(_path_,_error_code_) {	\
 if(core->last_error_ptr == NULL) {		\
  gchar *msg = g_strdup_printf(			\
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",						\
   (_path_), g_strerror(_error_code_)		\
  );						\
  edv_vfs_object_op_set_error(core, msg);	\
  g_free(msg);					\
 }						\
}

	if(edv_unlink(path))
	{
		const gint error_code = (gint)errno;
		switch(error_code)
		{
#ifdef EFAULT
		  /* Segmentation fault */
		  case EFAULT:
			SET_LAST_ERROR(path, error_code);
			return(-3);
			break;
#endif
#ifdef EACCES
		  /* Access denied */
		  case EACCES:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
#ifdef EPERM
		  /* Permission denied */
		  case EPERM:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
#ifdef ENAMETOOLONG
		  /* Invalid name */
		  case ENAMETOOLONG:
			SET_LAST_ERROR(path, error_code);
			return(-2);
			break;
#endif
#ifdef ENOENT
		  /* Object does not exist */
		  case ENOENT:
			/* Do not process this as an error */
			break;
#endif
#ifdef ENOTDIR
		  /* Object does not exist */
		  case ENOTDIR:
			/* Do not process this as an error */
			break;
#endif
#ifdef EISDIR
		  /* The object is a directory */
		  case EISDIR:
			SET_LAST_ERROR(path, error_code);
			return(-2);
			break;
#endif
#ifdef ENOMEM
		  /* Out of memory */
		  case ENOMEM:
			SET_LAST_ERROR(path, error_code);
			return(-3);
			break;
#endif
#ifdef EROFS
		  /* The object is on a read only filesystem */
		  case EROFS:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
#ifdef ELOOP
		  /* Too many symbolic links encountered */
		  case ELOOP:
			SET_LAST_ERROR(path, error_code);
			return(-3);
			break;
#endif
#ifdef EIO
		  /* IO error */
		  case EIO:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
#ifdef EBUSY
		  /* The system is currently busy */
		  case EBUSY:
			SET_LAST_ERROR(path, error_code);
			return(-3);
			break;
#endif
#ifdef ETXTBUSY
		  /* The file is currently in use */
		  case ETXTBUSY:
			SET_LAST_ERROR(path, error_code);
			return(-3);
			break;
#endif
#ifdef EINTR
		  /* Operation interrupted */
		  case EINTR:
			return(-4);
			break;
#endif
#ifdef EMULTIHOP
		  /* Multihop */
		  case EMULTIHOP:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
#ifdef ENOLINK
		  /* No link */
		  case ENOLINK:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
#endif
		  /* All other errors */
		  default:
			SET_LAST_ERROR(path, error_code);
			return(-1);
			break;
		}
	}
#undef SET_LAST_ERROR

	return(0);
}

/*
 *	Deletes the directory and all the objects in that directory.
 *
 *	The path specifies the full path to the directory to delete.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero if the directory or any
 *	object in the directory could not be deleted.
 */
static gint edv_vfs_object_op_rmdir(
	EDVCore *core,
	const gchar *path
)
{
	gint status;
	GList *names_list;

	if(core == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	/* If no path was specified then return success */
	if(path == NULL)
		return(0);

	status = 0;

	/* Get the list of objects in the directory */
	names_list = edv_directory_list(
		path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		/* Delete each object in this directory */
		const gchar *name;
		gchar *child_path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Format the full path to this object */
			child_path = g_strconcat(
				path,
				G_DIR_SEPARATOR_S,
				name,
				NULL
			);
			if(child_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Get this object's local statistics */
			obj = edv_vfs_object_lstat(child_path);
			if(obj == NULL)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
					child_path,
					g_strerror(error_code)
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				g_free(child_path);
				status = -1;
				break;
			}

			if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				/* Delete all objects in this directory and the
				 * directory itself
				 */
				status = edv_vfs_object_op_rmdir(
					core,
					child_path
				);
			}
	 	else
			{
				/* Delete this object */
				status = edv_vfs_object_op_unlink(
					core,
					child_path
				);
			}

			edv_vfs_object_delete(obj);
			g_free(child_path);

			/* Stop deleting if an error occured */
			if(status != 0)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}

	/* Do not delete this directory if an error occured */
	if(status != 0)
		return(status);

	/* Remove this directory */
	if(edv_directory_remove(
		path,
		FALSE,				/* Do not recurse */
		FALSE,				/* Do not force */
		NULL,				/* No objects list return */
		NULL, NULL
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to delete the directory:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		switch(error_code)
		{
#ifdef EFAULT
		  /* Segmentation fault */
		  case EFAULT:
			status = -3;
			break;
#endif
#ifdef EACCES
		  /* Access denied */
		  case EACCES:
			status = -1;
			break;
#endif
#ifdef EPERM
		  /* Permission denied */
		  case EPERM:
			status = -1;
			break;
#endif
#ifdef ENAMETOOLONG
		  /* Name too long */
		  case ENAMETOOLONG:
			status = -2;
			break;
#endif
#ifdef ENOENT
		  /* No such object */
		  case ENOENT:
			/* This is still an error because we just checked
			 * and it existed
			 */
			status = -1;
			break;
#endif
#ifdef ENOTDIR
		  /* The directory to remove is not, in fact, a directory */
		  case ENOTDIR:
			/* This is still an error because we just checked
			 * and it was a directory
			 */
			status = -2;
			break;
#endif
#ifdef ENOTEMPTY
		  /* The directory is not empty */
		  case ENOTEMPTY:
			status = -1;
			break;
#endif
#ifdef EBUSY
		  /* The system is currently busy */
		  case EBUSY:
			status = -3;
			break;
#endif
#ifdef ENOMEM
		  /* Out of memory */
		  case ENOMEM:
			status = -3;
			break;
#endif
#ifdef EROFS
		  /* The directory is on a read-only filesystem */
		  case EROFS:
			status = -1;
			break;
#endif
#ifdef ELOOP
		  /* Too many symbolic links encountered */
		  case ELOOP:
			status = -3;
			break;
#endif
		  default:
			status = -1;
			break;
		}
	}

	return(status);
}


/*
 *	Coppies the file.
 *
 *	The src_path specifies the full path of the file to be
 *	coppied. The src_path must be a file.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_file(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	const gboolean originally_move
)
{
	FILE		*tar_fp,
			*src_fp;
	gint		status,
			tar_fd;
	gulong		io_buf_len,
			file_size,
			coppied_size;
	guint8 *io_buf;
	EDVVFSObject *tar_obj;

	/* Open the source file for reading */
	src_fp = fopen((const char *)src_path, "rb");
	if(src_fp == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to open the file for reading:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Create/open the target file for writing */
	tar_fp = fopen((const char *)tar_path, "wb");
	if(tar_fp == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to open the file for writing:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		(void)fclose(src_fp);
		return(-1);
	}

	tar_fd = (gint)fileno(tar_fp);

	/* Get the source file's statistics and the I/O buffer size */
	if(src_lobj != NULL)
	{
		file_size = src_lobj->size;
		io_buf_len = src_lobj->block_size;

		/* If archive is TRUE then copy the permissions of the
		 * source file to the target file
		 */
		if(archive)
			(void)edv_permissions_set_fd(
				tar_fd,
				src_lobj->permissions
			);
	}
	else
	{
		file_size = 0l;
		io_buf_len = 0l;
	}

	/* Get the target file's statistics */
	tar_obj = edv_vfs_object_fstat(tar_fd);
	if(tar_obj != NULL)
	{
		/* Need to use a smaller I/O buffer? */
		if(tar_obj->block_size < io_buf_len)
			io_buf_len = tar_obj->block_size;

		edv_vfs_object_delete(tar_obj);
	}

	/* Copy one block at a time? */
	if(io_buf_len > 1l)
	{
		/* Allocate the I/O buffer to copy each block */
		io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
		if(io_buf == NULL)
		{
			const gint error_code = (gint)errno;
			gchar	*io_buf_size_str = STRDUP(edv_str_size_delim(
							(gulong)io_buf_len
			)),
					*file_size_str = STRDUP(edv_str_size_delim(
							(gulong)file_size
			)),
					*msg = g_strdup_printf(
"Unable to allocate %s %s for the I/O buffer\n\
to %s the file:\n\
\n\
    %s (%s %s)\n\
\n\
%s.",
				io_buf_size_str,
				(io_buf_len == 1l) ? "byte" : "bytes",
				originally_move ? "move" : "copy",
				src_path,
				file_size_str,
				(file_size == 1l) ? "byte" : "bytes",
				g_strerror(error_code)
			);
			g_free(io_buf_size_str);
			g_free(file_size_str);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			(void)fclose(src_fp);
			(void)fclose(tar_fp);
			return(-3);
		}
	}
	else
	{
		io_buf = NULL;			/* Copy one byte at a time */
	}

	status = 0;
	coppied_size = 0l;

	/* Copy one block at a time */
	if(io_buf != NULL)
	{
		size_t	units_read,
			units_written;

		while(!feof(src_fp))
		{
			/* Read this block from the source file */
			units_read = fread(
				io_buf,
				sizeof(guint8),
				(size_t)io_buf_len,
				src_fp
			);
			/* Did an error occur while reading from the source
			 * file?
			 */
			if(ferror(src_fp))
			{
				/* Read error */
				const gint error_code = (gint)errno;

				/* Query the user to try again, but do not check
				 * *yes_to_all or else it may cause an infinite
				 * looping if errors keep reoccuring
				 */
				if(interactive)
				{
					gint response;
					gchar	*file_size_str = STRDUP(edv_str_size_delim(
						(gulong)file_size
					)),
							*pos_str = STRDUP(edv_str_size_delim(
						(gulong)coppied_size
					)),
							
							*msg = g_strdup_printf(
"Unable to read from the file:\n\
\n\
    %s (%s %s)\n\
\n\
At position %s %s.\n\
\n\
%s.",
						src_path,
						file_size_str,
						 (file_size == 1l) ? "byte" : "bytes",
						pos_str,
						(coppied_size == 1l) ? "byte" : "bytes",
						g_strerror(error_code)
					);
					g_free(file_size_str);
					g_free(pos_str);
					edv_play_sound_warning(core);
					CDialogSetTransientFor(toplevel);
					response = CDialogGetResponse(
						"Read Error",
						msg,
						NULL,
						CDIALOG_ICON_WARNING,
						CDIALOG_BTNFLAG_RETRY |
							CDIALOG_BTNFLAG_IGNORE |
							CDIALOG_BTNFLAG_ABORT,
						CDIALOG_BTNFLAG_RETRY
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
					switch(response)
					{
					  case CDIALOG_RESPONSE_RETRY:
						status = -99;    /* Try again */
						break;
					  case CDIALOG_RESPONSE_IGNORE:
						status = -5;    /* Ignore */
						break;
					  case CDIALOG_RESPONSE_ABORT:
						status = -4;    /* Abort/cancel */
						break;
					  default:
						status = -5;    /* Response not available */
						break;
					}
					/* Try again? */
					if(status == -99)
					{
						/* Clear the EOF and error on the source
						 * and seek it back to the previous
						 * position to try again
						 */
						clearerr(src_fp);
						if(fseek(
							src_fp,
							(long)coppied_size,
							SEEK_SET
						))
						{
							/* Failed to seek back to the previous
							 * position
							 */
							const gint error_code = (gint)errno;
							gchar	*file_size_str = STRDUP(edv_str_size_delim(
								(gulong)file_size
							)),
									*pos_str = STRDUP(edv_str_size_delim(
								(gulong)coppied_size
							)),
									*msg = g_strdup_printf(
"Unable to seek the file:\n\
\n\
    %s (%s %s)\n\
\n\
To position %s %s.\n\
\n\
%s.",
								src_path,
								file_size_str,
								(file_size == 1l) ? "byte" : "bytes",
								pos_str,
								(coppied_size == 1l) ? "byte" : "bytes",
								g_strerror(error_code)
							);
							g_free(file_size_str);
							g_free(pos_str);
							edv_vfs_object_op_set_error(core, msg);
							g_free(msg);
							status = -1;	
							break;
						}
						status = 0;
						continue;
					}
				}
				else
				{
					/* Not interactive, report the read error,
					 * but do not try again
					 */
					gchar	*file_size_str = STRDUP(edv_str_size_delim(
						(gulong)file_size
					)),
							*pos_str = STRDUP(edv_str_size_delim(
						(gulong)coppied_size
					)),
							
							*msg = g_strdup_printf(
"Unable to read from the file:\n\
\n\
    %s (%s %s)\n\
\n\
At position %s %s.\n\
\n\
%s.",
						src_path,
						file_size_str,
						(file_size == 1l) ? "byte" : "bytes",
						pos_str,
						(coppied_size == 1l) ? "byte" : "bytes",
						g_strerror(error_code)
					);
					g_free(file_size_str);
					g_free(pos_str);
					edv_vfs_object_op_set_error(core, msg);
					g_free(msg);
					status = -1;
				}

				/* Attempt to write any remaining data that was read */
				if((units_read > 0l) && (units_read <= io_buf_len))
				{
					const size_t units_written = fwrite(
						io_buf,
						sizeof(guint8),
						units_read,
						tar_fp
					);
					if(units_written > 0l)
						coppied_size += units_written;
				}
				break;
			}
			/* No data read? */
			else if(units_read <= 0l)
			{
				/* Check for the EOF marker at the top of this
				 * while() loop
				 */
				continue;
			}

			/* More than the requested number of units read? */
			if(units_read > io_buf_len)
			{
				gchar	*file_size_str = STRDUP(edv_str_size_delim(
					(gulong)file_size
				)),
							*pos_str = STRDUP(edv_str_size_delim(
					(gulong)coppied_size
				)),
							*req_size_str = STRDUP(edv_str_size_delim(
					(gulong)io_buf_len
				)),
							*got_size_str = STRDUP(edv_str_size_delim(
					(gulong)(units_read * sizeof(guint8))
				)),
							*msg = g_strdup_printf(
"Too many bytes were read from the file:\n\
\n\
    %s (%s %s)\n\
\n\
At position %s %s.\n\
\n\
Requested to read %s %s, but read %s %s instead.",
					src_path,
					file_size_str,
					(file_size == 1l) ? "byte" : "bytes",
					pos_str,
					(coppied_size == 1l) ? "byte" : "bytes",
					req_size_str,
					(io_buf_len == 1l) ? "byte" : "bytes",
					got_size_str,
					(units_read == 1l) ? "byte" : "bytes"
				);
				g_free(file_size_str);
				g_free(pos_str);
				g_free(req_size_str);
				g_free(got_size_str);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				status = -3;
				break;
			}

			/* Write this block to the target file */
			units_written = fwrite(
				io_buf,
				sizeof(guint8),
				units_read,
				tar_fp
			);
			if(units_written != units_read)
			{
				/* Write error */
				const gint error_code = (gint)errno;
				gchar	*pos_str = STRDUP(edv_str_size_delim(
					(gulong)coppied_size
				)),
							*msg = g_strdup_printf(
"Unable to write to the file:\n\
\n\
    %s\n\
\n\
At position %s %s.\n\
\n\
%s.",
					tar_path,
					pos_str,
					(coppied_size == 1l) ? "byte" : "bytes",
					g_strerror(error_code)
				);
				g_free(pos_str);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				if(units_written > 0l)
					coppied_size += units_written;
				status = -1;
				break;
			}

			/* Count the number of bytes coppied */
			coppied_size += units_written;

			/* Report progress */
			if(ProgressDialogIsQuery() && (file_size > 0l))
			{
				ProgressDialogUpdate(
					NULL, NULL,
					NULL, NULL,
					(gfloat)coppied_size / (gfloat)file_size,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);

				/* User aborted? */
				if(ProgressDialogStopCount() > 0)
				{
					status = -4;
					break;
				}
			}
		}

		/* Delete the I/O buffer */
		g_free(io_buf);
	}
	else
	{
		/* Copy one byte at a time */
		gulong copy_cnt = 0l;

		/* Read the first character from the source file */
		gint c = (gint)fgetc(src_fp);
		while((int)c != EOF)
		{
			/* Write this character to the target file */
			if(fputc((int)c, tar_fp) == EOF)
			{
				const gint error_code = (gint)errno;
				gchar	*pos_str = STRDUP(edv_str_size_delim(
					(gulong)coppied_size
				)),
							*msg = g_strdup_printf(
"Unable to write to the file:\n\
\n\
    %s\n\
\n\
At position %s %s.\n\
\n\
%s.",
					tar_path,
					pos_str,
					(coppied_size == 1l) ? "byte" : "bytes",
					g_strerror(error_code)
				);
				g_free(pos_str);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				status = -1;
				break;
			}

			/* Increment the copy counter */
			copy_cnt++;

			/* Count the number of bytes coppied */
			coppied_size++;

			/* Time to report the progress? */
			if(copy_cnt >= 10000l)
			{
				/* Report the progress */
				if(ProgressDialogIsQuery() && (file_size > 0l))
				{
					ProgressDialogUpdate(
						NULL, NULL,
						NULL, NULL,
						(gfloat)coppied_size / (gfloat)file_size,
						EDV_PROGRESS_BAR_NTICKS,
						TRUE
					);

					/* User aborted? */
					if(ProgressDialogStopCount() > 0)
					{
						status = -4;
						break;
					}
				}

				copy_cnt = 0l;
			}

			/* Read the next character from the source file */
			c = (gint)fgetc(src_fp);
		}
		/* Did an error occur while reading from the source file? */
		if(ferror(src_fp))
		{
			/* Read error */
			const gint error_code = (gint)errno;

			/* Query the user to try again, but do not check
			 * *yes_to_all or else it may cause an infinite
			 * looping if errors keep reoccuring
			 */
			if(interactive)
			{
				gint response;
				gchar	*file_size_str = STRDUP(edv_str_size_delim(
					(gulong)file_size
				)),
							*pos_str = STRDUP(edv_str_size_delim(
					(gulong)coppied_size
				)),
					
							*msg = g_strdup_printf(
"Unable to read from the file:\n\
\n\
    %s (%s %s)\n\
\n\
At position %s %s.\n\
\n\
%s.",
					src_path,
					file_size_str,
					(file_size == 1l) ? "byte" : "bytes",
					pos_str,
					(coppied_size == 1l) ? "byte" : "bytes",
					g_strerror(error_code)
				);
				g_free(file_size_str);
				g_free(pos_str);
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Read Error",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_IGNORE |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
				switch(response)
				{
				  case CDIALOG_RESPONSE_RETRY:
					status = -99;		/* Try again */
					break;
				  case CDIALOG_RESPONSE_IGNORE:
					status = -5;		/* Ignore */
					break;
				  case CDIALOG_RESPONSE_ABORT:
					status = -4;		/* Abort/cancel */
					break;
				  default:
					status = -5;		/* Response not available */
					break;
				}
			}
			else
			{
				/* Not interactive, report the read error but
				 * do not try again
				 */
				gchar	*file_size_str = STRDUP(edv_str_size_delim(
					(gulong)file_size
				)),
							*pos_str = STRDUP(edv_str_size_delim(
					(gulong)coppied_size
				)),
							*msg = g_strdup_printf(
"Unable to read from the file:\n\
\n\
    %s (%s %s)\n\
\n\
At position %s %s.\n\
\n\
%s.",
					src_path,
					file_size_str,
					(file_size == 1l) ? "byte" : "bytes",
					pos_str,
					(coppied_size == 1l) ? "byte" : "bytes",
					g_strerror(error_code)
				);
				g_free(file_size_str);
				g_free(pos_str);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				status = -1;
			}
		}
	}

	/* Close the source file */
	(void)fclose(src_fp);

	/* Close the target file */
	if(fclose(tar_fp))
	{
		/* Mark that the coppied size is less than the source
		 * file's size so that an incomplete copy gets detected
		 * below
		 */
		if((coppied_size >= file_size) && (file_size > 0l))
			coppied_size = file_size - 1l;
	}

	/* Did not copy all of the data? */
	if(coppied_size < file_size)
	{
		/* No previous error encountered or user abort? */
		if(status == 0)
		{
			/* Report the incomplete copy as an error */
			gchar	*file_size_str = STRDUP(edv_str_size_delim(
				(gulong)file_size
			)),
					*coppied_size_str = STRDUP(edv_str_size_delim(
				(gulong)coppied_size
			)),
					*msg = g_strdup_printf(
"Unable to %s the entire file:\n\
\n\
    %s (%s %s)\n\
\n\
Only %s %s were %s.",
				originally_move ? "move" : "copy",
				src_path,
				file_size_str,
				(file_size == 1l) ? "byte" : "bytes",
				coppied_size_str,
				(coppied_size == 1l) ? "byte" : "bytes",
				originally_move ? "moved" : "coppied"
			);
			g_free(file_size_str);
			g_free(coppied_size_str);
			if(interactive)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					originally_move ? "Move Object Error" : "Copy Object Error",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_IGNORE |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				CDialogSetTransientFor(NULL);
				switch(response)
				{
				  case CDIALOG_RESPONSE_RETRY:
					status = -99;		/* Try again */
					break;
				  case CDIALOG_RESPONSE_IGNORE:
					status = -5;		/* Ignore */
					break;
				  case CDIALOG_RESPONSE_ABORT:
					status = -4;		/* Abort/cancel */
					break;
				  default:
					status = -5;		/* Response not available */
					break;
				}
			}
			else
			{
				edv_vfs_object_op_set_error(core, msg);
				status = -1;
			}
			g_free(msg);
		}
	}

	return(status);
}

/*
 *	Coppies the link.
 *
 *	The src_path specifies the full path of the link to be
 *	coppied. The src_path must be a link.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_link(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
)
{
	gchar *v;

	/* Report the initial progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Get the link's value */
	v = edv_link_get_target(src_path);
	if(v == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to read from the link:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Report progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.5f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			g_free(v);
			return(-4);
		}
	}


	/* Create the new link */
	if(edv_link_create(
		tar_path,				/* Path */
		v					/* Target */
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to copy the link:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			tar_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		g_free(v);
		return(-1);
	}

	/* Delete the link's value */
	g_free(v);

	/* Links do not have any permissions to restore */

	/* Report the final progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			1.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	return(0);
}

/*
 *	Coppies the FIFO pipe.
 *
 *	The src_path specifies the full path of the FIFO pipe to be
 *	coppied. The src_path must be a FIFO pipe.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_fifo(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
)
{
#if defined(S_IFFIFO) || defined(S_IFIFO)
	const mode_t m = (src_lobj != NULL) ?
		edv_edv_permissions_to_stat_mode(src_lobj->permissions) : 0;

	/* Report the initial progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    


	/* Create the new FIFO pipe */
	if(mknod(
		(const char *)tar_path,
#ifdef S_IFFIFO
		m | S_IFFIFO
#else
		m | S_IFIFO
#endif
		,
		0					/* Ignored for FIFO */
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to copy the FIFO pipe:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			tar_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Report progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.5f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	/* Restore the original permissions */
	if(src_lobj != NULL)
	{
		(void)edv_permissions_set(
			tar_path,
			src_lobj->permissions
		);
	}

	/* Report the final progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			1.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	return(0);
#else
	core->last_error_ptr = "Unsupported object type.";
	return(-2);
#endif
}

/*
 *	Coppies the block device.
 *
 *	The src_path specifies the full path of the block device to be
 *	coppied. The src_path must be a block device.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_device_block(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
)
{
#if defined(S_IFBLK)
	const mode_t m = (src_lobj != NULL) ?
		edv_edv_permissions_to_stat_mode(src_lobj->permissions) : 0;

	/* Report the initial progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	/* Create the new block device */
	if(mknod(
		(const char *)tar_path,
		m | S_IFBLK,
		(src_lobj != NULL) ? src_lobj->device_type : 0
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to copy the block device:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			tar_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Report progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.5f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	/* Restore the original permissions */
	if(src_lobj != NULL)
	{
		(void)edv_permissions_set(
			tar_path,
			src_lobj->permissions
		);
	}

	/* Report the final progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			1.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}    

	return(0);
#else
	core->last_error_ptr = "Unsupported object type.";
	return(-2);
#endif
}

/*
 *	Coppies the character device.
 *
 *	The src_path specifies the full path of the character device
 *	to be coppied. The src_path must be a character device.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_device_character(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
)
{
#if defined(S_IFCHR)
	const mode_t m = (src_lobj != NULL) ?
		edv_edv_permissions_to_stat_mode(src_lobj->permissions) : 0;

	/* Report the initial progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Create the new character device */
	if(mknod(
		(const char *)tar_path,
		m | S_IFCHR,
		(src_lobj != NULL) ? src_lobj->device_type : 0
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to copy the character device:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			tar_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Report progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.5f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Restore the original permissions */
	if(src_lobj != NULL)
	{
		(void)edv_permissions_set(
			tar_path,
			src_lobj->permissions
		);
	}

	/* Report the final progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			1.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	return(0);
#else
	core->last_error_ptr = "Unsupported object type.";
	return(-2);
#endif
}

/*
 *	Coppies the socket.
 *
 *	The src_path specifies the full path of the socket to be
 *	coppied. The src_path must be a socket.
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_socket(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj
)
{
#if defined(S_IFSOCK)
	const mode_t m = (src_lobj != NULL) ?
		edv_edv_permissions_to_stat_mode(src_lobj->permissions) : 0;

	/* Report the initial progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Create the new socket */
	if(mknod(
		(const char *)tar_path,
		m | S_IFSOCK,
		0					/* Ignored for sockets */
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to copy the socket:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			src_path, tar_path, g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Report progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			0.5f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Restore the original permissions */
	if(src_lobj != NULL)
	{
		(void)edv_permissions_set(
			tar_path,
			src_lobj->permissions
		);
	}

	/* Report the final progress */
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL,
			NULL, NULL,
			1.0f, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	return(0);
#else
	core->last_error_ptr = "Unsupported object type.";
	return(-2);
#endif
}


/*
 *	Determines the object type and coppies it accordingly (except
 *	directories).
 *
 *	The src_path specifies the full path to the object to copy
 *	(except directories).
 *
 *	The tar_path specifies the full path to the copy to location,
 *	there must not be an object already existing at this location.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_vfs_object_op_copy_nexus(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	const gboolean originally_move
)
{
	EDVObjectType type;
	EDVVFSObject *src_obj_ptr;

	if(STRISEMPTY(src_path) || STRISEMPTY(tar_path))
	{
		core->last_error_ptr = "Path not specified.";
		return(-2);
	}

	/* Choose the source object's statistics, if archive is
	 * TRUE then get its local statistics, otherwise get its
	 * destination's statistics
	 */
	src_obj_ptr = archive ? src_lobj : src_obj;
	if(src_obj_ptr == NULL)
	{
		core->last_error_ptr =
"The source object's statistics was not specified.";
		return(-1);
	}

	type = src_obj_ptr->type;

	/* Copy the source object by its type (except directories) */
	if(type == EDV_OBJECT_TYPE_FILE)
	{
		gint status;
		do {
			status = edv_vfs_object_op_copy_file(
				core,
				src_path,
				tar_path,
				src_lobj,
				src_obj,
				toplevel,
				show_progress,
				interactive,
				archive,
				yes_to_all,
				originally_move
			);
		} while(status == -99);
		return(status);
	}
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
		return(edv_vfs_object_op_copy_link(
			core,
			src_path,
			tar_path,
			src_lobj,
			src_obj
		));
	}
	else if(type == EDV_OBJECT_TYPE_FIFO)
	{
		return(edv_vfs_object_op_copy_fifo(
			core,
			src_path,
			tar_path,
			src_lobj,
			src_obj
		));
	}
	else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
	{
		return(edv_vfs_object_op_copy_device_block(
			core,
			src_path,
			tar_path,
			src_lobj,
			src_obj
		));
	}
	else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	{
		return(edv_vfs_object_op_copy_device_character(
			core,
			src_path,
			tar_path,
			src_lobj,
			src_obj
		));
	}
	else if(type == EDV_OBJECT_TYPE_SOCKET)
	{
		return(edv_vfs_object_op_copy_socket(
			core,
			src_path,
			tar_path,
			src_lobj,
			src_obj
		));
	}
	/* Add support for other objects here (except directories) */
	else
	{
		/* Unsupported object, ignore */
		core->last_error_ptr = "Unsupported object type.";
		return(-2);
	}
}


/*
 *	Moves the object.
 *
 *	The src_path specifies the full path of the object to be moved.
 *
 *	The tar_path specifies the full path of where the object is
 *	to be moved to. If it refers to an existing object then the
 *	user will be prompted to overwrite it. If if refers to an
 *	existing directory or a link to a directory then -2 will
 *	be returned and *need_copy will be set to TRUE.
 *
 *	If the source and target objects are on different devices then
 *	-1 will be returned and *need_copy will be set to TRUE.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	The *status specifies the return value for the status. 0 is
 *	set on success and non-zero is set on error. If the operation
 *	requires a copy instead of a move then -2 will be set and
 *	*need_copy will be set to TRUE.
 *
 *	Returns *status.
 */
static gint edv_vfs_object_op_move_objects_local(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	gint *status,
	gboolean *need_copy
)
{
	gboolean tar_path_exists_locally;
	CfgList *cfg_list = core->cfg_list;
	EDVVFSObject	*tar_lobj,
					*src_obj_ptr;

	/* Is there already an error and it was not a user response
	 * of "no"?
	 */
	if((*status != 0) && (*status != -5))
		return(*status);

	/* Clear the need_copy flag */
	*need_copy = FALSE;

	/* Check if the source and target object paths are identical */
	if(!strcmp((const char *)src_path, (const char *)tar_path))
	{
		core->last_error_ptr =
"The source object and the target object are the same.";
		*status = -2;
		return(*status);
	}

	/* Choose the source object's statistics, if archive is
	 * TRUE then get its local statistics, otherwise get its
	 * destination's statistics
	 */
	src_obj_ptr = archive ? src_lobj : src_obj;
	if(src_obj_ptr == NULL)
	{
		core->last_error_ptr =
"The source object's statistics was not specified.";
		*status = -2;
		return(*status);
	}

	/* Map the Progress Dialog? */
	if(show_progress)
	{
		gchar	*p1 = edv_path_shorten(
			src_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*p2 = edv_path_shorten(
			tar_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Mover:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dmnagement:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bewegen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Trasloco:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mover:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Flytting:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Moving:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			,
			p1,
			p2
		);
		edv_progress_dialog_map_move_animated(
			cfg_list,
			msg,
			0.0f,
			toplevel,
			FALSE
		);
		g_free(msg);
		g_free(p1);
		g_free(p2);

		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
			*status = -4;
			return(*status);
		}
	}

	/* Get the target object's local statistics */
	tar_lobj = edv_vfs_object_lstat(tar_path);
	if(tar_lobj != NULL)
	{
		/* The target object exists locally */
		EDVVFSObject *tar_obj;

		tar_path_exists_locally = TRUE;

		/* Check if the source and target objects are the same
		 * locally according to their local statistics
		 */
		if(edv_vfs_object_op_objects_same(src_lobj, tar_lobj))
		{
			edv_vfs_object_delete(tar_lobj);
			core->last_error_ptr =
"The source object and the target object are the same.";
			*status = -2;
			return(*status);
		}

		/* Get the target object's destination statistics */
		tar_obj = edv_vfs_object_stat(tar_path);

		/* Check if the source object is the same as the target
		 * object's destination
		 */
		if(edv_vfs_object_op_objects_same(src_obj_ptr, tar_obj))
		{
			edv_vfs_object_delete(tar_lobj);
			edv_vfs_object_delete(tar_obj);
			core->last_error_ptr =
"The source object and the target object are the same.";
			*status = -2;
			return(*status);
		}

		/* Does the target object exist and its destination is
		 * a directory?
		 */
		if((tar_obj != NULL) ? EDV_VFS_OBJECT_IS_DIRECTORY(tar_obj) : FALSE)
		{
			/* If the source object is also a directory then mark
			 * that this operation needs to be changed to a copy
			 * operation
			 */
			if(EDV_VFS_OBJECT_IS_DIRECTORY(src_obj_ptr))
			{
				/* Mark that the source directory needs to be
				 * coppied to the target directory
				 */
				*need_copy = TRUE;
				*status = -2;
			}
			else
			{
				/* The source object is not a directory, cannot
				 * move a non-directory to a directory
				 */
				gchar *msg = g_strdup_printf(
"The target object:\n\
\n\
    %s\n\
\n\
Already exists and it is a directory.",
					tar_path
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				*status = -2;
			}
			edv_vfs_object_delete(tar_lobj);
			edv_vfs_object_delete(tar_obj);
			return(*status);
		}

		/* The target object exists locally so we need to confirm
		 * overwrite
		 */
		if(interactive && !(*yes_to_all))
		{
			const gint response = edv_vfs_object_op_confirm_overwrite(
				core,
				src_path,
				tar_path,
				src_lobj,
				tar_lobj,
				toplevel
			);
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
			  case CDIALOG_RESPONSE_OK:
				break;
			  case CDIALOG_RESPONSE_CANCEL:
				/* Respond with user abort error code, this will
				 * cause the calling function to detect an error
				 * and stop any looping operations
				 */
				edv_vfs_object_delete(tar_lobj);
				edv_vfs_object_delete(tar_obj);
				*status = -4;
				return(*status);
				break;
			  default:
				/* Respond with no or not available code, the
				 * calling function will see this as an error but
				 * should still continue with futher loop operations
				 */
				edv_vfs_object_delete(tar_lobj);
				edv_vfs_object_delete(tar_obj);
				*status = -5;
				return(*status);
				break;
			}
		}

		edv_vfs_object_delete(tar_lobj);
		edv_vfs_object_delete(tar_obj);
	}
	else
	{
		/* Unable to get the target object's local statistics,
		 * check why
		 */
		const gint error_code = (gint)errno;
		switch(error_code)
		{
#ifdef ENOENT
		  /* Does not exist */
		  case ENOENT:
			tar_path_exists_locally = FALSE;
			break;
#endif
#ifdef ENOTDIR
		  /* Does not exist */
		  case ENOTDIR:
			tar_path_exists_locally = FALSE;
			break;
#endif
#ifdef EACCES
		  /* Access denied */
		  case EACCES:
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Access denied to the target location:\n\
\n\
    %s",
					tar_path
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
			}
			*status = -1;
			return(*status);
			break;
#endif
#ifdef EPERM
		  /* Permission denied */
		  case EPERM:
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Permission denied to the target location:\n\
\n\
    %s",
					tar_path
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
			}
			*status = -1;
			return(*status);
			break;
#endif
		  /* All else assume that it might exist */
		  default:
			tar_path_exists_locally = TRUE;
			break;
		}
	}

	/* If the target object exists locally then it must be
	 * deleted before we can move the source there (user
	 * confirmation has already been made)
	 */
	if(tar_path_exists_locally)
	{
		const gint status2 = edv_vfs_object_op_unlink(
			core,
			tar_path
		);
		if(status2 != 0)
		{
			*status = status2;
			return(*status);
		}
	}

	/* Move the source object to the target object */
	if(edv_rename(
		src_path,
		tar_path
	))
	{
		const gint error_code = (gint)errno;
#define SET_LAST_ERROR(_error_code_)	{	\
 if(core->last_error_ptr == NULL) {		\
  gchar *msg = g_strdup_printf(			\
"Unable to move the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",						\
   src_path,					\
   tar_path,					\
   g_strerror(_error_code_)			\
  );						\
  edv_vfs_object_op_set_error(core, msg);	\
  g_free(msg);					\
 }						\
}
		switch(error_code)
		{
#ifdef EISDIR
		  /* The target is a directory but the source is not */
		  case EISDIR:
			SET_LAST_ERROR(error_code);
			*status = -2;
			break;
#endif
#ifdef EXDEV
		  /* The source and the target objects are not different
		   * physical devices
		   */
		  case EXDEV:
			/* Need to copy the object across a physical device,
			 * set *need_copy to TRUE and *status to -2 so that
			 * the calling function knows that it failed and
			 * needs to copy the object instead
			 */
			*need_copy = TRUE;

			/* Clear the error message since we don't want one
			 * being generated because of this
			 */
			edv_vfs_object_op_set_error(core, NULL);

			*status = -2;
			break;
#endif
#if defined(ENOTEMPTY)
		  /* The target location is not an empty directory */
		  case ENOTEMPTY:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#elif defined(EEXIST)
		  /* The target location is not an empty directory */
		  case EEXIST:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef EBUSY
		  /* The object is currently in use */
		  case EBUSY:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef EINVAL
		  /* Invalid target location */
		  case EINVAL:
			SET_LAST_ERROR(error_code);
			*status = -2;
			break;
#endif
#ifdef EMLINK
		  /* The source object already has the maximum number of links to it */
		  case EMLINK:
			SET_LAST_ERROR(error_code);
			*status = -2;
			break;
#endif
#ifdef ENOTDIR
		  /* A compoent of the path is not, in fact, a directory or
		   * the source object is a directory and the target object
		   * is not
		   */
		  case ENOTDIR:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef EFAULT
		  /* Segmentation fault */
		  case EFAULT:
			SET_LAST_ERROR(error_code);
			*status = -3;
			break;
#endif
#ifdef EACCES
		  /* Access denied */
		  case EACCES:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef EPERM
		  /* Permission denied */
		  case EPERM:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef ENAMETOOLONG
		  /* Name too long */
		  case ENAMETOOLONG:
			SET_LAST_ERROR(error_code);
			*status = -2;
			break;
#endif
#ifdef ENOENT
		  /* The source object or a compoent of the target object
		   * does not exist
		   */
		  case ENOENT:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef ENOMEM
		  /* Out of memory */
		  case ENOMEM:
			SET_LAST_ERROR(error_code);
			*status = -3;
			break;
#endif
#ifdef EROFS
		  /* The source object or the target object is on a read-only filesystem */
		  case EROFS:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
#endif
#ifdef ELOOP
		  /* Too many symbolic links encountered */
		  case ELOOP:
			SET_LAST_ERROR(error_code);
			*status = -3;
			break;
#endif
#ifdef ENOSPC
		  /* Out of disk space */
		  case ENOSPC:
			SET_LAST_ERROR(error_code);
			*status = -3;
			break;
#endif
		  default:
			SET_LAST_ERROR(error_code);
			*status = -1;
			break;
		}
#undef SET_LAST_ERROR
	}

	return(*status);
}

/*
 *	Coppies the object.
 *
 *	The src_path specifies the full path to the object to be coppied.
 *	It must not locally be a directory.
 *
 *	The tar_path specifies the full path to where the object is to
 *	be coppied to. If this location already exists and interactive
 *	is TRUE then tue user will be prompted to overwrite the
 *	existing object.
 *
 *	If the tar_path already exists locally as a directory then -2
 *	will be returned.
 *
 *	If originally_move is TRUE then all messages will indicate
 *	that this is a move instead of a copy. This is needed in cases
 *	where the a directory is being moved to a directory or an
 *	object is being moved across a physical device.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns *status.
 */
static gint edv_vfs_object_op_copy_objects_local(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gboolean *yes_to_all,
	gint *status,
	const gboolean originally_move
)
{
	gboolean tar_path_exists_locally;
	gint status2;
	CfgList *cfg_list = core->cfg_list;
	EDVVFSObject	*tar_lobj,
					*src_obj_ptr;

	/* Is there already an error and it was not a user response
	 * of "no"?
	 */
	if((*status != 0) && (*status != -5))
		return(*status);

	/* Check if paths are the same */
	if(!strcmp((const char *)src_path, (const char *)tar_path))
	{
		core->last_error_ptr =
"The source object and the target object are the same.";
		*status = -2;
		return(*status);
	}

	/* Choose the source object's statistics to use, if archive
	 * is TRUE then use its local statistics, otherwise use its
	 * destination statistics
	 */
	src_obj_ptr = archive ? src_lobj : src_obj;
	if(src_obj_ptr == NULL)
	{
		core->last_error_ptr =
"The source object's statistics was not specified.";
		*status = -2;
		return(*status);
	}

	/* Update the progress dialog? */
	if(show_progress)
	{
		gchar	*p1 = edv_path_shorten(
			src_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
					*p2 = edv_path_shorten(
			tar_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Mover" : "Copiar",
#elif defined(PROG_LANGUAGE_FRENCH)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Dmnagement" : "Copier",
#elif defined(PROG_LANGUAGE_GERMAN)
"%s:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n",
			originally_move ? "Bewegen" : "Kopieren",
#elif defined(PROG_LANGUAGE_ITALIAN)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Il Trasloco" : "Copiare",
#elif defined(PROG_LANGUAGE_DUTCH)
"%s:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n",
			originally_move ? "Bewegen" : "Kopiren",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Mover" : "Copiar",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"%s:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n",
			originally_move ? "Flytting" : "Kopiering",
#else
"%s:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
			originally_move ? "Moving" : "Copying",
#endif
			p1,
			p2
		);
		g_free(p1);
		g_free(p2);
		if(originally_move)
			edv_progress_dialog_map_move_animated(
				cfg_list,
				msg,
				0.0f,
				toplevel,
				FALSE
			);
		else
			edv_progress_dialog_map_copy_animated(
				cfg_list,
				msg,
				0.0f,
				toplevel,
				FALSE
			);
		g_free(msg);

		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
			*status = -4;
			return(*status);
		}
	}

	/* Get the target object's local statistics */
	tar_lobj = edv_vfs_object_lstat(tar_path);
	if(tar_lobj != NULL)
	{
		/* The target object exists locally */
		EDVVFSObject *tar_obj;

		tar_path_exists_locally = TRUE;

		/* Check if source and target objects are the same locally */
		if(edv_vfs_object_op_objects_same(src_lobj, tar_lobj))
		{
			edv_vfs_object_delete(tar_lobj);
			core->last_error_ptr =
"The source object and the target object are the same.";
			*status = -2;
			return(*status);
		}

		/* Get the target object's destination statistics */
		tar_obj = edv_vfs_object_stat(tar_path);

		/* Check if source and target object destinations are
		 * the same
		 */
		if(edv_vfs_object_op_objects_same(src_obj_ptr, tar_obj))
		{
			edv_vfs_object_delete(tar_lobj);
			edv_vfs_object_delete(tar_obj);
			core->last_error_ptr =
"The source object and the target object are the same.";
			*status = -2;
			return(*status);
		}

		/* Is target object locally a directory? */
		if(EDV_VFS_OBJECT_IS_DIRECTORY(tar_lobj))
		{
			gchar *msg = g_strdup_printf(
"The target object:\n\
\n\
    %s\n\
\n\
Already exists and it is a directory.",
				tar_path
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			edv_vfs_object_delete(tar_lobj);
			edv_vfs_object_delete(tar_obj);
			*status = -2;
			return(*status);
		}

		/* Target object exists and is not a directory, ask user for
		 * confirmation on overwriting it?
		 */
		if(interactive && !(*yes_to_all))
		{
			const gint response = edv_vfs_object_op_confirm_overwrite(
				core,
				src_path,
				tar_path,
				src_lobj,
				tar_lobj,
				toplevel
			);
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
			  case CDIALOG_RESPONSE_OK:
				break;

			  case CDIALOG_RESPONSE_CANCEL:
				/* Respond with the user abort error code, this
				 * will cause the calling function to detect an
				 * error and stop any looping operations
				 */
				edv_vfs_object_delete(tar_lobj);
				edv_vfs_object_delete(tar_obj);
				*status = -4;
				return(*status);
				break;

			  default:
				/* Respond with the no or not available error code,
				 * the calling function will see this as an error
				 * but should still continue with futher loop
				 * operations
				 */
				edv_vfs_object_delete(tar_lobj);
				edv_vfs_object_delete(tar_obj);
				*status = -5;
				return(*status);
				break;
			}
		}

		edv_vfs_object_delete(tar_lobj);
		edv_vfs_object_delete(tar_obj);
	}
	else
	{
		/* Unable to get the target object's local statistics,
		 * check why
		 */
		const gint error_code = (gint)errno;
		switch(error_code)
		{
#ifdef ENOENT
		  /* Does not exist */
		  case ENOENT:
			tar_path_exists_locally = FALSE;
			break;
#endif
#ifdef ENOTDIR
		  /* Does not exist */
		  case ENOTDIR:
			tar_path_exists_locally = FALSE;
			break;
#endif
#ifdef EACCES
		  /* Access denied */
		  case EACCES:
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Access denied to the target location:\n\
\n\
    %s",
					tar_path
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
			}
			*status = -1;
			return(*status);
			break;
#endif
#ifdef EPERM
		  /* Permission denied */
		  case EPERM:
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Permission denied to the target location:\n\
\n\
    %s",
					tar_path
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
			}
			*status = -1;
			return(*status);
			break;
#endif
		  /* All else assume that it might exist */
		  default:
			tar_path_exists_locally = TRUE;
			break;
		}
	}

	/* If the target object exists locally then it must be
	 * deleted before we can move the source there
	 */
	if(tar_path_exists_locally)
	{
		const gint status2 = edv_vfs_object_op_unlink(
			core,
			tar_path
		);
		if(status2 != 0)
		{
			*status = status2;
			return(*status);
		}
	}

	/* Copy the source object to the target object
	 *
	 * This will check the source object's type and copy it
	 * accordingly
	 *
	 * The tar_path must not exist
	 */
	status2 = edv_vfs_object_op_copy_nexus(
		core,
		src_path,
		tar_path,
		src_lobj,
		src_obj,
		toplevel,
		show_progress,
		interactive,
		archive,
		yes_to_all,
		originally_move
	);
	if(status2 != 0)
	{
		*status = status2;
	}
	else
	{
		/* If the operation was originally a move or archive is
		 * TRUE then restore the original ownership and time
		 * stamps
		 */
		if(originally_move || archive)
		{
			(void)edv_lchown(
				tar_path,
				src_obj_ptr->owner_id,
				src_obj_ptr->group_id
			);

			/* Restore the time stamps only if the object was not
			 * a link, since links do not have time stamps
			 */
			if(!EDV_VFS_OBJECT_IS_LINK(src_obj_ptr))
			{
				(void)edv_utime(
					tar_path,
					src_obj_ptr->access_time,
					src_obj_ptr->modify_time
				);
			}
		}
	}

	return(*status);
}


/*
 *	Coppies the directory recursively.
 *
 *	The src_dir specifies the full path to the directory to copy.
 *	All contents within it will be coppied.
 *
 *	The tar_dir specifies the full path to where the directory is
 *	to be coppied to.
 *
 *	If originally_move is TRUE then all messages will indicate
 *	that this is a move instead of a copy.
 *
 *	If an error occures then last_error will be set to the error
 *	message that describes what error occured.
 *
 *	Returns *status.
 */
static gint edv_vfs_object_op_copy_directory_iterate(
	EDVCore *core,
	const gchar *src_dir,
	const gchar *tar_dir,
	EDVVFSObject *src_lobj,
	EDVVFSObject *src_obj,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	const gboolean archive,
	gint *status,
	gboolean *yes_to_all,
	const gboolean originally_move
)
{
	GList *names_list;
	CfgList *cfg_list = core->cfg_list;
	EDVVFSObject *tar_obj;

	/* Is there already an error and it was not a user response
	 * of "no"?
	 */
	if((*status != 0) && (*status != -5))
		return(*status);

	/* Check if the source and target paths are identical */
	if(!strcmp((const char *)src_dir, (const char *)tar_dir))
	{
		core->last_error_ptr =
"The source directory and the target directory are the same.";
		*status = -2;
		return(*status);
	}

	/* Must have the source object's destination statistics */
	if(src_obj == NULL)
	{
		core->last_error_ptr =
"The source directory's statistics was not specified.";
		*status = -2;
		return(*status);
	}

	/* Map the Progress Dialog? */
	if(show_progress)
	{
		gchar	*p1 = edv_path_shorten(
			src_dir,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
					*p2 = edv_path_shorten(
			tar_dir,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Mover" : "Copiar",
#elif defined(PROG_LANGUAGE_FRENCH)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Dmnagement" : "Copier",
#elif defined(PROG_LANGUAGE_GERMAN)
"%s:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n",
			originally_move ? "Bewegen" : "Kopieren",
#elif defined(PROG_LANGUAGE_ITALIAN)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Il Trasloco" : "Copiare",
#elif defined(PROG_LANGUAGE_DUTCH)
"%s:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n",
			originally_move ? "Bewegen" : "Kopiren",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"%s:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
			originally_move ? "Mover" : "Copiar",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"%s:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n",
			originally_move ? "Flytting" : "Kopiering",
#else
"%s:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
			originally_move ? "Moving" : "Copying",
#endif
			p1,
			p2
		);
		g_free(p1);
		g_free(p2);
		if(originally_move)
			edv_progress_dialog_map_move_animated(
				cfg_list,
				msg,
				0.0f,
				toplevel,
				FALSE
			);
		else
			edv_progress_dialog_map_copy_animated(
				cfg_list,
				msg,
				0.0f,
				toplevel,
				FALSE
			);
		g_free(msg);

		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
			*status = -4;
			return(*status);
		}
	}


	/* Get the target directory's destination statistics and
	 * check if it is the same as thet source object
	 */
	tar_obj = edv_vfs_object_stat(tar_dir);
	if(tar_obj != NULL)
	{
		/* Check if the source and target directories are the same */
		if(edv_vfs_object_op_objects_same(src_obj, tar_obj))
		{
			edv_vfs_object_delete(tar_obj);
			core->last_error_ptr =
"The source directory and the target directory are the same.";
			*status = -2;
			return(*status);
		}
	}

	/* The specified source destination is a directory so we do not
	 * need to check if the source and target are the same locally
	 */


	/* Create the target directory as needed */
	if(edv_directory_create(
		tar_dir,
		FALSE,				/* Do not create parents */
		NULL				/* No new paths list return */
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to create the directory:\n\
\n\
    %s\n\
\n\
%s.",
			tar_dir,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		switch(error_code)
		{
		  case EEXIST:
			/* The target object exists, check if it is a
			 * directory or a link who's destination is a valid
			 * directory
 		 */
			if((tar_obj != NULL) ?
				EDV_VFS_OBJECT_IS_DIRECTORY(tar_obj) : FALSE
			)
			{
				/* The target object is an existing directory or
				 * a link who's destination is a valid directory,
				 * so do not indicate any error and continue
				 * through
				 */
				edv_vfs_object_op_set_error(core, NULL);
			}
			else
			{
				gchar *msg = g_strdup_printf(
"The target object:\n\
\n\
    %s\n\
\n\
Exists and is not a directory.",
					tar_dir
				);
				edv_vfs_object_op_set_error(core, msg);
				g_free(msg);
				edv_vfs_object_delete(tar_obj);
				*status = -1;
				return(*status);
			}
			break;
		  case EFAULT:
			edv_vfs_object_delete(tar_obj);
			*status = -3;
			return(*status);
			break;
		  case EACCES:
			edv_vfs_object_delete(tar_obj);
			*status = -2;
			return(*status);
			break;
		  case ENAMETOOLONG:
			edv_vfs_object_delete(tar_obj);
			*status = -2;
			return(*status);
			break;
		  case ENOENT:
		  case ENOTDIR:
			edv_vfs_object_delete(tar_obj);
			*status = -2;
			return(*status);
			break;
		  case ENOMEM:
			edv_vfs_object_delete(tar_obj);
			*status = -3;
			return(*status);
			break;
		  case EROFS:
			edv_vfs_object_delete(tar_obj);
			*status = -2;
			return(*status);
			break;
		  case ELOOP:
			edv_vfs_object_delete(tar_obj);
			*status = -3;
			return(*status);
			break;
		  case ENOSPC:
			edv_vfs_object_delete(tar_obj);
			*status = -3;
			return(*status);
		  default:
			edv_vfs_object_delete(tar_obj);
			*status = -1;
			return(*status);
			break;
		}
	}

	if(show_progress)
	{
		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(tar_obj);
			*status = -4;
			return(*status);
		}
	}


	/* Get the listing of the contents in the source directory
	 * and copy each one to the target directory
	 */
	names_list = edv_directory_list(
		src_dir,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gchar *name;
		gchar	*src_child,
			*tar_child;
		GList *glist;
		EDVVFSObject	*src_child_lobj,
				*src_child_obj;

		/* Copy each object in this directory */
		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			src_child = edv_paths_join(
				src_dir,
				name
			);
			if(src_child == NULL)
			{
				g_free(glist->data);
				continue;
			}

			tar_child = edv_paths_join(
				tar_dir,
				name
			);
			if(tar_child == NULL)
			{
				g_free(src_child);
				g_free(glist->data);
				continue;
			}

			/* Get this object's local statistics */
			src_child_lobj = edv_vfs_object_lstat(src_child);
			if(src_child_lobj == NULL)
			{
				g_free(src_child);
				g_free(tar_child);
				g_free(glist->data);
				continue;
			}

			/* Get this object's destination statistics but do
			 * not continue or break if there is an error
			 */
			src_child_obj = edv_vfs_object_stat(src_child);

			/* Check if the source object is locally a directory
			 * or if the destination is a directory and archive
			 * is FALSE, if so then we will recurse into this
			 * directory
			 */
			if(archive ?
				EDV_VFS_OBJECT_IS_DIRECTORY(src_child_lobj) :
				EDV_VFS_OBJECT_IS_DIRECTORY(src_child_obj)
			)
			{
				/* Copy this directory and all of its contents
				 * to the target directory
				 */
				(void)edv_vfs_object_op_copy_directory_iterate(
					core,
					src_child,
					tar_child,
					src_child_lobj,
					src_child_obj,
					toplevel,
					show_progress,
					interactive,
					archive,
					status,
					yes_to_all,
					originally_move
				);
			}
			else
			{
				/* Copy this source object to the target
				 * directory
				 */
				(void)edv_vfs_object_op_copy_objects_local(
					core,
					src_child,
					tar_child,
					src_child_lobj,
					src_child_obj,
					toplevel,
					show_progress,
					interactive,
					archive,
					yes_to_all,
					status,
					originally_move
				);
			}

			edv_vfs_object_delete(src_child_lobj);
			edv_vfs_object_delete(src_child_obj);
			g_free(src_child);
			g_free(tar_child);

			/* Error encountered and it was not a user "no" or
			 * "not available" response?
			 */
			if((*status != 0) && (*status != -5))
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}

	/* If the original operation was a move or archive is TRUE
	 * then restore the original ownership and time stamps on
	 * the target directory
	 */
	if(originally_move || archive)
	{
		(void)edv_permissions_set(
			tar_dir,
			src_obj->permissions
		);

		(void)edv_chown(
			tar_dir,
			src_obj->owner_id,
			src_obj->group_id
		);

		(void)edv_utime(
			tar_dir,
			src_obj->access_time,
			src_obj->modify_time
		);
	}

	edv_vfs_object_delete(tar_obj);

	return(*status);
}


/*
 *	Coppies or moves the object.
 *
 *	The src_path specifies the full path to the object to be
 *	coppied or moved.
 *
 *	The tar_path specifies the full path to where the object is to
 *	be coppied or moved to.
 *
 *	If new_path_rtn is not NULL then a dynamically allocated string
 *	describing the full path of the new object will be returned.
 *
 *	If the src_path is a directory then that directory and of its
 *	contents will be recursivly copied or moved.
 *
 *	If new_path_rtn is not NULL then it will be set with a
 *	dynamically allocated string describing the newly added
 *	object's path.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped as needed and updated. The calling function needs to 
 *	unmap the progress dialog.
 *
 *	If interactive is TRUE then the user will be prompted for
 *	overwrites as needed.
 *	
 *	If is_copy is TRUE then the operation will be a copy, otherwise
 *	the operation will be a move unless src_path and tar_path are
 *	on different devices in which case the operation will become a
 *	copy but with messages stating that it is still a move and
 *	src_path will be removed upon success.
 *
 *	Returns 0 on success and non-zero on error.
 */
static gint edv_vfs_object_op_copy_or_move(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_copy
)
{
	const gulong	time_start = edv_time();
	gboolean	need_copy = is_copy,
					originally_move = !is_copy;
	gint status = 0;
	gchar		*lsrc_path,
					*ltar_path;
	GList *glist;
	EDVVFSObject	*src_lobj = NULL,
					*src_obj = NULL;

	if(new_paths_list_rtn != NULL)
		*new_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(src_path) ||
	   STRISEMPTY(tar_path) || (yes_to_all == NULL)
	)
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 edv_vfs_object_delete(src_lobj);	\
 edv_vfs_object_delete(src_obj);	\
 g_free(lsrc_path);			\
 g_free(ltar_path);			\
									\
 return(_v_);				\
}

	/* Copy the source and target paths */
	lsrc_path = STRDUP(src_path);
	ltar_path = STRDUP(tar_path);
	if((lsrc_path == NULL) || (ltar_path == NULL))
	{
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}

	/* Simplify the source and target paths */
	edv_path_simplify(lsrc_path);
	edv_path_simplify(ltar_path);


	/* Make sure that the paths are not exactly the same, this will
	 * be checked again later when the paths are prefixed as needed
	 * but an initial check needs to be done first to user does not
	 * see any of the more exotic errors
	 */
	if(!strcmp((const char *)lsrc_path, (const char *)ltar_path))
	{
		core->last_error_ptr = "The source object and the target object are the same.";
		core->op_level--;
		CLEANUP_RETURN(-2);
	}

	/* Make sure that the source object is not a parent or grand
	 * parent of the target object since it is impossible to copy
	 * or move a parent into a child
	 */
	if(edv_path_is_parent(
		lsrc_path,				/* Parent */
		ltar_path				/* Child */
	))
	{
		if(need_copy)
			core->last_error_ptr = "Unable to copy a parent into its child.";
		else
			core->last_error_ptr = "Unable to move a parent into its child.";
		core->op_level--;
		CLEANUP_RETURN(-2);
	}


	/* Get the source object's local and destination statistics */
	src_lobj = edv_vfs_object_lstat(lsrc_path);
	src_obj = edv_vfs_object_stat(lsrc_path);
	if(archive ? (src_lobj == NULL) : (src_obj == NULL))
	{
		/* Unable to get the statistics for the source object */
		core->last_error_ptr =
"Unable to get the source object's statistics.";
		core->op_level--;
		CLEANUP_RETURN(-2);
	}

	/* Got the source object statistics, now perform the copy/move
	 * by the source object's type
	 *
	 * Check if the source object is a directory
	 */
	if(archive ?
		EDV_VFS_OBJECT_IS_DIRECTORY(src_lobj) :
		EDV_VFS_OBJECT_IS_DIRECTORY(src_obj)
	)
	{
		/* The source object is a directory */
		gboolean need_remove_src = FALSE;
		gint response;

		/* For directories, always set the target child path by
		 * prefixing the specified target path to the source
		 * object's name (so in this case the target child path
		 * always has the source object's name)
		 */
		gchar *tar_child_path = edv_paths_join(
			ltar_path,
			g_basename(lsrc_path)
		);
		if(tar_child_path == NULL)
		{
			core->last_error_ptr =
"Unable to generate the full path to the target.";
			core->op_level--;
			CLEANUP_RETURN(-1);
		}

		/* Need to create any non-existant parent directories in
		 * the target path?
		 */
		response = edv_vfs_object_op_rmkdir(
			core,
			ltar_path,
			new_paths_list_rtn,
			interactive,
			yes_to_all,
			toplevel
		);
		if(response == -4)
		{
			core->op_level--;
			CLEANUP_RETURN(-4);
		}
		else if(response == -2)
		{
			gchar *msg = g_strdup_printf(
"Not a directory:\n\
\n\
    %s",
				ltar_path
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			core->op_level--;
			CLEANUP_RETURN(-2);
		}
		else if(response != 0)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to create the parent directories of:\n\
\n\
    %s\n\
\n\
%s.",
				ltar_path,
				g_strerror(error_code)
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			core->op_level--;
			CLEANUP_RETURN(-1);
		}

		/* Begin copying or moving the source directory and all
		 * of its contents to the target child path location
		 *
		 * Is this a move?
		 */
		if(!need_copy)
		{
			(void)edv_vfs_object_op_move_objects_local(
				core,
				lsrc_path,
				tar_child_path,
				src_lobj,
				src_obj,
				toplevel,
				show_progress,
				interactive,
				archive,
				yes_to_all,
				&status,
				&need_copy
			);

			/* If the above operation changed from a move to a
			 * copy then make note that the source object needs
			 * to be removed later
			 */
			if(need_copy)
				need_remove_src = TRUE;
		}

		/* Is a copy requested or needed?
		 *
		 * The need_copy can either be specified as TRUE or set
		 * to TRUE when edv_vfs_object_op_move_objects_local() was called
		 * to indicate that the source object needs to be coppied
		 * instead of moved
		 */
		if(need_copy)
		{
			/* Reset the status incase this was originally a move
			 * and a copy needs to be performed instead
			 */
			if(status != 0)
				status = 0;
			(void)edv_vfs_object_op_copy_directory_iterate(
				core,
				lsrc_path,
				tar_child_path,
				src_lobj,
				src_obj,
				toplevel,
				show_progress,
				interactive,
				archive,
				&status,
				yes_to_all,
				originally_move
		    );
		}
		/* Source directory needs to be removed (if operation was
		 * originally a move and a copy across devices took place
		 * instead)?
		 *
		 * Also do not remove if there was an error detected
		 */
		if(need_remove_src && (status == 0))
			(void)edv_vfs_object_op_rmdir(
				core,
				lsrc_path
			);

		/* Add the child object path to the list of new paths */
		if(new_paths_list_rtn != NULL)
			*new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn,
				g_strdup(tar_child_path)
			);

		g_free(tar_child_path);
	}
	else
	{
		/* The source object is not a directory */
		gboolean need_remove_src = FALSE;
		EDVVFSObject *tar_obj;

		/* Set the path to the target child exactly as the
		 * specified target path, change later if the target
		 * child path refers to a directory
		 */
		gchar *tar_child_path = g_strdup(ltar_path);
		if(tar_child_path == NULL)
		{
			core->last_error_ptr =
"Unable to generate the full path to the target.";
			core->op_level--;
			CLEANUP_RETURN(-1);
		}

		/* When the source object is not a directory then the target
		 * object's existance needs to be checked before appending
		 * the source object's name to the target path
		 *
		 * If and only if the target path exists and is a directory
		 * should the source object's name be appended to the target
		 * path, this is to allow objects being moved/coppied to a
		 * destination with a new name
		 */
		tar_obj = edv_vfs_object_stat(ltar_path);
		if(tar_obj != NULL)
		{
			/* The target path exists, check if it is a directory */
			if(EDV_VFS_OBJECT_IS_DIRECTORY(tar_obj))
			{
				/* Change the target child path to that of the
				 * specified target path with the source object's
				 * name postfixed to it
				 */
				g_free(tar_child_path);
				tar_child_path = edv_paths_join(
					ltar_path,
					g_basename(lsrc_path)
				);
				if(tar_child_path == NULL)
				{
					edv_vfs_object_delete(tar_obj);
					core->last_error_ptr =
"Unable to generate the full path to the target.";
					core->op_level--;
					CLEANUP_RETURN(-1);
				}
			}

			edv_vfs_object_delete(tar_obj);
		}
		else
		{
			/* The target path does not exist or has an error,
			 * check if its parent directories need to be created
			 */
			gchar *parent_path = g_dirname(ltar_path);
			if(parent_path != NULL)
			{
				const gint response = edv_vfs_object_op_rmkdir(
					core,
					parent_path,
					new_paths_list_rtn,
					interactive,
					yes_to_all,
					toplevel
				);
				if(response == -4)
				{
					g_free(parent_path);
					core->op_level--;
					CLEANUP_RETURN(-4);
				}
				else if(response == -2)
				{
					gchar *msg = g_strdup_printf(
"Not a directory:\n\
\n\
    %s",
						parent_path
					);
					edv_vfs_object_op_set_error(core, msg);
					g_free(msg);
					g_free(parent_path);
					core->op_level--;
					CLEANUP_RETURN(-2);
				}
				else if(response != 0)
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to create the parent directories of:\n\
\n\
    %s\n\
\n\
%s.",
						parent_path,
						g_strerror(error_code)
					);
					edv_vfs_object_op_set_error(core, msg);
					g_free(msg);
					g_free(parent_path);
					core->op_level--;
					CLEANUP_RETURN(-1);
				}
				g_free(parent_path);
			}
		}

		/* Begin copying or moving the source object to the
		 * location specified by tar_child_path
		 *
		 * Is this a move?
		 */
		if(!need_copy)
		{
			(void)edv_vfs_object_op_move_objects_local(
				core,
				lsrc_path,
				tar_child_path,
				src_lobj,
				src_obj,
				toplevel,
				show_progress,
				interactive,
				archive,
				yes_to_all,
				&status,
				&need_copy
			);

			/* If the above operation changed from a move to a
			 * copy then the source object needs to be removed
			 */
			if(need_copy)
				need_remove_src = TRUE;
		}

		/* Is a copy requested or needed?
		 *
		 * The need_copy can either be specified as TRUE or set
		 * to TRUE when edv_vfs_object_op_move_objects_local() was called
		 * to indicate that the source object needs to be coppied
		 * instead of moved
		 */
		if(need_copy)
		{
			/* Reset the status incase this was originally a move
			 * and a copy needs to be performed instead
			 */
			if(status != 0)
				status = 0;
			(void)edv_vfs_object_op_copy_objects_local(
				core,
				lsrc_path,
				tar_child_path,
				src_lobj,
				src_obj,
				toplevel,
				show_progress,
				interactive,
				archive,
				yes_to_all,
				&status,
				originally_move
		    );
		}
		/* Does the source object need to be removed (if the
		 * operation was originally a move and a copy across
		 * devices took place instead)?
		 *
		 * Also do not remove if there was an error detected
		 */
		if(need_remove_src && (status == 0))
			(void)edv_vfs_object_op_unlink(
				core,
				lsrc_path
			);

		/* Add the child object to the list of new paths */
		if(new_paths_list_rtn != NULL)
			*new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn,
				g_strdup(tar_child_path)
			);

		g_free(tar_child_path);
	}

	/* Record history */
	glist = (new_paths_list_rtn != NULL) ?
		g_list_last(*new_paths_list_rtn) : NULL;
	edv_append_history(
		core,
		originally_move ?
			EDV_HISTORY_VFS_OBJECT_MOVE : EDV_HISTORY_VFS_OBJECT_COPY,
		time_start,
		edv_time(),
		status,
		lsrc_path,				/* Source */
		(glist != NULL) ?			/* Target */
			(const gchar *)glist->data : ltar_path,
		core->last_error_ptr		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Coppies the object.
 *
 *	The src_path specifies the full path to the object to copy.
 *
 *	The tar_path specifies the full path to the destination.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * paths
 *	describing the objects that have been created will be returned.
 *	The calling function must delete the returned list and each
 *	string.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_vfs_object_op_copy(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(edv_vfs_object_op_copy_or_move(
		core,
		src_path,
		tar_path,
		new_paths_list_rtn,
		archive,
		toplevel,
		show_progress,
		interactive,
		yes_to_all,
		TRUE				/* Copy */
	));
}

/*
 *	Moves the object.
 *
 *	The src_path specifies the full path to the object to move.
 *
 *	The tar_path specifies the full path to the destination.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * paths
 *	describing the objects that have been created will be returned.
 *	The calling function must delete the returned list and each
 *	string.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_vfs_object_op_move(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	GList **new_paths_list_rtn,
	const gboolean archive,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(edv_vfs_object_op_copy_or_move(
		core,
		src_path,
		tar_path,
		new_paths_list_rtn,
		archive,
		toplevel,
		show_progress,
		interactive,
		yes_to_all,
		FALSE				/* Move */
	));
}


/*
 *	Generates a new link name.
 *
 *	The path specifies the full path to the link or a directory
 *	where the link is to be created in.
 *
 *	The dest_name specifies the name of the target object (without
 *	a path).
 *
 *	Returns a dynamically allocated string describing the absolute
 *	path to a new link. If path refers to a non-existant object
 *	then a copy of path is returned, otherwise if path refers to
 *	an existing object then it must be a directory in which case
 *	a copy of path postfixed with dest_name and possibily a number
 *	will be returned.
 *
 *	If the path refers to a non-existant object then the returned
 *	string is just a copy of path.
 */
static gchar *edv_vfs_object_op_generate_new_link_path(
	const gchar *path,
	const gchar *dest_name
)
{
	gint counter = 0, counter_max = 10000;
	gchar	*full_path,
			*parent,
			*name,
			*new_path;
	EDVVFSObject *obj;

	if(STRISEMPTY(path) || STRISEMPTY(dest_name))
		return(NULL);

	/* If the path does not exist locally then just return a copy
	 * of it
	 */
	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
#ifdef ENOENT
		if(error_code == ENOENT)
#else
		if(FALSE)
#endif
			return(g_strdup(path));
		else
			return(NULL);
	}

	edv_vfs_object_delete(obj);

	/* Get the path's destination statistics */
	obj = edv_vfs_object_stat(path);
	if(obj == NULL)
	{
		/* Path exists locally but its destination does not exist */
		return(NULL);
	}

	/* Does the path lead to a directory? */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		/* The path is a directory */
		parent = g_strdup(path);
		name = g_strdup(dest_name);
		if((parent == NULL) || (name == NULL))
		{
			g_free(parent);
			g_free(name);
			edv_vfs_object_delete(obj);
			return(NULL);
		}
	}
	else
	{
		/* The path does not lead to a directory */
		edv_vfs_object_delete(obj);
		return(NULL);
	}

	edv_vfs_object_delete(obj);

	new_path = NULL;

	/* The copy of the target path is now a directory, now find
	 * an available non-existant link name in that directory
	 * derived from the copy of the source object's name
	 */
	for(counter = 1; counter < counter_max; counter++)
	{
		full_path = edv_paths_join(
			parent,
			name
		);
		if(full_path == NULL)
			continue;

		/* If this is not the first try then append a number after
		 * the link's name
		 */
		if(counter > 1)
		{
			gchar	*s,
					num_str[40];
			g_snprintf(
				num_str, sizeof(num_str),
				"%u",
				counter
			);
			s = g_strconcat(
				full_path,
				num_str,
				NULL
			);
			if(s != NULL)
			{
				g_free(full_path);
				full_path = s;
			}
		}

		/* Check if this potential new link name exists locally */
		obj = edv_vfs_object_lstat(full_path);
		if(obj != NULL)
		{
			/* This new new link refers to an existing object */
			edv_vfs_object_delete(obj);
			g_free(full_path);
			continue;
		}
		else
		{
			/* Unable to obtain the object's statistics */
			const gint error_code = (gint)errno;
#ifdef ENOENT
			if(error_code != ENOENT)
			{
				g_free(full_path);
				continue;
			}
#endif
		}

		/* All checks passed, transfer full_path to new_path */
		g_free(new_path);
		new_path = full_path;

		break;
	}

	/* Unable to find available link name? */
	if(counter >= counter_max)
	{
		g_free(new_path);
		new_path = NULL;
	}

	g_free(parent);
	g_free(name);

	return(new_path);
}

/*
 *	Creates a link.
 *
 *	The path specifies the full path to the link or a directory
 *	where the link is to be created in. If path refers to an
 *	existing object then it must be a directory and the new link
 *	will be created there, otherwise if path refers to a
 *	non-existing object then it will be treated as the full path
 *	to the link that is to be created.
 *
 *	The dest_path specifies the full path to the object that is
 *	to be the link's target.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * paths
 *	describing the objects that have been created will be returned.
 *	The calling function must delete the returned list and each
 *	string.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_link(
	EDVCore *core,
	const gchar *path,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gboolean archive = TRUE;
	gint status;
	gchar		*lpath,
					*ldest_path;
	GList *glist;
	EDVVFSObject	*dest_lobj = NULL,
					*dest_obj = NULL;

	if(new_paths_list_rtn != NULL)
		*new_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(path) ||
	   STRISEMPTY(dest_path) || (yes_to_all == NULL)
	)
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 edv_vfs_object_delete(dest_lobj);	\
 edv_vfs_object_delete(dest_obj);	\
 g_free(lpath);				\
 g_free(ldest_path);			\
									\
 return(_v_);				\
}

	/* Copy the path and the target path */
	lpath = g_strdup(path);
	ldest_path = g_strdup(dest_path);
	if((lpath == NULL) || (ldest_path == NULL))
	{
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}

	edv_path_simplify(lpath);
	edv_path_simplify(ldest_path);

	/* Get the statistics of the destination */
	dest_lobj = edv_vfs_object_lstat(ldest_path);
	dest_obj = edv_vfs_object_stat(ldest_path);
	if(archive ? (dest_lobj == NULL) : (dest_obj == NULL))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to link to:\n\
\n\
    %s\n\
\n\
%s.",
			ldest_path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		core->op_level--;
		CLEANUP_RETURN(-1);
	}

	status = 0;

	if(TRUE)
	{
		const gchar *dest_name = g_basename(ldest_path);
		gchar	*child_path,
					*dest_value;

		/* Generate the full path for a new link */
		child_path = edv_vfs_object_op_generate_new_link_path(
			lpath,				/* Path */
			dest_name			/* Destination name */
		);
		if(child_path == NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to generate a new link name for:\n\
\n\
    %s",
				lpath
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			status = -1;
			core->op_level--;
			CLEANUP_RETURN(status);
		}

		/* Generate the target value for this link */
		dest_value = edv_path_plot_relative(
			child_path,
			ldest_path
		);
		if(dest_value == NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to plot a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
				child_path,
				ldest_path
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			g_free(child_path);
			status = -1;
			core->op_level--;
			CLEANUP_RETURN(status);
		}

		/* Create the link */
		if(edv_link_create(
			child_path,			/* Path */
			dest_value			/* Target */
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to create the link:\n\
\n\
    %s\n\
\n\
%s.",
				child_path,
				g_strerror(error_code)
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
		}

		/* Set the full path of the new link return */
		if(new_paths_list_rtn != NULL)
			*new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn,
				g_strdup(child_path)
			);

		g_free(child_path);
		g_free(dest_value);
	}

	/* Record history */
	glist = (new_paths_list_rtn != NULL) ?
		g_list_last(*new_paths_list_rtn) : NULL;
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_LINK,
		time_start,
		edv_time(),
		status,
		(glist != NULL) ?			/* Source */
			(const gchar *)glist->data : ldest_path,
		ldest_path,				/* Target */
		core->last_error_ptr		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Set the link's value.
 *
 *	The path specifies the full path to the link who's value is
 *	to be set.
 *
 *	The new_dest_value specifies the new link value.
 *
 *	If modified_paths_list_rtn is not NULL then a dynamically
 *	allocated GList of gchar * paths describing the modified
 *	objects objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_relink(
	EDVCore *core,
	const gchar *path,
	const gchar *new_dest_value,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	gchar *lpath;
	EDVVFSObject *obj = NULL;

	if(modified_paths_list_rtn != NULL)
		*modified_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(path) ||
	   STRISEMPTY(new_dest_value) || (yes_to_all == NULL)
	)
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 edv_vfs_object_delete(obj);		\
 g_free(lpath);				\
									\
 return(_v_);				\
}

	/* Get copy of target object path */
	lpath = g_strdup(path);
	if(lpath == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}

	edv_path_simplify(lpath);

	/* Get the link's stats */
	obj = edv_vfs_object_lstat(lpath);
	if(obj != NULL)
	{
		/* Is it a link? */
		if(EDV_VFS_OBJECT_IS_LINK(obj))
		{
			/* Remove the existing link */
			if(edv_vfs_object_op_unlink(
				core,
				lpath
			))
			{
				core->op_level--;
				CLEANUP_RETURN(-1);
			}
		}
		else
		{
			core->last_error_ptr = "The object to relink is not a link.";
			core->op_level--;
			CLEANUP_RETURN(-2);
		}
	}

	status = 0;

	if(TRUE)
	{
		/* Create the link */
		if(edv_link_create(
			lpath,				/* Path */
			new_dest_value			/* Target */
		))
		{
			/* Error creating the link */
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to relink:\n\
\n\
    %s\n\
\n\
%s.",
				lpath,
				g_strerror(error_code)
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			status = -1;
		}
		else
		{
			/* Successfully created the new link */
			if(obj != NULL)
			{
				/* Restore the original ownership */
				(void)edv_lchown(
					path,
					obj->owner_id,
					obj->group_id
				);
			}

			/* Add this object to the list of modified paths */
			if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					g_strdup(lpath)
				);
		}
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_LINK,
		time_start,
		edv_time(),
		status,
		lpath,				/* Source */
		new_dest_value,			/* Target */
		core->last_error_ptr		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Rename.
 *
 *	The path specifies the full path to the object to rename.
 *
 *	The new_name specifies the new name of the object (without
 *	the path).
 *
 *	If modified_paths_list_rtn is not NULL then a dynamically
 *	allocated GList of gchar * paths describing the modified
 *	objects objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_rename(
	EDVCore *core,
	const gchar *path,
	const gchar *new_name,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	gchar		*parent_path,
					*lpath = NULL,
					*ltar_path = NULL;
	EDVVFSObject *tar_lobj = NULL;

	if(modified_paths_list_rtn != NULL)
		*modified_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(path) || STRISEMPTY(new_name) ||
 	   (yes_to_all == NULL)
	)
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 edv_vfs_object_delete(tar_lobj);	\
 g_free(lpath);				\
 g_free(ltar_path);			\
									\
 return(_v_);				\
}

	/* Check if the new name contains any invalid characters */
	if(!edv_is_object_name_valid(new_name))
	{
		gchar *msg = g_strdup_printf(
"Invalid name:\n\
\n\
    %s\n\
\n\
This name contains one or more invalid characters.",
			new_name
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		core->op_level--;
		CLEANUP_RETURN(-1);
	}

	/* Copy the object's path */
	lpath = g_strdup(path);
	if(lpath == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}
	edv_path_simplify(lpath);

	/* Get the object's parent */
	parent_path = g_dirname(lpath);
	if(parent_path == NULL)
	{
		core->last_error_ptr = "Unable to obtain the object's parent path.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}

	/* Format the full path to the new object */
	ltar_path = edv_paths_join(
		parent_path,
		new_name
	);

	g_free(parent_path);

	if(ltar_path == NULL)
	{
		core->last_error_ptr = "Unable to format the full path to the new object.";
		core->op_level--;
		CLEANUP_RETURN(-3);
	}

	edv_path_simplify(ltar_path);

	/* Are the names are the same? */
	if(!strcmp((const char *)lpath, (const char *)ltar_path))
	{
		/* If the names are the same then do not rename and
		 * return success
		 *
		 * Add this object to the modified paths list
		 */
		if(modified_paths_list_rtn != NULL)
			*modified_paths_list_rtn = g_list_append(
				*modified_paths_list_rtn, STRDUP(ltar_path)
			);

		core->op_level--;
		CLEANUP_RETURN(0);
	}


	/* Check if the new name already exists locally */
	tar_lobj = edv_vfs_object_lstat(ltar_path);
	if(tar_lobj != NULL)
	{
		core->last_error_ptr = "An object by that name already exists.";
		core->op_level--;
		CLEANUP_RETURN(-1);
	}

	/* Rename the object */
	status = 0;
	if(edv_rename(
		lpath,
		ltar_path
	))
	{
		/* Rename error */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to rename:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			g_basename(lpath),
			g_basename(ltar_path),
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		status = -1;
	}
	else
	{
		/* Renamed successfully
		 *
		 * Add this object to the modified paths list
		 */
		if(modified_paths_list_rtn != NULL)
			*modified_paths_list_rtn = g_list_append(
				*modified_paths_list_rtn,
				g_strdup(ltar_path)
			);
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_MOVE,
		time_start,
		edv_time(),
		status,
		lpath,				/* Source */
		ltar_path,				/* Target */
		core->last_error_ptr		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change permissions iteration.
 */
static gint edv_vfs_object_op_chmod_iterate(
	EDVCore *core,
	const gchar *path,
	const mode_t m,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
)
{
	gint status;
	EDVVFSObject *lobj;

	if(path == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar	*p1 = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),		*msg = g_strdup_printf(
"Changing the permissions of:\n\
\n\
    %s\n",
			p1
		);
		g_free(p1);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Permissions",
				msg,
				(const guint8 **)icon_chmod_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Set the permissions of this object
	 *
	 * Note that the archive value is ignored because there is
	 * no lchown since links themselves do not have permissions
	 */
	if(edv_chmod(
		path,
		m
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to change the permissions of:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Get this object's local statistics */
	lobj = edv_vfs_object_lstat(path);
	if(lobj != NULL)
	{
		/* Is this object a link? */
		if(EDV_VFS_OBJECT_IS_LINK(lobj))
		{
			/* Report the link's target as being modified */
			if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					edv_link_get_target_full(path)
				);
		}
		else
		{
			/* Add this object to the modified paths list */
	 	if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					g_strdup(path)
				);
		}
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(lobj);
			return(-4);
		}
	}

	status = 0;

	/* Recurse? */
	if(recursive)
	{
		/* Get the object's destination statistics */
		EDVVFSObject *obj = edv_vfs_object_stat(path);
	
		/* Is this object a directory? */
		if(archive ?
			EDV_VFS_OBJECT_IS_DIRECTORY(lobj) :
			EDV_VFS_OBJECT_IS_DIRECTORY(obj)
		)
		{
			/* Set the permissions of each object in this directory */
			EDVDirectory *dp = edv_directory_open(
				path,
				FALSE,			/* Unsorted */
				FALSE			/* Exclude notations */
			);
			if(dp != NULL)
			{
				const gchar *name;
				gchar *full_path;

				for(name = edv_directory_next(dp);
					name != NULL;
					name = edv_directory_next(dp)
				)
				{
					full_path = edv_paths_join(
						path,
						name
					);
					status = edv_vfs_object_op_chmod_iterate(
						core,
						full_path,
						m,
						modified_paths_list_rtn,
						toplevel,
						show_progress,
						interactive,
						yes_to_all,
						recursive,
						archive,
						nobjs_processed,
						nobjs
					);
					g_free(full_path);

					if(status != 0)
						break;
				}

				edv_directory_close(dp);
			}
		}

		edv_vfs_object_delete(obj);
	}

	edv_vfs_object_delete(lobj);

	return(status);
}

/*
 *	Change permissions.
 *
 *	The path specifies the full path to the object to change the
 *	permissions of.
 *
 *	The permissions specifies the new permissions to set, which
 *	can be any of EDV_PERMISSION_*.
 *
 *	If modified_paths_list_rtn is not NULL then a dynamically
 *	allocated GList of gchar * paths describing the modified
 *	objects objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	If recursive is TRUE then this operation will recurse to any
 *	subdirectories.
 *
 *	If archive is FALSE then any links that are encountered will
 *	be dereferenced.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_chmod(
	EDVCore *core,
	GList *paths_list,
	const EDVPermissionFlags permissions,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
)
{
	gulong time_start;
	gint status, nobjs_processed, nobjs;
	mode_t m = 0;
	gchar *permissions_s = NULL;
	const gchar *path;
	GList *glist;

	if(modified_paths_list_rtn != NULL)
		*modified_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || (paths_list == NULL) || (yes_to_all == NULL))
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* check if an operation is already in progress */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(permissions_s);		\
									\
 return(_v_);				\
}

	/* Count the number of objects to be processed */
	nobjs_processed = 0;
	nobjs = 0;
	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
		nobjs += edv_vfs_object_op_count_objects(
			(const gchar *)glist->data,
			recursive,
			archive
		);

	/* Convert the permissions mask to stat()'s mode_t format */
	m = edv_edv_permissions_to_stat_mode(permissions);

	/* Get the permissions string */
	permissions_s = edv_str_permissions(permissions);

	/* Iterate through the paths list and change the permissions
	 * of each object
	 */
	status = 0;
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		path = (const gchar *)glist->data;

		/* Change the permissions of this object */
		status = edv_vfs_object_op_chmod_iterate(
			core,
			path,
			m,
			modified_paths_list_rtn,
			toplevel,
			show_progress, interactive,
			yes_to_all,
			recursive, archive,
			&nobjs_processed, nobjs
		);

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_VFS_OBJECT_CHMOD,
			time_start,
			edv_time(),
			status,
			path,				/* Source */
			permissions_s,		/* Target */
			core->last_error_ptr		/* Comment */
		);

		if(status != 0)
			break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change ownership iteration.
 */
static gint edv_vfs_object_op_chown_iterate(
	EDVCore *core,
	const gchar *path,
	const gint owner_id, const gint group_id,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
)
{
	gint status;
	EDVVFSObject *lobj;

	if(path == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar	*p1 = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),		*msg = g_strdup_printf(
"Changing the ownership of:\n\
\n\
    %s\n",
			p1
		);
		g_free(p1);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Ownership",
				msg,
				(const guint8 **)icon_owned_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Set the ownership of this object */
	if(archive)
	{
		if(edv_lchown(
			path,
			owner_id,
			group_id
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the ownership of:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			return(-1);
		}
	}
	else
	{
		if(edv_chown(
			path,
			(gint)owner_id,
			(gint)group_id
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the ownership of:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_vfs_object_op_set_error(core, msg);
			g_free(msg);
			return(-1);
		}
	}

	/* Get this object's local statistics */
	lobj = edv_vfs_object_lstat(path);

	if(archive)
	{
		/* Add this object to the modified paths list */
		if(modified_paths_list_rtn != NULL)
			*modified_paths_list_rtn = g_list_append(
				*modified_paths_list_rtn,
				g_strdup(path)
			);
	}
	else
	{
		/* Is this object a link? */
		if(EDV_VFS_OBJECT_IS_LINK(lobj))
		{
			/* Report the link's target as being modified */
			if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					edv_link_get_target_full(path)
				);
		}
		else
		{
			/* Add this object to the modified paths list */
			if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					g_strdup(path)
				);
		}
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(lobj);
			return(-4);
		}
	}

	status = 0;

	/* Recurse? */
	if(recursive)
	{
		/* Get this object's destination statistics */
		EDVVFSObject *obj = edv_vfs_object_stat(path);

		/* Is this object a directory? */
		if(archive ?
			EDV_VFS_OBJECT_IS_DIRECTORY(lobj) :
			EDV_VFS_OBJECT_IS_DIRECTORY(obj)
		)
		{
			/* Set the permissions of each object in this directory */
			EDVDirectory *dp = edv_directory_open(
				path,
				FALSE,			/* Unsorted */
				FALSE			/* Exclude notations */
			);
			if(dp != NULL)
			{
				const gchar *name;
				gchar *full_path;

				for(name = edv_directory_next(dp);
					name != NULL;
					name = edv_directory_next(dp)
				)
				{
					full_path = edv_paths_join(
						path,
						name
					);
					status = edv_vfs_object_op_chown_iterate(
						core,
						full_path,
						owner_id, group_id,
						modified_paths_list_rtn,
						toplevel,
						show_progress,
						interactive,
						yes_to_all,
						recursive,
						archive,
						nobjs_processed,
						nobjs
					);
					g_free(full_path);

					if(status != 0)
						break;
				}

				edv_directory_close(dp);
			}
		}

		edv_vfs_object_delete(obj);
	}    

	edv_vfs_object_delete(lobj);

	return(status);
}

/*
 *	Change ownership.
 *
 *	The path specifies the full path to the object to change the
 *	ownership of.
 *
 *	The owner_id specifies the user id of the new owner.
 *
 *	The group_id specifies the group id of the new group.
 *
 *	If modified_paths_list_rtn is not NULL then a dynamically
 *	allocated GList of gchar * paths describing the modified
 *	objects objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	If recursive is TRUE then this operation will recurse to any
 *	subdirectories.
 *
 *	If archive is FALSE then any links that are encountered will
 *	be dereferenced.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_chown(
	EDVCore *core,
	GList *paths_list,
	const gint owner_id, const gint group_id,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
)
{
	gulong time_start;
	gint		status,
					nobjs_processed,
					nobjs;
	gchar		*owner_s,
					*group_s,
					*owner_group_s = NULL;
	const gchar *path;
	GList *glist;

	if(modified_paths_list_rtn != NULL)
		*modified_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || (paths_list == NULL) || (yes_to_all == NULL))
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Check if an operation is already in progress */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(owner_group_s);			\
									\
 return(_v_);				\
}

	/* Count the number of objects to be processed */
	nobjs_processed = 0;
	nobjs = 0;
	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
		nobjs += edv_vfs_object_op_count_objects(
			(const gchar *)glist->data,
			recursive,
			archive
		);

	/* Format the ownership string for logging */
	owner_s = edv_uid_uid_to_name(
		core->uids_list,
		owner_id, NULL
	);
	group_s = edv_gid_gid_to_name(
		core->gids_list,
		group_id, NULL
	);
	owner_group_s = g_strconcat(
		owner_s, ".", group_s, NULL
	);
	g_free(owner_s);
	g_free(group_s);

	/* Iterate through the paths list and change the ownership of
	 * each object
	 */
	status = 0;
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		path = (const gchar *)glist->data;

		/* Change the ownership of this object */
		status = edv_vfs_object_op_chown_iterate(
			core,
			path,
			owner_id, group_id,
			modified_paths_list_rtn,
			toplevel,
			show_progress, interactive,
			yes_to_all,
			recursive, archive,
			&nobjs_processed, nobjs
		);

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_VFS_OBJECT_CHOWN,
			time_start,
			edv_time(),
			status,
			path,				/* Source */
			owner_group_s,			/* Target */
			core->last_error_ptr		/* Comment */
		);

		if(status != 0)
			break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change time stamps iteration.
 */
static gint edv_vfs_object_op_chtime_iterate(
	EDVCore *core,
	const gchar *path,
	const gulong atime,
	const gulong mtime,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive,
	gint *nobjs_processed,
	const gint nobjs
)
{
	gint status;
	EDVVFSObject *lobj;

	if(path == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar	*p1 = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),		*msg = g_strdup_printf(
"Changing the time stamps of:\n\
\n\
    %s\n",
			p1
		);
		g_free(p1);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Time Stamps",
				msg,
				(const guint8 **)icon_time_stamp_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
			return(-4);
	}

	/* Change the time stamps of this object */
	if(edv_utime(
		path,
		atime,
		mtime
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to change the time stamps of:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_vfs_object_op_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Get this object's local statistics */
	lobj = edv_vfs_object_lstat(path);
	if(lobj != NULL)
	{
		/* Is this object a link? */
		if(EDV_VFS_OBJECT_IS_LINK(lobj))
		{
			/* Report the link's target as being modified */
			if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					edv_link_get_target_full(path)
				);
		}
		else
		{
			/* Add this object to the modified paths list */
	 	if(modified_paths_list_rtn != NULL)
				*modified_paths_list_rtn = g_list_append(
					*modified_paths_list_rtn,
					g_strdup(path)
				);
		}
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(lobj);
			return(-4);
		}
	}

	status = 0;

	/* Recurse? */
	if(recursive)
	{
		/* Get this object's destination statistics */
		EDVVFSObject *obj = edv_vfs_object_stat(path);

		/* Is this object a directory? */
		if(archive ?
			EDV_VFS_OBJECT_IS_DIRECTORY(lobj) :
			EDV_VFS_OBJECT_IS_DIRECTORY(obj)
		)
		{
			/* Set the permissions of each object in this directory */
			EDVDirectory *dp = edv_directory_open(
				path,
				FALSE,			/* Unsorted */
				FALSE			/* Exclude notations */
			);
			if(dp != NULL)
			{
				const gchar *name;
				gchar *full_path;

				for(name = edv_directory_next(dp);
					name != NULL;
					name = edv_directory_next(dp)
				)
				{
					full_path = edv_paths_join(
						path,
						name
					);
					status = edv_vfs_object_op_chtime_iterate(
						core,
						full_path,
						atime,
						mtime,
						modified_paths_list_rtn,
						toplevel,
						show_progress,
						interactive,
						yes_to_all,
						recursive,
						archive,
						nobjs_processed,
						nobjs
					);
					g_free(full_path);

					if(status != 0)
						break;
				}

				edv_directory_close(dp);
			}
		}

		edv_vfs_object_delete(obj);
	}    

	edv_vfs_object_delete(lobj);

	return(status);
}

/*
 *	Change time stamps.
 *
 *	The path specifies the full path to the object to change the
 *	time stamps of.
 *
 *	The atime specifies the access time stamp in seconds since
 *	EPOCH.
 *
 *	The mtime specifies the modify time stamp in seconds since
 *	EPOCH.
 *
 *	If modified_paths_list_rtn is not NULL then a dynamically
 *	allocated GList of gchar * paths describing the modified
 *	objects objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	If recursive is TRUE then this operation will recurse to any
 *	subdirectories.
 *
 *	If archive is FALSE then any links that are encountered will
 *	be dereferenced.
 *
 *      Returns 0 on success and non-zero on error.
 */
gint edv_vfs_object_op_chtime(
	EDVCore *core,       
	GList *paths_list,
	const gulong atime, const gulong mtime,
	GList **modified_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean archive
)
{
	gint		status,
			nobjs_processed,
			nobjs;
	gulong time_start;
	const gchar	*format,
			*path;
	gchar *time_str = NULL;
	GList *glist;
	CfgList *cfg_list;
	EDVDateRelativity relativity;

	if(modified_paths_list_rtn != NULL)
		*modified_paths_list_rtn = NULL;

	/* Clear the last error message */
	edv_vfs_object_op_set_error(core, NULL);

	if((core == NULL) || (paths_list == NULL) || (yes_to_all == NULL))
	{
		edv_vfs_object_op_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Check if an operation is already in progress */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;
	relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

#define CLEANUP_RETURN(_v_)	{	\
 g_free(time_str);			\
									\
 return(_v_);				\
}

	/* Count the number of objects to be processed */
	nobjs_processed = 0;
	nobjs = 0;
	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
		nobjs += edv_vfs_object_op_count_objects(
			(const gchar *)glist->data,
			recursive,
			archive
		);

	/* Format the time string */
	time_str = edv_date_string_format(
		mtime,
		format, relativity
	);

	/* Iterate through the paths list and change the time stamps
	 * of each object
	 */
	status = 0;
	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		path = (const gchar *)glist->data;

		/* Change the time stamps of this object */
		status = edv_vfs_object_op_chtime_iterate(
			core,
			path,
			atime,
			mtime,
			modified_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			archive,
			&nobjs_processed,
			nobjs
		);

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_VFS_OBJECT_CHTIME,
			time_start,
			edv_time(),
			status,
			path,				/* Source */
			time_str,			/* Target */
			core->last_error_ptr		/* Comment */
		);

		if(status != 0)
			break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
