#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>				/* mknod() */
#include <sys/types.h>				/* mknod() */
#include <sys/stat.h>				/* mknod() */
#include <gtk/gtk.h>
#include <unistd.h>				/* mknod() */

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_link.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_vfs_obj_create.h"
#include "edv_utils_gtk.h"
#include "endeavour2.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.                        
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.    
 *	-4	User responded with "Cancel".                     
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.    
 */


/* Error Message */  
const gchar *EDVVFSObjectCreateGetError(EDVCore *core);
static void EDVVFSObjectCreateCopyErrorMessage(
	EDVCore *core, const gchar *msg
);

/* New Path */
static gchar *EDVVFSObjectCreateGenerateNewPath(
	const gchar *path,
	const EDVObjectType type
);

/* Create New File Query Name */
gint EDVVFSObjectCreateFileQueryName(
	EDVCore *core,
	const gchar *parent_path,
	const gchar *suggested_name,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gboolean *yes_to_all
);

/* Create New Object Nexus */
static gint EDVVFSObjectCreateNexus(
	EDVCore *core,
	const gchar *path,
	const EDVObjectType type,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);

/* Create New File */
gint EDVVFSObjectCreateFile(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Directory */
gint EDVVFSObjectCreateDirectory(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Link */
gint EDVVFSObjectCreateLink(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New FIFO Pipe */
gint EDVVFSObjectCreateFifo(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Block Device */
gint EDVVFSObjectCreateDeviceBlock(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Character Device */
gint EDVVFSObjectCreateDeviceCharacter(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Socket */
gint EDVVFSObjectCreateSocket(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *EDVVFSObjectCreateGetError(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
static void EDVVFSObjectCreateCopyErrorMessage(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Generates a path to a new object.
 *
 *	The path specifies the full path to the directory at where
 *	the new object is to be created.
 *
 *	The type specifies the type of object to be created. The type
 *	will be used as part of the generated path's name.
 *
 *	Returns a dynamically allocated string describing the full
 *	path to the new object.
 */
static gchar *EDVVFSObjectCreateGenerateNewPath(
	const gchar *path,
	const EDVObjectType type
)
{
	guint16 i;
	gchar		*type_str,
			*new_path;

	if(STRISEMPTY(path))
		return(NULL);

	/* Get the type string from the type code */
	type_str = STRDUP(edv_object_type_to_object_name(type));
	if(type_str == NULL)
		return(NULL);

	g_strdown(type_str);

	/* Generate a new path */
	new_path = g_strconcat(
		path,
		G_DIR_SEPARATOR_S,
		"new_",
		type_str,
		NULL
	);
	if(new_path == NULL)
	{
		g_free(type_str);
		return(NULL);
	}

	/* Does the new path refer to a non-existant object? */
	if(!edv_path_lexists(new_path))
	{
		g_free(type_str);
		return(new_path);
	}

	/* An object already exists at the generated new path, so
	 * regenerate a new pth with a number appended to it and
	 * regenerate in the same way as needed until a path is
	 * generated that does not refer to an existing object
	 */
	for(i = 2; i < (guint16)-1; i++)
	{
		/* Delete the previously generated new path */
		g_free(new_path);

		/* Generate a new path with a number appended to it */
		new_path = g_strdup_printf(
			"%s%cnew_%s%u",
			path, G_DIR_SEPARATOR, type_str, i
		);
		if(new_path == NULL)
			break;

		/* Does the new path refer to a non-existant object? */
		if(!edv_path_lexists(new_path))
		{
			g_free(type_str);
			return(new_path);
		}
	}

	/* If this point is reached then we failed to generate a new
	 * path to a non-existent object
	 *
	 * Delete the last generated new path and return NULL
	 */
	g_free(new_path);
	g_free(type_str);

	return(NULL);
}


/*
 *	Prompts the user to create a new file.
 *
 *	The parent_path specifies the full path to the directory at
 *	which the new object is to be created at.
 *
 *	The suggested_name specifies the suggested name for the new
 *	file or NULL.
 *
 *	If new_path_rtn is not NULL then a dynamically allocated
 *	string describing the full path to the new object will be set
 *	to it.
 *
 *	If an error occures then the last error will be set to the
 *	error message that describes what error occured.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVVFSObjectCreateFileQueryName(
	EDVCore *core,
	const gchar *parent_path,
	const gchar *suggested_name,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gboolean *yes_to_all
)
{
	const gboolean interactive = TRUE;
	gint		status,
					strc;
	gulong time_start;
	gchar		*name,
					*path,
					**strv;

	if(new_path_rtn != NULL)
		*new_path_rtn = NULL;

	/* Clear the last error message */
	EDVVFSObjectCreateCopyErrorMessage(core, NULL);

	if((core == NULL) || STRISEMPTY(parent_path) ||
	   (yes_to_all == NULL)
	)
	{
		EDVVFSObjectCreateCopyErrorMessage(core, "Invalid value.");
		return(-2);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

	if(!g_path_is_absolute(parent_path))
	{
		core->last_error_ptr = "Invalid value.";
		core->op_level--;
		return(-2);
	}

	/* Prompt the user for a file name */
	PDialogDeleteAllPrompts();
	PDialogSetTransientFor(toplevel);
	PDialogAddPrompt(NULL, "New Name:", suggested_name);
	PDialogSetSize(320, -1);
	strv = PDialogGetResponse(
		"Create New File",
		NULL, NULL,
		PDIALOG_ICON_FILE,
		"Create", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	);
	PDialogSetTransientFor(NULL);
	if((strv == NULL) || (strc < 1))
	{
		PDialogDeleteAllPrompts();
		core->op_level--;
		return(-4);
	}

	/* Get a copy of the name */
	name = STRDUP(strv[0]);

	PDialogDeleteAllPrompts();

	/* Unable to get a copy of the name? */
	if(name == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		return(-3);
	}

	/* Generate the full path to the new object to be created */
	path = g_strconcat(
		parent_path,
		G_DIR_SEPARATOR_S,
		name,
		NULL
	);
	if(path == NULL)
	{
		g_free(name);
		core->last_error_ptr = "Memory allocation error.";
		core->op_level--;
		return(-3);
	}

	edv_path_simplify(path);

	/* Check if the object already exists */
	if(edv_path_lexists(path))
	{
		if(interactive)
		{
			/* Confirm overwrite */
			gint response;
			gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
				path
			);
			edv_play_sound_warning(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"   
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
				, msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_NO
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
			if((response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_YES_TO_ALL)
			)
			{
				g_free(name);
				g_free(path);
		        core->op_level--;
		        return(-5);
			}
			if(response == CDIALOG_RESPONSE_YES_TO_ALL)
				*yes_to_all = TRUE;
		}

		/* Remove the existing file */
		if(edv_unlink(path))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(EDV_OBJECT_TYPE_FILE),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			g_free(name);
			g_free(path);
		    core->op_level--;
		    return(-1);
		}
	}

	time_start = edv_time();
	status = 0;

	/* Create the new file */
	if(edv_touch(
		path,
		time_start,				/* Access time */
		TRUE				/* Create */
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
			"Unable to create a new ",
			edv_object_type_to_object_name_lower(EDV_OBJECT_TYPE_FILE),
			", ",
			g_strerror(error_code),
			".",
		    NULL
		);
		EDVVFSObjectCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	}

	/* Update new path return if creation of object was successful */
	if((status == 0) && (new_path_rtn != NULL))
	{
		g_free(*new_path_rtn);
		*new_path_rtn = g_strdup(path);
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_CREATE,
		time_start,
		edv_time(),
		status,
		parent_path,			/* Source */
		path,				/* Target */
		core->last_error_ptr		/* Comment */
	);

	g_free(name);
	g_free(path);

	core->op_level--;

	return(status);
}


/*
 *	Creates a new object of the specified type.
 *
 *	The path specifies the full path to either a non-existant
 *	object to be created or an existing directory at which to
 *	create the new object in.
 *
 *	If new_path_rtn is not NULL then a dynamically allocated
 *	string describing the full path to the new object will be set
 *	to it.
 *
 *	If an error occures then the last error will be set to the
 *	error message that describes what error occured.
 *
 *	Returns 0 on success and non-zero on error.
 */
static gint EDVVFSObjectCreateNexus(
	EDVCore *core,
	const gchar *path,
	const EDVObjectType type,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	gchar		*lpath = NULL,
					*new_path = NULL;
	EDVVFSObject *obj;

	if(new_path_rtn != NULL)
		*new_path_rtn = NULL;

	/* Clear the last error message */
	EDVVFSObjectCreateCopyErrorMessage(core, NULL);

	if((core == NULL) || STRISEMPTY(path) ||
	   (yes_to_all == NULL)
	)
	{
		EDVVFSObjectCreateCopyErrorMessage(core, "Invalid value.");
		return(-2);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(lpath);					\
 g_free(new_path);				\
						\
 return(_v_);					\
}

	/* Copy the specified path */
	lpath = g_strdup(path);

	/* Simplify the path */
	edv_path_simplify(lpath);

	/* Get the statistics of the object at the specified path */
	obj = edv_vfs_object_stat(lpath);
	if(obj != NULL)
	{
		/* The specified path exists
		 *
		 * Check if its destination is not a directory, in which
		 * case no new object can be created
		 */
		if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
		{
			edv_vfs_object_delete(obj);
			core->last_error_ptr =
"The location to create the new object at is not a directory.";
			core->op_level--;
			CLEANUP_RETURN(-1);
		}

		/* Generate a new name based on the specified path */
		new_path = EDVVFSObjectCreateGenerateNewPath(
			lpath,
			type
		);

		edv_vfs_object_delete(obj);
	}
	else
	{
		const gint error_code = (gint)errno;
#ifdef ENOENT
		if(error_code == ENOENT)
#else
		if(FALSE)
#endif
		{
			/* No such object exists at the specified path's
			 * location so use it as the path to the new object
			 */
			new_path = g_strdup(lpath);
		}
		else
		{
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			core->op_level--;
			CLEANUP_RETURN(-1);
		}
	}

	/* Unable to generate the full path to the new object? */
	if(new_path == NULL)
	{
		core->last_error_ptr = "Unable to generate the full path for the new object.";
		core->op_level--;
		CLEANUP_RETURN(-1);
	}

	/* At this point new_path specifies the full path to the new
	 * object that is to be created
	 *
	 * Begin creating by type
	 */
	status = 0;

	/* File? */
	if(type == EDV_OBJECT_TYPE_FILE)
	{
		if(edv_touch(
			new_path,
			time_start,			/* Access time */
			TRUE				/* Create */
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
	}
	/* Directory */
	else if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
		if(edv_directory_create(
			new_path,
			FALSE,				/* Do not create parents */
			NULL				/* No new paths list return */
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
	}
	/* Link */
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
		if(edv_link_create(
			new_path,			/* Path */
			"undefined"			/* Target */
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
	}
	/* FIFO Pipe */
	else if(type == EDV_OBJECT_TYPE_FIFO)
	{
#if defined(S_IFFIFO) || defined(S_IFIFO)
		const guint m = edv_get_umask();
		if(mknod(
			(const char *)new_path,
#if defined(S_IFFIFO)
			S_IFFIFO | ((mode_t)(~m) &
				(S_IRUSR | S_IWUSR |
				 S_IRGRP | S_IWGRP |
				 S_IROTH | S_IWOTH)
			),
#else
			S_IFIFO | ((mode_t)(~m) &
				(S_IRUSR | S_IWUSR |
				 S_IRGRP | S_IWGRP |
				 S_IROTH | S_IWOTH)
			),
#endif
			0
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
#else
		core->last_error_ptr = "Unsupported object type.";
		status = -2;
#endif
	}
	/* Block Device */
	else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
	{
#if defined(S_IFBLK)
		const guint m = edv_get_umask();
		const gint dev_num = edv_device_numbers_format(1, 1);
		if(mknod(
			(const char *)new_path,
			S_IFBLK | ((mode_t)(~m) &
				(S_IRUSR | S_IWUSR |
				 S_IRGRP | S_IWGRP |
				 S_IROTH | S_IWOTH)
			),
			(dev_t)dev_num
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
#else
		core->last_error_ptr = "Unsupported object type.";
		status = -2;
#endif
	}
	/* Character Device */
	else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	{
#if defined(S_IFCHR)
		const guint m = edv_get_umask();
		const gint dev_num = edv_device_numbers_format(1, 1);
		if(mknod(
			(const char *)new_path,
			S_IFCHR | ((mode_t)(~m) &
				(S_IRUSR | S_IWUSR |
				 S_IRGRP | S_IWGRP |
				 S_IROTH | S_IWOTH)
			),
			(dev_t)dev_num
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
#else
		core->last_error_ptr = "Unsupported object type.";
		status = -2;
#endif
	}
	/* Socket */
	else if(type == EDV_OBJECT_TYPE_SOCKET)
	{
#if defined(S_IFSOCK)
		const guint m = edv_get_umask();
		if(mknod(
			(const char *)new_path,
			S_IFSOCK | ((mode_t)(~m) &
				(S_IRUSR | S_IWUSR |
				 S_IRGRP | S_IWGRP |
				 S_IROTH | S_IWOTH)
			),
			0
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strconcat(
				"Unable to create a new ",
				edv_object_type_to_object_name_lower(type),
				", ",
				g_strerror(error_code),
				".",
				NULL
			);
			EDVVFSObjectCreateCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
		}
#else
		core->last_error_ptr = "Unsupported object type.";
		status = -2;
#endif
	}
	else
	{
		core->last_error_ptr = "Unsupported object type.";
		status = -2;
	}

	/* Update new path return if creation of object was successful */
	if((status == 0) && (new_path_rtn != NULL))
	{
		g_free(*new_path_rtn);
		*new_path_rtn = STRDUP(new_path);
	}


	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_CREATE,
		time_start,
		edv_time(),
		status,
		path,				/* Source */
		new_path,				/* Target */
		core->last_error_ptr		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Create a new file.
 */
gint EDVVFSObjectCreateFile(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_FILE,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new directory.
 */
gint EDVVFSObjectCreateDirectory(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_DIRECTORY, new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new link.
 */
gint EDVVFSObjectCreateLink(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_LINK,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new FIFO pipe.
 */
gint EDVVFSObjectCreateFifo(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_FIFO,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new block device.
 */
gint EDVVFSObjectCreateDeviceBlock(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core, path,
		EDV_OBJECT_TYPE_DEVICE_BLOCK,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new character device.
 */
gint EDVVFSObjectCreateDeviceCharacter(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_DEVICE_CHARACTER,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}

/*
 *	Create a new socket.
 */
gint EDVVFSObjectCreateSocket(
	EDVCore *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVVFSObjectCreateNexus(
		core,
		path, EDV_OBJECT_TYPE_SOCKET,
		new_path_rtn,
		toplevel, show_progress, interactive,
		yes_to_all
	));
}
