#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "edv_pixmap.h"
#include "edv_cursor.h"
#include "edv_history.h"
#include "edv_history_list.h"
#include "edv_utils_gtk.h"
#include "vfs_browser.h"
#include "image_browser.h"
#include "archiver.h"
#include "recycle_bin.h"
#include "edv_cb.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_wildcards_32x32.xpm"

#include "images/icon_endeavour_recycle_bin_empty_16x16.xpm"
#include "images/icon_endeavour_recycle_bin_16x16.xpm"
#include "images/icon_endeavour_recycle_bin_empty_20x20.xpm"
#include "images/icon_endeavour_recycle_bin_20x20.xpm"
#include "images/icon_endeavour_recycle_bin_empty_32x32.xpm"
#include "images/icon_endeavour_recycle_bin_32x32.xpm"
#include "images/icon_endeavour_recycle_bin_empty_48x48.xpm"
#include "images/icon_endeavour_recycle_bin_48x48.xpm"


typedef struct _EDVScrolledWindow	EDVScrolledWindow;
#define EDV_SCROLLED_WINDOW(p)		((EDVScrolledWindow *)(p))
#define EDV_SCROLLED_WINDOW_KEY		"EDV/ScrolledWindow"

#define EDV_MENU_BUTTON_MAP_KEY		"EDV/MenuButton/Map"


/* Accelerator Key Matching */
gint edv_match_accel_key_op_id(
	CfgList *cfg_list,
	const gchar *parm,
	const guint accel_key, const guint accel_mods
);
CfgAccelkey *edv_match_accel_key(
	CfgList *cfg_list,
	const gchar *parm,
	const gint op_id
);

/* Pixmaps */
EDVPixmap *edv_load_pixmap_from_data(
	EDVCore *core,
	edv_pixmap_data *data,
	const gchar *name
);
EDVPixmap *edv_open_pixmap_from_file(
	EDVCore *core,
	const gchar *path
);
EDVPixmap *edv_new_progress_pixmap(
	GdkWindow *window,
	GtkStyle *style,
	const gint width, const gint height,
	const gfloat coeff,
	const gboolean draw_value,
	const GtkOrientation orientation,
	const gboolean reverse
);
GtkWidget *edv_new_pixmap_widget(
	EDVCore *core,
	edv_pixmap_data *data,
	const gchar *name 
);
gchar *edv_get_recycle_bin_icon_path(
	EDVCore *core,
	const EDVIconSize size,
	const gint nrecycled_objects
);
edv_pixmap_data *edv_get_recycle_bin_icon_data(
	const EDVIconSize size,
	const gint nrecycled_objects,
	const gchar **c_declaration_id_rtn
);

/* Cursors */
GdkCursor *edv_get_cursor(
	EDVCore *core,
	const EDVCursorCode code
);

/* GtkCTree Nodes */
gint edv_node_get_level(GtkCTreeNode *node);
GtkCTreeNode *edv_node_get_parent(GtkCTreeNode *node);
GtkCTreeNode *edv_node_get_child(GtkCTreeNode *node);
GtkCTreeNode *edv_node_get_sibling(GtkCTreeNode *node);
GtkCTreeNode *edv_ctree_node_get_toplevel(GtkCTree *ctree);

/* GtkCList & GtkCTree Utilities */
gint edv_clist_get_selected_last(
	GtkCList *clist,
	GtkCListRow **row_ptr_rtn
);
GtkCTreeNode *edv_ctree_get_selected_last(
	GtkCTree *ctree,
	GtkCTreeRow **row_ptr_rtn
);
gboolean edv_ctree_node_get_index(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	gint *row_rtn, gint *column_rtn
);
GtkCTreeNode *edv_ctree_node_get_by_coordinates(
	GtkCTree *ctree,
	const gint x, const gint y
);
void edv_clist_scroll_to_position(
	GtkCList *clist,
	const gfloat hpos, const gfloat vpos
);

/* Endeavour Window Matching */
gboolean edv_match_window_from_toplevel(
	EDVCore *core,
	GtkWidget *toplevel,
	gint *browser_num,
	gint *imbr_num,
	gint *archiver_num,
	gint *recbin_num
);

/* Object Checking */
gboolean edv_is_object_name_valid(const gchar *name);
gboolean edv_is_object_hidden(EDVVFSObject *obj);

/* Permission Checking */
gboolean edv_is_object_accessable(
	EDVCore *core,
	EDVVFSObject *obj
);
gboolean edv_is_object_readable(
	EDVCore *core,
	EDVVFSObject *obj
);
gboolean edv_is_object_writable(
	EDVCore *core,
	EDVVFSObject *obj
);
gboolean edv_is_object_executable(
	EDVCore *core,
	EDVVFSObject *obj
);

/* Internal Supported Object Type Checking */
gboolean edv_check_archive_format_supported(
	EDVCore *core,
	const gchar *name
);

/* Message Output */
void edv_message(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
);
void edv_message_info(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
);
void edv_message_warning(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
);
void edv_message_error(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
);
void edv_message_object_op_error(
	const gchar *title, 
	const gchar *message, 
	const gchar *path,
	GtkWidget *toplevel
);
void edv_message_from_file(
	const gchar *path,
	const gchar *title,
	gint cdialog_icon,
	GtkWidget *toplevel
);

/* Play Sound */
void edv_beep(EDVCore *core);
void edv_play_sound_beep(EDVCore *core);
void edv_play_sound_info(EDVCore *core);
void edv_play_sound_question(EDVCore *core);
void edv_play_sound_warning(EDVCore *core);
void edv_play_sound_error(EDVCore *core);
void edv_play_sound_completed(EDVCore *core);

/* Menu & Button Interaction */
static void EDVMenuButtonMenuHideCB(GtkWidget *widget, gpointer data);
static void EDVMenuButtonMenuPositionCB(
	GtkMenu *menu,
	gint *x_rtn, gint *y_rtn,
	gpointer data
);
void edv_map_menu_to_button(
	GtkWidget *menu,
	GtkWidget *button
);

/* Window Centering */
void edv_center_window_to_window(
	GtkWidget *w1,
	GtkWidget *w2
);
void edv_center_window_to_window_offset(
	GtkWidget *w1, GtkWidget *w2, gint x_offset, gint y_offset
);

/* GtkEditable & GtkEntry Behavior Additions */
void edv_entry_set_dnd(EDVCore *core, GtkWidget *w);
void edv_entry_set_complete_path(EDVCore *core, GtkWidget *w);

/* Scrolled Window Setup Utilities */
GtkWidget *edv_new_scrolled_window(
	GtkPolicyType hscrollbar_policy,
	GtkPolicyType vscrollbar_policy,
	GtkWidget **event_box_rtn,
	GtkWidget **vbox_rtn
);
static gint EDVScrolledWindowEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVScrolledWindowDelete(gpointer data);

/* Copy Object Path Utilities */
void edv_copy_paths_to_clipboard(
	EDVCore *core,
	GList *objs_list,
	GtkWidget *owner
);
void edv_copy_urls_to_clipboard(
	EDVCore *core,
	GList *objs_list,
	GtkWidget *owner
);

/* Style Utilities */
GtkRcStyle *edv_new_rc_style_from_cfg(const CfgStyle *style);

/* History Utilities */
void edv_append_history(
	EDVCore *core,
	const EDVHistoryType type,
	const gulong start_time,
	const gulong end_time,
	const gint status,
	const gchar *source,
	const gchar *target,
	const gchar *comments
);

/* Write Protect Utilities */
gboolean edv_check_master_write_protect(
	EDVCore *core,
	const gboolean verbose,
	GtkWidget *toplevel
);

/* Query Name Filter */
gboolean edv_query_name_filter(
	EDVCore *core,
	gchar **filter_rtn,
	GtkWidget *toplevel
);


/*
 *	GtkScrolledWindow Data:
 */
struct _EDVScrolledWindow {
	gint		button,
			drag_last_x,
			drag_last_y;
	GdkCursor	*translate_cur;
};

/*
 *	GtkEditable Popup Menu Data:
 */
struct _EDVEditablePopupMenuData {

	GtkWidget	*menu,
			*editable,
			*cut_mi,
			*copy_mi,
			*paste_mi,
			*delete_mi,
			*select_all_mi,
			*unselect_all_mi;

	gint		freeze_count;

};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the Operation ID of the CfgAccelkey.
 *
 *	The cfg_list specifies the CfgList that contains the
 *	CfgAccelkeyList.
 *
 *	The parm specifies the CfgAccelkeyList's CfgItem parameter name.
 *
 *	The accel_key and accel_mods specifies the accelerator keys to
 *	match the CfgAccelkey in the CfgAccelkeyList.
 *
 *	Returns the matched CfgAccelkey's Operation ID or 0 if there
 *	was no match.
 */
gint edv_match_accel_key_op_id(
	CfgList *cfg_list,
	const gchar *parm,
	const guint accel_key, const guint accel_mods
)
{
	guint uc_accel_key, clean_accel_mods;
	CfgAccelkeyList *ak_list;

	if((cfg_list == NULL) || STRISEMPTY(parm))
	{
		errno = EINVAL;
		return(0);
	}

	/* Make sure that the specified accel key is upper case for
	 * easy case insensitive matching
	 */
	uc_accel_key = gdk_keyval_to_upper(accel_key);

	/* Make sure that the specified modifier keys only contain
	 * alt, ctrl, or shift
	 */
	clean_accel_mods = (accel_mods & (GDK_SHIFT_MASK |
		GDK_CONTROL_MASK |
		GDK_MOD1_MASK)
	);

	/* Get the CfgAccelkeyList specified the parameter */
	ak_list = CFGItemListGetValueAccelkeyList(
		cfg_list,
		parm
	);
	if(ak_list != NULL)
	{
		/* Found a matching CfgAccelkeyList, now check if the
		 * specified accelerator key matches any of the
		 * CfgAccelkeyList's CfgAccelkeys
		 */
		GList *glist;
		CfgAccelkey *ak;
		for(glist = ak_list->list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			ak = CFG_ACCELKEY(glist->data);
			if(ak == NULL)
				continue;

			if(ak->op_id <= 0)
				continue;

			if((gdk_keyval_to_upper(ak->key) == uc_accel_key) &&
			   (ak->modifiers == clean_accel_mods)
			)
				return(ak->op_id);
		}
	}

	errno = ENOENT;

	return(0);
}

/*
 *	Gets the CfgAccelkey by Operation ID.
 *
 *	The cfg_list specifies the CfgList that contains the
 *	CfgAccelkeyList.
 *
 *	The parm specifies the CfgAccelkeyList's CfgItem parameter name.
 *
 *	The opid specifies the Operation ID to match the CfgAccelkey
 *	in the CfgAccelkeyList with.
 *
 *	Returns the matched CfgAccelkey or NULL if there was no match.
 */
CfgAccelkey *edv_match_accel_key(
	CfgList *cfg_list,
	const gchar *parm,
	const gint op_id 
)
{
	CfgAccelkeyList *ak_list;

	if((cfg_list == NULL) || STRISEMPTY(parm) || (op_id <= 0))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Get the CfgAccelkeyList specified the parameter */
	ak_list = CFGItemListGetValueAccelkeyList(
		cfg_list,
		parm
	);
	if(ak_list != NULL)
	{
		/* Found a matching CfgAccelkeyList, now check if the
		 * specified operation ID matches any of the
		 * CfgAccelkeyList's CfgAccelkeys
		 */
		GList *glist;
		CfgAccelkey *ak;
		for(glist = ak_list->list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			ak = CFG_ACCELKEY(glist->data);
			if(ak == NULL)
				continue;

			if(ak->op_id == op_id)
				return(ak);
		}
	}

	errno = ENOENT;

	return(NULL);
}


/*
 *	Gets the EDVPixmap of source type EDV_PIXMAP_SOURCE_DATA from
 *	the core's list of EDVPixmaps.
 *
 *	If no existing EDVPixmap is found then a new EDVPixmap will be
 *	loaded and appended to the list.
 *
 *	The core specifies the list of EDVPixmaps.
 *
 *	The data specifies the XPM data.
 *
 *	The name specifies the XPM data's C declaration ID that will
 *	be used to find an existing EDVPixmap with. If a new EDVPixmap
 *	is created then the name specifies the new EDVPixmap's
 *	C declaration ID.
 *
 *	Returns the EDVPixmap with one reference count added or NULL
 *	on error.
 */
EDVPixmap *edv_load_pixmap_from_data(
	EDVCore *core,
	edv_pixmap_data *data,
	const gchar *name
)
{
	EDVPixmap *p;

	if((core == NULL) || (data == NULL) || (name == NULL))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Create a new EDVPixmap of source type EDV_PIXMAP_SOURCE_DATA
	 * with two reference counts (one for the list and one for the
	 * calling function) or get an existing matching EDVPixmap from
	 * the list with one reference count added
	 */
	core->pixmaps_list = edv_pixmaps_list_append_from_data(
		core->pixmaps_list,
		data,
		name,
		&p
	);
	if(p == NULL)
		return(NULL);

	return(p);
}

/*
 *	Gets the EDVPixmap of source type EDV_PIXMAP_SOURCE_FILE from
 *	the core's list of EDVPixmaps.
 *
 *	If no existing EDVPixmap is found then a new EDVPixmap will be
 *	opened and appended to the list.
 *
 *	The core specifies the list of EDVPixmaps.
 *
 *	The path specifies the path to the XPM file that will be used
 *	to find an existing EDVPixmap with. If no existing EDVPixmap is
 *	found and a new EDVPixmap is created then the path specifies the
 *	XPM file to open the new EDVPixmap with.
 *
 *	Returns the EDVPixmap with one reference count added or NULL
 *	on error.
 */
EDVPixmap *edv_open_pixmap_from_file(
	EDVCore *core,
	const gchar *path
)
{
	EDVPixmap *p;

	if((core == NULL) || (path == NULL))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Create a new EDVPixmap of source type EDV_PIXMAP_SOURCE_FILE
	 * with two reference counts (one for the list and one for the
	 * calling function) or get an existing matching EDVPixmap from
	 * the list with one reference count added
	 */
	core->pixmaps_list = edv_pixmaps_list_append_from_file(
		core->pixmaps_list,
		path,
		&p
	);
	if(p == NULL)
		return(NULL);

	return(p);
}

/*
 *	Creates a new progress pixmap.
 *
 *	The width and height specifies the size of the pixmap in pixels.
 *
 *	The coeff specifies the coefficient from 0.0 to 1.0. If the
 *	coeff is -1.0 then an unknown value will be drawn.
 *
 *	If draw_value is NULL then the value will be drawn.
 *
 *	The orientation specifies the orientation of the progress.
 *
 *	If reverse is TRUE then the progress will be drawn from right
 *	to left or bottom to top based on the orientation.
 *
 *	The style specifies the reference GtkStyle to use. If this is
 *	NULL then the default GtkStyle will be used.
 *
 *	The *mask_rtn will be set to the created mask.
 *
 *	Returns the progress pixmap or NULL on error.
 */
EDVPixmap *edv_new_progress_pixmap(
	GdkWindow *window,
	GtkStyle *style,
	const gint width, const gint height,
	const gfloat coeff,
	const gboolean draw_value,
	const GtkOrientation orientation,
	const gboolean reverse
)
{
	const gint	def_width = 80,
			def_height = 80,
			min_width = 16,
			min_height = 16;
	gint		_width = width,
			_height = height;
	GdkRectangle	trough_rect,
			bar_rect;
	GdkGC		*pixmap_gc,
			*mask_gc;
	const GtkStateType state = GTK_STATE_NORMAL;
	EDVPixmap *p;

	if(window == NULL)
		window = gdk_window_get_root();
	if(window == NULL)
		return(NULL);

	if(style == NULL)
		style = gtk_widget_get_default_style();
	if(style == NULL)
		return(NULL);

	/* Use the default width? */
	if(_width <= 0)
	{
		switch(orientation)
		{
		  case GTK_ORIENTATION_HORIZONTAL:
			_width = def_width;
			break;
		  case GTK_ORIENTATION_VERTICAL:
			_width = min_width;
			break;
		}
	}
	if(_width < min_width)
		_width = min_width;
	/* Use the default height? */
	if(_height <= 0)
	{
		switch(orientation)
		{
		  case GTK_ORIENTATION_HORIZONTAL:
			_height = min_height;
			break;
		  case GTK_ORIENTATION_VERTICAL:
			_height = def_height;
			break;
		}
	}
	if(_height < min_height)
		_height = min_height;

	/* Create a new EDVPixmap */
	p = edv_pixmap_new_ref();
	if(p == NULL)
		return(NULL);

	p->pixmap = gdk_pixmap_new(
		window,
		_width, _height,
		-1
	);
	if(p->pixmap == NULL)
	{
		(void)edv_pixmap_unref(p);
		return(NULL);
	}

	p->mask = gdk_pixmap_new(
		window,
		_width, _height,
		1
	);
	if(p->mask == NULL)
	{
		(void)edv_pixmap_unref(p);
		return(NULL);
	}

	pixmap_gc = gdk_gc_new((GdkWindow *)p->pixmap);
	if(pixmap_gc == NULL)
	{
		(void)edv_pixmap_unref(p);
		return(NULL);
	}
	mask_gc = gdk_gc_new((GdkWindow *)p->mask);
	if(mask_gc == NULL)
	{
		(void)GDK_GC_UNREF(pixmap_gc);
		(void)edv_pixmap_unref(p);
		return(NULL);
	}

	/* Draw the background */
	gdk_gc_set_function(mask_gc, GDK_CLEAR);
	gdk_draw_rectangle(
		(GdkDrawable *)p->mask,
		mask_gc,
		TRUE,				/* Fill */
		0, 0,
		_width, _height
	);
#if 0
	gdk_gc_set_foreground(pixmap_gc, &style->base[state]);
	gdk_draw_rectangle(
		(GdkDrawable *)p->pixmap,
		pixmap_gc,
		TRUE,				/* Fill */
		0, 0,
		_width, _height
	);
#endif

	/* Draw the outline */
	gdk_gc_set_function(mask_gc, GDK_SET);
	gdk_draw_rectangle(
		(GdkDrawable *)p->mask,
		mask_gc,
		FALSE,				/* Outline */
		0, 0,
		_width - 1, _height - 1
	);
	gdk_gc_set_foreground(pixmap_gc, &style->fg[state]);
	gdk_draw_rectangle(
		(GdkDrawable *)p->pixmap,
		pixmap_gc,
		FALSE,				/* Outline */
		0, 0,
		_width - 1, _height - 1
	);

	/* Calculate the geometry of the trough */
	trough_rect.x = 2;
	trough_rect.y = 2;
	trough_rect.width = _width - (2 * trough_rect.x);
	trough_rect.height = _height - (2 * trough_rect.y);
	if((trough_rect.width <= 0) || (trough_rect.height <= 0))
	{
		(void)GDK_GC_UNREF(mask_gc);
		(void)GDK_GC_UNREF(pixmap_gc);
		errno = EINVAL;
		return(p);
	}

	/* Calculate the geometry of the bar */
	switch(orientation)
	{
	  case GTK_ORIENTATION_HORIZONTAL:
		bar_rect.width = (coeff >= 0.0f) ?
			(gint)(trough_rect.width * MIN(coeff, 1.0f)) :
			trough_rect.width;
		bar_rect.height = trough_rect.height;
		if(reverse)
		{
			bar_rect.x = _width - trough_rect.x - bar_rect.width;
			bar_rect.y = trough_rect.y;
		}
		else
		{
			bar_rect.x = trough_rect.x;
			bar_rect.y = trough_rect.y;
		}
		break;
	  case GTK_ORIENTATION_VERTICAL:
		bar_rect.width = trough_rect.width;
		bar_rect.height = (coeff >= 0.0f) ?
			(gint)(trough_rect.height * MIN(coeff, 1.0f)) :
			trough_rect.height;
		if(reverse)
		{
			bar_rect.x = trough_rect.x;
			bar_rect.y = _height - trough_rect.y - bar_rect.height;
 	    }
		else
		{
			bar_rect.x = trough_rect.x;
			bar_rect.y = trough_rect.y;
		}
		break;
	}

	/* Draw the trough */
	gdk_draw_rectangle(
		(GdkDrawable *)p->mask,
		mask_gc,
		TRUE,				/* Fill */
		trough_rect.x, trough_rect.y,
		trough_rect.width, trough_rect.height
	);
	gdk_gc_set_foreground(pixmap_gc, &style->base[state]);
	gdk_draw_rectangle(
		(GdkDrawable *)p->pixmap,
		pixmap_gc,
		TRUE,				/* Fill */
		trough_rect.x, trough_rect.y,
		trough_rect.width, trough_rect.height
	);

	/* Draw the bar */
	if((bar_rect.width > 0) && (bar_rect.height > 0))
	{
		gdk_draw_rectangle(
			(GdkDrawable *)p->mask,
			mask_gc,
			TRUE,				/* Fill */
			bar_rect.x, bar_rect.y,
			bar_rect.width, bar_rect.height
		);
		gdk_gc_set_foreground(pixmap_gc, &style->bg[GTK_STATE_SELECTED]);
		gdk_draw_rectangle(
			(GdkDrawable *)p->pixmap,
			pixmap_gc,
			TRUE,				/* Fill */
			bar_rect.x, bar_rect.y,
			bar_rect.width, bar_rect.height
		);
	}

	/* Draw the value? */
	if(draw_value && (style->font != NULL))
	{
		gint x, y;
		gchar *s;
		GdkFont *font = style->font;
		const gint font_height = font->ascent + font->descent;
		GdkTextBounds b;

		gdk_gc_set_font(mask_gc, font);
		gdk_gc_set_font(pixmap_gc, font);

		s = g_strdup_printf(
			"%.0f%%",
			coeff * 100.0f
		);
		gdk_string_bounds(font, s, &b);
		x = 0;
		y = 0;
		switch(orientation)
		{
		  case GTK_ORIENTATION_HORIZONTAL:
			x = trough_rect.x + ((trough_rect.width - b.width) / 2);
			y = trough_rect.y + ((trough_rect.height - font_height) / 2);
 		break;
		  case GTK_ORIENTATION_VERTICAL:
			x = trough_rect.x + ((trough_rect.width - b.width) / 2);
			y = trough_rect.y + ((trough_rect.height - font_height) / 2);
			break;
		}

		gdk_draw_string(
			(GdkDrawable *)p->mask,
			font,
			mask_gc,
			x - b.lbearing,
			y + font->ascent,
			s
		);
		gdk_gc_set_foreground(pixmap_gc, &style->text[state]);
		gdk_draw_string(
			(GdkDrawable *)p->pixmap,
			font,
			pixmap_gc,
			x - b.lbearing,
			y + font->ascent,
			s
		);

		gdk_gc_set_clip_rectangle(pixmap_gc, &bar_rect);
		gdk_gc_set_foreground(pixmap_gc, &style->text[GTK_STATE_SELECTED]);
		gdk_draw_string(
			(GdkDrawable *)p->pixmap,
			font,
			pixmap_gc,
			x - b.lbearing,
			y + font->ascent,
			s
		);
		gdk_gc_set_clip_rectangle(pixmap_gc, NULL);

		g_free(s);
	}

	(void)GDK_GC_UNREF(mask_gc);
	(void)GDK_GC_UNREF(pixmap_gc);

	return(p);
}

/*
 *	Creates a new GtkPixmap from the specified XPM data.
 *
 *	An EDVPixmap from the core's list of pixmaps will be used
 *	if one exists matching the specified name or a new EDVPixmap
 *	will be added to the list if one does not exist.
 */
GtkWidget *edv_new_pixmap_widget(
	EDVCore *core,
	edv_pixmap_data *data,
	const gchar *name 
)
{
	GtkWidget *w;
	EDVPixmap *p = edv_load_pixmap_from_data(
		core,
		data,
		name
	);
	if(p == NULL)
		return(NULL);

	if(p->pixmap == NULL)
	{
		(void)edv_pixmap_unref(p);
		errno = EINVAL;
		return(NULL);
	}

	/* Create the new GtkPixmap */
	w = gtk_pixmap_new(p->pixmap, p->mask);

	/* Unref the EDVPixmap since it is no longer needed */
	(void)edv_pixmap_unref(p);

	return(w);
}

/*
 *	Gets the path to the Recycle Bin icon file.
 *
 *	Returns a dynamically allocated string describing the path
 *	to the Recycle Bin icon XPM file.
 */
gchar *edv_get_recycle_bin_icon_path(
	EDVCore *core,
	const EDVIconSize size,
	const gint nrecycled_objects
)
{
	gchar		*icons_path,
					*icon_path;
	if(core != NULL)
	{
		CfgList *cfg_list = core->cfg_list;
		icons_path = g_strconcat(
			EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
			G_DIR_SEPARATOR_S,
			EDV_NAME_ICONS_SUBDIR,
			NULL
		);
	}
	else
	{
		icons_path = g_strconcat(
			EDV_PATH_DEF_GLOBAL_DATA_DIR,
			G_DIR_SEPARATOR_S,
			EDV_NAME_ICONS_SUBDIR,
			NULL
		);
	}

	icon_path = NULL;
	switch(size)
	{
	  case EDV_ICON_SIZE_16:
		icon_path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			(nrecycled_objects > 0) ?
				"icon_endeavour_recycle_bin_16x16.xpm" :
				"icon_endeavour_recycle_bin_empty_16x16.xpm",
			NULL
		);
		break;
	  case EDV_ICON_SIZE_20:
		icon_path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			(nrecycled_objects > 0) ?
				"icon_endeavour_recycle_bin_20x20.xpm" :
				"icon_endeavour_recycle_bin_empty_20x20.xpm",
			NULL
		);
		break;
	  case EDV_ICON_SIZE_32:
		icon_path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			(nrecycled_objects > 0) ?
				"icon_endeavour_recycle_bin_32x32.xpm" :
				"icon_endeavour_recycle_bin_empty_32x32.xpm",
			NULL
		);
		break;
	  case EDV_ICON_SIZE_48:
		icon_path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			(nrecycled_objects > 0) ?
				"icon_endeavour_recycle_bin_48x48.xpm" :
				"icon_endeavour_recycle_bin_empty_48x48.xpm",
			NULL
		);
		break;
	}

	g_free(icons_path);

	return(icon_path);

}

/*
 *	Gets the XPM data for the Recycle Bin icon.
 *
 *	Returns the XPM data or NULL on error. The returned pointer is
 *	shared and should not be modified or deleted.
 */
edv_pixmap_data *edv_get_recycle_bin_icon_data(
	const EDVIconSize size,
	const gint nrecycled_objects,
	const gchar **c_declaration_id_rtn
)
{
	switch(size)
	{
	  case EDV_ICON_SIZE_16:
		if(nrecycled_objects > 0)
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_16x16_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_16x16_xpm);
		}
		else
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_empty_16x16_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_empty_16x16_xpm);
		}
		break;
	  case EDV_ICON_SIZE_20:
		if(nrecycled_objects > 0)
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_20x20_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_20x20_xpm);
		}
		else
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_empty_20x20_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_empty_20x20_xpm);
		}
		break;
	  case EDV_ICON_SIZE_32:
		if(nrecycled_objects > 0)
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_32x32_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_32x32_xpm);
		}
		else
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_empty_32x32_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_empty_32x32_xpm);
		}
		break;
	  case EDV_ICON_SIZE_48:
		if(nrecycled_objects > 0)
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_48x48_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_48x48_xpm);
		}
		else
		{
			if(c_declaration_id_rtn != NULL)
				*c_declaration_id_rtn = "icon_endeavour_recycle_bin_empty_48x48_xpm";
			return((edv_pixmap_data *)icon_endeavour_recycle_bin_empty_48x48_xpm);
		}
		break;
	}

	if(c_declaration_id_rtn != NULL)
		*c_declaration_id_rtn = NULL;

	return(NULL);
}


/*
 *	Gets the GdkCursor from the cursors by the EDVCursorCode.
 *
 *	The code specifies the EDVCursorCode, which can be one of
 *	EDV_CURSOR_CODE_*.
 *
 *	Returns the GdkCursor or NULL on error.
 */
GdkCursor *edv_get_cursor(
	EDVCore *core,
	const EDVCursorCode code
)
{
	GList *glist;
	EDVCursor *cursor;

	if(core == NULL)
		return(NULL);

	for(glist = core->cursors_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cursor = EDV_CURSOR(glist->data);
		if(cursor == NULL)
			continue;

		if(cursor->code == code)
			return(cursor->cursor);
	}

	return(NULL);
}

/*
 *	Returns the level of the given node or 0 if the given node
 *	is NULL.
 */
gint edv_node_get_level(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->level : 0);
}

/*
 *	Returns the parent of the given node or NULL if the given node
 *	does not have a parent.
 */
GtkCTreeNode *edv_node_get_parent(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->parent : NULL);
}

/*
 *	Returns the first child of the given node or NULL if the node
 *	does not have any children.
 */
GtkCTreeNode *edv_node_get_child(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->children : NULL);
}

/*
 *	Returns the next sibling of the given node or NULL if the node
 *	does not have a next sibling.
 */
GtkCTreeNode *edv_node_get_sibling(GtkCTreeNode *node)
{
	GtkCTreeRow *row_ptr = (node != NULL) ? GTK_CTREE_ROW(node) : NULL;
	return((row_ptr != NULL) ? row_ptr->sibling : NULL);
}

/*
 *	Returns the first toplevel node on the given GtkCTree widget
 *	or NULL if the GtkCTree does not have one.
 */
GtkCTreeNode *edv_ctree_node_get_toplevel(GtkCTree *ctree)
{
	/* Here we just return the first node, since it is always
	 * gauranteed to be the toplevel node.
	 */
	return((ctree != NULL) ? gtk_ctree_node_nth(ctree, 0) : NULL);
}


/*
 *	Returns the index of the last selected row on the given clist.
 *
 *	If row_ptr_rtn is not NULL and there exists a selected row then
 *	*row_ptr_rtn will be updated with the pointer to the row
 *	structure.
 *
 *	Returns -1 on error or no row selected, if a valid row index is
 *	returned it can be considered to be valid and allocated.
 */
gint edv_clist_get_selected_last(
	GtkCList *clist, GtkCListRow **row_ptr_rtn
)
{
	gint row;
	GList *glist;

	if(row_ptr_rtn != NULL)
		*row_ptr_rtn = NULL;

	if(clist == NULL)
		return(-1);

	/* Get last selected row index or -1 if there is none */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Check if we need to get a row pointer and that the selected
	 * row index is in bounds
	 */
	if((row >= 0) && (row < clist->rows) &&
	   (row_ptr_rtn != NULL) && (clist->row_list != NULL)
	)
	{
		glist = g_list_nth(clist->row_list, row);
		*row_ptr_rtn = (glist != NULL) ?
			(GtkCListRow *)glist->data : NULL;
		if(*row_ptr_rtn == NULL)
			row = -1;
	}

	return(row);
}

/*
 *	Returns the pointer of the last selected node on the given ctree.
 *
 *	If row_ptr_rtn is not NULL and there exists a selected node then
 *	*row_ptr_rtn will be updated with the pointer to the node's row
 *	structure.
 *
 *	Returns NULL on error or no node selected, if a valid node is
 *	returned it can be considered to be valid and allocated.
 */
GtkCTreeNode *edv_ctree_get_selected_last(
	GtkCTree *ctree,
	GtkCTreeRow **row_ptr_rtn
)
{
	GList *glist;
	GtkCList *clist;
	GtkCTreeNode *node;

	if(row_ptr_rtn != NULL)
		*row_ptr_rtn = NULL;

	if(ctree == NULL)
		return(NULL);

	clist = GTK_CLIST(ctree);
	if(clist == NULL)
		return(NULL);

	/* Get last selected node or -1 if there is none */
	glist = clist->selection_end;
	node = (glist != NULL) ? (GtkCTreeNode *)glist->data : NULL;

	/* Check if we need to get a row pointer and that there is a
	 * selected node
	 */
	if((node != NULL) && (row_ptr_rtn != NULL))
	{
		*row_ptr_rtn = GTK_CTREE_ROW(node);
		if(*row_ptr_rtn == NULL)
			node = NULL;
	}

	return(node);
}


/*
 *	Returns the column and row indexes of the given GtkCTreeNode node
 *	relative to the given GtkCTree widget.
 *
 *	Returns TRUE if the indexes are found or FALSE on failed match.
 */
gboolean edv_ctree_node_get_index(
	GtkCTree *ctree, GtkCTreeNode *node,
	gint *row_rtn, gint *column_rtn
)
{
	gint row;
	const GList *glist;
	GtkCList *clist;
	const GtkCTreeRow *row_ptr;


	/* Reset returns */
	if(row_rtn != NULL)
		*row_rtn = -1;
	if(column_rtn != NULL)
		*column_rtn = -1;

	if((ctree == NULL) || (node == NULL))
		return(FALSE);

	clist = GTK_CLIST(ctree);

	/* Get the row pointer the given node */
	row_ptr = GTK_CTREE_ROW(node);
	if(row_ptr == NULL)
		return(FALSE);

	/* Count rows until we encounter the given node's row */
	row = 0;
	glist = clist->row_list;
	while(glist != NULL)
	{
		if(row_ptr == (const GtkCTreeRow *)glist->data)
			break;

		row++;
		glist = g_list_next(glist);
	}
	/* Failed to find row? */
	if(glist == NULL)
		return(FALSE);

	/* Update returns */
	if(row_rtn != NULL)
		*row_rtn = row;
	/* Column is always the ctree column */
	if(column_rtn != NULL)
		*column_rtn = ctree->tree_column;

	return(TRUE);
}

/*
 *	Returns the tree node that matches the given coordinates on the
 *	given GtkCTree widget.
 */
GtkCTreeNode *edv_ctree_node_get_by_coordinates(
	GtkCTree *ctree, const gint x, const gint y
)
{
	gint row, column;
	GtkCList *clist;

	if(ctree == NULL)
		return(NULL);

	clist = GTK_CLIST(ctree);

	/* Find row and column based on given coordinates */
	if(!gtk_clist_get_selection_info(
		clist, x, y, &row, &column
	))
	{
		row = clist->rows;
		column = 0;
	}
	if(row < 0)
		row = 0;

	/* Get tree node matching the calculated row */
	return(gtk_ctree_node_nth(ctree, row));
}

/*
 *	Scroll the given clist to the given position.
 *
 *	A "value_changed" signal will be emitted to each of the clist's
 *	horizontal and vertical GtkAdjustments.
 */
void edv_clist_scroll_to_position(
	GtkCList *clist, const gfloat hpos, const gfloat vpos
)
{
	GtkAdjustment *adj;

	if(clist == NULL)
		return;

	adj = clist->hadjustment;
	if(adj != NULL)
	{
		gfloat clipped_hpos = hpos;
		if(clipped_hpos > (adj->upper - adj->page_size))
			clipped_hpos = adj->upper - adj->page_size;
		if(clipped_hpos < adj->lower)
			clipped_hpos = adj->lower;
		gtk_adjustment_set_value(adj, clipped_hpos);
	}

	adj = clist->vadjustment;
	if(adj != NULL)
	{
		gfloat clipped_vpos = vpos;
		if(clipped_vpos > (adj->upper - adj->page_size))
			clipped_vpos = adj->upper - adj->page_size;
		if(clipped_vpos < adj->lower)
			clipped_vpos = adj->lower;
		gtk_adjustment_set_value(adj, clipped_vpos);
	}
}


/*
 *	Checks which window the given toplevel GtkWindow belongs to.
 *
 *	Returns TRUE on match or FALSE on failed match.
 */
gboolean edv_match_window_from_toplevel(
	EDVCore *core,
	GtkWidget *toplevel,
	gint *browser_num,
	gint *imbr_num,
	gint *archiver_num,
	gint *recbin_num
)
{
	gint i;
	const EDVVFSBrowser *browser;
	const EDVImageBrowser *imbr;
	const EDVArchiver *archiver;
	const edv_recbin_struct *recbin;

	if(browser_num != NULL)
		*browser_num = -1;
	if(imbr_num != NULL)
		*imbr_num = -1;
	if(archiver_num != NULL)
		*archiver_num = -1;
	if(recbin_num != NULL)
		*recbin_num = -1;

	if((core == NULL) || (toplevel == NULL))
		return(FALSE);

	/* Begin checking which window's toplevel GtkWindow matches the
	 * specified toplevel GtkWindow
	 */

	/* VFS Browsers */
	for(i = 0; i < core->total_browsers; i++)
	{
		browser = core->browser[i];
		if(browser == NULL)
			continue;

		if(browser->toplevel == toplevel)
		{
			if(browser_num != NULL)
				*browser_num = i;
			return(TRUE);
		}
	}

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
	{
		imbr = core->imbr[i];
		if(imbr == NULL)
			continue;

		if(imbr->toplevel == toplevel)
		{
			if(imbr_num != NULL)
				*imbr_num = i;
			return(TRUE);
		}
	}

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
	{
		archiver = core->archiver[i];
		if(archiver == NULL)
			continue;

		if(archiver->toplevel == toplevel)
		{
			if(archiver_num != NULL)
				*archiver_num = i;
			return(TRUE);
		}
	}

	/* Recycle Bin */
	recbin = core->recbin;
	if(recbin != NULL)
	{
		if(recbin->toplevel == toplevel)
		{
			if(recbin_num != NULL)
				*recbin_num = i;
			return(TRUE);
		}
	}

	return(FALSE);
}


/*
 *	Returns TRUE if the given disk object name has a valid notation.
 */
gboolean edv_is_object_name_valid(const gchar *name)
{
	if(STRISEMPTY(name))
		return(FALSE);

	/* No special notations allowed */
	if(!strcmp(name, ".."))
		return(FALSE);
	if(!strcmp(name, "."))
		return(FALSE);
	if(!strcmp(name, "?"))
		return(FALSE);
	if(!strcmp(name, "*"))
		return(FALSE);

	/* No blanks allowed as first char */
	if(*name == ' ')
		return(FALSE);
	if(*name == '\t')
		return(FALSE);

	/* No deliminators allowed */
	if(strchr(name, G_DIR_SEPARATOR) != NULL)
		return(FALSE);

	return(TRUE);
}

/*
 *	Checks if the object is hidden (its name starting with a '.'
 *      character).
 */
gboolean edv_is_object_hidden(EDVVFSObject *obj)
{
	return((obj != NULL) ? edv_path_is_hidden(obj->name) : FALSE);
}


/*
 *	Checks if the user or group ID of the process can access the
 *	specified directory object.
 *
 *	Access is defined as the directory object being executable,
 *	since any directory set x means it is accessable.
 */
gboolean edv_is_object_accessable(
	EDVCore *core,
	EDVVFSObject *obj
)
{
	gint euid, eguid;
	EDVPermissionFlags p;

	if((core == NULL) || (obj == NULL))
		return(FALSE);

	euid = core->effective_user_id;
	eguid = core->effective_group_id;
	p = obj->permissions;

	/* Sticky and at least one catagory has execute permission? */
	if((p & EDV_PERMISSION_STICKY) &&
	   (p & (EDV_PERMISSION_UX | EDV_PERMISSION_GX |
			 EDV_PERMISSION_OX))
	)
		return(TRUE);	

	/* Anyone can execute? */
	if(p & EDV_PERMISSION_OX)
		return(TRUE);

	/* Group can execute and is member of group? */
	if((p & EDV_PERMISSION_GX) &&
	   (eguid == obj->group_id)
	)
		return(TRUE);

	/* Owner can execute, is root, or is the owner? */
	if((p & EDV_PERMISSION_UX) &&
	   ((euid == 0) || (euid == obj->owner_id))
	)
		return(TRUE);

	return(FALSE);
}

/*
 *	Checks if the user or group ID of the process can read the
 *	specified object.
 */
gboolean edv_is_object_readable(
	EDVCore *core,
	EDVVFSObject *obj
)
{
	gint euid, eguid;
	EDVPermissionFlags p;

	if((core == NULL) || (obj == NULL))
		return(FALSE);

	euid = core->effective_user_id;
	eguid = core->effective_group_id;
	p = obj->permissions;

	/* Sticky and at least one catagory has read permission? */
	if((p & EDV_PERMISSION_STICKY) &&
	   (p & (EDV_PERMISSION_UR | EDV_PERMISSION_GR |
			 EDV_PERMISSION_OR))
	)
		return(TRUE);

	/* Anyone can read? */
	if(p & EDV_PERMISSION_OR)
		return(TRUE);

	/* Group can read and is member of group? */
	if((p & EDV_PERMISSION_GR) &&
	   (eguid <= obj->group_id)
	)
		return(TRUE);

	/* Owner can read and is a owner? */
	if((p & EDV_PERMISSION_UR) &&
	   ((euid == 0) || (euid == obj->owner_id))
	)
		return(TRUE);

	return(FALSE);
}

/*
 *	Checks if the user or group ID of the process can write the
 *	specified object.
 */
gboolean edv_is_object_writable(
	EDVCore *core,
	EDVVFSObject *obj
)
{
	gint euid, eguid;
	EDVPermissionFlags p;

	if((core == NULL) || (obj == NULL))
		return(FALSE);

	euid = core->effective_user_id;
	eguid = core->effective_group_id;
	p = obj->permissions;

	/* Sticky and at least one catagory has write permission? */
	if((p & EDV_PERMISSION_STICKY) &&
	   (p & (EDV_PERMISSION_UW | EDV_PERMISSION_GW |
			 EDV_PERMISSION_OW))
	)
		return(TRUE);

	/* Anyone can write? */
	if(p & EDV_PERMISSION_OW)
		return(TRUE);

	/* Group can write and is member of group? */
	if((p & EDV_PERMISSION_GW) &&
	   (eguid <= obj->group_id)
	)
		return(TRUE);

	/* Owner can write and is a owner? */
	if((p & EDV_PERMISSION_UW) &&
	   ((euid == 0) || (euid == obj->owner_id))
	)
		return(TRUE);

	return(FALSE);
}

/*
 *	Checks if the user or group ID of the process can execute the
 *	specified object.
 */
gboolean edv_is_object_executable(
	EDVCore *core,
	EDVVFSObject *obj
)
{
	gint euid, eguid;
	EDVPermissionFlags p;

	if((core == NULL) || (obj == NULL))
		return(FALSE);

	euid = core->effective_user_id;
	eguid = core->effective_group_id;
	p = obj->permissions;

	/* Sticky and at least one catagory has execute permission? */
	if((p & EDV_PERMISSION_STICKY) &&
	   (p & (EDV_PERMISSION_UX | EDV_PERMISSION_GX |
			 EDV_PERMISSION_OX))
	)
		return(TRUE);

	/* Anyone can execute? */
	if(p & EDV_PERMISSION_OX)
		return(TRUE);

	/* Group can execute and is member of group? */
	if((p & EDV_PERMISSION_GX) &&
	   (eguid <= obj->group_id)
	)
		return(TRUE);

	/* Owner can execute and is a owner? */
	if((p & EDV_PERMISSION_UX) &&
	   ((euid == 0) || (euid == obj->owner_id))
	)
		return(TRUE);

	return(FALSE);
}


/*
 *	Returns TRUE if the object specified by path is an archive
 *	supported by Endeavour's archiver.
 */
gboolean edv_check_archive_format_supported(
	EDVCore *core,
	const gchar *name
)
{
	gint i;
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;

	if((core == NULL) || STRISEMPTY(name))
		return(FALSE);

	/* Check if this is an image file by checking the extension
	 * portion of the name
	 */
	for(i = 0; ext_list[i] != NULL; i += 2)
	{
		if(edv_name_has_extension(name, ext_list[i]))
			return(TRUE);
	}

	return(FALSE);
}


/*
 *	Displays the specified message on the CDialog.
 */
void edv_message(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
)
{
	edv_message_info(
		title,
		message,
		details,
		toplevel
	);
}
void edv_message_info(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel
)
{
	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
		title,
		message,
		details,
		CDIALOG_ICON_INFO,
		CDIALOG_BTNFLAG_OK |
		((details != NULL) ? CDIALOG_BTNFLAG_HELP : 0),
		CDIALOG_BTNFLAG_OK
	);
	CDialogSetTransientFor(NULL);
}
void edv_message_warning(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel 
)
{
	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
		title,
		message,
		details,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK |
		((details != NULL) ? CDIALOG_BTNFLAG_HELP : 0),
		CDIALOG_BTNFLAG_OK
	);
	CDialogSetTransientFor(NULL);
}
void edv_message_error(
	const gchar *title,
	const gchar *message,
	const gchar *details,
	GtkWidget *toplevel                                            
)
{
	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
		title,
		message,
		details,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK |
		((details != NULL) ? CDIALOG_BTNFLAG_HELP : 0),
		CDIALOG_BTNFLAG_OK  
	);
	CDialogSetTransientFor(NULL);
}
void edv_message_object_op_error(
	const gchar *title,
	const gchar *message, 
	const gchar *path,
	GtkWidget *toplevel
)
{
	gchar		*s,
			*error_msg = STRDUP(message);

	if(error_msg != NULL)
	{
		const gint len = (gint)strlen((const char *)error_msg);

		*error_msg = (gchar)toupper((char)*error_msg);

		if(len > 0)
		{
			if(error_msg[len - 1] == '.')
				error_msg[len - 1] = '\0';
		}
	}

	if(path != NULL)
		s = g_strdup_printf(
"%s:\n\
\n\
    %s",
			(error_msg != NULL) ? error_msg : "Operation error",
			path
		);
	else
		s = g_strdup_printf(
			"%s.",
			(error_msg != NULL) ? error_msg : "Operation error"
		);

	g_free(error_msg);

	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
		title,
		s,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK  
	);
	g_free(s);
	CDialogSetTransientFor(NULL);
}

/*
 *	Displays the contents of the specified file on the CDialog.
 *
 *	If the file does not exist or cannot be opened then nothing will
 *	be displayed.
 */
void edv_message_from_file(
	const gchar *path,
	const gchar *title,
	gint cdialog_icon,
	GtkWidget *toplevel
)
{
	size_t units_read;
	FILE *fp;
	struct stat stat_buf;
	const gint max_filesize = 80 * 100;
	gint buf_len;
	gchar *buf;

	if(STRISEMPTY(path))
		return;

	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return;

	if(fstat(fileno(fp), &stat_buf))
	{
		(void)fclose(fp);
		return;
	}

	/* Get size of file and limit it to max_filesize */
	buf_len = MIN(
		(gint)stat_buf.st_size,
		max_filesize
	);
	if(buf_len < 0)
		buf_len = 0;
	buf = (gchar *)g_malloc((buf_len + 1) * sizeof(gchar));
	if(buf == NULL)
	{
		(void)fclose(fp);
		return;
	}

	/* Read the file */
	units_read = fread(
		buf,
		sizeof(gchar),
		(size_t)buf_len,
		fp
	);

	/* Close the file */
	(void)fclose(fp);

	if(units_read <= 0l)
	{
		g_free(buf);
		return;
	}

	/* Null terminate */
	if(units_read <= buf_len)
		buf[units_read] = '\0';
	else
		buf[buf_len] = '\0';

	/* File contents empty? */
	if(*buf == '\0')
	{
		g_free(buf);
		return;
	}

	/* Display file contents on the CDialog */
	CDialogSetTransientFor(toplevel);
	CDialogGetResponse(
		title,
		buf,
		NULL,
		cdialog_icon,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	);
	g_free(buf);
	CDialogSetTransientFor(NULL);
}


/*
 *	Plays the "Beep" sound.
 */
void edv_beep(EDVCore *core)
{
	edv_play_sound_beep(core);
}
void edv_play_sound_beep(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
		gdk_beep();
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_BEEP);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}

/*
 *	Plays the "Info" sound.
 */
void edv_play_sound_info(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
/*	    gdk_beep(); */
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{   
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_INFO);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}

/*
 *	Plays the "Question" sound.
 */
void edv_play_sound_question(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
/*	    gdk_beep(); */
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{   
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_QUESTION);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}

/*
 *	Plays the "Warning" sound.
 */
void edv_play_sound_warning(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
/*	    gdk_beep(); */
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{   
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_WARNING);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}

/*
 *	Plays the "Error" sound.
 */
void edv_play_sound_error(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
/*	    gdk_beep(); */
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{   
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_ERROR);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}

/*
 *	Plays the "Completed" sound.
 */
void edv_play_sound_completed(EDVCore *core)
{
	CfgList *cfg_list;
	EDVSoundOutput sound_output;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	sound_output = (EDVSoundOutput)EDV_GET_I(EDV_CFG_PARM_SOUND_OUTPUT);

	if(sound_output == EDV_SOUND_OUTPUT_GDK)
	{
/*	    gdk_beep(); */
	}
	else if(sound_output == EDV_SOUND_OUTPUT_COMMAND)
	{   
		const gchar *cmd = EDV_GET_S(EDV_CFG_PARM_SOUND_PLAY_COMPLETED);
		if(!STRISEMPTY(cmd))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,			/* Parse escapes */
				TRUE			/* Parse quotes */
			);
		    (void)edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
		    g_free(shell_prog);
		}
	}
}


/*
 *	GtkMenu "hide" signal callback.
 *
 *	Used to release the GtkButton associated with the the GtkMenu
 *	and disconnect this signal.
 */
static void EDVMenuButtonMenuHideCB(GtkWidget *widget, gpointer data)
{ 
	guint s;
	GList *glist;
	GtkWidget *w;
	if(widget == NULL)
		return;

	/* Get the meuu and button map data from the GtkMenu */
	glist = (GList *)gtk_object_get_data(
		GTK_OBJECT(widget), EDV_MENU_BUTTON_MAP_KEY
	);
	w = (glist != NULL) ? (GtkWidget *)glist->data : NULL;
	glist = g_list_next(glist);
	s = (glist != NULL) ? (guint)glist->data : 0;

	/* Release the GtkButton */
	if((w != NULL) ? GTK_IS_BUTTON(w) : FALSE)
	{
		GtkButton *button = GTK_BUTTON(w);
		button->in_button = 0;
		gtk_signal_emit_by_name(GTK_OBJECT(w), "released");
	}

	/* Disconnect this signal */
	if(s > 0)
		gtk_signal_disconnect(GTK_OBJECT(widget), s);

	/* Remove the menu and button and map data */
	gtk_object_remove_data(
		GTK_OBJECT(widget), EDV_MENU_BUTTON_MAP_KEY
	);
}

/*
 *	GtkMenu position callback.
 *
 *	Moves the GtkMenu to the proper position based on the GtkButton
 *	specified by data.
 */
static void EDVMenuButtonMenuPositionCB(
	GtkMenu *menu,
	gint *x_rtn, gint *y_rtn,
	gpointer data
)
{
	gint width, height;
	GdkWindow *window;
	GtkWidget *w = (GtkWidget *)data;
	if((menu == NULL) || (x_rtn == NULL) || (y_rtn == NULL) || (w == NULL))
		return;

	window = w->window;
	if(window == NULL)
		return;

	gdk_window_get_root_position(window, x_rtn, y_rtn);
	gdk_window_get_size(window, &width, &height);
	*y_rtn = (*y_rtn) + height;
}

/*
 *	Maps a GtkMenu relative to a GtkButton.
 *
 *	This is used to map shared GtkMenus (GtkMenus used by multiple
 *	windows or in multiple places) relative to a GtkButton. This
 *	is often used to map a GtkMenu relative to a GtkButton on a
 *	ToolBar (TOOL_BAR_ITEM_BUTTON) instead of creating a
 *	TOOL_BAR_ITEM_MENU_BUTTON on the ToolBar since its GtkMenu is
 *	not shared.
 *
 *	The menu specifies the GtkMenu.
 *
 *	The button specifies the GtkButton.
 */
void edv_map_menu_to_button(
	GtkWidget *menu,
	GtkWidget *button
)
{
	if(menu == NULL)
		return;

	/* Ungrab the GtkButton */
	if(button != NULL)
	{
		if(GTK_WIDGET_HAS_GRAB(button))
			gtk_grab_remove(button);

/*
		gtk_events_process();
 */

		if(GTK_IS_BUTTON(button))
		{
			/* Send a "pressed" signal to the GtkButton to keep
			 * it pressed once the grab for the GtkButton has been
			 * removed and the GtkMenu has been mapped
			 */
			GTK_BUTTON(button)->in_button = 1;
			GTK_BUTTON(button)->button_down = 1;
			gtk_signal_emit_by_name(GTK_OBJECT(button), "pressed");
		}
	}

	/* Map the menu */
	if(GTK_IS_MENU(menu))
	{
		/* Connect the hide signal */
		guint s = gtk_signal_connect(
			GTK_OBJECT(menu), "hide",
			GTK_SIGNAL_FUNC(EDVMenuButtonMenuHideCB), button
		);
		GList *data_list = NULL;

		/* Set object data on the menu */
		data_list = g_list_append(data_list, button);
		data_list = g_list_append(data_list, (gpointer)s);
		gtk_object_set_data_full(
			GTK_OBJECT(menu), EDV_MENU_BUTTON_MAP_KEY,
			data_list, (GtkDestroyNotify)g_list_free
		);

		/* Map the menu */
		gtk_menu_popup(
			GTK_MENU(menu), NULL, NULL,
			EDVMenuButtonMenuPositionCB,	/* Position Callback */
			button,			/* Data */
			1,			/* Button Number */
			GDK_CURRENT_TIME	/* Time */
		);
	}
}


/*
 *	Moves the GtkWindow w2 to be centered to GtkWindow w1.
 */
void edv_center_window_to_window(
	GtkWidget *w1,
	GtkWidget *w2
)
{
	edv_center_window_to_window_offset(
		w1, w2,
		0, 0
	);
}

/*
 *	Moves the GtkWindow w2 to be centered to GtkWindow w1 plus
 *	the given offsets.
 */
void edv_center_window_to_window_offset(
	GtkWidget *w1,
	GtkWidget *w2,
	const gint x_offset, const gint y_offset
)
{
	gint		x, y,
			width1, height1,
			width2, height2;
	GdkWindow	*window1,
			*window2;

	if((w1 == NULL) || (w2 == NULL))
		return;

	if(!GTK_IS_WINDOW(w1) || !GTK_IS_WINDOW(w2))
		return;

	window1 = w1->window;
	window2 = w2->window;
	if((window1 == NULL) || (window2 == NULL))
		return;

	gdk_window_get_root_origin(
		window1,
		&x, &y
	);
	gdk_window_get_size(
		window1,
		&width1, &height1
	);
	gdk_window_get_size(
		window2,
		&width2, &height2
	);

	gtk_widget_set_uposition(
		w2,
		((width2 > 0) ?
			(x + (width1 / 2) - (width2 / 2)) : (x + 20)
		) + x_offset,
		((height2 > 0) ?
			(y + (height1 / 2) - (height2 / 2)) : (y + 20)
		) + y_offset
	);
}


/*
 *	Sets up the GtkEntry w to allow it to receive drag and drop.
 */
void edv_entry_set_dnd(EDVCore *core, GtkWidget *w)
{
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};

	if((core == NULL) || (w == NULL))
		return;

	if(!GTK_IS_ENTRY(w))
		return;

	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY,		/* Actions */
		GDK_ACTION_COPY,		/* Default action if same */
		GDK_ACTION_COPY,		/* Default action */
		edv_entry_drag_data_received_cb,
		core,
		TRUE				/* Highlight */
	);
}

/*
 *	Sets up the GtkEntry w to complete path when the TAB key is
 *	pressed.
 */
void edv_entry_set_complete_path(EDVCore *core, GtkWidget *w)
{
	if((core == NULL) || (w == NULL))
		return;

	if(!GTK_IS_ENTRY(w))
		return;

	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_entry_complete_path_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_entry_complete_path_cb), core
	);
}


/*
 *	Creates a new GtkScrolledWindow set up for improved scrolling.
 */
GtkWidget *edv_new_scrolled_window(
	GtkPolicyType hscrollbar_policy,
	GtkPolicyType vscrollbar_policy,
	GtkWidget **event_box_rtn,
	GtkWidget **vbox_rtn
)
{
	EDVScrolledWindow *d = EDV_SCROLLED_WINDOW(g_malloc0(
		sizeof(EDVScrolledWindow)
	));
	GtkWidget *parent, *w = gtk_scrolled_window_new(NULL, NULL);
	GtkScrolledWindow *sw = GTK_SCROLLED_WINDOW(w);

	d->button = 0;
	d->drag_last_x = 0;
	d->drag_last_y = 0;
	d->translate_cur = gdk_cursor_new(GDK_FLEUR);

	gtk_scrolled_window_set_policy(
		sw, hscrollbar_policy, vscrollbar_policy
	);
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_SCROLLED_WINDOW_KEY,
		d, EDVScrolledWindowDelete
	);
	parent = w;

	w = gtk_event_box_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_add_events(
		w,
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(EDVScrolledWindowEventCB), sw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(EDVScrolledWindowEventCB), sw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(EDVScrolledWindowEventCB), sw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(EDVScrolledWindowEventCB), sw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(EDVScrolledWindowEventCB), sw
	);
	gtk_scrolled_window_add_with_viewport(sw, w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, 0);
	if(vbox_rtn != NULL)
		*vbox_rtn = w;
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

	return(GTK_WIDGET(sw));
}

/*
 *	Scrolled window GtkEventBox event signal callback.
 */
static gint EDVScrolledWindowEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	EDVScrolledWindow *d;
	gboolean press;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GtkScrolledWindow *sw = (GtkScrolledWindow *)data;
	if((widget == NULL) || (event == NULL) || (sw == NULL))
		return(status);

#define ADJ_SET_EMIT(adj,v)	{		\
 gfloat v2 = (v);				\
 if(v2 > ((adj)->upper - (adj)->page_size))	\
  v2 = (adj)->upper - (adj)->page_size;		\
 if(v2 < (adj)->lower)				\
  v2 = (adj)->lower;				\
 gtk_adjustment_set_value((adj), v2);		\
}
#define ADJ_SET_DELTA_EMIT(adj,d)	{	\
 ADJ_SET_EMIT((adj), (adj)->value + (d));	\
}

	switch((gint)event->type)
	{
	  case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(focus->window == widget->window)
		{
			if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		    {
				GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
				gtk_widget_queue_draw(widget);
			}
			else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
			{
				GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
				gtk_widget_queue_draw(widget);
			}
			status = TRUE;
		}
		break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
		key = (GdkEventKey *)event;
		press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
#define SIGNAL_EMIT_STOP        {               \
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );						\
}
		switch(key->keyval)
		{
		  case GDK_Up:
		  case GDK_KP_Up:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_DELTA_EMIT(adj, -adj->step_increment);
			}
			SIGNAL_EMIT_STOP
			status = TRUE;
			break;
		  case GDK_Down:
		  case GDK_KP_Down:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_DELTA_EMIT(adj, adj->step_increment);
			}
			SIGNAL_EMIT_STOP
			status = TRUE;
			break;
		  case '-':
		  case 'b':
		  case 'p':
		  case GDK_Page_Up:
		  case GDK_KP_Page_Up:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_DELTA_EMIT(adj, -adj->page_increment);
			}
			status = TRUE;
			break;
		  case 'n':
		  case GDK_space:
		  case GDK_Page_Down:
		  case GDK_KP_Page_Down:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_DELTA_EMIT(adj, adj->page_increment);
			}
			status = TRUE;
			break;
		  case GDK_Home:
		  case GDK_KP_Home:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_EMIT(adj, adj->lower);
			}
			status = TRUE;
			break;
		  case GDK_End:
		  case GDK_KP_End:
			if(press)
			{
				GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(sw);
				ADJ_SET_EMIT(adj, adj->upper - adj->page_size);
			}
			status = TRUE;
			break;
		}
#undef SIGNAL_EMIT_STOP
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		if(button->window == widget->window)
		{
			if(!GTK_WIDGET_HAS_FOCUS(widget))
				gtk_widget_grab_focus(widget);
		}
		switch(button->button)
		{
		  case GDK_BUTTON1:
		  case GDK_BUTTON2:
			d = EDV_SCROLLED_WINDOW(
				GTK_OBJECT_GET_DATA(
					sw,
					EDV_SCROLLED_WINDOW_KEY
				)
			);
			if((button->window == widget->window) &&
			   (d != NULL)
			)
			{
				gdk_pointer_grab(
					button->window,
					FALSE,
					GDK_BUTTON_PRESS_MASK |
						GDK_BUTTON_RELEASE_MASK |
						GDK_POINTER_MOTION_MASK |
						GDK_POINTER_MOTION_HINT_MASK,
					NULL,
					d->translate_cur,
					button->time
				);
				d->button = button->button;
				d->drag_last_x = (gint)button->x;
				d->drag_last_y = (gint)button->y;
				status = TRUE;
			}
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON1:
		  case GDK_BUTTON2:
			d = EDV_SCROLLED_WINDOW(
				GTK_OBJECT_GET_DATA(
					sw,
					EDV_SCROLLED_WINDOW_KEY
				)
			);
			if(d != NULL)
			{
				if(d->button > 0)
				{
					d->button = 0;
					d->drag_last_x = 0;
					d->drag_last_y = 0;
					status = TRUE;
				}
			}
			if(gdk_pointer_is_grabbed())
				gdk_pointer_ungrab(button->time);
			break;
		}
		break;

	  case GDK_MOTION_NOTIFY:
		motion = (GdkEventMotion *)event;
		d = EDV_SCROLLED_WINDOW(
			GTK_OBJECT_GET_DATA(
				sw,
				EDV_SCROLLED_WINDOW_KEY
			)
		);
		if((d != NULL) ? ((d->button == GDK_BUTTON1) ||
						  (d->button == GDK_BUTTON2)) : FALSE
		)
		{
			gint dx, dy;
			GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment(sw),
						  *vadj = gtk_scrolled_window_get_vadjustment(sw);
			if(motion->is_hint)
			{
				gint x, y;
				GdkModifierType mask;
				gdk_window_get_pointer(
					motion->window, &x, &y, &mask
				);
				dx = x - d->drag_last_x;
				dy = y - d->drag_last_y;
			}
			else
			{
				dx = (gint)(motion->x - d->drag_last_x);
				dy = (gint)(motion->y - d->drag_last_y);
			}
#define DO_SCROLL(adj,d)	{			\
 if((d) != 0) {						\
  if(((adj)->upper - (adj)->page_size) > (adj)->lower) {\
   gtk_adjustment_set_value(				\
    adj,						\
    CLIP(						\
     (adj)->value - (d),				\
     (adj)->lower, (adj)->upper - (adj)->page_size	\
    )							\
   );							\
  }							\
 }							\
}
			DO_SCROLL(hadj, dx);
			DO_SCROLL(vadj, dy);

#undef DO_SCROLL
			status = TRUE;
		}
		break;
	}

#undef ADJ_SET_DELTA_EMIT
#undef ADJ_SET_EMIT

	return(status);
}

static void EDVScrolledWindowDelete(gpointer data)
{
	EDVScrolledWindow *d = EDV_SCROLLED_WINDOW(data);
	if(d == NULL)
		return;

	GDK_CURSOR_DESTROY(d->translate_cur);
	g_free(d);
}


/*
 *	Coppies the object paths to the DDE as a space-separated
 *	list of absolute paths.
 *
 *	The objs_list specifies the list of EDVVFSObject *
 *	objects.
 *
 *	The owner specifies the owner GtkWidget.
 */
void edv_copy_paths_to_clipboard(
	EDVCore *core,
	GList *objs_list,
	GtkWidget *owner
)
{
	gchar *s, *s2;
	GList *glist;
	EDVVFSObject *obj;

	if((core == NULL) || (objs_list == NULL))
		return;

	s = NULL;
	for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		if(s != NULL)
			s2 = g_strconcat(
				s,
				" ",
				obj->path,
				NULL
			);
		else
			s2 = g_strdup(obj->path);
		if(s2 != NULL)
		{
			g_free(s);
			s = s2;
		}
	}

	if(s != NULL)
		GUIDDESetDirect(
			owner,				/* Owner */
			GDK_SELECTION_PRIMARY,		/* Selection */
			GDK_CURRENT_TIME,		/* Time */
			gdk_atom_intern(GUI_TARGET_NAME_STRING, FALSE),
			s, STRLEN(s)			/* Data */
		);
}

/*
 *	Coppies the object paths to the DDE as a space-separated
 *	list of URLs (without the localhost address specified).
 *
 *	The objs_list specifies the list of EDVVFSObject *
 *	objects.
 *
 *	The owner specifies the owner GtkWidget.
 */
void edv_copy_urls_to_clipboard(
	EDVCore *core,
	GList *objs_list,
	GtkWidget *owner
)
{
	gchar *s, *s2;
	GList *glist;
	EDVVFSObject *obj;

	if((core == NULL) || (objs_list == NULL))
		return;

	s = NULL;
	for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		if(s != NULL)
			s2 = g_strconcat(
				s,
				" ",
				"file://",
				obj->path,
				NULL
			);
		else
			s2 = g_strconcat(
				"file://",
				obj->path,
				NULL
			);
		if(s2 != NULL)
		{
			g_free(s);
			s = s2;
		}
	}

	if(s != NULL)
		GUIDDESetDirect(
			owner,				/* Owner */
			GDK_SELECTION_PRIMARY,		/* Selection */
			GDK_CURRENT_TIME,		/* Time */
			gdk_atom_intern(GUI_TARGET_NAME_STRING, FALSE),
			s, STRLEN(s)			/* Data */
		);
}


/*
 *	Creates a new GtkRcStyle from the given CfgStyle.
 */
GtkRcStyle *edv_new_rc_style_from_cfg(const CfgStyle *style)
{
	gint i;
	guint src_color_flags;
	GdkColor *tar_c;
	const CfgColor *src_c;
	GtkRcStyle *tar_style;
	const CfgStyle *src_style = style;

#define STRDUP2(s)	((STRISEMPTY(s)) ? NULL : g_strdup(s))

	if(src_style == NULL)
		return(NULL);

	tar_style = gtk_rc_style_new();

	g_free(tar_style->font_name);
	tar_style->font_name = STRDUP2(src_style->font_name);

	for(i = 0; i < CFG_STYLE_STATES; i++)
	{
		src_color_flags = src_style->color_flags[i];
		if(src_color_flags & CFG_STYLE_FG)
		{
			tar_style->color_flags[i] |= GTK_RC_FG;
			src_c = &src_style->fg[i];
			tar_c = &tar_style->fg[i];
			GDK_COLOR_SET_COEFF(
				tar_c, src_c->r, src_c->g, src_c->b
			)
		}
		if(src_color_flags & CFG_STYLE_BG)
		{
			tar_style->color_flags[i] |= GTK_RC_BG;
			src_c = &src_style->bg[i];
			tar_c = &tar_style->bg[i];
			GDK_COLOR_SET_COEFF(
				tar_c, src_c->r, src_c->g, src_c->b
			) 
		}
		if(src_color_flags & CFG_STYLE_TEXT)
		{
			tar_style->color_flags[i] |= GTK_RC_TEXT;
			src_c = &src_style->text[i];
			tar_c = &tar_style->text[i];
			GDK_COLOR_SET_COEFF(
				tar_c, src_c->r, src_c->g, src_c->b
			) 
		}
		if(src_color_flags & CFG_STYLE_BASE)
		{
			tar_style->color_flags[i] |= GTK_RC_BASE;
			src_c = &src_style->base[i];
			tar_c = &tar_style->base[i];
			GDK_COLOR_SET_COEFF(
				tar_c, src_c->r, src_c->g, src_c->b
			) 
		}

		g_free(tar_style->bg_pixmap_name[i]);
		tar_style->bg_pixmap_name[i] = STRDUP2(
			src_style->bg_pixmap_name[i]
		);
	}

	return(tar_style);
#undef STRDUP2
}

/*
 *	Appends a history event to the history list.
 *
 *	The core specifies the core containing the history list to
 *	append to.
 *
 *	The type specifies the event type.
 *
 *	The start_time specifies the start time in seconds since EPOCH.
 *
 *	The end_time specifies the end time in seconds since EPOCH.
 *
 *	The status specifies the resulting status of the history event,
 *	valid values are:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User aborted.
 *	-5	User responded with "no" to a query.
 *	-6	An operation is already in progress.
 *	-7	Operation not supported.
 *
 *	The source specifies the source path, name, or value.
 *
 *	The target specifies the target path, name, or value.
 *
 *	The comments specifies any comments or error message pertaining
 *	to the history event.
 */
void edv_append_history(
	EDVCore *core,
	const EDVHistoryType type,
	const gulong start_time,
	const gulong end_time,
	const gint status,
	const gchar *source,
	const gchar *target,
	const gchar *comments
)
{
	CfgList *cfg_list;
	EDVHistory *h;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Create a new history event */
	h = edv_history_new();
	if(h == NULL)
		return;

	/* Set the values for the new history event */
	h->type = type;
	h->start_time = start_time;
	h->end_time = end_time;
	h->status = status;
	h->source = STRDUP(source);
	h->target = STRDUP(target);
	h->comments = STRDUP(comments);

	/* Append this history event to the list (which is on file) */
	edv_history_list_file_append(
		EDV_GET_S(EDV_CFG_PARM_FILE_HISTORY),
		h
	);

	/* Notify about this history event being appended */
	edv_emit_history_added(core, -1, h);

	/* Delete the history event */
	edv_history_delete(h);
}



/*
 *	Checks if the master write protect is on.
 *
 *	If verbose is set to TRUE and the master write protect is on
 *	then a message will be displayed to the user indicating that
 *	the operation was denied because the master write protect is
 *	on and how to turn if off.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	Returns TRUE if the master write protect is on or FALSE if the
 *	master write protect is off or an error occured.
 */
gboolean edv_check_master_write_protect(
	EDVCore *core,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	gboolean write_protect;

	if(core == NULL)
		return(FALSE);

	/* Get configuration */
	write_protect = CFGItemListGetValueI(
		core->cfg_list,
		EDV_CFG_PARM_WRITE_PROTECT
	);
	/* Is write protect enabled and verbose specified? */
	if(write_protect && verbose)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Escriba La Operacin Negada",
"La operacin especificada no se permite porque escribe\n\
protege est en.",
"Si usted desea realizar la operacin de thr entonces\n\
usted debe girar escribe lejos protege yendo a\n\
Escenarios->Escribe Protege.",
#elif defined(PROG_LANGUAGE_FRENCH)
"L'Opration D'criture Nie",
"L'opration spcifie n'est pas permise parce que protge\n\
en criture est sur.",
"Si vous souhaitez excuter l'opration de thr alors vous\n\
devez tourner de protge en criture en allant aux\n\
Montages->Protge En criture.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreiben Sie Betrieb",
"Verweigert wird, der. Der angegebene betrieb ist nicht\n\
zugelassen, weil schreibgeschtzt auf ist.",
"Wenn sie wnschen, thr betrieb durchzufhren, dann\n\
mssen sie schreibgeschtzt durch gehen zu\n\
Settings->Schreibgeschtzt Ausschalten.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere L'Operazione Negata",
"L'operazione specificata non  permessa perch scrive\n\
protegge  su.",
"Se lei desidera eseguire l'operazione di thr poi lei\n\
deve spegnere scrive protegge da andare ai\n\
Montaggi->Scrive Protegge.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf Werking Ontkende",
"De gespecificeerde werking is niet toegestaan omdat\n\
schrijft op beschermt, is.",
"Indien u wenst thr werking te verrichten dan beschermt\n\
u door gaan aan Settings->moet uitschakelen schrijft,\n\
beschermt, Schrijft.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva Operao Negado",
"A operao especificada no  permitida porque escreve\n\
protege est em.",
"Se deseja executar operao de thr ento voc deve\n\
desligar deve escrever protege por ir a\n\
Settings->Escreve Protege.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skriv Nektet Drift",
"Den spesifiserte driften tillater ikke fordi\n\
skrivebeskyttet er p.",
"Om de nsker gjennomfre thr drift skrur da de av\n\
skrivebeskyttet ved  dra til Innstillinger->Skrivebeskyttet.",
#else
"Write Operation Denied",
"The specified operation is not permitted because the\n\
Master Write Protect is turned on.",
"The Master Write Protect is a software level write\n\
protection feature that prevents any modifications\n\
to any of the disks.\n\
\n\
If you wish to perform this operation then you need\n\
to turn off the Master Write Protect by going to\n\
Settings->Master Write Protect.",
#endif
			toplevel
		);
	}

	return(write_protect);
}


/*
 *	Queries the user to set the name filter.
 *
 *	The filter_rtn specifies the current and return filter string
 *	value.
 *
 *	Returns TRUE if the user clicked on "Set" or FALSE if the
 *	user clicked on "Cancel" or an error occured.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
gboolean edv_query_name_filter(
	EDVCore *core,
	gchar **filter_rtn,
	GtkWidget *toplevel
)
{
	gboolean response;
	gint strc;
	gchar **strv;

	if((core == NULL) || PDialogIsQuery())
		return(FALSE);

	PDialogDeleteAllPrompts();
	PDialogSetTransientFor(toplevel);
	PDialogAddPrompt(
		NULL,
		"Name Filter:",
		(filter_rtn != NULL) ? *filter_rtn : NULL
	);
	PDialogSetPromptTip(
		-1,
		"Enter the name pattern for the objects that you want to\
 show in the list.\
 Valid wild card characters are * ? [ ]\
 Escape character \\ is allowed.\
 Path deliminators / are not allowed.\
 Case is sensitive.\
 For example, to match all objects whos names end with .txt use *.txt"
	);
	PDialogSetSize(320, -1);
	strv = PDialogGetResponseIconData(
		"Set Name Filter",
		NULL, NULL,
		(guint8 **)icon_wildcards_32x32_xpm,
		"Set", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	);
	PDialogSetTransientFor(NULL);
	response = FALSE;
	if((strv != NULL) && (strc > 0))
	{
		if((strc > 0) && (filter_rtn != NULL))
		{
			g_free(*filter_rtn);
			*filter_rtn = STRDUP(strv[0]);
		}

		response = TRUE;
	}

	PDialogDeleteAllPrompts();

	return(response);
}
