#include <gtk/gtk.h>

#include "guiutils.h"

#include "cfg.h"

#include "edv_types.h"
#include "edv_pixmap.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_write_protect_on_16x16.xpm"
#include "images/icon_write_protect_off_16x16.xpm"
#include "images/icon_purge_16x16.xpm"


typedef struct _EDVStatusBar		EDVStatusBar;
#define EDV_STATUS_BAR(p)		((EDVStatusBar *)(p))
#define EDV_STATUS_BAR_KEY		"EDV/StatusBar"


/*
 *	Flags:
 */
typedef enum {
	EDV_STATUS_BAR_MAPPED		= (1 << 0),
	EDV_STATUS_BAR_REALIZED		= (1 << 1)
} EDVStatusBarFlags;


/* Callbacks */
static void edv_status_bar_destroy_cb(gpointer data);
static void edv_status_bar_realize_cb(GtkWidget *widget, gpointer data);
static void edv_status_bar_show_cb(GtkWidget *widget, gpointer data);
static void edv_status_bar_hide_cb(GtkWidget *widget, gpointer data);
static gint edv_status_bar_master_write_protect_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint edv_status_bar_delete_method_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint edv_status_bar_master_write_protect_crossing_event_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint edv_status_bar_delete_method_crossing_event_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* Status Bar */
static EDVStatusBar *edv_status_bar_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *edv_status_bar_new(EDVCore *core);
void edv_status_bar_message(
	GtkWidget *w, 
	const gchar *msg,
	const gboolean allow_gtk_iteration
);
void edv_status_bar_progress(
	GtkWidget *w, 
	const gfloat v,
	const gboolean allow_gtk_iteration
);
void edv_status_bar_update_display(GtkWidget *w);


#define EDV_STATUS_BAR_HEIGHT		26
#define EDV_STATUS_BAR_PROGRESS_BAR_WIDTH	\
									100
#define EDV_STATUS_BAR_PROGRESS_BAR_HEIGHT	\
									(EDV_STATUS_BAR_HEIGHT - 6)


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Status Bar
 */
struct _EDVStatusBar {

	GtkWidget	*toplevel;		/* GtkFrame */
	gint		freeze_count;
	EDVCore	*core;
	EDVStatusBarFlags	flags;

	GtkWidget	*master_write_protect_eb,	/* GtkEventBox */
			*master_write_protect_pm,	/* GtkPixmap */
			*delete_method_eb,	/* GtkEventBox */
			*delete_method_pm,	/* GtkPixmap */
			*progress_bar,		/* GtkProgressBar */
			*label_frame,		/* GtkFrame */
			*label;			/* GtkLabel */

	EDVPixmap	*master_write_protect_on_icon,
			*master_write_protect_off_icon,
			*delete_method_recycle_empty_icon,
			*delete_method_recycle_icon,
			*delete_method_purge_icon;

	gchar		*master_write_protect_tooltip,
			*delete_method_tooltip;

	gboolean	last_master_write_protect_state;
	gint		last_nrecycle_bin_items;
	EDVDeleteMethod	last_delete_method;
	gfloat		last_progress_position;

};


/*
 *	Toplevel GtkObject data "destroy" signal callback.
 */
static void edv_status_bar_destroy_cb(gpointer data)
{
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if(bar == NULL)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	(void)edv_pixmap_unref(bar->master_write_protect_on_icon);
	(void)edv_pixmap_unref(bar->master_write_protect_off_icon);
	(void)edv_pixmap_unref(bar->delete_method_recycle_empty_icon);
	(void)edv_pixmap_unref(bar->delete_method_recycle_icon);
	(void)edv_pixmap_unref(bar->delete_method_purge_icon);

	g_free(bar->master_write_protect_tooltip);
	g_free(bar->delete_method_tooltip);

	bar->freeze_count--;

	g_free(bar);
}

/*
 *	Toplevel GtkWidget "realize" signal callback.
 */
static void edv_status_bar_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	if(bar->flags & EDV_STATUS_BAR_REALIZED)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	window = widget->window;
	if(window != NULL)
	{

	}

	/* Mark the EDVStatusBar as realized */
	bar->flags |= EDV_STATUS_BAR_REALIZED;

	bar->freeze_count--;
}

/*
 *	Toplevel GtkWidget "show" signal callback.
 */
static void edv_status_bar_show_cb(GtkWidget *widget, gpointer data)
{
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVStatusBar as mapped */
	bar->flags |= EDV_STATUS_BAR_MAPPED;
}

/*
 *	Toplevel GtkWidget "hide" signal callback.
 */
static void edv_status_bar_hide_cb(GtkWidget *widget, gpointer data)
{
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVStatusBar not mapped */
	bar->flags &= ~EDV_STATUS_BAR_MAPPED;
}

/*
 *	Master Write Protect GtkEventBox "button_press_event" or
 *	"button_release_event" signal callback.
 */
static gint edv_status_bar_master_write_protect_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	CfgList *cfg_list;
	EDVCore *core;
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (button == NULL) || (bar == NULL))
		return(FALSE);

	if(bar->freeze_count > 0)
		return(FALSE);

	bar->freeze_count++;

	core = bar->core;
	cfg_list = core->cfg_list;

	switch((gint)button->type)
	{
	  case GDK_BUTTON_PRESS:
		switch(button->button)
		{
		  case GDK_BUTTON1:
			if(bar->last_master_write_protect_state == TRUE)
			{
				EDV_SET_B(
					EDV_CFG_PARM_WRITE_PROTECT,
					FALSE
				);
				edv_queue_emit_master_write_protect_changed(core);
			}
			else
			{
				EDV_SET_B(
					EDV_CFG_PARM_WRITE_PROTECT,
					TRUE
				);
				edv_queue_emit_master_write_protect_changed(core);
			}
			break;
		}
		break;
	}

	bar->freeze_count--;

	return(TRUE);
}

/*
 *	Delete Method GtkEventBox "button_press_event" or
 *	"button_release_event" signal callback.
 */
static gint edv_status_bar_delete_method_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	CfgList *cfg_list;
	EDVCore *core;
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (button == NULL) || (bar == NULL))
		return(FALSE);

	if(bar->freeze_count > 0)
		return(FALSE);

	bar->freeze_count++;

	core = bar->core;
	cfg_list = core->cfg_list;

	switch((gint)button->type)
	{
	  case GDK_BUTTON_PRESS:
		switch(button->button)
		{
		  case GDK_BUTTON1:
			if(bar->last_delete_method == EDV_DELETE_METHOD_PURGE)
			{
				EDV_SET_I(
					EDV_CFG_PARM_DELETE_METHOD,
					EDV_DELETE_METHOD_RECYCLE
				);
				edv_queue_emit_delete_method_changed(core);
			}
			else
			{
				EDV_SET_I(
					EDV_CFG_PARM_DELETE_METHOD,
					EDV_DELETE_METHOD_PURGE
				);
				edv_queue_emit_delete_method_changed(core);
			}
			break;
		}
		break;
	}

	bar->freeze_count--;

	return(TRUE);
}

/*
 *	Master Write Protect GtkEventBox "enter_notify_event" or
 *	"leave_notify_event" signal callback.
 */
static gint edv_status_bar_master_write_protect_crossing_event_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (bar == NULL))
		return(FALSE);

	if(bar->freeze_count > 0)
		return(FALSE);

	bar->freeze_count++;

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
		edv_status_bar_message(
			bar->toplevel,
			bar->master_write_protect_tooltip,
			FALSE
		);
		break;

	  case GDK_LEAVE_NOTIFY:
		edv_status_bar_message(
			bar->toplevel,
			NULL,
			FALSE
		);
		break;
	}

	bar->freeze_count--;

	return(TRUE);
}

/*
 *	Delete Method GtkEventBox "enter_notify_event" or
 *	"leave_notify_event" signal callback.
 */
static gint edv_status_bar_delete_method_crossing_event_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVStatusBar *bar = EDV_STATUS_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (bar == NULL))
		return(FALSE);

	if(bar->freeze_count > 0)
		return(FALSE);

	bar->freeze_count++;

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
		edv_status_bar_message(
			bar->toplevel,
			bar->delete_method_tooltip,
			FALSE
		);
		break;

	  case GDK_LEAVE_NOTIFY:
		edv_status_bar_message(
			bar->toplevel,
			NULL,
			FALSE
		);
		break;
	}

	bar->freeze_count--;

	return(TRUE);
}


/*
 *	Gets the EDVStatusBar data from the GtkWidget.
 */
static EDVStatusBar *edv_status_bar_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = EDV_STATUS_BAR_KEY;
	EDVStatusBar *bar;

	if(w == NULL)
		return(NULL);

	bar = EDV_STATUS_BAR(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(bar == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(bar);
}

/*
 *	Creates a new Status Bar.
 */
GtkWidget *edv_status_bar_new(EDVCore *core)
{
	const gint border_minor = 2;
	gchar *path;
	GtkAdjustment *adj;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	GtkProgress *progress;
	GtkProgressBar *progress_bar;
	EDVPixmap *icon;
	EDVStatusBar *bar;

	if(core == NULL)
		return(NULL);

	/* Create a new Status Bar */
	bar = EDV_STATUS_BAR(g_malloc0(sizeof(EDVStatusBar)));
	if(bar == NULL)
		return(NULL);

	bar->toplevel = toplevel = gtk_frame_new(NULL);
/*	bar->freeze_count = 0; */
	bar->core = core;
	bar->master_write_protect_on_icon = edv_load_pixmap_from_data(
		core,
		(guint8 **)icon_write_protect_on_16x16_xpm,
		"icon_write_protect_on_16x16_xpm"
	);
	bar->master_write_protect_off_icon = edv_load_pixmap_from_data(
		core,
		(guint8 **)icon_write_protect_off_16x16_xpm,
		"icon_write_protect_off_16x16_xpm"
	);

	path = edv_get_recycle_bin_icon_path(
		core,
		EDV_ICON_SIZE_16,
		0				/* Empty */
	);
	bar->delete_method_recycle_empty_icon = edv_open_pixmap_from_file(
		core,
		path
	);
	g_free(path);

	path = edv_get_recycle_bin_icon_path(
		core,
		EDV_ICON_SIZE_16,
		1				/* Not empty */
	);
	bar->delete_method_recycle_icon = edv_open_pixmap_from_file(
		core,
		path
	);
	g_free(path);

	bar->delete_method_purge_icon = edv_load_pixmap_from_data(
		core,
		(guint8 **)icon_purge_16x16_xpm,
		"icon_purge_16x16_xpm"
	);
/*
	bar->master_write_protect_tooltip = NULL;
	bar->delete_method_tooltip = NULL;
 */
	bar->last_master_write_protect_state = -1;
	bar->last_nrecycle_bin_items = -1;
	bar->last_delete_method = -1;
	bar->last_progress_position = -1.0f;

	bar->freeze_count++;

	/* Toplevel GtkFrame */
	w = toplevel;
	gtk_widget_set_name(w, EDV_WIDGET_NAME_STATUS_BAR);
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_STATUS_BAR_KEY,
		bar, edv_status_bar_destroy_cb
	);
	gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_status_bar_realize_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "show",
		GTK_SIGNAL_FUNC(edv_status_bar_show_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "hide",
		GTK_SIGNAL_FUNC(edv_status_bar_hide_cb), bar
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	parent = w;

	/* Main GtkTable */
	w = gtk_table_new(1, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Master Write Protect GtkEventBox */
	bar->master_write_protect_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
		GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_status_bar_master_write_protect_button_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_status_bar_master_write_protect_crossing_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_status_bar_master_write_protect_crossing_event_cb), bar
	);
	gtk_table_attach(
		GTK_TABLE(parent),
		w,
		0, 1,
		0, 1,
		0,
		0,
		border_minor, 0
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent), 0, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura habilitada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est En-Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist ermglicht"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  permesso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf beschermt, is gegeven de gelegenheid"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  capacitado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet muliggjrer"
#else
"Master write protect is on"
#endif
	);
	gtk_widget_show(w);
	parent2 = w;

	/* Master Write Protect GtkPixmap */
	icon = edv_pixmap_ref(bar->master_write_protect_on_icon);
	if(edv_pixmap_is_loaded(icon))
	{
		bar->master_write_protect_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
/*			gtk_widget_set_usize(
				w,
				icon->width, icon->height
			);
 */
			gtk_container_add(GTK_CONTAINER(parent2), w);
			gtk_widget_show(w);
		}
	}
	(void)edv_pixmap_unref(icon);

	/* Delete Method GtkEventBox */
	bar->delete_method_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
		GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_status_bar_delete_method_button_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_status_bar_delete_method_crossing_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_status_bar_delete_method_crossing_event_cb), bar
	);
	gtk_table_attach(
		GTK_TABLE(parent),
		w,
		1, 2,
		0, 1,
		0,
		0,
		border_minor, 0
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent), 1, 1);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado seleccionado a reciclar"
#elif defined(PROG_LANGUAGE_FRENCH)
"La mthode pour Supprimer est positionne sur mettre  la Corbeille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen sie methode ist gesetzt wiederzuverwerten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per riciclare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te recyclen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto reciclar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter resirkulere"
#else
"Delete method is set to \"recycle\""
#endif
	);
	gtk_widget_show(w);
	parent2 = w;

	/* Delete Method GtkPixmap */
	icon = edv_pixmap_ref(bar->delete_method_recycle_empty_icon);
	if(edv_pixmap_is_loaded(icon))
	{
		bar->delete_method_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
/*			gtk_widget_set_usize(
				w,
				icon->width, icon->height
			);
 */
			gtk_container_add(GTK_CONTAINER(parent2), w);
			gtk_widget_show(w);
		}
	}
	(void)edv_pixmap_unref(icon);


	/* GtkProgressBar
	 *
	 * The GtkAdjustment range is from 1.0 to 100.0 (in units of
	 * percent), 0.0 is used as "no value" (so that no progress
	 * is shown)
	 */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 1.0f, 100.0f,
		0.0f, 0.0f, 0.0f
	);
	bar->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
	progress = GTK_PROGRESS(w);
	progress_bar = GTK_PROGRESS_BAR(w);
	gtk_widget_set_usize(
		w,
		EDV_STATUS_BAR_PROGRESS_BAR_WIDTH,
		EDV_STATUS_BAR_PROGRESS_BAR_HEIGHT
	);
	gtk_progress_bar_set_orientation(
		progress_bar,
		GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
		progress_bar,
		GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
		progress,
		FALSE
	);
	gtk_table_attach(
		GTK_TABLE(parent),
		w,
		2, 3,
		0, 1,
		0,
		0,
		0, 1
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent), 2, 0);
	gtk_widget_show(w);

	/* Label GtkFrame */
	bar->label_frame = w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_table_attach(
		GTK_TABLE(parent),
		w,
		3, 4,
		0, 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		1,				/* Use X padding as 1 so that
						 * both sides get padded since
						 * this is the last column and
						 * the right edge needs padding
						 * that can only be set this
						 * way */
		1
	);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkLabel */
	bar->label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_minor);
	gtk_widget_show(w);

	bar->freeze_count--;

	return(bar->toplevel);
}

/*
 *	Sets the Status Bar's message.
 */
void edv_status_bar_message(
	GtkWidget *w,
	const gchar *msg,
	const gboolean allow_gtk_iteration
)
{
	EDVStatusBar *bar = edv_status_bar_get_widget_data(
		w,
		"edv_status_bar_message"
	);
	if(bar == NULL)
		return;

	/* If the Status Bar is not mapped then do not update it
	 * or else undesired resizing queues will occure
	 */
	if(!(bar->flags & EDV_STATUS_BAR_MAPPED))
		return;

	gtk_label_set_text(
		GTK_LABEL(bar->label),
		(msg != NULL) ? msg : ""
	);

	/* Manage any GTK events so that the value changes to the
	 * GtkWidgets take affect immediately
	 */
	if(allow_gtk_iteration)
		gtk_events_process();
}

/*
 *	Sets the Status Bar's progress.
 *
 *	The v specifies the progress value from 0.0 to 1.0. If v is
 *	-1.0 then unknown progress is displayed.
 */
void edv_status_bar_progress(
	GtkWidget *w,
	const gfloat v,
	const gboolean allow_gtk_iteration
)
{
	gfloat value = v;
	GtkProgress *progress;
	GtkProgressBar *progress_bar;
	EDVStatusBar *bar = edv_status_bar_get_widget_data(
		w,
		"edv_status_bar_progress"
	);
	if(bar == NULL)
		return;

	/* If the Status Bar is not mapped then do not update it
	 * or else undesired resizing queues will occure
	 */
	if(!(bar->flags & EDV_STATUS_BAR_MAPPED))
		return;

	progress = GTK_PROGRESS(bar->progress_bar);
	progress_bar = GTK_PROGRESS_BAR(progress);

	/* Do activity? */
	if(value < 0.0f)
	{
		GtkAdjustment *adj = progress->adjustment;

		/* Get new value based on unknown position */
		value = gtk_progress_get_value(progress) + 1.0f;
		if(value > adj->upper)
			value = adj->lower;

		/* Update progress bar for `activity mode' (unknown limit)
		 * and set new value
		 */
		gtk_progress_set_activity_mode(
			progress,
			TRUE
		);
		gtk_progress_set_show_text(
			progress,
			FALSE
		);
		gtk_progress_set_value(
			progress,
			value
		);

		/* Reset the last progress position to -1.0, indicating
		 * that there is no known last position since we are
		 * doing activity mode
		 */
		bar->last_progress_position = -1.0f;
	}
	else
	{
		/* Clip the value to [0.0, 1.0] */
		if(value > 1.0f)
			value = 1.0f;

		/* Reset the last progress position if it is greater than
		 * the specified value, implying that the progress has
		 * warped back to the beginning
		 */
		if(bar->last_progress_position > value)
			bar->last_progress_position = -1.0f;

		/* Has the value not sufficiently changed? (less than
		 * 0.009 change in value)
		 */
		if((bar->last_progress_position > 0.0f) &&
		   ((value - bar->last_progress_position) < 0.009f)
		)
			return;

		/* Update progress */
		gtk_progress_set_activity_mode(
			progress,
			FALSE
		);
		gtk_progress_set_format_string(
			progress,
			"%p%%"
		);
		/* Display text only if value is positive */
		gtk_progress_set_show_text(
			progress,
			(value > 0.0f) ? TRUE : FALSE
		);
		gtk_progress_bar_update(
			progress_bar,
			value
		);

		/* Record the last progress value */
		bar->last_progress_position = value;
	}

	/* Manage any GTK events so that the value changes to the
	 * GtkWidgets take affect immediately
	 */
	if(allow_gtk_iteration)
		gtk_events_process();
}

/*
 *	Updates the Status Bar widgets to reflect the current values.
 */
void edv_status_bar_update_display(GtkWidget *w)
{
	gboolean master_write_protect_state;
	CfgList *cfg_list;
	EDVPixmap *recycle_bin_icon;
	EDVDeleteMethod delete_method;
	EDVCore *core;
	EDVStatusBar *bar = edv_status_bar_get_widget_data(
		w,
		"edv_status_bar_update_display"
	);
	if(bar == NULL)
		return;

	bar->freeze_count++;

	core = bar->core;
	cfg_list = core->cfg_list;

	/* Get the current configuration values */
	master_write_protect_state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	delete_method = (EDVDeleteMethod)EDV_GET_I(EDV_CFG_PARM_DELETE_METHOD);

	recycle_bin_icon = (core->last_nrecycle_bin_items > 0) ?
		bar->delete_method_recycle_icon : bar->delete_method_recycle_empty_icon;

	/* Master Write Protect state changed? */
	if(bar->last_master_write_protect_state != master_write_protect_state)
	{
		bar->last_master_write_protect_state = master_write_protect_state;
		if(master_write_protect_state)
		{
			edv_pixmap_set_gtk_pixmap(
				bar->master_write_protect_on_icon,
				bar->master_write_protect_pm
			);
			g_free(bar->master_write_protect_tooltip);
			bar->master_write_protect_tooltip = g_strdup(
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura habilitada, haga clic para anularla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture En Service, cliquer pour annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist, klicken ermglicht, unwirksam es zu machen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  permesso, lo scatto esserlo incapace"
#elif defined(PROG_LANGUAGE_DUTCH)
"De meester schrijf bescherm is op, klik om het uitdoen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  capacitado, estalido incapacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet muliggjrer, klikk udyktiggjre det"
#else
"Master write protect is on, click to turn it off"
#endif
			);
			GUISetWidgetTip(
				bar->master_write_protect_eb,
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura habilitada, haga clic para anularla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture En Service, cliquer pour annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist, klicken ermglicht, unwirksam es zu machen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  permesso, lo scatto esserlo incapace"
#elif defined(PROG_LANGUAGE_DUTCH)
"De meester schrijf bescherm is op, klik om het uitdoen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  capacitado, estalido incapacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet muliggjrer, klikk udyktiggjre det"
#else
"Master write protect is on"
#endif
			);
		}
		else
		{
			edv_pixmap_set_gtk_pixmap(
				bar->master_write_protect_off_icon,
				bar->master_write_protect_pm
			);
			g_free(bar->master_write_protect_tooltip);
			bar->master_write_protect_tooltip = g_strdup(
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura deshabilitada, haga clic para habilitarla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est Hors Service, cliquer pour la mettre En Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist unwirksam, klicken gemacht, es zu ermglichen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  stato incapace, lo scatto permetterlo"
#elif defined(PROG_LANGUAGE_DUTCH)
"De meester schrijf bescherm is van, klik om het aan te zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  incapacitado, estalido capacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet udyktiggjr, klikk muliggjre det"
#else
"Master write protect is off, click to turn it on"
#endif
			);
			GUISetWidgetTip(
				bar->master_write_protect_eb,
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura deshabilitada, haga clic para habilitarla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est Hors Service, cliquer pour la mettre En Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist unwirksam, klicken gemacht, es zu ermglichen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  stato incapace, lo scatto permetterlo"
#elif defined(PROG_LANGUAGE_DUTCH)
"De meester schrijf bescherm is van, klik om het aan te zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  incapacitado, estalido capacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet udyktiggjr, klikk muliggjre det"
#else
"Master write protect is off"
#endif
			);
		}
	}

	/* Recycled objects changed? */
	if((delete_method == EDV_DELETE_METHOD_RECYCLE) &&
	   (bar->last_nrecycle_bin_items != core->last_nrecycle_bin_items)
	)
	{
		edv_pixmap_set_gtk_pixmap(
			recycle_bin_icon,
			bar->delete_method_pm
		);
		bar->last_nrecycle_bin_items = core->last_nrecycle_bin_items;
	}
	
	/* Delete Method Changed? */
	if(bar->last_delete_method != delete_method)
	{
		bar->last_delete_method = delete_method;
		switch(delete_method)
		{
		  case EDV_DELETE_METHOD_RECYCLE:
			edv_pixmap_set_gtk_pixmap(
				recycle_bin_icon,
				bar->delete_method_pm
			);
			g_free(bar->delete_method_tooltip);
			bar->delete_method_tooltip = g_strdup(
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a reciclar, hacer clic para ponerlo a purgar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer la methode est reglee a \"recycle\", le declic pour le regler\
 \"purge\""
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt wiederzuverwerten, zu klicken, es\
 zu setzen, zu reinigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per riciclare, lo scatto regolarlo\
 per prosciogliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te recyclen, klik het te zetten te\
 reinigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto reciclar, clicar por purgar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter resirkulere, klikk sette det rense"
#else
"Delete method is set to \"recycle\", click to set it to \"purge\""
#endif
			);
			GUISetWidgetTip(
				bar->delete_method_eb,
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a reciclar, hacer clic para ponerlo a purgar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer la methode est reglee a \"recycle\", le declic pour le regler\
 \"purge\""
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt wiederzuverwerten, zu klicken, es\
 zu setzen, zu reinigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per riciclare, lo scatto regolarlo\
 per prosciogliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te recyclen, klik het te zetten te\
 reinigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto reciclar, clicar por purgar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter resirkulere, klikk sette det rense"
#else
"Delete method is set to \"recycle\""
#endif
			);
			break;

		  case EDV_DELETE_METHOD_PURGE:
			edv_pixmap_set_gtk_pixmap(
				bar->delete_method_purge_icon,
				bar->delete_method_pm
			);
			g_free(bar->delete_method_tooltip);
			bar->delete_method_tooltip = g_strdup(
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a purgar, hacer clic para ponerlo a reciclar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer la methode est reglee a \"purge\", le declic pour le regler a\
 \"recycle\""
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt zu reinigen, zu klicken, es zu\
 setzen, wiederzuverwerten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per prosciogliere, lo scatto\
 regolarlo per riciclare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te reinigen, klik het te zetten te recyclen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto purgar, clicar por reciclar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter rense, klikk sette det resirkulere"
#else
"Delete method is set to \"purge\", click to set it to \"recycle\""
#endif
			);
			GUISetWidgetTip(
				bar->delete_method_eb,
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a purgar, hacer clic para ponerlo a reciclar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer la methode est reglee a \"purge\", le declic pour le regler a\
 \"recycle\""
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt zu reinigen, zu klicken, es zu\
 setzen, wiederzuverwerten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per prosciogliere, lo scatto\
 regolarlo per riciclare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te reinigen, klik het te zetten te recyclen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto purgar, clicar por reciclar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter rense, klikk sette det resirkulere"
#else
"Delete method is set to \"purge\""
#endif
			);
			break;
		}
	}

	bar->freeze_count--;
}
