#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_utils_gtk.h"
#include "edv_recycled_obj_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_chmod_32x32.xpm"
#include "images/icon_owned_32x32.xpm"
#include "images/icon_time_stamp_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_recycled_object_op_get_error(EDVCore *core);
static void edv_recycled_object_op_copy_error_message(
	EDVCore *core,
	const gchar *msg
);

/* Lock */
static gint edv_recycled_object_op_lock(
	EDVCore *core,
	const gchar *index_path,
	const gchar *op_name,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Relink */
gint edv_recycled_object_op_relink(
	EDVCore *core,
	const gulong index,
	const gchar *new_target,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Rename */
gint edv_recycled_object_op_rename(
	EDVCore *core,
	const gulong index,
	const gchar *new_name,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Permissions */
static gint edv_recycled_object_op_chmod_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const EDVPermissionFlags permissions,
	const gchar *permissions_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint edv_recycled_object_op_chmod(
	EDVCore *core,
	GList *indices_list,
	const EDVPermissionFlags permissions,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Ownership */
static gint edv_recycled_object_op_chown_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const gint owner_id, const gint group_id,
	const gchar *owner_group_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint edv_recycled_object_op_chown(
	EDVCore *core,
	GList *indices_list,
	const gint owner_id, const gint group_id,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Timestamps */
static gint edv_recycled_object_op_chtime_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const gulong atime, const gulong mtime, const gulong dtime,
	const gchar *time_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint edv_recycled_object_op_chtime(
	EDVCore *core,
	GList *indices_list,
	const gulong atime, const gulong mtime, const gulong dtime,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_recycled_object_op_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message specified by msg to the last error
 *	message buffer and sets last_error to point to that buffer.
 */
static void edv_recycled_object_op_copy_error_message(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Locks the Recycle Bin.
 */
static gint edv_recycled_object_op_lock(
	EDVCore *core,
	const gchar *index_path,
	const gchar *op_name,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint status;

	/* Attempt to lock the recycle bin */
	while(TRUE)
	{
		status = edv_recycle_bin_index_lock(
			index_path,
			core->pid
		);
		if(status == 0)
			break;

		/* Already locked? */
		if(status == -6)
		{
			/* Query the user to retry or abort */
			if(interactive)
			{
				gint response;
				gchar *title = g_strconcat(
					op_name,
					" Failed",
					NULL
				);
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					title,
"The Recycle Bin is currently locked and in use.\n\
\n\
Try again?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				g_free(title);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Retry */
					continue;
				}
				else
				{
					/* Abort */
					errno = EINTR;
					return(-4);
				}
			}
			else
			{
				core->last_error_ptr =
"The Recycle Bin is currently locked.";
				errno = EBUSY;
				return(-6);
			}
		}

		/* Other error occured while attempting to lock */
		core->last_error_ptr = "Unable to lock the Recycle Bin.";
		return(status);
	}

	return(0);
}


/*
 *	Relink.
 *
 *	The index specifies the recycled object to relink. The recycled
 *	object must be of type EDV_OBJECT_TYPE_LINK.
 *
 *	The new_target specifies the new target value of the recycled
 *	object (without the path).
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of gulong indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_recycled_object_op_relink(
	EDVCore *core,
	const gulong index,
	const gchar *new_target,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	const gchar *index_path;
	CfgList *cfg_list;
	EDVRecycledObject *obj = NULL;

	if(modified_indicies_list != NULL)
		*modified_indicies_list = NULL;

	/* Clear the last error message */
	edv_recycled_object_op_copy_error_message(core, NULL);

	if((core == NULL) || (index == 0l) ||
	   STRISEMPTY(new_target) || (yes_to_all == NULL)
	)
	{
		edv_recycled_object_op_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Attempt to lock the Recycle Bin */
	status = edv_recycled_object_op_lock(
		core,
		index_path,
		"Relink",
		toplevel,
		show_progress,
		interactive,
		yes_to_all
	);
	if(status != 0)
		return(status);

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
 edv_recycled_object_delete(obj);		\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Get the current statistics for the recycled object */
	obj = edv_recycled_object_stat(index_path, index);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
			edv_recycle_bin_index_get_error(),
			index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Object must be of type link */
	if(obj->type != EDV_OBJECT_TYPE_LINK)
	{
		gchar *msg = g_strdup_printf(
"Not a link:\n\
\n\
    %s",
			obj->name
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	}

	/* Set the new target */
	g_free(obj->link_target);
	obj->link_target = g_strdup(new_target);

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = edv_recycle_bin_index_set(
		index_path,
		index,
		obj
	);
	if(status != 0)
	{
		/* Unable to commit the changes */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
			edv_recycle_bin_index_get_error(),
			index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Add this index to the modified indicies list */
		if(modified_indicies_list != NULL)
			*modified_indicies_list = g_list_append(
				*modified_indicies_list,
				(gpointer)index
			);
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_LINK,
		time_start,
		edv_time(),
		status,
		obj->name,			/* Source */
		new_target,			/* Target */
		core->last_error_ptr		/* Comment */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Rename.
 *
 *	The index specifies the recycled object to rename.
 *
 *	The new_name specifies the new name of the recycled object
 *	(without the path).
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of gulong indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_recycled_object_op_rename(
	EDVCore *core,
	const gulong index,
	const gchar *new_name,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	gchar *old_name = NULL;
	const gchar *index_path;
	CfgList *cfg_list;
	EDVRecycledObject *obj = NULL;

	if(modified_indicies_list != NULL)
		*modified_indicies_list = NULL;

	/* Clear the last error message */
	edv_recycled_object_op_copy_error_message(core, NULL);

	if((core == NULL) || (index == 0l) ||
	   STRISEMPTY(new_name) || (yes_to_all == NULL)
	)
	{
		edv_recycled_object_op_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Attempt to lock the Recycle Bin */
	status = edv_recycled_object_op_lock(
		core,
		index_path,
		"Rename",
		toplevel,
		show_progress,
		interactive,
		yes_to_all
	);
	if(status != 0)
		return(status);

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
 edv_recycled_object_delete(obj);		\
 g_free(old_name);				\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Check if the new name contains any invalid characters */
	if(!edv_is_object_name_valid(new_name))
	{
		gchar *msg = g_strdup_printf(
"Invalid name:\n\
\n\
    %s\n\
\n\
This name contains one or more invalid characters.",
			new_name
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	}

	/* Get the current statistics for the recycled object */
	obj = edv_recycled_object_stat(index_path, index);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
			edv_recycle_bin_index_get_error(),
			index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Record the old name */
	old_name = STRDUP(obj->name);

	/* Set the new name */
	g_free(obj->name);
	obj->name = g_strdup(new_name);

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = edv_recycle_bin_index_set(
		index_path,
		index,
		obj
	);
	if(status != 0)
	{
		/* Unable to commit the changes */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
			edv_recycle_bin_index_get_error(),
			index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Add this index to the modified indicies list */
		if(modified_indicies_list != NULL)
			*modified_indicies_list = g_list_append(
				*modified_indicies_list,
				(gpointer)index
			);
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_RENAME,
		time_start,
		edv_time(),
		status,
		old_name,			/* Source */
		new_name,			/* Target */
		core->last_error_ptr		/* Comment */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change permissions iteration.
 */
static gint edv_recycled_object_op_chmod_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const EDVPermissionFlags permissions,
	const gchar *permissions_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{		\
 return(_v_);					\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar       *name_shortened = edv_path_shorten(
			obj->name,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),          *msg = g_strdup_printf(
"Changing permissions of:\n\
\n\
    %s\n",
			name_shortened
		);
		g_free(name_shortened);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Permissions",
				msg,
				(const guint8 **)icon_chmod_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	/* Set the new permissions */
	obj->permissions = permissions;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = edv_recycle_bin_index_set(
		index_path,
		obj->index,
		obj
	);
	if(status != 0)
	{
		/* Unable to commit the changes */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %ld",
			edv_recycle_bin_index_get_error(),
			obj->index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Add this index to the modified indicies list */
		if(modified_indicies_list != NULL)
			*modified_indicies_list = g_list_append(
				*modified_indicies_list,
				(gpointer)obj->index
			);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change permissions.
 *
 *	The indicies_list specifies a GList of gulong indicies
 *	describing the recycled objects to change the permissions of.
 *
 *	The permissions specifies the new permissions which can be
 *	any of EDV_PERMISSION_*.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of gulong indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_recycled_object_op_chmod(
	EDVCore *core,
	GList *indices_list,
	const EDVPermissionFlags permissions,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint		status,
			nobjs_processed,
			nobjs;
	gulong index;
	gulong time_start;
	gchar		*permissions_s = NULL,
			*index_s;
	const gchar *index_path;
	GList *glist;
	CfgList *cfg_list;
	EDVRecycledObject *obj;

	if(modified_indicies_list != NULL)
		*modified_indicies_list = NULL;

	/* Clear the last error message */
	edv_recycled_object_op_copy_error_message(core, NULL);

	if((core == NULL) || (indices_list == NULL) ||
	   (yes_to_all == NULL)
	)
	{
		edv_recycled_object_op_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Attempt to lock the Recycle Bin */
	status = edv_recycled_object_op_lock(
		core,
		index_path,
		"Change Permissions",
		toplevel,
		show_progress,
		interactive,
		yes_to_all
	);
	if(status != 0)
		return(status);

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
 g_free(permissions_s);				\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Format the permissions string */
	permissions_s = edv_str_permissions(permissions);

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		index = (gulong)glist->data;
		if(index == 0)
			continue;

		/* Format the index string to describe this recycled object */
		index_s = g_strdup_printf("#%ld", index);

		/* Get the current statistics for this recycled object */
		obj = edv_recycled_object_stat(index_path, index);
		if(obj != NULL)
		{
			/* Change this recycled object's permissions */
			status = edv_recycled_object_op_chmod_iterate(
				core,
				index_path,
				obj,
				permissions,
				permissions_s,
				modified_indicies_list,
				toplevel,
				show_progress,
				interactive,
				yes_to_all,
				&nobjs_processed,
				nobjs,
				time_start
			);
		}
		else
		{
			/* Unable to obtain the recycled object's statistics */
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
				edv_recycle_bin_index_get_error(),
				index
			);
			edv_recycled_object_op_copy_error_message(core, msg);
			g_free(msg);
			status = -1;
		}

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_RECYCLED_OBJECT_CHMOD,
			time_start,
			edv_time(),
			status,
			(obj != NULL) ? obj->name : index_s,	/* Source */
			permissions_s,		/* Target */
			core->last_error_ptr	/* Comment */
		);

		/* Delete the recycled object's statistics */
		edv_recycled_object_delete(obj);

		g_free(index_s);

		if(status != 0)
			break;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change ownership iteration.
 */
static gint edv_recycled_object_op_chown_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const gint owner_id, const gint group_id,
	const gchar *owner_group_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{		\
 return(_v_);					\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar       *name_shortened = edv_path_shorten(
			obj->name,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),          *msg = g_strdup_printf(
"Changing ownership of:\n\
\n\
    %s\n",
			name_shortened
		);
		g_free(name_shortened);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Ownership",
				msg,
				(const guint8 **)icon_owned_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	/* Set the new ownership */
	obj->owner_id = owner_id;
	obj->group_id = group_id;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = edv_recycle_bin_index_set(
		index_path,
		obj->index,
		obj
	);
	if(status != 0)
	{
		/* Unable to commit the changes */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
			edv_recycle_bin_index_get_error(),
			obj->index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Add this index to the modified indicies list */
		if(modified_indicies_list != NULL)
			*modified_indicies_list = g_list_append(
				*modified_indicies_list,
				(gpointer)obj->index
			);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change ownership.
 *
 *	The indicies_list specifies a GList of gulong indicies
 *	describing the recycled objects to change the ownership of.
 *
 *	The owner_id specifies the ID of the new owner.
 *
 *	The group_id specifies the ID of the new group.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of gulong indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_recycled_object_op_chown(
	EDVCore *core,
	GList *indices_list,
	const gint owner_id, const gint group_id,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint		status,
			nobjs_processed,
			nobjs;
	gulong		index,
			time_start;
	gchar		*owner_s, *group_s,
			*owner_group_s = NULL;
	gchar *index_s;
	const gchar *index_path;
	GList *glist;
	CfgList *cfg_list;
	EDVRecycledObject *obj;

	if(modified_indicies_list != NULL)
		*modified_indicies_list = NULL;

	/* Clear the last error message */
	edv_recycled_object_op_copy_error_message(core, NULL);

	if((core == NULL) || (indices_list == NULL) ||
	   (yes_to_all == NULL)
	)
	{
		edv_recycled_object_op_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Attempt to lock the Recycle Bin */
	status = edv_recycled_object_op_lock(
		core,
		index_path,
		"Change Ownership",
		toplevel,
		show_progress,
		interactive,
		yes_to_all
	);
	if(status != 0)
		return(status);

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
 g_free(owner_group_s);				\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Format the owner/group string */
	owner_s = edv_uid_uid_to_name(
		core->uids_list,
		owner_id, NULL
	);
	group_s = edv_gid_gid_to_name(
		core->gids_list,
		group_id, NULL
	);
	owner_group_s = g_strconcat(
		owner_s, ".", group_s, NULL
	);
	g_free(owner_s);
	g_free(group_s);

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		index = (gulong)glist->data;
		if(index == 0)
			continue;

		/* Format the index string to describe this recycled object */
		index_s = g_strdup_printf("#%ld", index);

		/* Get the current statistics for this recycled object */
		obj = edv_recycled_object_stat(index_path, index);
		if(obj != NULL)
		{
			/* Change this recycled object's ownership */
			status = edv_recycled_object_op_chown_iterate(
				core,
				index_path,
				obj,
				owner_id, group_id,
				owner_group_s,
				modified_indicies_list,
				toplevel,
				show_progress,
				interactive,
				yes_to_all,
				&nobjs_processed,
				nobjs,
				time_start
			);
		}
		else
		{
			/* Unable to obtain the recycled object's statistics */
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
				edv_recycle_bin_index_get_error(),
				index
			);
			edv_recycled_object_op_copy_error_message(core, msg);
			g_free(msg);
			status = -1;
		}

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_RECYCLED_OBJECT_CHOWN,
			time_start,
			edv_time(),
			status,
			(obj != NULL) ? obj->name : index_s,	/* Source */
			owner_group_s,		/* Target */
			core->last_error_ptr	/* Comment */
		);

		/* Delete the recycled object's statistics */
		edv_recycled_object_delete(obj);

		g_free(index_s);

		if(status != 0)
			break;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change time stamps iteration.
 */
static gint edv_recycled_object_op_chtime_iterate(
	EDVCore *core,
	const gchar *index_path,
	EDVRecycledObject *obj,
	const gulong atime, const gulong mtime, const gulong dtime,
	const gchar *time_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{		\
 return(_v_);					\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		gchar       *name_shortened = edv_path_shorten(
			obj->name,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),          *msg = g_strdup_printf(
"Changing time stamps of:\n\
\n\
    %s\n",
			name_shortened
		);
		g_free(name_shortened);
		if(ProgressDialogIsQuery())
		{
			ProgressDialogUpdate(
				NULL, msg, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		}
		else
		{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Changing Time Stamps",
				msg,
				(const guint8 **)icon_time_stamp_32x32_xpm,
				"Stop"
			);
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				progress, EDV_PROGRESS_BAR_NTICKS, TRUE
			);
			gdk_flush();
		}
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	/* Set the new time stamps */
	obj->access_time = atime;
	obj->modify_time = mtime;
	obj->deleted_time = dtime;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = edv_recycle_bin_index_set(
		index_path,
		obj->index,
		obj
	);
	if(status != 0)
	{
		/* Unable to commit the changes */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %ld",
			edv_recycle_bin_index_get_error(),
			obj->index
		);
		edv_recycled_object_op_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Add this index to the modified indicies list */
		if(modified_indicies_list != NULL)
			*modified_indicies_list = g_list_append(
				*modified_indicies_list,
				(gpointer)obj->index
			);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
		const gfloat progress = (nobjs > 0) ?
			((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			errno = EINTR;
			CLEANUP_RETURN(-4);
		}
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change time stamps.
 *
 *	The indicies_list specifies a GList of gulong indicies
 *	describing the recycled objects to change the time stamps
 *	of.
 *
 *	The atime specifies the new access time in seconds since
 *	EPOCH.
 *
 *	The mtime specifies the new modify time in seconds since
 *	EPOCH.
 *
 *	The dtime specifies the new delete time in seconds since
 *	EPOCH.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of gulong indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_recycled_object_op_chtime(
	EDVCore *core,
	GList *indices_list,
	const gulong atime, const gulong mtime, const gulong dtime,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint		status,
			nobjs_processed,
			nobjs;
	gulong		index,
			time_start;
	gchar		*time_s = NULL,
			*index_s;
	const gchar	*format,
			*index_path;
	GList *glist;
	CfgList *cfg_list;
	EDVDateRelativity relativity;
	EDVRecycledObject *obj;

	if(modified_indicies_list != NULL)
		*modified_indicies_list = NULL;

	/* Clear the last error message */
	edv_recycled_object_op_copy_error_message(core, NULL);

	if((core == NULL) || (indices_list == NULL) ||
	   (yes_to_all == NULL)
	)
	{
		edv_recycled_object_op_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Attempt to lock the Recycle Bin */
	status = edv_recycled_object_op_lock(
		core,
		index_path,
		"Change Timestamps",
		toplevel,
		show_progress,
		interactive,
		yes_to_all
	);
	if(status != 0)
		return(status);

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
 g_free(time_s);				\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Format the time string */
	relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	time_s = edv_date_string_format(
		mtime,
		format, relativity
	);

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		time_start = edv_time();

		index = (gulong)glist->data;
		if(index == 0)
			continue;

		/* Format the index string to describe this recycled object */
		index_s = g_strdup_printf("#%ld", index);

		/* Get the current statistics for this recycled object */
		obj = edv_recycled_object_stat(index_path, index);
		if(obj != NULL)
		{
			/* Change this recycled object's time stamps */
			status = edv_recycled_object_op_chtime_iterate(
				core,
				index_path,
				obj,
				atime, mtime, dtime,
				time_s,
				modified_indicies_list,
				toplevel,
				show_progress,
				interactive,
				yes_to_all,
				&nobjs_processed,
				nobjs,
				time_start
			);
		}
		else
		{
			/* Unable to obtain the recycled object's statistics */
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%ld",
				edv_recycle_bin_index_get_error(),
				index
			);
			edv_recycled_object_op_copy_error_message(core, msg);
			g_free(msg);
			status = -1;
		}

		/* Record history */
		edv_append_history(
			core,
			EDV_HISTORY_RECYCLED_OBJECT_CHTIME,
			time_start,
			edv_time(),
			status,
			(obj != NULL) ? obj->name : index_s,	/* Source */
			time_s,			/* Target */
			core->last_error_ptr	/* Comment */
		);

		/* Delete the recycled object's statistics */
		edv_recycled_object_delete(obj);

		g_free(index_s);

		if(status != 0)
			break;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
