#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_convert_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_recycle_obj.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_recycle_object_get_error(EDVCore *core);
static void edv_recycle_object_copy_error_message(
	EDVCore *core,
	const gchar *msg
);

/* Internal "No Delete" Check */
static gboolean edv_recycle_object_no_delete_check(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gchar *path,
	EDVVFSObject *obj
);

/* Progress Callback */
static gint edv_recycle_object_progress_cb(
	gpointer data, const gulong pos, const gulong total
);

/* Counting */
static gint edv_recycle_object_count_objects(const gchar *path);

/* Delete */
static gint edv_recycle_object_purge_iterate(
	EDVCore *core,
	const gchar *index_path,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_purged_rtn,
	const gint nobjs
);
static gulong edv_recycle_object_recycle_iterate(
	EDVCore *core,
	const gchar *index_path,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn
);
gint edv_recycle_object(
	EDVCore *core,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_recycle_object_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Sets the last error message.
 */
static void edv_recycle_object_copy_error_message(
	EDVCore *core, const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Checks if the object may not be deleted according to the
 *	EDV_NO_DELETE_LIST (list of objects that may not be deleted).
 *
 *	Returns TRUE if the object may not be deleted.
 */
static gboolean edv_recycle_object_no_delete_check(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gchar *path,
	EDVVFSObject *obj
)
{
typedef struct {
	gchar		*path;
	gchar		*message;
	gint		error_level;
} EDVNoDeleteObject;
	gboolean no_delete;
	gint i;
	const gchar *s;
	gchar *parent_path;
	CfgList *cfg_list = core->cfg_list;
	EDVNoDeleteObject *nd_ptr = NULL;
	EDVNoDeleteObject no_delete_list[] = EDV_NO_DELETE_LIST;

	if(STRISEMPTY(path))
		return(TRUE);

	/* Iterate through no delete list, checking if an entry has a
	 * path that matches the given path
	 */
	i = 0;
	no_delete = FALSE;
	while(no_delete_list[i].path != NULL)
	{
		nd_ptr = &no_delete_list[i];

		/* Given object path matches one in the no delete list? */
		if(!strcmp((const char *)nd_ptr->path, (const char *)path))
		{
			/* Handle by error level of this no delete entry */
			switch(nd_ptr->error_level)
			{
			  case 0:		/* Only warn user */
				if(interactive)
				{


				}
				break;

			  case 1:		/* Query user for no delete */
				if(interactive && !(*yes_to_all))
				{
					gint response;
					gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se debe borrar, la razn:\n\
\n\
%s.\n\
\n\
Usted est seguro que usted quiere borrar este objeto?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne devrait pas tre effac, la raison:\n\
\n\
%s.\n\
\n\
Etes-vous sr que vous voulez effacer cet objet? "
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" sollte, Grund nicht gelscht werden:\n\
\n\
%s.\n\
\n\
Sind Sie sicher Sie dieses Objekt wollen lschen?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non dovrebbe essere cancellato, la ragione:\n\
\n\
%s.\n\
\n\
Lei sono sicuro che lei vuole cancellare quest'oggetto?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" zou, reden niet moeten geschrappet worden:\n\
\n\
%s.\n\
\n\
Bent u zeker u deze voorwerp wil schrappen?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao deve ser anulado, razo:\n\
\n\
%s.\n\
\n\
Esto seguro quer anular este objeto?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" stryker ikke, grunn:\n\
\n\
%s.\n\
\n\
Er De sikker De stryker dette objektet?"
#else
"The object \"%s\" should not be deleted, reason:\n\
\n\
%s.\n\
\n\
Are you sure you want to delete this object?"
#endif
						, path,
						(nd_ptr->message != NULL) ?
							nd_ptr->message :
							"Object is marked \"no_delete\" by this process"
					);
					edv_play_sound_question(core);
					CDialogSetTransientFor(toplevel);
					response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Advertir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer L'Avertissement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Warnen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare L'Avvertimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Waarschuwen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Aviso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Warning"
#else
"Delete Warning"
#endif
						, msg, NULL,
						CDIALOG_ICON_WARNING,
						CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
						CDIALOG_BTNFLAG_NO
					);
					CDialogSetTransientFor(NULL);
					g_free(msg);

					switch(response)
					{
					  case CDIALOG_RESPONSE_YES_TO_ALL:
						*yes_to_all = TRUE;
					  case CDIALOG_RESPONSE_YES:
						break;

					  default:
						no_delete = TRUE;
						break;
					}
				}
				else
				{
					no_delete = TRUE;
				}
				break;

			  default:			/* No delete under any case */
				no_delete = TRUE;
				if(interactive)
				{
					gchar *msg = g_strdup_printf(
"Unable to delete object \"%s\".\n\
\n\
%s.\n",
						path,
						(nd_ptr->message != NULL) ?
							nd_ptr->message :
							"Object is marked \"no_delete\" by this process"
					);
					edv_play_sound_error(core);
					edv_message_error(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
						, msg, NULL,
						toplevel
					);
					g_free(msg);
				}
				break;
			}
			/* Stop iterating through no delete list and return
			 * response
			 */
			return(no_delete);
		}

		i++;
	}


	/* Check for other paths that are defined at run time and may
	 * not be deleted
	 */

	/* Special notations? */
	s = g_basename(path);
	if(!STRISEMPTY(s))
	{
		/* Current or parent directory? */
		if(!strcmp((const char *)s, ".") ||
		   !strcmp((const char *)s, "..")
		)
		{
			if(interactive)
			{
				gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Las anotaciones objetivas especiales tal como \"%s\"\n\
no se estn acostumbrados a reffer a un objeto que\n\
deber ser borrado."
#elif defined(PROG_LANGUAGE_FRENCH)
"Les notations spciales d'objet telles que \"%s\"\n\
ne peut pas tre utilis  reffer  un objet qui\n\
va tre effac."
#elif defined(PROG_LANGUAGE_GERMAN)
"Besondere Objekt Aufzeichnungen kann wie zum\n\
Beispiel \"%s\" zu reffer zu einem Objekt nicht benutzt\n\
werden, das gelscht werden soll."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le notazioni di oggetto speciali come \"%s\" non\n\
possono essere usate al reffer a un oggetto che \n\
essere cancellato."
#elif defined(PROG_LANGUAGE_DUTCH)
"Speciale voorwerp aantekeningen zal zoals \"%s\"\n\
te reffer aan een voorwerp niet misschien gebruikt\n\
worden dat geschrappet worden zal."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anotaes especiais de objeto tal como \"%s\"\n\
nao pode ser usado a reffer a um objeto que  ser\n\
anulado."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesiell objektnotasjoner slik som \"%s\" ikke\n\
brukt til reffer til et objekt stryker som."
#else
"Special object notations such as \"%s\" may not be\n\
used to reffer to an object that is to be deleted."
#endif
					, s
				);
				edv_play_sound_error(core);
				edv_message_error(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
					, msg, NULL,
					toplevel
				);
				g_free(msg);
			}
			return(TRUE);
		}
	}

	/* Home directory? */
	s = core->home_path;
	if(!STRISEMPTY(s) ?
		!strcmp((const char *)s, (const char *)path) : FALSE
	)
	{
		if(interactive)
		{
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se puede borrar, la razn:\n\
\n\
La gua buscadora del usuario."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne peut pas tre effac, la raison:\n\
\n\
L'annuaire de maison de l'utilisateur."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" kann, Grund nicht gelscht werden:\n\
\n\
Das Heim von Verbraucher Verzeichnis."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non pu essere cancellato, la ragione:\n\
\n\
L'elenco di casa dell'operatore."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" kan niet, reden is geschrappet:\n\
\n\
Het huis van gebruiker gids."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao pode ser anulado, razo:\n\
\n\
O guia de lar do operador."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" strket ikke, grunn:\n\
\n\
Brukers hjemkatalog."
#else
"Unable to delete object \"%s\".\n\
\n\
Object is the user's home directory.\n"
#endif
				, path
			);
			edv_play_sound_error(core);
			edv_message_error(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
				, msg, NULL,
				toplevel
			);
			g_free(msg);
		}
		return(TRUE);
	}

	/* Is the object part of the recycled objects system?
	 *
	 * Check this by getting the parent of the recycled objects
	 * index file, this parent directory will be checked to see if
	 * it is the parent of the specified object
	 *
	 * If it is then we know that the object is in within the
	 * recycled objects directory and therefore is a part of the
	 * recycled objects system (which may not be deleted)
	 */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	parent_path = (s != NULL) ? g_dirname(s) : NULL;
	if(parent_path != NULL)
	{
		if(edv_path_is_parent(parent_path, path))
		{
			if(interactive)
			{
				gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se puede borrar, la razn:\n\
\n\
El objeto forma parte del sistema de objetos de\n\
recycled (purga de uso en lugar)."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne peut pas tre effac, la raison:\n\
\n\
L'objet fait partie du systme d'objets recycl\n\
(la purge d'usage plutt)."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" kann, Grund nicht gelscht werden:\n\
\n\
Objekt ist teil des wiederverwerteten objekte systems\b\
(gebrauch reinigt anstatt)."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non pu essere cancellato, la ragione:\n\
\n\
L'oggetto  la parte del sistema di oggetti riciclato\n\
(la purga di uso invece)."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" kan niet, reden is geschrappet:\n\
\n\
Voorwerp is deel van de gerecyclde voorwerpen systeem\n\
(gebruik zuivering in plaats van)."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao pode ser anulado, razo:\n\
\n\
O objeto  parte do sistema de objetos de recycled\n\
(purge de uso contrriamente)."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" strket ikke, grunn:\n\
\n\
Objekt er del av det resirkuleredde objektsystem renser\n\
(bruk istedenfor)."
#else
"Unable to delete object \"%s\".\n\
\n\
Object is part of the recycled objects system.\n"
#endif
					,
					path
				);
				edv_play_sound_error(core);
				edv_message_error(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
					,
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
			}

			g_free(parent_path);

			return(TRUE);
		}
	}

	g_free(parent_path);

	return(FALSE);
}


/*
 *	Recycle recover or delete progress callback.
 */
static gint edv_recycle_object_progress_cb(
	gpointer data, const gulong pos, const gulong total
)
{
	gint status = 0;
/*	EDVCore *core = EDV_CORE(data); */

	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			((total > 0) && (pos >= 0)) ?
				((gfloat)pos / (gfloat)total) : -1.0f,
			EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			status = -1;
	}

	return(status);
}


/*
 *	Counts the number of objects recursively.
 *
 *	The path specifies the object to count.
 */
static gint edv_recycle_object_count_objects(const gchar *path)
{
	const gboolean	recursive = TRUE,
			archive = TRUE;
	gint nobjs = 0;
	EDVVFSObject *obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
		return(nobjs);

	/* Count this object */
	nobjs++;

	/* Directory? */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		EDVDirectory *dp = edv_directory_open(
			path,
			FALSE,			/* Unsorted */
			FALSE			/* Exclude notations */
		);
		if(dp != NULL)
		{
			const gchar *name;
			gchar *full_path;

			for(name = edv_directory_next(dp);
			    name != NULL;
			    name = edv_directory_next(dp)
			)
			{
				full_path = g_strconcat(
					path,
					G_DIR_SEPARATOR_S,
					name,
					NULL
				);
				nobjs += edv_recycle_object_count_objects(full_path);
				g_free(full_path);
			}

			edv_directory_close(dp);
		}
	}
	/* Link? */
	else if(EDV_VFS_OBJECT_IS_LINK(obj))
	{
		/* Count links as the link target? */
		if(!archive && recursive)
		{
			/* Get the target object's statistics */
			EDVVFSObject *obj = edv_vfs_object_stat(path);
			if(obj != NULL)
			{
				/* Is the target a directory? */
				if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
				{
					EDVDirectory *dp = edv_directory_open(
						path,
						FALSE,		/* Unsorted */
						FALSE		/* Exclude notations */
					);
					if(dp != NULL)
					{
						const gchar *name;
						gchar *full_path;

						for(name = edv_directory_next(dp);
							name != NULL;
							name = edv_directory_next(dp)
						)
						{
							full_path = g_strconcat(
								path,
								G_DIR_SEPARATOR_S,
								name,
								NULL
							);
							nobjs += edv_recycle_object_count_objects(full_path);
							g_free(full_path);
						}

						edv_directory_close(dp);
					}
				}

				edv_vfs_object_delete(obj);
			}
		}
	}

	edv_vfs_object_delete(obj);

	return(nobjs);
}


/*
 *	Permanently deletes the VFS object from the VFS and does not
 *	place it into the Recycle Bin.
 *
 *	If the object is a directory then all of its child objects
 *	will also be permanently deleted.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_recycle_object_purge_iterate(
	EDVCore *core,
	const gchar *index_path,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_purged_rtn,
	const gint nobjs
)
{
	gboolean no_delete;
	gint status;
	GList *names_list;
	CfgList *cfg_list = core->cfg_list;
	EDVPermissionFlags permissions;
	EDVVFSObject	*obj_lvalues,
			*obj_values;

	/* Get this object's local statistics */
	obj_lvalues = edv_vfs_object_lstat(path);
	if(obj_lvalues == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_recycle_object_copy_error_message(core, msg);
		g_free(msg);
		errno = (int)error_code;
		return(-1);
	}

	/* Check if this object's permissions specify that it is
	 * read-only and query the user to confirm deleting of this
	 * object
	 */
	permissions = obj_lvalues->permissions;
	if(!(permissions & EDV_PERMISSION_UW))
	{
		/* Query the user to remove this read-only object
		 * only if interactive is specified
		 */
		if(interactive && !(*yes_to_all))
		{
			gint response;
			gchar *msg = g_strdup_printf(
"Delete read-only %s \"%s\"?",
				edv_object_type_to_object_name_lower(obj_lvalues->type),
				obj_lvalues->name
			);
		        edv_play_sound_question(core);
		        CDialogSetTransientFor(toplevel);
		        response = CDialogGetResponse(
"Confirm Delete Read-Only Object",
				msg,
				NULL,
				CDIALOG_ICON_QUESTION,
			        CDIALOG_BTNFLAG_YES |
					CDIALOG_BTNFLAG_YES_TO_ALL |
				        CDIALOG_BTNFLAG_NO |
					CDIALOG_BTNFLAG_CANCEL,
				CDIALOG_BTNFLAG_NO
			);
		        g_free(msg);
		        CDialogSetTransientFor(NULL);
			switch(response)
			{
			   case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			   case CDIALOG_RESPONSE_YES:
				break;
			   case CDIALOG_RESPONSE_NO:
		                edv_vfs_object_delete(obj_lvalues);
				errno = EINTR;
				return(-5);
				break;
			   default:	/* CDIALOG_RESPONSE_CANCEL */
		                edv_vfs_object_delete(obj_lvalues);
				errno = EINTR;
				return(-4);
				break;
			}
		}
	}

	/* Do not check if the effective user ID of this process can
	 * delete this object or not, allow the system to determine
	 * that when the actual deleting calls take place
	 */

	/* Get this object's destination statistics and check if this
	 * object may not be deleted by this program because it is
	 * part of the integraty of the system or similar reason
	 */
	obj_values = edv_vfs_object_stat(path);
	no_delete = edv_recycle_object_no_delete_check(
		core,
		toplevel,
		interactive,
		yes_to_all,
		path,
		obj_values
	);
	if(no_delete)
	{
		/* This object may not be deleted because it is part of
		 * the integraty of the system or similar reason
		 */

		/* Set the last error only if not interactive since
		 * calling edv_recycle_object_no_delete_check() would
		 * have already displayed appropriate the error message
		 */
		if(!interactive)
		{
			gchar *msg = g_strdup_printf(
"The object:\n\
\n\
    %s\n\
\n\
Is part of the function of this process or system\n\
and may not be deleted.",
				path
			);
			edv_recycle_object_copy_error_message(core, msg);
			g_free(msg);
		}
		edv_vfs_object_delete(obj_lvalues);
		edv_vfs_object_delete(obj_values);
		errno = EPERM;
		return(-1);
	}

	edv_vfs_object_delete(obj_values);

	/* If the specified object to delete is locally a directory
	 * then get the contents of that directory and delete
	 * recursively
	 */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj_lvalues))
		names_list = edv_directory_list(
			path,
			FALSE,			/* Unsorted */
			FALSE			/* Exclude notations */
		);
	else
		names_list = NULL;
	status = 0;
	if(names_list != NULL)
	{
		gint status2;
		const gchar *name;
		gchar *child_path;
		GList *glist;

		/* Purge all the objects in this directory */
		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			child_path = edv_paths_join(
				path,
				name
			);
			if(child_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Purge this object and any of its children */
			status2 = edv_recycle_object_purge_iterate(
				core,
				index_path,
				child_path,
				index_list_rtn,
				toplevel,
				show_progress,
				interactive,
				yes_to_all,
				nobjs_purged_rtn,
				nobjs
			);
			if(status2 != 0)
				status = status2;

			g_free(child_path);

			/* Stop purging if an error occured or the
			 * user aborted (except "no" response)
			 */
			if((status2 != 0) && (status2 != -5))
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}
		g_list_free(names_list);
	}

	/* Error, user aborted, or user responded with "no" while
	 * deleting any child objects?
	 */
	if(status != 0)
	{
		edv_vfs_object_delete(obj_lvalues);
		return(status);
	}

	/* Map/update the progress dialog? */
	if(show_progress)
	{
		gchar	*path_shortened = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Borrar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrappen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anular:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryking:\n\
\n\
    %s\n"
#else
"Deleting:\n\
\n\
    %s\n"
#endif
			,
			path_shortened
		);
		g_free(path_shortened);
		edv_progress_dialog_map_delete_animated(
			cfg_list,
			msg,
			(nobjs > 0) ?
				((gfloat)(*nobjs_purged_rtn) / (gfloat)nobjs) : -1.0f,
			toplevel,
			FALSE,
			EDV_DELETE_METHOD_PURGE
		);
		g_free(msg);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(obj_lvalues);
			errno = EINTR;
			return(-4);
		}
	}

	/* Permanently purge this object by its type
	 *
	 * Directory?
	 */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj_lvalues))
	{
		if(edv_directory_remove(
			path,
			FALSE,			/* Do not recurse */
			FALSE,			/* Do not force */
			NULL,			/* No removed paths list return */
			NULL, NULL
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to delete the directory:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_recycle_object_copy_error_message(core, msg);
			g_free(msg);
			edv_vfs_object_delete(obj_lvalues);
			errno = (int)error_code;
			return(-1);
		}
	}
	/* All else, assume it can be unlink'ed */
	else
	{
		if(edv_unlink(path))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_recycle_object_copy_error_message(core, msg);
			g_free(msg);
			edv_vfs_object_delete(obj_lvalues);
			errno = (int)error_code;
			return(-1);
		}
	}

	/* Successfully deleted the object, add a 0 index to the
	 * recycled objects index list so that the calling function
	 * knows how many objects were purged
	 */
	if(index_list_rtn != NULL)
		*index_list_rtn = g_list_append(
			*index_list_rtn,
			(gpointer)0l		/* gulong */
		);

	/* Count this object as being purged */
	*nobjs_purged_rtn = (*nobjs_purged_rtn) + 1;

	/* Update the progress */
	if(show_progress && ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			(nobjs > 0) ?
				((gfloat)(*nobjs_purged_rtn) / (gfloat)nobjs) : -1.0f,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(obj_lvalues);
			errno = EINTR;
			return(-4);
		}
	}

	edv_vfs_object_delete(obj_lvalues);

	return(0);
}

/*
 *	Places the VFS object to the Recycle Bin.
 *
 *	Returns the recycled object index on success or 0 on error.
 */
static gulong edv_recycle_object_recycle_iterate(
	EDVCore *core,
	const gchar *index_path,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn
)
{
	gboolean no_delete;
	gulong		index,
			cur_time;
	gchar *parent_path;
	GList *names_list;
	CfgList *cfg_list = core->cfg_list;
	EDVPermissionFlags permissions;
	EDVVFSObject	*obj_values,
			*obj_lvalues;
	EDVRecycledObject *obj;

	/* Get this object's local statistics */
	obj_lvalues = edv_vfs_object_lstat(path);
	if(obj_lvalues == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_recycle_object_copy_error_message(core, msg);
		g_free(msg);
		errno = (int)error_code;
		*status_rtn = -1;
		return(0l);
	}

	/* Check if this object's permissions specify that it is
	 * read-only and query the user to confirm deleting of this
	 * object
	 */
	permissions = obj_lvalues->permissions;
	if(!(permissions & EDV_PERMISSION_UW))
	{
		/* Query the user to remove this read-only object
		 * only if interactive is specified
		 */
		if(interactive && !(*yes_to_all))
		{
			gint response;
			gchar *msg = g_strdup_printf(
"Delete read-only %s \"%s\"?",
				edv_object_type_to_object_name_lower(obj_lvalues->type),
				obj_lvalues->name
			);
		        edv_play_sound_question(core);
		        CDialogSetTransientFor(toplevel);
		        response = CDialogGetResponse(
"Confirm Delete Read-Only Object",
				msg,
				NULL,
				CDIALOG_ICON_QUESTION,
			        CDIALOG_BTNFLAG_YES |
					CDIALOG_BTNFLAG_YES_TO_ALL |
				        CDIALOG_BTNFLAG_NO |
					CDIALOG_BTNFLAG_CANCEL,
				CDIALOG_BTNFLAG_NO
			);
		        g_free(msg);
		        CDialogSetTransientFor(NULL);
			switch(response)
			{
			   case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			   case CDIALOG_RESPONSE_YES:
				break;
			   case CDIALOG_RESPONSE_NO:
		                edv_vfs_object_delete(obj_lvalues);
				*status_rtn = -5;
				errno = EINTR;
				return(0l);
				break;
			   default:	/* CDIALOG_RESPONSE_CANCEL */
		                edv_vfs_object_delete(obj_lvalues);
				*status_rtn = -4;
				errno = EINTR;
				return(0l);
				break;
			}
		}
	}

	/* Do not check if the effective user ID of this process can
	 * delete this object or not, allow the system to determine
	 * that when the actual deleting calls take place
	 */

	/* Get this object's destination statistics and check if this
	 * object may not be deleted by this program because it is
	 * part of the integraty of the system or similar reason
	 */
	obj_values = edv_vfs_object_stat(path);
	no_delete = edv_recycle_object_no_delete_check(
		core,
		toplevel,
		interactive,
		yes_to_all,
		path,
		obj_values
	);
	if(no_delete)
	{
		/* This object may not be deleted because it is part
		 * of the integraty of the system or similar reason
		 *
		 * Set the last error only if not interactive since
		 * calling edv_recycle_object_no_delete_check() would
		 * have already displayed appropriate the error
		 * message
		 */
		if(!interactive)
		{
			gchar *msg = g_strdup_printf(
"The object:\n\
\n\
    %s\n\
\n\
Is part of the function of this process or system\n\
and may not be deleted.",
				path
			);
			edv_recycle_object_copy_error_message(core, msg);
			g_free(msg);
		}
		edv_vfs_object_delete(obj_lvalues);
		edv_vfs_object_delete(obj_values);
		*status_rtn = -1;
		errno = EPERM;
		return(0l);
	}

	edv_vfs_object_delete(obj_values);

	/* If the specified object to delete is locally a directory
	 * then get the contents of that directory and delete
	 * recursively
	 */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj_lvalues))
		names_list = edv_directory_list(
			path,
			FALSE,			/* Unsorted */
			FALSE			/* Exclude notations */
		);
	else
		names_list = NULL;
	if(names_list != NULL)
	{
		gint status;
		const gchar *name;
		gchar *child_path;
		GList *glist;

		/* Recycle all the objects in this directory */
		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			child_path = edv_paths_join(
				path,
				name
			);
			if(child_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Reset the status each time before calling
			 * edv_recycle_object_recycle_iterate() so
			 * that it does not use our status return
			 * which may cause confusion with our level
			 * of recusion's errors
			 */
			status = 0;

			/* Recycle this object and all of its
			 * children
			 */
			index = edv_recycle_object_recycle_iterate(
				core,
				index_path,
				child_path,
				index_list_rtn,
				toplevel,
				show_progress,
				interactive,
				yes_to_all,
				&status
			);
			if(status != 0)
				*status_rtn = status;

			g_free(child_path);

			/* Stop recycling if an error occured or the
			 * user aborted (except a "no" response)
			 */
			if((status != 0) && (status != -5))
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}
		g_list_free(names_list);
	}

	/* Error, user aborted, or user responded with "no" while
	 * deleting any child objects?
	 */
	if(*status_rtn != 0)
	{
		edv_vfs_object_delete(obj_lvalues);
		return(0l);
	}

	/* Map/update the progress dialog? */
	if(show_progress)
	{
		gchar	*path_shortened = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Borrar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrappen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anular:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryking:\n\
\n\
    %s\n"
#else
"Deleting:\n\
\n\
    %s\n"
#endif
			,
			path_shortened
		);
		g_free(path_shortened);
		edv_progress_dialog_map_delete_animated(
			cfg_list,
			msg,
			0.0f,
			toplevel,
			FALSE,
			EDV_DELETE_METHOD_RECYCLE
		);
		g_free(msg);
		if(ProgressDialogStopCount() > 0)
		{
			edv_vfs_object_delete(obj_lvalues);
			*status_rtn = -4;
			errno = EINTR;
			return(0l);
		}
	}

	/* Record the current time to mark the start time of deletion */
	cur_time = edv_time();

	/* Convert the local VFS object statistics to a recycled object */
	obj = edv_convert_vfs_object_to_recycled_object(obj_lvalues);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_recycle_object_copy_error_message(core, msg);
		g_free(msg);
		edv_vfs_object_delete(obj_lvalues);
		*status_rtn = -3;
		return(0);
	}

	/* Get this object's location */
	parent_path = g_dirname(path);
	if(parent_path == NULL)
		parent_path = g_strdup("/");

	/* Set the values from this object to the new recycled object
	 * that were not converted or are not suppose to be converted
	 */
	obj->index = 0l;			/* Not known yet */
	obj->deleted_time = cur_time;

	/* Add an entry to the recycled objects index file */
	index = edv_recycle_bin_index_add(
		index_path,
		obj
	);
	if(index != 0l)
	{
		/* Recycle this object */
		*status_rtn = edv_recycle_bin_index_recycle(
			index_path,
			index,
			path,
			show_progress ? edv_recycle_object_progress_cb : NULL,
			core
		);
		/* User aborted operation? */
		if((*status_rtn == -4) || (*status_rtn == -5))
		{
			(void)edv_recycle_bin_index_remove(
				index_path,
				index
			);
			index = 0l;
		}
		/* Other error occured? */
		else if(*status_rtn != 0)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				edv_recycle_bin_index_get_error()
			);
			edv_recycle_object_copy_error_message(core, msg);
			g_free(msg);
			(void)edv_recycle_bin_index_remove(
				index_path,
				index
			);
			index = 0l;
			errno = (int)error_code;
		}
		else
		{
			/* Successfully recycled the object, append the
			 * recycled object's index to the recycled
			 * objects index list
			 */
			if(index_list_rtn != NULL)
				*index_list_rtn = g_list_append(
					*index_list_rtn,
					(gpointer)index	/* gulong */
				);
		}
	}
	else
	{
		core->last_error_ptr =
"Unable to generate a new recycled object index.";
		*status_rtn = -1;
	}

	edv_vfs_object_delete(obj_lvalues);
	edv_recycled_object_delete(obj);
	g_free(parent_path);

	return(index);
}

/*
 *	Recycles or purges the VFS object.
 *
 *	If the global configuration specifies purge on delete then
 *	the object will be deleted instead of being put in to the
 *	recycle bin, otherwise the object will be put into the recycle
 *	bin.
 *
 *	The path specifies the full path to the object to delete.
 *
 *	The index_list_rtn specifies a pointer to a GList * that will
 *	be set to a list of guint indices describing all of the
 *	objects deleted. If index_list_rtn is NULL then no list will
 *	be returned. The calling function must delete the returned
 *	list.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	shown during this operation.
 *
 *	If interactive is TRUE then the user will be notified and
 *	queried if any problems arise.
 *
 *	The yes_to_all specifies a pointer to the preinitialized
 *	current gboolean "yes to all" value. This value may be
 *	set to TRUE during this call if the user responds with
 *	"yes to all" to a query.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_object(
	EDVCore *core,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint		status,
			nobjs_purged,
			nobjs;
	gulong index;
	const gchar *index_path;
	gchar *dpath = NULL;
	CfgList *cfg_list;
	EDVDeleteMethod delete_method;

	/* Reset the return values */
	if(index_list_rtn != NULL)
		*index_list_rtn = NULL;

	/* Clear the last error message */
	edv_recycle_object_copy_error_message(core, NULL);

	if((core == NULL) || STRISEMPTY(path) ||
	   (yes_to_all == NULL)
	)
	{
		edv_recycle_object_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get path to recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	{
		core->last_error_ptr = "Unable to get the recycled objects index file.";
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	while(TRUE)
	{
		status = edv_recycle_bin_index_lock(
			index_path,
			core->pid
		);
		if(status == 0)
			break;

		/* Already locked? */
		if(status == -6)
		{
			/* Query the user to retry or abort */
			if(interactive)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Delete Failed",
"The Recycle Bin is currently locked and in use.\n\
\n\
Try again?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Retry */
					continue;
				}
				else
				{
					/* Abort */
					errno = EINTR;
					return(-4);
				}
			}
			else
			{
				core->last_error_ptr =
"The Recycle Bin is currently locked.";
				errno = EBUSY;
				return(-6);
			}
		}

		/* Other error occured while attempting to lock */
		core->last_error_ptr = "Unable to lock the Recycle Bin.";
		return(status);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 /* Unlock the recycle bin */			\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
						\
 g_free(dpath);					\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Get the delete method which will tell us if we should
	 * recycle (place the VFS object to the Recycle Bin) or purge
	 * (permanently delete the VFS object from the VFS)
	 */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);


	/* Get copy of path of object to be deleted */
	dpath = g_strdup(path);
	if(dpath == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		CLEANUP_RETURN(-3);
	}

	status = 0;

	/* Delete this object by the delete method*/
	switch(delete_method)
	{
	    case EDV_DELETE_METHOD_PURGE:
		nobjs_purged = 0;
		nobjs = edv_recycle_object_count_objects(dpath);
		status = edv_recycle_object_purge_iterate(
			core,
			index_path,
			dpath,
			index_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			&nobjs_purged,
			nobjs
		);
		break;
	  case EDV_DELETE_METHOD_RECYCLE:
		index = edv_recycle_object_recycle_iterate(
			core,
			index_path,
			dpath,
			index_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			&status
		);
		break;
	  default:
		status = -2;
		break;
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_VFS_OBJECT_DELETE,
		time_start,
		edv_time(),
		status,
		dpath,				/* Source */
		NULL,				/* No target */
		core->last_error_ptr		/* Comments */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
